/*
//====================================================================
// MWT_SiriusReceiverFrameDecoder.c
// Implementation of the Sirius Receiver Frame decoder ADT
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//
//====================================================================
*/

/*
//====================================================================
// Required includes
//====================================================================
*/

#include <stdio.h>
#include "MWT_SiriusReceiverFrameDecoder.h"
#include "MWT_SiriusDPUAssembler.h"

/*
//====================================================================
// ADT data
//====================================================================
*/

#define MAXIMUM_ALLOWED_FRAME_LENGTH_BYTES 4096
#define FRAME_BORDER_FLAG_BYTE 0x7E
#define FRAME_TRANSPARENCY_FLAG_BYTE 0x7D

/* The CRC table */
static TTwoByteInteger_Unsigned crc_table[256];

/* Is the CRC table initialized? */
static TByte_Unsigned bool_isCRCTableInitialized = 0;

/* The frame buffer*/
static TByte_Unsigned MWTSiriusFrameDecoder_frameBuffer[MAXIMUM_ALLOWED_FRAME_LENGTH_BYTES];

/*
The number of bytes of the current frame held in the frame buffer
We do not hold the flag bytes in the frame buffer
*/
static TFourByteInteger_Unsigned MWTSiriusFrameDecoder_numFrameBytes = 0;


/* If this is non-zero, then we are reading in frames from the beginning
Otherwise, we have to skip bytes until the next frame border flag byte
to get resynchronized with the beginning of a frame
*/
static TByte_Unsigned MWTSiriusFrameDecoder_bool_frameSynchronized = 0;

/* If this is non-zero, then the next byte of input is transparency encoded*/
static TByte_Unsigned MWTSiriusFrameDecoder_bool_transparencyFlagOn = 0;




/*
//====================================================================
// Private methods
//====================================================================
*/

/*
//====================================================================
// MWTSiriusFrameDecoder_Internal_desynchronized:
//
// Call this if an error occurs in the transport protocol or in
// processing a frame.  This resets the decoder so that it will
// not make the assumption that it is still frame synchronized.
// Bytes may have dropped from the input pipe so this is very
// important.
//
//
//====================================================================
*/

void MWTSiriusFrameDecoder_Internal_desynchronized()
{
	/* Reset DPU processor */
	MWTSiriusDPUAssembler_reset();

	/* Reset the frame buffer to empty*/
	MWTSiriusFrameDecoder_numFrameBytes = 0;

	/* Transparency flag is off */
	MWTSiriusFrameDecoder_bool_transparencyFlagOn = 0;

	/* Not frame synchronized*/
	MWTSiriusFrameDecoder_bool_frameSynchronized = 0;

}

/*
//====================================================================
// MWTSiriusFrameDecoder_Internal_resynchronized:
//
// Call this method if the MWTSiriusFrameDecoder_bool_frameSynchronized
// variable is 0 and a frame border byte is encountered.
//
// This resets all the variables of the decoder state to handle
// aligning with a new product.
//====================================================================
*/

TMW_ResultCode MWTSiriusFrameDecoder_Internal_resynchronized()
{
	/* Reset DPU processor */
	MWTSiriusDPUAssembler_reset();

	/* Reset the frame buffer to empty*/
	MWTSiriusFrameDecoder_numFrameBytes = 0;

	/* Transparency flag is off */
	MWTSiriusFrameDecoder_bool_transparencyFlagOn = 0;

	/* Now frame synchronized*/
	MWTSiriusFrameDecoder_bool_frameSynchronized = 1;


	/*Done*/
	return MWRes_Success;
}

/*
//====================================================================
// MWTSiriusFrameDecoder_Internal_initializeCRCTable:
//
// This method initializes the CRC table. It need only be called
// once.  Once this method returns, the CRC table is ready for use.
//
// SIDE EFFECTS:
//
//	Modifies the following variables:
//		bool_isCRCTableInitialized
//
// RETURNS:
//
//	nothing
//====================================================================
*/
void MWTSiriusFrameDecoder_Internal_initializeCRCTable()
{
	/*
	//##############################################################
	// Generate CRC table
	//##############################################################
	*/

	TTwoByteInteger_Unsigned c;
	TFourByteInteger_Signed n, k;
	for (n = 0; n < 256; n++)
	{
		c = (unsigned short) n<<8;
		for (k = 0; k < 8; k++)
		{
			if (c & 0x8000)
				c = 0x1021 ^ (c<<1);
			else
				c = c<<1;
		}
		crc_table[n] = c;
	}

	/*
	//##############################################################
	// Note that table is initialized
	//##############################################################
	*/

	/* Table is now initialized */
	bool_isCRCTableInitialized = 1;


	/*
	//##############################################################
	// Done
	//##############################################################
	*/
}

/*
//====================================================================
// MWTSiriusFrameDecoder_Internal_LogMessage:
//
// Given a file, this writes the given string to it in logging format.
//====================================================================
*/

void MWTSiriusFrameDecoder_Internal_LogMessage
(
	FILE* p_file,
	const TASCIIChar* pstr_message
)
{
	if ((p_file != NULL) && (pstr_message != NULL))
	{
		fprintf (p_file, "%s\n", pstr_message);
	}
}

/*
//====================================================================
// MWTSiriusFrameDecoder_Internal_processFrame:
//
// This method processes the frame held in the frame buffer.
// It performs the following operations:
//
//		It checks the CRC of the frame to see if it was received
//		 correctly.
//
//		Calls DPU Assembler.
//
//
// PARAMETERS:
//
//	numProductStreamBufferBytes:
//		The number of bytes maximum that can be stored in the caller's
//		product assembly buffer.
//
//	p_productStreamBuffer:
//		The pointer to the first byte of the caller's product assembly
//		buffer.
//
//	p_inout_numProductStreamBufferBytesUsed
//		Pointer to a variable which tracks the number of bytes
//		of the product stream buffer filled.  This will be
//		modified with each call.
//		This will automatically be reset to 0 whenever a new
//		product instance is found in the frame stream.
//
// RETURNS:
//
//	MWRes_Success:
//		If the frame was processed but no product was completed.
//
//	MWRes_EndOfIteration:
//		The frame was processed correctly and a product was completed.
//		The caller's product assembly buffer will contain the
//		finished product.
//
//	MWRes_DataStructureFull:
//		The caller's product assembly buffer was not large enough to
//		hold the product.
//
//	MWRes_TransportProtocolError:
//		If the CRC value computed for the frame did not match
//		the CRC value in the frame.
//
//
//====================================================================
*/
TMW_ResultCode MWTSiriusFrameDecoder_Internal_processFrame
(
	TFourByteInteger_Unsigned numProductStreamBufferBytes,
	TByte_Unsigned* p_productStreamBuffer,
	TFourByteInteger_Unsigned* p_inout_numProductStreamBufferBytesUsed,
	FILE* p_headerInfoFile
)
{
	TMW_ResultCode MWRes;

	/* Gets CRC value that we calculate */
	TTwoByteInteger_Unsigned CalculatedCRCValue;

	/* Gets the CRC value embedded at the end of the frame*/
	TTwoByteInteger_Unsigned CRCValueInFrame;

	/* For iterating the frame buffer to calculate crc */
	TTwoByteInteger_Unsigned frameBufferIndex;

	/*
	//##############################################################
	// Ignore empty frames (idling link)
	//##############################################################
	*/
	if (MWTSiriusFrameDecoder_numFrameBytes == 0)
	{
		return MWRes_Success;
	}

	// Frame must be big enough to hold DPU header and 2 byte CRC
	if (MWTSiriusFrameDecoder_numFrameBytes < (sizeof(TWSI_Output_DPUHeader) + 2))
	{
		/* Not enough length for CRC*/
		MWTSiriusFrameDecoder_Internal_LogMessage (p_headerInfoFile, "Frame length too short to be valid");
		return MWRes_TransportProtocolError;
	}
	/*
	//##############################################################
	// Get CRC value embedded in the frame
	//##############################################################
	*/

	// CRC is stored as 2 byte unsigned integer in LOHI order
	CRCValueInFrame = (MWTSiriusFrameDecoder_frameBuffer[MWTSiriusFrameDecoder_numFrameBytes -2]) ;
	CRCValueInFrame += ((TTwoByteInteger_Unsigned)(MWTSiriusFrameDecoder_frameBuffer[MWTSiriusFrameDecoder_numFrameBytes - 1])) * 256;

	/*
	//##############################################################
	// Calculate CRC
	//##############################################################
	*/
	if (bool_isCRCTableInitialized == 0)
	{
		MWTSiriusFrameDecoder_Internal_initializeCRCTable();
	}


	/* CRC starting value */
	CalculatedCRCValue = 0xFFFF;

	/*
	// Buffer does not hold the starting and ending frame flags so
	// iterate all the bytes in the buffer up to the CRC which should be the
	// last value
	*/
	for (frameBufferIndex = 0; frameBufferIndex < MWTSiriusFrameDecoder_numFrameBytes-2; frameBufferIndex ++)
	{
		CalculatedCRCValue = (CalculatedCRCValue<<8) ^ crc_table[(unsigned char)MWTSiriusFrameDecoder_frameBuffer[frameBufferIndex] ^(unsigned char)(CalculatedCRCValue>>8)];
	}

	/*
	//##############################################################
	// Compare CRCs
	//##############################################################
	*/
	if (CalculatedCRCValue != CRCValueInFrame)
	{
		/* CRC doesn't check out */
		MWTSiriusFrameDecoder_Internal_LogMessage (p_headerInfoFile, "Calculated CRC did not match transmitted CRC");
		return MWRes_TransportProtocolError;
	}

	/*
	//##############################################################
	// Process the DPU
	// The DPU header begins the frame after the frame flag.
	// That is followed by the payload, which runs up until
	// the frame CRC (up until 3rd last byte of frame)
	//##############################################################
	*/
	MWRes = MWTSiriusDPUAssembler_processDPU
	(
		(TWSI_Output_DPUHeader*) &(MWTSiriusFrameDecoder_frameBuffer[0]), /* DPU header */
		(MWTSiriusFrameDecoder_numFrameBytes - (sizeof(TWSI_Output_DPUHeader) + 2) ), /* Length of payload in bytes */
		&(MWTSiriusFrameDecoder_frameBuffer[sizeof(TWSI_Output_DPUHeader)]), /* First byte of payload*/
		numProductStreamBufferBytes,
		p_productStreamBuffer,
		p_inout_numProductStreamBufferBytesUsed,
		p_headerInfoFile
	);


	/* Done with frame */
	MWTSiriusFrameDecoder_numFrameBytes = 0;

	/*
	//##############################################################
	// Return result
	//##############################################################
	*/
	return MWRes;
}

/*
//====================================================================
// MWTSiriusFrameDecoder_Internal_processInputByte:
//
// This method proceses a single byte of input and handles all of
// the frame synchronization/decoding logic from there.
//
// PARAMETERS:
//
//	p_byte:
//		Pointer to the byte to be processed.
//
//	numProductStreamBufferBytes:
//		The number of bytes of length of the product stream buffer.
//
//	p_productStreamBuffer:
//		Pointer to the first byte of the product stream buffer.
//
//	p_inout_numProductStreamBufferBytesUsed:
//		Pointer to a variable which tracks the number of bytes
//		of the product stream buffer filled.  This will be
//		modified with each call.
//		This will automatically be reset to 0 whenever a new
//		product instance is found in the frame stream.
//
//	p_out_bool_productInstanceCompleted:
//		Pointer to a variable which is set to non-zero by this method
//		if a product instance is completed, or is set to 0 by this
//		method otherwise.
//
// RETURNS:
//
//	MWRes_Success
//		If the byte is processed without error
//
//	MWRes_TransportProtocolError:
//		If something is wrong with the byte relative to the current
//		decoder state.  You should resynchronize the decoder.
//
//====================================================================
*/
TMW_ResultCode MWTSiriusFrameDecoder_Internal_processInputByte
(
	TByte_Unsigned* p_byte,
	TFourByteInteger_Unsigned numProductStreamBufferBytes,
	TByte_Unsigned* p_productStreamBuffer,
	TFourByteInteger_Unsigned* p_inout_numProductStreamBufferBytesUsed,
	TByte_Unsigned* p_out_bool_productInstanceCompleted,
	FILE* p_headerInfoFile
)
{
	/*
	//##############################################################
	// Some variables
	//##############################################################
	*/
	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// Branch based on byte contents
	//##############################################################
	*/

	if (*p_byte == FRAME_BORDER_FLAG_BYTE)
	{
		/* This should not occur directly after a transparency flag*/
		if (MWTSiriusFrameDecoder_bool_transparencyFlagOn != 0)
		{
			MWTSiriusFrameDecoder_Internal_desynchronized();
			MWTSiriusFrameDecoder_Internal_LogMessage (p_headerInfoFile, "Frame border flag immediately followed transparency flag");
			/* return error*/
			return MWRes_TransportProtocolError;
		}


		/* If we were not frame synchronized, we now are frame synchronized
			Also reset callers output buffer as we cannot be mid product if just
			synchronizing.

		   If we are frame synchronized, and we just completed a frame
		   (frame buffer is not empty) then process the frame.
		*/
		if (MWTSiriusFrameDecoder_bool_frameSynchronized == 0)
		{
			MWTSiriusFrameDecoder_Internal_resynchronized();
			p_inout_numProductStreamBufferBytesUsed = 0;
		}
		else if (MWTSiriusFrameDecoder_numFrameBytes > 0)
		{
			/* Process the frame */
			MWRes = MWTSiriusFrameDecoder_Internal_processFrame
			(
				numProductStreamBufferBytes,
				p_productStreamBuffer,
				p_inout_numProductStreamBufferBytesUsed,
				p_headerInfoFile
			);
			if (MWRes == MWRes_EndOfIteration)
			{
				/* A product was completed*/
				*p_out_bool_productInstanceCompleted = 1;

				/* Return success */
				return MWRes_Success;
			}
			else if (MWRes != MWRes_Success)
			{
				/* Bad frame, no longer frame synchronized*/
				MWTSiriusFrameDecoder_Internal_desynchronized();

				/* return error*/
				return MWRes;
			}

		}

		/*
		If we have no frame bytes, then the signal is idling by
		sending only border flags or there are multiple flags
		between frames, both of which are allowed.
		*/
	}
	else if (*p_byte == FRAME_TRANSPARENCY_FLAG_BYTE)
	{
		/* Should not get 2 transparency flags in a row */
		if (MWTSiriusFrameDecoder_bool_transparencyFlagOn != 0)
		{
			MWTSiriusFrameDecoder_Internal_desynchronized();

			/* return error*/
			MWTSiriusFrameDecoder_Internal_LogMessage (p_headerInfoFile, "Transparency flag immediately followed transparency flag");
			return MWRes_TransportProtocolError;
		}

		/* Next byte should be transparency decoded */
		MWTSiriusFrameDecoder_bool_transparencyFlagOn = 1;

	}
	else
	{
		/* Is there room for it ?*/
		if (MWTSiriusFrameDecoder_numFrameBytes >= MAXIMUM_ALLOWED_FRAME_LENGTH_BYTES)
		{
			MWTSiriusFrameDecoder_Internal_desynchronized();

			/* return error*/
			MWTSiriusFrameDecoder_Internal_LogMessage (p_headerInfoFile, "Frame exceeded maximum specified length");
			return MWRes_TransportProtocolError;
		}

		/* Transparency decode byte from stream if necessary, otherwise fetch from stream*/
		if (MWTSiriusFrameDecoder_bool_transparencyFlagOn != 0)
		{
			/* Transparency decode this byte and add it to the frame*/
			MWTSiriusFrameDecoder_frameBuffer[MWTSiriusFrameDecoder_numFrameBytes] = (*p_byte) ^0x20;

			/* The flagged transparency decode was done*/
			MWTSiriusFrameDecoder_bool_transparencyFlagOn = 0;
		}
		else
		{
			MWTSiriusFrameDecoder_frameBuffer[MWTSiriusFrameDecoder_numFrameBytes] = (*p_byte);
		}

		/* One more byte in frame */
		MWTSiriusFrameDecoder_numFrameBytes ++;

	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/

	return MWRes_Success;

}

/*
//====================================================================
// Public interface
//====================================================================
*/

void MWTSiriusFrameDecoder_reset()
{
	MWTSiriusFrameDecoder_Internal_desynchronized();
}



TMW_ResultCode MWTSiriusFrameDecoder_extractProduct
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_inout_numBytesUsed,
	TFourByteInteger_Unsigned numProductStreamBufferBytes,
	TByte_Unsigned* p_productStreamBuffer,
	TFourByteInteger_Unsigned* p_inout_numProductStreamBufferBytesUsed,
	FILE* p_headerInfoFile
)
{
	TMW_ResultCode MWRes;
	TFourByteInteger_Unsigned byteIndex;
	TByte_Unsigned bool_productCompleted;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(p_bytes == NULL) ||
		(p_inout_numBytesUsed == NULL) ||
		(p_productStreamBuffer == NULL) ||
		(p_inout_numProductStreamBufferBytesUsed == NULL)
	)
	{
		return MWRes_NullPointer;
	}


	/*
	//##############################################################
	// Pass the bytes through the frame processing routine
	// We exit internally from the loop on frame error or on
	// completion of a product instance.
	//##############################################################
	*/
	byteIndex = *p_inout_numBytesUsed;
	bool_productCompleted = 0;

	while (byteIndex < numBytes)
	{
		/*
		//##############################################################
		// Process the new byte
		//##############################################################
		*/
		MWRes = MWTSiriusFrameDecoder_Internal_processInputByte
		(
			&(p_bytes[byteIndex]),
			numProductStreamBufferBytes,
			p_productStreamBuffer,
			p_inout_numProductStreamBufferBytesUsed,
			&bool_productCompleted,
			p_headerInfoFile
		);

		/* One more byte used */
		byteIndex ++;

		/* Handle errors */
		if (MWRes != MWRes_Success)
		{
			*p_inout_numBytesUsed = byteIndex;
			return MWRes;
		}
		else if (bool_productCompleted != 0)
		{
			/*
			// A product was completed
			// Return so caller can get it out of assembly
			// buffer before we replace it with the next product.
			*/
			*p_inout_numBytesUsed = byteIndex;
			return MWRes_Success;
		}

	} /* Process next byte*/


	/*
	//##############################################################
	// We ran out of input
	//##############################################################
	*/
	*p_inout_numBytesUsed = byteIndex;
	return MWRes_InputBufferExhausted;



}

/*//--------------------------------------------------------------------*/

TMW_ResultCode MWTSiriusFrameDecoder_deframeProductFile
(
	const TASCIIChar* pstr_inputFrameFilePath,
	const TASCIIChar* pstr_outputProductFilePath,
	TByte_Unsigned bool_outputHeaderInfoToFile,
	const TASCIIChar* pstr_headerInfoFilePath
)
{
#define NUM_FILE_BUFFER_BYTES 4096

	TMW_ResultCode MWRes;

	FILE* p_inputFile;
	FILE* p_outputFile;
	FILE* p_headerInfoFile;

	TByte_Unsigned inputFileBuffer[NUM_FILE_BUFFER_BYTES];
	TFourByteInteger_Unsigned numInputFileBufferBytes;
	TFourByteInteger_Unsigned numInputFileBufferBytesUsed;

	TByte_Unsigned outputFileBuffer[NUM_FILE_BUFFER_BYTES];
	TFourByteInteger_Unsigned numOutputFileBufferBytesUsed;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(pstr_inputFrameFilePath == NULL) ||
		(pstr_outputProductFilePath == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	if
	(
		(bool_outputHeaderInfoToFile != 0) &&
		(pstr_headerInfoFilePath == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Open the input and output files
	//##############################################################
	*/
	p_inputFile = fopen (pstr_inputFrameFilePath, "rb");
	if (p_inputFile == NULL)
	{
		return MWRes_CommunicationsError;
	}

	p_outputFile = fopen (pstr_outputProductFilePath, "wb");
	if (p_outputFile == NULL)
	{
		fclose (p_inputFile);
		return MWRes_CommunicationsError;
	}

	p_headerInfoFile = NULL;
	if (bool_outputHeaderInfoToFile)
	{
		p_headerInfoFile = fopen (pstr_headerInfoFilePath, "wb");
		if (p_headerInfoFile == NULL)
		{
			fclose (p_inputFile);
			fclose (p_outputFile);
			return MWRes_CommunicationsError;
		}
	}

	numOutputFileBufferBytesUsed = 0;
	numInputFileBufferBytes = 0;
	numInputFileBufferBytesUsed = 0;

	/*
	//##############################################################
	// Read in bytes and process through output until all input
	// consumed.  Then read more input.  Finish on error or end
	// of input file.
	//##############################################################
	*/

	while (1)
	{
		/*
		//##############################################################
		// Do we need to read more input?
		//##############################################################
		*/
		if (numInputFileBufferBytesUsed >= numInputFileBufferBytes)
		{
			/* If at end of input, then error */
			if (feof(p_inputFile))
			{
				fclose (p_inputFile);
				fclose (p_outputFile);
				if (p_headerInfoFile != NULL)
				{
					fclose (p_headerInfoFile);
				}
				return MWRes_TransportProtocolError;
			}

			/* Read input*/
			numInputFileBufferBytes = fread (inputFileBuffer, 1, NUM_FILE_BUFFER_BYTES, p_inputFile);
			if (numInputFileBufferBytes < NUM_FILE_BUFFER_BYTES)
			{
				if (ferror(p_inputFile))
				{
					// File error
					fclose (p_inputFile);
					fclose (p_outputFile);
					if (p_headerInfoFile != NULL)
					{
						fclose (p_headerInfoFile);
					}
					return MWRes_CommunicationsError;
				}
			}

			/* No input used yet */
			numInputFileBufferBytesUsed = 0;
		}

		/*
		//##############################################################
		// Process input into output
		//##############################################################
		*/
		numOutputFileBufferBytesUsed = 0;

		MWRes = MWTSiriusFrameDecoder_extractProduct
		(
			numInputFileBufferBytes,
			inputFileBuffer,
			&numInputFileBufferBytesUsed,
			NUM_FILE_BUFFER_BYTES,
			outputFileBuffer,
			&numOutputFileBufferBytesUsed,
			p_headerInfoFile
		);
		if
		(
			(MWRes != MWRes_Success) &&
			(MWRes != MWRes_InputBufferExhausted) &&
			(MWRes != MWRes_DataStructureFull)
		)
		{
			/* Its an error in the frame format */
			fclose (p_inputFile);
			fclose (p_outputFile);
			if (p_headerInfoFile != NULL)
			{
				fclose (p_headerInfoFile);
			}
			return MWRes;
		}

		/*
		//##############################################################
		// Write output
		//##############################################################
		*/
		if (numOutputFileBufferBytesUsed > 0)
		{
			if (fwrite (outputFileBuffer, 1, numOutputFileBufferBytesUsed, p_outputFile) != numOutputFileBufferBytesUsed)
			{
				/* Error writing out */
				fclose (p_inputFile);
				fclose (p_outputFile);
				if (p_headerInfoFile != NULL)
				{
					fclose (p_headerInfoFile);
				}
				return MWRes_CommunicationsError;
			}
		}

		/* Output buffer is now empty */
		numOutputFileBufferBytesUsed = 0;

		/*
		//##############################################################
		// If result of extraction operation was MWRes_Success, then
		// we are done.
		//##############################################################
		*/
		if (MWRes == MWRes_Success)
		{
			fclose(p_inputFile);
			fclose(p_outputFile);
			if (p_headerInfoFile != NULL)
			{
				fclose (p_headerInfoFile);
			}
			return MWRes_Success;
		}

	} // Loop around for more IO

}


/*//--------------------------------------------------------------------*/

TMW_ResultCode MWTSiriusFrameDecoder_undoTransparencyEncoding
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_inout_numBytesUsed,
	TFourByteInteger_Unsigned numOutputBufferBytes,
	TByte_Unsigned* p_outputBuffer,
	TFourByteInteger_Unsigned* p_inout_numOutputBufferBytesUsed
)
{

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(p_bytes == NULL) ||
		(p_inout_numBytesUsed == NULL) ||
		(p_outputBuffer == NULL) ||
		(p_inout_numOutputBufferBytesUsed == NULL)
	)
	{
		return MWRes_NullPointer;
	}


	/*
	//##############################################################
	// Pass the bytes through the frame processing routine
	// We exit internally from the loop on error or on
	// exhaustion of output buffer space.
	//##############################################################
	*/

	while
	(
		(*p_inout_numBytesUsed < numBytes) &&
		(*p_inout_numOutputBufferBytesUsed < numOutputBufferBytes)
	)
	{
		if (p_bytes[*p_inout_numBytesUsed] == FRAME_BORDER_FLAG_BYTE)
		{
			/* Always copy out frame borders as is, regardless of transparency state */
			p_outputBuffer[*p_inout_numOutputBufferBytesUsed] =
				p_bytes[*p_inout_numBytesUsed];


			/* More output used */
			*p_inout_numOutputBufferBytesUsed =
				(*p_inout_numOutputBufferBytesUsed) + 1;

		}
		else if (p_bytes[*p_inout_numBytesUsed] == FRAME_TRANSPARENCY_FLAG_BYTE)
		{
			/* Should not get 2 transparency flags in a row */
			if (MWTSiriusFrameDecoder_bool_transparencyFlagOn != 0)
			{
				MWTSiriusFrameDecoder_bool_transparencyFlagOn = 0;

				/* error*/
				*p_inout_numBytesUsed = (*p_inout_numBytesUsed) + 1;
				return MWRes_TransportProtocolError;
			}
			else
			{
				MWTSiriusFrameDecoder_bool_transparencyFlagOn = 1;
			}

		}
		else
		{
			if (MWTSiriusFrameDecoder_bool_transparencyFlagOn != 0)
			{
				p_outputBuffer[*p_inout_numOutputBufferBytesUsed] =
					p_bytes[*p_inout_numBytesUsed] ^ 0x20;

				/* Transparency code done */
				MWTSiriusFrameDecoder_bool_transparencyFlagOn = 0;
			}
			else
			{
				p_outputBuffer[*p_inout_numOutputBufferBytesUsed] =
					p_bytes[*p_inout_numBytesUsed];
			}

			/* More output used */
			*p_inout_numOutputBufferBytesUsed =
				(*p_inout_numOutputBufferBytesUsed) + 1;


		}

		/* More input used */
		*p_inout_numBytesUsed = (*p_inout_numBytesUsed) + 1;

	} // Get next byte

	/*
	//##############################################################
	// Did we run out of input or output?
	//##############################################################
	*/
	if (*p_inout_numBytesUsed < numBytes)
	{
		/* out of output */
		return MWRes_DataStructureFull;
	}
	else
	{
		return MWRes_Success;
	}

}

/*//--------------------------------------------------------------------*/

TMW_ResultCode MWTSiriusFrameDecoder_removeTransparencyFromFrameFile
(
	const TASCIIChar* pstr_inputFrameFilePath,
	const TASCIIChar* pstr_outputProductFilePath
)
{
#define NUM_FILE_BUFFER_BYTES 4096

	TMW_ResultCode MWRes;

	FILE* p_inputFile;
	FILE* p_outputFile;

	TByte_Unsigned inputFileBuffer[NUM_FILE_BUFFER_BYTES];
	TFourByteInteger_Unsigned numInputFileBufferBytes;
	TFourByteInteger_Unsigned numInputFileBufferBytesUsed;

	TByte_Unsigned outputFileBuffer[NUM_FILE_BUFFER_BYTES];
	TFourByteInteger_Unsigned numOutputFileBufferBytesUsed;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(pstr_inputFrameFilePath == NULL) ||
		(pstr_outputProductFilePath == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Open the input and output files
	//##############################################################
	*/
	p_inputFile = fopen (pstr_inputFrameFilePath, "rb");
	if (p_inputFile == NULL)
	{
		return MWRes_CommunicationsError;
	}

	p_outputFile = fopen (pstr_outputProductFilePath, "wb");
	if (p_outputFile == NULL)
	{
		fclose (p_inputFile);
		return MWRes_CommunicationsError;
	}

	numOutputFileBufferBytesUsed = 0;
	numInputFileBufferBytes = 0;
	numInputFileBufferBytesUsed = 0;

	/*
	//##############################################################
	// Read in bytes and process through output until all input
	// consumed.  Then read more input.  Finish on error or end
	// of input file.
	//##############################################################
	*/

	while (1)
	{
		/*
		//##############################################################
		// Do we need to read more input?
		//##############################################################
		*/
		if (numInputFileBufferBytesUsed >= numInputFileBufferBytes)
		{
			/* If at end of input, then done */
			if (feof(p_inputFile))
			{
				fclose (p_inputFile);
				fclose (p_outputFile);
				return MWRes_Success;
			}

			/* Read input*/
			numInputFileBufferBytes = fread (inputFileBuffer, 1, NUM_FILE_BUFFER_BYTES, p_inputFile);
			if (numInputFileBufferBytes < NUM_FILE_BUFFER_BYTES)
			{
				if (ferror(p_inputFile))
				{
					// File error
					fclose (p_inputFile);
					fclose (p_outputFile);
					return MWRes_CommunicationsError;
				}
			}

			/* No input used yet */
			numInputFileBufferBytesUsed = 0;
		}

		/*
		//##############################################################
		// Process input into output
		//##############################################################
		*/
		numOutputFileBufferBytesUsed = 0;

		MWRes = MWTSiriusFrameDecoder_undoTransparencyEncoding
		(
			numInputFileBufferBytes,
			inputFileBuffer,
			&numInputFileBufferBytesUsed,
			NUM_FILE_BUFFER_BYTES,
			outputFileBuffer,
			&numOutputFileBufferBytesUsed
		);
		if
		(
			(MWRes != MWRes_Success) &&
			(MWRes != MWRes_DataStructureFull)
		)
		{
			/* Its an error */
			fclose (p_inputFile);
			fclose (p_outputFile);
			return MWRes;
		}

		/*
		//##############################################################
		// Write output
		//##############################################################
		*/
		if (numOutputFileBufferBytesUsed > 0)
		{
			if (fwrite (outputFileBuffer, 1, numOutputFileBufferBytesUsed, p_outputFile) != numOutputFileBufferBytesUsed)
			{
				// Error writing out
				fclose (p_inputFile);
				fclose (p_outputFile);
				return MWRes_CommunicationsError;
			}
		}

		/* Output buffer is now empty */
		numOutputFileBufferBytesUsed = 0;


	} // Loop around for more IO

}
