/*
//====================================================================
// MWT_SiriusDPUAssembler.h:
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//
// This is the public interface for an ADT which handles the assembly
// of Data Payload Units (DPUs) into a product instance.
//
// It monitors the sequence of arriving DPUs and handles their
// payload concatenation.  If DPUs arrive out of order, the
// assembler resets its state.
//
// If all the DPUs for a product instance arrive in order and without
// anything interleaved, this notes the completion of the product
// for the caller.
//
// The product assembly output goes into a buffer supplied by the
// caller.
//
//====================================================================
*/
#ifndef MWT_SIRIUISDPUASSEMBLER_HEADER
#define MWT_SIRIUISDPUASSEMBLER_HEADER

/*
//====================================================================
// Required includes
//====================================================================
*/
#include "MW_DataFieldTypes.h"
#include <stdio.h>

/*
//====================================================================
// DPU header structure definition
//====================================================================
*/

#pragma pack (push,1)
typedef struct
{
	TByte_Unsigned productID_LOWByte;
	TByte_Unsigned productID_HIGHByte;

	TByte_Unsigned instanceID_LOWByte;
	TByte_Unsigned instanceID_HIGHByte;

	TByte_Unsigned DPUSequenceNumber_LOWByte;
	TByte_Unsigned DPUSequenceNumber_HIGHByte;

	TByte_Unsigned finalDPUThisProductInstance; // 0 is not, 1 is true

} TWSI_Output_DPUHeader;

#pragma pack(pop)

/*
//====================================================================
// Public methods
//====================================================================
*/

/*
//====================================================================
// MWTSiriusDPUAssembler_reset:
//
// This method resets the assembler so that it no longer considers
// itslef mid-assembly of a product.
//
//====================================================================
*/
void MWTSiriusDPUAssembler_reset();

/*
//====================================================================
// MWTSiriusDPUAssembler_processDPU:
//
// Given a pointer to a DPU Header, a DPU payload length, and the DPU
// payload, processes the DPU through the product assembly
// process into the caller's output buffer.
//
// The following processing occurs:
//
//		If it is the next DPU in the currently assembling product,
//		it performs concatenation of the product payload into
//		the caller's product buffer. If it is the last DPU
//		of the product instance, signals this to the
//		caller via the return value (see Returns section).
//
//		If not the next DPU in the currently assembling product,
//		but it is the DPU of a product instance, then
//		this product becomes the new assembling instance and the
//		DPU payload is placed at the front of the caller's
//		assembly buffer.
//
//		If it is not the first DPU of a product and is not the next
//		expected DPU for the assembling product instance then
//		the DPU is ignored and the assembling product instance
//		is reset to none.
//
// RETURNS:
//
//	MWRes_Success:
//		If the DPU was processed but no product was completed.
//
//	MWRes_EndOfIteration:
//		The DPU was processed correctly and a product was completed.
//		The caller's product assembly buffer will contain the
//		finished product.
//
//	MWRes_DataStructureFull:
//		The caller's product assembly buffer was not large enough to
//		hold the product.
//
//====================================================================
*/
TMW_ResultCode MWTSiriusDPUAssembler_processDPU
(
	TWSI_Output_DPUHeader* p_DPUHeader,
	TFourByteInteger_Unsigned numPayloadBytes,
	TByte_Unsigned* p_payloadBytes,
	TFourByteInteger_Unsigned numProductStreamBufferBytes,
	TByte_Unsigned* p_productStreamBuffer,
	TFourByteInteger_Unsigned* p_inout_numProductStreamBufferBytesUsed,
	FILE* p_headerInfoFile
);


/* Done header wrapper */
#endif
