/*
//====================================================================
// MW_WxHuffmanRunDecoder.h:
//
// This provides an Abstract data type for reading in
// run information from a weather huffman source.
//
// It handles the weather huffman tables (default or custom)
// and run state tracking.
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//====================================================================
*/

#ifndef MW_WX_HUFFMAN_TABLE_HEADER
#define MW_WX_HUFFMAN_TABLE_HEADER

/*
====================================================================
// Required Includes
====================================================================
*/
#include "MW_DataFieldTypes.h"

/*
====================================================================
// MWWxHuffmanRunDecoder_reset:
//
// This resets the entire decoder, clearing the states of the tables
// and run decoding information.
//
// RETURNS:
//	MWRes_Success
//
====================================================================
*/
TMW_ResultCode MWWxHuffmanRunDecoder_reset();

/*
====================================================================
// MWWxHuffmanRunDecoder_readTables:
//
// This method reads in the WxHuffmanTables defined in the
// product stream.
//
// PARAMETERS:
//
//	numWeatherHuffmanEncodingBytes:
//		The number of bytes in this WSI Weather Huffman encoding
//
//	p_WxHuffmanBody:
//		Pointer to the first byte of the contigous array of WSI Weather
//		Huffman encoding bytes (must be at least numWeatherHuffmanEncodingBytes
//		bytes long)
//
//	numTablesExpected:
//		The number of tables declared in the TMW_WHBlockHeader of
//		the raster plane
//
//	maximumPossiblePixelValue:
//		The maximum possible pixel value in the raster
//
//	inout_p_totalBitsConsumed:
//		Pointer to the counter of the number of bits from p_WxHuffmanBody
//		already processed. We update this as we consume bits.
//
// RETURNS:
//
//	MWRes_Success
//		If all the tables were read in.
//
//	MWRes_DataStructureFull
//		If there are too many WxHuffman tables in the product stream.
//
//	MWRes_NullPointer:
//		If a pointer parameter is NULL
//
//	MWRes_ProductContentsInvalid:
//		If something is wrong with the product contents
//
====================================================================
*/
TMW_ResultCode MWWxHuffmanRunDecoder_readTables
(
	TFourByteInteger_Unsigned numWeatherHuffmanEncodingBytes,
	TByte_Unsigned* p_WxHuffmanBody,
	TByte_Unsigned numTablesExpected,
	TByte_Unsigned maximumPossiblePixelValue,
	TFourByteInteger_Unsigned* inout_p_totalBitsConsumed
);

/*
====================================================================
// MWWxHuffmanRunDecoder_getPixels:
//
// This method is used to retreive pixels from the raster run codes.
// The caller can specify the maximum number of pixels they wish
// to receive.  This method will return <= numRequestedPixels,
// depending on the run structure of the image, as we return
// runs of pixels.
//
// The caller may call this again and again until all the desired
// pixels have been retrieved.
//
====================================================================
*/
TMW_ResultCode MWWxHuffmanRunDecoder_getPixels
(
	TFourByteInteger_Unsigned numWeatherHuffmanEncodingBits,
	TByte_Unsigned* p_WxHuffmanBody,
	TFourByteInteger_Unsigned* inout_p_totalBitsConsumed,
	TByte_Unsigned maxPixelValue,
	TFourByteInteger_Unsigned numRequestedPixels,
	TFourByteInteger_Unsigned* p_out_runLength,
	TByte_Unsigned* p_out_pixelValue
);

/*
====================================================================
// MWWxHuffmanRunDecoder_prepareHilbertWalk:
//
// Prepares for a hilbert walk iteration given the size of the raster
// to be walked.  If the raster is not a dyadic square (square where
// both dimensions are the same power of 2) then not all points in
// the walk will fall on the raster.  Points which do not fall
// on the raster are not considered part of any runs and should
// be skipped over.
//
// Use this to start a hilbert walk iteration.
//
// PARAMETERS:
//
//	rasterWidth
//		The width of the raster in pixels
//
//	rasterHeight:
//		The height of the raster in pixels
//
// RETURNS:
//
//	MWRes_Success
//
====================================================================
*/
TMW_ResultCode MWWxHuffmanRunDecoder_prepareHilbertWalk
(
	TTwoByteInteger_Unsigned rasterWidth,
	TTwoByteInteger_Unsigned rasterHeight
);

/*
====================================================================
// MWWxHuffmanRunDecoder_getNextHilbertWalkPoint:
//
// Gets the next point in the hilbert walk given the current point.
//
// The first point of the walk is always at the origin (0,0) and
// this method is NOT called to retreive it.  Call it to retrieve
// the second and later points.
//
// Coordinates have an origin at the upper-left pixel of the raster,
// with positive X coordinates moving right and positive Y
// coordinates moving down.
//
//
// PARAMETERS:
//
//	inout_p_coordX:
//		Pass in a pointer to the current X coordinate.
//		This method modifies the X coordinate to the next one
//		in the walk order.
//		0 is left edge of the raster.
//
//	inout_p_coordY:
//		Pass in a pointer to the current Y coordinate.
//		This method modifies the Y coordinate to the next one
//		in the walk order.
//		0 is top edge of the raster.
//
// RETURNS:
//
//	MWRes_Success:
//		If a point was retrieved
//
//	MWRes_EndOfIteration:
//		If there are no more points, so no point was gotten
//		(end of the walk)
//
====================================================================
*/
TMW_ResultCode MWWxHuffmanRunDecoder_getNextHilbertWalkPoint
(
	TTwoByteInteger_Unsigned* inout_p_coordX,
	TTwoByteInteger_Unsigned* inout_p_coordY
);



/* End header wrapper */
#endif
