/*
//====================================================================
// MW_WxHuffmanRunDecoder.c:
//
// Implementation of the WeatherHuffman run decoder ADT.
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/
#include "MW_WxHuffmanRunDecoder.h"
#include "MW_BitFieldOperators.h"

/*
//====================================================================
//====================================================================
// Defines
//====================================================================
//====================================================================
*/


/* 63 tables leads to about 9KB of memory */
#define MAX_WX_HUFFMAN_TABLES 64

/*
// Max entries needed in a weather huffman table
// 0-63 is 64, plus S1 and S2
*/
#define MAX_WX_HUFFMAN_TABLE_ENTRIES 66

/* The maximum ever pixel value is 1 byte long */
#define MAX_EVER_PIXEL_VALUE 255

/* Indexes of S1 and S2 code words */
#define S1_TABLE_INDEX 0
#define S2_TABLE_INDEX 1
#define NUM_TABLE_INDICES_BEFORE_0 2

/* Huffman codes can't be longer than 7 bits in Wx Huffman */
#define MAX_ALLOWED_CODE_LENGTH 7

/*
//====================================================================
// Bitfields which define tables
//====================================================================
*/

/* These are all in bits */
#define TABLE_ID_BITFIELD_LENGTH 2
#define TABLE_LONGESTCODELENGTH_BITFIELD_LENGTH 3

/*
//====================================================================
// Bitfields used in run codes
//====================================================================
*/

#define S2_ITERATION_FIELD_LENGTH_BITS 8

/*
//====================================================================
//====================================================================
// "Other run length" options
//====================================================================
//====================================================================
*/

typedef struct
{
	/*
	// Number of bits required for a short code
	// 0 if only one code length (long) is used
	// Note, if numBitsShort is > 0 then the actual
	// code bits are preceeded by 1 bit
	// that state if it is a short (0) or long (1) code
	*/
	TByte_Unsigned numBitsShort;

	/* The number of bits required for a long code */
	TByte_Unsigned numBitsLong;


} TMW_WxHuffman_OtherRunLengthOptions;

/*--------------------------------------------------------------------*/

#define NUM_OTHER_RUN_LENGTH_OPTIONS 8

/*--------------------------------------------------------------------*/

TMW_WxHuffman_OtherRunLengthOptions OtherRunLengthOptionDefinitions[NUM_OTHER_RUN_LENGTH_OPTIONS] =
{
	{0, 6},
	{3, 6},
	{2, 6},
	{0, 5},
	{2, 5},
	{0, 4},
	{0, 3},
	{0, 2}
};


/*
//====================================================================
//====================================================================
// Table structure
//====================================================================
//====================================================================
*/

/* The tables */
typedef struct
{
	TByte_Unsigned tableID;
	TByte_Unsigned otherRunLengthOption;

	TByte_Unsigned numEntries;
	TByte_Unsigned codeLengths[MAX_WX_HUFFMAN_TABLE_ENTRIES];
	TByte_Unsigned codes[MAX_WX_HUFFMAN_TABLE_ENTRIES];

} TMW_WxHuffmanTable;


/*
//====================================================================
//====================================================================
// Default wx huffman tables
//====================================================================
//====================================================================
*/

/* There are 3 default tables to choose from for each pixel level */
#define NUM_DEFAULT_TABLES_FOR_A_PIXEL_LEVEL 3


typedef struct
{
	/* Includes S1 and S2 */
	TTwoByteInteger_Unsigned numCodeWordsInTable;

	/* First is S1 then S2, then run-lengths in increasing order from 0 */
	TByte_Unsigned lengthsInBits[11];
	TByte_Unsigned codes[11];

} TMW_DefaultRunCodeTable;


/*--------------------------------------------------------------------*/

/* Default tables for pixel level 0*/

TMW_DefaultRunCodeTable DefaultTables_Level0[NUM_DEFAULT_TABLES_FOR_A_PIXEL_LEVEL] =
{
	/* Option 0 */
	{
		/* count including S1 and S2 */
		10,

		/* code lengths */
		{ 4, 1, 0, 4, 4, 4, 4, 4, 4, 4, 0},

		/* codes */
		{ 0x08, 0x00, 0x00, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00}
	},

	/* Option 1 */
	{
		/* count including S1 and S2 */
		2,

		/* Code Lengths */
			{1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0},

		/* Codes */
			{0x00, 0x01, 0, 0, 0, 0, 0, 0, 0, 0, 0}

	},

	/* Option 2 */
	{
		11, /* count = 9 + 2 */

		/* Code Lengths */
			{ 2, 2, 0, 4, 4, 4, 4, 4, 4, 4, 4},

		/* Run codes */
			{ 0x01, 0x00, 0, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F}

	}



};

/*--------------------------------------------------------------------*/

/* Default tables for pixel level 1 and 2 */
TMW_DefaultRunCodeTable DefaultTables_Levels1And2[NUM_DEFAULT_TABLES_FOR_A_PIXEL_LEVEL] =
{
	/* Option 0 */
	{
		/* Count is 6 + 2 */
		8,

		/* Lengths */
			{ 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0},

		/* Run codes */
			{0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x00, 0x00}

	},

	/* Option 1 */
	{
		2, /* Count = 0 + 2 */

		/* Code Lengths */
			{1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0},

		/* Codes */
			{0x00, 0x01, 0, 0, 0, 0, 0, 0, 0, 0, 0}
	},

	/* Option 2 */
	{
		10, /* Count = 8 + 2 */

		/* Code Lengths */
			{ 2, 2, 4, 4, 4, 4, 4, 4, 4, 4, 0},

		/* Run codes */
			{ 0x00, 0x01, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00}

	}
};

/*--------------------------------------------------------------------*/

/* Default tables for pixel level 3, 4, 5 and 6 */

TMW_DefaultRunCodeTable DefaultTables_Levels3To6[NUM_DEFAULT_TABLES_FOR_A_PIXEL_LEVEL] =
{
	/* Option 0 */
	{
		/* Count is 3 + 2 */
		5,


		/* Code Lengths */
			{ 3, 3, 2, 2, 2, 0, 0, 0, 0, 0, 0},

		/* Codes */
			{0x06, 0x07, 0x00, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}

	},

	/* Option 1 */
	{
		5, /* Count = 3 + 2 */

		/* Code Lengths */
			{ 1, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0},

		/* Codes */
			{0x00, 0x07, 0x04, 0x05, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}

	},

	/* Option 2 */
	{
		7, /* Lmax = 5 + 2 */

		/* Code Lengths */
			{ 2, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0},

		/* Codes */
			{0x00, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x00, 0x00, 0x00}

	}

};

/*--------------------------------------------------------------------*/

/* Default tables for pixel level 7 (and higher for us maybe) */

TMW_DefaultRunCodeTable DefaultTables_Level7[NUM_DEFAULT_TABLES_FOR_A_PIXEL_LEVEL] =
{
	/* Option 0 */
	{
		/* Count is 4 + 2 */
		6,

		/* Code Lengths */
			{ 3, 3, 0, 2, 2, 2, 0, 0, 0, 0, 0},

		/* Codes */
			{0x06, 0x07, 0x00, 0x00, 0x01, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00}

	},

	/* Option 1 */
	{
		6, /* Count is 4 + 2 */

		/* Code Lengths */
			{ 1, 3, 0, 3, 3, 3, 0, 0, 0, 0, 0},

		/* Codes */
			{0x00, 0x07, 0x00, 0x04, 0x05, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00}

	},

	/* Option 2 */
	{
		8, /* Count is 6 + 2 */

		/* Code Lengths */
			{ 2, 3, 0, 3, 3, 3, 3, 3, 0, 0, 0},

		/* Codes */
			{0x00, 0x02, 0x00, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x00, 0x00}

	}

};

/*
//====================================================================
//====================================================================
// Custom tables sent with product
// Note, these are also used for referring to default tables when
// those are specified as in use by the product instance
//====================================================================
//====================================================================
*/

/* The number of tables in this product */
static TByte_Unsigned numWxHuffmanTables;

/* The number of pixel values represented per WxHuffmanTable */
static TByte_Unsigned numPixelValuesPerHuffmanTable;

/* The tables themselves */
static TMW_WxHuffmanTable WxHuffmanTables[MAX_WX_HUFFMAN_TABLES];

/*
//====================================================================
//====================================================================
// Run decoding state
//====================================================================
//====================================================================
*/

/* Run length information */
static TFourByteInteger_Unsigned MWWxHuffmanRunDecoder_currentRunLength;
static TByte_Unsigned MWWxHuffmanRunDecoder_currentRunPixel;

/* Is it the first run in the raster, that we are going to be reading? */
static TByte_Unsigned MWWxHuffmanRunDecoder_bool_firstRunInRasterNotYetRead;


/*
//====================================================================
//====================================================================
// Interior methods
//====================================================================
//====================================================================
*/

/*
//====================================================================
// MWWxHuffmanRunDecoder_Internal_readS1CodeValue:
//
// This method reads whatever data is required after an S1 code
// value in order to determine the run length.
//
// PARAMETERS:
//
//	numWeatherHuffmanEncodingBits:
//		The number of bits in the encoding data
//
//	p_WxHuffmanBody:
//		The start of the encoding bits
//
//	inout_p_totalBitsConsumed
//		Pass in a pointer to the total number of bits consumed so far.
//		The value pointed to is updated for any bits consumed by this
//		method.
//
//	otherRunLengthOptionFlag:
//		Pass in the value of the other run length option in effect
//		at the current pixel level.  It determines how many bits
//		follow an S1 flag and what they mean.
//
//	p_out_runLength:
//		Pass in a pointer to the variable that will receive the
//		run-length indicated by this S1 code
//
//====================================================================
*/

TMW_ResultCode MWWxHuffmanRunDecoder_Internal_readS1CodeValue
(
	TFourByteInteger_Unsigned numWeatherHuffmanEncodingBits,
	TByte_Unsigned* p_WxHuffmanBody,
	TFourByteInteger_Unsigned* inout_p_totalBitsConsumed,
	TByte_Unsigned otherRunLengthOptionFlag,
	TFourByteInteger_Unsigned* p_out_runLength
)
{
	TMW_ResultCode MWRes;
	TByte_Unsigned bool_useLongCode;
	TByte_Unsigned numCodeBits;
	TByte_Unsigned codeBits;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/

	/* Flag is 0 based */
	if
	(
		(otherRunLengthOptionFlag >= NUM_OTHER_RUN_LENGTH_OPTIONS) ||
		(otherRunLengthOptionFlag < 0)
	)
	{
		return MWRes_BadParamValue;
	}

	/*
	//##############################################################
	// Which code to use, long or short
	//##############################################################
	*/
	bool_useLongCode = 0;

	if (OtherRunLengthOptionDefinitions[otherRunLengthOptionFlag].numBitsShort > 0)
	{
		MWRes = readBitField
		(
			&bool_useLongCode,
			1,
			numWeatherHuffmanEncodingBits,
			p_WxHuffmanBody,
			inout_p_totalBitsConsumed
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

	}
	else
	{
		bool_useLongCode = 1;
	}

	/* How many code bits is that? */
	if (bool_useLongCode != 0)
	{
		numCodeBits = OtherRunLengthOptionDefinitions[otherRunLengthOptionFlag].numBitsLong;
	}
	else
	{
		numCodeBits = OtherRunLengthOptionDefinitions[otherRunLengthOptionFlag].numBitsShort;
	}

	/*
	//##############################################################
	// Read the code
	//##############################################################
	*/
	MWRes = readBitField
	(
		&codeBits,
		numCodeBits,
		numWeatherHuffmanEncodingBits,
		p_WxHuffmanBody,
		inout_p_totalBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Code is the run length
	//##############################################################
	*/
	*p_out_runLength = codeBits;

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;



}

/*
//====================================================================
// MWWxHuffmanRunDecoder_Internal_readRunLength:
//
//	This inner run reading function handles the run length only.
//  The current pixel value is passed in to this method.
//
//	Any S1 or S2 code decoding happens within this method.
//  Run lengths >= 0 are returned through the out parameter.
//
// PARAMETERS:
//
//	numWeatherHuffmanEncodingBits:
//		The number of bits in the encoding data
//
//	p_WxHuffmanBody:
//		The start of the encoding bits
//
//	inout_p_totalBitsConsumed
//		Pass in a pointer to the total number of bits consumed so far.
//		The value pointed to is updated for any bits consumed by this
//		method.
//
//	pixelValue:
//		Pass in the current pixel value. This determines which WxHuffman
//		table is used.
//
//	p_out_runLength:
//		Pass in a pointer to the variable that will receive the run-length
//
// RETURNS:
//====================================================================
*/

TMW_ResultCode MWWxHuffmanRunDecoder_Internal_readRunLength
(
	TFourByteInteger_Unsigned numWeatherHuffmanEncodingBits,
	TByte_Unsigned* p_WxHuffmanBody,
	TFourByteInteger_Unsigned* inout_p_totalBitsConsumed,
	TByte_Unsigned pixelValue,
	TFourByteInteger_Unsigned* p_out_runLength
)
{
	TMW_ResultCode MWRes;

	TFourByteInteger_Unsigned runLengthThisCode;
	TTwoByteInteger_Unsigned bitsReadInCode;

	TByte_Unsigned bitCode;
	TByte_Unsigned tempBit;
	TMW_WxHuffmanTable* p_tableToUse;
	TMW_DefaultRunCodeTable* p_defaultTableToUse;

	/* Fields from target table used to search for the codes */
	TTwoByteInteger_Unsigned numCodes;
	TByte_Unsigned* p_codeLengths;
	TByte_Unsigned* p_codes;

	/* Variable to store the modified S2 code follow-on byte */
	TByte_Unsigned num64s;


	/* The code search index */
	TTwoByteInteger_Unsigned codeSearchIndex;

	/*
	//##############################################################
	// Prepare
	//##############################################################
	*/

	/* No run read so far */
	*p_out_runLength = 0;
	runLengthThisCode = 0;

	/* No code read so far */
	bitsReadInCode = 0;
	bitCode = 0;

	/*
	//##############################################################
	// Determine table to use
	//##############################################################
	*/
	if (pixelValue >= numWxHuffmanTables)
	{
		return MWRes_BadParamValue;
	}

	p_tableToUse = &(WxHuffmanTables[pixelValue]);

	/* If its a default table, get pointer to default table to use */
	if (p_tableToUse->tableID < 3)
	{

		if (pixelValue == 0)
		{
			p_defaultTableToUse = &(DefaultTables_Level0[p_tableToUse->tableID]);
		}
		else if ((pixelValue == 1) || (pixelValue == 2))
		{
			p_defaultTableToUse = &(DefaultTables_Levels1And2[p_tableToUse->tableID]);
		}
		else if ((pixelValue >= 3) && (pixelValue <= 6))
		{
			p_defaultTableToUse = &(DefaultTables_Levels3To6[p_tableToUse->tableID]);
		}
		else
		{
			p_defaultTableToUse = &(DefaultTables_Level7[p_tableToUse->tableID]);
		}

		/*
		//##############################################################
		// Get fields from default table
		//##############################################################
		*/
		numCodes = p_defaultTableToUse->numCodeWordsInTable;
		p_codeLengths = p_defaultTableToUse->lengthsInBits;
		p_codes = p_defaultTableToUse->codes;

	}
	else
	{
		/*
		//##############################################################
		// Get fields from custom table
		//##############################################################
		*/

		numCodes = p_tableToUse->numEntries;
		p_codeLengths = p_tableToUse->codeLengths;
		p_codes = p_tableToUse->codes;
	}

	/*
	//##############################################################
	// Begin decoding
	//##############################################################
	*/

	bitsReadInCode = 0;
	bitCode = 0;

	while (bitsReadInCode < MAX_ALLOWED_CODE_LENGTH)
	{
		/* Read new bit */
		tempBit = 0;
		MWRes = readBitField
		(
			&tempBit,
			1,
			numWeatherHuffmanEncodingBits,
			p_WxHuffmanBody,
			inout_p_totalBitsConsumed
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/* Add bit to bit code */
		bitCode <<= 1;
		bitCode |= tempBit;
		bitsReadInCode ++;

		/* Find this code in the table */
		for (codeSearchIndex = 0; codeSearchIndex < numCodes; codeSearchIndex ++)
		{
			if
			(
				(p_codeLengths[codeSearchIndex] == bitsReadInCode) &&
				(p_codes[codeSearchIndex] == bitCode)
			)
			{
				/* Found a match */
				if (codeSearchIndex == S1_TABLE_INDEX)
				{
					MWRes = MWWxHuffmanRunDecoder_Internal_readS1CodeValue
					(
						numWeatherHuffmanEncodingBits,
						p_WxHuffmanBody,
						inout_p_totalBitsConsumed,
						p_tableToUse->otherRunLengthOption,
						&runLengthThisCode
					);
					if (MWRes != MWRes_Success)
					{
						return MWRes;
					}
					else
					{
						/* Add to total */
						*p_out_runLength = (*p_out_runLength) + runLengthThisCode;

						/* Success */
						return MWRes_Success;
					}

				}
				else if (codeSearchIndex == S2_TABLE_INDEX)
				{
					/*
					// Read in S2 trailer that tells us how many
					// 64s in a row there are. Then start a new code.
					*/

					num64s = 0;
					MWRes = readBitField
					(
						&num64s,
						S2_ITERATION_FIELD_LENGTH_BITS,
						numWeatherHuffmanEncodingBits,
						p_WxHuffmanBody,
						inout_p_totalBitsConsumed
					);
					if (MWRes != MWRes_Success)
					{
						return MWRes;
					}

					*p_out_runLength = (*p_out_runLength) + ( ( ( (TTwoByteInteger_Unsigned) num64s) + 1) * 64);

					/* Start a new code */
					bitsReadInCode = 0;
					bitCode = 0;

					/* Done code search loop */
					break;
				}
				else
				{
					/* Its a value */
					*p_out_runLength = (*p_out_runLength) + (codeSearchIndex - NUM_TABLE_INDICES_BEFORE_0);

					/* Success */
					return MWRes_Success;
				}
			}
		}



	} /* No code match yet, read another code bit */

	/*
	//##############################################################
	// Error, no matching code
	//##############################################################
	*/

	return MWRes_ProductContentsInvalid;

}

/*
//====================================================================
// MWWxHuffmanRunDecoder_Internal_readRun:
//
// This outer run reading function handles the transitions in pixel
// levels between runs. It calls a subfunction to read the next
// run length for the current pixel level.
//
// PARAMETERS:
//
//	numWeatherHuffmanEncodingBits:
//		The number of bits in the encoding data
//
//	p_WxHuffmanBody:
//		The start of the encoding bits
//
//	inout_p_totalBitsConsumed
//		Pass in a pointer to the total number of bits consumed so far.
//		The value pointed to is updated for any bits consumed by this
//		method.
//
//	maxPixelValue:
//		The maximum possible pixel value for this raster
//
//	bool_IsFirstRunInRaster:
//		Set this to 1 if it is the first run in the raster.
//		Otherwise, set this to 0.
//
//====================================================================
*/

TMW_ResultCode MWWxHuffmanRunDecoder_Internal_readRun
(
	TFourByteInteger_Unsigned numWeatherHuffmanEncodingBits,
	TByte_Unsigned* p_WxHuffmanBody,
	TFourByteInteger_Unsigned* inout_p_totalBitsConsumed,
	TByte_Unsigned maxPixelValue,
	TByte_Unsigned bool_IsFirstRunInRaster
)
{
	TMW_ResultCode MWRes;

	/* Used for reading in level change bit fields */
	TByte_Unsigned tempByte;

	/*
	// We make this 1 if the pixel level increased,
	//	-1 if it decreased.
	// We also make it 1 if it is the first run of the raster.
	// (It is always either -1 or 1)
	*/

	TByte_Signed pixelLevelDelta;

	/*
	//##############################################################
	// Based on what the current pixel level is, do we go up or down
	//##############################################################
	*/

	/* No pixel delta yet */
	pixelLevelDelta = 0;

	if (bool_IsFirstRunInRaster == 0)
	{
		if (maxPixelValue == 0)
		{
			/* Can't go up or down*/
			pixelLevelDelta = 0;
		}
		else if ( MWWxHuffmanRunDecoder_currentRunPixel == 0)
		{
			/* It must be going up to 1 */
			pixelLevelDelta = 1;
			MWWxHuffmanRunDecoder_currentRunPixel ++;

		}
		else if (MWWxHuffmanRunDecoder_currentRunPixel >= maxPixelValue)
		{
			/* It must be going down */
			pixelLevelDelta = -1;
			MWWxHuffmanRunDecoder_currentRunPixel --;
		}
		else
		{
			/* Read a single bit to decide */
			tempByte = 0;
			MWRes = readBitField
			(
				&tempByte,
				1, /* a single bit */
				numWeatherHuffmanEncodingBits,
				p_WxHuffmanBody,
				inout_p_totalBitsConsumed
			);
			if (MWRes != MWRes_Success)
			{
				return MWRes;
			}

			/* 0 is up, 1 is down */
			if (tempByte == 0)
			{
				pixelLevelDelta = 1;
				MWWxHuffmanRunDecoder_currentRunPixel ++;
			}
			else
			{
				pixelLevelDelta = -1;
				MWWxHuffmanRunDecoder_currentRunPixel --;
			}
		}

	} // End is not first run in raster
	else
	{
		/*
		// Is first run in raster.
		// We don't move up immediately,
		// but any 0 length runs we encounter do so.
		*/
		pixelLevelDelta = 1;
	}

	/*
	//##############################################################
	// Get a run length given the new pixel level
	//##############################################################
	*/

	MWWxHuffmanRunDecoder_currentRunLength = 0;

	while (MWWxHuffmanRunDecoder_currentRunLength == 0)
	{
		MWRes = MWWxHuffmanRunDecoder_Internal_readRunLength
		(
			numWeatherHuffmanEncodingBits,
			p_WxHuffmanBody,
			inout_p_totalBitsConsumed,
			MWWxHuffmanRunDecoder_currentRunPixel,
			&MWWxHuffmanRunDecoder_currentRunLength
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/*
		//##############################################################
		// If its a zero length run, it is a continuation of the previous
		// pixel level change.
		//##############################################################
		*/

		if (MWWxHuffmanRunDecoder_currentRunLength == 0)
		{
			if (pixelLevelDelta < 0)
			{
				if (MWWxHuffmanRunDecoder_currentRunPixel == 0)
				{
					/* Can't drop below 0 */
					return MWRes_ProductContentsInvalid;
				}

				MWWxHuffmanRunDecoder_currentRunPixel --;
			}
			else if (pixelLevelDelta > 0)
			{
				if (MWWxHuffmanRunDecoder_currentRunPixel >= maxPixelValue)
				{
					/* Can't rise above maximum */
					return MWRes_ProductContentsInvalid;
				}

				MWWxHuffmanRunDecoder_currentRunPixel ++;
			}
			else
			{
				/* pixelLevelDelta should never be 0 */
				return MWRes_CodeFault;
			}
		}

	} /* Loop around to get non-0 length run */

	/*
	//##############################################################
	// Done
	//##############################################################
	*/

	return MWRes_Success;

}

/*
//====================================================================
// MWWxHuffmanRunDecoder_Internal_DetermineCustomCodes:
//
// This method is used to calculate code words for a custom
// WxHuffman table, given that the table's codeLengths have
// all been set.
//
// PARAMETERS:
//
//	inout_p_table:
//		Pass in a wx huffman table with the code lengths set.
//		This method sets the code patterns based on the code lengths
//
// RETURNS:
//
//	MWRes_Success
//		On successful setting of the codes
//
//	MWRes_ProductContentsInvalid:
//		If a code length value in the table was invalid
//
//====================================================================
*/

TMW_ResultCode MWWxHuffmanRunDecoder_Internal_DetermineCustomCodes
(
	TMW_WxHuffmanTable* inout_p_table
)
{
	/*
	// This array is used to sort the table by code length.
	// Each entry in this array is the index of an entry in the
	// actual table.
	*/
	TByte_Unsigned lengthSortedCodeIndices[MAX_WX_HUFFMAN_TABLE_ENTRIES];

	/* Flag used for bubble sort */
	TByte_Unsigned bool_StartingOver;

	/* Index used for iterating the codes */
	TByte_Unsigned i;

	/* Temporary value used in swapping the code-lengths during sort */
	TByte_Unsigned temp;

	/* For building the codes */
	TByte_Unsigned bool_firstCodeWordFound;
	TByte_Unsigned previousCodeWord;
	TByte_Unsigned lastLength=0;
	TTwoByteInteger_Signed lengthDiff;

	/*
	//##############################################################
	// Create unsorted mapping
	//##############################################################
	*/

	for ( i = 0; i < inout_p_table->numEntries; i ++)
	{
		lengthSortedCodeIndices[i] = i;
	}

	/*
	//##############################################################
	// Sort the mappings by code length
	// This currently uses bubble-sort.
	// We can change it if it needs to be faster.
	//##############################################################
	*/

	/* Sort mappings by length of codeWord */
	bool_StartingOver = 0;

	for (i = 0; i < (inout_p_table->numEntries -1); i ++)
	{
		/* Handle unsigned index starting over */
		if (bool_StartingOver != 0)
		{
			i = 0;
			bool_StartingOver = 0;
		}

		/* Is this pair of lengths in the wrong order */
		if
		(
			inout_p_table->codeLengths[lengthSortedCodeIndices[i]] >
			inout_p_table->codeLengths[lengthSortedCodeIndices[i+1]]
		)
		{

			/* Swap mappings */
			temp = lengthSortedCodeIndices[i];
			lengthSortedCodeIndices[i] = lengthSortedCodeIndices[i+1];
			lengthSortedCodeIndices[i+1] = temp;

			/* Start over */
			i = 0;
			bool_StartingOver = 1;

		}

	} /* Continue iteration */

	/*
	// Mappings are now sorted by length of codeword
	// Hence, index 3 into mappings is the 4th shortest code word
	*/

	/*
	//##############################################################
	// Iterate code words from last to first, growing code as we go.
	//##############################################################
	*/

	/* No code words generated yet */
	bool_firstCodeWordFound = 0;
	previousCodeWord = 0;


	for (i = 0; i < inout_p_table->numEntries; i ++)
	{
		/*
		//##############################################################
		// Is this a used code?
		//##############################################################
		*/
		if ( inout_p_table->codeLengths[lengthSortedCodeIndices[i]] > 0)
		{
			// Test code length
			if (inout_p_table->codeLengths[lengthSortedCodeIndices[i]] > 7)
			{
				/* Illegal code length */
				return MWRes_ProductContentsInvalid;
			}

			/* Is it the first code or a subsequent code? */
			if (bool_firstCodeWordFound == 0)
			{
				/* Shortest codeword is all 0s */
				inout_p_table->codes[lengthSortedCodeIndices[i]] = 0;
				previousCodeWord = 0;
				lastLength = inout_p_table->codeLengths[lengthSortedCodeIndices[i]];
				bool_firstCodeWordFound = 1;
			}
			else
			{
				/* Extend previous code word */

				/* Add 1 to previous code (literally) */
				previousCodeWord += 1;

				/* Add 0's for diference in length */
				lengthDiff = inout_p_table->codeLengths[lengthSortedCodeIndices[i]] - lastLength;
				if (lengthDiff > 0)
				{
					previousCodeWord <<= lengthDiff;
				}

				/* Set code word */
				inout_p_table->codes[lengthSortedCodeIndices[i]] = previousCodeWord;

				/* Leave previous code word for building next, set lastLength */
				lastLength = inout_p_table->codeLengths[lengthSortedCodeIndices[i]];

			}
		}
		else
		{
			/* Code is unused, set to 0 */
			inout_p_table->codes[lengthSortedCodeIndices[i]] = 0;
		}

	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;

}

/*
//====================================================================
// MWWxHuffmanRunDecoder_Internal_readTable:
//
// This method reads in single WxHuffman Table from the product
// stream.
//
// PARAMETERS:
//
//	numWeatherHuffmanEncodingBytes:
//		The number of bytes in this WSI Weather Huffman encoding
//
//	p_WxHuffmanBody:
//		Pointer to the first byte of the contigous array of WSI Weather
//		Huffman encoding bytes (must be at least numWeatherHuffmanEncodingBytes
//		bytes long)
//
//	p_out_WxHuffmanTable:
//		The pointer to the WxHuffmanTable structure that is to be read.
//		The current contents of the structure are overwritten.
//
//	inout_p_totalBytesConsumed:
//		Pointer to the counter of the number of bytes from p_WxHuffmanBody
//		already processed. We update this as we consume bytes.
//
// RETURNS:
//
//	MWRes_Success
//		If all the table was read successfully
//
//	MWRes_NullPointer:
//		If a pointer parameter is NULL
//
//	MWRes_ProductContentsInvalid:
//		If something is wrong with the product contents
//
//====================================================================
*/

TMW_ResultCode MWWxHuffmanRunDecoder_Internal_readTable
(
	TFourByteInteger_Unsigned numWeatherHuffmanEncodingBytes,
	TByte_Unsigned* p_WxHuffmanBody,
	TMW_WxHuffmanTable* out_p_table,
	TFourByteInteger_Unsigned* inout_p_totalBitsConsumed
)
{
	TMW_ResultCode MWRes;

	TFourByteInteger_Unsigned availableBits;
	TByte_Unsigned longestCodeLength;
	TByte_Unsigned tempByte;

	/*
	//##############################################################
	// How many bits are available
	//##############################################################
	*/
	availableBits = numWeatherHuffmanEncodingBytes * 8;

	/*
	//##############################################################
	// We begin by reading the table ID, this is a 3 bit field
	//##############################################################
	*/
	MWRes = readBitField
	(
		&(out_p_table->tableID),
		TABLE_ID_BITFIELD_LENGTH,
		availableBits,
		p_WxHuffmanBody,
		inout_p_totalBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// If it is a custom table, we must read in the code lengths
	//##############################################################
	*/
	if (out_p_table->tableID == 3)
	{
		/*
		//##############################################################
		// Read 3 bits for longest code length
		//##############################################################
		*/
		MWRes = readBitField
		(
			&longestCodeLength,
			TABLE_LONGESTCODELENGTH_BITFIELD_LENGTH,
			availableBits,
			p_WxHuffmanBody,
			inout_p_totalBitsConsumed
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/*
		//##############################################################
		// Table empty so far
		//##############################################################
		*/
		out_p_table->numEntries = 0;

		tempByte = 1;
		while (tempByte != 0)
		{
			/*
			//##############################################################
			// Read 1 bit to see if this one exists or not
			//##############################################################
			*/
			MWRes = readBitField
			(
				&tempByte,
				1,
				availableBits,
				p_WxHuffmanBody,
				inout_p_totalBitsConsumed
			);
			if (MWRes != MWRes_Success)
			{
				return MWRes;
			}

			/*
			//##############################################################
			// If it exists, read the length (3 bits long)
			//##############################################################
			*/
			if (tempByte != 0)
			{
				/* Test for table overflow */
				if (out_p_table->numEntries >= MAX_WX_HUFFMAN_TABLE_ENTRIES)
				{
					/* Overflow */
					return MWRes_ProductContentsInvalid;
				}

				/* There is another code length */
				MWRes = readBitField
				(
					&(out_p_table->codeLengths[out_p_table->numEntries]),
					3,
					availableBits,
					p_WxHuffmanBody,
					inout_p_totalBitsConsumed
				);
				if (MWRes != MWRes_Success)
				{
					return MWRes;
				}

				/* One more entry in the table */
				out_p_table->numEntries ++;
			}

		} /* Continue until single bit flag is 0, indicating no more entries */

	} /* Was a custom table */

	/*
	//##############################################################
	// Read in "other run length" option flag
	//##############################################################
	*/
	MWRes = readBitField
	(
		&(out_p_table->otherRunLengthOption),
		3,
		availableBits,
		p_WxHuffmanBody,
		inout_p_totalBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// If it was a custom table, we need to determine the codes
	// from the code lengths
	//##############################################################
	*/
	if (out_p_table->tableID == 3)
	{
		MWRes = MWWxHuffmanRunDecoder_Internal_DetermineCustomCodes
		(
			out_p_table
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}
	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;

}


/*
//====================================================================
//====================================================================
// Public methods
//====================================================================
//====================================================================
*/

TMW_ResultCode MWWxHuffmanRunDecoder_reset()
{
	/* No tables  */
	numWxHuffmanTables = 0;
	numPixelValuesPerHuffmanTable = 1;

	/* Reset run state */
	MWWxHuffmanRunDecoder_currentRunLength = 0;
	MWWxHuffmanRunDecoder_currentRunPixel = 0;
	MWWxHuffmanRunDecoder_bool_firstRunInRasterNotYetRead = 1;

	/* Done */
	return MWRes_Success;
}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWWxHuffmanRunDecoder_readTables
(
	TFourByteInteger_Unsigned numWeatherHuffmanEncodingBytes,
	TByte_Unsigned* p_WxHuffmanBody,
	TByte_Unsigned numTablesExpected,
	TByte_Unsigned maximumPossiblePixelValue,
	TFourByteInteger_Unsigned* inout_p_totalBitsConsumed
)
{
	TMW_ResultCode MWRes;

	TByte_Unsigned numTablesReadSoFar;

	/*
	//##############################################################
	// Test parameters (as this is the earliest method of this ADT
	// which gets to see these parameters)
	//
	//##############################################################
	*/
	if
	(
		(p_WxHuffmanBody == NULL) ||
		(inout_p_totalBitsConsumed == NULL)
	)
	{
		return MWRes_Success;
	}

	/*
	//##############################################################
	// No tables yet
	//##############################################################
	*/
	numWxHuffmanTables = 0;

	/*
	//##############################################################
	// This means that there is 1 table for every
	// (maximumPossiblePixelValue+1) / numTableExpected pixels
	//##############################################################
	*/
	if (numTablesExpected == 0)
	{
		return MWRes_BadParamValue;
	}

	if ((maximumPossiblePixelValue+1) % numTablesExpected)
	{
		numPixelValuesPerHuffmanTable =
			(maximumPossiblePixelValue / numTablesExpected) + 1;
	}
	else
	{
		numPixelValuesPerHuffmanTable = (maximumPossiblePixelValue+1) / numTablesExpected;
	}

	/*
	//##############################################################
	// Read in each table
	//##############################################################
	*/

	for (numTablesReadSoFar = 0; numTablesReadSoFar < numTablesExpected; numTablesReadSoFar ++)
	{
		/*
		//##############################################################
		// Read this table
		//##############################################################
		*/
		MWRes = MWWxHuffmanRunDecoder_Internal_readTable
		(
			numWeatherHuffmanEncodingBytes,
			p_WxHuffmanBody,
			&(WxHuffmanTables[numTablesReadSoFar]),
			inout_p_totalBitsConsumed
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}
	}

	/*
	//##############################################################
	// Set number of tables
	//##############################################################
	*/
	numWxHuffmanTables = numTablesReadSoFar;

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;

}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWWxHuffmanRunDecoder_getPixels
(
	TFourByteInteger_Unsigned numWeatherHuffmanEncodingBits,
	TByte_Unsigned* p_WxHuffmanBody,
	TFourByteInteger_Unsigned* inout_p_totalBitsConsumed,
	TByte_Unsigned maxPixelValue,
	TFourByteInteger_Unsigned numRequestedPixels,
	TFourByteInteger_Unsigned* p_out_runLength,
	TByte_Unsigned* p_out_pixelValue
)
{
	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// Is there a run left?
	//##############################################################
	*/
	if (MWWxHuffmanRunDecoder_currentRunLength == 0)
	{
		/* We need to read a run */
		MWRes = MWWxHuffmanRunDecoder_Internal_readRun
		(
			numWeatherHuffmanEncodingBits,
			p_WxHuffmanBody,
			inout_p_totalBitsConsumed,
			maxPixelValue,
			MWWxHuffmanRunDecoder_bool_firstRunInRasterNotYetRead
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/* First run in raster has been read */
		MWWxHuffmanRunDecoder_bool_firstRunInRasterNotYetRead = 0;
	}

	/*
	//##############################################################
	// Return the run info, as much as fits in caller's request
	//##############################################################
	*/
	if (MWWxHuffmanRunDecoder_currentRunLength <= numRequestedPixels)
	{
		*p_out_runLength = MWWxHuffmanRunDecoder_currentRunLength;

		/* Run used up */
		MWWxHuffmanRunDecoder_currentRunLength = 0;

	}
	else
	{
		*p_out_runLength = numRequestedPixels;

		/* Reduce current run length by amount returned to caller */
		MWWxHuffmanRunDecoder_currentRunLength -= numRequestedPixels;

	}


	/* Set the pixel value for the caller */
	*p_out_pixelValue = MWWxHuffmanRunDecoder_currentRunPixel;

	/*
	//##############################################################
	// Success
	//##############################################################
	*/

	return MWRes_Success;

}

/*
//====================================================================
//====================================================================
// Hilbert walk state
//====================================================================
//====================================================================
*/

#define MAX_HILBERT_WALK_ORDER 12

/*--------------------------------------------------------------------*/

#define HW_YNEG		0
#define HW_XPOS		1
#define HW_YPOS		2
#define HW_XNEG		3

/*--------------------------------------------------------------------*/

typedef struct
{
	// The length of a side at this order
	TTwoByteInteger_Unsigned sideLength;

	// The transitions at this order
	TByte_Unsigned transitions[4];

	// The moves at this order, there are 4
	TByte_Unsigned moves[3];

	// The current move index
	TByte_Unsigned moveIndex;

} TMWWx_HilbertWalkState;

/*--------------------------------------------------------------------*/

/* The power of 2 that each side of the walked surface is in length */
static TTwoByteInteger_Unsigned hilbertWalkOrder;

/* The state of the hilbert walk at each order */
static TMWWx_HilbertWalkState hilbertWalkStates[MAX_HILBERT_WALK_ORDER+1];

/* The number of moves retrieved */
static TFourByteInteger_Unsigned numWalkMovesIterated;


/*
//====================================================================
//====================================================================
// Hilbert walk methods
//====================================================================
//====================================================================
*/

/*
//====================================================================
// MWWxHuffmanRunDecoder_initHilbertWalkState:
//
// PARAMETERS:
//
//	walkOrderIndex:
//		The 0 based index of the walk state to be initialized.
//
//	sideLength:
//		The length of the side corresponding to this order
//		(2 << (order-1))
//
//	childQuadrant:
//		The 0 based index of the 4 possible child quadrants
//		being initialized from the parent quadrant.
//		(0..3)
//
//	ParentRDir:
//		The parent's R direction (transition0)
//
//	ParentUDir:
//		The parent's U direction (transition1)
//
//	ParentLDir:
//		The parent's L direction (transition2)
//
//	ParentUDir:
//		The parent's U direction (transition3)
//
//====================================================================
*/

TMW_ResultCode MWWxHuffmanRunDecoder_initHilbertWalkState
(
	TByte_Unsigned walkOrderIndex,
	TTwoByteInteger_Unsigned sideLength,
	TByte_Unsigned childQuadrant,
	TByte_Unsigned ParentRDir,
	TByte_Unsigned ParentUDir,
	TByte_Unsigned ParentLDir,
	TByte_Unsigned ParentDDir
)
{
	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// Set side length
	//##############################################################
	*/
	hilbertWalkStates[walkOrderIndex].sideLength = sideLength;

	/*
	//##############################################################
	// Set transitions
	//##############################################################
	*/
	if (childQuadrant == 0)
	{
		hilbertWalkStates[walkOrderIndex].transitions[0] = ParentUDir;
		hilbertWalkStates[walkOrderIndex].transitions[1] = ParentRDir;
		hilbertWalkStates[walkOrderIndex].transitions[2] = ParentDDir;
		hilbertWalkStates[walkOrderIndex].transitions[3] = ParentLDir;
	}
	else if (childQuadrant == 1)
	{
		hilbertWalkStates[walkOrderIndex].transitions[0] = ParentRDir;
		hilbertWalkStates[walkOrderIndex].transitions[1] = ParentUDir;
		hilbertWalkStates[walkOrderIndex].transitions[2] = ParentLDir;
		hilbertWalkStates[walkOrderIndex].transitions[3] = ParentDDir;
	}
	else if (childQuadrant == 2)
	{
		hilbertWalkStates[walkOrderIndex].transitions[0] = ParentRDir;
		hilbertWalkStates[walkOrderIndex].transitions[1] = ParentUDir;
		hilbertWalkStates[walkOrderIndex].transitions[2] = ParentLDir;
		hilbertWalkStates[walkOrderIndex].transitions[3] = ParentDDir;
	}
	else if (childQuadrant == 3)
	{
		hilbertWalkStates[walkOrderIndex].transitions[0] = ParentDDir;
		hilbertWalkStates[walkOrderIndex].transitions[1] = ParentLDir;
		hilbertWalkStates[walkOrderIndex].transitions[2] = ParentUDir;
		hilbertWalkStates[walkOrderIndex].transitions[3] = ParentRDir;
	}
	else
	{
		return MWRes_BadParamValue;
	}

	/*
	//##############################################################
	// Moves values are always the same pattern of parent's directions
	//##############################################################
	*/
	hilbertWalkStates[walkOrderIndex].moves[0] = hilbertWalkStates[walkOrderIndex].transitions[0];
	hilbertWalkStates[walkOrderIndex].moves[1] = hilbertWalkStates[walkOrderIndex].transitions[1];
	hilbertWalkStates[walkOrderIndex].moves[2] = hilbertWalkStates[walkOrderIndex].transitions[2];

	/*
	//##############################################################
	// We start at first move
	//##############################################################
	*/
	hilbertWalkStates[walkOrderIndex].moveIndex = 0;

	/*
	//##############################################################
	// Modify children (recursive function with depth from here
	// of current walk order. Hence max depth is walkOrder)
	//##############################################################
	*/
	if (walkOrderIndex > 0)
	{
		MWRes = MWWxHuffmanRunDecoder_initHilbertWalkState
		(
			(TByte_Unsigned) (walkOrderIndex-1),
			(TTwoByteInteger_Unsigned) (sideLength/2),
			(TByte_Unsigned) 0, // children start in quadrant 0
			hilbertWalkStates[walkOrderIndex].transitions[0],
			hilbertWalkStates[walkOrderIndex].transitions[1],
			hilbertWalkStates[walkOrderIndex].transitions[2],
			hilbertWalkStates[walkOrderIndex].transitions[3]
		);
		return MWRes;
	}
	else
	{
		/* Done */
		return MWRes_Success;
	}
}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWWxHuffmanRunDecoder_advanceHilbertWalk
(
	TByte_Unsigned walkOrderIndex,
	TTwoByteInteger_Unsigned* inout_p_X,
	TTwoByteInteger_Unsigned* inout_p_Y
)
{
	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// Advance the transition. If we reach the last one, then we
	// have to advance the parent (next lower resolution)
	//##############################################################
	*/
	if (hilbertWalkStates[walkOrderIndex].moveIndex < 3)
	{
		/*
		// Modify the point by the current move
		// The next child is always one pixel away, because the child has
		// filled a sideLength/2 by sideLength/2 pixel area.
		*/

		switch (hilbertWalkStates[walkOrderIndex].moves[hilbertWalkStates[walkOrderIndex].moveIndex])
		{
		case HW_YNEG:
			*inout_p_Y = (*inout_p_Y) - 1;
			break;

		case HW_YPOS:
			*inout_p_Y = (*inout_p_Y) + 1;
			break;

		case HW_XPOS:
			*inout_p_X = (*inout_p_X) + 1;
			break;

		case HW_XNEG:
			*inout_p_X = (*inout_p_X) - 1;
			break;

		default:
			return MWRes_CodeFault;

		}

		/* This move has been used up */
		hilbertWalkStates[walkOrderIndex].moveIndex ++;

		/* Init childrens moves which follow this parent move */
		if (walkOrderIndex > 0)
		{
			/*
			//##############################################################
			// Init child next order down based on new parent
			// quadrant index (0-3)
			//##############################################################
			*/
			MWRes = MWWxHuffmanRunDecoder_initHilbertWalkState
			(
				(TByte_Unsigned) (walkOrderIndex-1),
				(TTwoByteInteger_Unsigned) (hilbertWalkStates[walkOrderIndex].sideLength / 2),
				hilbertWalkStates[walkOrderIndex].moveIndex,
				hilbertWalkStates[walkOrderIndex].transitions[0],
				hilbertWalkStates[walkOrderIndex].transitions[1],
				hilbertWalkStates[walkOrderIndex].transitions[2],
				hilbertWalkStates[walkOrderIndex].transitions[3]
			);
			return MWRes;

		}
		else
		{
			/* Success */
			return MWRes_Success;
		}

	}
	else if (walkOrderIndex >= (hilbertWalkOrder-1))
	{
		/* We are at the end of the walk */
		return MWRes_EndOfIteration;
	}
	else
	{
		/* Recurse upward */
		MWRes = MWWxHuffmanRunDecoder_advanceHilbertWalk
		(
			(TByte_Unsigned) (walkOrderIndex + 1),
			inout_p_X,
			inout_p_Y
		);
		return MWRes;
	}



}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWWxHuffmanRunDecoder_prepareHilbertWalk
(
	TTwoByteInteger_Unsigned rasterWidth,
	TTwoByteInteger_Unsigned rasterHeight
)
{
	TTwoByteInteger_Unsigned sideLength;
	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// No points iterated
	//##############################################################
	*/

	numWalkMovesIterated = 0;

	/*
	//##############################################################
	// Determine block order of hilbert walk
	//##############################################################
	*/

	/* Start with order of 1 */
	hilbertWalkOrder = 0;
	sideLength = 1;

	while
	(
		(sideLength < rasterWidth) ||
		(sideLength < rasterHeight)
	)
	{
		/* Is the walk requested too large? */
		if (hilbertWalkOrder > MAX_HILBERT_WALK_ORDER)
		{
			/*
			// Not enough memory dedicated to create a walk of
			// this size
			*/
			return MWRes_OutOfMemory;
		}

		hilbertWalkOrder ++;
		sideLength <<= 1;

	}

	if (hilbertWalkOrder == 0)
	{
		/* Only one point in the walk! */
		return MWRes_Success;
	}

	/*
	//##############################################################
	// Set initial hilbert walk states
	//
	// Order 1 state is at index 0
	// Order 2 state is at index 1 etc..
	// Therefore we subtract 1 from the walk order when using it
	// as an index.
	//
	//##############################################################
	*/
	MWRes = MWWxHuffmanRunDecoder_initHilbertWalkState
	(
		(TByte_Unsigned) (hilbertWalkOrder-1),
		(TTwoByteInteger_Unsigned) sideLength,
		(TByte_Unsigned) 0, /* starts in upper left quadrant */
		(TByte_Unsigned) HW_YPOS,
		(TByte_Unsigned) HW_XPOS,
		(TByte_Unsigned) HW_YNEG,
		(TByte_Unsigned) HW_XNEG
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;

}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWWxHuffmanRunDecoder_getNextHilbertWalkPoint
(
	TTwoByteInteger_Unsigned* inout_p_coordX,
	TTwoByteInteger_Unsigned* inout_p_coordY
)
{
	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// Special case, 1 point in the walk
	//##############################################################
	*/

	if (hilbertWalkOrder == 0)
	{
		return MWRes_EndOfIteration;
	}

	/*
	//##############################################################
	// Advance walk at finest resolution. This works its way
	// up the hierarchy as much as possible.
	// (Index 0 holds the Order 1 walk, which is 2 pixels by 2 pixels
	// in size)
	//##############################################################
	*/

	MWRes = MWWxHuffmanRunDecoder_advanceHilbertWalk
	(
		0,
		inout_p_coordX,
		inout_p_coordY
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// One more move iterated
	//##############################################################
	*/

	numWalkMovesIterated ++;

	/*
	//##############################################################
	// Done
	//##############################################################
	*/

	return MWRes_Success;

}
