/*
//====================================================================
// MW_WxHuffmanProcessor.c:
//
// This implements an ADT which can
// process Weather Huffman compressed rasters into other
// output formats.
//
// All source code was built from inspection of the DOT document
// describing Weather Huffman.  It may not be the same as
// other Weather Huffman implementations. Therefore the
// format is referred to as WSI Weather Huffman.
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//
//
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/

/* Interface of this ADT */
#include "MW_WxHuffmanProcessor.h"

/* Interface for general raster handling methods */
#include "MW_GeneralRasterProcessing.h"

/* For RLE output */
#include "MW_WSIRLE.h"

/* For final DPU and frame output */
#include "MW_OutputEncoder.h"

/* For reading pixels from the wxhuffman data */
#include "MW_WxHuffmanRunDecoder.h"

/* For handling raster rows */
#include "MW_MemorySpaceManager.h"

/* For reading bit fields */
#include "MW_BitFieldOperators.h"

/*
//====================================================================
// Defines
//====================================================================
*/

/*
// The maximum number of upsampling iterations which we
// are prepared to handle (Wx Huffman has a max of 3, but
// we may not need that many)
*/

#define MAX_EVER_PIXEL_VALUE 255

#define MAX_PACKED_PIXEL_MAPPINGS 255

/*
// Uncomment the define below to have the system read through any
// extra padding rows at the end of the image to confirm that
// all of the body is correctly consumed.  This is a good correctness
// check for the parser, but it isn't necessary.  Comment this out
// once testing is completed to reduce workload on the processor
*/
#define TEST_PRODUCT_SIZE_CONSUMED

/*
// Number of bits per packed pixel mapping in packed pixel mapping
// data
*/
#define PACKED_PIXEL_MAPPING_BITFIELD_LENGTH 8

/* EBA field sizes and values (all field sizes are in bits) */
#define EBA_FIELD_LENGTH_NUM_PIXEL_VALUE_CATEGORIES 8
#define EBA_FIELD_LENGTH_NUM_PIXELS_PER_EBACATEGORY 5

/* These EBA fields occur per EBA pss */
#define EBA_FIELD_LENGTH_MAX_SCORE 5
#define EBA_FIELD_LENGTH_COUNT_LIMITED_OR_NOT 1
#define EBA_FIELD_LENGTH_NUM_CORRECTIONS 9

#define EBA_FIELD_VALUE_MAX_SCORE_NO_EBA 31

/*
//====================================================================
// Structures in the stream, these are byte aligned
//====================================================================
*/

#pragma pack(push)
#pragma pack(1)

/* The weather huffman block header */
typedef struct
{
	TFourByteInteger_Unsigned numBytesInBlock;
	TFourByteInteger_Unsigned bitOffsetOfEBAStart;
	TTwoByteInteger_Unsigned downsampledWidth;
	TTwoByteInteger_Unsigned downsampledHeight;
	TTwoByteInteger_Unsigned maxPossiblePixelValue;
	TByte_Unsigned bool_performSmoothingOnUpsample;
	TByte_Unsigned numTables;
	TByte_Unsigned numPackedPixelMappings;

} TMW_WHBlockHeader;

#pragma pack(pop)

/*
//====================================================================
// Private ADT data
//====================================================================
*/

/*
// Scoring regimes for each superpixel quadrant regarding
// 8 nearest neighbors of superpixel
*/
static TByte_Unsigned EBAQuadrantScores[4][9] =
{
	/* In left to right, top to bottom scanline order */
	{4, 3, 2, 3, 0, 1, 2, 1, 1},
	{2, 3, 4, 1, 0, 3, 1, 1, 2},
	{2, 1, 1, 3, 0, 1, 4, 3, 2},
	{1, 1, 2, 1, 0, 3, 2, 3, 4}
};

/*
// The packed pixel mappings
// The index into the array is the pixel in the raster as transmitted.
// The corresponding value in the array is the pixel that the value
// should become
*/
static TByte_Unsigned packedPixelMappings[MAX_PACKED_PIXEL_MAPPINGS];

/*
//====================================================================
//====================================================================
// Internal Methods
//====================================================================
//====================================================================
*/

/*
//====================================================================
// MWWxHuffmanProcessor_Internal_startNewRaster:
//
// Prepares the ADT to start a new raster
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_UnrecognizedTransmogrificationType
//
//	Other errors returned from encoder initialization results
//====================================================================
*/
TMW_ResultCode MWWxHuffmanProcessor_Internal_startNewRaster()
{
	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// Initialize the memory usage manager.
	//##############################################################
	*/

	/*
	// It is safe to initialize this no matter what its state if the
	// memory is no longer being used.  This module is the only one
	// that uses the memory, so we know it is no longer being used.
	*/
	MWRes = MWGeneralRasterProcessing_reset();
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Reset the weather huffman run decoder
	//##############################################################
	*/
	MWRes = MWWxHuffmanRunDecoder_reset();
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Start new output encoding block
	//##############################################################
	*/
	return MWOutputEncoder_startNewEncodingBlock();

}



/*
//====================================================================
// MWWxHuffmanProcessor_Internal_extraBitAlgorithmPass
//
// This method preforms the extra bit algorithm on an upsampled
// raster from its originating (downsampled) raster.
//
// PARAMETERS:
//
//	p_EBABitsStartByte:
//		The byte in which the EBA bits start
//
//	p_inout_numEBABitsUsed:
//		Pointer to the variable which counts how many bits
//		from the EBA start byte have already been read.
//		If the EBA bits do not start at the MSB bit of the
//		byte at p_EBABitsStartByte then this counter starts
//		as > 0.
//
//	maxNumEBABits:
//		The number of bits from the MSB bit of the EBA start byte
//		where the last EBA bit is found. Bits beyond that
//		are not EBA information.
//
//	minimumPixelValueToBeCorrectedThisPass:
//		The minimum pixel value to be considered for correction in this
//		pass.
//
//	p_upsampledPixels:
//		Pointer to the upper left pixel of the upsampled raster
//		plane which was derived from the downsampled raster plane.
//
//	numUpsampledRows:
//		The number of rows in the upsampled raster plane.
//
//	numUpsampledColumns:
//		The number of columns in the upsampled raster plane.
//
//	p_downsampledPixels:
//		Pointer to the upper left pixel of the downsampled raster
//		plane from which the upsampled raster was derived.
//
//	numDownsampledRows:
//		The number of rows in the downsampled raster plane.
//
//	numDownsampledColumns:
//		The number of columns in the downsampled raster plane.
//
//
//====================================================================
*/

TMW_ResultCode MWWxHuffmanProcessor_Internal_extraBitAlgorithmPass
(
	TByte_Unsigned* p_EBABitsStartByte,
	TFourByteInteger_Unsigned* p_inout_numEBABitsUsed,
	TFourByteInteger_Unsigned maxNumEBABits,
	TByte_Unsigned minimumPixelValueToBeCorrectedThisPass,
	TByte_Unsigned* p_upsampledPixels,
	TTwoByteInteger_Unsigned numUpsampledRows,
	TTwoByteInteger_Unsigned numUpsampledColumns,
	TByte_Unsigned* p_downsampledPixels,
	TTwoByteInteger_Unsigned numDownsampledRows,
	TTwoByteInteger_Unsigned numDownsampledColumns

)
{
	TMW_ResultCode MWRes;

	TTwoByteInteger_Unsigned downsampledRow;
	TTwoByteInteger_Unsigned downsampledColumn;

	TTwoByteInteger_Unsigned upsampledRow;
	TTwoByteInteger_Unsigned upsampledColumn;

	TByte_Unsigned* p_upsampledPixelCursor;
	TByte_Unsigned* p_downsampledPixelCursor;

	/*
	// The index of the EBA neighbor being tested (there are 8 neighbors, 9
	// including the pixel itself)
	*/
	TByte_Unsigned EBANeighborScoringIndex;

	/* The index of the super-pixel's sub-pixel quadrant being examined */
	TByte_Unsigned EBAQuadrantIndex;

	/*
	// The indices for iterating the neighbors of the superpixel
	// being examined
	*/
	TByte_Signed EBANeighborRow;
	TByte_Signed EBANeighborColumn;

	/* The value of the pixel at the upsampled result quadrant being examined. */
	TByte_Unsigned subPixelValue;

	/*
	// The value of the super pixel from which the sub-pixel was
	//	derived.
	*/
	TByte_Unsigned superPixelValue;

	/* The EBA score of the superpixel.*/
	TByte_Unsigned EBAScore;

	/* EBA state */
	TByte_Unsigned maximumEBAScoreCorrected;
	TTwoByteInteger_Unsigned maximumEBAScoreCorrected_Count;
	TTwoByteInteger_Unsigned maximumEBAScoreCorrected_CountLimit;

	/* General purpose short-term byte */
	TByte_Unsigned tempBytes[2];

	/* For debugging only */
	TFourByteInteger_Unsigned pixelBitCount;

	/*
	//##############################################################
	// Initialize some variables
	//##############################################################
	*/

	maximumEBAScoreCorrected_Count = 0;
	maximumEBAScoreCorrected_CountLimit = 0;
	maximumEBAScoreCorrected = 0;

	pixelBitCount = 0;

	/*
	//##############################################################
	// Read in the maximum EBA score corrected
	//##############################################################
	*/

	MWRes = readBitField
	(
		&maximumEBAScoreCorrected,
		EBA_FIELD_LENGTH_MAX_SCORE,
		maxNumEBABits,
		p_EBABitsStartByte,
		p_inout_numEBABitsUsed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/* If the result is 31, thats the code for No EBA */
	if (maximumEBAScoreCorrected == EBA_FIELD_VALUE_MAX_SCORE_NO_EBA)
	{
		return MWRes_Success;
	}
	else if (maximumEBAScoreCorrected > 18)
	{
		return MWRes_ProductContentsInvalid;
	}

	/*
	//##############################################################
	// Read in the bit that indicates if there is a limit
	// in the EBA count or not
	//##############################################################
	*/

	tempBytes[0] = 0;
	MWRes = readBitField
	(
		tempBytes,
		EBA_FIELD_LENGTH_COUNT_LIMITED_OR_NOT,
		maxNumEBABits,
		p_EBABitsStartByte,
		p_inout_numEBABitsUsed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	if (tempBytes[0] == 0)
	{
		/* No limit */
		maximumEBAScoreCorrected_CountLimit = 0;
	}
	else
	{
		/*
		//##############################################################
		// Read in the maximum count
		//##############################################################
		*/
		MWRes = readBitField
		(
			(TByte_Unsigned*) &maximumEBAScoreCorrected_CountLimit,
			EBA_FIELD_LENGTH_NUM_CORRECTIONS,
			maxNumEBABits,
			p_EBABitsStartByte,
			p_inout_numEBABitsUsed
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}


	}

	/*
	//##############################################################
	// Iterate the downsampled raster, scoring superpixels
	// by the EBA algorithm rules.
	//##############################################################
	*/

	p_upsampledPixelCursor = p_upsampledPixels;
	p_downsampledPixelCursor = p_downsampledPixels;
	downsampledColumn = 0;
	downsampledRow = 0;

	for ( upsampledRow = 0; upsampledRow < numUpsampledRows; upsampledRow ++)
	{

		for ( upsampledColumn = 0; upsampledColumn < numUpsampledColumns; upsampledColumn ++)
		{

			/*
			//##############################################################
			// If the downsampled pixel value (superpixel value) is at least
			// at the minimum , consider it, using the minimum categories
			// EBA data.
			//##############################################################
			*/

			if (*p_downsampledPixelCursor >= minimumPixelValueToBeCorrectedThisPass)
			{

				/*
				//##############################################################
				// Which quadrant is this?
				//##############################################################
				*/

				EBAQuadrantIndex = (upsampledColumn %2) + ((upsampledRow % 2) * 2);

				/* What is the upsampled sub-pixel (quadrant) value */
				subPixelValue = *p_upsampledPixelCursor;

				/* Total up scores */
				EBANeighborScoringIndex = 0;
				EBAScore = 0;

				downsampledRow = upsampledRow / 2;
				downsampledColumn = upsampledColumn / 2;

				for (EBANeighborRow = -1; EBANeighborRow < 2; EBANeighborRow ++)
				{
					for (EBANeighborColumn = -1; EBANeighborColumn < 2; EBANeighborColumn ++)
					{
						if
						(
							( (EBANeighborRow >= 0) || (downsampledRow > 0) ) &&
							( (EBANeighborColumn >= 0) || (downsampledColumn > 0) ) &&
							( (EBANeighborRow <= 0) || (downsampledRow < (numDownsampledRows-1)) ) &&
							( (EBANeighborColumn <= 0 ) || (downsampledColumn < (numDownsampledColumns-1)) )
						)
						{

							/*
							//##############################################################
							// This neighbor exists, check for scoring
							//##############################################################
							*/

							superPixelValue =
								*(
									p_downsampledPixelCursor +
									(EBANeighborRow * numDownsampledColumns) +
									EBANeighborColumn
								);

							/*
							// If parent super pixel is at least as much as the sub-Pixel,
							// add this neighbor to the score
							*/
							if (superPixelValue >= subPixelValue)
							{
								EBAScore += EBAQuadrantScores[EBAQuadrantIndex][EBANeighborScoringIndex];
							}

						} /* Neighbor not outside of raster bounds */

						/* One more neighbor traversed */
						EBANeighborScoringIndex ++;

					} /* Neighbor column iteration */

				} /* Neighbor row iteration */

				/*
				//##############################################################
				// Does the score meet the requirements for EBA?
				//##############################################################
				*/
				if
				(
					(EBAScore < maximumEBAScoreCorrected) ||
					(
						(EBAScore == maximumEBAScoreCorrected) &&
						(
							(maximumEBAScoreCorrected_CountLimit == 0) || // Limit of 0 means unlimited
							(maximumEBAScoreCorrected_Count < maximumEBAScoreCorrected_CountLimit)
						)
					)
				)
				{
					/* Note if this was at the maximum EBA score */
					if (EBAScore == maximumEBAScoreCorrected)
					{
						maximumEBAScoreCorrected_Count ++;
					}

					/* Read a bit from the corrections and correct this quadrant subpixel */
					tempBytes[0] = 0;

					MWRes = readBitField
					(
						tempBytes,
						1, /* a single bit */
						maxNumEBABits,
						p_EBABitsStartByte,
						p_inout_numEBABitsUsed
					);
					if (MWRes != MWRes_Success)
					{
						return MWRes;
					}

					pixelBitCount ++;

					/*
					// Change quadrant based on value if necessary
					// 1 indicates we should be at or higher than our super pixel
					// 0 indicates we should be lower than our super pixel
					*/

					if (tempBytes[0] != 0)
					{
						if (subPixelValue < *p_downsampledPixelCursor)
						{
							*p_upsampledPixelCursor = subPixelValue + 1;
						}
					}
					else if (subPixelValue >= *p_downsampledPixelCursor)
					{
						*p_upsampledPixelCursor = subPixelValue - 1;
					}

				}


			} /* The super pixel EBAPixelValueCategory was >= the minimum EBAPixelValueCategory for this pass */


			/*
			//##############################################################
			// Increment cursors
			//##############################################################
			*/

			if ((upsampledColumn %2) != 0)
			{
				p_downsampledPixelCursor ++;
			}
			p_upsampledPixelCursor = p_upsampledPixelCursor + 1;


		} /* Next column */


		/*
		//##############################################################
		// Modify cursors
		//##############################################################
		*/


		if ((upsampledRow % 2) == 0)
		{
			/* The scanned downsampled row is the parent of the next row as well */
			/* Depending on wether the last upsampled column index was odd or even,
			   we either did or did not advance beyond the last downsampled column */

			if (numUpsampledColumns %2)
			{
				/* Odd number of columns ends on even index */
				p_downsampledPixelCursor -= (numDownsampledColumns-1);
			}
			else
			{
				/* Even number of columns ends on odd index */
				p_downsampledPixelCursor -= numDownsampledColumns;
			}


		}


	} /* Next row */

	/*
	//##############################################################
	// Done this pass
	//##############################################################
	*/

	return MWRes_Success;



}

/*
//====================================================================
// MWWxHuffmanProcessor_Internal_extraBitAlgorithm:
//
// This performs the extra bit algorithm which corrects upsampled
// sub-pixels based on the nature of the neighbors of the source
// downsampled superpixel and a small bit encoding in the
// source data.
//
// PARAMETERS:
//
//	p_EBABitsStartByte:
//		The byte in which the EBA bits start
//
//	p_inout_numEBABitsUsed:
//		Pointer to the variable which counts how many bits
//		from the EBA start byte have already been read.
//		If the EBA bits do not start at the MSB bit of the
//		byte at p_EBABitsStartByte then this counter starts
//		as > 0.
//
//	maxNumEBABits:
//		The number of bits from the MSB bit of the EBA start byte
//		where the last EBA bit is found. Bits beyond that
//		are not EBA information.
//
//	maxPossiblePixelValue:
//		The maximum possible pixel value in the raster plane
//
//	numPixelLevelCategories:
//		The number of pixel levels grouped into each category
//		for weather huffman table and EBA data sharing.
//
//	p_upsampledPixels:
//		Pointer to the upper left pixel of the upsampled raster
//		plane which was derived from the downsampled raster plane.
//
//	numUpsampledRows:
//		The number of rows in the upsampled raster plane.
//
//	numUpsampledColumns:
//		The number of columns in the upsampled raster plane.
//
//	p_downsampledPixels:
//		Pointer to the upper left pixel of the downsampled raster
//		plane from which the upsampled raster was derived.
//
//	numDownsampledRows:
//		The number of rows in the downsampled raster plane.
//
//	numDownsampledColumns:
//		The number of columns in the downsampled raster plane.
//
//
//
//====================================================================
*/

TMW_ResultCode MWWxHuffmanProcessor_Internal_extraBitAlgorithm
(
	TByte_Unsigned* p_EBABitsStartByte,
	TFourByteInteger_Unsigned* p_inout_numEBABitsUsed,
	TFourByteInteger_Unsigned maxNumEBABits,
	TTwoByteInteger_Unsigned maximumPossiblePixelLevel,
	TTwoByteInteger_Unsigned numPixelLevelsPerCategory,
	TByte_Unsigned* p_upsampledPixels,
	TTwoByteInteger_Unsigned numUpsampledRows,
	TTwoByteInteger_Unsigned numUpsampledColumns,
	TByte_Unsigned* p_downsampledPixels,
	TTwoByteInteger_Unsigned numDownsampledRows,
	TTwoByteInteger_Unsigned numDownsampledColumns

)
{
	TMW_ResultCode MWRes;
	TByte_Unsigned minimumPixelLevelThisPass;
	TByte_Unsigned pixelLevelsInHighestCategory;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(p_inout_numEBABitsUsed == NULL) ||
		(p_upsampledPixels == NULL) ||
		(p_downsampledPixels == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// How many pixels are in the highest category
	//##############################################################
	*/
	pixelLevelsInHighestCategory = (maximumPossiblePixelLevel +1) % numPixelLevelsPerCategory;
	if (pixelLevelsInHighestCategory == 0)
	{
		pixelLevelsInHighestCategory = (TByte_Unsigned) numPixelLevelsPerCategory;
	}

	/*
	//##############################################################
	// Perform an EBA pass for each pixel level category
	//##############################################################
	*/
	minimumPixelLevelThisPass = maximumPossiblePixelLevel - (pixelLevelsInHighestCategory);

	while (minimumPixelLevelThisPass >= 1)
	{
		/*
		//##############################################################
		// Perform pass
		//##############################################################
		*/

		MWRes = MWWxHuffmanProcessor_Internal_extraBitAlgorithmPass
		(
			p_EBABitsStartByte,
			p_inout_numEBABitsUsed,
			maxNumEBABits,
			(TByte_Unsigned) minimumPixelLevelThisPass,
			p_upsampledPixels,
			numUpsampledRows,
			numUpsampledColumns,
			p_downsampledPixels,
			numDownsampledRows,
			numDownsampledColumns
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/*
		//##############################################################
		// Extend minimum downward
		//##############################################################
		*/
		minimumPixelLevelThisPass -= numPixelLevelsPerCategory;


	}

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;

}


/*
//====================================================================
// MWWxWeatherHuffmanProcessor_Internal_readPackedPixelMappings:
//
// In our modificaiton of weather huffman, we transformthe pixels
// values used in the source data into a contigous range of pixel
// values by using a mapping table.  The table is transmitted with
// the product.  The purpose is to drastically lower the number
// of 0 length runs required in the transmitted raster data.
//
// This reads in the mapping table.
//====================================================================
*/

TMW_ResultCode MWWxWeatherHuffmanProcessor_Internal_readPackedPixelMappings
(
	TByte_Unsigned numPackedPixelMappings,
	TFourByteInteger_Unsigned numInputBits,
	TByte_Unsigned* p_inputBits,
	TFourByteInteger_Unsigned* inout_p_numInputBitsUsed
)
{
	TByte_Unsigned packedPixelIndex;
	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if (numPackedPixelMappings > MAX_PACKED_PIXEL_MAPPINGS)
	{
		return MWRes_BadParamValue;
	}

	/*
	//##############################################################
	// Read the packed pixel values
	//##############################################################
	*/
	for (packedPixelIndex = 0; packedPixelIndex < numPackedPixelMappings; packedPixelIndex ++)
	{
		MWRes = readBitField
		(
			&(packedPixelMappings[packedPixelIndex]),
			PACKED_PIXEL_MAPPING_BITFIELD_LENGTH,
			numInputBits,
			p_inputBits,
			inout_p_numInputBitsUsed
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}
	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;

}

/*
//====================================================================
// MWWxHuffmanProcessor_Internal_performPixelMappings:
//
// Maps the pixels in the raster from the source values to the
// replacement values given a mapping table.  The source pixel value
// is the index into the mapping table and the value in the table
// is the replacement pixel value.
//
// The operation is performed in-place in the source raster.
//
// PARAMETERS:
//
//	numPixelMappings:
//		If this value is set to 0, it indicates that there are no
//		pixel mappings, and pixel mapping is not performed.
//
//		Otherwise, indicates the number of pixel mappings.
//		Any pixel value >= in the raster on input will be
//		considered a product error.
//
//	p_pixelMappings:
//		The pixel mappings array, at least numPixelMappings elements
//		long
//
//	numRows:
//		The number of rows in the raster to be read
//
//	numColumns:
//		The number of columns in the raster to be read
//
//	p_rasterPixels:
//		The contiguous bytes in which the raster pixel values
//		should be modified inplace.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_ProductContentsInvalid:
//		If a pixel value >= number of pixels in the mapping table
//
//====================================================================
*/

TMW_ResultCode MWWxHuffmanProcessor_Internal_performPixelMappings
(
	TByte_Unsigned numPixelMappings,
	TByte_Unsigned* p_pixelMappings,
	TTwoByteInteger_Unsigned numRows,
	TTwoByteInteger_Unsigned numColumns,
	TByte_Unsigned* p_rasterPixels
)
{
	TFourByteInteger_Unsigned numPixels;
	TFourByteInteger_Unsigned pixelIndex;

	/*
	//##############################################################
	// No pixel mappings, no change
	//##############################################################
	*/
	if (numPixelMappings == 0)
	{
		return MWRes_Success;
	}

	/*
	//##############################################################
	// Iterate the raster
	//##############################################################
	*/

	numPixels = numRows * numColumns;

	for (pixelIndex = 0; pixelIndex < numPixels; pixelIndex++)
	{
		if ( (*p_rasterPixels) >= numPixelMappings)
		{
			return MWRes_ProductContentsInvalid;
		}
		else
		{
			*p_rasterPixels = p_pixelMappings[*p_rasterPixels];
		}

		// Next pixel
		p_rasterPixels ++;
	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;

}

/*
//====================================================================
// MWWxHuffmanProcessor_Internal_readRaster:
//
// This method is used to read from the Wx Huffman run codes into
// a raster given by the caller.
//
// PARAMETERS:
//
//	maxPossiblePixelValue:
//		The maximum possible pixel value encoded in the raster
//
//	numRows:
//		The number of rows in the raster to be read
//
//	numColumns:
//		The number of columns in the raster to be read
//
//	p_rasterPixels:
//		The contiguous bytes in which the raster pixel values
//		should be stored.
//
//	numInputBits:
//		The number of bits of input available in total
//
//	p_inputBits:
//		Pointer to the start of the input bits
//
//	inout_p_numInputBitsUsed:
//		The number of bits of input already consumed from
//		the total number given by (numInputBits).  The next
//		bit to be read occurs after this index.  This counter
//		is also updated by this method for any bits consumed
//		by it.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_InputFormatInvalid:
//		On error reading the run codes or if expected size
//		is not reached
//
//====================================================================
*/

TMW_ResultCode MWWxHuffmanProcessor_Internal_readRaster
(
	TByte_Unsigned maxPossiblePixelValue,
	TTwoByteInteger_Unsigned numRows,
	TTwoByteInteger_Unsigned numColumns,
	TByte_Unsigned* p_rasterPixels,
	TFourByteInteger_Unsigned numInputBits,
	TByte_Unsigned* p_inputBits,
	TFourByteInteger_Unsigned* inout_p_numInputBitsUsed
)
{
	/*
	//##############################################################
	// Variables
	//##############################################################
	*/

	TMW_ResultCode MWRes;

	TFourByteInteger_Unsigned numPixelsLeft;
	TFourByteInteger_Unsigned numPixelsRead;

	TFourByteInteger_Unsigned runLength;
	TByte_Unsigned pixelValue;


	/* Hilbert walk variables */
	TTwoByteInteger_Unsigned walkPointX;
	TTwoByteInteger_Unsigned walkPointY;

	TFourByteInteger_Unsigned pixelIndex;

	/*
	//##############################################################
	// Prepare the hilbert walk
	//##############################################################
	*/
	MWRes = MWWxHuffmanRunDecoder_prepareHilbertWalk
	(
		numColumns,
		numRows
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Read using the decoder
	//##############################################################
	*/
	numPixelsLeft = numRows * numColumns;
	numPixelsRead = 0;
	walkPointX = 0;
	walkPointY = 0;

	while (numPixelsLeft > 0)
	{
		/*
		//##############################################################
		// Read input
		//##############################################################
		*/
		runLength = 0;


		MWRes = MWWxHuffmanRunDecoder_getPixels
		(
			numInputBits,
			p_inputBits,
			inout_p_numInputBitsUsed,
			maxPossiblePixelValue,
			numPixelsLeft,
			&runLength,
			&pixelValue
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/*
		//##############################################################
		// Store the run into the raster using the walk order
		//##############################################################
		*/

		pixelIndex = 0;
		while (pixelIndex < runLength)
		{
			/* It only counts if it is inside the raster bounds */
			if ((walkPointX < numColumns) && (walkPointY < numRows))
			{
				p_rasterPixels[(walkPointY * numColumns) + walkPointX] = pixelValue;

				/* Used a run pixel */
				pixelIndex ++;
			}

			/*
			// If we will be continuing the run or there is a next run,
			// get the next walk point
			*/
			if ((pixelIndex < runLength) || (numPixelsLeft > runLength))
			{
				/* Get next walk point */
				MWRes = MWWxHuffmanRunDecoder_getNextHilbertWalkPoint
				(
					&walkPointX,
					&walkPointY
				);
				if (MWRes != MWRes_Success)
				{
					/*
					// Ran out of walk points early
					// Run length is invalid
					*/
					return MWRes_ProductContentsInvalid;
				}
			}

		}

		/*
		//##############################################################
		// Modify pixel read counter
		//##############################################################
		*/
		if (runLength > numPixelsLeft)
		{
			return MWRes_ProductContentsInvalid;
		}

		numPixelsLeft -= runLength;
		numPixelsRead += runLength;

	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;

}

/*
//====================================================================
//====================================================================
// Public Methods
//====================================================================
//====================================================================
*/

TMW_ResultCode MWWxHuffmanProcessor_transformFormat
(
	TByte_Unsigned* p_WxHuffmanBody,
	TByte_Unsigned bool_productEndsWithThisBody,
	TTwoByteInteger_Unsigned numRasterColumns,
	TTwoByteInteger_Unsigned numRasterRows,
	TByte_Unsigned numBitsPerPixel,
	TFourByteInteger_Unsigned* p_out_numBitsConsumed
)
{
	TMW_ResultCode MWRes;

	/* The weather huffman header */
	TMW_WHBlockHeader* p_WHHeader;

	/*
	// For tracking how many bits from p_WxHuffmanBody we have
	// consumed
	*/
	TFourByteInteger_Unsigned totalBytesConsumed;

	TFourByteInteger_Unsigned numPixels;

	/* Size of the raster after downsampling (size sent to us) */
	TTwoByteInteger_Unsigned downsampledWidth;
	TTwoByteInteger_Unsigned downsampledHeight;
	TByte_Unsigned* p_workingRaster1;

	/*
	// Size of the raster after final upsampling (with extra trim
	// of value 0 pixels around the right and bottom edges, in order
	// to make final upsampled size a power of 2 of the transmitted
	// downsampled size)
	*/
	TTwoByteInteger_Unsigned upsampledWidth;
	TTwoByteInteger_Unsigned upsampledHeight;
	TByte_Unsigned* p_workingRaster2;

	/* Temporary pixel pointer used for pointer swap */
	TByte_Unsigned* p_tempSwapPointer;

	/* Pixels of final result */
	TByte_Unsigned* p_finalPixels=NULL;

	/* How many levels of upsampling are required */
	TByte_Unsigned levelsOfUpsamplingRequired;

	/* For iterating upsamples and calculating upsample result sizes */
	TByte_Unsigned upsamplingIndex;
	TByte_Unsigned downsamplingIndex;

	/* For creating second raster */
	TFourByteInteger_Unsigned numR2Rows, numR2Columns;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(p_WxHuffmanBody == NULL) ||
		(p_out_numBitsConsumed == NULL)
	)
	{
		return MWRes_NullPointer;
	}


	if (numRasterColumns * numRasterRows > MAX_NUM_PIXELS)
	{
		/* Not enough memory set aside to handle such long rows */
		return MWRes_OutOfMemory;
	}

	/*
	//##############################################################
	// No bits consumed so far
	//##############################################################
	*/
	*p_out_numBitsConsumed = 0;

	/*
	//##############################################################
	// Get ready for processing
	//##############################################################
	*/

	MWRes = MWWxHuffmanProcessor_Internal_startNewRaster();
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Read in the weather huffman block header
	//##############################################################
	*/

	p_WHHeader = (TMW_WHBlockHeader*) p_WxHuffmanBody;

	*p_out_numBitsConsumed = sizeof(TMW_WHBlockHeader) * 8;

	/* What is downsampled size as transmitted */
	downsampledWidth = p_WHHeader->downsampledWidth;
	downsampledHeight = p_WHHeader->downsampledHeight;

	/*
	//##############################################################
	// Read in the packed pixel mappings
	//##############################################################
	*/
	MWRes = MWWxWeatherHuffmanProcessor_Internal_readPackedPixelMappings
	(
		p_WHHeader->numPackedPixelMappings,
		p_WHHeader->numBytesInBlock * 8,
		p_WxHuffmanBody,
		p_out_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Read in the weather huffman tables
	//##############################################################
	*/
	MWRes = MWWxHuffmanRunDecoder_readTables
	(
		p_WHHeader->numBytesInBlock,
		p_WxHuffmanBody,
		p_WHHeader->numTables,
		(TByte_Unsigned) (p_WHHeader->maxPossiblePixelValue),
		p_out_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Determine the number of levels of upsampling required and
	// the final upsampled dimensions (>= actual dimensions, with
	// extra trim of pixel value 0 around right and bottom edges)
	//##############################################################
	*/

	upsampledWidth = downsampledWidth;
	upsampledHeight = downsampledHeight;
	for (levelsOfUpsamplingRequired = 0; levelsOfUpsamplingRequired < 4; levelsOfUpsamplingRequired ++)
	{
		if
		(
			(upsampledWidth >= numRasterColumns) &&
			(upsampledHeight >= numRasterRows)
		)
		{
			/* Done */
			break;
		}
		else
		{
			upsampledWidth *= 2;
			upsampledHeight *= 2;
		}

	}

	/*
	//##############################################################
	// Wx Huffman supports at most 3 levels of upsampling
	//##############################################################
	*/

	if (levelsOfUpsamplingRequired >= 4)
	{
		return MWRes_ProductContentsInvalid;
	}

	/*
	//##############################################################
	// Allocate first raster to be last downsampled required size
	// in length, or final size if no dowsampling was done
	//##############################################################
	*/

	if (downsampledWidth < numRasterColumns)
	{
		numR2Rows = (numRasterRows / 2);
		if (numRasterRows %2)
		{
			numR2Rows ++;
		}

		numR2Columns = (numRasterColumns / 2);
		if (numRasterColumns %2)
		{
			numR2Columns ++;
		}

		numPixels = numR2Rows * numR2Columns;

	}
	else
	{
		numPixels = numRasterRows * numRasterColumns;
	}

	MWRes = MWGeneralRasterProcessing_allocateRaster
	(
		numPixels,
		&p_workingRaster1
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// If there was downsampling, allocate second raster to be
	// final size
	//##############################################################
	*/
	if (downsampledWidth < numRasterColumns)
	{
		numPixels = numRasterRows * numRasterColumns;
		MWRes = MWGeneralRasterProcessing_allocateRaster
		(
			numPixels,
			&p_workingRaster2
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}
	}

	/*
	//##############################################################
	// Read into the raster such that the final raster after
	// upsampling will be the one that can handle the full size
	//##############################################################
	*/

	/* Make sure full size raster gets final upsample */
	if
	(
		(levelsOfUpsamplingRequired != 0) &&
		((levelsOfUpsamplingRequired %2) == 0)
	)
	{
		/*
		// At this point raster 2 is the larger one, so we have
		// to use it as the first downsampled raster so that
		// the final upsample goes into the full size
		*/

		p_tempSwapPointer = p_workingRaster1;
		p_workingRaster1 = p_workingRaster2;
		p_workingRaster2 = p_tempSwapPointer;

	}

	/*
	//##############################################################
	// Read the transmitted raster
	//##############################################################
	*/
	MWRes = MWWxHuffmanProcessor_Internal_readRaster
	(
		(TByte_Unsigned) (p_WHHeader->maxPossiblePixelValue),
		downsampledHeight,
		downsampledWidth,
		p_workingRaster1,
		p_WHHeader->numBytesInBlock * 8,
		p_WxHuffmanBody,
		p_out_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Perform each level of upsampling
	//##############################################################
	*/
	for (upsamplingIndex = 0; upsamplingIndex < levelsOfUpsamplingRequired; upsamplingIndex ++)
	{
		/*
		// Working raster 1 holds downsampled source
		// Working raster 2 will hold upsampled results
		*/

		/*
		//##############################################################
		// What is the upsampled size
		//##############################################################
		*/

		/*
		// We can't just double the downsampled size as that does not
		// account for non-even sizes during original downsampling.
		*/
		upsampledWidth = numRasterColumns;
		upsampledHeight = numRasterRows;

		/*
		// Note that levelsOfUpsamplingRequired cannot ever be 0 at this code point
		*/
		for (downsamplingIndex = upsamplingIndex; downsamplingIndex < (levelsOfUpsamplingRequired-1); downsamplingIndex ++)
		{
			TTwoByteInteger_Unsigned tempUpsampledWidth, tempUpsampledHeight;

			tempUpsampledWidth = upsampledWidth / 2;
			if ((tempUpsampledWidth * 2) < upsampledWidth)
			{
				upsampledWidth = tempUpsampledWidth + 1;
			}
			else
			{
				upsampledWidth = tempUpsampledWidth;
			}

			tempUpsampledHeight = upsampledHeight / 2;
			if ((tempUpsampledHeight * 2) < upsampledHeight)
			{
				upsampledHeight = tempUpsampledHeight + 1;
			}
			else
			{
				upsampledHeight = tempUpsampledHeight;
			}

		}

		/*
		//##############################################################
		// Perform upsampling
		//##############################################################
		*/
		MWRes = MWGeneralRasterProcessing_upsample
		(
			downsampledHeight,
			downsampledWidth,
			p_workingRaster1,
			upsampledHeight,
			upsampledWidth,
			p_workingRaster2,
			p_WHHeader->bool_performSmoothingOnUpsample
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/*
		//##############################################################
		// Perform EBA
		//##############################################################
		*/
		MWRes = MWWxHuffmanProcessor_Internal_extraBitAlgorithm
		(
			p_WxHuffmanBody,
			p_out_numBitsConsumed,
			p_WHHeader->numBytesInBlock * 8,
			p_WHHeader->maxPossiblePixelValue,
			(TTwoByteInteger_Unsigned) ((p_WHHeader->maxPossiblePixelValue + 1) / (p_WHHeader->numTables)), // 1 EBA category per table
			p_workingRaster2,
			upsampledHeight,
			upsampledWidth,
			p_workingRaster1,
			downsampledHeight,
			downsampledWidth
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}


		/*
		//##############################################################
		// Prepare for next iteration of upsampling
		//##############################################################
		*/
		if (upsamplingIndex != (levelsOfUpsamplingRequired-1))
		{
			/* Swap downsampled and upsampled roles */
			p_tempSwapPointer = p_workingRaster1;
			p_workingRaster1 = p_workingRaster2;
			p_workingRaster2 = p_tempSwapPointer;

			/*
			// New downsampled width and height are the last upsampling
			// result width and height
			*/
			downsampledWidth = upsampledWidth;
			downsampledHeight = upsampledHeight;

		}
		else
		{
			/*
			// That was the final round, upsample result is the
			// final set of pixels
			*/
			p_finalPixels = p_workingRaster2;
		}

	}

	// Case where no upsampling was required, final pixes are the
	// ones we read in.
	if (levelsOfUpsamplingRequired == 0)
	{
		p_finalPixels = p_workingRaster1;
	}


	/*
	//##############################################################
	// Perform pixel mappings
	//##############################################################
	*/



	MWRes = MWWxHuffmanProcessor_Internal_performPixelMappings
	(
		p_WHHeader->numPackedPixelMappings,
		packedPixelMappings,
		numRasterColumns,
		numRasterRows,
		p_finalPixels
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Encode the pixels to the output
	//##############################################################
	*/

	MWRes = MWOutputEncoder_outputEncodedBytes
	(
		(numRasterRows * numRasterColumns),
		p_finalPixels,
		1, /* Is the end of the encoding block */
		bool_productEndsWithThisBody

	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Body blocks are byte aligned, so read in padding bits until
	// end of byte
	//##############################################################
	*/

	if ((*p_out_numBitsConsumed) % 8 != 0)
	{
		/*
		// Done with upsamplingIndex, so use it to read in the padding
		// bits
		*/

		MWRes = readBitField
		(
			&upsamplingIndex,
			(8 - ((*p_out_numBitsConsumed) % 8) ),
			p_WHHeader->numBytesInBlock * 8,
			p_WxHuffmanBody,
			p_out_numBitsConsumed
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}
	}

	/*
	//##############################################################
	// Test product size consumed if compile time define is declared
	//##############################################################
	*/
#ifdef TEST_PRODUCT_SIZE_CONSUMED

	totalBytesConsumed = (*p_out_numBitsConsumed) / 8;

	if ( totalBytesConsumed != p_WHHeader->numBytesInBlock )
	{
		return MWRes_ProductNotExpectedSize;
	}

#endif

	/*
	//##############################################################
	// Done
	//##############################################################
	*/

	return MWRes_Success;

}
