/*
====================================================================
// MW_ZLib.h:
//
// This forms a simplified interface to ZLib which handles zlib
// memory usage in a way suitable for embedded environments.
//
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//
====================================================================
*/

#ifndef MW_ZLIB_HEADER
#define MW_ZLIB_HEADER

/*
====================================================================
// Required includes
====================================================================
*/

#include <zlib.h>
#include "MW_DataFieldTypes.h"
#include "MW_MemorySpaceManager.h"

/*
====================================================================
// Defines
====================================================================
*/
#define INTERNAL_INFLATION_BUFFER_LENGTH 1024

/*
====================================================================
// Some types
====================================================================
*/

typedef struct
{
	// internal buffer used for inflating data
	TFourByteInteger_Unsigned numInflatedBytesInternal;
	TFourByteInteger_Unsigned numInflatedBytesInternalConsumed;
	TByte_Unsigned inflatedBytesInternal[INTERNAL_INFLATION_BUFFER_LENGTH];

	// Pointer to the memory space manager which will be used for zAlloc and zFree calls
	TMW_MemorySpaceManager* p_memorySpaceManager;

	// Z stream
	z_stream zStream;

	// The last inflate call result
	TFourByteInteger_Signed ZRes;

} TMWZLib_SessionStructure;

/*
====================================================================
// The interface functions
====================================================================
*/


/*
====================================================================
// MWZLib_inflateStart:
//
// This is used to begin an inflation session. The caller must indicate
// the number of bytes available for the inflation source, and also, the
// location of the source.  This is required for zlib to read in the deflation
// session header.
//
// It is not necessary that this be ALL of the bytes in the deflated source data.
// The data can be handed to this object in chunks by later calling
// MWZLib_ResupplyInputBuffer in response to MWRes_InputBufferExhausted
// result codes.  However, the buffer passed in here should be large enough for
// the ZLib session to be initiated or an error will be returned.
//
// PARAMETERS:
//
//		p_zlibSession:
//			Pass in a pointer to zlib session tracking structure to be used.
//			On return of MWRes_Success, that same structure instance should be used
//			in calls to MWZLib_InflateBytes and MWZLib_InflateEnd for this session.
//			Note that calling MWZLib_InflateEnd always invalidates the session handle.
//
//		numBytes:
//			The number of bytes of available input in the source data.
//
//		p_sourceByteArray:
//			Pointer to the start of the available bytes for input to be INFLATEd.
//
//		p_memorySpaceManager:
//			Pointer to the memory space manager that should be used for this ZLib session.
//			It is critical that this memory space manager is used ONLY by this particular ZLib session
//			until this session ends.
//
// RETURNS:
//
//	MWRes_Success
//		On success
//
//	MWRes_NullPointer:
//		If p_zlibSession is NULL or
//		If p_sourceByteArray is NULL
//
//	MWRes_InflationError:
//		On failure to start the inflation session, due to invalid source data.
//
//	MWRes_OutOfMemory:
//		If the internal memory used for inflate session memory requests
//		was not large enough (compile time setting)
//
====================================================================
*/
TMW_ResultCode MWZLib_inflateStart
(
	TMWZLib_SessionStructure* p_zlibSession,
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_sourceByteArray,
	TMW_MemorySpaceManager* p_memorySpaceManager
);

/*
====================================================================
// MWZLib_resupplyInputBuffer:
//
// This is used when the input buffer supplied in MWZLib_InflateStart does
// not contain the entire deflated stream.  In that case, the method
// MWZLib_InflateBytes will eventually return MWRes_InputBufferExhausted and
// this method can be used to supply the next part of the input stream.
//
// Note that you can keep re-using the same p_sourceByteArray, so long as you
// overwrite it with the new data and set the numBytes parameter to represent
// the size of the new input chunk.
//
// PARAMETERS:
//
//	p_zlibSession:
//		Pointer to the INFLATE session tracking structure, as given in the original call to MWZLib_InflateStart.
//
//	numBytes:
//		The number of bytes of available in this next section of input source data.
//
//	p_sourceByteArray:
//		Pointer to the start of the available bytes for input to be INFLATEd.
//
// RETURNS:
//
//	MWRes_Success
//		On success
//
//	MWRes_NullPointer:
//		If p_zlibSession is NULL or
//		If p_sourceByteArray is NULL
//
//	MWRes_BadCallingOrder:
//		If the zlib session state does not indicate that the available input had
//		been exhausted.
//
//	MWRes_BadParamValue:
//		If numBytes == 0
//
====================================================================
*/
TMW_ResultCode MWZLib_resupplyInputBuffer
(
	TMWZLib_SessionStructure* p_zlibSession,
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_sourceByteArray
);

/*
====================================================================
// MWZLib_inflateEnd:
//
// This is used to end an inflation session, given its handle.
// Once this method is called, this handle should no longer be considered valid
// whatever the return result.
//
// PARAMETERS:
//
//	p_zlibSession:
//		Handle to the zlib session tracking structure.
//
// RETURNS:
//
//	MWRes_Success
//		On success
//
//	MWRes_InflationError:
//		On failure to end the session.
//
//	MWRes_OutOfMemory:
//		If the internal memory used for inflate session memory requests
//		was not large enough (compile time setting)
//
====================================================================
*/
TMW_ResultCode MWZLib_inflateEnd
(
	TMWZLib_SessionStructure* p_zlibSession
);

/*
====================================================================
// MWZLib_inflateBytes:
//
// This is used to inflate a particular number of bytes into an array
// given by the caller.
//
// PARAMETERS:
//
//	p_zlibSession:
//		Pointer to the INFLATE session tracking structure, as given in the original call to MWZLib_InflateStart.
//
//	numBytesToInflate:
//		The number of bytes to inflate into the destination array.  Note that this is the
//		number of destination bytes, not the number of source bytes.
//
//	p_destinationArray:
//		Pointer to the bytes into which the inflated data will be stored.  The
//		data storage always begins at the first byte indicated by this pointer.
//		You can store data further into an array by externally moving the pointer
//		that you pass into this parameter.
//
//	p_out_numBytesInflated:
//		On return of the function, the variable pointed to by this parameter will contain
//		the number of bytes inflated into the destination array by this function call.
//		This number will always be <= numBytesToInflate.
//		If the return code is MWRes_Success, it will be numBytesToInflate.
//
//	p_out_numBytesConsumed:
//		On return of the function, the variable pointed to by this parameter will contain
//		the number of source bytes (deflated bytes) consumed by Inflation.
//
// RETURNS:
//
//	MWRes_Success
//		On success.  The expected bytes were inflated.
//		p_out_numBytesInflated will give a number equal to
//		numBytesToInflate.
//
//	MWRes_InflationError:
//		On failure to inflate the requested number of bytes, due
//		to some sort of Inflation error other than the end of the
//		deflated data being encountered.
//
//	MWRes_EndOfDeflatedData:
//		If the end of the deflated data was found during this inflation
//		request. p_out_numBytesInflated will give a number less than
//		numBytesToInflate.
//
//	MWRes_InputBufferExhausted:
//		If the input buffer ran out, and MWZLib_ResupplyInputBuffer should
//		be called.  If there is no more input to give using
//		MWZLib_ResupplyInputBuffer, then there is some sort of error in
//		the source data.
//
//	MWRes_OutOfMemory:
//		If the internal memory used for inflate session memory requests
//		was not large enough (compile time setting)
//
====================================================================
*/
TMW_ResultCode MWZLib_inflateBytes
(
	TMWZLib_SessionStructure* p_zlibSession,
	TFourByteInteger_Unsigned numBytesToInflate,
	TByte_Unsigned* p_destinationArray,
	TFourByteInteger_Unsigned* p_out_numBytesInflated,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed
);



#endif

