/*
====================================================================
// MW_WSIRLE.h:
//
// This is the public interface definition for an ADT
// (Abstract Data Type) module which performs encoding
// of raster data into the WSI Run Length Encoding
// format.
//
// The ADT exists as a single instance.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
====================================================================
*/
#ifndef MW_WSIRLE_INTERFACE
#define MW_WSIRLE_INTERFACE

/*
====================================================================
// Required includes
====================================================================
*/
#include "MW_DataFieldTypes.h"


/*
====================================================================
// MWWSIRLE_beginRaster:
//
// This method is used whenever a new raster is being
// compressed or decompressed.
//
// It resets the encoder/decoder state and prepares it to
// encode a raster according to the WSI RLE rules.
//
// PARAMETERS:
//		none
//
// RETURNS:
//
//	MWRes_Success
//
====================================================================
*/
TMW_ResultCode MWWSIRLE_beginRaster();

/*
====================================================================
// MWWSIRLE_compressPixels:
//
// This method is used to pass pixels to the compressor.
// The caller's output buffer is filled with the resulting
// run codes.  The input buffer parameters are adjusted
// to reflect the amount of input data which was consumed
// before the end of the available input data was reached
// or the output buffer was filled.
//
// PARAMETERS:
//
//	numPixels:
//		This parameter should be set to the number of pixels
//		in the callers p_Pixels array.
//
//	p_Pixels:
//		This pointer to an array of pixels is used to
//		indicate the first of the numPixels in the pixel
//		array.
//
//	p_inout_numPixelsConsumed:
//		The counter referred to by this pointer is used to
//		track the number of input pixels in the callers array
//		which have already been consumed and should not be
//		encoded.  It is passed in as the value which reflects
//		the state of the pixels in the buffer, and
//		updated by the method before return.
//
//	maxOutputBytes:
//		This indicates the maximum number of bytes that
//		can be stored in the output run code buffer
//		from the p_outputByteBuffer position forward.
//
//	p_outputByteBuffer:
//		This points to an array of bytes supplied by the
//		caller which is to be filled with run code
//		byte values based on the input pixels.
//
//	p_inout_numOutputBytesFilled:
//		This points to a counter of the number of
//		bytes in the caller's output byte buffer
//		which are already filled.  The method
//		will begin writing at the first byte after
//		the last one indicated by this counter. The
//		method will update the counter to refer to
//		any newly added bytes before this method
//		returns.
//
// RETURNS:
//
//	MWRes_Success:
//		This is returned if no errors are encountered
//		and the input buffer was entirely consumed
//		without requiring more output buffer space.
//
//	MWRes_DataStructureFull:
//		This is returned if the output buffer was filled
//		and more output room is needed to completely
//		express all of the output generated from the
//		input that was consumed. Some, all, or none
//		of the input may have been consumed depending
//		on the encoder state on method entry
//		and the amount of input provided.
//
//
//	MWRes_NullPointer:
//		One of the pointers passed in was NULL
====================================================================
*/
TMW_ResultCode MWWSIRLE_compressPixels
(
	TFourByteInteger_Unsigned numPixels,
	TByte_Unsigned* p_Pixels,
	TFourByteInteger_Unsigned* p_inout_numPixelsConsumed,
	TFourByteInteger_Unsigned maxOutputBytes,
	TByte_Unsigned* p_outputByteBuffer,
	TFourByteInteger_Unsigned* p_inout_numOutputBytesFilled
);



/*
====================================================================
// MWWSIRLE_flushRun:
//
// This method should be called when all of the pixel
// data in the raster has been processed, or when a run
// termination point in the run length encoding process
// has been reached.  (For example, some RLE codecs specify
// that a run must end at the end of a scan line.)
// This terminates the current run and flushes remaining
// run code data into the output buffer.
//
// PARAMETERS:
//
//	maxOutputBytes:
//		This indicates the maximum number of bytes that
//		can be stored in the output run code buffer
//		from the p_outputByteBuffer position forward.
//
//	p_outputByteBuffer:
//		This points to an array of bytes supplied by the
//		caller which is to be filled with run code
//		byte values based on any run state information
//		remaining to be expressed by the encoder.
//
//	p_inout_numOutputBytesFilled:
//		This points to a counter of the number of
//		bytes in the caller's output byte buffer
//		which are already filled.  The method
//		will begin writing at the first byte after
//		the last one indicated by this counter. The
//		method will update the counter to refer to
//		any newly added bytes before this method
//		returns.
//
// RETURNS:
//
//	MWRes_Success:
//		The method completed and all remaining run code
//		data was flushed into the output buffer.
//
//	MWRes_DataStructureFull:
//		The method completed, but there is still more
//		data to be flushed into an output buffer.
//		Call this method again with available output buffer
//		bytes.
//
====================================================================
*/
TMW_ResultCode MWWSIRLE_flushRun
(
	TFourByteInteger_Unsigned maxOutputBytes,
	TByte_Unsigned* p_outputByteBuffer,
	TFourByteInteger_Unsigned* p_inout_numOutputBytesFilled
);

/*
====================================================================
// MWWSIRLE_decompressProductBodyBlock:
//
// Perform the decompression of the run data. Decompressed
// data is passed directly to the MWOutputEncoder_outputEncodedBytes
// method.
//
// PARAMETERS:
//
//	numBytes:
//		The number of bytes in the RLE data to be decompressed.
//
//	p_bytes:
//		The bytes in the RLE data.
//
//	p_out_numBytesConsumed:
//		Pointer to a variable which will return the number of bytes
//		consumed out of the numBytes given by the caller.
//
//	bool_isLastBodyBlockInProduct:
//		Is the body block the last in the product.
//		Set to 1 if it is, 0 otherwise.
//
//
// RETURNS:
//
//
====================================================================
*/
TMW_ResultCode MWWSIRLE_decompressProductBodyBlock
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed,
	TByte_Unsigned bool_isLastBodyBlockInProduct
);

/*
====================================================================
// MWWSIRLE_decompressPartialProductBodyBlock:
//
// Perform the decompression of the run data for
// part of a product body block. Decompressed data is passed
// directly to the MWOutputEncoder_outputEncodedBytes method.
//
// PARAMETERS:
//
//	numBytes:
//		The number of bytes in the RLE data to be decompressed.
//
//	p_bytes:
//		The bytes in the RLE data.
//
//	p_out_numBytesConsumed:
//		Pointer to a variable which will return the number of bytes
//		consumed out of the numBytes given by the caller.
//
//	bool_containsEndOfProductBodyBlock:
//		Set to 1 to indicate that the end of the body block is found
//		in the bytes given this call.
//		Set to 0 otherwise.
//
//	bool_isLastBodyBlockInProduct:
//		Is the body block the last in the product.
//		Set to 1 if it is, 0 otherwise.
//
//
// RETURNS:
//
//
====================================================================
*/
TMW_ResultCode MWWSIRLE_decompressPartialProductBodyBlock
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed,
	TByte_Unsigned bool_containsEndOfProductBodyBlock,
	TByte_Unsigned bool_isLastBodyBlockInProduct
);

/*
====================================================================
// MWWSIRLE_decompressPartialProductBodyBlockIntoRaster
//
// Perform the decompression of the run data for
// part of a product body block. Decompressed data is written
// into the raster specified by the caller.
//
// PARAMETERS:
//
//	numBytes:
//		The number of bytes in the RLE data to be decompressed.
//
//	p_bytes:
//		The bytes in the RLE data.
//
//	p_out_numBytesConsumed:
//		Pointer to a variable which will return the number of bytes
//		consumed out of the numBytes given by the caller.
//
//	numPixelsInDestination:
//		The number of pixels expected in the destination raster
//
//	p_destinationRaster:
//		Pointer to the upper left pixel of the destination raster.
//
//	p_inout_numPixelWritten:
//		Pointer to a variable which tracks the number of pixels written
//		to the destination raster.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_InputBufferExhausted
//
//	MWRes_DataStructureFull:
//		If the number of pixels would exceed numPixelsInDestination
//
====================================================================
*/
TMW_ResultCode MWWSIRLE_decompressPartialProductBodyBlockToRaster
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed,
	TFourByteInteger_Unsigned numPixelsInDestination,
	TByte_Unsigned* p_destinationRaster,
	TFourByteInteger_Unsigned* p_inout_numPixelsWritten
);


/* End header wrapper */
#endif

