/*
//====================================================================
// MW_Transmogrifier.c:
//
// Implementation of the transmogrifier ADT.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
//
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/

/* Public interface of this ADT */
#include "MW_Transmogrifier.h"

/* Interface of ADT used to output results */
#include "MW_OutputEncoder.h"

/* Output DPU Encoder interface for setting final output pipe callback*/
#include "MW_OutputDPUEncoder.h"

/* Interface of the ADT used to provide memory for zlib inflation */
#include "MW_MemorySpaceManager.h"

/* Interface of ADT used to handle DEFLATED bodies*/
#include "MW_DEFLATECODEC.h"

/* Interface of ADT used to handle RLE bodies*/
#include "MW_WSIRLE.h"

/* Interface of ADT used to handle decoding Wx Weather Huffman rasters */
#include "MW_WxHuffmanProcessor.h"

/* Interface of ADT used to handle decoding Downsampled Deflated RLE rasters */
#include "MW_DownsampledDeflatedRLEProcessor.h"

/* Product Headers */
#include "MW_ProductHeaders.h"

/* For reading bit aligned fields */
#include "MW_BitFieldOperators.h"

/*
//====================================================================
// Defines
//====================================================================
*/



/*
//====================================================================
// ADT internal data
//====================================================================
*/



/*
//====================================================================
//====================================================================
// Internal methods
//====================================================================
//====================================================================
*/


/*
//====================================================================
// MWTransmogrifier_Internal_unpackGenericProductHeader:
//
// Reads in the product header, unpacking it and returning it
// to the caller in the memory. This assumes that the headerType
// single bit field has already been read.
//
// PARAMETERS:
//
//	numBytesInProduct:
//		The number of bytes in the product stream
//
//	p_bytesOfProduct:
//		Array holding the entire product stream
//
//	inout_p_numBitsConsumed:
//		Pointer to a counter of the number of bits consumed from
//		the product stream before this call.
//		This count is updated by this method in the counter
//		variable.
//
//	p_out_UnpackedHeader:
//		Pointer to a TMW_GenericProductHeader_Output structure
//		that is filled from the bitpacked version in the data stream.
//
// SIDE EFFECTS:
//
//	Parses past the end of the header in the data stream
//
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_ProductHeaderFieldInvalid:
//		If field values are outside the expected ranges
//====================================================================
*/

TMW_ResultCode MWTransmogrifier_Internal_unpackGenericProductHeader
(
	TFourByteInteger_Unsigned numBytesInProduct,
	TByte_Unsigned* p_bytesOfProduct,
	TFourByteInteger_Unsigned* inout_p_numBitsConsumed,
	TMW_GenericProductHeader_Output* p_out_UnpackedHeader
)
{
	TMW_ResultCode MWRes;
	TFourByteInteger_Unsigned numBitsInProduct;

#ifndef HAS_MEMCPY
	TFourByteInteger_Unsigned byteIndex;
#endif

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if ((p_out_UnpackedHeader == NULL) || (p_bytesOfProduct == NULL))
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Clear structure to all off bits
	//##############################################################
	*/
#ifdef HAS_MEMCPY
	memset (p_out_UnpackedHeader, 0, sizeof(TMW_GenericProductHeader_Output));
#else
	for (byteIndex = 0; byteIndex < sizeof(TMW_GenericProductHeader_Output); byteIndex++)
	{
		((TByte_Unsigned*) p_out_UnpackedHeader)[byteIndex] = 0;
	}
#endif

	/*
	//##############################################################
	// Header type is known
	//##############################################################
	*/
	p_out_UnpackedHeader->headerType = Generic_ProductHeaderType;
	numBitsInProduct = numBytesInProduct * 8;

	/*
	//##############################################################
	// Product ID
	//##############################################################
	*/

	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->productID),
		GPH_PRODUCTID_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Issue Time
	//##############################################################
	*/
	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->issueMonth),
		GPH_MONTH_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}


	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->issueDayOfMonth),
		GPH_DAYOFMONTH_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->issueHour),
		GPH_HOUR_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->issueMinute),
		GPH_MINUTE_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Valid Time
	//##############################################################
	*/

	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->validMonth),
		GPH_MONTH_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}


	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->validDayOfMonth),
		GPH_DAYOFMONTH_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->validHour),
		GPH_HOUR_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->validMinute),
		GPH_MINUTE_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Georeference
	//##############################################################
	*/

	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->maxLatitude),
		GPH_LATITUDE_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	MWRes = signExtendBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->maxLatitude),
		GPH_LATITUDE_NUMBITS
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->minLongitude),
		GPH_LONGITUDE_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->minLatitude),
		GPH_LATITUDE_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	MWRes = signExtendBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->minLatitude),
		GPH_LATITUDE_NUMBITS
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}


	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->maxLongitude),
		GPH_LONGITUDE_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Test fields
	//##############################################################
	*/

	if
	(
		(p_out_UnpackedHeader->issueMonth < 1) ||
		(p_out_UnpackedHeader->issueMonth > 12) ||
		(p_out_UnpackedHeader->validMonth < 1) ||
		(p_out_UnpackedHeader->validMonth > 12) ||
		(p_out_UnpackedHeader->issueDayOfMonth < 1) ||
		(p_out_UnpackedHeader->issueDayOfMonth > 31) ||
		(p_out_UnpackedHeader->validDayOfMonth < 1) ||
		(p_out_UnpackedHeader->validDayOfMonth > 31) ||
		(p_out_UnpackedHeader->issueHour > 23) ||
		(p_out_UnpackedHeader->validHour > 23) ||
		(p_out_UnpackedHeader->issueMinute > 59) ||
		(p_out_UnpackedHeader->validMinute > 59) ||
		(p_out_UnpackedHeader->maxLatitude > 9000) ||
		(p_out_UnpackedHeader->maxLatitude < -9000) ||
		(p_out_UnpackedHeader->minLatitude > 9000) ||
		(p_out_UnpackedHeader->minLatitude < -9000) ||
		(p_out_UnpackedHeader->maxLongitude > 18000) ||
		(p_out_UnpackedHeader->maxLongitude < -18000) ||
		(p_out_UnpackedHeader->minLongitude > 18000) ||
		(p_out_UnpackedHeader->minLongitude < -18000)
	)
	{
		return MWRes_ProductHeaderFieldInvalid;
	}

	/*
	//##############################################################
	// Done
	//##############################################################
	*/

	return MWRes_Success;

}

/*
//====================================================================
// MWTransmogrifier_Internal_unpackGenericRasterProductHeader:
//
// Reads in the product header, unpacking it and returning it
// to the caller in the memory. This assumes that the headerType
// single bit field has already been read.
//
// PARAMETERS:
//
//	numBytesInProduct:
//		The number of bytes in the product stream
//
//	p_bytesOfProduct:
//		Array holding the entire product stream
//
//	inout_p_numBitsConsumed:
//		Pointer to a counter of the number of bits consumed from
//		the product stream before this call.
//		This count is updated by this method in the counter
//		variable.
//
//	p_out_UnpackedHeader:
//		Pointer to a TMW_GenericRasterProductHeader_Output structure
//		that is filled from the bitpacked version in the data stream.
//
//	p_fixedScaleSpecifiers:
//		Pointer to an array of 9 fixed scale specifiers.  A raster
//		can have up to MAX_PLANES_IN_RASTER planes.
//		Each plane has a fixed scale specifier
//		which specifies a fixed scale encoding of the data values.
//		For each plane specified in the numPlanes header field which this
//		method reads, the fixed scale specifier will be read into this
//		array.  Any further fixedScaleSpecifiers in the array
//		are not touched.
//
// SIDE EFFECTS:
//
//	Parses past the end of the header in the data stream
//
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_ProductHeaderFieldInvalid:
//		If field values are outside the expected ranges
//
//====================================================================
*/

TMW_ResultCode MWTransmogrifier_Internal_unpackGenericRasterProductHeader
(
	TFourByteInteger_Unsigned numBytesInProduct,
	TByte_Unsigned* p_bytesOfProduct,
	TFourByteInteger_Unsigned* inout_p_numBitsConsumed,
	TMW_GenericRasterProductHeader_Output* p_out_UnpackedHeader,
	TMW_GenericRasterProductHeader_FixedScaleSpecifier p_fixedScaleSpecifiers[MAX_PLANES_IN_RASTER]
)
{
	TMW_ResultCode MWRes;
	TFourByteInteger_Unsigned numBitsInProduct;
	TByte_Unsigned planeIndex;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if ((p_out_UnpackedHeader == NULL) || (p_bytesOfProduct == NULL))
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Call GenericProductHeader version, as it starts the same
	//##############################################################
	*/
	MWRes = MWTransmogrifier_Internal_unpackGenericProductHeader
	(
		numBytesInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed,
		(TMW_GenericProductHeader_Output*) p_out_UnpackedHeader
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Change header type is known
	//##############################################################
	*/
	p_out_UnpackedHeader->headerType = GenericRaster_ProductHeaderType;
	numBitsInProduct = numBytesInProduct * 8;

	/*
	//##############################################################
	// Read in the raster specific fields at the end
	//##############################################################
	*/

	/* Number of rows */
	p_out_UnpackedHeader->numRows = 0;
	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->numRows),
		GPRH_NUMROWS_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/* Number of columns */
	p_out_UnpackedHeader->numColumns = 0;
	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->numColumns),
		GPRH_NUMCOLUMNS_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/* Bits per pixel */
	p_out_UnpackedHeader->bitsPerPixel = 0;
	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->bitsPerPixel),
		GPRH_BITSPERPIXEL_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/* Bit field encodes value -1 */
	p_out_UnpackedHeader->bitsPerPixel += 1;


	/* number of planes */
	p_out_UnpackedHeader->numPlanes = 0;
	MWRes = readBitField
	(
		(TByte_Unsigned*) &(p_out_UnpackedHeader->numPlanes),
		GPRH_NUMPLANES_NUMBITS,
		numBitsInProduct,
		p_bytesOfProduct,
		inout_p_numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/* Bit field encodes value -1 */
	p_out_UnpackedHeader->numPlanes += 1;

	/* Due to bit length, this test isn't strictly necessary, but just incase...*/
	if (p_out_UnpackedHeader->numPlanes > MAX_PLANES_IN_RASTER)
	{
		return MWRes_ProductHeaderFieldInvalid;
	}

	/* Precision Specifiers for each plane */
	for (planeIndex = 0; planeIndex < p_out_UnpackedHeader->numPlanes; planeIndex ++)
	{
		/* Value Precision */
		p_fixedScaleSpecifiers[planeIndex].precisionSpecifier = 0;
		MWRes = readBitField
		(
			(TByte_Unsigned*) &(p_fixedScaleSpecifiers[planeIndex].precisionSpecifier),
			GPRH_PRECISION_NUMBITS,
			numBitsInProduct,
			p_bytesOfProduct,
			inout_p_numBitsConsumed
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/* Value Offset */
		p_fixedScaleSpecifiers[planeIndex].offsetSpecifier = 0;
		MWRes = readBitField
		(
			(TByte_Unsigned*) &(p_fixedScaleSpecifiers[planeIndex].offsetSpecifier),
			GPRH_OFFSET_NUMBITS,
			numBitsInProduct,
			p_bytesOfProduct,
			inout_p_numBitsConsumed
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}
	}

	/*
	//##############################################################
	// Test values
	//##############################################################
	*/
	if
	(
		(p_out_UnpackedHeader->bitsPerPixel > 8) ||
		(p_out_UnpackedHeader->bitsPerPixel < 1) ||
		(p_out_UnpackedHeader->numPlanes > 32) ||
		(p_out_UnpackedHeader->numPlanes < 1)
	)
	{
		/* Error */
		return MWRes_ProductHeaderFieldInvalid;
	}


	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;


}

/*
//====================================================================
// MWTransmogrifier_processBodyBlock:
//
//
// This method transforms a particular body block.
//
// PARAMETERS:
//
//	numBytesInProduct:
//		The number of bytes in the entire product
//
//	p_bytesOfProduct:
//		The bytes of the entire product
//
//	p_inout_numBytesConsumed:
//		A pointer to a counter of the number of bytes from the
//		product that have been processed. On input, this should
//		be the number of bytes consumed, the body block is
//		expected to begin with the very next byte.  This method
//		updates the counter to reflect the body block bytes
//		consumed.
//
//	numColumns:
//		If it is a raster product, the number of columns in the
//		raster.  Unused otherwise.
//
//	numRows:
//		If it is a raster product, the number of rows in the
//		raster.  Unused otherwise.
//
//	numBitsPerPixel:
//		If it is a raster product, the number of bits in each
//		pixel.
//
//	bool_isLastBodyBlock:
//		Set to 1 if this is the last body block in the product.
//		Otherwise set to 0.
//
// RETURNS:
//
//	MWRes_Success:
//
//
//	MWRes_UnrecognizedTransmogrificationType:
//		If the input format to output format transformation
//		specified in the body block header is not one that this
//		software was designed to perform.
//
//	MWRes_ProductFormatInvalid:
//		If the product decoding failed due to some sort of
//		malformation in the input data.
//
//	MWRes_ProductNotExpectedSize:
//		If the decoding failed because the output of the
//		decoder was not the expected size.
//
//	MWRes_CommunicationsError:
//		If an error occurred in the output of the
//		encoder over the final pipe.
//
//	MWRes_ProductHeaderFieldInvalid:
//		If an internal header in the body block contained
//		an invalid field value.
//
//====================================================================
*/

TMW_ResultCode MWTransmogrifier_processBodyBlock
(
	TFourByteInteger_Unsigned numBytesInProduct,
	TByte_Unsigned* p_bytesOfProduct,
	TFourByteInteger_Unsigned* p_inout_numBytesConsumed,
	TTwoByteInteger_Unsigned numColumns,
	TTwoByteInteger_Unsigned numRows,
	TByte_Unsigned numBitsPerPixel,
	TByte_Unsigned bool_isLastBodyBlock
)
{
	TMW_ResultCode MWRes;

	TByte_Unsigned inputCompressionFlag;
	TByte_Unsigned outputCompressionFlag;
	TMW_CompressionType decodeType;

	TFourByteInteger_Unsigned numExpectedBytesInBodyBlock;
	TFourByteInteger_Unsigned numBytesConsumed;
	TFourByteInteger_Unsigned numBytesConsumedInBodyBlock;
	TFourByteInteger_Unsigned numBitsConsumedInBodyBlock;

	/* Is there enough room for another block */
	if (numBytesInProduct < (*p_inout_numBytesConsumed + 5))
	{
		return MWRes_ProductNotExpectedSize;
	}

	/*
	//##############################################################
	// Read the input and output compression flags.
	// Each is 4 bits.
	// Input is in MSB bits, output is in LSB bits
	//##############################################################
	*/

	numBytesConsumed = *p_inout_numBytesConsumed;

	inputCompressionFlag = (*(p_bytesOfProduct + numBytesConsumed)) & 0xF0;
	inputCompressionFlag >>= 4;


	outputCompressionFlag = (*(p_bytesOfProduct + numBytesConsumed)) & 0x0F;

	/* I/O compression format flags consumed */
	numBytesConsumed += 1;

    // Write value byte-wise since pointer arithmetic does
    // not guarantee that all CPU platforms can actually access
    // words on non-word aligned memory addresses.

#if 0
    // Don't do this!
    /* Four byte length of block in INTEL LOHI order */
    numExpectedBytesInBodyBlock = *((TFourByteInteger_Unsigned*) (p_bytesOfProduct + numBytesConsumed));
#else
    // Do this instead...
    {
        /* Four byte length of block in INTEL LOHI order */
        TByte_Unsigned* tSrc =
            p_bytesOfProduct + numBytesConsumed;
        numExpectedBytesInBodyBlock   =
            (TFourByteInteger_Unsigned)*tSrc++;
        numExpectedBytesInBodyBlock  +=
            (((TFourByteInteger_Unsigned)*tSrc++) << 8);
        numExpectedBytesInBodyBlock  +=
            (((TFourByteInteger_Unsigned)*tSrc++) << 16);
        numExpectedBytesInBodyBlock  +=
            (((TFourByteInteger_Unsigned)*tSrc++) << 24);
    }
#endif

	numBytesConsumed += 4;

	if (numExpectedBytesInBodyBlock > (numBytesInProduct - numBytesConsumed))
	{
		/* Not enough bytes to hold this block in this product*/
		/* Something wrong with it */
		*p_inout_numBytesConsumed = numBytesConsumed;
		return MWRes_ProductNotExpectedSize;
	}



	/*
	//##############################################################
	// Determine input Decoder and output Encoder that will be used.
	//
	// There is a large matrix of possibilities. However, only
	// certain transforms are supported. See the Testing document
	// for the matrix of supported conversions.
	//##############################################################
	*/

	/*
	// In the simplest case, the input and output formats are the same,
	// so no encoding or decoding is necessary.
	*/
	if (inputCompressionFlag == outputCompressionFlag)
	{
		/* Pass-through */

		/* No decode needed */
		decodeType = None_MWCompressionType;

		/* No encode needed */
		MWRes = MWOutputEncoder_setOutputEncoder
		(
			None_MWCompressionType
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}
	}
	else if (inputCompressionFlag == DEFLATED_MWCompressionType)
	{
		/*
		// We will undo the deflate, and then apply whatever
		// output encoder transforms into the specified output
		// format.
		*/
		decodeType = DEFLATED_MWCompressionType;

		/* Put into whatever output format is */
		MWRes = MWOutputEncoder_setOutputEncoder
		(
			outputCompressionFlag
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}
	}
	else if (inputCompressionFlag == WSI_RLE_DEFLATE_MWCompressionType)
	{

		/* Handle output*/
		if (outputCompressionFlag == WSI_RLE_MWCompressionType)
		{
			/* Remove the deflate */
			decodeType = DEFLATED_MWCompressionType;

			// No encoding needed after deflate removed
			MWRes = MWOutputEncoder_setOutputEncoder
			(
				None_MWCompressionType
			);
			if (MWRes != MWRes_Success)
			{
				return MWRes;
			}

		}
		else
		{
			/*
			// Must undo the DEFLATE and then the RLE
			// At that point it is unencoded and we apply the output
			// method.
			*/
			decodeType = WSI_RLE_DEFLATE_MWCompressionType;

			/* Set output encoder */
			MWRes = MWOutputEncoder_setOutputEncoder
			(
				outputCompressionFlag
			);
			if (MWRes != MWRes_Success)
			{
				return MWRes;
			}

		}

	}
	else if (inputCompressionFlag == Downsampled_WSI_RLE_DEFLATE_MWCompressionType)
	{
		/*
		//  Remove the downsampled, WSI RLEd, then Defalted compression and
		//  apply whatever the output compression is.
		*/
		decodeType = Downsampled_WSI_RLE_DEFLATE_MWCompressionType;

		/* Set output encoder */
		MWRes = MWOutputEncoder_setOutputEncoder
		(
			outputCompressionFlag
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}


	}
	else if (inputCompressionFlag == WeatherHuffman_MWCompressionType)
	{


		/*
		// Remove the weather huffman and then encode in whatever
		// the output is
		*/
		decodeType = WeatherHuffman_MWCompressionType;

		MWRes = MWOutputEncoder_setOutputEncoder
		(
			outputCompressionFlag
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}


	}
	else if (inputCompressionFlag == WSI_RLE_MWCompressionType)
	{
		/*
		// Remove the RLE and add whatever the encoding is.
		*/
		decodeType = WSI_RLE_MWCompressionType;

		MWRes = MWOutputEncoder_setOutputEncoder
		(
			outputCompressionFlag
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

	}
	else
	{
		/* We don't know how to do that */
		return MWRes_UnrecognizedTransmogrificationType;
	}

	/*
	//##############################################################
	// The encoder is configured.  Pass the source bytes to
	// the proper decoder.
	//
	// The decoders automatically pass the data to the output
	// encoder ADT which sends the data to its configured target
	// (which we set above using MWOutputEncoder_setOutputEncoder)
	//##############################################################
	*/
	if (decodeType == None_MWCompressionType)
	{
		/*
		// Output the product block
		*/

		MWRes = MWOutputDPUEncoder_outputBytes
		(
			numExpectedBytesInBodyBlock,
			p_bytesOfProduct + numBytesConsumed,
			bool_isLastBodyBlock
		);

		/* Always consumes number of bytes we tell it to */
		numBytesConsumed += numExpectedBytesInBodyBlock;

		/* Update total in callers inout parameter */
		*p_inout_numBytesConsumed = numBytesConsumed;

		/* Done */
		return MWRes;

	}
	else if (decodeType == DEFLATED_MWCompressionType)
	{
		numBytesConsumedInBodyBlock = 0;

		MWRes = MWDEFLATECODEC_inflateProductBodyBlock
		(
			numExpectedBytesInBodyBlock,
			p_bytesOfProduct + numBytesConsumed,
			bool_isLastBodyBlock,
			&numBytesConsumedInBodyBlock
		);

		/* Add bytes consumed from this block to total*/
		numBytesConsumed += numBytesConsumedInBodyBlock;

		/* Update total in callers inout parameter */
		*p_inout_numBytesConsumed = numBytesConsumed;

		/* Done */
		return MWRes;

	}
	else if (decodeType == WeatherHuffman_MWCompressionType)
	{
		/*
		// WxHuffman processor requires at least 4 bytes that indicate
		// wx encoding size.
		*/
		if ((numBytesInProduct - numBytesConsumed) < 4)
		{
			return MWRes_ProductNotExpectedSize;
		}

		/* Perform transformation using WxHuffmanProcessor ADT */
		numBitsConsumedInBodyBlock = 0;
		MWRes = MWWxHuffmanProcessor_transformFormat
		(
			p_bytesOfProduct + numBytesConsumed,
			bool_isLastBodyBlock,
			numColumns,
			numRows,
			numBitsPerPixel,
			&numBitsConsumedInBodyBlock
		);

		/* Product body block should have ended on a byte boundary */
		if ((numBitsConsumedInBodyBlock %8) != 0)
		{
			return MWRes_ProductContentsInvalid;
		}

		/* Add number of bytes consumed to total */
		numBytesConsumed += (numBitsConsumedInBodyBlock / 8);

		/* Update total in callers inout parameter */
		*p_inout_numBytesConsumed = numBytesConsumed;

		return MWRes;

	}
	else if (decodeType == WSI_RLE_MWCompressionType)
	{
		/* We need to remove WSI RLE */
		MWRes = MWWSIRLE_decompressProductBodyBlock
		(
			numExpectedBytesInBodyBlock,
			p_bytesOfProduct + numBytesConsumed,
			&numBytesConsumedInBodyBlock,
			bool_isLastBodyBlock
		);

		/* Add bytes consumed from this block to total*/
		numBytesConsumed += numBytesConsumedInBodyBlock;

		/* Update total in callers inout parameter */
		*p_inout_numBytesConsumed = numBytesConsumed;


		/* Done */
		return MWRes;

	}
	else if (decodeType == WSI_RLE_DEFLATE_MWCompressionType)
	{
		numBytesConsumedInBodyBlock = 0;

		MWRes = MWDEFLATECODEC_inflateWSIRLEProductBodyBlock
		(
			numExpectedBytesInBodyBlock,
			p_bytesOfProduct + numBytesConsumed,
			bool_isLastBodyBlock,
			&numBytesConsumedInBodyBlock
		);

		/* Add bytes consumed from this block to total*/
		numBytesConsumed += numBytesConsumedInBodyBlock;

		/* Update total in callers inout parameter */
		*p_inout_numBytesConsumed = numBytesConsumed;

		/* Done */
		return MWRes;


	}
	else if (decodeType == Downsampled_WSI_RLE_DEFLATE_MWCompressionType)
	{
		numBytesConsumedInBodyBlock = 0;

		MWRes = MWDDRLE_transformFormat
		(
			numExpectedBytesInBodyBlock,
			p_bytesOfProduct + numBytesConsumed,
			bool_isLastBodyBlock,
			numColumns,
			numRows,
			numBitsPerPixel,
			&numBytesConsumedInBodyBlock
		);

		/* Add bytes consumed from this block to total*/
		numBytesConsumed += numBytesConsumedInBodyBlock;

		/* Update total in callers inout parameter */
		*p_inout_numBytesConsumed = numBytesConsumed;

		/* Done */
		return MWRes;
	}

	/*
	//##############################################################
	// We don't know how to handle this pair
	//##############################################################
	*/
	return MWRes_UnrecognizedTransmogrificationType;


}

/*
//====================================================================
//====================================================================
// Public methods
//====================================================================
//====================================================================
*/

TMW_ResultCode MWTransmogrifier_setOutputCallback
(
	TWSIMW_OutputFunction* p_outputFunction
)
{
	/* Set callback as output function of output encoder */
	return MWOutputDPUEncoder_setOutputFunction
	(
		p_outputFunction
	);
}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWTransmogrifier_processProduct
(
	TFourByteInteger_Unsigned numBytesInProduct,
	TByte_Unsigned* p_bytesOfProduct
)
{
	TMW_ResultCode MWRes;
	TByte_Unsigned headerType;
	TByte_Unsigned numBodyBlocks;
	TByte_Unsigned bodyBlockIndex;
	TFourByteInteger_Unsigned numBytesConsumed;
	TByte_Unsigned bool_isLastBodyBlock;

	TFourByteInteger_Unsigned numBitsConsumed;

	/* For handling raster products, unused for others */
	TTwoByteInteger_Unsigned numRasterColumns=0;
	TTwoByteInteger_Unsigned numRasterRows=0;
	TByte_Unsigned numRasterBitsPerPixel=0;

	TMW_GenericRasterProductHeader_Output genericRasterProductHeader;
	TMW_GenericProductHeader_Output genericProductHeader;
	TMW_GenericRasterProductHeader_FixedScaleSpecifier output_fixedScaleSpecifiers[MAX_PLANES_IN_RASTER];

#ifdef TEST_PARAMETERS
	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if (p_bytesOfProduct == NULL)
	{
		return MWRes_NullPointer;
	}
#endif

	/*
	//##############################################################
	// Read in the header type
	//##############################################################
	*/
	numBitsConsumed = 0;
	headerType = 0;
	MWRes = readBitField
	(
		&headerType,
		GPH_HEADERTYPE_NUMBITS,
		numBytesInProduct,
		p_bytesOfProduct,
		&numBitsConsumed
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Read the rest of the header
	//##############################################################
	*/

	if (headerType == GenericRaster_ProductHeaderType)
	{
		/*
		//##############################################################
		// Read in the header
		//##############################################################
		*/

		MWRes = MWTransmogrifier_Internal_unpackGenericRasterProductHeader
		(
			numBytesInProduct,
			p_bytesOfProduct,
			&numBitsConsumed,
			&genericRasterProductHeader,
			output_fixedScaleSpecifiers
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}


		/*
		//##############################################################
		// Starting new product
		//##############################################################
		*/
		MWRes = MWOutputEncoder_startNewProduct
		(
			genericRasterProductHeader.productID
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}



		/*
		//##############################################################
		// Get fields we will need to process the body blocks
		//##############################################################
		*/

		numRasterColumns = genericRasterProductHeader.numColumns;
		numRasterRows = genericRasterProductHeader.numRows;

		numRasterBitsPerPixel = genericRasterProductHeader.bitsPerPixel;

		// One body block per plane
		numBodyBlocks = genericRasterProductHeader.numPlanes;

		/*
		//##############################################################
		// Pass unpacked header to output
		//##############################################################
		*/
		MWRes = MWOutputEncoder_outputUnencodedBytes
		(
			sizeof (TMW_GenericRasterProductHeader_Output),
			(TByte_Unsigned*) (&genericRasterProductHeader),
			0 /* Not end of product instance */
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/*
		//##############################################################
		// Pass unpacked fixed scale specifiers
		//##############################################################
		*/
		MWRes = MWOutputEncoder_outputUnencodedBytes
		(
			(genericRasterProductHeader.numPlanes) * sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier) ,
			(TByte_Unsigned*) (output_fixedScaleSpecifiers),
			0 /* Not end of product instance */
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

	}
	else
	{
		/*
		//##############################################################
		// Read in the header
		//##############################################################
		*/

		MWRes = MWTransmogrifier_Internal_unpackGenericProductHeader
		(
			numBytesInProduct,
			p_bytesOfProduct,
			&numBitsConsumed,
			&genericProductHeader
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/* Generic products have one body block */
		numBodyBlocks = 1;

		/*
		//##############################################################
		// Starting new product
		//##############################################################
		*/
		MWRes = MWOutputEncoder_startNewProduct
		(
			genericProductHeader.productID
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}


		/*
		//##############################################################
		// Pass unpacked header to output
		//##############################################################
		*/
		MWRes = MWOutputEncoder_outputUnencodedBytes
		(
			sizeof (TMW_GenericProductHeader_Output),
			(TByte_Unsigned*) (&genericProductHeader),
			0 /* Not end of product instance */
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

	}

	/*
	//##############################################################
	// Byte alignment after header
	//##############################################################
	*/
	numBytesConsumed = numBitsConsumed / 8;
	if ((numBitsConsumed % 8) != 0)
	{
		numBytesConsumed ++;
	}

	/*
	//##############################################################
	// Process each body block
	//##############################################################
	*/

	/* Not true until last block */
	bool_isLastBodyBlock = 0;

	for (bodyBlockIndex = 0; bodyBlockIndex < numBodyBlocks; bodyBlockIndex ++)
	{
		/* Is it the last block in the product? */
		if (bodyBlockIndex == numBodyBlocks-1)
		{
			bool_isLastBodyBlock = 1;
		}


		/* Transmogrify the block */
		MWRes = MWTransmogrifier_processBodyBlock
		(
			numBytesInProduct,
			p_bytesOfProduct,
			&numBytesConsumed,
			numRasterColumns,
			numRasterRows,
			numRasterBitsPerPixel,
			bool_isLastBodyBlock
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}
	}

	/*
	//##############################################################
	// santity check
	//##############################################################
	*/
	if (numBytesConsumed != numBytesInProduct)
	{
		return MWRes_ProductNotExpectedSize;
	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes;

}
