/*
====================================================================
// MW_OutputEncoder.h:
//
// This singleton ADT pipes data from the various decoders to
// the configured encoder.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
====================================================================
*/

#ifndef MW_OUTPUTENCODER_HEADER
#define MW_OUTPUTENCODER_HEADER

/*
//====================================================================
// Required includes
//====================================================================
*/
#include "MW_DataFieldTypes.h"
#include "MW_ProductHeaders.h"

/*
//====================================================================
// MWOutputEncoder_startNewProduct:
//
// Call this before MWOutputEncoder_setOutputFormat when a new
// product is started.
//
// PARAMETERS:
//
//	productID:
//		The product ID of the product being started
//
// RETURNS:
//
//		MWRes_Success
//
//		or Error Result from encoder
//====================================================================
*/
TMW_ResultCode MWOutputEncoder_startNewProduct
(
	TTwoByteInteger_Unsigned productID
);

/*
//====================================================================
// MWOutputEncoder_setOutputFormat:
//
// This sets the encoder through which the output is to be passed
// before going to the DPU framing ADT.
//
// Call this method before the decompressor object begins it work,
// as it will call the other public method of this ADT in order
// to pass it bytes, and this  method is what tells this ADT
// what to do with them.
//
// PARAMETERS:
//
//	outputFormat:
//		The format into which the output should be transformed.
//
// RETURNS:
//
//	MWRES_Success
//
//	MWRes_UnrecognizedTransmogrificationType
//		If the output format is not supported (there is no encoder
//		for that format)
//
//====================================================================
*/
TMW_ResultCode MWOutputEncoder_setOutputEncoder
(
	TMW_CompressionType outputFormat
);



/*
//====================================================================
// MWOutputEncoder_startNewEncodingBlock:
//
// Call this whenever the a new block of encoded format starts.
//
// MWOutputEncoder_outputUnecodedBytes or
// MWOutputEncoder_outputEncodedBytes.
//
// RETURNS:
//
//	MWRes_Success
//
//	or Error Result from encoder
//====================================================================
*/
TMW_ResultCode MWOutputEncoder_startNewEncodingBlock();


/*
//====================================================================
// MWOutputEncoder_outputUnecodedBytes:
//
// This method hands bytes to this ADT to be routed directly to
// the DPU output frame formatter without using the currently
// configured Output Format Encoder.
//
// PARAMETERS:
//
//	numBytes:
//		The number of bytes being given
//
//	p_bytes:
//		The bytes being given
//
//	bool_isEndOfProduct:
//		Set to 1 if this is the last group of bytes in the product.
//			This signals the DPU framer to flush its buffers after
//			processing the last byte. This also terminates
//			the current output frame and sets the last frame flag
//			to true.
//
//		Set to 0 otherwise
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_NullPointer:
//		if p_bytes is NULL
//
//	Other errors returned by the encoders
//====================================================================
*/
TMW_ResultCode MWOutputEncoder_outputUnencodedBytes
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TByte_Unsigned bool_isEndOfProduct
);

/*
//====================================================================
// MWOutputEncoder_outputEncodedBytes:
//
// This method hands bytes to this ADT to be routed to the currently
// set output encoder.
//
// PARAMETERS:
//
//	numBytes:
//		The number of bytes being given
//
//	p_bytes:
//		The bytes being given
//
//	bool_isEndOfEncodingBlock:
//		Set to 1 if this is the end of a particular encoding
//		block.
//
//		Set to 0 otherwise
//
//		Note, this cannot be 0 if bool_isEndOfProduct is not 0
//
//	bool_isEndOfProduct:
//		Set to 1 if this is the last group of bytes in the product.
//			This signals the encoder to flush its buffers after
//			processing the last byte. This also terminates
//			the current output frame and sets the last frame flag
//			to true.
//
//		Set to 0 otherwise
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_BadParamValue:
//		If bool_isEndOfProduct is != 0 but bool_isEndOfEncdingBlock is 0
//
//	MWRes_NullPointer:
//		if p_bytes is NULL
//
//	Other errors returned by the encoders
//====================================================================
*/
TMW_ResultCode MWOutputEncoder_outputEncodedBytes
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TByte_Unsigned bool_isEndOfEncodingBlock,
	TByte_Unsigned bool_isEndOfProduct
);


#endif

