/*
====================================================================
// MW_OutputDPUEncoder.h:
//
// This is the public interface definition for an ADT
// (Abstract Data Type) module which performs encoding
// of output DPUs (data payload units) inside of Output Frames.
//
// The ADT exists as a single instance.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
====================================================================
*/
#ifndef MW_OUTPUTDPU_INTERFACE
#define MW_OUTPUTDPU_INTERFACE

/*
====================================================================
// Required includes
====================================================================
*/
#include "MW_DataFieldTypes.h"


/*
====================================================================
// TMWOutputDPUEncoder_OutputFunction:
//
// Format of the callback accepted by the output encoder for its
// output pipe callback.
//
// The output bytes will have been formatted by the output encoder
// in the frame format specified in the receiver output format
// section of the WSI for Sirius receiver system specification.
//
// PARAMETERS:
//
// The first parameter holds the number of bytes to be output.
//
// The second parameter points to the contiguous array of bytes
// to be output.
//
// RETURNS:
//
// If the callback returns anything other than MWRes_Success,
// then the pipe output is considered to have been a failure and
// the return code is returned out of the output system to its
// callers. This effectively cancels processing on the current
// product.
//
//
====================================================================
*/
typedef TMW_ResultCode (TMWOutputDPUEncoder_OutputFunction) (TFourByteInteger_Unsigned,
                                                             TByte_Unsigned*);

/*
====================================================================
// MWOutputDPUEncoder_setOutputFunction:
//
// This method hands the output encoder a pointer to a function
// that it can use to pass framed bytes to an output pipe.
//
// This must be called succesfully at least once before
// MWOutputDPUEncoder_startNewProduct or a calling order error
// will result.
//
//
// PARAMETERS:
//
//	p_outputFunction:
//		Pass in the address of a function of the declared format
//		which accepts bytes to be written out over the output pipe.
//
//		The bytes passed to the function will be in the frame
//		format specified for output of WSI products from the
//		Sirius receiver.
//
// RETURNS:
//
//	MWRes_Success:
//		If the output function has been set.
//
//	MWRes_NullPointer:
//		If the function pointer was NULL
//
====================================================================
*/
TMW_ResultCode MWOutputDPUEncoder_setOutputFunction
(
	TMWOutputDPUEncoder_OutputFunction* p_outputFunction
);

/*
====================================================================
// MWOutputDPUEncoder_startNewProduct:
//
// This method should be called before beginnning encoding the first
// DPU of a product.
//
// Note: If a product ends prematurely due to an error, just
// call this method to start the new product.  The product instance ID
// will change, and the DPU sequence will reset, both of which indicate
// to the recipient that there are missing DPUs.
//
// PARAMETERS:
//
//	productID:
//		The ID number of the type of product being started.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_BadCallingOrder:
//		If MWOutputDPUEncoder_setOutputCallback has not yet been
//		successfully called.
//
====================================================================
*/
TMW_ResultCode MWOutputDPUEncoder_startNewProduct
(
	TTwoByteInteger_Unsigned productID
);

/*
====================================================================
// MWOutputDPUEncoder_outputBytes:
//
// Call this method any number of times after a single call to
// MWOutputDPUEncoder_startNewProduct to encode the product into DPUs.
//
// The last call during the product should set the bool_endOfProductInstance
// flag to 1.  For all other calls, the flag should be set to 0.
//
// PARAMETERS:
//
//	numBytes:
//		The number of bytes to output from the callers p_outputBytes parameter
//		You can pass in 0 as the value, which is particularly useful
//		in combination with the bool_endOfProductInstance to mark the
//		end of the product.
//
//	p_outputBytes:
//		Pointer to an array of bytes at least numBytesLong.  The first byte
//		should be the first byte intended for output this call.
//
//	bool_endOfProductInstance:
//		Set this to 0 if the bytes given are not the last in the product instance.
//		Set this to non-zero if the bytes given are the last in the product instance.
//
//		If you are uncertain, set this to 0.  The frame containing
//		the last byte passed in via this method is never sent
//		until the next call to this method unless bool_endOfProductInstance
//		is non-0.  Thus, you can safely wait and call this
//		method again with numBytes = 0 and bool_endOfProductInstance = 1
//		if the last call turned out to be the end of the product.
//
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_NullPointer:
//
//		If one of the pointer parameters is NULL
====================================================================
*/
TMW_ResultCode MWOutputDPUEncoder_outputBytes
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_outputBytes,
	TByte_Unsigned bool_endOfProductInstance
);



/* end header wrapper */
#endif
