/*
//====================================================================
// MW_OutputDPUEncoder.c:
//
// This holds the implementation of the Output DPU encoder ADT.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
//====================================================================
*/

/*
//====================================================================
// includes
//====================================================================
*/

#include "MW_OutputDPUEncoder.h"

/*
//====================================================================
// DPU header structure definition
//====================================================================
*/

#pragma pack (push,1)
typedef struct
{
	TByte_Unsigned productID_LOWByte;
	TByte_Unsigned productID_HIGHByte;

	TByte_Unsigned instanceID_LOWByte;
	TByte_Unsigned instanceID_HIGHByte;

	TByte_Unsigned DPUSequenceNumber_LOWByte;
	TByte_Unsigned DPUSequenceNumber_HIGHByte;

	TByte_Unsigned finalDPUThisProductInstance; // 0 is not, 1 is true

} TWSI_Output_DPUHeader;

#pragma pack(pop)

/*
//====================================================================
// Defined constants
//====================================================================
*/

/* The flag byte which borders frames */
#define FRAME_BORDER_FLAG_BYTE 0x7E

/* The flag byte which signals a transparency code */
#define FRAME_TRANSPARENCY_FLAG_BYTE 0x7D

/* The maximum number of bytes allowed in a single frame */
#define MAX_OUTPUT_FRAME_LENGTH 4096

/* The initial value of the CRC before any bytes are run through*/
#define INITIAL_CRC_VALUE 0xFFFF

/*
//====================================================================
// Internal ADT variables
//====================================================================
*/

/* The buffer used for storing a DPU */
static TByte_Unsigned p_frameBuffer[MAX_OUTPUT_FRAME_LENGTH];

/* The number of bytes in the DPU buffer */
static TFourByteInteger_Unsigned numBytesInFrameBuffer = 0;

/* The index of this DPU in the product */
static TTwoByteInteger_Unsigned DPUIndex = 0;

/* The product instance */
static TTwoByteInteger_Unsigned productInstanceCount = 65525; // so will start at 0

/* The CRC value*/
static TTwoByteInteger_Unsigned CRCValue = INITIAL_CRC_VALUE;

/* The DPU header used during framing */
static TWSI_Output_DPUHeader pduHeader;

/* The CRC table */
static TTwoByteInteger_Unsigned crc_table[256];

/* Is the CRC table initialized? */
static TByte_Unsigned bool_isCRCTableInitialized = 0;


/*
// The output callback method that is used to output the DPU encoding
// results over the caller's pipe
*/
TMWOutputDPUEncoder_OutputFunction* p_MWOutputDPUEncoder_outputCallback = NULL;

/*
//====================================================================
//====================================================================
// Internal methods
//====================================================================
//====================================================================
*/

/*
//====================================================================
// MWOutputDPUEncoder_Internal_sendFrameOverLink:
//
//
//====================================================================
*/

TMW_ResultCode MWOutputDPUEncoder_setOutputFunction
(
	TMWOutputDPUEncoder_OutputFunction* p_outputFunction
)
{
	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if (p_outputFunction == NULL)
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Set the output callback
	//##############################################################
	*/
	p_MWOutputDPUEncoder_outputCallback = p_outputFunction;

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;

}

/*
//====================================================================
// MWOutputDPUEncoder_Internal_sendFrameOverLink:
//
// This method is used to send a completed frame over the link.
// Its implementation is not yet built.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_OperationFailed:
//		On failure to make call to final output method
//
//  MWRes_BadCallingOrder:
//		If the frame buffer is empty
//====================================================================
*/

TMW_ResultCode MWOutputDPUEncoder_Internal_sendFrameOverLink()
{

	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// Test state
	//##############################################################
	*/
	if (p_MWOutputDPUEncoder_outputCallback == NULL)
	{
		return MWRes_BadCallingOrder;
	}

	if (numBytesInFrameBuffer == 0)
	{
		return MWRes_BadCallingOrder;
	}

	/*
	//##############################################################
	// Use callback
	//##############################################################
	*/
	MWRes = p_MWOutputDPUEncoder_outputCallback
	(
		numBytesInFrameBuffer,
		p_frameBuffer
	);

	/*
	//##############################################################
	// Buffer is now empty
	//##############################################################
	*/
	numBytesInFrameBuffer = 0;

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes;
}


/*
//====================================================================
// MWOutputDPUEncoder_Internal_initializeCRCTable:
//
// This method initializes the CRC table. It need only be called
// once.  Once this method returns, the CRC table is ready for use.
//
// SIDE EFFECTS:
//
//	Modifies the following variables:
//		bool_isCRCTableInitialized
//
// RETURNS:
//
//	nothing
//====================================================================
*/

void MWOutputDPUEncoder_Internal_initializeCRCTable()
{
	/*
	//##############################################################
	// Generate CRC table
	//##############################################################
	*/

	TTwoByteInteger_Unsigned c;
	TFourByteInteger_Signed n, k;
	for (n = 0; n < 256; n++)
	{
		c = (unsigned short) n<<8;
		for (k = 0; k < 8; k++)
		{
			if (c & 0x8000)
				c = 0x1021 ^ (c<<1);
			else
				c = c<<1;
		}
		crc_table[n] = c;
	}

	/*
	//##############################################################
	// Note that table is initialized
	//##############################################################
	*/

	/* Table is now initialized */
	bool_isCRCTableInitialized = 1;


	/*
	//##############################################################
	// Done
	//##############################################################
	*/
}


/*
//====================================================================
// MWOutputDPUEncoder_Internal_encodeFrameBodyBytes:
//
// This method copies bytes from the source, into the frame buffer
// while performing transparency encoding.
//
// The count of bytes in the frame buffer is bumped up for each
// byte written to it. Due to transparency encoding the number
// of bytes written to the frame buffer may be > numBytes.
//
// The CRC is updated for each byte added to the frame
// through this method.
//
// PARAMETERS:
//
//	numBytes:
//		The number of source bytes (not transparency encoded) to
//		be written to the frame buffer
//
//	p_bytes:
//		The bytes to be written.
//
//	p_out_numBytesWritten:
//		On return of MWRes_Success, indicates the number of bytes
//		which fit into the current frame.
//		The remainder of the bytes were not written to any frame yet.
//
//	bool_isCRC:
//		Set this to 1 only if this is the CRC at the end of the frame.
//		Set this to 0 otherwise.
//
// RETURNS:
//
//	MWRes_Success
//		If the data was encoded into the frame buffer
//
//====================================================================
*/

TMW_ResultCode MWOutputDPUEncoder_Internal_encodeFrameBodyBytes
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_out_numBytesWritten,
	TByte_Unsigned bool_isCRC
)
{
	TFourByteInteger_Unsigned byteIndex;

	/*
	//##############################################################
	// Transparency encode each byte
	//##############################################################
	*/

	for (byteIndex = 0; byteIndex < numBytes; byteIndex++)
	{
		/*
		//##############################################################
		// Encode this byte
		//##############################################################
		*/
		if
		(
			(p_bytes[byteIndex] == FRAME_BORDER_FLAG_BYTE) ||
			(p_bytes[byteIndex] == FRAME_TRANSPARENCY_FLAG_BYTE)
		)
		{
			/* Must have enough for 2 bytes, plus 2 (4 with possible transparency) byte CRC and 1 byte flag at end of frame */
			/* 2 + 4 + 1 = 7 if not the crc */
			if ( (bool_isCRC == 0) && (numBytesInFrameBuffer < MAX_OUTPUT_FRAME_LENGTH-7) )
			{
				/* Must do transprency encoding */
				p_frameBuffer[numBytesInFrameBuffer] = FRAME_TRANSPARENCY_FLAG_BYTE;
				p_frameBuffer[numBytesInFrameBuffer +1] = p_bytes[byteIndex] ^= 0x20;
				numBytesInFrameBuffer += 2;
			}
			else if ( (bool_isCRC != 0) && (numBytesInFrameBuffer < MAX_OUTPUT_FRAME_LENGTH-2))
			{
				/* If it is the CRC, then fail if we can't leave one byte at end for flag */
				/* That means that after adding our 2, we should still have at least 1 room left */
				p_frameBuffer[numBytesInFrameBuffer] = FRAME_TRANSPARENCY_FLAG_BYTE;
				p_frameBuffer[numBytesInFrameBuffer +1] = p_bytes[byteIndex] ^= 0x20;
				numBytesInFrameBuffer += 2;
			}
			else
			{
				break;
			}
		}
		else
		{
			/*
			// No transparency code this byte
			// Must have enough for 1 bytes, plus 2 (to 4 w/ trans encoding) byte CRC and 1 byte flag at end of frame
			*/
			if ( (bool_isCRC == 0) && (numBytesInFrameBuffer < MAX_OUTPUT_FRAME_LENGTH-5) )
			{
				p_frameBuffer[numBytesInFrameBuffer] = p_bytes[byteIndex];
				numBytesInFrameBuffer += 1;

			}
			else if ( (bool_isCRC != 0) && (numBytesInFrameBuffer < MAX_OUTPUT_FRAME_LENGTH-1) )
			{
				p_frameBuffer[numBytesInFrameBuffer] = p_bytes[byteIndex];
				numBytesInFrameBuffer += 1;
			}
			else
			{
				break;
			}
		}

	} /* Next input byte */

	/*
	//##############################################################
	// Note number of bytes written for caller
	//##############################################################
	*/

	*p_out_numBytesWritten = byteIndex;

	/*
	//##############################################################
	// Done
	//##############################################################
	*/

	return MWRes_Success;

}

/*
//====================================================================
// MWOutputDPUEncoder_Internal_startFrame:
//
// Call this method when a new frame should be started.
//
// SIDE EFFECTS:
//	Effects the following ADT variables:
//
//		numBytesInFrameBuffer
//		p_frameBuffer contents
//		CRCValue
//
// RETURNS:
//
//		MWRes_Success
//====================================================================
*/

TMW_ResultCode MWOutputDPUEncoder_Internal_startFrame()
{
	TMW_ResultCode MWRes;
	TFourByteInteger_Unsigned numBytesWrittenToFrame;

	/*
	//##############################################################
	// Frame starts at beginning of buffer with frame border flag
	// byte
	//##############################################################
	*/

	p_frameBuffer[0] = FRAME_BORDER_FLAG_BYTE;
	numBytesInFrameBuffer = 1;

	/* CRC is reset */
	CRCValue = INITIAL_CRC_VALUE;

	/*
	//##############################################################
	// If the crc table is not initialized, then we must initialize
	// it now
	//##############################################################
	*/
	if (bool_isCRCTableInitialized == 0)
	{
		MWOutputDPUEncoder_Internal_initializeCRCTable();
	}

	/*
	//##############################################################
	// Put frame index in DPU header
	//##############################################################
	*/
	pduHeader.DPUSequenceNumber_LOWByte = DPUIndex % 256;
	pduHeader.DPUSequenceNumber_HIGHByte = DPUIndex / 256;

	/*
	//##############################################################
	// Encode DPU header into frame
	//##############################################################
	*/

	numBytesWrittenToFrame = 0;

	MWRes = MWOutputDPUEncoder_Internal_encodeFrameBodyBytes
	(
		sizeof(pduHeader),
		(TByte_Unsigned*) &pduHeader,
		&numBytesWrittenToFrame,
		0 /* not the CRC */
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/* Buffer should be big enough to hold transparency encoded DPU header (13 bytes or so), or this will never work */
	if (numBytesWrittenToFrame != sizeof(pduHeader))
	{
		return MWRes_CodeFault;
	}

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;


}

/*
//====================================================================
// MWOutputDPUEncoder_Internal_endFrame:
//
// Call this when a frame body has been filled out.
//
// It calculates the CRC for the frame contents not including the frame
// border flags and appends the two byte CRC in LOW HIGH order to the
// end of the frame.  It then appends the ending frame border flag.
//
// PARAMETERS:
//
//	bool_isLastInProductInstance:
//		Pass in 0 if this is not the last frame in the product instance
//		Pass in 1 if it is
//====================================================================
*/
TMW_ResultCode MWOutputDPUEncoder_Internal_endFrame
(
	TByte_Unsigned bool_isLastInProductInstance
)
{
	TMW_ResultCode MWRes;

	/* Index for iterating frame buffer in CRC calculation */
	TFourByteInteger_Unsigned frameBufferIndex;

	/* For finding the "last in product instance" byte of DPU header */
	TFourByteInteger_Unsigned headerByteCount;

	/* For adding the CRC with transparency encoding */
	TFourByteInteger_Unsigned numBytesWritten;

	/* Value used to encode CRC value in LOHI byte order */
	TByte_Unsigned CRCTemp[2];

	/* This is set to 1 during CRC calculation if previous byte was a transparency flag*/
	TByte_Unsigned bool_CRCTransparencyOn = 0;

	/*
	//##############################################################
	// We must set the correct value of the DPU header to
	// indicate if it is the last DPU/frame in the product instance
	//##############################################################
	*/

	/*
	// It defaulted to off (0) when the frame was started, so we
	// need only set it to 1 if the parameter indicates it is true.
	*/

	if (bool_isLastInProductInstance != 0)
	{
		/*
		// Since it is a stand-alone byte with a value of 0 or 1, it
		// cannot affect the transparency encoding
		// So we can always just set the byte without rippling any changes
		// down the buffer.
		*/

		// Read past flag
		frameBufferIndex = 1;

		/* .. and  then first 6 bytes of header that may be transparency encoded */
		for (headerByteCount = 0; headerByteCount < 6; headerByteCount++)
		{
			if (p_frameBuffer[frameBufferIndex] == FRAME_TRANSPARENCY_FLAG_BYTE)
			{
				frameBufferIndex += 2;
			}
			else
			{
				frameBufferIndex += 1;
			}
		}

		/* Now set value at this index to 1, since it is the boolean "last DPU" flag */
		p_frameBuffer[frameBufferIndex] = 1;

	}

	/*
	//##############################################################
	// Calculate the CRC without transparency encoding.
	// We couldn't do this as we went, because we wern't sure if
	// the last frame in product flag was going to be 0 or 1 until
	// just now.
	//##############################################################
	*/
	CRCValue = INITIAL_CRC_VALUE;
	bool_CRCTransparencyOn = 0;
	for (frameBufferIndex = 1; frameBufferIndex < numBytesInFrameBuffer; frameBufferIndex ++)
	{
		if (p_frameBuffer[frameBufferIndex] == FRAME_TRANSPARENCY_FLAG_BYTE)
		{
			bool_CRCTransparencyOn = 1;
		}
		else if (bool_CRCTransparencyOn != 0)
		{
			CRCValue = (CRCValue<<8) ^ crc_table[(unsigned char)(p_frameBuffer[frameBufferIndex] ^0x20)
				^(unsigned char)(CRCValue>>8)];

			bool_CRCTransparencyOn = 0;
		}
		else
		{
			CRCValue = (CRCValue<<8) ^ crc_table[(unsigned char)(p_frameBuffer[frameBufferIndex])
				^(unsigned char)(CRCValue>>8)];
		}
	}

	/*
	//##############################################################
	// Append CRC at end in LOHI order. It too must be transparency
	// encoded.
	//##############################################################
	*/
	CRCTemp[0] = CRCValue % 256;
	CRCTemp[1] = CRCValue / 256;

	numBytesWritten = 0;
	MWRes = MWOutputDPUEncoder_Internal_encodeFrameBodyBytes
	(
		2,
		(TByte_Unsigned*) CRCTemp,
		&numBytesWritten,
		1 /* This is the CRC */
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	if (numBytesWritten < 2)
	{
		// Not enough room in frame for CRC
		return MWRes_CodeFault;
	}

	/*
	//##############################################################
	// Append a frame flag
	//##############################################################
	*/
	p_frameBuffer[numBytesInFrameBuffer] = FRAME_BORDER_FLAG_BYTE;
	numBytesInFrameBuffer ++;

	/*
	//##############################################################
	// Call method which sends frames over link
	//##############################################################
	*/
	MWRes = MWOutputDPUEncoder_Internal_sendFrameOverLink();

	/*
	//##############################################################
	// Bump up DPU counters
	//##############################################################
	*/
	DPUIndex ++;

	/*
	//##############################################################
	// If not the last DPU, then start a new one
	//##############################################################
	*/
	if  (bool_isLastInProductInstance == 0)
	{
		TMW_ResultCode MWResTemp = MWOutputDPUEncoder_Internal_startFrame();
		if (MWResTemp != MWRes_Success)
		{
			return MWResTemp;
		}
	}

	/*
	//##############################################################
	// Return result of output
	//##############################################################
	*/

	return MWRes;


}

/*
//====================================================================
//====================================================================
// Public methods
//====================================================================
//====================================================================
*/

TMW_ResultCode MWOutputDPUEncoder_startNewProduct
(
	TTwoByteInteger_Unsigned productID
)
{
	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// Reset the DPU sequence and start with the new product
	//##############################################################
	*/
	DPUIndex = 0;

	/* Bump up instance code */
	productInstanceCount ++;
	if (productInstanceCount > 65525) /* Max value is 65525 in spec. document */
	{
		/* Wrap around to 0 as described in spec */
		productInstanceCount = 0;
	}

	/*
	//##############################################################
	// Fill out DPU header
	//##############################################################
	*/

	/* Fill out product ID */
	pduHeader.productID_LOWByte = productID % 256;
	pduHeader.productID_HIGHByte = productID / 256;

	/* Fill out instance ID */
	pduHeader.instanceID_LOWByte = productInstanceCount % 256;
	pduHeader.instanceID_HIGHByte = productInstanceCount / 256;

	/* Fill out DPU count */
	pduHeader.DPUSequenceNumber_LOWByte = 0;
	pduHeader.DPUSequenceNumber_HIGHByte = 0;

	/* Not final one that we know of (yet) */
	pduHeader.finalDPUThisProductInstance = 0;

	/*
	//##############################################################
	// Start new frame
	//##############################################################
	*/
	MWRes = MWOutputDPUEncoder_Internal_startFrame();

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes;

}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWOutputDPUEncoder_outputBytes
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_outputBytes,
	TByte_Unsigned bool_endOfProductInstance
)
{
	TMW_ResultCode MWRes;
	TFourByteInteger_Unsigned numBytesWritten;
	TFourByteInteger_Unsigned numBytesWrittenThisPass;

	/*
	//##############################################################
	// Loop until all bytes are consumed
	//##############################################################
	*/
	numBytesWritten = 0;

	while (numBytesWritten < numBytes)
	{
		/*
		//##############################################################
		// Add as many bytes as possible to the current frame/DPU
		//##############################################################
		*/
		MWRes = MWOutputDPUEncoder_Internal_encodeFrameBodyBytes
		(
			(numBytes - numBytesWritten),
			(p_outputBytes + numBytesWritten),
			&numBytesWrittenThisPass,
			0 /* not CRC */
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/* Add bytes written this pass to the total */
		numBytesWritten += numBytesWrittenThisPass;

		/*
		//##############################################################
		// If there are bytes left over from the caller's parameters,
		// then we must end (and hence send) this frame and loop around
		// again in a new one.
		//
		//##############################################################
		*/
		if (numBytesWritten < numBytes)
		{
			/*
			// End that frame
			// This starts a new frame since we are not saying it is the
			// last in the product instance
			*/
			MWRes = MWOutputDPUEncoder_Internal_endFrame
			(
				0 /* 0 means it is NOT the last in the product instance */
			);
			if (MWRes != MWRes_Success)
			{
				return MWRes;
			}

		}
	}

	//##############################################################
	// Force frame ending if it is the end of the data
	//##############################################################
	if
	(
		(numBytesWritten >= numBytes) &&
		(bool_endOfProductInstance != 0)
	)
	{
		/*
		// We have consumed all of the input.  If the caller specified
		// that this is the last of the input, then we should end the frame.
		*/
		MWRes = MWOutputDPUEncoder_Internal_endFrame
		(
			1 /* 1 indicates it IS the last in the product instance */
		);
		return MWRes;
	}


	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;

}

