/*
====================================================================
// MW_MemorySpaceManager:
//
// This module handles run-time allocations and deallocations of sub-sections
// from an array of memory bytes.  It can run in a fixed size block mode
// which prevents internal fragmentation.
//
// Note that this is an Abstract Data Type module implemented in C for
// portability between platforms.
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//
====================================================================
*/


#ifndef MW_MEMORYSPACEMANAGER_HEADER
#define MW_MEMORYSPACEMANAGER_HEADER

/*
====================================================================
// Includes
====================================================================
*/

#include "MW_DataFieldTypes.h"


/*
====================================================================
// The memory block node structure
====================================================================
*/

typedef struct
{
	/* The previous and next nodes. Void* is necessary because of stupid
		C limitation */
	void* p_nextBlock;
	void* p_previousBlock;

	// Length of the block, in bytes, EXCLUDING this header
	TFourByteInteger_Unsigned lengthBytes;

} TMW_MemorySpaceManager_BlockNode;

/*
====================================================================
// The ADT structure
====================================================================
*/

typedef struct
{
	/* The number of bytes available in the managed memory */
	TFourByteInteger_Unsigned arrayLengthBytes;

	/* The first byte of the managed memory */
	TByte_Unsigned* p_arrayStart;

	/* If this is non-zero then all allocations will use a block of this size plus
		the size of the memory block node header.  Allocation attempts larger than a single
		block size will fail. */
	TFourByteInteger_Unsigned fixedBlockSize;

	/* The first unallocated block */
	TMW_MemorySpaceManager_BlockNode* p_firstUnallocated;


} TMW_MemorySpaceManager;

/*
====================================================================
// The ADT methods
====================================================================
*/

/*
====================================================================
// MemorySpaceManager_initialize:
//
// This is used to initally set up a memory space manager.
// In this method, the manager is handed an array of bytes which is
// the memory space it will control.
//
// The user can specify a fixed block size, which forces all
// allocations to internally use the same block size.  This
// limits the maximum allocation size, but prevents internal
// fragmentation.
//
// Please note the size of the memory block node, which will always
// occur an overhead inside of each block.  Hence, if a fixed block
// size of 1024 is chosen, then an allocation of 1 to 1024 bytes will
// use 1024 + sizeof(TMW_MemorySpaceManager_BlockHeader) bytes of
// memory.
//
// PARAMETERS
//
//	p_memorySpaceManager:
//		Pointer to the ADT instance being initialized.
//
//	in_numBytes:
//		The number of bytes which are being handed to this manager
//
//	in_p_bytes:
//		The contiguous array of bytes, at least in_numBytes in length, that
//		is being handed to this manager.
//
//	in_fixedBlockSize:
//		If this parameter is 0, then no fixed block size is in effect. Allocations
//		will use the number of requested bytes plus the size of the block
//		node header.  Allocations will succeed as long as there is enough
//		free contiguous memory.  However, internal fragmentation will result after
//		deallocations, and the time to allocate memory will not be fixed, but rather,
//		based on a search for a large enough free memory block.
//
//		If this parameter is a positive number, then each allocation will always use
//		in_fixedBlockSize bytes plus the size of the block header node.  Allocations
//		will succeed as long as there is a free block of the required size. This
//		can be internally wasteful if allocations smaller than the block size are
//		performed, but the fixed size prevents internal fragmentation and ensures
//		a constant time for each allocation.
//
// RETURNS:
//
//	MWRes_Success:
//		On success
//
//	MWRes_NullPointer:
//		if p_memorySpaceManager is NULL or
//		if in_p_bytes is NULL
//
//	MWRes_BadParamValue:
//		if in_numBytes <= in_fixedBlockSize
//
====================================================================
*/
TMW_ResultCode MemorySpaceManager_initialize
(
	TMW_MemorySpaceManager* p_memorySpaceManager,
	TFourByteInteger_Unsigned in_numBytes,
	TByte_Unsigned* in_p_bytes,
	TFourByteInteger_Unsigned in_fixedBlockSize
);


/*
====================================================================
// MemorySpaceManager_reset:
//
// This method resets the memory space manager such that all memory
// is considered free.  Note, if anyone is still actively using the memory,
// this method will leave dangling references to it.  Do not call this
// unless nothing is referencing any of the memory managed in this instance.
//
// PARAMETERS:
//
//	p_memorySpaceManager:
//		Pointer to the memory space manager on which this call is being made.
//
// RETURNS:
//
//	MWRes_Success:
//		on success
//
====================================================================
*/
TMW_ResultCode MemorySpaceManager_reset
(
	TMW_MemorySpaceManager* p_memorySpaceManager
);

/*
====================================================================
// MemorySpaceManager_allocate:
//
// This method requests an allocation of memory.
//
// PARAMETERS:
//
//	p_memorySpaceManager:
//		Pointer to the memory space manager on which this call is being made.
//
//	numBytes:
//		The number of bytes requested
//
//	p_out_p_bytes:
//		Pointer to a voide pointer.  The pointer should hold the address of a void pointer
//		which is to be set to the first externally usable byte of the allocated memory.
//		The externally used part will then consist of the contiguous numBytes starting
//		with the one indicated by the pointer.
//
// RETURNS:
//
//	MWRes_Success:
//		on success
//
//	MWRes_OutOfMemory:
//		if not enough memory was available for the request
//
//	MWRes_CodeFault:
//		if the manager was configured to use fixed size blocks and the
//		amount of memory requested is larger than the block size.
//
====================================================================
*/
TMW_ResultCode MemorySpaceManager_allocate
(
	TMW_MemorySpaceManager* p_memorySpaceManager,
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned** p_out_p_bytes
);

/*
====================================================================
// MemorySpaceManager_deallocate:
//
// This method returns the indicated memory back to the pool of unused memory.
//
// PARAMETERS:
//
//	p_memorySpaceManager:
//		Pointer to the memory space manager on which this call is being made.
//
//	p_bytes:
//		A pointer to the first byte of the usable portion of the memory block
//		being deallocated.  This is the same value given to the caller
//		via the p_out_p_bytes parameter of MemorySpacemanager_allocate.
//
// RETURNS:
//
//	MWRes_Success:
//		on success
//
//	MWRes_NullPointer:
//		if p_bytes is NULL
====================================================================
*/
TMW_ResultCode MemorySpaceManager_deallocate
(
	TMW_MemorySpaceManager* p_memorySpaceManager,
	TByte_Unsigned* p_bytes
);


/* End header wrapper */
#endif


