/*
//====================================================================
// MW_MemorySpaceManager.c:
//
// Implementation of the MW_MemorySpaceManager ADT methods.
//
// Copyright 2005: WSI Corp
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/

#include "stddef.h"
#include "standard.h"
#include "MW_MemorySpaceManager.h"

/*
//====================================================================
// Initialization
//====================================================================
*/

TMW_ResultCode MemorySpaceManager_initialize
(
	TMW_MemorySpaceManager* p_memorySpaceManager,
	TFourByteInteger_Unsigned in_numBytes,
	TByte_Unsigned* in_p_bytes,
	TFourByteInteger_Unsigned in_fixedBlockSize
)
{
	/*
	//########################################
	// Test parameters
	//########################################
	*/

	if (p_memorySpaceManager == NULL)
	{
		return MWRes_NullPointer;
	}

	if (in_numBytes < (in_fixedBlockSize + sizeof(TMW_MemorySpaceManager_BlockNode) ) )
	{
		return MWRes_BadParamValue;
	}

	/*
	//########################################
	// Set internal state
	//########################################
	*/

	p_memorySpaceManager->p_arrayStart = in_p_bytes;
	p_memorySpaceManager->arrayLengthBytes = in_numBytes;
	p_memorySpaceManager->fixedBlockSize = in_fixedBlockSize;

	/* Only 1 block, and it is all in the unallocated group */
	p_memorySpaceManager->p_firstUnallocated = (TMW_MemorySpaceManager_BlockNode*) (p_memorySpaceManager->p_arrayStart);
	p_memorySpaceManager->p_firstUnallocated->p_nextBlock = NULL;
	p_memorySpaceManager->p_firstUnallocated->p_previousBlock = NULL;
	p_memorySpaceManager->p_firstUnallocated->lengthBytes = p_memorySpaceManager->arrayLengthBytes - sizeof(TMW_MemorySpaceManager_BlockNode);

	/*
	//########################################
	// Done
	//########################################
	*/

	return MWRes_Success;

}

/*
//====================================================================
// Reset to empty
//====================================================================
*/

TMW_ResultCode MemorySpaceManager_reset
(
	TMW_MemorySpaceManager* p_memorySpaceManager
)
{
	/*
	########################################
	# Test parameters
	########################################
	*/

	if (p_memorySpaceManager == NULL)
	{
		return MWRes_NullPointer;
	}

	/*
	//########################################
	// Reset internal state to all unallocated
	//########################################
	*/

	/* Only 1 block, and it is all in the unallocated group */
	p_memorySpaceManager->p_firstUnallocated = (TMW_MemorySpaceManager_BlockNode*) (p_memorySpaceManager->p_arrayStart);
	p_memorySpaceManager->p_firstUnallocated->p_nextBlock = NULL;
	p_memorySpaceManager->p_firstUnallocated->p_previousBlock = NULL;
	p_memorySpaceManager->p_firstUnallocated->lengthBytes = p_memorySpaceManager->arrayLengthBytes  - sizeof(TMW_MemorySpaceManager_BlockNode);

	/*
	//########################################
	// Success
	//########################################
	*/
	return MWRes_Success;

}

/*
//====================================================================
// Allocation / Deallocation requests
//====================================================================
*/

TMW_ResultCode MemorySpaceManager_unlinkUnallocatedBlock
(
	TMW_MemorySpaceManager* p_memorySpaceManager,
	TMW_MemorySpaceManager_BlockNode* p_block
)
{
	/*
	//########################################
	// Test parameters
	//########################################
	*/
	if
	(
		(p_memorySpaceManager == NULL) ||
		(p_block == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	/*
	//########################################
	// Remove it
	//########################################
	*/
	if (p_block == p_memorySpaceManager->p_firstUnallocated)
	{
		p_memorySpaceManager->p_firstUnallocated = (TMW_MemorySpaceManager_BlockNode*) (p_block->p_nextBlock);
		if (p_block->p_previousBlock != NULL)
		{
			return MWRes_CodeFault;
		}
	}

	if (p_block->p_previousBlock != NULL)
	{
		((TMW_MemorySpaceManager_BlockNode*) (p_block->p_previousBlock))->p_nextBlock = p_block->p_nextBlock;
	}

	if (p_block->p_nextBlock != NULL)
	{
		((TMW_MemorySpaceManager_BlockNode*) (p_block->p_nextBlock))->p_previousBlock = p_block->p_previousBlock;
	}

	/*
	//########################################
	// Success
	//########################################
	*/
	return MWRes_Success;

}

//-----------------------------------------------------------------------------

TMW_ResultCode MemorySpaceManager_allocate
(
	TMW_MemorySpaceManager* p_memorySpaceManager,
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned** p_out_p_bytes
)
{

	TMW_ResultCode MWRes;

	/* For searching unallocated blocks */
	TMW_MemorySpaceManager_BlockNode* p_unallocatedBlock;

	/* Properties of the block being used */
	TMW_MemorySpaceManager_BlockNode* p_newBlock;
	TFourByteInteger_Unsigned blockSize;

	/*
	//########################################
	// Test parameters
	//########################################
	*/
	if
	(
		(p_memorySpaceManager == NULL) ||
		(p_out_p_bytes == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	if (numBytes == 0)
	{
		*p_out_p_bytes = NULL;
		return MWRes_CodeFault;
	}

	if
	(
		(p_memorySpaceManager->fixedBlockSize > 0) &&
		(numBytes > p_memorySpaceManager->fixedBlockSize)
	)
	{
		return MWRes_CodeFault;
	}


	/* Determine block size we will be creating */
	if (p_memorySpaceManager->fixedBlockSize > 0)
	{
		blockSize = p_memorySpaceManager->fixedBlockSize;
	}
	else
	{
		/*
		// Granular block size
		*/
#if 0
	    // Don't do this!
		blockSize = numBytes;
#else
	    // Do this instead...

        // Align on addressable boundaries
        ALIGN_SIZE(numBytes);
        blockSize = numBytes;
#endif
	}

	/*
	//########################################
	// Get a block that is large enough
	//########################################
	*/
	p_unallocatedBlock = p_memorySpaceManager->p_firstUnallocated;
	while (p_unallocatedBlock != NULL)
	{
		/*
		//########################################
		// Is this block large enough?
		//########################################
		*/

		if
		(
			(p_unallocatedBlock->lengthBytes == blockSize) ||
			(
				(p_unallocatedBlock->lengthBytes >= blockSize +  sizeof(TMW_MemorySpaceManager_BlockNode) ) &&
				(p_unallocatedBlock->lengthBytes > p_memorySpaceManager->fixedBlockSize)
			)
		)
		{
			/*
			// This block is large enough
			// Prepare
			*/
			p_newBlock = NULL;


			/* Is the block already the same size as the new one? */
			if (p_unallocatedBlock->lengthBytes == blockSize)
			{
				/* No need to split, re-use */
				MWRes = MemorySpaceManager_unlinkUnallocatedBlock
				(
					p_memorySpaceManager,
					p_unallocatedBlock
				);
				if (MWRes != MWRes_Success)
				{
					return MWRes;
				}

				/* Have new block */
				p_newBlock = p_unallocatedBlock;

			}
			else
			{
				/* Reduce size of unallocted block to free up end for new block */
				p_unallocatedBlock->lengthBytes -= (blockSize + sizeof(TMW_MemorySpaceManager_BlockNode));

				/* Create the new block from end part of old block */
				p_newBlock = (TMW_MemorySpaceManager_BlockNode*) ((TByte_Unsigned*) (p_unallocatedBlock) + sizeof(TMW_MemorySpaceManager_BlockNode) + p_unallocatedBlock->lengthBytes);
				p_newBlock->lengthBytes = blockSize;

				/* No longer linked to unallocated blocks */
				p_newBlock->p_nextBlock = NULL;
				p_newBlock->p_previousBlock = NULL;
			}

			/* Set output parameters for caller */
			*p_out_p_bytes = (TByte_Unsigned*)(p_newBlock) + sizeof(TMW_MemorySpaceManager_BlockNode);

			/* Success */
			return MWRes_Success;

		}
		else
		{
			/* Iterate cursor to next block of unallocated bytes */
			p_unallocatedBlock = (TMW_MemorySpaceManager_BlockNode*) (p_unallocatedBlock->p_nextBlock);
		}

	} /* Try next block */

	/*
	//########################################
	// None found, we have no memory to
	// meet the request
	//########################################
	*/
	return MWRes_OutOfMemory;

}

//-----------------------------------------------------------------------------

TMW_ResultCode MemorySpaceManager_deallocate
(
	TMW_MemorySpaceManager* p_memorySpaceManager,
	TByte_Unsigned* p_bytes
)
{
	/* Pointer to the block header of the block being deallocated */
	TMW_MemorySpaceManager_BlockNode* p_blockHeader;


	/*
	//########################################
	// Test parameters
	//########################################
	*/
	if
	(
		(p_memorySpaceManager == NULL) ||
		(p_bytes == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	/*
	//########################################
	// Get block header pointer
	//########################################
	*/
	p_blockHeader = (TMW_MemorySpaceManager_BlockNode*) (p_bytes - sizeof(TMW_MemorySpaceManager_BlockNode));

	/*
	//########################################
	// Move this block to the unallocated list
	//########################################
	*/

	/* This block is the new first block */
	p_blockHeader->p_previousBlock = NULL;
	p_blockHeader->p_nextBlock = (void*) (p_memorySpaceManager->p_firstUnallocated);

	if (p_memorySpaceManager->p_firstUnallocated != NULL)
	{
		p_memorySpaceManager->p_firstUnallocated->p_previousBlock = p_blockHeader;
	}

	p_memorySpaceManager->p_firstUnallocated = p_blockHeader;

	/*
	//########################################
	// Done
	//########################################
	*/
	return MWRes_Success;
}

