/*
//====================================================================
// MW_GeneralRasterProcessing.h:
//
// This defines a block of memory and a memory space manager that
// are used by the encoders/decoders that require holding entire
// rasters.
//
// It also defines some basic operations on a raster which are
// used by various raster CODECs (upsample, upsample with WH smooth
// etc...)
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
//====================================================================
*/

/* Header wrapper */
#ifndef MW_GENERALRASTERPROCESSING_HEADER
#define MW_GENERALRASTERPROCESSING_HEADER

/*
//====================================================================
// Required includes
//====================================================================
*/
#include "MW_DataFieldTypes.h"

/*
//====================================================================
// Defines
//====================================================================
*/

/* TODO: Consult on this limit */
#define MAX_NUM_PIXELS (1024 * 1024)

#define MAX_UPSAMPLING_ITERATIONS 3




/*
//====================================================================
//====================================================================
// Common raster manipulation methods
//====================================================================
//====================================================================
*/



/*
//====================================================================
// MWGeneralRasterProcessing_reset:
//
// Call this before working with any new product. This resets the
// raster memory in a fragmentation free fashion.
//
// RETURNS:
//
//	MWRes_Success
//====================================================================
*/
TMW_ResultCode MWGeneralRasterProcessing_reset();



/*
//====================================================================
// MWGeneralRasterProcessing_allocateRaster:
//
// This method is used to allocate the memory for a raster.
//
// PARAMETERS:
//
//	numBytes:
//		The number of bytes in the raster surface
//
//	p_out_p_rasterSurface:
//		Pass in a pointer to a raster memory pointer.  The raster
//		memory pointer will be assigned the address of the first
//		byte of the allocated raster memory surface.  The returned
//		raster memory will be a contiguous array of bytes numBytes
//		long.
//
// RETURNS:
//
//	MWRes_Success:
//
//	MWRes_NullPointer:
//		If pp_out_rasterSurface is NULL
//
//	MWRes_BadParamValue:
//		If numBytes is 0
//
//	MWRes_OutOfMemory
//		If there is not enough memory put aside for rasters to
//		accomodate a request of this size.
//
//====================================================================
*/
TMW_ResultCode MWGeneralRasterProcessing_allocateRaster
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned** p_out_p_rasterSurface
);

/*
//====================================================================
// MWGeneralRasterProcessing_upsample:
//
// PARAMETERS:
//
//	numRowsInSource:
//		The number of rows in the source (low resolution)
//
//	numColumnsInSource:
//		The number of columns at the source (low resolution)
//
//	p_sourcePixels:
//		The source pixels
//
//	numRowsInDestination:
//		The number of rows in the destination (high resolution)
//
//	numColumnsInDestination:
//		The number of columns in the destination (high resolution)
//
//	p_destinationPixels:
//		The destination pixels
//
//	bool_performWHWeatherSmoothing:
//		If non-zero, Weather Huffman weather smoothing is performed
//		using a convolustion on the source to modify the destination.
//
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_BadParamValue
//		If numColumnsInDestination does not meet its restrictions
//		(see Parameters section)
//
//	MWRes_NullPointer:
//		if p_sourcePixels is NULL
//		if p_destinationPixels is NULL
//====================================================================
*/
TMW_ResultCode MWGeneralRasterProcessing_upsample
(
	TTwoByteInteger_Unsigned numRowsInSource,
	TTwoByteInteger_Unsigned numColumnsInSource,
	TByte_Unsigned* p_sourcePixels,
	TTwoByteInteger_Unsigned numRowsInDestination,
	TTwoByteInteger_Unsigned numColumnsInDestination,
	TByte_Unsigned* p_destinationPixels,
	TByte_Unsigned bool_performWHWeatherSmoothing
);


/* End header wrapper */
#endif

