/*
//====================================================================
// MW_GeneralRasterProcessing.c:
//
// Implementation of the general raster processing methods.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/
#include "MW_GeneralRasterProcessing.h"

/* For the raster buffer memory fragmentation free allocator */
#include "MW_MemorySpaceManager.h"

/*
//====================================================================
// Defines
//====================================================================
*/

/* This can be derived from the MAX_NUM_PIXELS AND MAX_UPSMAPLING_ITERATIONS values */
#define MAX_RASTER_BUFFER_BYTES ( MAX_NUM_PIXELS + (MAX_NUM_PIXELS / 2))



/*
//====================================================================
// Data resources
//====================================================================
*/

/* The buffer space used for rasters */
static TByte_Unsigned rasterBufferMemory[MAX_RASTER_BUFFER_BYTES];

/* ADT which is used to allocate memory from rasterBufferMemory */
static TMW_MemorySpaceManager rasterBufferManager;



/*
//====================================================================
//====================================================================
// Internal methods
//====================================================================
//====================================================================
*/




/*
//====================================================================
//====================================================================
// Public methods
//====================================================================
//====================================================================
*/

TMW_ResultCode MWGeneralRasterProcessing_reset()
{
	/* Reset memory manager */
	return MemorySpaceManager_initialize
	(
		&rasterBufferManager,
		MAX_RASTER_BUFFER_BYTES,
		rasterBufferMemory,
		0 /* Not using fixed block sizes, we never deallocate so no fragmnetation */
	);

}

//--------------------------------------------------------------------

TMW_ResultCode MWGeneralRasterProcessing_allocateRaster
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned** p_out_p_rasterSurface
)
{
	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/

	if (p_out_p_rasterSurface == NULL)
	{
		return MWRes_NullPointer;
	}

	if (numBytes == 0)
	{
		return MWRes_BadParamValue;
	}

	/*
	//##############################################################
	// Get memory from raster buffer manager
	//##############################################################
	*/
	return MemorySpaceManager_allocate
	(
		&rasterBufferManager,
		numBytes,
		p_out_p_rasterSurface
	);
}

//--------------------------------------------------------------------

TMW_ResultCode MWGeneralRasterProcessing_upsample
(
	TTwoByteInteger_Unsigned numRowsInSource,
	TTwoByteInteger_Unsigned numColumnsInSource,
	TByte_Unsigned* p_sourcePixels,
	TTwoByteInteger_Unsigned numRowsInDestination,
	TTwoByteInteger_Unsigned numColumnsInDestination,
	TByte_Unsigned* p_destinationPixels,
	TByte_Unsigned bool_performWHWeatherSmoothing
)
{

	/* For iterating the source */
	TTwoByteInteger_Unsigned sourceRow;
	TTwoByteInteger_Unsigned sourceColumn;

	/* Same for destination */
	TTwoByteInteger_Unsigned destinationRow;
	TTwoByteInteger_Unsigned destinationColumn;

	/* The cursors */
	TByte_Unsigned *p_sourceCursor;
	TByte_Unsigned *p_destinationCursor;

	/* The pixel value of the superpixel (low resolution) */
	TByte_Unsigned pixelValue;

	/*
	// Smoothing rule test results for each sub-pixel derived from
	// one source row pixel and its source neighbors.
	*/
	TByte_Unsigned b_ULSmooth;
	TByte_Unsigned b_URSmooth;
	TByte_Unsigned b_LLSmooth;
	TByte_Unsigned b_LRSmooth;

	/* After combinations of b_XXSmooth cancelling each other out */
	TByte_Unsigned b_ULSmoothFinal;
	TByte_Unsigned b_URSmoothFinal;
	TByte_Unsigned b_LLSmoothFinal;
	TByte_Unsigned b_LRSmoothFinal;

	/* Pixel values that smoothed pixels become */
	TByte_Unsigned smoothingReplaceValue[4] = {0,0,0,0};

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if ((p_sourcePixels == NULL) || (p_destinationPixels == NULL))
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Read in each source pixel, compare its neighboring source
	// pixels, and upsample with weather-huffman smoothing.
	//##############################################################
	*/
	destinationRow = 0;
	p_sourceCursor = p_sourcePixels;
	p_destinationCursor = p_destinationPixels;

	for (sourceRow = 0; sourceRow < numRowsInSource; sourceRow ++)
	{
		destinationColumn = 0;

		for (sourceColumn = 0; sourceColumn < numColumnsInSource; sourceColumn ++)
		{

			/* No smoothing until tested */
			b_ULSmooth = 0;
			b_URSmooth = 0;
			b_LLSmooth = 0;
			b_LRSmooth = 0;

			/*
			//##############################################################
			// Get the pixel value
			//##############################################################
			*/
			pixelValue = *p_sourceCursor;

			/*
			//##############################################################
			// Does upper left sub-pixel qualify for smoothing?
			//##############################################################
			*/
			if
			(
				(sourceRow > 0) &&
				(sourceColumn >= 1)
			)
			{
				/* Does it qualify */
				if
				(
					(bool_performWHWeatherSmoothing != 0) &&
					(*(p_sourceCursor -1) < pixelValue) &&
					(*(p_sourceCursor - numColumnsInSource)  < pixelValue) &&
					(*(p_sourceCursor - (numColumnsInSource +1)) < pixelValue)
				)
				{
					b_ULSmooth = 1;

					/* What would be the highest neighbor? */
					smoothingReplaceValue[0] = *(p_sourceCursor -1);
					if (*(p_sourceCursor - numColumnsInSource) > smoothingReplaceValue[0])
					{
						smoothingReplaceValue[0] = *(p_sourceCursor - numColumnsInSource);
					}
					else if (*(p_sourceCursor - (numColumnsInSource+1)) > smoothingReplaceValue[0])
					{
						smoothingReplaceValue[0] = *(p_sourceCursor - (numColumnsInSource + 1));
					}
				}

			}

			/*
			//##############################################################
			// Same for upper right
			//##############################################################
			*/
			if
			(
				(sourceRow > 0) &&
				(sourceColumn < (numColumnsInSource-1))
			)
			{
				/* Does it qualify */
				if
				(
					(bool_performWHWeatherSmoothing != 0) &&
					(*(p_sourceCursor +1) < pixelValue) &&
					(*(p_sourceCursor - numColumnsInSource) < pixelValue) &&
					(*(p_sourceCursor - (numColumnsInSource-1)) < pixelValue)
				)
				{
					b_URSmooth = 1;

					/* What would be the highest neighbor? */
					smoothingReplaceValue[1] = *(p_sourceCursor +1);
					if (*(p_sourceCursor - numColumnsInSource) > smoothingReplaceValue[1])
					{
						smoothingReplaceValue[1] = *(p_sourceCursor - numColumnsInSource);
					}
					else if (*(p_sourceCursor - (numColumnsInSource-1)) > smoothingReplaceValue[1])
					{
						smoothingReplaceValue[1] = *(p_sourceCursor - (numColumnsInSource-1));
					}
				}

			}

			/*
			//##############################################################
			// Lower left
			//##############################################################
			*/

			if
			(
				(sourceRow < (numRowsInSource-1)) &&
				(sourceColumn >= 1)
			)
			{
				if
				(
					(bool_performWHWeatherSmoothing != 0) &&
					(*(p_sourceCursor - 1) < pixelValue) &&
					(*(p_sourceCursor + numColumnsInSource) < pixelValue) &&
					(*(p_sourceCursor + (numColumnsInSource -1)) < pixelValue)
				)
				{
					b_LLSmooth = 1;

					/* What would be the highest neighbor */
					smoothingReplaceValue[2] = *(p_sourceCursor - 1);
					if (*(p_sourceCursor + numColumnsInSource) > smoothingReplaceValue[2])
					{
						smoothingReplaceValue[2] = *(p_sourceCursor + numColumnsInSource);
					}
					else if (*(p_sourceCursor + (numColumnsInSource-1)) > smoothingReplaceValue[2])
					{
						smoothingReplaceValue[2] = *(p_sourceCursor + (numColumnsInSource-1));
					}
				}

			}

			/*
			//##############################################################
			// Lower right
			//##############################################################
			*/
			if
			(
				(sourceRow < (numRowsInSource-1)) &&
				(sourceColumn < (numColumnsInSource-1))
			)
			{
				if
				(
					(bool_performWHWeatherSmoothing != 0) &&
					(*(p_sourceCursor +1) < pixelValue) &&
					(*(p_sourceCursor + numColumnsInSource) < pixelValue) &&
					(*(p_sourceCursor + (numColumnsInSource+1)) < pixelValue)
				)
				{
					b_LRSmooth = 1;

					/* What would be the highest neighbor? */
					smoothingReplaceValue[3] = *(p_sourceCursor +1);
					if (*(p_sourceCursor + numColumnsInSource) > smoothingReplaceValue[3])
					{
						smoothingReplaceValue[3] = *(p_sourceCursor + numColumnsInSource);
					}
					else if (*(p_sourceCursor + (numColumnsInSource+1)) > smoothingReplaceValue[3])
					{
						smoothingReplaceValue[3] = *(p_sourceCursor + (numColumnsInSource+1));
					}
				}

			}

			/*
			//##############################################################
			// Some combinations cancel each other out
			//##############################################################
			*/

			b_ULSmoothFinal = b_ULSmooth;
			if  ( ( b_URSmooth != 0) || (b_LLSmooth != 0) )
			{
				b_ULSmoothFinal = 0;
			}

			b_URSmoothFinal = b_URSmooth;
			if (( b_ULSmooth != 0) || ( b_LRSmooth != 0))
			{
				b_URSmoothFinal = 0;
			}

			b_LLSmoothFinal = b_LLSmooth;
			if ( (b_ULSmooth!= 0)  || (b_LRSmooth!= 0))
			{
				b_LLSmoothFinal = 0;
			}

			b_LRSmoothFinal = b_LRSmooth;
			if ( (b_LLSmooth != 0) || (b_URSmooth != 0) )
			{
				b_LRSmoothFinal = 0;
			}

			/*
			//##############################################################
			// Set superpixel values in the two destination rows
			//##############################################################
			*/

			// Upper left sub-pixel
			if (b_ULSmoothFinal != 0)
			{
				*p_destinationCursor = smoothingReplaceValue[0];
			}
			else
			{
				*p_destinationCursor = *p_sourceCursor;

			}

			/* Upper right sub-pixel */
			if (destinationColumn < (numColumnsInDestination-1))
			{
				if (b_URSmoothFinal != 0)
				{
					*(p_destinationCursor+1) = smoothingReplaceValue[1];
				}
				else
				{
					*(p_destinationCursor+1) = *p_sourceCursor;
				}
			}

			/* Lower left and right */
			if (destinationRow < (numRowsInDestination-1))
			{
				/* Lower left sub-pixel */
				if (b_LLSmoothFinal != 0)
				{
					*(p_destinationCursor + numColumnsInDestination) = smoothingReplaceValue[2];
				}
				else
				{
					*(p_destinationCursor + numColumnsInDestination) = *p_sourceCursor;
				}

				/* Lower right sub-pixel */
				if (destinationColumn < (numColumnsInDestination-1))
				{
					if (b_LRSmoothFinal != 0)
					{
						*(p_destinationCursor + numColumnsInDestination + 1) = smoothingReplaceValue[3];
					}
					else
					{
						*(p_destinationCursor + numColumnsInDestination + 1) = *p_sourceCursor;
					}
				}

			}

			/*
			//##############################################################
			// Advance cursors and indices
			//##############################################################
			*/
			p_sourceCursor ++;
			if (destinationColumn < (numColumnsInDestination-1))
			{
				p_destinationCursor = p_destinationCursor + 2;
			}
			else
			{
				p_destinationCursor = p_destinationCursor + 1;
			}

			destinationColumn += 2;

			/* Ready for next column */

		} /* Next column */

		/* Advance destination cursor to skip over row already written */
		p_destinationCursor += numColumnsInDestination;
		destinationRow += 2;


	} /* Next row */

	/*
	//##############################################################
	// Done
	//##############################################################
	*/

	return MWRes_Success;

}

