/*
//====================================================================
// MW_DownsampledDeflatedRLEProcessor.c:
//
// Implementation of the MWDDRLE format processor.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/

/* Header for this ADT */
#include "MW_DownsampledDeflatedRLEProcessor.h"

/* For general raster methods */
#include "MW_GeneralRasterProcessing.h"

/* For decompression */
#include "MW_DEFLATECODEC.h"

/* For output */
#include "MW_OutputEncoder.h"

/*
//====================================================================
// Defines
//====================================================================
*/


/*
//====================================================================
// Data resources
//====================================================================
*/

/* All structures here use byte alignment */
#pragma pack (push)
#pragma pack (1)

typedef struct
{
	TTwoByteInteger_Unsigned numDownsampledRows;
	TTwoByteInteger_Unsigned numDownsampledColumns;
	TByte_Unsigned bool_useWHSmoothOnUpsample;

} TMW_MWDDRLE_BlockHeader;


/* Done byte alignment */
#pragma pack (pop)

/*
//====================================================================
//====================================================================
// Internal methods
//====================================================================
//====================================================================
*/

/*
//====================================================================
// MWDDRLE_Internal_readBlockHeader:
//
// Parses in the header block of the MWDDRLE format.
// The header block occurs just after the 4 byte length of the block
// encoded before the block.
//
// PARAMETERS:
//
//	numBodyBytes:
//		This indicates the number of deflated body bytes in the caller's
//		array.  This is assumed to be the entire product body as
//		compressed in broadcast receipt.
//
//	p_bytes:
//		This is the array of bytes holding the product body, starting at
//		the first byte of the product body (after the header and ICF and OCF
//		flags.
//
//	p_inout_numBytesConsumed:
//		Pointer to a variable that tracks how many bytes from p_bytes
//		have been used.  We update it to reflect any bytes used by
//		this method.
//
//	p_out_blockHeader:
//		Pointer to the block header structure into which we will be
//		reading.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_ProductContentsInvalid:
//		If there are not enough bytes left in the body block for
//		the expected header.
//====================================================================
*/

TMW_ResultCode MWDDRLE_Internal_readBlockHeader
(
	TFourByteInteger_Unsigned numBodyBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_inout_numBytesConsumed,
	TMW_MWDDRLE_BlockHeader* p_out_blockHeader
)
{

#ifndef HAS_MEMCPY
	TFourByteInteger_Unsigned i;
#endif

	/*
	//##############################################################
	// Are there enough bytes left
	//##############################################################
	*/
	if ((numBodyBytes - *p_inout_numBytesConsumed) < sizeof(TMW_MWDDRLE_BlockHeader) )
	{
		return MWRes_ProductContentsInvalid;
	}

	/*
	//##############################################################
	// Parse it out
	//##############################################################
	*/

#ifdef HAS_MEMCPY
	memcpy (p_out_blockHeader, (p_bytes + *p_inout_numBytesConsumed), sizeof(TMW_MWDDRLE_BlockHeader));
#else

	for (i = 0; i < sizeof (TMW_MWDDRLE_BlockHeader); i ++)
	{
		( (TByte_Unsigned*) p_out_blockHeader)[i] = p_bytes[(*p_inout_numBytesConsumed) + i];
	}
#endif

	/*
	//##############################################################
	// Note bytes consumed
	//##############################################################
	*/

	*p_inout_numBytesConsumed = (*p_inout_numBytesConsumed) + sizeof (TMW_MWDDRLE_BlockHeader);

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;

}



/*
//====================================================================
// MWDDRLE_Internal_decompressTransmittedRaster:
//
// This method decompresses the transmitted raster into the given
// raster surface.  The raster surface is expected to be large
// enough to hold the raster in its transmitted dimensions.
//
// PARAMETERS:
//
//	numBodyBytes:
//		This indicates the number of deflated body bytes in the caller's
//		array.  This is assumed to be the entire product body as
//		compressed in broadcast receipt.
//
//	p_bytes:
//		This is the array of bytes holding the product body, starting at
//		the first byte of the product body (after the header and ICF and OCF
//		flags.
//
//	p_inout_numBytesConsumed:
//		Pointer to a variable that tracks how many bytes from p_bytes
//		have been used.  We update it to reflect any bytes used by
//		this method.
//
//	numTransmittedColumns:
//		The number of columns in the transmitted resolution
//
//	numTransmittedRows:
//		The number of rows in the transmitted resolution
//
//	p_rasterSurface:
//		The raster surface into which the data should be written.
//
//====================================================================
*/
TMW_ResultCode MWDDRLE_Internal_decompressTransmittedRaster
(
	TFourByteInteger_Unsigned numBodyBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_inout_numBytesConsumed,
	TTwoByteInteger_Unsigned numTransmittedColumns,
	TTwoByteInteger_Unsigned numTransmittedRows,
	TByte_Unsigned* p_workingRaster
)
{
	TMW_ResultCode MWRes;
	TFourByteInteger_Unsigned numBytesOfDeflatedDataConsumed;
	TByte_Unsigned* p_firstByteOfDeflatedData;
	TFourByteInteger_Unsigned numPixelsFilled;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if (p_workingRaster == NULL)
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Enough size?
	//##############################################################
	*/
	if (numBodyBytes <= (*p_inout_numBytesConsumed))
	{
		return MWRes_BadParamValue;
	}

	/*
	//##############################################################
	// Use codec method to decompress into raster
	//##############################################################
	*/

	p_firstByteOfDeflatedData = p_bytes + (*p_inout_numBytesConsumed);
	numBytesOfDeflatedDataConsumed = 0;
	numPixelsFilled = 0;

	MWRes = MWDEFLATECODEC_decompressDeflatedWSIRLEProductBodyBlockIntoRaster
	(
		numBodyBytes - (*p_inout_numBytesConsumed),
		p_firstByteOfDeflatedData,
		&numBytesOfDeflatedDataConsumed,
		numTransmittedColumns * numTransmittedRows,
		p_workingRaster,
		&numPixelsFilled
	);


	/* Add bytes consumed to total */
	*p_inout_numBytesConsumed = (*p_inout_numBytesConsumed) + numBytesOfDeflatedDataConsumed;

	/* Handle error in decompression */
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// All pixels should have been consumed
	//##############################################################
	*/
	if (numPixelsFilled != (TFourByteInteger_Unsigned) (numTransmittedColumns * numTransmittedRows))
	{
		return MWRes_ProductContentsInvalid;
	}

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;

}

/*
//--------------------------------------------------------------------
*/

/*
//====================================================================
// MWDDRLE_Internal_performUpsampling:
//
// This method performs from 1 to N stages of upsampling given
// the source raster and the target size.
//
// PARAMETERS:
//
//	numUpsampledColumns:
//		The number of columns in the final upsampled size
//
//	numUpsampledRows:
//		The number of rows in teh final upsampled size
//
//	numDownsampledColumns:
//		The number of columns in the low resolution source
//
//	numDownsmapledRows:
//		The number of rows in the low resolution source
//
//	p_raster1:
//		Pointer to the first raster to be used. This always contains
//		the initial source.
//
//	p_raster2:
//		Pointer to the second raster to be used.
//
//	bool_useWHSmoothOnUpsample:
//		Set to 1 if upsampling should apply Weather Huffman
//		style smoothing. Set to 0 otherwise.
//====================================================================
*/
TMW_ResultCode MWDDRLE_Internal_performUpsampling
(
	TTwoByteInteger_Unsigned numUpsampledColumns,
	TTwoByteInteger_Unsigned numUpsampledRows,
	TTwoByteInteger_Unsigned numDownsampledColumns,
	TTwoByteInteger_Unsigned numDownsampledRows,
	TByte_Unsigned* p_raster1,
	TByte_Unsigned* p_raster2,
	TByte_Unsigned bool_useWHSmoothOnUpsample
)
{
	TMW_ResultCode MWRes;

	/* Size of upsample result for a given round */
	TTwoByteInteger_Unsigned upsampledWidthThisRound;
	TTwoByteInteger_Unsigned upsampledHeightThisRound;

	/* Pointers which alternate as we sample are way up */
	TByte_Unsigned* p_sourceRaster;
	TByte_Unsigned* p_upsampledRaster;
	TByte_Unsigned* p_tempRasterPointer;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/

	if ((p_raster1 == NULL) || (p_raster2 == NULL))
	{
		return MWRes_NullPointer;
	}

	/* Test this paramter, as a 0 could create an infinite loop */
	if ((numDownsampledColumns == 0) || (numDownsampledRows == 0))
	{
		return MWRes_BadParamValue;
	}

	/*
	//##############################################################
	// Perform each upsample
	//##############################################################
	*/

	/* First round goes from raster 1 to raster 2 */
	p_sourceRaster = p_raster1;
	p_upsampledRaster = p_raster2;

	while (numDownsampledColumns < numUpsampledColumns)
	{
		/*
		//##############################################################
		// Determine upsampled size this round
		//##############################################################
		*/
		upsampledWidthThisRound = numUpsampledColumns;
		upsampledHeightThisRound = numUpsampledRows;
		while ( (upsampledWidthThisRound / 2) > numDownsampledColumns)
		{
			upsampledWidthThisRound /= 2;
			upsampledHeightThisRound /= 2;
		}

		/*
		//##############################################################
		// Upsample
		//##############################################################
		*/

		MWRes = MWGeneralRasterProcessing_upsample
		(
			numDownsampledRows,
			numDownsampledColumns,
			p_sourceRaster,
			upsampledHeightThisRound,
			upsampledWidthThisRound,
			p_upsampledRaster,
			bool_useWHSmoothOnUpsample
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}


		/*
		//##############################################################
		// Downsampled size for next round is upsample result of this
		// round
		//##############################################################
		*/
		numDownsampledRows = upsampledHeightThisRound;
		numDownsampledColumns = upsampledWidthThisRound;

		/*
		//##############################################################
		// Switch roles of the two rasters.
		//##############################################################
		*/
		p_tempRasterPointer = p_sourceRaster;
		p_sourceRaster = p_upsampledRaster;
		p_upsampledRaster = p_tempRasterPointer;
	}

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;


}

/*
//====================================================================
//====================================================================
// Public methods
//====================================================================
//====================================================================
*/

TMW_ResultCode MWDDRLE_transformFormat
(
	TFourByteInteger_Unsigned numBodyBytes,
	TByte_Unsigned* p_bytes,
	TByte_Unsigned bool_productEndsWithThisBody,
	TTwoByteInteger_Unsigned numUpsampledColumns,
	TTwoByteInteger_Unsigned numUpsampledRows,
	TByte_Unsigned numBitsPerPixel,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed
)
{
	TMW_ResultCode MWRes;
	TMW_MWDDRLE_BlockHeader blockHeader;

	/* The number of times the transmitted raster must be upsampled
	   to reach the final specified upsampled size */
	TTwoByteInteger_Unsigned numRoundsOfUpsamplingRequired;

	TTwoByteInteger_Unsigned tempWidth;
	TTwoByteInteger_Unsigned tempHeight;
	TFourByteInteger_Unsigned tempNumPixels;

	/* The raster, at its transmitted resolution, and also during upsampling */
	TByte_Unsigned* p_workingRaster = NULL;

	/* The raster surface of the raster at its final resolution */
	TByte_Unsigned* p_finalRaster = NULL;

	/* This pointer gets assigned to be either p_workingRaster or
		p_finalRaster depending on whether the number of upsampling
		iterations is odd or even. */
	TByte_Unsigned* p_readRaster = NULL;
	TByte_Unsigned* p_nonReadRaster = NULL;

	/* For creating second raster */
	TFourByteInteger_Unsigned numR2Rows, numR2Columns;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if ((p_out_numBytesConsumed == NULL) || (p_bytes == NULL))
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// No bytes consumed yet
	//##############################################################
	*/
	*p_out_numBytesConsumed = 0;

	/*
	//##############################################################
	// Starting new output block
	//##############################################################
	*/
	MWRes = MWOutputEncoder_startNewEncodingBlock();
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Reset raster processing
	//##############################################################
	*/
	MWRes = MWGeneralRasterProcessing_reset();
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Read in the MWDDRLE block header
	//##############################################################
	*/
	MWRes = MWDDRLE_Internal_readBlockHeader
	(
		numBodyBytes,
		p_bytes,
		p_out_numBytesConsumed,
		&blockHeader
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/

	if
	(
		(numUpsampledColumns < blockHeader.numDownsampledColumns) ||
		(numUpsampledRows < blockHeader.numDownsampledRows)
	)
	{
		return MWRes_ProductContentsInvalid;
	}

	/*
	//##############################################################
	// Determine the number of rounds of upsampling required
	//##############################################################
	*/

	numRoundsOfUpsamplingRequired = 0;
	tempWidth = blockHeader.numDownsampledColumns;
	tempHeight = blockHeader.numDownsampledRows;

	while ((tempWidth < numUpsampledColumns) || (tempHeight < numUpsampledRows))
	{
		tempWidth = tempWidth * 2;
		tempHeight = tempHeight * 2;
		numRoundsOfUpsamplingRequired ++;

		/* Just in case, a stop-gap */
		if (numRoundsOfUpsamplingRequired > 100)
		{
			return MWRes_ProductContentsInvalid;
		}
	}

	/*
	//##############################################################
	// If there is any upsampling to be done, then we need two
	// raster buffers.
	//##############################################################
	*/
	if (numRoundsOfUpsamplingRequired > 0)
	{

		numR2Rows = (numUpsampledRows / 2);
		if (numUpsampledRows %2)
		{
			numR2Rows ++;
		}

		numR2Columns = (numUpsampledColumns / 2);
		if (numUpsampledColumns %2)
		{
			numR2Columns ++;
		}

		tempNumPixels = numR2Rows * numR2Columns;

		MWRes = MWGeneralRasterProcessing_allocateRaster
		(
			tempNumPixels,
			&p_workingRaster
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

	}

	//##############################################################
	// Always need final raster
	//##############################################################

	/* No upsampling required, so read into final buffer */
	tempNumPixels = numUpsampledColumns * numUpsampledRows;

	MWRes = MWGeneralRasterProcessing_allocateRaster
	(
		tempNumPixels,
		&p_finalRaster
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Read transmitted raster into one of the rasters, such that
	// the upsampling can use the two rasters back and forth
	// and the final upsampling will always wind up in the
	// final raster.
	//##############################################################
	*/

	if (numRoundsOfUpsamplingRequired % 2)
	{
		/*
		// Odd number of upsamples (1, 3, 5, etc...)
		// Read into working raster
		*/
		p_readRaster = p_workingRaster;
		p_nonReadRaster = p_finalRaster;
	}
	else
	{
		/*
		// Even number of upsamples (0, 2, 4, 6)
		// Read into final raster
		*/
		p_readRaster = p_finalRaster;
		p_nonReadRaster = p_workingRaster;
	}

	MWRes = MWDDRLE_Internal_decompressTransmittedRaster
	(
		numBodyBytes,
		p_bytes,
		p_out_numBytesConsumed,
		blockHeader.numDownsampledColumns,
		blockHeader.numDownsampledRows,
		p_readRaster
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Perform upsampling
	//##############################################################
	*/
	if (numRoundsOfUpsamplingRequired > 0)
	{
		MWRes = MWDDRLE_Internal_performUpsampling
		(
			numUpsampledColumns,
			numUpsampledRows,
			blockHeader.numDownsampledColumns,
			blockHeader.numDownsampledRows,
			p_readRaster,
			p_finalRaster,
			blockHeader.bool_useWHSmoothOnUpsample
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}
	}

	/*
	//##############################################################
	// output the final raster
	//##############################################################
	*/

	/* Starting new output encoding block (1 per raster plane) */
	MWRes = MWOutputEncoder_startNewEncodingBlock();
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}


	MWRes = MWOutputEncoder_outputEncodedBytes
	(
		numUpsampledColumns * numUpsampledRows,
		p_finalRaster,
		1, /* End of this block is at end of this data */
		bool_productEndsWithThisBody
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;




}
