/*
//====================================================================
// MW_DEFLATECODEC.h:
//
// This ADT performs INFLATION of DEFLATED data.
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//====================================================================
*/

#ifndef MW_DEFLATECODEC_HEADER
#define MW_DEFLATECODEC_HEADER

/*
//====================================================================
// Required Includes
//====================================================================
*/
#include "MW_DataFieldTypes.h"

/*
//====================================================================
// MWDEFLATECODEC_inflateProductBodyBlock:
//
// This method handles the INFLATION of a product body block.
//
// ASSUMES:
//	That MWOutputEncoder_startProduct has already been called.
//
// PARAMETERS:
//
//	numBodyBytes:
//		This indicates the number of deflated body bytes in the caller's
//		array.  This is assumed to be the entire product body as
//		compressed in broadcast receipt.
//
//	p_bytes:
//		This is the array of bytes holding the product body, starting at
//		the first byte of the product body (after the header and ICF and OCF
//		flags.
//
//	bool_EndOfInflationIsEndOfProduct:
//		If 1, then when inflation block ends, the product is done.
//		If 0, the product is not ended when this method returns.
//
//	p_out_numBytesConsumed:
//		This returns the number of body bytes bytes consumed in the inflation.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_NullPointer:
//		if p_bytes is NULL
//
//	MWRES_BadParamValue
//		if numBodyBytes < 1
//
//====================================================================
*/
TMW_ResultCode MWDEFLATECODEC_inflateProductBodyBlock
(
	TFourByteInteger_Unsigned numBodyBytes,
	TByte_Unsigned* p_bytes,
	TByte_Unsigned bool_EndOfInflationIsEndOfProduct,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed
);


/*
//====================================================================
// MWDEFLATECODEC_inflateWSIRLEProductBodyBlock:
//
// This method handles the INFLATION of a product body block and
// then passes the INFLATED bytes to the WSI RLE decoder for
// removal of a WSI RLE layer.
//
// ASSUMES:
//	That MWOutputEncoder_startProduct has already been called.
//
// PARAMETERS:
//
//	numBodyBytes:
//		This indicates the number of deflated body bytes in the caller's
//		array.  This is assumed to be the entire product body as
//		compressed in broadcast receipt.
//
//	p_bytes:
//		This is the array of bytes holding the product body, starting at
//		the first byte of the product body (after the header and ICF and OCF
//		flags.
//
//	bool_EndOfInflationIsEndOfProduct:
//		If 1, then when inflation block ends, the product is done.
//		If 0, the product is not ended when this method returns.
//
//	p_out_numBytesConsumed:
//		This returns the number of body bytes bytes consumed in the inflation.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_NullPointer:
//		if p_bytes is NULL
//
//	MWRES_BadParamValue
//		if numBodyBytes < 1
//
//====================================================================
*/
TMW_ResultCode MWDEFLATECODEC_inflateWSIRLEProductBodyBlock
(
	TFourByteInteger_Unsigned numBodyBytes,
	TByte_Unsigned* p_bytes,
	TByte_Unsigned bool_EndOfInflationIsEndOfProduct,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed
);



/*
//====================================================================
// MWDEFLATECODEC_decompressDeflatedWSIRLEProductBodyBlockIntoRaster
//
// This method handles the INFLATION of a product body block and
// then passes the INFLATED bytes through a WSI RLE deccoder. The
// resulting pixels are written into the given raster.
// The configured output encoder is not used.
//
//
// PARAMETERS:
//
//	numBodyBytes:
//		This indicates the number of deflated body bytes in the caller's
//		array.  This is assumed to be the number of bytes in the
//		deflated portion of the product block.
//
//	p_bytes:
//		This is the array of bytes holding the product body, starting at
//		the first byte of the deflated data.
//
//	p_out_numBytesConsumed:
//		This returns the number of body bytes bytes consumed in the
//		inflation.
//
//	numPixelsInDestination:
//		The number of pixels expected in the destination raster
//
//	p_destinationRaster:
//		Pointer to the upper left pixel of the destination raster.
//
//	p_out_numPixelWritten:
//		Pointer to a variable which indicates the number of pixels written
//		to the destination raster.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_NullPointer:
//		if p_bytes is NULL
//
//	MWRES_BadParamValue
//		if numBodyBytes < 1
//
//	MWRes_DataStructureFull:
//		If the number of pixels would exceed numPixelsInDestination
//
//====================================================================
*/
TMW_ResultCode MWDEFLATECODEC_decompressDeflatedWSIRLEProductBodyBlockIntoRaster
(
	TFourByteInteger_Unsigned numBodyBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed,
	TFourByteInteger_Unsigned numPixelsInDestination,
	TByte_Unsigned* p_destinationRaster,
	TFourByteInteger_Unsigned* p_out_numPixelsWritten
);



/* end of header wrapper */
#endif

