/*
//====================================================================
// MW_DEFLATECODEC.c:
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/
#include "MW_DataFieldTypes.h"

/* Interface of ADT used to handle zlib Inflation */
#include "MW_ZLib.h"

// For passing through RLE decoder before output
#include "MW_WSIRLE.h"

/* Output target multiplexor*/
#include "MW_OutputEncoder.h"

/*
//====================================================================
// ADT internal data
//====================================================================
*/

/*
// Max ever seen for any of the products passed through here is 65KB.
// Lets use 80KB to be safe
*/
#define NUM_ZLIB_INFLATION_HEAP_BYTES 81920

/* Number of bytes to hold in the decompression buffer at any one time */
#define NUM_DECOMPRESSION_BUFFER_BYTES 1024


/* The decompression buffer */
static TByte_Unsigned decompressionBuffer[NUM_DECOMPRESSION_BUFFER_BYTES];

/* The memory manager for zlib */
static TMW_MemorySpaceManager inflationMemoryManager;
static TByte_Unsigned ZLibInflationHeap[NUM_ZLIB_INFLATION_HEAP_BYTES];

/* The ADT instance used for zlib */
static TMWZLib_SessionStructure inflationADT;

#if 0
/* Has the ZLib session been started */
static TByte_Unsigned bool_zlibHeaderRead = 0;
#endif

/*
//====================================================================
// Public methods
//====================================================================
*/

TMW_ResultCode MWDEFLATECODEC_inflateProductBodyBlock
(
	TFourByteInteger_Unsigned numBodyBytes,
	TByte_Unsigned* p_bytes,
	TByte_Unsigned bool_EndOfInflationIsEndOfProduct,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed
)
{
	TMW_ResultCode MWRes;

	TFourByteInteger_Unsigned numBytesInflated;
	TFourByteInteger_Unsigned numBytesInflatedTotal;
	TFourByteInteger_Unsigned numBytesConsumedThisRound;
	TByte_Unsigned bool_endOfBodyInThisOutput;
	TByte_Unsigned bool_endOfProductInThisOutput;
	TMW_ResultCode outputRes;

	/*
	//##############################################################
	// Set up inflation heap
	//##############################################################
	*/
	MWRes = MemorySpaceManager_initialize
	(
		&inflationMemoryManager,
		NUM_ZLIB_INFLATION_HEAP_BYTES,
		ZLibInflationHeap,
		0
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Set up inflator
	//##############################################################
	*/
	MWRes = MWZLib_inflateStart
	(
		&inflationADT,
		numBodyBytes,
		p_bytes,
		&inflationMemoryManager
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Start output encoder
	//##############################################################
	*/
	MWRes = MWOutputEncoder_startNewEncodingBlock();
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Inflate bytes and send them through the OutputDPU encoder
	// until all deflated input is consumed.
	//##############################################################
	*/
	numBytesInflatedTotal = 0;
	bool_endOfBodyInThisOutput = 0;
	bool_endOfProductInThisOutput = 0;

	while (1)
	{

		/* Inflate the bytes */
		numBytesInflated = 0;
		numBytesConsumedThisRound = 0;
		MWRes = MWZLib_inflateBytes
		(
			&inflationADT,
			NUM_DECOMPRESSION_BUFFER_BYTES,
			decompressionBuffer,
			&numBytesInflated,
			&numBytesConsumedThisRound
		);

		/* Add bytes inflated to total */
		numBytesInflatedTotal += numBytesInflated;

		/* Add number of bytes consumed this round to the total */
		*p_out_numBytesConsumed = (*p_out_numBytesConsumed) + numBytesConsumedThisRound;

		/* Was end of product reached? */
		if
		(
			(MWRes == MWRes_EndOfDeflatedData)
		)
		{
			// This is the end of this product block
			bool_endOfBodyInThisOutput = 1;

			// Is it the end of the entire product?
			if (bool_EndOfInflationIsEndOfProduct != 0)
			{
				bool_endOfProductInThisOutput = 1;
			}

			/* End inflation session */
			MWZLib_inflateEnd
			(
				&inflationADT
			);

			/* Reset memory manager */
			MemorySpaceManager_reset
			(
				&inflationMemoryManager
			);

		}

		/*
		//##############################################################
		// Handle non-error cases by outputting results
		//##############################################################
		*/
		if
		(
			(MWRes == MWRes_Success) ||
			(MWRes == MWRes_EndOfDeflatedData )
		)
		{

			/* Output inflated bytes */
			outputRes = MWOutputEncoder_outputEncodedBytes
			(
				numBytesInflated,
				decompressionBuffer,
				bool_endOfBodyInThisOutput,
				bool_endOfProductInThisOutput
			);
			if (outputRes != MWRes_Success)
			{
				return MWRes;
			}

			/* If it was the last data, we are done */
			if (MWRes == MWRes_EndOfDeflatedData)
			{
				return MWRes_Success;
			}
		}
		else
		{
			/*
			// An error occured
			// MWRes_InputBufferExhausted is included here because we have supposedly
			// made available all the deflated bytes for inflation.
			*/

			return MWRes;
		}

	}

}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWDEFLATECODEC_inflateWSIRLEProductBodyBlock
(
	TFourByteInteger_Unsigned numBodyBytes,
	TByte_Unsigned* p_bytes,
	TByte_Unsigned bool_EndOfInflationIsEndOfProduct,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed
)
{
	TMW_ResultCode MWRes;

	TFourByteInteger_Unsigned numBytesInflated;
	TFourByteInteger_Unsigned numBytesInflatedTotal;
	TFourByteInteger_Unsigned numBytesConsumedThisRound;
	TByte_Unsigned bool_endOfBodyInThisOutput;
	TByte_Unsigned bool_endOfProductInThisOutput;
	TFourByteInteger_Unsigned numInflatedBytesConsumed;
	TFourByteInteger_Unsigned numInflatedBytesLeftOverFromLastRound;
	TMW_ResultCode outputRes;

	/*
	//##############################################################
	// Set up inflation heap
	//##############################################################
	*/
	MWRes = MemorySpaceManager_initialize
	(
		&inflationMemoryManager,
		NUM_ZLIB_INFLATION_HEAP_BYTES,
		ZLibInflationHeap,
		0
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Set up inflator
	//##############################################################
	*/
	MWRes = MWZLib_inflateStart
	(
		&inflationADT,
		numBodyBytes,
		p_bytes,
		&inflationMemoryManager
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Start output encoder
	//##############################################################
	*/
	MWRes = MWOutputEncoder_startNewEncodingBlock();
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Inflate bytes and send them through the WSI RLE decoder
	// until all deflated input is consumed.
	//##############################################################
	*/
	numBytesInflatedTotal = 0;
	bool_endOfBodyInThisOutput = 0;
	bool_endOfProductInThisOutput = 0;
	numInflatedBytesLeftOverFromLastRound = 0;

	while (1)
	{

		/*
		Inflate the bytes. We may have a partial run left at the beginning
		of the buffer, so don't overwrite it.
		*/

		numBytesInflated = 0;
		numBytesConsumedThisRound = 0;
		MWRes = MWZLib_inflateBytes
		(
			&inflationADT,
			NUM_DECOMPRESSION_BUFFER_BYTES - numInflatedBytesLeftOverFromLastRound,
			decompressionBuffer + numInflatedBytesLeftOverFromLastRound,
			&numBytesInflated,
			&numBytesConsumedThisRound
		);

		/* Add bytes inflated to total */
		numBytesInflatedTotal += numBytesInflated;

		/* Also process bytes left over from last round */
		numBytesInflated += numInflatedBytesLeftOverFromLastRound;

		/* Add bytes consumed to the total*/
		*p_out_numBytesConsumed = (*p_out_numBytesConsumed) + numBytesConsumedThisRound;

		/* Was end of product reached? */
		if
		(
			(MWRes == MWRes_EndOfDeflatedData)
		)
		{
			// This is the end of this product block
			bool_endOfBodyInThisOutput = 1;

			// Is it the end of the entire product?
			if (bool_EndOfInflationIsEndOfProduct != 0)
			{
				bool_endOfProductInThisOutput = 1;
			}

			/* End inflation session */
			MWZLib_inflateEnd
			(
				&inflationADT
			);

			/* Reset memory manager */
			MemorySpaceManager_reset
			(
				&inflationMemoryManager
			);

		}

		/*
		//##############################################################
		// Handle non-error cases by outputting results
		//##############################################################
		*/
		if
		(
			(MWRes == MWRes_Success) ||
			(MWRes == MWRes_EndOfDeflatedData )
		)
		{

			/* Output inflated bytes, including those leftover from
			   last round
			*/
			numInflatedBytesConsumed = 0;
			outputRes = MWWSIRLE_decompressPartialProductBodyBlock
			(
				numBytesInflated,
				decompressionBuffer,
				&numInflatedBytesConsumed,
				bool_endOfBodyInThisOutput,
				bool_endOfProductInThisOutput
			);
			if (outputRes == MWRes_InputBufferExhausted)
			{
				/*
				// Some bytes at the end of the inflated data
				// are a partial run. We need to copy them
				// down to the beginning of the inflated buffer
				// to prepend the next inflated data.
				// Due to the way the RLE scheme works, there should
				// never be more than 3 of them.
				*/


#ifdef HAS_MEMCPY
				numInflatedBytesLeftOverFromLastRound =
					numBytesInflated - numInflatedBytesConsumed;

				memcpy (decompressionBuffer, decompressionBuffer + numInflatedBytesConsumed, numInflatedBytesLeftOverFromLastRound);
#else
				numInflatedBytesLeftOverFromLastRound = 0;
				for (; numInflatedBytesConsumed < numBytesInflated; numInflatedBytesConsumed ++)
				{
					decompressionBuffer[numInflatedBytesLeftOverFromLastRound] = decompressionBuffer[numInflatedBytesConsumed];
					numInflatedBytesLeftOverFromLastRound ++;
				}
#endif


			}
			else if (outputRes != MWRes_Success)
			{
				return MWRes;
			}

			/* If it was the last data, we are done */
			if (MWRes == MWRes_EndOfDeflatedData)
			{
				return MWRes_Success;
			}
		}
		else
		{
			/*
			// An error occured
			// MWRes_InputBufferExhausted is included here because we have supposedly
			// made available all the deflated bytes for inflation.
			*/

			return MWRes;
		}

	}

}

/*
//--------------------------------------------------------------------
*/

TMW_ResultCode MWDEFLATECODEC_decompressDeflatedWSIRLEProductBodyBlockIntoRaster
(
	TFourByteInteger_Unsigned numBodyBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed,
	TFourByteInteger_Unsigned numPixelsInDestination,
	TByte_Unsigned* p_destinationRaster,
	TFourByteInteger_Unsigned* p_out_numPixelsWritten
)
{
	TMW_ResultCode MWRes;

	TFourByteInteger_Unsigned numBytesInflated;
	TFourByteInteger_Unsigned numBytesInflatedTotal;
	TFourByteInteger_Unsigned numBytesConsumedThisRound;
	TByte_Unsigned bool_endOfBodyInThisOutput;
	TByte_Unsigned bool_endOfProductInThisOutput;
	TFourByteInteger_Unsigned numInflatedBytesConsumed;
	TFourByteInteger_Unsigned numInflatedBytesLeftOverFromLastRound;
	TMW_ResultCode outputRes;

	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if ((p_destinationRaster == NULL) || (p_out_numPixelsWritten == NULL))
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Starting new raster so reset RLE encoder state
	//##############################################################
	*/
	MWRes = MWWSIRLE_beginRaster();
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Set up inflation heap
	//##############################################################
	*/
	MWRes = MemorySpaceManager_initialize
	(
		&inflationMemoryManager,
		NUM_ZLIB_INFLATION_HEAP_BYTES,
		ZLibInflationHeap,
		0
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}

	/*
	//##############################################################
	// Set up inflator
	//##############################################################
	*/
	MWRes = MWZLib_inflateStart
	(
		&inflationADT,
		numBodyBytes,
		p_bytes,
		&inflationMemoryManager
	);
	if (MWRes != MWRes_Success)
	{
		return MWRes;
	}


	/*
	//##############################################################
	// Inflate bytes and send them through the WSI RLE decoder
	// into the destination raster until all deflated input is
	// consumed or we get an error.
	//##############################################################
	*/
	numBytesInflatedTotal = 0;
	bool_endOfBodyInThisOutput = 0;
	bool_endOfProductInThisOutput = 0;
	numInflatedBytesLeftOverFromLastRound = 0;

	*p_out_numPixelsWritten = 0;

	while (1)
	{

		/*
		Inflate the bytes. We may have a partial run left at the beginning
		of the buffer, so don't overwrite it.
		*/

		numBytesInflated = 0;
		numBytesConsumedThisRound = 0;
		MWRes = MWZLib_inflateBytes
		(
			&inflationADT,
			NUM_DECOMPRESSION_BUFFER_BYTES - numInflatedBytesLeftOverFromLastRound,
			decompressionBuffer + numInflatedBytesLeftOverFromLastRound,
			&numBytesInflated,
			&numBytesConsumedThisRound
		);

		/* Add bytes inflated to total */
		numBytesInflatedTotal += numBytesInflated;

		/* Also process bytes left over from last round */
		numBytesInflated += numInflatedBytesLeftOverFromLastRound;

		/* Add bytes consumed to the total*/
		*p_out_numBytesConsumed = (*p_out_numBytesConsumed) + numBytesConsumedThisRound;

		/* Was end of product reached? */
		if
		(
			(MWRes == MWRes_EndOfDeflatedData)
		)
		{
			// This is the end of this product block
			bool_endOfBodyInThisOutput = 1;

			/* End inflation session */
			MWZLib_inflateEnd
			(
				&inflationADT
			);

			/* Reset memory manager */
			MemorySpaceManager_reset
			(
				&inflationMemoryManager
			);

		}

		/*
		//##############################################################
		// Handle non-error cases by outputting results
		//##############################################################
		*/
		if
		(
			(MWRes == MWRes_Success) ||
			(MWRes == MWRes_EndOfDeflatedData )
		)
		{

			/* Output inflated bytes, including those leftover from
			   last round
			*/

			numInflatedBytesConsumed = 0;
			outputRes = MWWSIRLE_decompressPartialProductBodyBlockToRaster
			(
				numBytesInflated,
				decompressionBuffer,
				&numInflatedBytesConsumed,
				numPixelsInDestination,
				p_destinationRaster,
				p_out_numPixelsWritten
			);
			if (outputRes == MWRes_InputBufferExhausted)
			{
				/*
				// Some bytes at the end of the inflated data
				// are a partial run. We need to copy them
				// down to the beginning of the inflated buffer
				// to prepend the next inflated data.
				// Due to the way the RLE scheme works, there should
				// never be more than 3 of them.
				*/


#ifdef HAS_MEMCPY
				numInflatedBytesLeftOverFromLastRound =
					numBytesInflated - numInflatedBytesConsumed;

				memcpy (decompressionBuffer, decompressionBuffer + numInflatedBytesConsumed, numInflatedBytesLeftOverFromLastRound);
#else
				numInflatedBytesLeftOverFromLastRound = 0;
				for (; numInflatedBytesConsumed < numBytesInflated; numInflatedBytesConsumed ++)
				{
					decompressionBuffer[numInflatedBytesLeftOverFromLastRound] = decompressionBuffer[numInflatedBytesConsumed];
					numInflatedBytesLeftOverFromLastRound ++;
				}
#endif

				/* Test state, this shouldn't happen at end of inflated data */
				if (bool_endOfBodyInThisOutput != 0)
				{
					return MWRes_ProductContentsInvalid;
				}


			}
			else if (outputRes != MWRes_Success)
			{
				return MWRes;
			}
			else
			{
				// No bytes leftover
				numInflatedBytesLeftOverFromLastRound  = 0;
			}

			/* If it was the last data, we are done */
			if (MWRes == MWRes_EndOfDeflatedData)
			{
				return MWRes_Success;
			}
		}
		else
		{
			/*
			// An error occured
			// MWRes_InputBufferExhausted is included here because we have supposedly
			// made available all the deflated bytes for inflation.
			*/

			return MWRes;
		}

	}


}


