/*
====================================================================
// MW_BitFieldOperators.c
//
// Implementation of bit aligned field utility functions.
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//
====================================================================
*/


/*
====================================================================
 #includes
====================================================================
*/

#include "MW_BitFieldOperators.h"


/*
====================================================================
// Defines
====================================================================
*/

/*
// If this is defined, then multi-byte bit fields are converted to little endian
// byte order before being returned to the caller.
*/
#ifdef OSAL_CPU_LITTLE_ENDIAN
#define LITTLE_ENDIAN_MACHINE
#elif defined OSAL_CPU_BIG_ENDIAN
#else
    #error Error! You must define OSAL_CPU_LITTLE_ENDIAN or OSAL_CPU_BIG_ENDIAN
#endif


/*
====================================================================
====================================================================
// Internal methods
====================================================================
====================================================================
*/

/*
====================================================================
// readBitFieldByte
//
// This method reads up to a single byte of a bit field. It is
// used by readBitField to read in the seperate bytes for each
// 8 bits in the input field.
//
// PARAMETERS:
//
//	p_fieldStorage:
//		Pointer to the byte in which the bits are to be stored
//
//	numBitsInField:
//		The number of bits in the field. This must be >= 1 and <= 8
//
//	numSourceBits:
//		The number of bits of input available for reading
//
//	p_sourceBits:
//		The array of bytes containing the bits for reading
//
//	inout_p_bitsConsumed:
//		A pointer to a counter which tracks how many bits
//		of input were alredy consumed before this call. It is
//		modified by this call to reflect any bits consumed
//		by this operation.
//
// RETURNS:
//
//	MWRes_Success
//		If the field was read.
//
//	MWRes_InputBufferExhausted:
//		If there were not enough input bits.
//
//	MWRes_BadParamValue:
//		If numBitsInField > 8 or < 1
//
====================================================================
*/

TMW_ResultCode readBitFieldByte
(
	// Field storage and length
	TByte_Unsigned* p_fieldStorage,
	TFourByteInteger_Unsigned numBitsInField,
	// Input source and length
	TFourByteInteger_Unsigned numSourceBits,
	TByte_Unsigned* p_sourceBits,
	// Current state
	TFourByteInteger_Unsigned* inout_p_bitsConsumed
)
{


	TFourByteInteger_Unsigned numBitsReadIntoField;

	TFourByteInteger_Unsigned inputByteIndex;
	TFourByteInteger_Unsigned inputBitOffset;
	TFourByteInteger_Unsigned bitsLeftInField;
	TFourByteInteger_Unsigned bitsToMove;
	//TFourByteInteger_Unsigned outputByteIndex;
	//TFourByteInteger_Unsigned outputBitOffset;

	TByte_Unsigned inputMask;
	TByte_Unsigned inputByte;

	/*
	################################################################
	// Test parameters
	################################################################
	*/
	if ((numBitsInField < 1) || (numBitsInField > 8))
	{
		return MWRes_BadParamValue;
	}

	/*
	##############################################################
	// How many bits do we need? Reduce this by the number of
	// bits available if that is limiting.
	##############################################################
	*/
	numBitsReadIntoField = 0;

	/*
	##############################################################
	// Start with empty output byte
	##############################################################
	*/
	*p_fieldStorage = 0;

	/*
	##############################################################
	// Enough bits left?
	##############################################################
	*/
	if (numSourceBits - (*inout_p_bitsConsumed) < numBitsInField)
	{
		/* Not enough input */
		return MWRes_InputBufferExhausted;

	}

	while (numBitsReadIntoField < numBitsInField)
	{
		/*
		##############################################################
		// Determine byte index of input and bit offset of input
		##############################################################
		*/
		inputByteIndex = (*inout_p_bitsConsumed) / 8;
		inputBitOffset = (*inout_p_bitsConsumed) % 8;

		/*
		##############################################################
		// How many bits should we move from this byte to the next
		##############################################################
		*/
		bitsLeftInField = numBitsInField - numBitsReadIntoField;

		bitsToMove = bitsLeftInField;
		if (bitsToMove > (numSourceBits - (*inout_p_bitsConsumed)) )
		{
			bitsToMove = numSourceBits - (*inout_p_bitsConsumed);
		}

		if (bitsToMove + inputBitOffset >= 8)
		{
			bitsToMove = 8 - inputBitOffset;
		}

		/* Prepare input mask */
		inputMask = 0xFF;
		inputMask >>= (8-bitsToMove); // LSB aligned # of bits
		inputMask <<= (8-inputBitOffset- bitsToMove  ); // In location of input

		/*
		// Read input into temp byte using mask
		// Only the bits to be read can be 1
		*/
		inputByte = p_sourceBits[inputByteIndex] & inputMask;

		/*
		// Shift the temp byte to align it with the area of the
		// field left to be read. Remember, we are reading in from
		// the input in MSB to LSB order.
		// First bit of input byte is at inputBitOffset from MSB
		// First bit is supposed to align with (8-bitsLeftInField)
		//   from MSB. Shift input by the difference
		*/
		if (inputBitOffset < (8-bitsLeftInField))
		{
			inputByte >>= ((8-bitsLeftInField) - inputBitOffset);
		}
		else
		{
			inputByte <<= (inputBitOffset - (8-bitsLeftInField));
		}

		/* Store what we got */
		*p_fieldStorage |= inputByte;

		/*
		##############################################################
		// Note number of bits consumed of input and
		// written of output
		##############################################################
		*/

		/* Output written */
		numBitsReadIntoField += bitsToMove;

		/* Input consumed */
		(*inout_p_bitsConsumed) = (*inout_p_bitsConsumed) + bitsToMove;
	}

	/*
	##############################################################
	// Done
	##############################################################
	*/

	return MWRes_Success;

}

/*
====================================================================
====================================================================
* Public methods
====================================================================
====================================================================
*/

TMW_ResultCode signExtendBitField
(
	TByte_Unsigned* p_fieldBytes,
	TFourByteInteger_Unsigned numBitsInField
)
{
	/* Number of bytes in the destination field */
	TFourByteInteger_Unsigned numBytesOfField;
	TByte_Unsigned bitModulus;
	TFourByteInteger_Unsigned indexOfMSBByte;
	TByte_Unsigned SignMask = 0x01;


	/*
	//##############################################################
	// How many bytes are in the field
	//##############################################################
	*/

	numBytesOfField = numBitsInField / 8;
	bitModulus = (TByte_Unsigned) (numBitsInField %8);
	if (bitModulus > 0)
	{
		numBytesOfField ++;
	}
	else
	{
		/* Don't need to sign extend if its MSB byte fills an entire byte*/
		return MWRes_Success;
	}

	/*
	//##############################################################
	// Get MSB byte
	//##############################################################
	*/

#ifdef LITTLE_ENDIAN_MACHINE
	indexOfMSBByte = numBytesOfField-1;
#else
	indexOfMSBByte = 0;
#endif

	SignMask = 0x01 << (bitModulus-1);

	if ((SignMask & p_fieldBytes[indexOfMSBByte]) != 0)
	{
		/* Sign extend 1s */
		for (SignMask = SignMask << 1; SignMask != 0; SignMask = SignMask << 1)
		{
			p_fieldBytes[indexOfMSBByte] |= SignMask;
		}
	}

	//##############################################################
	// Done
	//##############################################################
	return MWRes_Success;


}

//--------------------------------------------------------------------

TMW_ResultCode readBitField
(
	/* Field storage and length */
	TByte_Unsigned* p_fieldStorage,
	TFourByteInteger_Unsigned numBitsInField,
	/* Input source and length */
	TFourByteInteger_Unsigned numSourceBits,
	TByte_Unsigned* p_sourceBits,
	/* Current state */
	TFourByteInteger_Unsigned* inout_p_bitsConsumed
)
{
	TMW_ResultCode MWRes;
	TFourByteInteger_Unsigned numBitsReadIntoField;

	/* Number of bytes in the destination field */
	TFourByteInteger_Unsigned numBytesOfField;
	TByte_Unsigned bitModulus;

	TFourByteInteger_Unsigned firstByteIndex;
	TFourByteInteger_Unsigned fieldByteIndex;

	TByte_Unsigned NumBitsInThisByteEver;
	TByte_Unsigned NumBitsInThisByteBeforeRead;
	TByte_Unsigned NumBitsInThisByteCurrently;

#ifdef LITTLE_ENDIAN_MACHINE
	TByte_Unsigned tempByte;
#endif

	/*
	##############################################################
	// No bits read yet
	##############################################################
	*/

	numBitsReadIntoField = 0;

	/*
	##############################################################
	// Test parameters
	##############################################################
	*/
	if (numBitsInField < 1)
	{
		return MWRes_BadParamValue;
	}

	/*
	##############################################################
	// How many input bytes are there?
	##############################################################
	*/

	numBytesOfField = numBitsInField / 8;
	bitModulus = (TByte_Unsigned) (numBitsInField %8);
	if (bitModulus > 0)
	{
		numBytesOfField ++;
	}

	/*
	##############################################################
	// What is our first byte offset, because we may have already
	// read in a part of this field
	##############################################################
	*/

	firstByteIndex = 0;
	if (numBitsReadIntoField >= bitModulus)
	{
		// Modulus bits are in first byte, beyond that are in second byte and onward
		firstByteIndex = ((numBitsReadIntoField - bitModulus)  / 8);
		if (bitModulus != 0)
		{
			firstByteIndex += 1;
		}
	}

	/*
	// Because we are reading into an LSB alligned field, the remainder
	// bits > 8 are always in the first byte.
	*/

	for (fieldByteIndex = firstByteIndex; fieldByteIndex < numBytesOfField; fieldByteIndex ++)
	{
		/*
		##############################################################
		// How many bits to get left in this byte of field and how
		// many are there total in this byte of the field.
		##############################################################
		*/

		NumBitsInThisByteEver = 8;
		NumBitsInThisByteBeforeRead = 0;

		/*
		// How many bits go into this byte for LSB alignment of field
		// as a Big Endian number
		*/

		if ((fieldByteIndex == 0) && (bitModulus != 0))
		{
			NumBitsInThisByteEver = bitModulus;
			NumBitsInThisByteBeforeRead = (TByte_Unsigned) numBitsReadIntoField;
		}
		else
		{
			NumBitsInThisByteEver = 8;
			NumBitsInThisByteBeforeRead = (TByte_Unsigned) ((numBitsReadIntoField - bitModulus) %8);
		}


		/* Call single byte version */
		NumBitsInThisByteCurrently = NumBitsInThisByteBeforeRead;
		MWRes = readBitFieldByte
		(
			&(p_fieldStorage[fieldByteIndex]),
			NumBitsInThisByteEver,
			numSourceBits,
			p_sourceBits,
			inout_p_bitsConsumed
		);

		/* Update number of bits in this byte currently */
		NumBitsInThisByteCurrently = NumBitsInThisByteEver;

		/* Update the number of bits read into the field, in total */
		numBitsReadIntoField +=
			(NumBitsInThisByteCurrently - NumBitsInThisByteBeforeRead);

		/* Failure? */
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

	} /* Read next byte (or partial byte) of the bit field */

#ifdef LITTLE_ENDIAN_MACHINE

	/*
	##############################################################
	// Convert big endian to little endian
	##############################################################
	*/
	if (numBytesOfField > 1)
	{
		/*
		// Go from HILO to LOHI order
		// Swap pairs of bytes. If there is an odd number of bytes, one in the
		// middle gets left alone
		*/
		for (fieldByteIndex = 0; fieldByteIndex < (numBytesOfField/2); fieldByteIndex ++)
		{
			tempByte = p_fieldStorage[fieldByteIndex];
			p_fieldStorage[fieldByteIndex] = p_fieldStorage[numBytesOfField-(fieldByteIndex+1)];
			p_fieldStorage[numBytesOfField-(fieldByteIndex+1)] = tempByte;
		}

	}
#endif

	/*
	##############################################################
	// Success
	##############################################################
	*/

	return MWRes_Success;

}
