/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_time.c
* \author Leslie French
* \date 8/20/2013
* \brief This file contain functions related to time operations. Please, refer
*        to \ref sxe_time_page for implementation details. 
*
* \page sxe_time_page Time
*
* Operating systems and hardware will vary in their support of time-of-day functions.
* To simplify implementations, and provide a common framework for all modules,
* the SXe SDK uses only the SXi radio Time monitor as its source of time information. 
* The time indications are only accurate to the nearest minute, but this resolution is
* quite adequate for the broad-range of timeouts (typically 20-30 minutes or longer)
* handled by the modules.  More accurate timing may be required by the application,
* but this is outside the scope of the SXe SDK.
*
* The internal value is stored as "minutes since the start of 2012" which can easily
* be converted into standard unixtime, or another representation. The timestamps
* are in the local time of the radio.
*
*******************************************************************************/
#include "sxm_build.h"

#include <sxm_sdk.h>
#if (SXM_USE_GEN8)
#include <sxi8/sxm_sxi8_internal.h>
#else
#include <sxi/sxm_sxi_internal.h>
#endif

#include <util/sxm_time_internal.h>


#if defined(SXM_USE_SMS)
    #include "osal.h"
#endif

#ifndef WIN32
    #include <sys/time.h>
    #include <unistd.h>
    #include <sys/select.h>
#else
    #include <time.h>
    #include <Windows.h>
#endif

/** Define the offset in seconds between SXe own time
 *  and Unix one.
 */
#define  SXM_SXE_TIME_TO_UNIX_OFFSET (1325376000)

/** Number of days in 4 years */
#define DAYS_CYCLE (366 + 365 * 3)

/** Number of days starting from 2012 (2012 - 2012 = 0, 2013 - 2012 = 366 and etc) */
static uint dsf[] = {0, 366, 366 + 365, 366 + 365 + 365};

/** Number of days in months incremented in a sequence
 * 31 28 31 30 31 30 31 31 30 31 30 31
 */
static uint msf[] = {
      0,  31,  59,  90,
    120, 151, 181, 212,
    243, 273, 304, 334
};

/** Time bias in minutes (set by sxm_sxe_time_set_epoch_bias)
 * \warning This value is used only for development and code verification
 * purposes
 */
static uint sxeTimeBias = 0U;

/** Defines SXe time offset to get local time from UTC */
static uint utcOffset = 0U;

/***************************************************************************//**
* This function returns local time
*
* \return Value of local time in minutes since 00:00 Jan 1, 2012
*         0 in case of an error
*
*******************************************************************************/
SXESDK_INTERNAL_API uint sxm_sxe_time_now(void)
{
#ifndef SXM_USE_SMS
    SXMTime t;
    uint time = 0U;
    if (sxm_sxi_extract_time(&t) == SXM_E_OK) {
        time = sxm_sxe_time_encode(&t) + sxeTimeBias - utcOffset; 
    }
    return time;
#else
    UN32 un32Sec;

    if (OSAL.eTimeGet(&un32Sec) == OSAL_SUCCESS) {
        // Convert from seconds since 00:00:00 UTC January 1, 1970 to 
        // minutes since 00:00:00 UTC January 1, 2012 (1325376000)
        un32Sec -= SXM_SXE_TIME_TO_UNIX_OFFSET;
    }
    else {
        un32Sec = 0U;
    }

    return (un32Sec / SECONDS_IN_MINUTE) + sxeTimeBias - utcOffset;
#endif
}

/***************************************************************************//**
* This function returns UTC time
*
* \return Value of UTC time in minutes since 00:00 Jan 1, 2012
*         0 in case of an error
*
*******************************************************************************/
SXESDK_INTERNAL_API uint sxm_sxe_time_now_utc(void)
{
#ifndef SXM_USE_SMS
    SXMTime t;
    uint time;
    /* get the current time */
    if (sxm_sxi_extract_time(&t) == SXM_E_OK) {
        /* encode the time */
        time = sxm_sxe_time_encode(&t) + sxeTimeBias;
    }
    else {
        time = 0U;
    }
    return time;
#else
    UN32 un32Sec;

    if (OSAL.eTimeGet(&un32Sec) == OSAL_SUCCESS) {
        // Convert from seconds since 00:00:00 UTC January 1, 1970 to 
        // minutes since 00:00:00 UTC January 1, 2012 (1325376000)
        un32Sec -= SXM_SXE_TIME_TO_UNIX_OFFSET;
    }
    else {
        un32Sec = 0U;
    }

    return (un32Sec / SECONDS_IN_MINUTE) + sxeTimeBias;
#endif
}

/***************************************************************************//**
* Converts SXMTime to time in minutes since 00:00 Jan 1, 2012
*
* \param[in] t address of the SXMTine structure the caller wished to encode
*
* \return Time in minutes since 00:00 Jan 1, 2012
*
*******************************************************************************/
SXESDK_INTERNAL_API uint sxm_sxe_time_encode(SXMTime *t)
{
    uint lc, li;
    uint days;

    lc = (t->year - 2012U) / 4U;
    li = (t->year - 2012U) & 3U;
    days = DAYS_CYCLE * lc + dsf[li] + msf[t->mon-1] + ((li == 0) && (t->mon > 2)) + t->day - 1;
    return (days * 24 + t->hour) * 60 + t->min;
}

/***************************************************************************//**
* Converts SXMTime to local time in minutes since 00:00 Jan 1, 2012
*
* \param[in] t address of the SXMTine structure the caller wished to encode
*
* \return Local time in minutes since 00:00 Jan 1, 2012
*
*******************************************************************************/
SXESDK_INTERNAL_API uint sxm_sxe_time_encode_local(SXMTime *t)
{
    return (sxm_sxe_time_encode(t) + sxeTimeBias - utcOffset);
}

/*******************************************************************************
* Converts time in minutes since 00:00 Jan 1, 2012 to SXMTime format
*
* \param[in] now time in minutes,
* \param[out] t struct to store result
*
*******************************************************************************/
SXESDK_INTERNAL_API void sxm_sxe_time_decode(uint now, SXMTime *t)
{
    uint tmin = now % MINUTES_IN_HOUR;
    uint thour = (now / MINUTES_IN_HOUR) % HOURS_IN_DAY;
    uint tday = now / MINUTES_IN_DAY;
    uint lc = tday / DAYS_CYCLE;
    uint rest = tday % DAYS_CYCLE;
    uint yy = 0U;
    int mm = 0;

    memset(t, 0, sizeof(SXMTime));

    while (yy < 3U)
    {
        if (rest >= dsf[yy + 1U])
        {
            ++yy;
        }
        else
        {
            break;
        }
    }
    rest -= dsf[yy];

    while (mm < 11)
    {
        if (rest >= (msf[mm + 1] + ((yy == 0U) && (mm > 0))))
        {
            ++mm;
        }
        else
        {
            break;
        }
    }

    rest -= (msf[mm] + ((yy == 0U) && (mm > 1)));

    t->year = (ushort)(2012U + lc * 4U + yy);
    t->mon = (byte)(mm + 1);
    t->day = (byte)(rest + 1U);
    t->hour = (byte)thour;
    t->min = (byte)tmin;
    t->sec = 0;
}

/***************************************************************************//**
* This function returns absolute difference between two times in milliseconds
*
* \note the \p t1 shall be later than \p t2
*
* \param[in] t1 the first time
* \param[in] t2 the second time
*
* \return difference in milliseconds
*
*******************************************************************************/
SXESDK_INTERNAL_API time_t sxm_time_get_absolute_time_diff(const SXMTimeSpec *t1,
                                                           const SXMTimeSpec *t2)
{
    time_t sec = t1->tv_sec - t2->tv_sec;
    long nsec;
    /* Does it have enough nanoseconds ? */
    if (t1->tv_nsec >= t2->tv_nsec) {
        nsec = t1->tv_nsec - t2->tv_nsec;
    }
    /* Calculate the leftover from the borrowed second */
    else if (sec > 0) {
        --sec; /* Borrow one second */
        /* Subtract the leftover */
        nsec = (long)SXM_NANOSEC_PER_SEC - (t2->tv_nsec - t1->tv_nsec); 
    }
    else {
        nsec = 0U;
    }
    /* Add half of the interval to have math rounded value up to the nearest
     * millisecond
     */
    nsec += (SXM_NANOSEC_PER_MILLISEC / 2);
    /* Convert into milliseconds */
    return (sec * SXM_MILLISEC_PER_SEC) + (time_t)(nsec / SXM_NANOSEC_PER_MILLISEC);
}

/***************************************************************************//**
* This function sets positive epoch bias for the SXe time.
*
* \param[in] epochBias Epoch bias (number of days)
*
*******************************************************************************/
SXESDK_INTERNAL_API void sxm_sxe_time_set_epoch_bias(ushort epochBias)
{
    sxeTimeBias = epochBias * (uint)MINUTES_IN_DAY;
}

/*******************************************************************************//**
 * Sets UTC offset to get device's local time. The application is responsible
 * for keeping eye on current timezone and DST to provide the current offset.
 * 
 * \note this function shall be called prior any time related operation, but
 * after the SXe is started.
 *
 * \param[in] mins  negative value as offset in minutes from UTC to local time.
 *                  The value shall be in range from -24H to 0 (aka back to UTC)
 *
 * \retval SXM_E_OK       success
 * \retval SXM_E_INVAL    invalid parameter encountered
 *******************************************************************************/
SXESDK_API int sxm_time_set_utc_offset(int mins)
{
    SXMResultCode rc;
    if ((mins < -MINUTES_IN_DAY) || (mins > 0)) {
        rc = SXM_E_INVAL;
    }
    else {
        /* The value is kept positive while the passed value
         * has to be negative
         */
        utcOffset = (uint)(-mins);
        rc = SXM_E_OK;
    }
    return rc;
}

/*******************************************************************************//**
 * Gets current UTC offset which is used for adjusting device's local time.
 *
 * \sa sxm_audio_set_utc_offset
 * \return Current UTC offset in minutes.
 * 
 *******************************************************************************/
SXESDK_API int sxm_time_get_utc_offset(void)
{
    return -((int)utcOffset);
}

/***************************************************************************//**
* This function returns local time
*
* \return Value of local time in seconds since 00:00:00 Jan 1, 1970
*         0 in case of an error
*
*******************************************************************************/
SXESDK_API time_t sxm_time_now(void)
{
    const time_t now = (time_t)sxm_sxe_time_now();
    return (now != 0) ? (now * SECONDS_IN_MINUTE + SXM_SXE_TIME_TO_UNIX_OFFSET) : 0;
}

/***************************************************************************//**
* This function returns UTC time
*
* \return Value of UTC time in seconds since 00:00:00 Jan 1, 1970
*         0 in case of an error
*
*******************************************************************************/
SXESDK_API time_t sxm_time_now_utc(void)
{
    const time_t now = (time_t)sxm_sxe_time_now_utc();
    return (now != 0) ? (now * SECONDS_IN_MINUTE + SXM_SXE_TIME_TO_UNIX_OFFSET) : 0;
}

/***************************************************************************//**
* Converts SXMTime to time in seconds since 00:00:00 Jan 1, 1970
*
* \param[in] t address of the SXMTine structure the caller wished to encode
*
* \return Time in seconds since 00:00:00 Jan 1, 1970
*
*******************************************************************************/
SXESDK_API time_t sxm_time_encode(SXMTime *t)
{
    const time_t encoded = (time_t)sxm_sxe_time_encode(t);
    return encoded * SECONDS_IN_MINUTE +
           t->sec +
           SXM_SXE_TIME_TO_UNIX_OFFSET;
}

/***************************************************************************//**
* Converts time in seconds since 00:00:00 Jan 1, 1970 to SXMTime format
*
* \param[in] now time in seconds to decode
* \param[out] t struct to store result
*
*******************************************************************************/
SXESDK_API void sxm_time_decode(time_t now, SXMTime *t)
{
    sxm_sxe_time_decode((uint)((now - SXM_SXE_TIME_TO_UNIX_OFFSET) / SECONDS_IN_MINUTE), t);
    t->sec = (byte)(now % SECONDS_IN_MINUTE);
}
