/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_tfile.h
* \author Leslie French
* \date 8/20/2013
*
* Public header file for T-File definitions (Refer to \ref tfile_page for
* implementation details.
*
*******************************************************************************/

#ifndef SXM_TFILE_H
#define SXM_TFILE_H

#include <stdio.h>
#include <sxm_build.h>
#include <sxm_typ.h>


#ifdef __cplusplus
extern "C"
{
#endif

enum {
    /** T-File magic sequence [\p SXMT - big endian, \p TMXS - little endian] */
    SXM_TFILE_MAGIC = (('S' << 24) | ('X' << 16) | ('M' << 8) | 'T'),
    /** T-File version for modern layouts */
    SXM_TFILE_VERSION = 0x0002U,
    /** Defines the space reserved for internal needs */
    SXM_TFILE_BLOCK_RESERVED = 4U,
    /** Supported file format length */
    SXM_TFILE_FORMAT_LEN = 16U,
    /** Defines number of transactions */
    SXM_TFILE_TRANSACTION_MAX = 2U,
    /** Defines T-File header size */
    SXM_TFILE_HEADER_BYTELEN = 8192U,
    /** Defines allocation map size */
    SXM_TFILE_ALLOC_MAP_BITLEN = 16384U,
    /** Defines sign for file with dynamic size during creation only */
    SXM_TFILE_DYNAMIC_NO_OF_BLOCKS = (uint)-1
};

/** T-File Transaction structure */
typedef struct {
    uint seq;           //!< transaction sequence number
    uint bsize;         //!< block size
    uint fsize;         //!< file size
    uint bbptr;         //!< <b>CURRENTLY UNUSED</b>
    uint bbcrc;         //!< CRC of allocation map
    uint rootptr;       //!< index of user root block
    uint rootsize;      //!< size of user root block
    uint rootcrc;       //!< CRC of user root block
    uint committs;      //!< Timestamp of the transaction commit
    uint user[118];     //!< Available for user
    /** CRC of the previous bytes (until this field).
     * \warning this field must be the last field in the \ref SXMTFileBlock
     */
    uint crc;
} SXMTFileBlock;

/** T-File Allocation Map structure */
typedef struct {
    /** Allocation bit mask */
    uint bits[(SXM_TFILE_ALLOC_MAP_BITLEN / (8 /* SXM_ARCH_BITS_IN_BYTE */  * sizeof(uint)))]; 
} SXMTFileAllocMap;

/*!
 * \brief T-File Header structure
 * \note Shall be less or equal to \ref SXM_TFILE_HEADER_BYTELEN bytes
 */
typedef struct {
    SXMTFileBlock tb[SXM_TFILE_TRANSACTION_MAX];     //!< Transaction blocks
    SXMTFileAllocMap map[SXM_TFILE_TRANSACTION_MAX]; //!< Allocation maps
    SXMTFileAllocMap temp;              //!< Intermediate allocation map
    uint magic;                         //!< T-File magic sequence
    uint tver;                          //!< T-File format version
    char format[SXM_TFILE_FORMAT_LEN + 1];  //!< User dependent content format
    uint sver;                          //!< Schema version related to the format
} SXMTFileHeader;

/** \name T-File masks
 * @{
 */
enum {
    /** Nothing  is set */
    SXM_TFILE_MASK_NONE         = 0x0000, 
    /** Transaction is active */
    SXM_TFILE_MASK_TRANSACTION  = 0x0001,
    /** File is being created */
    SXM_TFILE_MASK_CREATION     = 0x0002,
    /** Modification for the transaction detected */
    SXM_TFILE_MASK_MODIFIED     = 0x0004,
    /** File is being created in dynamic mode */
    SXM_TFILE_MASK_CREATION_DYN = 0x0008
};
/** @} */

/** T-File Handle */
typedef struct {
    SXMTFileHeader *hdr; //!< transaction header
    FILE *dbfile;        //!< file pointer
    void *root;          //!< user root block
    void *root_origin;   //!< Keeps original user root block during transaction 
    uint* user_origin;   //!< Keeps original user data during transaction
    uint dataptr;        //!< id of the first available block for t-file user
    byte active;         //!< active root block (0 or 1)
    int mask;            //!< T-File descriptive mask
#if !defined(SXM_DEBUG_PRODUCTION) || defined(SDKFILES_STANDALONE_BUILD)
    /** Keeps CRC16 for user data to detect modification on commit */
    ushort crcuser[SXM_TFILE_TRANSACTION_MAX];
#endif
} SXMTFile;

/** T-File statistics*/
typedef struct {
    uint tschema;  //!< T-File own schema
    uint dschema;  //!< Service's content schema version
    uint dversion; //!< Service's content data version
    uint seq;      //!< Active transaction seq number
    uint bsize;    //!< Block size
    uint ubsize;   //!< Block size available for the user data
    uint fsize;    //!< Total number of blocks in file
    uint usize;    //!< Number of used blocks
    BOOL transaction; //!< \p TRUE if transaction in process or \p FALSE
} SXMTFileStat;

/** T-File based stream */
typedef struct _sxm_tfile_stream SXMTFileStream;

/** T-File base stream states */
typedef enum {
    SXM_TFILE_STREAM_UNKNOWN = 0, //!< Unknown state, usually means API call errors
    SXM_TFILE_STREAM_OK,          //!< So far so good
    SXM_TFILE_STREAM_NOMEM,       //!< Recently failed due to no-memory condition
    SXM_TFILE_STREAM_NOBLOCKS,    //!< Recently failed due to lack of free blocks in t-file
    SXM_TFILE_STREAM_IO           //!< Recently failed due to I/O error
} SXMTFileStreamState;

/** Helper macro to get statically calculated size of the user data accommodated
 * by the block designed as \p _s byte(s) len
 * \param[in] _s designed size of the single block
 */
#define sxm_tfile_bsize_static(_s) \
    (uint)((_s) - SXM_TFILE_BLOCK_RESERVED)

extern SXESDK_INTERNAL_API uint sxm_tfile_bsize(const SXMTFile *);

extern SXESDK_INTERNAL_API int sxm_tfile_update_version(SXMTFile *,
                                                        uint);

extern SXESDK_INTERNAL_API int sxm_tfile_stat(const SXMTFile *,
                                              SXMTFileStat *);

extern SXESDK_INTERNAL_API void sxm_tfile_close(SXMTFile *);

extern SXESDK_INTERNAL_API int sxm_tfile_cancel(SXMTFile *);

extern SXESDK_INTERNAL_API int sxm_tfile_commit(SXMTFile *);

extern SXESDK_INTERNAL_API SXMTFile *sxm_tfile_create(char,
                                           const char *,
                                           const char *,
                                           const char *,
                                           uint,
                                           uint,
                                           uint,
                                           uint,
                                           uint,
                                           int *);

extern SXESDK_INTERNAL_API SXMTFile *sxm_tfile_open(char,
                                           const char *,
                                           const char *,
                                           const char *,
                                           const char *,
                                           uint *,
                                           int *);

extern SXESDK_INTERNAL_API uint *sxm_tfile_other_user(SXMTFile *);

extern SXESDK_INTERNAL_API void *sxm_tfile_root(SXMTFile *);

extern SXESDK_INTERNAL_API uint sxm_tfile_transactions(SXMTFile *);

extern SXESDK_INTERNAL_API int sxm_tfile_start(SXMTFile *);

extern SXESDK_INTERNAL_API uint *sxm_tfile_user(SXMTFile *);

extern SXESDK_INTERNAL_API BOOL sxm_tfile_transaction_state(const SXMTFile *);

/** T-File chain of blocks */
typedef struct _sxm_tfile_chain SXMTFileChain;

extern SXESDK_INTERNAL_API int sxm_tfile_alloc(SXMTFile *,
                                               uint,
                                               SXMTFileChain **);

extern SXESDK_INTERNAL_API int sxm_tfile_read(SXMTFile *,
                                      uint,
                                      uint,
                                      SXMTFileChain **,
                                      void *);

extern SXESDK_INTERNAL_API void *sxm_tfile_alloc_and_read(SXMTFile *,
                                      uint,
                                      uint,
                                      SXMTFileChain **,
                                      int *);

extern SXESDK_INTERNAL_API int sxm_tfile_free(SXMTFile *,
                                      SXMTFileChain *);

extern SXESDK_INTERNAL_API int sxm_tfile_write(SXMTFile *,
                                      SXMTFileChain *,
                                      uint,
                                      const void *);

extern SXESDK_INTERNAL_API int sxm_tfile_check(const SXMTFile *,
                                               const SXMTFileChain *);

extern SXESDK_INTERNAL_API ushort sxm_tfile_chain_start(const SXMTFileChain *);

extern SXESDK_INTERNAL_API ushort sxm_tfile_chain_count(const SXMTFileChain *);

extern SXESDK_INTERNAL_API void sxm_tfile_chain_free(SXMTFileChain *);

extern SXESDK_INTERNAL_API int sxm_tstream_create(SXMTFile *,
                                                  SXMTFileStream **);

extern SXESDK_INTERNAL_API int sxm_tstream_commit(SXMTFileStream *, uint *, uint *);

extern SXESDK_INTERNAL_API int sxm_tstream_destroy(SXMTFileStream *);

extern SXESDK_INTERNAL_API int sxm_tstream_write(SXMTFileStream *, const void *, size_t, BOOL);

extern SXESDK_INTERNAL_API int sxm_tstream_clean(SXMTFileStream *);

extern SXESDK_INTERNAL_API size_t sxm_tstream_tell(SXMTFileStream *);

extern SXESDK_INTERNAL_API int sxm_tstream_alignup(SXMTFileStream *, size_t);

extern SXESDK_INTERNAL_API SXMTFileStreamState sxm_tstream_state(const SXMTFileStream *);

/** Puts any data type in it's binary form directly to the stream.
 * \param[in] _s valid stream instance
 * \param[in] _v value to be put
 */
#define sxm_tstream_put(_s, _v) \
    sxm_tstream_write((_s), &(_v), sizeof(_v), FALSE)

/** Puts nil-terminated string to the stream.
 * \param[in] _s valid stream instance
 * \param[in] _v string to be put
 */
#define sxm_tstream_puts(_s, _v) \
    sxm_tstream_write((_s), (_v), strlen(_v) + 1, FALSE)

/** Puts blob data with exact byte length to the stream.
 * \param[in] _s valid stream instance
 * \param[in] _v value to be put
 * \param[in] _sz size of the data addressed by \p _v
 */
#define sxm_tstream_putb(_s, _v, _sz) \
    sxm_tstream_write((_s), (_v), (_sz), FALSE)

#ifdef __cplusplus
}
#endif

#endif // SXM_TFILE_H
