/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 * \file sxm_tds.c 
 * \author David Franklin
 * \date 10/30/2016
 *
 * TDS Data Protocol functions.
 ********************************************************************************/
static unsigned int debugLevel = 0;

#define DEBUG_VAR (debugLevel)
#define DEBUG_TAG "tds"

#include <sxm_stdtrace.h>
#include <sxm_common.h>
#include <util/sxm_tds_internal.h>
#include <util/sxm_noname_internal.h>

static int sxm_tds_bitbuff_int(SXMBitBuff *pkt, void *dstBuf, UINT32 dstItemSize, UINT32 size);
static int sxm_tds_bitbuff_array(SXMBitBuff *pkt, void *dstBuf, UINT32 dstItemSize, UINT32 dstBufSize, TdsLabel *label);
static int sxm_tds_bitbuff_scode(SXMBitBuff *pkt, void *dstBuf, UINT32 dstBufSize, UINT16 lbSize);
static int sxm_tds_bitbuff_reserved(SXMBitBuff *pkt, void *dstBuf, UINT32 dstItemSize, UINT32 size);
static int sxm_tds_skip_any_label(SXMBitBuff *pkt, TdsLabel *label);

/*****************************************************************************//**
 * Reads a TDS Entry of type ARRAY.
 * 
 * This function decodes the data bitstream and reads an ARRAY type TDS entry.
 * First it validates the size of the ARRAY against the size of the data where 
 * the entry is to be stored, checking both the overall maximum array size as
 * well as the size of each array member. If the provided entry is too small then 
 * an error is returned. If it is valid then the length of the array is read and
 * validated. If it is valid then the array members are read and stored.
 *
 * \param[in] pkt           Bitstream containing received SXM data to decode.
 * \param[out] dstBuf       Provided by the OEM developer as free space to copy the selected data into.
 * \param[in] dstItemSize   Size of dstBuf members in bytes.
 * \param[in] dstBufSize    Number of items in the dstBuf array.
 *
 * \retval SXM_E_OK     Data found.
 * \retval SXM_E_ERROR  Invalid size for array members.
 * \retval SXM_E_NOMEM  Array length exceeds defined maximum.
 * \retval SXM_E_FAULT  User-provided dstBuf array is not large enough.
 *
 ********************************************************************************/
static int sxm_tds_bitbuff_array(SXMBitBuff *pkt, void *dstBuf, UINT32 dstItemSize, UINT32 dstBufSize, TdsLabel *label) {
    int rc = SXM_E_OK;

    /* Check if label is valid */
    if (label->valid == 0) {
        DEBUG_PKT("Invalid label %d. Skipping", label->lbid);
        rc = sxm_tds_skip_any_label(pkt, label);
    }
    else {
        switch (0) { default: {
            UINT32 arrayLen;
            UINT32 labelSize;
            UINT32 tempData;

            DEBUG_PKT("Processing label %d", label->lbid);
            /* calculate number of bytes needed to store each item */
            labelSize = label->lbsize / SXM_ARCH_BITS_IN_BYTE;
            if ((label->lbsize % SXM_ARCH_BITS_IN_BYTE) != 0) {
                ++labelSize;
            }
            /* if array items are larger than size of application defined array item size
               then there is an error, stop processing the message */
            if (dstItemSize < labelSize) {
                rc = SXM_E_SIZE;
                break;
            }
            /* the first LENSIZE bits of the array contain the Array Length */
            NEXT(arrayLen, label->lensize);
            if (pkt->err) {
                rc = SXM_E_BAD_PACKET;
                break;
            }
            /* make sure array length does not exceed maximum allowed */
            if (arrayLen > label->arraymax) {
                rc = SXM_E_ARRAY_SIZE;
                break;
            }
            /* process array entries */
            while (arrayLen > 0) {
                /* read data until the local storage buffer is full or the end
                   of the data is reached */
                if (dstBufSize > 0) {
                    NEXT(tempData, label->lbsize);
                    if (pkt->err) {
                        rc = SXM_E_BAD_PACKET;
                        break;
                    }
                    switch (dstItemSize)
                    {
                    case sizeof(UINT8) :
                        *(UINT8 *)dstBuf = (UINT8)(tempData & 0xFFU);
                        dstBuf = (UINT8 *)dstBuf + 1;
                        break;
                    case sizeof(UINT16) :
                        *(UINT16 *)dstBuf = (UINT16)(tempData & 0xFFFFU);
                        dstBuf = (UINT16 *)dstBuf + 1;
                        break;
                    case sizeof(UINT32) :
                        *(UINT32 *)dstBuf = tempData;
                        dstBuf = (UINT32 *)dstBuf + 1;
                        break;
                    default:
                        /* invalid dstItemSize passed to function */
                        rc = SXM_E_SIZE;
                        break;
                    }
                }
                /* if the maximum size of the local storage array is exceeded then
                   skip remainder of data (as it may be possible to continue) and report an error */
                else {
                    SKIP(label->lbsize);
                    DEBUG_PKT("Skipping %d", label->lbsize);
                    rc = SXM_E_BUF_SIZE;
                }
                if (dstBufSize > 0) {
                    dstBufSize = dstBufSize - 1;
                }
                arrayLen = arrayLen - 1;
            }
        }};
    }

    LEAVE_PKT("%s", sxm_sdk_format_result(rc));
    return rc;
}

/*****************************************************************************//**
 * Reads a TDS Entry of type INTEGER.
 * 
 * This function decodes the data bitstream and reads an INTEGER type TDS entry.
 * First it validates the size of the INTEGER against the size of the data where 
 * the entry is to be stored. If the provided entry is too small then an error 
 * is returned. If it is valid then the appropriate number of bits, as defined
 * in the TDS Definition table, are read and saved.
 *
 * \param[in] pkt            Bitstream containing received SXM data to decode.
 * \param[out] dstBuf        Provided by the OEM developer as free space to copy
 *                           the selected data into.
 * \param[in] itemSize       Size of dstBuf in bytes.
 * \param[in] size           Expected number of bits to read.
 *
 * \retval SXM_E_OK          Data read.
 * \retval SXM_E_SIZE        Provided integer dstBuf is too small for the amount
 *                           of data to be read.
 * \retval SXM_E_BAD_PACKET  Errors while reading from buffer
 *
 ********************************************************************************/
static int sxm_tds_bitbuff_int(SXMBitBuff *pkt, void *dstBuf, UINT32 dstItemSize, UINT32 size) {
    int rc = SXM_E_OK;

    switch (0) { default: {
        UINT32 labelSize;
        UINT32 tempData;

        /* calculate number of bytes needed to store the entry */
        labelSize = size / SXM_ARCH_BITS_IN_BYTE;
        if ((size % SXM_ARCH_BITS_IN_BYTE) != 0) {
            ++labelSize;
        }
        /* if needed number of bytes greater than available then 
           data will not fit, return an error */
        if (dstItemSize < labelSize) {
            rc = SXM_E_SIZE;
            break;
        }
        NEXT(tempData, size);
        switch (dstItemSize)
        {
            case sizeof(UINT8):
                *(UINT8 *)dstBuf = (UINT8)(tempData & 0xFFU);
                break;
            case sizeof(UINT16):
                *(UINT16 *)dstBuf = (UINT16)(tempData & 0xFFFFU);
                break;
            case sizeof(UINT32):
                *(UINT32 *)dstBuf = tempData;
                break;
            default:
                /* invalid dstItemSize passed to function */
                rc = SXM_E_SIZE;
                break;
        }
        if (pkt->err) {
            rc = SXM_E_BAD_PACKET;
        }
    }}
    return rc;
}

/*****************************************************************************//**
 *
 * \brief skip a TDS Entry from a data bitstream.
 *
 * This function provides a generic data skipper for TDS entries. It determines the
 * type of entry then calls the appropriate function to skip (parse over) the entry.
 * The type of entry is specified by the dtype variable found in the Tdslabel provided.
 *
 * \param[in] pkt           Bitstream containing received SXM data to decode.
 * \param[in] label         Pointer to the TDS entry for the entry being read.
 *
 * \retval SXM_E_OK         Data successfully skipped
 * \retval SXM_E_ERROR      Data error occurred while skipping bits
 *
 ********************************************************************************/
static int sxm_tds_skip_any_label(SXMBitBuff *pkt, TdsLabel *label) {
    int rc = SXM_E_OK;

    ENTER_PKT("lbid:%d", label->lbid);
    /* Is the field present ? */
    if (FLAG()) {
        /* skip according to data type */
        switch (label->dtype) {
            case TDS_DATA_INT:
            {
                SKIP(label->lbsize);
                DEBUG_PKT("INT: skipped %d", label->lbsize);
                break;
            }
            case TDS_DATA_STRING:
            {
                ISOSKIP(label->lbsize);
                DEBUG_PKT("ISO: skipped %d", label->lbsize);
                break;
            }
            case TDS_DATA_BAUDOT:
            {
                BSKIP(BAUDOT_MODE_START_WITH_LETTERS);
                DEBUG_PKT("BAUDOT: skipped");
                break;
            }
            case TDS_DATA_ARRAY:
            {
                uint arraylen;

                NEXT(arraylen, label->lensize);
                SKIP(label->lbsize * arraylen);
                DEBUG_PKT("ARRAY: skipped %d", label->lbsize * arraylen);
                break;
            }
            case TDS_DATA_SAMPA:
            {
                UINT16 scode = 0;
                uint idx;

                idx = label->lbsize;
                while ((idx-- > 0) && (pkt->err == 0) && (scode != 0)) {
                    NEXT_TO(UINT16, scode, 6);
                }
                DEBUG_PKT("SAMPA: skipped");
                break;
            }
            default:
            {
                SKIP(label->lbsize);
                DEBUG_PKT("UNKNOWN: skipped %d", label->lbsize);
                break;
            }
        }

        if (pkt->err) {
            rc = SXM_E_ERROR;
        }
    }

    LEAVE_PKT("%s", sxm_sdk_format_result(rc));
    return rc;
}

/*****************************************************************************//**
 *
 * \brief Read a TDS Entry from a data bitstream.
 * 
 * This function provides a generic data reader for TDS entries. It determines the
 * type of entry then calls the appropriate function to decode the entry. The type
 * of entry is specified by the dtype variable found in the Tdslabel provided.
 *
 * \param[in] pkt           Bitstream containing received SXM data to decode.
 * \param[out] dstBuf       Provided by the OEM developer as free space to copy the selected data into.
 * \param[in] dstItemSize   Number of bytes for each item in the dstBuf. For example if dstBuf is an 
 *                          integer array of type UINT16, then dstItemSize is sizeof(UINT16).
 * \param[in] dstBufSize    Number of items in dstBuf. For example if dstBuf is a character array of 
 *                          36 characters then dstBufSize is 36.
 * \param[in] table         Pointer to the Definition Table for the entry being read.
 *
 * \retval SXM_E_OK             Data found.
 * \retval SXM_E_FAULT          User-provided memory is NULL.
 * \retval SXM_E_INVAL          Invalid type to decode (must be TDS_LABEL_VARIABLE).
 * \retval SXM_E_BAD_STRING     A BAUDOT or CSTRING operation failed.
 * \retval SXM_E_SIZE           The input parameter dstItemSize is smaller than the TDS Label size
 *                              or it is an invalid size (must be sizeof UINT8, UINT16 or UINT32).
 * \retval SXM_E_BAD_PACKET     An error occurred while reading a packet.
 * \retval SXM_E_BUF_SIZE       The value of dstBufSize is smaller than the TDS Array size.
 * \retval SXM_E_BAD_LABEL      A label type is invalid or unknown. Valid types are TDS_DATA_INT,
 *                              TDS_DATA_BAUDOT, TDS_DATA_STRING, TDS_DATA_ARRAY and TDS_DATA_SAMPA.
 * \retval SXM_E_ERROR  Error reading/decoding data.
 * \retval SXM_E_NOENT  The entry type is not known. 
 *
 ********************************************************************************/
int sxm_tds_read_label(SXMBitBuff *pkt, void *dstBuf, UINT32 dstItemSize, UINT32 dstBufSize, TdsLabel *table) {
    int rc = SXM_E_OK;

    switch (0) { default: {
        if ((table == NULL) || (pkt == NULL) || (dstBuf == NULL)) {
            rc = SXM_E_FAULT;
            break;
        }
        if (table->lbtype == TDS_LABEL_CONSTANT) {
            rc = SXM_E_INVAL;
            break;
        }
        switch (table->dtype) {
            case TDS_DATA_INT:
                rc = sxm_tds_bitbuff_int(pkt, dstBuf, dstItemSize, table->lbsize);
                break;
            case TDS_DATA_STRING:
                CSTRING((char *)dstBuf, table->lbsize);
                /* this does not return an error but it sets pkt->err to 2 (hardcoded so I cannot 
                   use a define value unless it is fixed elsewhere) */
                if (pkt->err == 2) {
                    rc = SXM_E_BAD_STRING;
                }
                break;
            case TDS_DATA_BAUDOT:
                rc = BAUDOT((char *)dstBuf, BAUDOT_MODE_START_WITH_LETTERS, table->lbsize, dstBufSize); 
                /* BAUDOT returns non-zero if an error occurred, lets standardize the return code */
                if (rc != SXM_E_OK) {
                    rc = SXM_E_BAD_STRING;
                }
                break;
            case TDS_DATA_ARRAY:
                rc = sxm_tds_bitbuff_array(pkt, dstBuf, dstItemSize, dstBufSize, table);
                break;
            case TDS_DATA_SAMPA:
                rc = sxm_tds_bitbuff_scode(pkt, dstBuf, dstBufSize, table->lbsize);
                break;
            default:
                rc = SXM_E_BAD_LABEL;
                (void)sxm_tds_bitbuff_reserved(pkt, dstBuf, dstBufSize, table->lbsize);
                break;
        }
    }};

    return rc;
}

/*****************************************************************************//**
 * Reads a TDS Entry of reserved type.
 * 
 * This function decodes the reserved type of data (DTYPE=5, 6 or 7) and put it into 
 * output buffer. Intention is to support possible future protocol enhancements
 * It validates the size of the label against the size of the data where 
 * the entry is to be stored. If the provided entry is too small then an error 
 * is returned. If it is valid then the appropriate number of bits, as defined
 * in the TDS Definition table, are read and saved.
 *
 * \param[in] pkt            Bitstream containing received SXM data to decode.
 * \param[out] dstBuf        Provided by the OEM developer as free space to copy the 
 *                           selected data into.
 * \param[in] itemSize       Size of dstBuf in bytes.
 * \param[in] size           Expected number of bits to read.
 *
 * \retval SXM_E_OK          Data read.
 * \retval SXM_E_SIZE        Provided integer dstBuf is too small for the amount of 
 *                           data to be read.
 * \retval SXM_E_BAD_PACKET  Errors while reading from buffer
 *
 ********************************************************************************/
static int sxm_tds_bitbuff_reserved(SXMBitBuff *pkt, void *dstBuf, UINT32 dstItemSize, UINT32 size) {
    int rc = SXM_E_OK;

    switch (0) { default: {
        UINT32 labelSize;
        UINT8 leftover, *dest = (UINT8*)dstBuf;

        /* calculate number of bytes needed to store the entry */
        labelSize = (size + SXM_ARCH_BITS_IN_BYTE - 1) / SXM_ARCH_BITS_IN_BYTE;
        if (labelSize == 0)
        {
            /* nothing to do */
            break;
        }

        /* if needed number of bytes greater than available then 
           data will not fit, return an error */
        if (dstItemSize < labelSize) {
            rc = SXM_E_SIZE;
            break;
        }

        do {
            /* Fetching next byte, or its remainder */
            leftover = MIN(SXM_ARCH_BITS_IN_BYTE, size);

            NEXT_TO(UINT8, *dest++, leftover);

            size -= leftover;
        } while (size > 0);

        if (pkt->err) {
            rc = SXM_E_BAD_PACKET;
        }
    }}
    return rc;
}


/** Defines SAMPA mapping table item */
typedef struct {
    /** ASCII representation */
    const char *const glyph;
    /** Glyph len excluding nul-terminator */
    uint len;
} TdsSampa;

/** Defines SAMPA mapping table item */
static const TdsSampa scodes_table[] = {
    /* 0x00 */ {"\0", 1}, {" ", 1}, {".", 1}, {"\"", 1},
    /* 0x04 */ {"%",  1}, {"~", 1}, {"2", 1}, {"3",  1},
    /* 0x08 */ {"4",  1}, {"9", 1}, {"@", 1}, {"{",  1},
    /* 0x0C */ {"a",  1}, {"b", 1}, {"d", 1}, {"e",  1},
    /* 0x10 */ {"f",  1}, {"g", 1}, {"h", 1}, {"?",  1},
    /* 0x14 */ {"i",  1}, {"j", 1}, {"j\\", 2}, {"k", 1},
    /* 0x18 */ {"l",  1}, {"m", 1}, {"n", 1}, {"o", 1},
    /* 0x1C */ {"p",  1}, {"r", 1}, {"s", 1}, {"t", 1},
    /* 0x20 */ {"u",  1}, {"v", 1}, {"w", 1}, {"x", 1},
    /* 0x24 */ {"y",  1}, {"z", 1}, {"A", 1}, {"B", 1},
    /* 0x28 */ {"D",  1}, {"E", 1}, {"G", 1}, {"H", 1},
    /* 0x2C */ {"I",  1}, {"J", 1}, {"Y", 1}, {"N", 1},
    /* 0x30 */ {"O",  1}, {"R", 1}, {"S", 1}, {"T", 1},
    /* 0x34 */ {"U",  1}, {"V", 1}, {"Z", 1}, {"tS", 2}, 
    /* 0x38 */ {"`",  1}, {"D@", 2}, {" \"", 2}, {"\"sports", 7},
    /* 0x3C */ {"hVn.drEd", 8}, {"re.di.o", 7}, {"sl.ri.@s", 8}, {"Eks.\"Em", 7}
};

/*****************************************************************************//**
 * Reads a TDS Entry of type SAMPA.
 * 
 * This function decodes the data bitstream and reads an SAMPA type TDS entry
 *
 * \param[in] pkt           Bitstream containing received SXM data to decode.
 * \param[out] dstBuf       Provided by the OEM developer as free space to copy the selected data into.
 * \param[in] dstBufSize    Number of items in the dstBuf array.
 * \param[in] lbSize        Expected scode sequence length to be read
 *
 * \retval SXM_E_OK          Data found.
 * \retval SXM_E_BAD_STRING  Either buffer is too small or the bitstream has not enough data. 
 * \retval SXM_E_FAULT       User-provided dstBuf array is not large enough.
 *
 ********************************************************************************/
static int sxm_tds_bitbuff_scode(SXMBitBuff *pkt, void *dstBuf, UINT32 dstBufSize, UINT16 lbSize)
{
    int rc = SXM_E_OK;
    UINT16 idx = lbSize;
    char *pDstBuff = (char*)dstBuf;
    const char * const pDstBuffEnd = pDstBuff + dstBufSize;
    while ((idx-- > 0) && !pkt->err) {
        UINT16 scode;
        const TdsSampa *pSCode;
        NEXT_TO(UINT16, scode, 6);
        pSCode = &scodes_table[scode];
        if ((pDstBuff + pSCode->len) > pDstBuffEnd) {
            rc = SXM_E_BAD_STRING;
            break;
        }
        memcpy(pDstBuff, pSCode->glyph, pSCode->len);
        pDstBuff += pSCode->len;
        if (!scode) {
            // End-of-string detected
            break;
        }
    }
    if (pkt->err != 0) {
        rc = SXM_E_BAD_STRING;
    }
    return rc;
}

/*****************************************************************************//**
 *
 * \brief Reads a TDS Entry of type ARRAY.
 * 
 * This function decodes the data bitstream and reads an ARRAY type TDS entry.
 * First it validates the size of the ARRAY against the size of the data where 
 * the entry is to be stored, checking both the overall maximum array size as
 * well as the size of each array member. If the provided entry is too small then 
 * an error is returned. If it is valid then the length of the array is read and
 * validated. If it is valid then the array members are read and stored.
 *
 * \param[in] pkt           Bitstream containing received SXM data to decode.
 * \param[out] dstBuf       Provided by the OEM developer as free space to copy the selected data into.
 * \param[in] dstItemSize   Size of dstBuf members in bytes.
 * \param[in] dstBufSize    Number of items in the dstBuf array.
 * \param[out] arrayReadLen Number of array elements read by the parser 
 
 * \retval SXM_E_OK     Data found.
 * \retval SXM_E_ERROR  Invalid size for array members.
 * \retval SXM_E_NOMEM  Array length exceeds defined maximum.
 * \retval SXM_E_FAULT  User-provided dstBuf array is not large enough.

 ********************************************************************************/
int sxm_tds_read_array(SXMBitBuff *pkt, void *dstBuf, UINT32 dstItemSize, UINT32 dstBufSize, TdsLabel *label, UINT32 *pArrayReadLen) {
    int rc = SXM_E_OK;

    /* Check if label is valid */
    if (label->valid == 0) {
        DEBUG_PKT("Invalid label %d. Skipping", label->lbid);
        rc = sxm_tds_skip_any_label(pkt, label);
    }
    else {
        UINT32 arrayLen;
        UINT32 labelSize;
        UINT32 tempData;

        /* initialize the number of array element read */
        *pArrayReadLen = 0;

        do {

            if (label->dtype != TDS_DATA_ARRAY) {
                rc = SXM_E_BAD_LABEL;
                break;
            }

            /* calculate number of bytes needed to store each item */
            labelSize = label->lbsize / SXM_ARCH_BITS_IN_BYTE;
            if ((label->lbsize % SXM_ARCH_BITS_IN_BYTE) != 0) {
                labelSize = labelSize + 1;
            }
            /* if array items are larger than size of application defined array item size
               then there is an error, stop processing the message */
            if (dstItemSize < labelSize) {
                rc = SXM_E_SIZE;
                break;
            }
            /* the first LENSIZE bits of the array contain the Array Length */
            NEXT(arrayLen, label->lensize);
            if (pkt->err) {
                rc = SXM_E_BAD_PACKET;
                break;
            }
            /* make sure array length does not exceed maximum allowed */
            if (arrayLen > label->arraymax) {
                rc = SXM_E_ARRAY_SIZE;
                break;
            }
            /* process array entries */
            while (arrayLen > 0) {
                /* read data until the local storage buffer is full or the end
                   of the data is reached */
                if (dstBufSize > 0) {
                    NEXT(tempData, label->lbsize);
                    if (pkt->err) {
                        rc = SXM_E_BAD_PACKET;
                        break;
                    }
                    if (dstItemSize == sizeof(UINT8)) {
                        *(UINT8 *)dstBuf = (UINT8)(tempData & 0xFFU);
                        dstBuf = (UINT8 *)dstBuf + 1;
                    }
                    else if (dstItemSize == sizeof(UINT16)) {
                        *(UINT16 *)dstBuf = (UINT16)(tempData & 0xFFFFU);
                        dstBuf = (UINT16 *)dstBuf + 1;
                    }
                    else if (dstItemSize == sizeof(UINT32)) {
                        *(UINT32 *)dstBuf = tempData;
                        dstBuf = (UINT32 *)dstBuf + 1;
                    }
                    /* invalid dstItemSize passed to function */
                    else {
                        rc = SXM_E_SIZE;
                        break;
                    }
                }
                /* if the maximum size of the local storage array is exceeded then
                   skip remainder of data (as it may be possible to continue) and report an error */
                else {
                    SKIP(label->lbsize);
                    rc = SXM_E_BUF_SIZE;
                }
                if (dstBufSize > 0) {
                    dstBufSize = dstBufSize - 1;
                    *pArrayReadLen = *pArrayReadLen + 1;
                }
                arrayLen = arrayLen - 1;
            }

        } while (0);
    }
    return rc;
}

/*****************************************************************************//**
 *
  * \brief skip a TDS Entry from a data bitstream.
 * 
 * This function provides a generic data skipper for TDS entries. It determines the
 * type of entry then calls the appropriate function to skip (parse over) the entry.
 * The type of entry is specified by the dtype variable found in the Tdslabel provided.
 * This function is to be used for Table Entry payloads processing, so only variable
 * label types are supported. No skipping happens for Constant labels
 *
 * \param[in] pkt           Bitstream containing received SXM data to decode.
 * \param[in] label         Pointer to the TDS entry for the entry being read.
 *
 * \retval SXM_E_OK         Data successfully skipped
 * \retval SXM_E_ERROR      Data error occurred while skipping bits
 *
 ********************************************************************************/
int sxm_tds_skip_label(SXMBitBuff *pkt, TdsLabel *label) {
    int rc = SXM_E_OK;

    ENTER_PKT("lbid:%d", label->lbid);
    // only variable type labels are processed
    if (label->lbtype != TDS_LABEL_CONSTANT) {
        rc = sxm_tds_skip_any_label(pkt, label);
    }

    LEAVE_PKT("%s", sxm_sdk_format_result(rc));
    return rc;
}

/*****************************************************************************//**
 *
 * This function sets debug level for utlilty functions
 *
 * \param[in] dLevel      required debugging level
 *
 ********************************************************************************/
void sxm_tds_set_debug(uint dLevel) {
    debugLevel = dLevel;
}

/*****************************************************************************//**
 *
 * This function returns debug level set for utlilty functions
 *
 ********************************************************************************/
uint sxm_tds_get_debug(void) {
    return debugLevel;
}
