/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 * 
 * \file sxm_states.c
 * \author Leslie French
 * \date 8/20/2013
 * \brief State and County MBR lookup. Index by SXM id
 *
 * \details The states file can be built and decoded using the sdkfiles utility.
 * The builder takes as input (-i) a PSV file containing one line for each state,
 * province or county to be included in the file.
 *
 * ~~~~~~
 * STATE|COUNTRY|SNAME|COUNTY|FIPS|MBR
 * AL|0|Alabama|-|1|-2899090,987772,-2781618,1147143
 * AL|0|-|Sumter|119|-2897394,1058650,-2878369,1081208
 * AL|0|-|Dallas|47|-2866317,1050122,-2844487,1072481
 * . . . . .
 * WY|0|-|Laramie|21|-3449837,1343424,-3409583,1365005
 * WY|0|-|Carbon|7|-3536649,1343459,-3475649,1390502
 * WY|0|-|Fremont|13|-3606237,1384820,-3522606,1442092
 * DC|0|District of Columbia|-|11|-2527060,1271124,-2520166,1277806
 * DC|0|-|District of Columbia|1|-2527060,1271124,-2520166,1277806
 * PR|3|Puerto Rico|-|-|[-67.998751,17.831509 -65.168503,18.568002]
 * AB|1|Alberta|-|-|[-120.000523,49.000000 -109.999855,60.000063]
 * BC|1|British Columbia|-|-|[-139.052198,48.308613 -114.054222,60.000064]
 * MB|1|Manitoba|-|-|[-102.000005,48.999950 -88.967225,60.000129]
 * ~~~~~~
 *
 * The first column gives the 2 or 3 letter abbreviation for the State or Province,
 * the second column gives the country code used in the states routines (0 - 3),
 * the third column is the state name, or '-' for a county, the fourth column is
 * the county name, or '-' for a state, the fifth column is the FIPS code for the
 * state or county and the final column gives the MBR (either as \ref fix integers,
 * or floating point values).
 *
 * The output  is the binary states file, which has the following structure.
 * It is organized as a transaction file, even though it is read-only. The root
 * block contains an array of \ref StateRec entries, one per state or province
 * in the service.
 *
 * The \ref StateRec entries are followed by the string table containing the
 * names of the states.
 *
 * The individual country records are indexed through the user entries
 * in the transaction block:
 *
 * |                 |                                                           |
 * |:----------------|:----------------------------------------------------------|
 * | user[0]         | version number                                            |
 * | user[1]		 | total number of state/province entries                    |
 * | user[2], user[3]| start of US entries and count of US entries               |
 * | user[4], user[5]| start of CA entries and count of CA entries               |
 * | user[6], user[7]| start of MX entries and count of MX entries               |
 * | user[8], user[9]| start of PR entries and count of PR entries               |
 *
 * Each county is a fixed-length structure defined as \ref CountyRec.
 *
 ******************************************************************************/

#define DEBUG_TAG "states"

#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>

/** States DB format */
#define STATES_SERVICE_NAME ("states")
/** State DB schema version */
#define STATES_TFILE_SCHEMA_VERSION (1)

/** State descriptor */
typedef struct {
    fix mbr[4]; //!< The state/province MBR
    ushort off; //!< The block number of the county records
    ushort bc;  //!< The number of blocks in the county records
    ushort count; //!< The number of county records
    ushort offn; //!< The offset to the state name in the string table
    ushort fips; //!< The FIPS code
    byte country; //!< Country id
    byte start;  //!< The index number in the SXM state table
    char abbr[4]; //!< The 2 or 3 character state abbreviation
} StateRec;

/** Each county is a fixed-length structure */
typedef struct {
    fix mbr[4]; //!< The state/province MBR
    ushort fips; //!< The FIPS code
    ushort offn; //!< The offset to the state name in the string table
} CountyRec;

/** T-File root block */
typedef union {
    struct {
        StateRec states[128]; //!< Array of state descriptors
        char snames[2048]; //!< State names data
    }d; //!< Real data
    struct {
        byte b[8192]; //!< Alignment data
    }b; //!< Alignment data
}RootRec;

/** Keeps currently opened t-file */
static SXMTFile *dbfile = NULL;
/** Keeps t-file root data */
static byte *root;
/** Keeps the t-file root data type-casted to real type */
static StateRec *rr;
/** Keeps reference to the state names from the t-file*/
static char *snames;

/** States component data protection */
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
/** State DB registration reference counter */
static byte states_usecnt = 0;

/* Static Prototypes */
static int init( void );

/***************************************************************************//**
 * Internal function used to open the states database.
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_NO_STATES No states DB
 * \retval SXM_E_BAD_STATES The Db file exists but it's cannot be used
 *
 ********************************************************************************/
static int init( void ) 
{
    int ret = SXM_E_OK;

    if  (dbfile == NULL) 
    {
        uint schema;

        dbfile = sxm_tfile_open('R', STATES_SERVICE_NAME,
                                STATES_SERVICE_NAME, "rb",
                                STATES_SERVICE_NAME, &schema, &ret);

        if  (ret == SXM_E_NO_DB)
            ret = SXM_E_NO_STATES;
        else if (ret == SXM_E_BAD_DB || ret == SXM_E_PIPE)
            ret = SXM_E_BAD_STATES;

        if  (dbfile == NULL)
        {
            PLOG_DBF("Failed to open states file. rc %s", sxm_sdk_format_result(ret));
        }
        else if (schema != STATES_TFILE_SCHEMA_VERSION)
        {
            PLOG_DBF("Unsupported schema version %u", schema);

            // Close the file
            sxm_tfile_close(dbfile);
            dbfile = NULL;
            ret = SXM_E_NO_STATES;
        }
        else 
        {
            root = (byte*)sxm_tfile_root(dbfile);
            rr = (StateRec*)(void*)root;
            snames = ((RootRec*)(void*)root)->d.snames;
        }
    }

    return ret;
}

/***************************************************************************//**
 * Registers a service to use states database.
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_NO_STATES No states DB
 * \retval SXM_E_BAD_STATES The Db file exists but it's cannot be used
 *
 ********************************************************************************/
int sxm_states_register( void ) 
{
    int rc = SXM_E_OK;

    LOCK(mutex);

    rc = init();

    if (rc == SXM_E_OK)
        states_usecnt++;
    
    UNLOCK(mutex);
    return rc;
}

/***************************************************************************//**
 * Unregisters a service to use states database.
 *
 * \note Resources returned from state api calls should not be 
 *       referenced after unregister is called.
 *
 ******************************************************************************/
void sxm_states_unregister( void ) 
{
    LOCK(mutex);
    if (states_usecnt && (dbfile != NULL))
        if (--states_usecnt == 0)
        {
            sxm_tfile_close(dbfile);
            dbfile = NULL;
        }
    
    UNLOCK(mutex);
}

/***************************************************************************//**
 * Determines which states are contained inside an mbr.
 *
 * \param[in] mbr pointer to the bounding rectangle to be checked.
 * \param[out] st 16 * 32 bit array. Bits set indicate that index is 
 *                inside mbr.
 * \param[in] doc Determines level of checking
 *                \c 0: check at state mbr
 *                \c otherwise: check each county mbr.
 *
 ******************************************************************************/
void sxm_states_lookup(SXMMBR *mbr, uint *st, int doc) 
{
    SXMFixMBR fbr;
    int i, j;

    memset(st, 0, 16);
    if  (dbfile == NULL)
        return;

#if SXM_USE_FIX == 1
    memcpy(&fbr, mbr, sizeof(SXMFixMBR));
#else
    fbr.ll.lon = float2fix(mbr->ll.lon);
    fbr.ll.lat = float2fix(mbr->ll.lat);
    fbr.ur.lon = float2fix(mbr->ur.lon);
    fbr.ur.lat = float2fix(mbr->ur.lat);
#endif

    for  (i = 0; i < SXM_STATES_INDEX_MAX; i++)
    {
        if  (sxm_fixmbr_intersects(&fbr, (SXMFixMBR *)&rr[i].mbr)) 
        {
            if (doc == 0)
                BITS(st, i + 1);
            else 
            {
                CountyRec *cc;
                int rc;

                LOCK(mutex);

                cc = (CountyRec *)sxm_tfile_alloc_and_read(dbfile, rr[i].off, rr[i].bc, NULL, &rc);
                if (!cc) 
                {
                    PLOG_DBF("Failed to load %u block(s) from %u in states db (rc=%d)",
                             rr[i].bc, rr[i].off, rc);
                }
                else 
                {
                    for  (j = 0; j < rr[i].count; j++)
                    {
                        if  (sxm_fixmbr_intersects(&fbr, (SXMFixMBR *)&cc[j].mbr)) 
                        {
                            BITS(st, i + 1);
                            break;
                        }
                    }
                    sxe_free(cc);
                }

                UNLOCK(mutex);
            }
        }
    }
}

/***************************************************************************//**
 * Returns the state abbreviation for a given index.
 *
 * \param[in] i State Index
 *
 * \return String pointer containing result, '??' for invalid.
 *
 ******************************************************************************/
char *sxm_states_abbr(int i) 
{
    if  (i < 1 || i > SXM_STATES_INDEX_MAX || dbfile == NULL || 0 == rr[i - 1].start)
        return "??";

    return rr[i-1].abbr;
}

/***************************************************************************//**
 * Returns the state name for a given index.
 *
 * \param[in] i State Index
 *
 * \return String pointer containing result, "(Unknown)" for invalid.
 *
 ******************************************************************************/
char *sxm_states_name(int i) 
{
    if  (i < 1 || i > SXM_STATES_INDEX_MAX || dbfile == NULL || 0 == rr[i - 1].start)
        return "(Unknown)";

    return (char *)(snames + rr[i-1].offn);
}

/***************************************************************************//**
 * Determines if a point lies within a given states bounds.
 *
 * \param[in] i State Index
 * \param[in] point a point structure to check. 
 *
 * \return not-0 if point is withing state, otherwise 0.
 *
 ******************************************************************************/
int sxm_states_inside(int i, SXMPoint *point) {
    SXMFixPoint fixpoint;

    if  (i < 1 || i > SXM_STATES_INDEX_MAX || dbfile == NULL || point == NULL)
        return 0;  // returning error

#if SXM_USE_FIX == 1
    memcpy(&fixpoint, point, sizeof(fixpoint));
#else
    fixpoint.lon = float2fix(point->lon);
    fixpoint.lat = float2fix(point->lat);
#endif

    return sxm_fixmbr_inside((SXMFixMBR *)&rr[i - 1].mbr, &fixpoint);
}

