/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_sdtp.c
* \author(s) Leslie French
* \date 8/20/2013
* \brief SDTP Packet re-assembly for SXM-SDK
*
* \details The sdtp packet handler re-assembles individual transport units into complete
* Access Units. Please, refer to \ref sxe_sdtp_page for implementation details.
*
* \page sxe_sdtp_page SDTP
* 
* The sdtp packet handler re-assembles individual transport units into complete
* Access Units.
*
* The SDTP layer is the main glue between the device interface, which delivers SXi
* DataPackets, and the protocol decoders, which expect complete Access Units.
* Whenever a DMI is enabled, it needs an SXMSdtp structure to hold the transport
* packets as they are built up into the Access Unit.  Once the AU is complete
* (a set of SDTP packets in the correct sequence has been accumulated), the add
* routine returns 1.  At the point the buffer contains the complete Access Unit,
* starting at the first payload byte (the one containing the Protocol Version Number),
* and ending with the last byte of the CRC-32 checksum.  The pl field contains the
* length of the Access Unit.  This is suitable for passing to the complete routine
* of the specific service.
*
* Note that the SDTP layer does not re-check the SDTP checksum, since that was done
* in the module, and the DataPacket is protected by a link-layer CRC, which has
* already been verified.  Also note that the AU CRC-32 has not been checked either. 
* This is to reduce processing requirements if the service module decides it does
* not need this AU.
*
*******************************************************************************/

#include <util/sxm_common_internal.h>
#include <util/sxm_list.h>

/***************************************************************************//**
* The routine adds a single SDTP transport packet into the
* Access Unit re-assembly .
*
* \note Refer to SX-9845-0019 for details on the SDTP format and decoding.
*
* \param[in] self Pointer to received SDTP packet structure.
* \param[in] b Pointer to SDTP byte stream
*
* \retval SXM_E_RESOURCE    Buffer is not enough to complete AU.
* \retval SXM_E_OK          Last SDTP of an AU received, AU should be complete.
* \retval SXM_E_NOENT       AU not complete yet, do not try to process.
* \retval SXM_E_FAULT       NULL parameter
*
*******************************************************************************/
int sxm_sdtp_add(SXMSdtp *self, byte *b) {
    uint flags, plen;
    int rc = SXM_E_NOENT;

//  there is no point checking the sdtp checksum, it was done in the module
    if ((!self) || (!self->buffer) || (!b)){
        return SXM_E_FAULT;
    }

    flags = (b[1] >> 6) & 3;
    plen = ((b[2] << 8) | b[3]) & 1023;

    if  (flags == 0) {
        //  middle packet
        if  (self->state == 0 || self->seq != plen) {
            self->state = 0;
        } else if  (self->pl+1024 >= self->max) {
            self->state = 0;
            rc = SXM_E_RESOURCE;
        } else {
            memcpy(self->buffer+self->pl, b+4, 1024);
            self->pl += 1024;
            self->seq = plen-1;
        }
    } else if (flags == 1) {
        //  end packet
        if  (self->state != 0 && self->seq == 0) {
            if  (plen == 0)
                plen = 1024;
            if  (self->pl+plen <= self->max) {
                memcpy(self->buffer+self->pl, b+4, plen);
                self->pl += plen;
                self->state = 0;
                rc = SXM_E_OK;
            } else {
                self->state = 0;
                rc = SXM_E_RESOURCE;
            }

        }
    } else if (flags == 2) {
        //  start of a new packet
        if (self->max < 1024) {
            self->state = 0;
            rc = SXM_E_RESOURCE;
        }
        else {
            memcpy(self->buffer, b+4, 1024);
            self->pl = 1024;
            self->seq = plen-1;
            self->state = 1;
        }
    } else {
        //  single packet
        if  (plen == 0)
            plen = 1024;
        if (self->max < plen) {
            self->state = 0;
            rc = SXM_E_RESOURCE;
        }
        else {
            memcpy(self->buffer, b+4, plen);
            self->pl = plen;
            self->state = 0;
            rc = SXM_E_OK;
        }
    }

    return rc;
}

/***************************************************************************//**
* The routine creates a new re-assembly structure. One structure will be needed
* for each DMI being re-assembled.
*
* \param[in] maxau - Maximum size in bytes for the AU to be built
*
* \return Pointer to memory allocated for SDTP storage.
*         \c NULL if allocation failed.
*
*******************************************************************************/
SXMSdtp *sxm_sdtp_new(uint maxau) {
    /* our element is a list entry, so allocating it as a list element with buffer 
       to reduce number of allocates */
    SXMSdtp *ret = sxm_list_allocate(sizeof(SXMSdtp) + maxau + 128);

    if (ret != NULL) {
        ret->max = maxau;
        ret->buffer = (byte *)(ret + 1);
    }

    return ret;
}

/***************************************************************************//**
* Frees SDTP memory allocated above. 
*
* \param[in] self Pointer to the memory to free
*
*******************************************************************************/
void sxm_sdtp_delete(SXMSdtp *self) {
    sxm_list_free(self);

    return;
}
