/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 *
 * \file sxm_noname_internal.h
 * \author(s) Leslie French
 * \date 8/20/2013
 * \brief These are helper macros designed to make the code more readable
 * but not following the SXM naming conventions.                  
 *
 *******************************************************************************/

#ifndef SXM_NONAME_INTERNAL_H
#define SXM_NONAME_INTERNAL_H

#ifdef __cplusplus
extern "C" {
#endif

/** Debug printouts buffer size */
#define SXM_PBUF_SIZE               (1024)
#define SXM_DUMP_NUM_BYTES_IN_LINE  (15)

/** Debug log message columns delimeter */
#define SXM_DBG_LOG_COLUMN_DELIMITER    '|'

/** Simple static assertion
 * \param[in] _t the statement which must be true
 * \param[in] _msg special string which allows see the assert reason
 */ 
#define STATIC_ASSERT(_t, _msg) \
    struct _msg##Assert{ int _msg: !!(_t); }

/** Simple static assertion for data type sizes
 * \warning Does not work with equations
 * \param[in] _t the type to be tested
 * \param[in] _s size requirement for the type
 */ 
#define TYPE_SIZE_STATIC_ASSERT(_t, _s) \
    STATIC_ASSERT(sizeof(_t) == (_s), _t##_must_be_##_s##_byte_len)

/** Simple static assertion or less or equal for data type sizes
 * \warning Does not work with equations
 * \param[in] _t the type to be tested
 * \param[in] _s size requirement for the type
 */ 
#define TYPE_SIZE_STATIC_LE_ASSERT(_t, _s) \
    STATIC_ASSERT(sizeof(_t) <= (_s), _t##_must_be_less_or_equal##_s##_byte_len)

/** Simple static assertion for minimum data type sizes
 * \warning Does not work with equations
 * \param[in] _v the value to be tested
 * \param[in] _r minimum required size of value
 */ 
#define SIZE_GE_STATIC_ASSERT(_v, _r) \
    STATIC_ASSERT((_v) >= (_r), _v##_must_be_greater_than_or_equal_to_##_r)

/** Simple static assertion for array sizes
 * \warning Does not work with equations
 * \param[in] _a the array to be tested
 * \param[in] _s size requirement for the type (in items)
 */
#define ARRAY_SIZE_STATIC_ASSERT(_a, _s) \
    STATIC_ASSERT(ARRAY_SIZE(_a) == (size_t)(_s), _a##_must_be_of_##_s##_items)

/** Make string from the passed argument
 * \param[in] a value to be a string
 */
#define STRINGIFY(a) #a

/** Calculates static array size
 * \param[in] _a array
 * \warning Don't use it for dynamic arrays
 */
#define ARRAY_SIZE(_a) (sizeof(_a) / sizeof((_a)[0]))

/** Calculates max value of two passed
 * \param[in] _a the first value
 * \param[in] _b the second value
 */
#define MAX(_a,_b) ((_a) > (_b) ? (_a) : (_b))

/** Calculates min value of two passed
 * \param[in] _a the first value
 * \param[in] _b the second value
 */
#define MIN(_a,_b) ((_a) < (_b) ? (_a) : (_b))

/** Round up to the nearest multiple of (1<<bit) */
#define MODUP(_val, _bit) ((((_val) + (1 << (_bit)) - 1) >> (_bit)) << (_bit))

/** Align a pointer to a uint type size boundary
 * \param[in] x the pointer to be aligned
 */
#define ALIGN_INT(x) ((byte *)((((size_t)(x) + sizeof(uint) - 1) / sizeof(uint)) * sizeof(uint)))

/** A bitfield is stored as an array of uints.  Use to calculate storage sizes. */
#define UINTS_FOR_BITS(_n) (MODUP(_n,5)>>5)

/** A bitfield is stored as an array of uints.  Use to calculate storage sizes. */
#define BITS_FOR_UINTS(_n) ((_n)<<5)

/** Makes a values with \c _j but set */
#define BIT(_j)                 (1U << (uint)(_j))
/** Checks bit \c _j inside the value \c _a
 * where \c _a is 32-bit values array */
#define BITP(_a, _j)            ((_a)[(uint)(_j) >> 5] & BIT((uint)(_j) & 31u))
/** Sets bit \c _j to the value \c _a
 * where \c _a is 32-bit values array */
#define BITS(_a, _j)            (_a)[(uint)(_j) >> 5] |= BIT((uint)(_j) & 31u)
/** Clears bit \c _j inside the value \c _a
 * where \c _a is 32-bit values array */
#define BITC(_a, _j)            (_a)[(uint)(_j) >> 5] &= ~BIT((uint)(_j) & 31u)
/** Set all bits inside the value \c _a
 * where \c _a is 32-bit values array */
#define BITSALL(_a)             (memset(_a, 0xFF, sizeof(_a)))

/** Sets bit \c b to the value \c a */
#define BITSET(a,b)             ((a) |= (1U << (uint)(b)))
#define BITSETT(type,a,b)       ((a) |= (type)(1 << (b)))
/** Clears bit \c b inside the value \c a */
#define BITCLR(a,b)             ((a) &= (~(1U << (uint)(b))))
#define BITCLRT(type,a,b)       ((a) &= (type)(~(type)(1 << (b))))
/** Checks bit \c b inside the value \c a */
#define ISBITSET(a,b)           ((a) & ((1U << (uint)(b))))

/** Simple values swapping macro
  * \param[in] _t data type
  * \param[in] _a the first value to swap
  * \param[in] _b the second value to swap
  */
#define SWAP_VALUES(_t, _a, _b) {_t tmp = _a; _a = _b; _b = tmp;}

/** range checking including boundaries
 * \param[in] i value to check
 * \param[in] min lower border
 * \param[in] max upper border
 */
#define IS_IN_RANGE(i, min, max) (((i) >= (min)) && ((i) <= (max)))

/** Converts \ref fix to float */
#define fix2float(_a)       (float)((_a)/32768.0)

/** Converts float to \ref fix*/
#define float2fix(_a)       (int)((_a)*32768.0)

/** Helper to wrap mutex locking function */
#define LOCK(_m)            pthread_mutex_lock(&_m);
/** Helper to wrap mutex locking function */
#define UNLOCK(_m)          pthread_mutex_unlock(&_m);

/** Helper macro to declare unused variable */
#define UNUSED_VAR(_v)  ((void)_v)

/**
 * \name byte buffer reading helpers
 * @{
 */
#define b1()                sxm_bytebuff_b1(bp)
#define b2()                sxm_bytebuff_b2(bp)
#define b4()                sxm_bytebuff_b4(bp)
#define bskip(_s)           sxm_bytebuff_skip(bp,  _s)
#define bstring(_t)         sxm_bytebuff_string(bp, (char *)_t, sizeof(_t))
/** @} */

/** Helper around strcmp() to check equal strings */
#define strequ(_a, _b) (strcmp(_a, _b) == 0)

/** Helper macro definition to peek 2-bytes value in big-endian
 * \param[in] _b byte buffer
 * \param[in] _o LSB index for that 2-bytes value
 */
#define x2(_b, _o)    (((_b)[(_o)] << 8 ) | (_b)[(_o)+1])

/** Helper macro definition to peek 4-bytes value in big-endian
 * \param[in] _b byte buffer
 * \param[in] _o LSB index for that 4-bytes value
 */
#define x4(_b, _o)    (((_b)[(_o)] << 24 ) | ((_b)[(_o)+1] << 16) | ((_b)[(_o)+2] << 8) | (_b)[(_o)+3])

/**
 * \name BitBuffer normal access and debug macros
 * @{
 */
#ifdef SXM_BITBUFF_DEBUG
    #define ALIGN()                     sxm_bitbuff_align_debug(pkt, 1)
    #define BAUDOT(_v, _m, _s, _i)      sxm_bitbuff_baudot_debug(pkt, #_v, _v, _m, _s, _i, 1)
    #define BSKIP(_m)                   sxm_bitbuff_skip_baudot(pkt, _m)
    #define CSTRING(_v, _m)             sxm_bitbuff_cstring_debug(pkt, #_v, _v, _m, 1)
    #define CSKIP()                     sxm_bitbuff_skip_cstring_debug(pkt, 1)
    #define ISOSTRING(_v, _m)           sxm_bitbuff_iso(pkt, _v, _m)
    #define ISOSKIP(_m)                 sxm_bitbuff_skip_iso(pkt, _m)  
    #define FLAG()                      sxm_bitbuff_flag_debug(pkt, 1)
    #define NEXT(_v, _n)                _v = sxm_bitbuff_read_debug(pkt, #_v, _n, 1)
    #define NEXT_TO(_t, _v, _n)         _v = (_t)sxm_bitbuff_read_debug(pkt, #_v, _n, 1)
    #define NEXT_TO_ADD(_t, _v, _n, _o) _v = (_t)(sxm_bitbuff_read_debug(pkt, #_v, _n, 1) + (_o))
    #define SKIP(_n)                    sxm_bitbuff_skip_debug(pkt, _n, 1)
    #define NEXT_TDS(_v, _m, _s, _i)    sxm_tds_read_label(pkt, _v, _m, _s, _i)
    #define SKIP_TDS(_i)                sxm_tds_skip_label(pkt,_i)
    #define NEXT_TDS_ARRAY(_v, _m, _s, _i, _j)    sxm_tds_read_array(pkt, _v, _m, _s, _i, _j)
    #define SKIP_LONG(_n)               sxm_bitbuff_skip_debug(pkt, _n, 1)
#else
    #define ALIGN()                     sxm_bitbuff_align(pkt)
    #define BAUDOT(_v, _m, _s, _i)      sxm_bitbuff_baudot(pkt, (char*)(_v), (uint)(_m), (int)(_s), (int)(_i))
    #define BSKIP(_m)                   sxm_bitbuff_skip_baudot(pkt, (uint)(_m))
    #define CSTRING(_v, _m)             sxm_bitbuff_cstring(pkt, (char*)(_v), (_m))
    #define CSKIP()                     sxm_bitbuff_skip_cstring(pkt)
    #define ISOSTRING(_v, _m)           sxm_bitbuff_iso(pkt, (char*)(_v), (uint)(_m))
    #define ISOSKIP(_m)                 sxm_bitbuff_skip_iso(pkt, (uint)(_m))
    #define FLAG()                      sxm_bitbuff_read(pkt, 1U)
    #define NEXT(_v, _n)                _v = sxm_bitbuff_read(pkt, (uint)(_n))
    #define NEXT_TO(_t, _v, _n)         _v = (_t)sxm_bitbuff_read(pkt, (uint)(_n))
    #define NEXT_TO_ADD(_t, _v, _n, _o) _v = (_t)(sxm_bitbuff_read(pkt, (uint)(_n)) + (_o))
    #define SKIP(_n)                    sxm_bitbuff_skip(pkt, (uint)(_n))
    #define NEXT_TDS(_v, _m, _s, _i)    sxm_tds_read_label(pkt, _v, (UINT32)(_m), (UINT32)(_s), _i)
    #define SKIP_TDS(_i)                sxm_tds_skip_label(pkt, _i)
    #define NEXT_TDS_ARRAY(_v, _m, _s, _i, _j)    sxm_tds_read_array(pkt, _v, (UINT32)(_m), (UINT32)(_s), _i, (UINT32*)(_j))
    #define SKIP_LONG(_n)               sxm_bitbuff_skip(pkt, (uint)(_n))
#endif
/** @} */

/** We will pick up debug through SKIP() usage.
 * \param[in] _n number of bytes to skip
 */
#define SKIPBYTES(_n)                   SKIP((_n) * 8)

/**
 * Provides aligned number of blocks in \c _d byte(s) length
 * \param[in] _size the size in bytes the caller wishes to be aligned
 * \param[in] _d the alignment criteria (block size)
 * \return number of blocks to fit the data with size \c _size
 */
#define NUM_ALIGNED_BLOCKS(_size, _d) \
    (((_size) + ((_d) - 1)) / (_d))

#ifdef __cplusplus
}
#endif

#endif /* SXM_NONAME_INTERNAL_H */
