/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_msg_queue_incl.h
* \author Eugene Lopatukhin
* \date 5/28/2013
* \brief Private header file for Message Queue implementation.
* 
********************************************************************************/

#ifndef SXM_MGS_QUEUE_INCL_H_
#define SXM_MGS_QUEUE_INCL_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <pthread.h>
#include <sxm_typ.h>
#include <util/sxm_list.h>
#include <util/sxm_platform_internal.h>

/** Queue Messages priority */
typedef enum {
    MSG_PRIO_NORMAL = 0, //!< Normal.
    MSG_PRIO_HIGH //!< High
} SXMQueueMsgPriority;

/** Defines advanced message content provider item */
typedef struct {
    uint size; //!< Size of the data referenced by \p pData
    const void *pData; //!< Piece of the data for the message  
} SXMQueueMsgData;

/** Pre-definition of the message master entry */
typedef struct _sxm_queue_msg_impl SXMQueueMsgImpl;

/** Queue message entry structure */
typedef struct {
    /** Type of message for routing */
    uint msgType;
    /** Size of message in bytes */
    uint msgSize;
    /** Special pointer to the original message instance, this should
     * not be used outside the message queue
     */
    const SXMQueueMsgImpl *impl;
} SXMQueueMsg;

/** Pre-definition of the message queue implementation specific */
typedef struct _sxm_msg_queue_impl_struct SXMQueueImpl;
 
/** Message queue structure */
typedef struct _sxm_msg_queue_struct {	
    /** Unique object signature for robustness */
    uint uniqueSignature;
    /** Queued messages */ 
    SXMList mEntries;
    /** Unused (free) messages */
    SXMList mFreeEntries;
    /** Currently processing messages, i.e. taken by the handler, but not
     * released yet
     */
    SXMList mProcessingEntries;
    /** Protection from multiple threads simultaneous access */
    pthread_mutex_t *pMutex;
    /** Synchronization mechanism */
    SXMSem_t semaphore;
    /** Implementation reference */
    const SXMQueueImpl *impl;

    /** \name Helpers
     * @{
     */
    /** Can be used to debug runtime max queue depth */
    uint maxQueued;
    /** Queue ID / name */
    const char *id;
    /** Debug Logging Level */
    const uint *pDebugLevel;
    /** @} */
} MSG_QUEUE_STRUCT;

/** Declares custom free function prototype
 * \param[in] pData message's content
 */
typedef void (*SXMQueueCustomFree)(void *pData);

/** Infinite wait timeout
 * \note The assumption that this value shall be the biggest positive
 *       one to be used internally with no dedicated checks against it.
 */
#define SXM_QUEUE_INFINITE_TIMEOUT ((uint)-1)

/** Legacy memory reservations */
#define SXM_QUEUE_RESERVE_LEGACY  (0) 
/** Adaptive memory reservation */
#define SXM_QUEUE_RESERVE_ADAPTIVE ((uint)-1)

SXESDK_INTERNAL_API int sxm_queue_create(MSG_QUEUE_STRUCT *pQueue, size_t numOfEntries,
                     pthread_mutex_t* pQueueMutex, size_t entryReserved,
                     const char *id, uint *pDebugLevel);

SXESDK_INTERNAL_API void sxm_queue_destroy(MSG_QUEUE_STRUCT *pQueue);

SXESDK_INTERNAL_API int sxm_queue_get(MSG_QUEUE_STRUCT *pQueue,
                                      SXMQueueMsg *pMsg, uint secs);

SXESDK_INTERNAL_API int sxm_queue_get_ms(MSG_QUEUE_STRUCT *pQueue,
                                         SXMQueueMsg *pMsg,
                                         uint millisecs);

SXESDK_INTERNAL_API void sxm_queue_flush(MSG_QUEUE_STRUCT *pQueue);

SXESDK_INTERNAL_API int sxm_queue_put(MSG_QUEUE_STRUCT *pQueue,
                                      const void *pData, uint msgType, uint msgSize,
                                      SXMQueueMsgPriority msgPriority);
SXESDK_INTERNAL_API int sxm_queue_put_custom(MSG_QUEUE_STRUCT *pQueue,
                                       const SXMQueueMsgData *pData, uint msgType,
                                       uint msgCount, SXMQueueMsgPriority msgPriority,
                                       SXMQueueCustomFree customFree);
SXESDK_INTERNAL_API int sxm_queue_get_msg_count(MSG_QUEUE_STRUCT *pQueue, uint *count);

SXESDK_INTERNAL_API int sxm_queue_get_max_msg_depth_count(MSG_QUEUE_STRUCT *pQueue, uint *count);

SXESDK_INTERNAL_API void* sxm_queue_msg_data(SXMQueueMsg *pMsg);

SXESDK_INTERNAL_API void sxm_queue_msg_release(SXMQueueMsg *pMsg);

#ifdef __cplusplus
}
#endif


#endif /* SXM_MGS_QUEUE_INCL_H_ */
