/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_msg_queue.c
* \author Eugene Lopatukhin
* \date 9/24/2013
* \brief Implementation of message queue using pthreads.
*
* \page msg_q_intro Introduction
*
* The message queue designed to provide thread-safe, based on queue owner's mutex,
* sync way of communication between several threads. This particular implementation
* supports one reader and several writers. 
*
* Writers put new messages to the queue specifying message type, message priority
* and message data if exists. Depending on the mode the queue processed provided
* message's content, attached it to the message and fills it by the caller's content.
* After that the caller can release it if it's not loner needed. This was done to
* improve queue users' experience and provide transparent way of data manipulations.
* 
* Readers should use function sxm_queue_get[_ms]() to monitor message within the
* queue.
*
* Once message detected the function returns \ref SXM_E_OK and provides message as
* SXMQueueMsg instance. After processing the messages the only thing the caller
* shall do is to call sxm_queue_msg_release() to perform graceful message's
* resources recycling.
*
* \section msg_q_rest Restrictions
* -# The queue support predefined number of messages and this number cannot be
*    increased dynamically.
* -# The max size of the single message is 65535 bytes;
*
* \section msg_q_types Queue Mode
* The queue supports three modes which are working with the heap differently.
* The type is set via sxm_queue_create() function by setting the \p entryReserved
* parameter.
*
* \subsection msg_q_legacy Legacy Mode
* The legacy mode is the simplest and straight forward queue type. The memory
* is allocated every time once the message queue is used for a new message to be
* sent. The one exception is if the message's content requires size less or equal
* to the pointer size - in this case the pointer itself is utilized as buffer
* sizeof(void*) bytes length.
*
* In order to activate this mode the queue should be created with the \p entryReserved
* equals to SXM_QUEUE_RESERVE_LEGACY.
*
* \subsection msg_q_res Reserved Mode
* This mode allows the queue owner specifying amount of bytes which should be
* preallocated for each entry. This is a useful for queue if most of the
* time it's being used for messages with the similar sizes. In the same time,
* if by some reason the queue has to process the message longer than the reserved
* amount of memory, dedicated heap size will be requested to serve the request.
*
* In order to activate this mode the queue should be created with the \p entryReserved
* equals to the size of the desired reserved space. In order to optimize heap
* usage on different platforms the queue creation routine check the desired
* size and if it's not greater than pointer size the mode automatically converted
* into \ref msg_q_legacy.
*
* \subsection msg_q_adp Adaptive Mode
* The advanced mode to save memory allocation via maximized reuse of already
* allocated memory. This is a useful for queue which is being used
* for the component where hard to determine what is the average size of the message.
* Each message is managing the biggest ever allocated for it the block of memory.
* Once the queue has to process a new messages the queue check the message's
* available memory and if it's enough use it as-is, if not - reallocate it to
* the required size. Thus, at some point the queue should have a collection of
* messages where each message is connected to it's buffer and most of the time
* this buffer is sufficient to serve queue needs.
*
********************************************************************************/

#define DEBUG_VAR (((NULL != pQueue) && (NULL != pQueue->pDebugLevel)) ? *pQueue->pDebugLevel : 0)
#define DEBUG_TAG "queue"

#include <stdlib.h>
#include <sxm_common.h>
#include <sxm_stdtrace.h>
#include <util/sxm_msg_queue_incl.h>
#include <util/sxm_time_internal.h>
#include <util/sxm_noname_internal.h>
#include <util/sxm_memory_internal.h>

enum {
    /** Queue protection signature */
    QUEUE_SIGNATURE     = 0x12345678U,
    /** Queue protection signature (inverted, aka invalid) */
    QUEUE_SIGNATURE_INV = ~QUEUE_SIGNATURE,
    /** Queue's message protection signature */
    QUEUE_MSG_SIGNATURE = 0xC0FFEE83U,
    /** Queue's message protection signature (inverted, aka invalid) */
    QUEUE_MSG_SIGNATURE_INV = ~QUEUE_MSG_SIGNATURE,
    /** Defines max message size in bytes */
    QUEUE_MSG_SIZE_MAX = (1 << (sizeof(ushort) * SXM_ARCH_BITS_IN_BYTE)) - 1
};

/**
 * \name Private macros
 * @{
 */

/** Locks queue's mutex
 * \param[in] q valid queue reference
 */
#define QUEUE_LOCK(q) pthread_mutex_lock((q)->pMutex)
/** Unlocks queue's mutex
 * \param[in] q valid queue reference
 */
#define QUEUE_UNLOCK(q) pthread_mutex_unlock((q)->pMutex)

/** Check that the message queue is valid
 * \param[in] q valid queue reference
 */
#define QUEUE_VALID(q) \
    (((q)->pMutex != NULL) && (q)->uniqueSignature == QUEUE_SIGNATURE)

/** @} */

/** This is an internal structure which keeps the internals along with
 * the message.
 */
struct _sxm_queue_msg_impl {
    /** Protection signature for the messages which keep valid content;
     * Valid messages keep value defined by #QUEUE_MSG_SIGNATURE
     */
    uint sig;
    /** Corresponding message content which is populated to the caller */
    SXMQueueMsg sMsg;
    /** Custom free function */
    SXMQueueCustomFree customFree;
    /** The host queue */
    MSG_QUEUE_STRUCT *pQueue;
    /** Reference to message's data */
    void *pData;
    /** Amount of memory allocated in the pData */
    ushort dataSize;
};

/** This is an internal structure which keeps the internals along with
* the message queue instance.
*/
struct _sxm_msg_queue_impl_struct {
    /** Prepare message */
    SXMResultCode (*msg_init)(SXMQueueMsgImpl *pImpl, uint msgSize);
    /** Provides message content */
    void* (*msg_data)(SXMQueueMsgImpl *pImpl);
    /** Release content */
    void (*msg_release)(SXMQueueMsgImpl *pImpl, BOOL bDeep);
};

/* Function pre-definition */
static void queue_clear_list(MSG_QUEUE_STRUCT *pQueue, SXMList *pList);
static void queue_msg_release(MSG_QUEUE_STRUCT *pQueue, SXMQueueMsgImpl *pImpl);

static SXMResultCode queue_msg_init_legacy(SXMQueueMsgImpl *pImpl, uint size);
static SXMResultCode queue_msg_init_reserved(SXMQueueMsgImpl *pImpl, uint size);
static SXMResultCode queue_msg_init_adaptive(SXMQueueMsgImpl *pImpl, uint size);

static void queue_msg_release_legacy(SXMQueueMsgImpl *pImpl, BOOL bDeep);
static void queue_msg_release_reserved(SXMQueueMsgImpl *pImpl, BOOL bDeep);
static void queue_msg_release_adaptive(SXMQueueMsgImpl *pImpl, BOOL bDeep);

static void* queue_msg_data_legacy(SXMQueueMsgImpl *pImpl);
static void* queue_msg_data_regular(SXMQueueMsgImpl *pImpl);

/** Legacy messages processing */
static const SXMQueueImpl sLegacyImpl = {
    &queue_msg_init_legacy,
    &queue_msg_data_legacy,
    &queue_msg_release_legacy
};

/** Reserved messages processing */
static const SXMQueueImpl sReservedImpl = {
    &queue_msg_init_reserved,
    &queue_msg_data_regular,
    &queue_msg_release_reserved
};

/** Adaptive messages processing */
static const SXMQueueImpl sAdaptiveImpl = {
    &queue_msg_init_adaptive,
    &queue_msg_data_regular,
    &queue_msg_release_adaptive
};

/***************************************************************************//**
 * Creates message queue
 *
 * The new instance resides within the caller's memory provided via \p pQueue
 * parameter.
 *
 * \param[in,out] pQueue pointer to the message queue structure
 * \param[in] numOfEntries maximum number of entries supported by queue
 * \param[in] queueMutex initialized mutex to provide
 * \param[in] entryReserved amount of memory reserved for each entry; must be less than 64KiB
 * \param[in] id queue identifier which can be useful for debug purposes. The 
 *               function doesn't do a deep copy of it thus the caller shall
 *               manage it by itself.
 * \param[in] pDebugLevel the reference to the debug level the caller controls or
 *                        \c NULL in case if need to use default level
 *
 * \retval SXM_E_OK on successful message queue creation
 * \retval SXM_E_INVAL Invalid argument
 * \retval SXM_E_FAULT in case of NULL argument
 * \retval SXM_E_NOMEM in case lack of memory
 * \retval SXM_E_BUSY the queue already created in this memory
 *
 ******************************************************************************/
int sxm_queue_create(MSG_QUEUE_STRUCT *pQueue, size_t numOfEntries,
                     pthread_mutex_t* pQueueMutex, size_t entryReserved,
                     const char *id, uint *pDebugLevel) {

    int rc;

    /* Check input */
    if ((pQueue == NULL) || (numOfEntries == 0) || (pQueueMutex == NULL)) {
        rc = SXM_E_FAULT;
    }
    else if ((entryReserved >= QUEUE_MSG_SIZE_MAX) && (entryReserved != SXM_QUEUE_RESERVE_ADAPTIVE)) {
        rc = SXM_E_INVAL;
    }
    else if (QUEUE_VALID(pQueue)) {
        rc = SXM_E_BUSY;
    }
    else {
        /* Make sure the queue content is fresh and clean */
        memset(pQueue, 0, sizeof(*pQueue));

        pQueue->pDebugLevel = pDebugLevel;

        ENTER_UTL("pQueue: %p, numOfEntries: %u, pQueueMutex: %p, entryReserved: %u, id: %s, pDebugLevel: %p",
            pQueue, numOfEntries, pQueueMutex, (uint)entryReserved, (NULL != id) ? id : "", pDebugLevel);
        DEBUG_UTL("sizeof(MSG_QUEUE_STRUCT)=%u, sizeof(SXMQueueMsg)=%u, sizeof(SXMQueueMsgImpl)=%u",
            (uint)sizeof(*pQueue), (uint)sizeof(SXMQueueMsg), (uint)sizeof(SXMQueueMsgImpl));

        /* Create lists for messages (queued, unused, processing) */
        rc = sxm_list_create(&pQueue->mEntries, SXM_LIST_PREALLOCATED);
        if (rc == SXM_E_OK) {
            rc = sxm_list_create(&pQueue->mFreeEntries, SXM_LIST_PREALLOCATED);
            if (rc == SXM_E_OK) {
                rc = sxm_list_create(&pQueue->mProcessingEntries, SXM_LIST_PREALLOCATED);
            }
        }
        if (rc == SXM_E_OK) {
            size_t preAllocatedSize;

            /* Pre-process the reserved size to simplify several cases:
            * - If the required size fits into the pointer's size there is no need
            *   in additional memory allocation, i.e. use legacy mode.
            */
            if (entryReserved <= sizeof(((SXMQueueMsgImpl*)0)->pData)) {
                entryReserved = SXM_QUEUE_RESERVE_LEGACY;
            }
            if (entryReserved == SXM_QUEUE_RESERVE_LEGACY) {
                preAllocatedSize = 0;
                pQueue->impl = &sLegacyImpl;
            }
            else if (entryReserved == SXM_QUEUE_RESERVE_ADAPTIVE) {
                preAllocatedSize = 0;
                pQueue->impl = &sAdaptiveImpl;
            }
            else {
                preAllocatedSize = entryReserved;
                pQueue->impl = &sReservedImpl;
            }

            /* Pre-allocate messages for the queue */
            while ((sxm_list_size(&pQueue->mFreeEntries) < numOfEntries) && (rc == SXM_E_OK)) {
                SXMQueueMsgImpl* pEntry;
                pEntry = (SXMQueueMsgImpl*)sxm_list_allocate(sizeof(SXMQueueMsgImpl) + preAllocatedSize);
                if (pEntry != NULL) {
                    rc = sxm_list_add(&pQueue->mFreeEntries, pEntry, TRUE, NULL);
                    if (rc != SXM_E_OK) {
                        (void)sxm_list_free(pEntry);
                    }
                    else {
                        pEntry->sig = (uint)(QUEUE_MSG_SIGNATURE_INV); /* not valid */
                        pEntry->dataSize = (ushort)preAllocatedSize; /* reserved size */
                    }
                }
                else {
                    ERROR_UTL("Failed to allocate entry");
                    rc = SXM_E_NOMEM;
                }
            }

            if (rc == SXM_E_OK) {
                /* Initialize pending messages semaphore */
                rc = sxm_sem_init(&pQueue->semaphore, 0U);
                if (rc != SXM_E_OK) {
                    ERROR_UTL("Failed to init sem, rc=%d", rc);
                    rc = SXM_E_ERROR;
                }
                else {
                    /* Initialize mutex */
                    pQueue->pMutex = pQueueMutex;

                    /* Initialize remaining fields */
                    pQueue->maxQueued = 0U;
                    pQueue->id = (id != NULL) ? id : "";
                    pQueue->uniqueSignature = QUEUE_SIGNATURE;
                }
            }
            if (rc != SXM_E_OK) {
                queue_clear_list(pQueue, &pQueue->mFreeEntries);
            }
        }
    }

    LEAVE_UTL("%s", sxm_sdk_format_result(rc));

    return rc;
}

/***************************************************************************//**
 * Destroys message queue and frees up all the resources associated with it
 *
 * \note The queue structure becomes invalid after this call and shall
 *       not be used for any queue API.
 *
 * \param[in] pQueue pointer to the message queue structure
 *
 ******************************************************************************/
void sxm_queue_destroy(MSG_QUEUE_STRUCT *pQueue) {

    /* make sure got valid pointer */
    if ((pQueue != NULL) && QUEUE_VALID(pQueue)) {
        BOOL bIsValid;

        QUEUE_LOCK(pQueue);

        ENTER_UTL("(pQueue: %p)", pQueue);

        /* Check if the queue is still valid - protection against multiple
         * destroy requests */
        bIsValid = QUEUE_VALID(pQueue) ? TRUE : FALSE;
        if (bIsValid == TRUE) {
            int rc;
            DEBUG_UTL("queue [%s] still valid", pQueue->id);

            /* Invalidate the queue */
            pQueue->uniqueSignature = (uint)(QUEUE_SIGNATURE_INV);
            pQueue->maxQueued = 0U;
            pQueue->pDebugLevel = NULL;

            /* Clean up both lists */
            queue_clear_list(pQueue, &pQueue->mEntries);
            queue_clear_list(pQueue, &pQueue->mFreeEntries);
            queue_clear_list(pQueue, &pQueue->mProcessingEntries);

            /* Notify the thread which may wait for the messages to unblock it
             * and let it stop on the queue mutex lock. Keep in mind that the
             * queue is designed to support only one reader. */
            rc = sxm_sem_post(&pQueue->semaphore);
            if (rc != SXM_E_OK) {
                ERROR_UTL("Failed to make a final post to the queue's sem, rc=%d", rc);
            }
        }

        LEAVE_UTL("");

        /* This unlock should unlock those threads which can seat of the queue's
         * mutex waiting for getting control over it. After this each thread will
         * get the return code from functions they were calling that the queue
         * no longer valid due to invalidated signature */
        QUEUE_UNLOCK(pQueue);

        if (bIsValid == TRUE) {
            /* free up semaphore if the queue used to be a valid one */
            (void) sxm_sem_destroy(&pQueue->semaphore);
        }
    }
    return;
}

/***************************************************************************//**
 * Flushes message queue and frees up all the data currently in the queue
 *
 * \param[in] pQueue pointer to the message queue structure
 *
 ******************************************************************************/
void sxm_queue_flush(MSG_QUEUE_STRUCT *pQueue) {
    /* make sure got valid pointer */
    if ((pQueue != NULL) && QUEUE_VALID(pQueue)) {
        QUEUE_LOCK(pQueue);
        ENTER_UTL("pQueue: %p", pQueue);

        if (QUEUE_VALID(pQueue)) {
            SXMListEntry *pListEntry;
            DEBUG_UTL("queue [%s] still valid", pQueue->id);
            /* go through the list of queued messages and remove data
             * associated with queue entry and move nodes to free list
             */
            while ((pListEntry = sxm_list_first(&pQueue->mEntries)) != NULL) {
                SXMQueueMsgImpl *pMsg = (SXMQueueMsgImpl *)sxm_list_data(pListEntry);
                queue_msg_release(pQueue, pMsg);
                (void) sxm_list_remove(&pQueue->mEntries, pListEntry);
                (void) sxm_list_add(&pQueue->mFreeEntries, pMsg, TRUE, NULL);
            }
        }

        LEAVE_UTL("");
        QUEUE_UNLOCK(pQueue);
    }
    return;
}

/***************************************************************************//**
 * Gets count of how many messages are currently in the queue
 *
 * \param[in] pQueue pointer to the message queue structure
 * \param[out] count if operation successful count will contain number of
 *                   messages in the queue.
 *
 * \retval SXM_E_OK if count was successfully retrieved
 * \retval SXM_E_INVAL if invalid queue reference
 *
 ******************************************************************************/
int sxm_queue_get_msg_count(MSG_QUEUE_STRUCT *pQueue, uint *count) {
    SXMResultCode rc;

    /* make sure got valid pointer */
    if ((pQueue != NULL) && QUEUE_VALID(pQueue)) {
        QUEUE_LOCK(pQueue);
        ENTER_UTL("pQueue: %p, count: %p", pQueue, count);
        if (QUEUE_VALID(pQueue)) {
            DEBUG_UTL("queue [%s] still valid", pQueue->id);
            *count = (uint)sxm_list_size(&pQueue->mEntries);
        }
        else {
            *count = 0U;
        }
        QUEUE_UNLOCK(pQueue);
        DEBUG_UTL("*count: %u", *count);
        rc = SXM_E_OK;
    }
    else {
        rc = SXM_E_INVAL;
    }

    LEAVE_UTL("%s", sxm_sdk_format_result(rc));
    return rc;
}

/***************************************************************************//**
 * Gets count of what was maximum message count in the queue. Can be
 * used to tweak queue sizes and performance evaluation.
 *
 * \param[in] pQueue pointer to the message queue structure
 * \param[out] count if operation successful count will contain number
 *                   of messages in the queue
 *
 * \retval SXM_E_OK if count was successfully retrieved
 * \retval SXM_E_INVAL if invalid queue reference
 *
 ******************************************************************************/
int sxm_queue_get_max_msg_depth_count(MSG_QUEUE_STRUCT *pQueue, uint *count) {
    SXMResultCode rc;

    /* make sure got valid pointer */
    if ((pQueue != NULL) && QUEUE_VALID(pQueue)) {
        QUEUE_LOCK(pQueue);
        ENTER_UTL("(pQueue: %p, count: %p)", pQueue, count);
        if (QUEUE_VALID(pQueue)) {
            DEBUG_UTL("queue [%s] still valid", pQueue->id);
            *count = pQueue->maxQueued;
        }
        else {
            *count = 0U;
        }
        DEBUG_UTL("*count: %u", *count);
        QUEUE_UNLOCK(pQueue);
        rc = SXM_E_OK;
    }
    else {
        rc = SXM_E_INVAL;
    }

    LEAVE_UTL("%s", sxm_sdk_format_result(rc));

    return rc;
}

/***************************************************************************//**
 * Puts message in the queue. This is a nonblocking call and if the queue is full
 * function returns with #SXM_E_RESOURCE return code.
 *
 * \param[in] pQueue pointer to the message queue structure
 * \param[in] pData pointer to message data being placed on the queue; Should be
 *                  valid if the \p msgSize is not zero.
 * \param[in] msgType type of message identifier
 * \param[in] msgSize size of message, in case of 0 the \p pData is ignored
 * \param[in] msgPriority priority of the message. High priority messages
 *                        (#MSG_PRIO_HIGH) are placed in front of the queue and
 *                        normal priority (#MSG_PRIO_NORMAL) messages are placed
 *                        on the back.
 *
 * \retval SXM_E_OK operation successful
 * \retval SXM_E_ERROR queue is broken
 * \retval SXM_E_NOMEM cannot allocate memory for message data
 * \retval SXM_E_INVAL invalid queue
 * \retval SXM_E_RESOURCE queue was full and message was not placed on the queue
 ******************************************************************************/
int sxm_queue_put(MSG_QUEUE_STRUCT *pQueue,
                  const void *pData,
                  uint msgType,
                  uint msgSize,
                  SXMQueueMsgPriority msgPriority)
{
    int rc;
    if (msgSize == 0U) {
        rc = sxm_queue_put_custom(pQueue, NULL, msgType, 0U, msgPriority, NULL);
    }
    else {
        SXMQueueMsgData sData;
        sData.size = msgSize;
        sData.pData = pData;
        rc = sxm_queue_put_custom(pQueue, &sData, msgType, 1U, msgPriority, NULL);
    }
    return rc;
}

/***************************************************************************//**
 * Puts message in the queue. This is a nonblocking call and if the queue is full
 * function returns with #SXM_E_RESOURCE return code.
 *
 * The message content is provided as set of SXMQueueMsgData items where each
 * item addresses single piece of data. The function combine them all together
 * into the solid block without any gaps between them and passes this block
 * through the queue. The number of element is defined by \p msgCount. In case
 * if there is no data to be sent the the \p msgCount should be 0; the \p pData
 * is not checked in this case and can have any value.
 *
 * \param[in] pQueue pointer to the message queue structure
 * \param[in] pData pointer to message data items array being placed on the queue.
 *                  Should be valid if the \p msgCount is not zero.
 * \param[in] msgType type of message identifier
 * \param[in] msgCount number of items referenced by \p pData; in case of 0 the
 *                     \p pData is ignored
 * \param[in] msgPriority priority of the message. High priority messages
 *                        (#MSG_PRIO_HIGH) are placed in front of the queue and
 *                        normal priority (#MSG_PRIO_NORMAL) messages are placed
 *                        on the back.
 * \param[in] customFree customer provided Free function, that will be used to 
                         free the \p pData pointer on message release.
 *
 * \retval SXM_E_OK operation successful
 * \retval SXM_E_ERROR queue is broken
 * \retval SXM_E_NOMEM cannot allocate memory for message data
 * \retval SXM_E_INVAL invalid queue
 * \retval SXM_E_RESOURCE queue was full and message was not placed on the queue
 ******************************************************************************/
int sxm_queue_put_custom(MSG_QUEUE_STRUCT *pQueue,
                         const SXMQueueMsgData *pData,
                         uint msgType,
                         uint msgCount,
                         SXMQueueMsgPriority msgPriority,
                         SXMQueueCustomFree customFree)
{
    int rc;
    uint msgSize = 0U, idx;
    const SXMQueueMsgData *pDataItr = pData;

    /* Count the total size of the message if the data has been provided */
    for (idx = 0U; idx < msgCount; ++idx, ++pDataItr) {
        msgSize += pDataItr->size;
    }
    if (msgSize >= QUEUE_MSG_SIZE_MAX) {
        rc = SXM_E_INVAL;
    }
    /* make sure got valid pointer */
    else if ((pQueue != NULL) && QUEUE_VALID(pQueue)) {
        QUEUE_LOCK(pQueue);

        ENTER_UTL("(pQueue: %p, pData: %p, msgType: %u, msgCount: %u, "
                  "msgPriority: %d, customFree: %p)",
                  pQueue, pData, msgType, msgCount, (int)msgPriority, customFree);

        if (QUEUE_VALID(pQueue)) {
            SXMListEntry *pEntry;
            BOOL bNewEntry = TRUE;

            DEBUG_UTL("queue [%s] still valid", pQueue->id);

            /* get entry from free list */
            pEntry = sxm_list_first(&pQueue->mFreeEntries);

            if ((msgPriority == MSG_PRIO_HIGH) && (pEntry == NULL)) {
                /* The queue is full but high priority message has to be added
                 * so throw away last queued message and reuse its slot.
                 */
                pEntry = sxm_list_last(&pQueue->mEntries);
                if (pEntry != NULL) {
                    SXMQueueMsgImpl *pMsg = (SXMQueueMsgImpl *)sxm_list_data(pEntry);
                    queue_msg_release(pQueue, pMsg);
                    (void) sxm_list_remove(&pQueue->mEntries, pEntry);
                    (void) sxm_list_add(&pQueue->mFreeEntries, pMsg, TRUE, &pEntry);
                    /* We use existing entry for high priority msg, let's update
                     * the new entry flag
                     */
                    bNewEntry = FALSE;
                }
            }

            /* check if got one */
            if (pEntry != NULL) {
                /* Fetch out real message content */
                SXMQueueMsgImpl *pMsg = (SXMQueueMsgImpl *)sxm_list_data(pEntry);

                DEBUG_UTL("Message has %u byte(s) in total", msgSize);
                
                /* Initialize the message's content holder */
                rc = pQueue->impl->msg_init(pMsg, msgSize);
                if (rc == SXM_E_OK) {
                    /* Fetch up free message */
                    rc = sxm_list_remove(&pQueue->mFreeEntries, pEntry);

                    if (rc == SXM_E_OK) {
                        
                        /* Fill up message fields */
                        pMsg->sMsg.msgSize = msgSize;
                        pMsg->sMsg.msgType = msgType;
                        pMsg->customFree = customFree;

                        /* Fill in the message content if needed */
                        if (msgSize > 0U) {
                            byte *pContent = pQueue->impl->msg_data(pMsg);
                            for (idx = 0U, pDataItr = pData; idx < msgCount; ++idx, ++pDataItr) {
                                /* copy the item's data */
                                memcpy(pContent, pDataItr->pData, (size_t)pDataItr->size);
                                /* move the destination */
                                pContent += pDataItr->size;
                            }
                        }

                        /* Put message into the queue based on priority */
                        rc = sxm_list_add(&pQueue->mEntries, pMsg,
                                          ((msgPriority == MSG_PRIO_HIGH) ? TRUE : FALSE), NULL);
                        if (rc == SXM_E_OK) {
                            /* Increment the semaphore to notify waiter */
                            if (bNewEntry == TRUE) {
                                rc = sxm_sem_post(&pQueue->semaphore);
                                if (rc == SXM_E_OK) {
                                    /* adjust runtime count if needed */
                                    if ((uint)sxm_list_size(&pQueue->mEntries) > pQueue->maxQueued) {
                                        pQueue->maxQueued = (uint)sxm_list_size(&pQueue->mEntries);
                                    }
                                }
                                else {
                                    ERROR_UTL("Failed to post sem, rc=%d", rc);
                                }
                            }

                            DEBUG_UTL("Sending message %p, total=%u, max=%u",
                                pMsg, (uint)sxm_list_size(&pQueue->mEntries),
                                pQueue->maxQueued);
                        }
                        else {
                            ERROR_UTL("List is broken");
                            pQueue->impl->msg_release(pMsg, TRUE);
                            sxm_list_free(pMsg); /* Just can release it */
                            rc = SXM_E_ERROR;
                        }
                    }
                    else {
                        ERROR_UTL("List is broken, ret=%d", rc);
                        rc = SXM_E_ERROR;
                    }
                }
                else {
                    ERROR_UTL("Failed to init message, rc=%d", rc);
                }
            }
            else {
                ERROR_UTL("The queue is full");
                rc = SXM_E_RESOURCE;
            }
        }
        else {
            DEBUG_UTL("Invalid queue");
            rc = SXM_E_INVAL;
        }

        LEAVE_UTL("%s", sxm_sdk_format_result(rc));

        QUEUE_UNLOCK(pQueue);
    }
    else {
        rc = SXM_E_INVAL;
    }

    return rc;
}

/***************************************************************************//**
 * Gets message from the queue. Will block for a specified timeout if message
 * is not available.
 *
 * In the case of success the message should be released via call to the
 * sxm_queue_msg_release API. After that the message content will be unavailable.
 * However, the message type and message size will no be forgotten for various
 * need on the caller's side.
 *
 * \param[in] pQueue pointer to the message queue structure
 * \param[in] pMsg pointer to message structure which will be populated with
 *                 queue message retrieved.
 * \param[in] secs Timeout wait in seconds if message is not available.
 *                 If set to #SXM_QUEUE_INFINITE_TIMEOUT there no timeout and
 *                 call to this function will block until message is retrieved.
 *
 * \retval SXM_E_TIMEOUT timeout occurred and no message retrieved
 * \retval SXM_E_INVAL wrong parameter
 * \retval SXM_E_ERROR general failure
 * \retval SXM_E_OK operation successful
 ******************************************************************************/
int sxm_queue_get(MSG_QUEUE_STRUCT *pQueue, SXMQueueMsg *pMsg, uint secs) {
    int rc;
    rc = sxm_queue_get_ms(pQueue, pMsg,
        ((secs != SXM_QUEUE_INFINITE_TIMEOUT) ? (secs * SXM_MILLISEC_PER_SEC) : SXM_QUEUE_INFINITE_TIMEOUT));
    return rc;
}

/***************************************************************************//**
 * Gets message from the queue. Will block for a specified timeout if message
 * is not available.
 *
 * In the case of success the message should be released via call to the
 * sxm_queue_msg_release API. After that the message content will be unavailable.
 * However, the message type and message size will no be forgotten for various
 * need on the caller's side.
 *
 * \param[in] pQueue pointer to the message queue structure
 * \param[out] pMsg pointer to message structure which will be populated with
 *                 queue message retrieved
 * \param[in] millisecs Timeout wait in milliseconds if message is not available.
 *                 If set to #SXM_QUEUE_INFINITE_TIMEOUT there no timeout and call to
 *                 this function will block until message is retrieved.
 *
 * \retval SXM_E_TIMEOUT timeout occurred and no message retrieved. 
 * \retval SXM_E_INVAL wrong parameter
 * \retval SXM_E_ERROR general failure
 * \retval SXM_E_OK operation successful
 ******************************************************************************/
int sxm_queue_get_ms(MSG_QUEUE_STRUCT *pQueue, SXMQueueMsg *pMsg, uint millisecs) {

    int rc;

    if ((pQueue != NULL) && QUEUE_VALID(pQueue) && (pMsg != NULL)) {
        /* check if there is timeout associated with get operation */
        if (millisecs != SXM_QUEUE_INFINITE_TIMEOUT) {
            /* wait for timeout or message to be added to message queue */
            rc = sxm_sem_timedwait(&pQueue->semaphore, millisecs);
        }
        else {
            rc = sxm_sem_wait(&pQueue->semaphore);
        }

        /* check if got out because of timeout or a message */
        if (rc != SXM_E_OK) {
#ifndef SXM_DEBUG_PRODUCTION
            QUEUE_LOCK(pQueue);
            DEBUG_UTL("sxm_sem_(timed)wait finished with %s",
                sxm_sdk_format_result(rc));
            QUEUE_UNLOCK(pQueue);
#endif
        }
        else {
            QUEUE_LOCK(pQueue);
            ENTER_UTL("pQueue: %p, pMsg: %p, millisecs: %u", pQueue, pMsg, millisecs);
            if (QUEUE_VALID(pQueue)) {
                SXMListEntry *pEntry;

                DEBUG_UTL("queue [%s] still valid", pQueue->id);

                /* got a message get reference to it */
                pEntry = sxm_list_first(&pQueue->mEntries);
                if (pEntry != NULL) {
                    SXMQueueMsgImpl *pQueueMsg = (SXMQueueMsgImpl *)sxm_list_data(pEntry);

                    /* move the entry to the processing list */
                    (void) sxm_list_remove(&pQueue->mEntries, pEntry);

                    /* restore links if the message has data */
                    if (pQueueMsg->sMsg.msgSize > 0U) {
                        /* Attach message attributes to be released later */
                        pQueueMsg->sig = (uint)QUEUE_MSG_SIGNATURE;
                        pQueueMsg->pQueue = pQueue;
                        pQueueMsg->sMsg.impl = pQueueMsg;
                        /* Put this list into the processing list since we assume
                         * that the caller should release this message explicitly.
                         */
                        (void)sxm_list_add(&pQueue->mProcessingEntries, pQueueMsg, TRUE, NULL);
                        DEBUG_UTL("Wating for release to message %p", pQueueMsg);
                    }
                    else {
                        pQueueMsg->sig = (uint)(QUEUE_MSG_SIGNATURE_INV);
                        pQueueMsg->pQueue = NULL;
                        pQueueMsg->sMsg.impl = NULL;
                        /* Put this message directly to the free as soon as it
                         * has no attached data, thus, there is no need to wait
                         * for the release request to let it be reused
                         */
                        (void)sxm_list_add(&pQueue->mFreeEntries, pQueueMsg, TRUE, NULL);
                        DEBUG_UTL("The message %p is recycled", pQueueMsg);
                    }

                    /* get data from the message */
                    *pMsg = pQueueMsg->sMsg;

                    rc = SXM_E_OK;
                }
                else {
                    ERROR_UTL("Failed to get message from queue");
                    rc = SXM_E_ERROR;
                }
            }
            else {
                DEBUG_UTL("Invalid queue provided");
                rc = SXM_E_INVAL;
            }

            LEAVE_UTL("%s", sxm_sdk_format_result(rc));

            QUEUE_UNLOCK(pQueue);
        }
    }
    else {
        rc = SXM_E_INVAL;
    }

    return rc;
}

/***************************************************************************//**
 * Function to get message data
 *
 * The caller is responsible for parsing the data based on the content which
 * corresponds to the message type. In the case of using sxm_queue_pus_custom()
 * function the caller also can rely on the items layout within the array back
 * to back with with no gaps, i.e. each elements can be found on offset which equals
 * to the sum of sizes of all prior elements.
 *
 * \note This is a wrapper for free which makes it clearer for programmer
 *       using this implementation that payload lays inside different storages
 *
 * \param[in] pMsg pointer to the queue message
 * \return address of the messages content.
 *
 ******************************************************************************/
void* sxm_queue_msg_data(SXMQueueMsg *pMsg) {
    void *pData;
    if ((pMsg != NULL) && (pMsg->msgSize > 0) && (pMsg->impl != NULL) && (pMsg->impl->sig == QUEUE_MSG_SIGNATURE)) {
        pData = pMsg->impl->pQueue->impl->msg_data((SXMQueueMsgImpl*)pMsg->impl);
    }
    else {
        pData = NULL;
    }
    return pData;
}

/***************************************************************************//**
 * Function to get message data (legacy)
 * \param[in] pImpl pointer to the queue message
 ******************************************************************************/
static void* queue_msg_data_legacy(SXMQueueMsgImpl *pImpl) {
    return (pImpl->sMsg.msgSize > sizeof(pImpl->pData)) ? pImpl->pData : &pImpl->pData;
}

/***************************************************************************//**
 * Function to get message data (reserved and adaptive)
 * \param[in] pImpl pointer to the queue message
 ******************************************************************************/
static void* queue_msg_data_regular(SXMQueueMsgImpl *pImpl) {
    return pImpl->pData;
}

/***************************************************************************//**
 * Function used to perform smart release for the queue messages.
 *
 * After this function call the message becomes invalid except message type and
 * content size since this information resides inside the caller's memory and
 * could be used for various need even after message's content release.
 *
 * \warning In the good scenario the same message instance should be release
 *          only once. However, if by some reason the caller makes a copy of
 *          of the message structure it can try to make a double release which
 *          will cause undefined behavior inside the messages handler code. In
 *          the same time multiple calls to the function for the same message
 *          instance causes no issues since the message will be invalidated after
 *          the call.
 *
 * \note This is a wrapper for free which makes it clearer for programmer
 *       using this implementation that payload needs to be allocated and freed.
 * \param[in] pMsg pointer to the queue message the caller wishes to release
 *
 ******************************************************************************/
void sxm_queue_msg_release(SXMQueueMsg *pMsg) {
    if ((pMsg != NULL) && (pMsg->impl != NULL) && (pMsg->impl->sig == QUEUE_MSG_SIGNATURE)) {
        MSG_QUEUE_STRUCT *pQueue = pMsg->impl->pQueue;
        /* Take out the message queue and put the message back to the queue */
        if ((pQueue != NULL) && QUEUE_VALID(pQueue)) {
            QUEUE_LOCK(pQueue);
            if (QUEUE_VALID(pQueue)) {
                SXMQueueMsgImpl * const pImpl = (SXMQueueMsgImpl *)pMsg->impl;
                SXMListEntry *pEntry = sxm_list_entry(pImpl);
                /* Release message content gracefully */
                queue_msg_release(pQueue, pImpl);

                /* recycle message */
                (void) sxm_list_remove(&pQueue->mProcessingEntries, pEntry);
                (void) sxm_list_add(&pQueue->mFreeEntries, pImpl, TRUE, NULL);
            }
            QUEUE_UNLOCK(pQueue);
            /* make sure the keeper of this message cannot re-use it. */
            pMsg->impl = NULL;
        }
    }
    return;
}

/***************************************************************************//**
 * Initializes the message content (legacy)
 * \param[in] pImpl pointer to the queue message implementation
 * \param[in] size message's content size
 ******************************************************************************/       
static SXMResultCode queue_msg_init_legacy(SXMQueueMsgImpl *pImpl, uint size) {
    SXMResultCode rc = SXM_E_OK;
    const BOOL bHeapData = (size > sizeof(pImpl->pData)) ? TRUE : FALSE;
    /* Clean up data reference */
    pImpl->pData = NULL;
    /* populate entry with with new data and don't connect it */
    if ((size > 0U) && (bHeapData == TRUE)) {
        pImpl->pData = sxe_malloc(size);
        if (pImpl->pData == NULL) {
            rc = SXM_E_NOMEM;
        }
    }
    return rc;
}

/***************************************************************************//**
 * Initializes the message content (reserved)
 * \param[in] pImpl pointer to the queue message implementation
 * \param[in] size message's content size
 * \param[out] ppData address of the available memory
 ******************************************************************************/   
static SXMResultCode queue_msg_init_reserved(SXMQueueMsgImpl *pImpl, uint size) {
    SXMResultCode rc = SXM_E_OK;
    if (size <= pImpl->dataSize) {
        pImpl->pData = pImpl + 1; /* adjust the pointer */
    }
    else {
        /* allocate new block since the messages requires more memory than
         * is reserved
         */
        pImpl->pData = sxe_malloc(size);
        if (pImpl->pData == NULL) {
            rc = SXM_E_NOMEM;
        }
    }
    return rc;
}

/***************************************************************************//**
 * Initializes the message content (adaptive)
 * \param[in] pImpl pointer to the queue message implementation
 * \param[in] size message's content size
 ******************************************************************************/  
static SXMResultCode queue_msg_init_adaptive(SXMQueueMsgImpl *pImpl, uint size) {
    SXMResultCode rc = SXM_E_OK;
    /* Release previous memory block if it too small for the new event */
    if ((size > pImpl->dataSize) && (pImpl->pData != NULL)) {
        /* Try to realloc the same memory block; if this operation failes the 
         * previous block could be releases, since the caller does need to
         * get bigger block that the message has attached
         */
        void *pNewData = sxe_realloc(pImpl->pData, size);
        if (pNewData == NULL) {
            /* release the prev memory */
            sxe_free(pImpl->pData);
            pImpl->pData = NULL;
            pImpl->dataSize = 0U;
            rc = SXM_E_NOMEM;
        }
        else {
            pImpl->pData = pNewData;
            pImpl->dataSize = (ushort)size;
        }
    }
    /* Allocate space for the message if needed; in case if the reallocation
     * has failed we still try to allocate required amount of memory
     * one more time using malloc
     */
    if ((size != 0U) && (pImpl->pData == NULL)) {
        pImpl->pData = sxe_malloc(size);
        if (pImpl->pData == NULL) {
            rc = SXM_E_NOMEM;
        }
        else {
            pImpl->dataSize = (ushort)size;
            rc = SXM_E_OK;
        }
    }
    return rc;
}

/***************************************************************************//**
 * Function used to perform smart release for the queue message with no extra
 * check.
 * \note This is a wrapper for free which makes it clearer for programmer
 *       using this implementation that payload needs to be allocated and freed.
 * \param[in] pQueue message queue
 * \param[in] pImpl pointer to the queue message implementation the caller wishes to release
 ******************************************************************************/
static void queue_msg_release(MSG_QUEUE_STRUCT *pQueue, SXMQueueMsgImpl *pImpl) {
    /* Check if the memory is part of the message */
    pQueue->impl->msg_release(pImpl, FALSE);
    /* make it detached from the implementation */
    pImpl->sig = (uint)(QUEUE_MSG_SIGNATURE_INV);
    pImpl->pQueue = NULL;
    pImpl->sMsg.impl = NULL;
    return;
}

/***************************************************************************//**
 * Function used to perform smart release for the queue message with no extra
 * check (legacy)
 * \param[in] pImpl pointer to the queue message implementation the caller wishes to release
 * \param[in] bDeep Deep releasing
 ******************************************************************************/
static void queue_msg_release_legacy(SXMQueueMsgImpl *pImpl, BOOL bDeep) {
    UNUSED_VAR(bDeep);
    /* here it is allowed to get rid of *const* and modify the value */
    if ((pImpl->pData != NULL) && (pImpl->sMsg.msgSize > sizeof(pImpl->pData))) {
        if (pImpl->customFree != NULL) {
            pImpl->customFree(pImpl->pData);
            pImpl->customFree = NULL;
        }
        sxe_free(pImpl->pData);
    }
    else if (pImpl->customFree != NULL) {
        pImpl->customFree(&pImpl->pData);
        pImpl->customFree = NULL;
    }
    pImpl->pData = NULL; /* invalidate the pointer */
    return;
}

/***************************************************************************//**
 * Function used to perform smart release for the queue message with no extra
 * check (reserved)
 * \param[in] pImpl pointer to the queue message implementation the caller wishes to release
 * \param[in] bDeep Deep releasing
 ******************************************************************************/
static void queue_msg_release_reserved(SXMQueueMsgImpl *pImpl, BOOL bDeep) {
    UNUSED_VAR(bDeep);
    if (pImpl->customFree != NULL) {
        pImpl->customFree(pImpl->pData);
        pImpl->customFree = NULL;
    }
    /* Check if the memory is not part of the message - release it */
    if (pImpl->pData != NULL) {
        if (pImpl->pData != (pImpl + 1)) {
            sxe_free(pImpl->pData);
        }
        pImpl->pData = NULL; /* invalidate the pointer */
    }
    return;
}

/***************************************************************************//**
 * Function used to perform smart release for the queue message with no extra
 * check (adaptive)
 * \param[in] pImpl pointer to the queue message implementation the caller wishes to release
 * \param[in] bDeep Deep releasing
 ******************************************************************************/
static void queue_msg_release_adaptive(SXMQueueMsgImpl *pImpl, BOOL bDeep) {
    if (pImpl->customFree != NULL) {
        pImpl->customFree(pImpl->pData);
        pImpl->customFree = NULL;
    }
    if ((bDeep == TRUE) && (pImpl->pData != NULL)) {
        sxe_free(pImpl->pData);
    }
    return;
}

/***************************************************************************//**
 * Helper function to remove all entries from the list of messages cleaning
 * up connected aux data.
 *
 * \param[in] pQueue list owner
 * \param[in] pList reference to the list to be cleaned up.
 ******************************************************************************/
static void queue_clear_list(MSG_QUEUE_STRUCT *pQueue, SXMList *pList) {
    SXMListEntry *pListEntry;
    const SXMQueueImpl *pImpl = pQueue->impl;

    ENTER_UTL("pQueue: %p, pList: %p, list has %u item(s)",
              pQueue, pList, (uint)sxm_list_size(pList));

    while ((pListEntry = sxm_list_first(pList)) != NULL) {
        SXMQueueMsgImpl *pMsg = (SXMQueueMsgImpl *)sxm_list_data(pListEntry);
        pImpl->msg_release(pMsg, TRUE);
        sxm_list_remove(pList, pListEntry);
        sxm_list_free(pMsg);
    }

    LEAVE_UTL("");

    return;
}
