/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
*   \file sxm_mempool.c
*   \author Leslie French
*   \date 9/23/2013
*   \brief SXM Memory Pool Implementation. Refer to \ref sxe_mempool_page for
*          implementation details.
*
*   \page sxe_mempool_page SXe Memory Pool
*
*   \brief The mempool utility manages a pool of fixed-sized blocks, but allows
*    allocations of more than one contiguous block at a time, up to 32-blocks
*    at a time.
*   \section sxe_mempool_page_intro Introduction
*
*    A SXM memory pool is a contiguous area of memory dynamically allocated
*    from the process heap, and partitioned into groups of 32 fixed sized
*    memory blocks.  The number of 32-block groups (wsize), and the size of
*    the blocks within the groups (bsize), are both specified when the memory
*    pool is created.  A memory pool is created by invoking the 
*    \ref sxm_mempool_new function with the desired sizing parameters.
*
*    Memory allocation from the SXM memory pool consumes one or more contiguous
*    memory blocks taken from a single 32-block group.  Memory allocation across
*    group boundaries is not supported.  Memory is allocated from the pool by
*    invoking the \ref sxm_mempool_alloc function.
*
*    As an example, consider a memory pool consisting of 2 32-block groups, with
*    a block size of 1K bytes (see below). The total capacity of the memory pool
*    is 64K bytes (2 groups x 32 blocks per group x 1K bytes per block). 
*    The maximum number of bytes per single allocation under this memory pool 
*    configuration is 32K, which is the capacity of a single 32-block group.
*    An attempt to allocate more than 32K bytes will fail, even though the 
*    memory pool has a 64K-byte capacity and all its blocks are available.  
*    Multiple allocations within a single group, however, is supported, such 
*    that 4 consecutive allocations  of 16K bytes will succeed, resulting in 
*    the entire 64K-byte memory pool being consumed, 2 16K-byte allocations 
*    from the first group, and two 16K-byte allocations from the second group.
*
*    \section sxe_mempool_page_example Example
*    
*    ~~~~~~~~~~~
*    SXM MEMORY POOL EXAMPLE : TWO 32-BLOCK GROUPS -- 1K BYTES PER BLOCK
*    ~~~~~~~~~~~
*
*    | 0 | 1 | 2 | 3 | 4 | 5 | . . . | 15 | 16 | 17 | 18 |....| 30 | 31 | (blocks) |
*    |:-:|:-:|:-:|:-:|:-:|:-:|:------|:--:|:--:|:--:|:--:|:---|:--:|:--:|:--------:|
*    | A | A | A | A | A | A | . . . |  A |  B |  B |  B |....|  B |  B | GROUP 0  |
*    | C | C | C | C | C | C | . . . |  C |  D |  D |  D |....|  D |  D | GROUP 1  |
*
*    For the four allocations (A -- D).
*
*    Note that allocations can be made and freed in any order, such that
*    allocation 'C' may be freed before allocation 'B' etc.  Note also, that
*    it is clear from the example above, that making allocations in multiples
*    of the block size minimizes unused (wasted) space in the pool.
*
********************************************************************************/

#define DEBUG_TAG "mempool"
#define DEBUG_VAR ((self->pDebugLevel) ? *self->pDebugLevel : 1)

#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>

/** pool overhead */
#define SXM_MEMPOOL_OVERHEAD 64

/***************************************************************************//**
* Allocate bytes from the SXM memory pool
*
* If the allocation is successful, the \a out structure is filled-in
* by the \ref sxm_mempool_alloc function, and is used to bind the allocation to the
* memory pool in question.  This structure is then used later for deallocation.
* Do not under any circumstances modify the members of this structure. Doing so
* will cause unpredictable results such as segmentation faults and application
* program crashes at worst, and memory leaks at best.  A pointer to the SXMMemLoc
* structure (or a pointer to a copy thereof) must then be used as input to the
* \ref sxm_mempool_free function.
*
* \note The maximum number of bytes per allocation cannot exceed \b 32 blocks.
*
* \param[in] self pointer to the memory pool
* \param[in] size number of bytes to allocate
* \param[in] out pointer to a tracking structure
*
* \return a valid pointer to the allocated memory or
*         \c NULL in case of failure - memory not available
*
********************************************************************************/
void *sxm_mempool_alloc(SXMMemPool *self, size_t size, SXMMemLoc *out) {
    uint i, j, k, mask;

    out->count = (ushort)((size + self->mask) >> self->shift);

    if (out->count > 32)
        return NULL;
    else if (out->count == 32)
        mask = 0xffffffffU;
    else
        mask = (1U << out->count) - 1U;

    j = 32U - out->count;

    //  find a block pool with 'out->count' contiguous free blocks
    for (i = 0; i < self->count; i++)
        for (k = 0; k <= j; k++)
            if ((self->bits[i] & (mask << k)) == 0) {
                self->bits[i] |= (mask << k);
                if (i * 32U + k + out->count > self->max)
                    self->max = i * 32U + k + out->count;

                out->index = (ushort)(i * 32U + k);
                return self->base + (out->index << self->shift);
            }

#ifdef _DEBUG
    // allocation failed
    ERROR_UTL("No free slots: self=%08x size=%d", self, size);
    for (i = 0; i < self->count; i++)
        DEBUG_UTL("%08x ", self->bits[i]);
#endif

    return NULL;
}

/***************************************************************************//**
* Allocate bytes from the SXM memory pool and copy data to it.
*
* \note The structure pointed to by the \a out parameter is initialized in
* order to track the allocated memory in the pool, and must be used unaltered
* when freeing this memory from the pool.  See sxm_mempool_alloc for more details.

* \param[in] self the memory pool to be used for the allocation
* \param[in] data the data to be saved in the pool
* \param[in] size the size of the data to be saved, in bytes
* \param[out] out the returned block-group reference for the saved data
*
* \return a valid pointer to the allocated memory or
*         \c NULL in case of failure - memory not available
*
********************************************************************************/
void *sxm_mempool_save(SXMMemPool *self, const void *data, size_t size, SXMMemLoc *out) {
    void *ret = sxm_mempool_alloc(self, size, out);

    if (ret)
        memcpy(ret, data, size);
    return ret;
}

/***************************************************************************//**
* Frees memory previously allocated with \ref sxm_mempool_alloc
*
* The \a memLoc structure pointer \b MUST point to the same structure that
* was used when \ref sxm_mempool_alloc was invoked to track the area of memory
* that is now being freed from the pool.
*
* \param[in] self the pool to which the group is to be returned
* \param[in,out] memLoc the block-group reference for the group
*
********************************************************************************/
void sxm_mempool_free(SXMMemPool *self, SXMMemLoc *memLoc) {
    if (memLoc->count)
        while (memLoc->count --> 0)
            BITC(self->bits, (memLoc->index + memLoc->count));
}

/***************************************************************************//**
* Deletes an SXM memory pool
*
* \param[in] self the pool to be freed
*
********************************************************************************/
void sxm_mempool_delete(SXMMemPool *self) {
    sxe_free(self);
}

/***************************************************************************//**
* Creates an SXM memory pool.
*
* \param[in] bsize the size of each memory block in the pool, in bytes
* \param[in] wsize the number of 32-block groups
* \param[in] pDebugLevel the reference to the debug level the caller controls or
*                        \c NULL in case if need to use default level
*
* \return valid address of the the SXM Memory Pool control structure or
*         \c NULL memory pool creation failed, memory not available or
*          parameters are invalid
*
********************************************************************************/
SXMMemPool *sxm_mempool_new(size_t bsize, size_t wsize, uint *pDebugLevel) {
    SXMMemPool *ret;

    // zero and negative sizes not allowed
    if ((wsize == 0) || (bsize == 0))
    return NULL;

    // allocate the memory pool
    ret = (SXMMemPool *)sxe_malloc(SXM_MEMPOOL_OVERHEAD + (wsize * sizeof(int)) +
                   (wsize * 32 * bsize));
    if (ret == NULL)
        return NULL;

    // initialize the controlling structure fields
    ret->bits = (uint *) (ret + 1);
    ret->count = (uint)wsize;
    ret->base = (byte*) (ret->bits + wsize);
    ret->max = 0;
    ret->mask = (uint)bsize - 1;
    ret->shift = 0;
    ret->pDebugLevel = pDebugLevel;
    while ((bsize >>= 1) > 0)
        ret->shift++;

    // clear the bit maps
    memset(ret->bits, 0, sizeof(int)*wsize);

    return ret;
}

/***************************************************************************//**
* Optionally log memory pool statistics and return empty state.
*
* \param[in] self the memory pool to be dumped
* \param[in] print if non-zero, log memory pool statistics
*
* \retval 0 the SXM memory pool is non-empty
* \retval not-0 the SXM memory pool is empty
*
********************************************************************************/
int sxm_mempool_dump_state(SXMMemPool *self, int print) {
    uint i;
    int empty = !0;

    if (print)
        DEBUG_UTL("Mempool(%08x): bsize=%d wsize=%d", self, self->mask + 1, self->count);

    for (i = 0; i < self->count; i++) {
        if (self->bits[i])
            empty = 0;
        if (print)
            DEBUG_UTL("%08x", self->bits[i]);
    }

    if (print)
        DEBUG_UTL("Mempool(%08x): %s", self, empty ? "IsEmpty" : "BlocksInUse");

    return empty;
}
