/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 *
 *  \file sxm_memory_internal.h
 *  \author Alexander Pylchagin
 *  \date 4/19/2016
 *  \brief Sharable memory related primitives
 *
 ******************************************************************************/

#ifndef SXM_MEMORY_INTERNAL_H_
#define SXM_MEMORY_INTERNAL_H_

#include "sxm_build.h"
#include "sxm_typ.h"

/************************************************************************
 *                                                                      *
 *            Memory Pool                                               *
 *            ===========                                               *
 *                                                                      *
 ************************************************************************/

/** Memory pool control structure */
typedef struct
{
    byte *base; //!< Memory base address
    uint *bits; //<! Allocation bitmask
    uint count; //!< Number to 32-block groups
    uint mask; //!< Block size mask
    uint shift; //!< Allocation mask shift
    uint max; //!< Max allocation blocks
    const uint *pDebugLevel; //!< debug level reference
} SXMMemPool;

/** Memory allocation descriptor */
typedef struct
{
    ushort index; //!< The first block index
    ushort count; //!< Number of blocks;
} SXMMemLoc;

extern SXESDK_INTERNAL_API SXMMemPool *sxm_mempool_new(size_t,      // bsize
                                                       size_t,      // wsize
                                                       uint*);      // pDebugLevel

extern SXESDK_INTERNAL_API void sxm_mempool_delete(SXMMemPool *);   // self


extern SXESDK_INTERNAL_API void *sxm_mempool_alloc(SXMMemPool *,    // self
                                                   size_t,          // size
                                                   SXMMemLoc *);    // out

extern SXESDK_INTERNAL_API void sxm_mempool_free(SXMMemPool *,      // self
                                                 SXMMemLoc *);      // memLoc

extern SXESDK_INTERNAL_API void *sxm_mempool_save(SXMMemPool *,     // self
                                                  const void *,     // data
                                                  size_t,           // size
                                                  SXMMemLoc *);     // out

extern SXESDK_INTERNAL_API int sxm_mempool_dump_state(SXMMemPool*,  // self
                                                      int);         // print

/***************************************************************************//**
 * Provides pool's block size in bytes
 * \param[in] _self a valid memory pool
 * \return block size in bytes
 ********************************************************************************/
#define sxm_mempool_get_bsize(_self) ((_self)->mask + 1)

 /***************************************************************************//**
 * Provides number of 32-block groups
 * \param[in] _self a valid memory pool
 * \return real number of blocks kept by the pool 
 ********************************************************************************/
#define sxm_mempool_get_wsize(_self) ((_self)->count)
     
/***************************************************************************//**
 * Provides real number of blocks kept by the pool
 * \param[in] _self a valid memory pool
 * \return real number of blocks kept by the pool
 ********************************************************************************/
#define sxm_mempool_get_bcount(_self) (sxm_mempool_get_wsize(_self) * 32)

 /***************************************************************************//**
 * Provides real size of the allocated block in bytes
 * \param[in] _self a valid memory pool
 * \param[in] _loc memory allocation descriptor
 * \return real number of blocks kept by the pool 
 ********************************************************************************/
#define sxm_mempool_get_memloc_size(_self, _loc) \
    (_loc.count * sxm_mempool_get_bsize(_self))

 /***************************************************************************//**
 * Resets the memory pool
 * \param[in] _self a valid memory pool
 ********************************************************************************/
#define sxm_mempool_reset(_self) \
    memset((_self)->bits, 0, sizeof(*(_self)->bits) * ((_self)->count))

/** \name Fake memory management functions
 * \brief This set is used to notify SXe developers that they try to use plain libc
 *        functions.
 * @{
 */
#if SXM_DEBUG_MEMORY_TRACKING
extern SXESDK_INTERNAL_API void* sxe_malloc_dbg(size_t size, const char* fl, size_t ln);

extern SXESDK_INTERNAL_API void* sxe_calloc_dbg(size_t count, size_t size,
                                                const char* fl, size_t ln);
extern SXESDK_INTERNAL_API void* sxe_realloc_dbg(void *p, size_t size,
                                                 const char* fl, size_t ln);

extern SXESDK_INTERNAL_API void sxe_free_dbg(void* p);

extern SXESDK_INTERNAL_API void sxe_mem_stat(size_t *totalBlocks, size_t *totalSize,
                                             size_t *maxBlocks, size_t *maxSize);
#ifndef sxe_calloc
#   define sxe_calloc(_c,_s)   sxe_calloc_dbg((size_t)(_c), (size_t)(_s), __FILE__, __LINE__)
#endif

#ifndef sxe_malloc
#   define sxe_malloc(_s)      sxe_malloc_dbg((size_t)(_s), __FILE__, __LINE__)
#endif

#ifndef sxm_realloc
#   define sxe_realloc(_p, _s) sxe_realloc_dbg((_p), (size_t)(_s), __FILE__, __LINE__) 
#endif

#ifndef sxe_free
#   define sxe_free(_p)        sxe_free_dbg((_p))
#endif
    
#ifdef malloc
#   undef malloc
#endif
#define malloc(_s) dont_use_libc_malloc_use_sxe_malloc

#ifdef calloc
#   undef calloc
#endif
#define calloc(_c, _s) dont_use_libc_calloc_use_sxe_calloc

#ifdef realloc
#   undef realloc
#endif
#define realloc(_s) dont_use_libc_realloc_use_sxe_realloc

#ifdef free
#   undef free
#endif
#define free(_p) dont_use_libc_free_use_sxe_free

#else
#   ifndef sxe_calloc
#       define sxe_calloc(_c, _s) calloc((size_t)(_c), (size_t)(_s))
#   endif
#   ifndef sxe_malloc
#       define sxe_malloc(_s) malloc((size_t)(_s))
#   endif
#   ifndef sxe_realloc
#       define sxe_realloc(_p, _s) realloc((_p), (size_t)(_s))
#   endif
#   ifndef sxe_free
#       define sxe_free(_p) free(_p)
#   endif
#endif

#endif /* SXM_MEMORY_H_ */
