/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 *
 * \file sxm_lli.c
 * \author Leslie French $
 * \date 8/1/2014
 * \brief Implementation of the common Low-level-interface API
 *        for components using the SDK directly.
 *
 ********************************************************************************/
#if !defined(SXM_USE_SMS)
#include <sxm_build.h>

#if (!SXM_USE_GEN8)

#define DEBUG_VAR (currService->ds->debug_level)
#define DEBUG_TAG ((currService != NULL) ? &currService->ds->service[0] : "")


#include <sxm_sdk.h>
#include <util/sxm_msg_queue_incl.h>
#include <sxi/sxm_sxi_internal.h>
#include "sxm_link.h"
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>
#include <util/sxm_iomanager_internal.h>

#include "lli.h"

/* LLI service thread message queue size */
#define LLI_MSG_QUEUE_SIZE          (16)

/** LLI Monitor Messages */
enum {
    LLI_DATAPACKET_MSG_TYPE    = 0xFFF9, //!< New data packet
    LLI_RFD_UPDATED_MSG_TYPE   = 0xFFFA, //!< RFD Updated
    LLI_TIME_MSG_TYPE          = 0xFFFB, //!< Time received
    LLI_CONN_READY_MSG_TYPE    = 0xFFFC, //!< Connection is ready 
    LLI_SUBS_UPDATE_MSG_TYPE   = 0xFFFD, //!< Subscription update
    LLI_LINK_LOSS_MSG_TYPE     = 0xFFFE, //!< Link/Connection loss
    LLI_STOP_MSG_TYPE          = 0xFFFF  //!< Stop message processing / stop thread
};

/** Defines size of the max supported message */
#define LLI_MAX_MSG_SIZE            (2048)
/** Defines thread name prefix */
#define LLI_THREAD_NAME_PREFIX "lli_"
/** Defines macro to pack dsi index into the message's content */
#define LLI_MAKE_PARAM(_dsi, _dsi_idx) (((_dsi) & 0xFFFF) | ((_dsi_idx) << 16))
/** Defines macro to unpack dsi index */
#define LLI_UNPACK_DSI(_v) ((_v) & 0xFFFF)
/** Defines macro to unpack original param aka dsi */
#define LLI_UNPACK_DSI_IDX(_v) (((_v) >> 16) & 0xFFFF)

/** Calculates index of assembly from dsi/dmi indexes */
#define LLI_ASSEMBLY_INDEX(_dsi_idx, _dmi_idx) (_dsi_idx * SXI_SP_MAX_DMI_CNT + _dmi_idx)

enum {
    /** Defines location of DMI in DataPacket message */
    SXM_LLI_DMI_OFFSET = 4,
    /** Defines location of Packet start in DataPacket message*/
    SXM_LLI_DATA_OFFSET = 8,
};

typedef struct sxmsdkservice
{
    struct sxmsdkservice *link; //!< Reference to the next item inside the list
    SXMDataService *ds;         //!< Reference to the DS structure
    pthread_t pid;              //!< Monitor thread handle
    unsigned int radio;         //!< Data provider device's handle
    MSG_QUEUE_STRUCT mq;        //!< LLI queue        
    pthread_mutex_t qmutex;     //!< Queue attached mutex
    pthread_mutex_t amutex;     //!< Assemblies protecting mutex
    SXMSdtp **assemblies;       //!< Array of assemblies for each service DSI/DMI
    uint asize;                 //!< Size of array of assemblies
    SXMList sar_pool;           //!< Pool of assemblies
    SXMRfdService rfd;          //!< RFD control structure
    uint timer_saved;           //!< Time of the latest timer notification
} SXMSDKService;

/** The structure used to pass payload data to a service monitor thread */
typedef struct {
    ushort size;                //!< Payload size
    byte dmi_idx;               //!< Index of related DMI in array of DMIs
    byte dsi_idx;               //!< Index of related DSI in array of DSIs
} LLIDataPacket;

/** Data services list protection */
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

/** Active data services linked list */
static SXMSDKService *root = NULL;

/***************************************************************************//**
 * Rebuilds a list of DMIs to filter for a given list of DSIs.
 *
 * \param[in,out] fl Pointer to a list of DSIs to filter.
 *
 ******************************************************************************/
static void rebuild_filter(SXMDataDsi *fl)
{
    ushort dmis[sizeof(fl->filter) * 8 /* bits per byte */];
    uint j;
    UINT8 dc = 0;

    for  (j = 0; (dc < ARRAY_SIZE(dmis)) && (j < fl->dmi.dmic); j++)
    {
        if  (BITP(fl->filter, j))
        {
            dmis[dc++] = fl->dmi.dmis[j];
        }
    }

    sxm_sxi_filter_dsi(SXI_FILTER_DMI_START, fl->dsi, dc, dmis);
    return;
}

/***************************************************************************//**
 * Removing assembly from DMI which is filtered out
 *
 * \param[in] currService main service structure
 * \param[in] idx index of assembly
 *
 ******************************************************************************/
static void remove_assembly(SXMSDKService *currService, uint idx)
{
    LOCK(currService->amutex);

    if (currService->assemblies[idx] != NULL)
    {
        /* placing assembly back to pool since its not needed */
        int rc;
        SXMSdtp *assembly = currService->assemblies[idx];
        currService->assemblies[idx] = NULL;
        rc = sxm_list_add(&currService->sar_pool, assembly, FALSE, NULL);
        if (rc != SXM_E_OK)
        {
            sxm_sdtp_delete(assembly);
            ERROR_LLI("Failed to return assembly to pool (rc=%d)", rc);
        }
    }

    UNLOCK(currService->amutex);
    
    return;
}

/***************************************************************************//**
 * Deallocating assemblies which are in the pool
 *
 * \param[in] pool List of assemblies
 *
 ******************************************************************************/
static void clean_pool(SXMList *pool)
{
    SXMListEntry *entry;

    entry = sxm_list_first(pool);
    while (entry != NULL)
    {
        int rc;

        rc = sxm_list_remove(pool, entry);
        if (rc == SXM_E_OK)
        {
            sxm_sdtp_delete((SXMSdtp *)sxm_list_data(entry));
            entry = sxm_list_first(pool);
        }
        else
        {
            entry = NULL;
        }
    }

    return;
}

/***************************************************************************//**
 * This function passes provided payloads to a service monitor thread
 *
 * \param[in] currService main service structure
 * \param[in] payload received payload
 * \param[in] size number of bytes in payload
 * \param[in] dsi_idx an index of DSI structure in a service list of DSIs 
 *                    for which payload is received
 * \param[in] dmi_idx an index of DMI in the DSI list of DMIs for
 *                    for which payload is received
 *
 * \retval SXM_E_OK data has been placed to queue
 * \retval SXM_E_NOMEM not enough memory to allocate buffer for data
 * \retval SXM_E_ERROR queue is broken
 * \retval SXM_E_RESOURCE queue is full
 *
 ******************************************************************************/
static int post_datapacket(SXMSDKService *currService, const void *payload, 
                           ushort size, byte dsi_idx, byte dmi_idx)
{
    int rc;
    LLIDataPacket packet;
    SXMQueueMsgData sData[2];
    packet.dmi_idx = dmi_idx;
    packet.dsi_idx = dsi_idx;
    packet.size = size;

    sData[0].pData = &packet;
    sData[0].size = sizeof(packet);
    sData[1].pData = payload;
    sData[1].size = size;

    /* We have got our data packet */
    rc = sxm_queue_put_custom(&currService->mq, &sData[0], LLI_DATAPACKET_MSG_TYPE,
                              ARRAY_SIZE(sData), MSG_PRIO_NORMAL, NULL);

    return rc;
}

/***************************************************************************//**
 * This function processes sdtp packets
 *
 * \param[in] currService main service structure
 * \param[in] packet A structure containing message info
 * \param[in] payload message's payload
 *
 ******************************************************************************/
static void process_sdtp_packet(SXMSDKService *currService, LLIDataPacket *packet, byte *payload)
{
    /* determining state of assembly for dmi */
    SXMSdtp *assembly = NULL;
    uint idx = LLI_ASSEMBLY_INDEX(packet->dsi_idx, packet->dmi_idx);

    LOCK(currService->amutex);

    if (currService->assemblies[idx] == NULL)
    {
        /* assembly is not assigned, looking in a pool */
        SXMListEntry *vacant = sxm_list_first(&currService->sar_pool);

        if (vacant == NULL)
        {
            /* in case if pool is empty, let's create new entry */
            assembly = sxm_sdtp_new(currService->ds->maxau);
            if (assembly == NULL)
            {
                ERROR_LLI("Cannot allocate assembly.");
            }
        }
        else
        {
            int rc = sxm_list_remove(&currService->sar_pool, vacant);
            if (rc != SXM_E_OK)
            {
                ERROR_LLI("Cannot take assembly from pool (rc=%d)", rc);
            }
            else
            {
                assembly = sxm_list_data(vacant);
            }
        }

        currService->assemblies[idx] = assembly;
    }
    else
    {
        assembly = currService->assemblies[idx];
    }

    UNLOCK(currService->amutex);

    if (assembly != NULL)
    {
        int result = sxm_sdtp_add(assembly, payload + SXM_LLI_DATA_OFFSET);

        if (result == SXM_E_OK)
        {
            int rc;

            DEBUG_LLI("completed sdtp packet %4d bytes", assembly->pl);
            if (currService->ds->complete_ex != NULL)
            {
                currService->ds->complete_ex(assembly->buffer, assembly->pl,
                    currService->ds->dsuser, currService->ds->dsis[packet->dsi_idx].dsi,
                    currService->ds->dsis[packet->dsi_idx].dmi.dmis[packet->dmi_idx]);
            }
            else
            {
                currService->ds->complete(assembly->buffer, (int)assembly->pl, currService->ds->dsuser);
            }

            LOCK(currService->amutex);
            /* packet has been assembled and processed by data service,
               assembly can be placed back to pool */
            if (currService->assemblies[idx] == assembly)
            {
                currService->assemblies[idx] = NULL;
                rc = sxm_list_add(&currService->sar_pool, assembly, FALSE, NULL);
                if (rc != SXM_E_OK)
                {
                    ERROR_LLI("cannot place assembly to pool (rc=%d), removing.",
                        currService->ds->service, rc);
                    sxm_sdtp_delete(assembly);
                }
            }
            UNLOCK(currService->amutex);
        }
        else if (result == SXM_E_RESOURCE)
        {
            ERROR_LLI("buffer is too small to assemble AU");
        }
        else
        {
            DEBUG_LLI("sxm_sdtp_add returned %d", result);
        }
    }

    return;
}

/***************************************************************************//**
 * Notifies the LLI about time update, called when TimeInd is received in
 * sxi thread.
 *
 * \param[in] t  Pointer to the current time in SXMTime structure 
 * 
 ******************************************************************************/
void sxm_lli_time_update(const SXMTime *t)
{
    SXMSDKService *currService;
    uint now = 0U;

    LOCK(mutex);
    for (currService = root; currService; currService = currService->link)
    {
        if (currService->ds->timer != NULL)
        {
            int rc;

            if (now == 0U)
            {
                now = sxm_sxe_time_encode_local((SXMTime*)t);
            }

            rc = sxm_queue_put(&currService->mq, &now, LLI_TIME_MSG_TYPE,
                               sizeof(now), MSG_PRIO_NORMAL);

            DEBUG_LLI("Posted time %u (rc=%d)", now, rc);
        }
    }
    UNLOCK(mutex);
    return;
}

/***************************************************************************//**
 *
 * This callback function receives the messages from Link Layer.
 *
 * \param[in] type Message type
 * \param[in] payload Message payload
 * \param[in] size Size of the payload
 * \param[in] user Pointer to the user defined data, message queue
 *
 ******************************************************************************/
static int sxm_lli_link_callback(int type, const byte * payload,
                                 uint size, void * user)
{
    int rc = SXM_E_OK;
    SXMSDKService *currService = (SXMSDKService *)user;

    switch (type) {
        case LNK_MSG_PAYLOAD_TYPE:
        {
            int op = x2(payload, LNK_OPCODE_MSB_OFFSET);
            if (LNK_UART_LINK_LOSS_OPCODE == op)
            {
                /* This is a link loss opcode, lets notify the thread */
                rc = sxm_queue_put(&currService->mq, NULL,
                                    LLI_LINK_LOSS_MSG_TYPE,
                                    0, MSG_PRIO_HIGH);
                if (rc != SXM_E_OK)
                {
                    PLOG_LLI("Failed to post link loss (rc=%d)", rc);
                }
                else
                {
                    ERROR_LLI("Posted link loss");
                }
            }
            break;
        }
        case SXI_DATA_MSG_PAYLOAD_TYPE:
        {
            const ushort dmi = (ushort)((payload[SXM_LLI_DMI_OFFSET] << 8) | payload[SXM_LLI_DMI_OFFSET + 1]);
            uint i;
            BOOL found = FALSE;

            for (i = 0; (i < ARRAY_SIZE(currService->ds->dsis)) && (found == FALSE); ++i)
            {
                if (currService->ds->dsis[i].dsi != 0U)
                {
                    uint j;
                    SXMDataDsi *fl = &currService->ds->dsis[i];
                    for (j = 0; (j < fl->dmi.dmic) && (found == FALSE); ++j)
                    {
                        if (fl->dmi.dmis[j] == dmi)
                        {
                            found = TRUE;
                            if (BITP(fl->filter, j))
                            {
                                /* We have got our data packet */
                                rc = post_datapacket(currService, payload, (ushort)size, (byte)i, (byte)j);
                                DEBUG_LLI("Posted datapacket DMI %u size %u"
                                          " (rc=%d)", dmi, size, rc);
                            }
                        }
                    }
                }
            }
            break;
        }
        default:
            break;
    }

    return rc;
}

/***************************************************************************//**
 *
 * This function collects and sets up the DMIs to monitor for the given Service.
 *
 * SXe this is running on the SXi thread, so post the callback data
 * into the service structure and let the monitor thread pick it up.
 *
 * \param[in] type Should be set to #SXM_CALLBACK_SUBS to indicate this is a
 *                 subscription based callback type or #SXM_CALLBACK_CONN_READY
 *                 that the connection is ready and the service start set up
 *                 monitors and filters.
 * \param[in] param Set to the DSI for the given Service.
 * \param[in] arg data service instance
 *
 ******************************************************************************/
static void sxm_callback(int type, int param, void *arg)
{ 
    SXMSDKService *currService = (SXMSDKService *)arg;
    uint i;

    if ((type != SXM_CALLBACK_SUBS) && (type != SXM_CALLBACK_CONN_READY))
    {
        /* This function has nothing to process anything except 
         * subscription status.
         */
        return;
    }

    /* The callback is called under the lock and can be modified under
     * lock only, thus, while this function is being executed the service
     * instance remains valid and there is no need to find the service
     * among those who are running now
     */
    for (i = 0; i < ARRAY_SIZE(currService->ds->dsis); ++i)
    {
        if (currService->ds->dsis[i].dsi == param)
        {
            uint msgType;
            /* pack the index of DSI into the parameter in addition
             * to the dsi which is already inside the param. Since
             * dsi can fit into 16 bit the upper 16 bits will be used
             * for dsi index within the service's structure
             */
            param = (int)LLI_MAKE_PARAM(param, i);
            msgType =
                (type == SXM_CALLBACK_SUBS) ? LLI_SUBS_UPDATE_MSG_TYPE :
                                              LLI_CONN_READY_MSG_TYPE;
            if (SXM_E_OK != sxm_queue_put(&currService->mq, &param,
                                          msgType, sizeof(param), MSG_PRIO_HIGH))
            {
                PLOG_LLI("sxm_queue_put() failed, message %u", msgType);
            }
            else {
                DEBUG_LLI("DSI %d found, posting message %d", LLI_UNPACK_DSI(param), msgType);
            }
            break;
        }
    }

    return;
}

/************************************************************************
 *                                                                      *
 *            The Data Service Monitor (I/O) Thread                     *
 *            =====================================                     *
 *                                                                      *
 ************************************************************************/

/***************************************************************************//**
 * The function checks if timer is set and invokes specified timer callback
 * if timeout exceeds.
 *
 * \param[in] currService Pointer to the Service Structure for the Service
 *                        calling this function.
 *
 ******************************************************************************/
static void sxm_lli_timer_handler(SXMSDKService *currService, uint now)
{
    if (currService->ds->timer != NULL)
    {
        if (now == 0U)
        {
            now = sxm_sxe_time_now();
        }

        /* Don't call the first time, when timer_saved is still 0 */
        if (0 != currService->timer_saved)
        {
            if ((now - currService->timer_saved) >= currService->ds->timer_mins)
            {
                currService->ds->timer(currService->ds->dsuser);

                currService->timer_saved = now;
            }
        }
        /* Don't call the first time, if time is not yet configured */
        else if (0U != now)
        {
            /* Call immediately if timeout is 0 */
            if (0U == currService->ds->timer_mins)
            {
                currService->ds->timer(currService->ds->dsuser);
            }

            currService->timer_saved = now;
        }
    }
    return;
}

/***************************************************************************//**
 * The function updates a service DSI structure on subscription update event
 *
 * \param[in] currService Pointer to the Service Structure for the Service
 *                        calling this function.
 * \param[in] dsi         Pointer to DSI structure to update
 *
 ******************************************************************************/
static void sxm_lli_update_dsi_status(SXMSDKService *curservice, SXMDataDsi *dsi)
{
    uint i, subs_mask;

    sxm_sxi_extract_dmis(dsi->dsi, &dsi->dmi);

    /* add any pending filter requests in to the filter mask */
    for (i = 0; i < dsi->pending.dmic; ++i)
    {
        uint k;
        for (k = 0; k < dsi->dmi.dmic; ++k)
        {
            if (dsi->dmi.dmis[k] == dsi->pending.dmis[i])
            {
                BITS(dsi->filter, k);
            }
        }
    }

    dsi->pending.dmic = 0;

    /* convert the subscription state to SXM_SUBS values */
    switch (dsi->dmi.status)
    {
        case SXI_FULLY_SUBSCRIBED:
            dsi->substate = SXM_SUBS_FULL;
            break;
        case SXI_PARTIALLY_SUBSCRIBED:
            dsi->substate = SXM_SUBS_PARTIAL;
            break;
        case SXI_UNSUBSCRIBED:
            dsi->substate = SXM_SUBS_NONE;
            break;
        default:
            dsi->substate = SXM_SUBS_UNKNOWN;
            break;
    }

    /* Check all states for all available DSIs */
    for (subs_mask = 0, i = 0; i < ARRAY_SIZE(curservice->ds->dsis); ++i)
    {
        if (curservice->ds->dsis[i].dsi != 0U)
        {
            BITSET(subs_mask, curservice->ds->dsis[i].substate);
        }
    }

    /* Check the power of 2 since only in this case all states are the same */
    if (!(subs_mask & (subs_mask - 1)))
    {
        /* All have the same state */
        curservice->ds->status->substate = dsi->substate;
    }
    else if (subs_mask & (BIT(SXM_SUBS_FULL) | BIT(SXM_SUBS_PARTIAL)))
    {
        /* There is something enabled thus it can be considered as
           partially subscribed. */
        curservice->ds->status->substate = SXM_SUBS_PARTIAL;
    }
    else
    {
        /* Consider not-subscribed if there is a combination
           of NONE and UNKNOWN states. */
        curservice->ds->status->substate = SXM_SUBS_NONE;
    }
    return;
}

/***************************************************************************//**
 * This function starts and monitors incoming data for a selected Service.
 * 
 * \note This function should continue to run until the Service is stopped.
 * \param[in] dsp Pointer to the structure for the given Data Service.
 * \return message type which caused thread stop
 *
 ******************************************************************************/
static void* sxm_monitor(void *dsp)
{
    SXMSDKService *currService = (SXMSDKService *)dsp;
    SXMDataService *ds = currService->ds;
    int rc;
    uint i;
    SXMQueueMsg msg = { 0U, 0U, NULL };

    /* Main thread's messages processing loop */
    do
    {
        rc = sxm_queue_get(&currService->mq, &msg, SXM_QUEUE_INFINITE_TIMEOUT);
        if (rc != SXM_E_OK)
        {
            if (rc != SXM_E_TIMEOUT)
            {
                /* exit immediately */
                PLOG_LLI("sxm_queue_get() failed (rc=%d), terminating", rc);
                break;
            }
            /* false signal, let's continue */
            DEBUG_LLI("False signal received from %s queue", currService->mq.id);
        }
        else {  
            /* process even type */
            switch (msg.msgType)
            {
                case LLI_STOP_MSG_TYPE:
                case LLI_LINK_LOSS_MSG_TYPE:
                {
                    DEBUG_LLI("Stopping by %u msg from %s queue.",
                              msg.msgType, currService->mq.id);
                }
                break;
                case LLI_CONN_READY_MSG_TYPE:
                {
                    const int param = *((int *)(sxm_queue_msg_data(&msg)));
                    const ushort dsi = (ushort)(LLI_UNPACK_DSI(param));
                    const int dsi_idx = LLI_UNPACK_DSI_IDX(param);

                    DEBUG_LLI("DSI %u permit to enable monitor received from %s queue, dsi_idx = %d",
                              dsi, currService->mq.id, dsi_idx);
                    if (dsi_idx >= (int)ARRAY_SIZE(ds->dsis)) {
                        PLOG_LLI("Invalid DSI index received %d", dsi_idx);
                        break;
                    }

                    if (currService->radio == 0)
                    {
                        if ((SXM_E_OK != sxm_link_subscribe(LNK_MSG_PAYLOAD_TYPE,
                                                     &currService->radio,
                                                     &sxm_lli_link_callback,
                                                     currService))
                            ||
                            (SXM_E_OK != sxm_link_subscribe(SXI_DATA_MSG_PAYLOAD_TYPE,
                                                     &currService->radio,
                                                     &sxm_lli_link_callback,
                                                     currService)))
                        {
                            if (currService->radio != 0) {
                                sxm_link_unsubscribe_all(currService->radio);
                                currService->radio = 0;
                            }
                            PLOG_LLI("Subscribing to LINK has failed");
                            break;
                        }
                    }

                    /* The default SXi state is to deliver all DMIs.
                       If the filter is set to zero, then disable this at the outset
                       This doesn't always work, but we'll try again when we get the DMI list */
                    if (ds->dsis[dsi_idx].dsi == dsi)
                    {
                        sxm_sxi_monitor_dsi(SXI_DATA_START_MONITOR, dsi);
                        for (i = 0U; i < ARRAY_SIZE(ds->dsis[dsi_idx].filter); ++i)
                        {
                            if (ds->dsis[dsi_idx].filter[0] != 0U)
                            {
                                break;
                            }
                        }
                        if (i == ARRAY_SIZE(ds->dsis[dsi_idx].filter)) 
                        {
                            sxm_sxi_filter_dsi(SXI_FILTER_DMI_START, dsi, 0, NULL);
                        }
                    }
                    else
                    {
                        PLOG_LLI("Service integrity failure, dsi %d is not at %d",
                                 (int)dsi, dsi_idx);
                    }
                }
                break;
                case LLI_SUBS_UPDATE_MSG_TYPE:
                {
                    const int param = *((int *)(sxm_queue_msg_data(&msg)));
                    const ushort dsi = (ushort)(LLI_UNPACK_DSI(param));
                    const int dsi_idx = LLI_UNPACK_DSI_IDX(param);

                    DEBUG_LLI("DSI %u subscription update received from %s queue, dsi_idx = %d",
                            dsi, currService->mq.id, dsi_idx);
                    if (dsi_idx >= (int)ARRAY_SIZE(ds->dsis)) {
                        PLOG_LLI("Invalid DSI index received %d", dsi_idx);
                        break;
                    }

                    if (ds->dsis[dsi_idx].dsi == dsi)
                    {
                        sxm_lli_update_dsi_status(currService, &ds->dsis[dsi_idx]);
                        rebuild_filter(&ds->dsis[dsi_idx]);
                        currService->ds->report(SXM_CALLBACK_SUBS,
                                                currService->ds->status->substate,
                                                currService->ds->dsuser);
                    }
                    else
                    {
                        PLOG_LLI("Service integrity failure, dsi %d is not at %d",
                            (int)dsi, dsi_idx);
                    }
                }
                break;
                case LLI_TIME_MSG_TYPE:
                {
                    sxm_lli_timer_handler(currService,
                                            *((uint *)sxm_queue_msg_data(&msg)));
                }
                break;
                case LLI_DATAPACKET_MSG_TYPE:
                {
                    LLIDataPacket *packet = (LLIDataPacket *)sxm_queue_msg_data(&msg);
                    byte *payload = (byte*)(packet + 1);
                    if (packet == NULL)
                    {
                        PLOG_LLI("Unexpected message content");
                        break;
                    }
                    if (ds->maxau == 0)
                    {
                        DEBUG_LLI("DMI=%3u %4d-byte au ", ds->dsis->dmi.dmis[packet->dmi_idx], packet->size - SXM_LLI_DATA_OFFSET);
                        ds->complete(payload + SXM_LLI_DATA_OFFSET, (int)(packet->size - SXM_LLI_DATA_OFFSET), ds->dsuser);
                    }
                    else
                    {
                        process_sdtp_packet(currService, packet, payload);
                    }
                }
                break;
                case LLI_RFD_UPDATED_MSG_TYPE:
                {
                    DEBUG_LLI("RFD updated msg received from %s queue",
                              currService->mq.id);
                    /* Do the DB update notification to the service */
                    ds->report(SXM_CALLBACK_DB_UPDATED, SXM_E_OK, ds->dsuser);
                }
                break;
                default:
                {
                    DEBUG_LLI("Custom message %u in %s queue",
                              msg.msgType, currService->mq.id);
                    if (ds->queue_callback != NULL)
                    {
                        ds->queue_callback(msg.msgType, msg.msgSize, sxm_queue_msg_data(&msg));
                    }
                }
                break;
            }
            /* Release message's data if any */
            (void) sxm_queue_msg_release(&msg);
        }
    } while ((LLI_STOP_MSG_TYPE != msg.msgType) && (LLI_LINK_LOSS_MSG_TYPE != msg.msgType));

    /* Unregister the LINK if it has been registered */
    if (currService->radio != 0U) {
        rc = sxm_link_unsubscribe_all(currService->radio);
        if (rc != SXM_E_OK) {
            ERROR_API("Unable to un-subscribe from LINK, radio=%u, rc=%d",
                currService->radio, rc);
        }
        currService->radio = 0U;
    }
 
    for (i = 0; i < ARRAY_SIZE(ds->dsis); i++)
    {
        if (ds->dsis[i].dsi != 0)
        {
            /* Remove the callback regardless of the reason for the
             * service stop
             */
            rc = sxm_sxi_set_dsi_callback(ds->dsis[i].dsi, NULL, NULL);
            if (rc != SXM_E_OK)
            {
                ERROR_API("Failed to un-reg DSI %d callback, rc=%d",
                    (int)ds->dsis[i].dsi, rc);
            }
            if (LLI_LINK_LOSS_MSG_TYPE != msg.msgType)
            {
                /* There is no need in this command if the connection lost
                 * since the chance to get unexpected situation is high while
                 * the SXi level is being stopped as well.
                 */
                rc = sxm_sxi_monitor_dsi(SXI_DATA_HALT_SERVICE_MONITOR, ds->dsis[i].dsi);
                if (rc != SXM_E_OK)
                {
                    ERROR_API("Failed to halt data service on DSI %u, rc=%d",
                        (uint)ds->dsis[i].dsi, rc);
                }
            }
        }
    }

    if (LLI_STOP_MSG_TYPE != msg.msgType)
    {
        ds->report(SXM_CALLBACK_ERROR, SXM_E_PIPE, ds->dsuser);
        if (LLI_LINK_LOSS_MSG_TYPE != msg.msgType)
        {
            ERROR_LLI("monitor. Exiting with rc=%d", rc);
        }
        else
        {
            ERROR_LLI("monitor. Exiting due to link loss");
        }
    }
    return (void*)((size_t)(msg.msgType));
}

/***************************************************************************//**
 * This function opens a device handle to receive data messages and
 * starts a thread to handle those messages as they are received.
 *
 * \param[in] ds Pointer to the Service Structure for the Service to start.
 * \param[out] handle Caller-provided pointer internally set to Service Structure.
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_NOMEM Failed to allocate memory
 * \retval SXM_E_PIPE Failed open corresponded device
 * \retval SXM_E_THREAD Failed to create monitor thread
 * \retval SXM_E_RESOURCE Failed to initialize internal resources
 *
 ******************************************************************************/
int sxm_lli_start(SXMDataService *ds, ptr *handle)
{
    SXMSDKService *currService =
        (SXMSDKService *)sxe_calloc(1, sizeof(SXMSDKService));
    uint i;
    char threadName[SXM_SERVICE_NAME_LEN_MAX] = LLI_THREAD_NAME_PREFIX;

    if (currService == NULL)
    {
        return SXM_E_NOMEM;
    }

    /* if service uses SDTP, need to create array for assemblies */
    if (ds->maxau > 0)
    {
        ushort count = 0;
        int rc;

        for (i = 0; i < ARRAY_SIZE(ds->dsis); ++i)
        {
            if (ds->dsis[i].dsi != 0)
            {
                ++count;
            }
        }

        /* calculating total number of DMIs for this service */
        count *= SXI_SP_MAX_DMI_CNT;

        currService->assemblies = (SXMSdtp**)sxe_calloc(count,
            sizeof(SXMSdtp*));
        if (currService->assemblies == NULL)
        {
            sxe_free(currService);
            return SXM_E_NOMEM;
        }

        currService->asize = count;

        rc = sxm_list_create(&currService->sar_pool, SXM_LIST_PREALLOCATED);
        if (rc != SXM_E_OK)
        {
            sxe_free(currService->assemblies);
            sxe_free(currService);
            return SXM_E_RESOURCE;
        }
    }

    if (SXM_E_OK != sxm_mutex_init(&currService->qmutex))
    {
        sxe_free(currService->assemblies);
        sxe_free(currService);
        return SXM_E_RESOURCE;
    }

    if (SXM_E_OK != sxm_mutex_init(&currService->amutex))
    {
        sxe_free(currService->assemblies);
        sxm_mutex_destroy(&currService->qmutex);
        sxe_free(currService);
        return SXM_E_RESOURCE;
    }

    currService->ds = ds;

    if (SXM_E_OK != sxm_queue_create(&currService->mq,
                                     LLI_MSG_QUEUE_SIZE,
                                     &currService->qmutex,
                                     SXM_QUEUE_RESERVE_ADAPTIVE,
                                     ds->service, &(DEBUG_VAR)))
    {
        sxe_free(currService->assemblies);
        sxm_mutex_destroy(&currService->qmutex);
        sxm_mutex_destroy(&currService->amutex);
        sxe_free(currService);
        return SXM_E_RESOURCE;
    }

    if (ds->iom.pFunc != NULL)
    {
        ds->iom.pArg = ds->dsuser;
        ds->iom.psTag = &ds->service[0];
        if (sxm_iomanager_reg(&ds->iom) != SXM_E_OK)
        {
            currService->radio = 0;
            sxm_queue_destroy(&currService->mq);
            sxe_free(currService->assemblies);
            sxm_mutex_destroy(&currService->qmutex);
            sxm_mutex_destroy(&currService->amutex);
            PLOG_LLI("Creation of %s%s thread failed",
                LLI_THREAD_NAME_PREFIX, currService->ds->service);
            sxe_free(currService);
            return SXM_E_RESOURCE;
        }
    }

    if (sxm_pthread_create(&currService->pid, NULL, &sxm_monitor, currService) != 0)
    {
        sxm_iomanager_unreg(&ds->iom);
        currService->radio = 0;
        sxm_queue_destroy(&currService->mq);
        sxe_free(currService->assemblies);
        sxm_mutex_destroy(&currService->qmutex);
        sxm_mutex_destroy(&currService->amutex);
        PLOG_LLI("Creation of %s%s thread failed",
	        LLI_THREAD_NAME_PREFIX, currService->ds->service);
        sxe_free(currService);

        return SXM_E_THREAD;
    }

    /* put new service into the least */
    LOCK(mutex);
    currService->link = root;
    root = currService;
    UNLOCK(mutex);
    /* populate the handle */
    *handle = currService;

    /* make sure the name is better distinguishable from other threads */
    strncpy(&threadName[sizeof(LLI_THREAD_NAME_PREFIX) - 1], currService->ds->service,
            (sizeof(threadName) - sizeof(LLI_THREAD_NAME_PREFIX)));
    sxe_thread_setname(currService->pid, &threadName[0]);

    for (i = 0; i < ARRAY_SIZE(ds->dsis); i++)
    {
        if (ds->dsis[i].dsi != 0)
        {
            sxm_sxi_set_dsi_callback(ds->dsis[i].dsi, &sxm_callback, currService);
        }
    }

    /* Transition service status to ok.
       SMS based services manage multiple Generic Data Service objects
       which have asynchronous startup in their lli.
       We want to keep status transition in the lli and not the module to keep
       target environment differences out of the core module implementation. */
    ds->status->service = SXM_SERVICE_OK;
    return SXM_E_OK;
}

/***************************************************************************//**
 * This function stops a selected Data Service.
 *
 * The Service Monitor thread is stopped, a Halt Service message is sent to
 * the Server, memory is cleaned up, and the Service status is set to reflect
 * that it is no longer running.
 *
 * \param[in] handle Pointer to the Low Level Interface handle for the selected Service.
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_BUSY The function is called from corresponded monitor thread
 * \retval SXM_E_STATE The handle belongs to the service which is stopped already
 *
 ******************************************************************************/
int sxm_lli_stop(ptr handle)
{
    SXMSDKService *currService = (SXMSDKService *)handle;
    void* threadExit = NULL;
    uint i;
    int found = 1;

    if (!handle)
    {
        return SXM_E_FAULT;
    }

    /* prevent the thread from stopping and killing itself */
    if (pthread_equal(currService->pid, pthread_self()))
    {
        return SXM_E_BUSY;
    }

    LOCK(mutex);
    if (root == currService)
    {
        root = root->link;
    }
    else
    {
        SXMSDKService *prev = root;
        while (prev->link && prev->link != currService)
        {
            prev = prev->link;
        }

        if (prev->link == currService)
        {
            prev->link = currService->link;
        }
        else
        {
            found = 0;
        }
    }
    UNLOCK(mutex);

    if (!found)
    {
        return SXM_E_STATE;
    }

    /* terminate (cleanly) the monitor thread */
    sxm_queue_put(&currService->mq, NULL, LLI_STOP_MSG_TYPE, 0, MSG_PRIO_HIGH);

    /* Be sure monitor thread is not reading from device while closing. */
    pthread_join(currService->pid, &threadExit);
    DEBUG_LLI("Thread exited by %d", (int)((size_t)threadExit));

    sxm_lli_rfd_stop(currService);

    sxm_iomanager_unreg(&currService->ds->iom);
    sxm_queue_destroy(&currService->mq);
    sxm_mutex_destroy(&currService->qmutex);
    sxm_mutex_destroy(&currService->amutex);

    /* clean up the assembly buffers */
    clean_pool(&currService->sar_pool);
    sxm_list_destroy(&currService->sar_pool);
    for (i = 0; i < currService->asize; i++)
    {
        sxm_sdtp_delete(currService->assemblies[i]);
    }

    sxe_free(currService->assemblies);
    sxe_free(currService);
    return SXM_E_OK;
}

/***************************************************************************//**
 * This function removes a given DMI from the calling Service.
 *
 * First, the desired DMI is found in the list, the filter is cleared
 * and the filter list then rebuilt, then the DMI is removed from
 * the Service.
 *
 * \param[in] handle Pointer to the calling Service Data Structure
 * \param[in] dsi The DSI that the DMI should be added to.
 * \param[in] dmi The DMI to be removed from the Service.
 *
 * \retval SXM_E_OK Slave started successfully.
 *
 ******************************************************************************/
int sxm_lli_remove(ptr handle, int dsi, int dmi)
{
    SXMSDKService *currService = (SXMSDKService *)handle;
    uint i, j;

    /* if we are not yet subscribed, ignore the request */

    for (i = 0; i < ARRAY_SIZE(currService->ds->dsis); i++)
    {
        if (currService->ds->dsis[i].dsi == dsi)
        {
            if (currService->ds->dsis[i].dmi.dmic == 0)
            {
                return SXM_E_OK;
            }
            else
            {
                for (j = 0; j < currService->ds->dsis[i].dmi.dmic; j++)
                {
                    if (currService->ds->dsis[i].dmi.dmis[j] == dmi)
                    {
                        if (BITP(currService->ds->dsis[i].filter, j) != 0)
                        {
                            BITC(currService->ds->dsis[i].filter, j);
                            rebuild_filter(&currService->ds->dsis[i]);

                            if (currService->ds->maxau > 0)
                            {
                                remove_assembly(currService,
                                    LLI_ASSEMBLY_INDEX(i, j));
                            }
                        }

                        return SXM_E_OK;
                    }
                }
            }
        }
    }
    return SXM_E_OK;
}

/***************************************************************************//**
 * This function adds a given DMI to the calling Service.
 *
 * First, the desired DSI is found in the list. Then if the DMI
 * has not already been set in the filter, it is set here. Then
 * the DMI is added to the list for the DSI and the filter is
 * updated.
 *
 * \param[in] handle Pointer to the calling Service Data Structure.
 * \param[in] dsi The DSI that the DMI should be added to.
 * \param[in] dmi The DMI to be removed from the Service.
 *
 * \retval SXM_E_OK DMI added successfully.
 * \retval SXM_E_FAULT NULL parameter.
 * \retval SXM_E_NOENT No entry was found for the value passed in dmi.
 *
 ******************************************************************************/
int sxm_lli_request(ptr handle, int dsi, int dmi)
{
    SXMSDKService *currService = (SXMSDKService *)handle;
    uint i, j;

    if (!handle)
    {
        return SXM_E_FAULT;
    }

    DEBUG_LLI("(handle: %p, dsi: %d, dmi: %d)", handle, dsi, dmi);

    for (i = 0; i < ARRAY_SIZE(currService->ds->dsis); i++)
    {
        if (currService->ds->dsis[i].dsi == dsi)
        {
            if (currService->ds->dsis[i].dmi.dmic == 0)
            {
                currService->ds->dsis[i].pending.dmis[
                    currService->ds->dsis[i].pending.dmic++] = (ushort)dmi;
                return SXM_E_OK;
            }
            else
            {
                for (j = 0; j < currService->ds->dsis[i].dmi.dmic; j++)
                {
                    if (currService->ds->dsis[i].dmi.dmis[j] == (ushort)dmi)
                    {
                        if (BITP(currService->ds->dsis[i].filter, j) == 0)
                        {
                            BITS(currService->ds->dsis[i].filter, j);
                            rebuild_filter(&currService->ds->dsis[i]);
                        }
                        return SXM_E_OK;
                    }
                }
            }
        }
    }
    return SXM_E_NOENT;
}

/***************************************************************************//**
 * This function modifies DMI list for a given service.
 *
 * First, the desired DSI is found in the list. Then if the DMI
 * has not already been set in the filter, it is set here. If DMI is set 
 * already, no effect for given DMI.
 * If DMI has been set before and not present in provided list, it will be 
 * disabled.
 * 
 * Any DMIs not related to service will be silently ignored.
 *
 * \param[in] handle Pointer to the calling Service Data Structure.
 * \param[in] dsi The DSI which DMI list should be changed
 * \param[in] dmic The number of DMIs in the list
 * \param[in] dmis The list of DMIs to be enabled
 *
 * \retval SXM_E_OK DMI list updated successfully.
 * \retval SXM_E_FAULT NULL parameter.
 * \retval SXM_E_INVAL Parameter is wrong or out of range
 * \retval SXM_E_NOENT No entry was found for the value passed in handle.
 *
 ******************************************************************************/
int sxm_lli_update(ptr handle, int dsi, ushort dmic, ushort *dmis)
{
    SXMSDKService *currService = (SXMSDKService *)handle;
    uint i, idx = 0;
    SXMDataDsi *dsidata = NULL;

    if ((handle == NULL) || ((dmic != 0U) && (dmis == NULL))) {
        return SXM_E_FAULT;
    }

    if ((currService->ds->maxau != 0) && (dmic > SXI_SP_MAX_DMI_CNT)) {
        return SXM_E_INVAL;
    }

    DEBUG_LLI("(handle: %p, dsi: %d, dmic: %u, dmis: %p)",
              handle, dsi, dmic, dmis);

    for (i = 0; i < ARRAY_SIZE(currService->ds->dsis); i++) {
        if (currService->ds->dsis[i].dsi == dsi)
        {
            dsidata = &currService->ds->dsis[i];
            idx = i;
            break;
        }
    }

    if (dsidata != NULL)
    {
        BOOL changed = FALSE;
        for (i = 0; i < dsidata->dmi.dmic; i++) {
            BOOL found = FALSE;
            uint j;
            for (j = 0; j < dmic; j++) {
                if (dsidata->dmi.dmis[i] == dmis[j]) {
                    found = TRUE;
                    if (BITP(dsidata->filter, i) == 0) {
                        /* dmi is disabled for now, need to enable */
                        BITS(dsidata->filter, i);
                        changed = TRUE;
                    }
                    break;
                }
            }

            if ((found == FALSE) && (BITP(dsidata->filter, i) != 0)) {
                /* dmi is enabled, need to disable */
                BITC(dsidata->filter, i);
                if (currService->ds->maxau > 0) {
                    remove_assembly(currService, LLI_ASSEMBLY_INDEX(idx, i));
                }
                changed = TRUE;
            }
        }

        if (changed == TRUE) {
            rebuild_filter(dsidata);
        }

        return SXM_E_OK;
    }

    return SXM_E_NOENT;
}

/***************************************************************************//**
 * This function compares the thread ID passed in to the thread ID 
 * for the calling thread.
 *
 * \param[in] handle Pointer to a Data Service structure.
 *
 * \retval 0 IDs do not match.
 * \retval Non-Zero IDs match.
 *
 ******************************************************************************/
int sxm_lli_on_thread(ptr handle)
{
    SXMSDKService *currService = (SXMSDKService *)handle;

    if (pthread_equal(currService->pid, pthread_self()))
    {
        return 1;
    }

    return 0;
}

/**
 * \name RFD Interface
 * @{
 */
/***************************************************************************//**
 * This function starts the RFD thread as needed.
 *
 * This is called when starting a Service that uses RFD.
 *
 * \param[in] handle Pointer to a Data Service structure.
 * \retval SXM_E_OK     on success
 * \retval SXM_E_NOENT  if RFD failed to start (e.g. RFD slot file issue)
 * \retval SXM_E_THREAD if RFD thread couldn't be started
 *
 ******************************************************************************/
int sxm_lli_rfd_init(ptr handle)
{
    int rc;
    SXMSDKService *currService = (SXMSDKService *)handle;

    rc = sxm_lli_core_rfd_init(&currService->rfd, currService->ds);
    return rc;
}

/***************************************************************************//**
 * Opens RFD segment for reading providing file handle
 *
 * \param[in] handle Pointer to a Data Service structure.
 * \param[in] meta Pointer to RFD file meta data.
 * \param[in] frag RFD file fragment.
 *
 * \return file handle to the RFD file or \c NULL in case of error
 *
 ******************************************************************************/
FILE *sxm_lli_rfd_open(ptr handle, SXMRfdMetadata *meta, int frag)
{
    SXMSDKService *currService = (SXMSDKService *)handle;

    return sxm_lli_core_rfd_open(&currService->rfd, meta, frag);
}

/***************************************************************************//**
 * This function starts collection of RFD data for a given Service.
 * 
 * \note In case of error this function internally fires up a notification
 *       through service callback about RFD issues.
 *
 * \param[in] handle Pointer to a Data Service structure.
 * \param[in] meta Pointer to RFD file meta data.
 *
 * \retval SXM_E_OK RFD started successfully.
 * \retval SXM_E_NOENT Cannot initialize RFD slot
 * \retval SXM_E_NOMEM Not enough memory to start RFD collection
 *
 ******************************************************************************/
int sxm_lli_rfd_start(ptr handle, SXMRfdMetadata* meta)
{
    SXMSDKService *currService = (SXMSDKService *)handle;
    return sxm_lli_core_rfd_start(&currService->rfd, currService->ds, meta);
}

/***************************************************************************//**
 * This function adds data to an RFD.
 *
 * \note In case of error this function internally fires up a notification
 *       through service callback about RFD issues.
 *
 * \param[in] handle Pointer to a Data Service structure.
 * \param[in] packet Pointer to the data packet to add to the RFD.
 * \param[in] pl Starting place in the packet for the data to add.
 *
 * \retval SXM_E_OK - Slave started successfully.
 *
 ******************************************************************************/
int sxm_lli_rfd_add(ptr handle, byte *packet, int pl)
{
    SXMSDKService *currService = (SXMSDKService *)handle;
    return sxm_lli_core_rfd_add(&currService->rfd, packet, pl);
}

/** @} */

/***************************************************************************//**
 * A timer that sets the expiration time (in minutes) and the function
 * to call when the timer expires.
 *
 * \param[in] handle Pointer to a Data Service structure that will use the timer.
 * \param[in] callback Pointer to the function to call when the timer expires.
 * \param[in] mins Timer duration in minutes.
 *
 * \retval SXM_E_OK Timer set
 * \retval SXM_E_RESOURCE Failed to place LLI_TIME_MSG_TYPE message to the queue
 *
 ******************************************************************************/
int sxm_lli_timer(ptr handle, void (*callback)(void *), uint mins)
{
    SXMSDKService *currService = (SXMSDKService *)handle;
    int rc;
    const uint now = 0U; /* special variable for the message's content */

    currService->ds->timer_mins = mins;
    currService->ds->timer = callback;
    currService->timer_saved = 0; /* Make sure the timer restarted */

    /* Notify the thread to setup the timer */
    rc = sxm_queue_put(&currService->mq, &now,
                       LLI_TIME_MSG_TYPE, sizeof(now), MSG_PRIO_NORMAL);
    if (SXM_E_OK != rc)
    {
        PLOG_LLI("sxm_queue_put() failed (rc=%d)", rc);
    }

    return rc;
}

/***************************************************************************//**
 * The function allows to send out event via LLI event queue.
 * In order to catch this event the service has to implement and provide 
 * callback \ref SXMDataService::queue_callback for it's own specific events. 
 *
 * \param[in] handle Pointer to a Data Service structure that
 *                   owns the desirable message queue.
 * \param[in] pData pointer to message data being placed on the queue
 * \param[in] msgType type of message identifier
 * \param[in] msgSize size of message
 *
 * \retval SXM_E_OK Event has been sent
 * \retval SXM_E_RESOURCE Failed to place the message to the queue
 *
 ******************************************************************************/
int sxm_lli_send_event(ptr handle, const void *pData, uint msgType, uint msgSize)
{
    int rc;
    SXMSDKService *currService = (SXMSDKService *)handle;
    /* Put the message to the queue */
    rc = sxm_queue_put(&currService->mq, pData, msgType, msgSize, MSG_PRIO_NORMAL);
    if (SXM_E_OK != rc)
    {
        PLOG_LLI("sxm_queue_put() failed, message %d (rc=%d)",
            msgType, rc);
    }
    return rc;
}

/***************************************************************************//**
 * This function returns status of Data Service RFD slot
 *
 * \param[in] handle Pointer to a Data Service structure.
 * \param[in] id Data Service slot number [0 to SXM_RFD_MAX_COLLECTION-1]
 * \param[out] meta RFD update metadata structure assigned to the slot.
 *
 * \retval SXM_E_OK Data Service slot is busy.
 * \retval SXM_E_NOENT Data Service slot is not used.
 *
 ******************************************************************************/
int sxm_lli_rfd_status(ptr handle, int id, SXMRfdMetadata *meta) {
    SXMSDKService *self = (SXMSDKService *)handle;

    return sxm_lli_core_rfd_status(&self->rfd, id, meta);
}

/***************************************************************************//**
 * This function returns the permil complete (percent complete times 10) of
 * and RFD fragment collection.
 *
 * Indicates how many of the needed blocks have been received for an RFD fragment.
 * Keep in mind that there are error cases which allow the permil complete to
 * go backwards. It can even get up to 1000 and then drop to -1 or 0.
 *
 * A -1 value means that the permil complete is unknown. Data blocks may 
 * or may not have been received (e.g. on a previous power cycle) but we don't
 * know yet how many.
 *
 * \param[in]  handle         Pointer to a SXMSDKService structure.
 * \param[in]  id             Data Service's RFD slot index
 * \param[out] permilComplete RFD update metadata structure assigned to the slot.
 *
 * \retval SXM_E_OK     Success, \p permilComplete now contains a value from -1 to 1000
 * \retval SXM_E_INVAL  If \p slotIndex is not valid
 * \retval SXM_E_FAULT  If \p permilComplete is NULL
 * \retval SXM_E_NOENT  If slot is not in use (empty slot)
 *
 ******************************************************************************/
int sxm_lli_rfd_permil_complete(ptr handle, int id, int *permilComplete) {
    SXMSDKService *self = (SXMSDKService *)handle;
    return sxm_lli_core_rfd_permil_complete(&(self->rfd), (uint)id, permilComplete);
}

/***************************************************************************//**
 * This function changes Data Service RFD slot status from busy to not used
 *
 * \param[in] handle Pointer to a Data Service structure.
 * \param[in] id Data Service slot number
 *
 * \retval SXM_E_OK Data Service slot status successfully changed.
 *
 ******************************************************************************/
int sxm_lli_rfd_remove(ptr handle, int id) {
    SXMSDKService *self = (SXMSDKService *)handle;

    return sxm_lli_core_rfd_remove(&self->rfd, id);
}

/***************************************************************************//**
 * This function deallocates and deinitializes resources used by RFD service
 *
 * \param[in] handle Pointer to a Data Service structure.
 *
 ******************************************************************************/
void sxm_lli_rfd_stop(ptr handle) {
    SXMSDKService *self = (SXMSDKService *)handle;

    sxm_lli_core_rfd_stop(&self->rfd);
}

/***************************************************************************//**
 * This function posts DB update notification event
 *
 * \param[in] handle Pointer to a Data Service structure.
 *
 ******************************************************************************/
void sxm_lli_rfd_set_db_updated(ptr handle) {
    SXMSDKService *currService = (SXMSDKService *)handle;
    int rc = sxm_queue_put(&currService->mq, NULL,
                           LLI_RFD_UPDATED_MSG_TYPE,
                           0, MSG_PRIO_HIGH);
    if (SXM_E_OK != rc)
    {
        PLOG_LLI("sxm_queue_put() failed (rc=%d)", rc);
    }
}
#endif /* #if (!SXM_USE_GEN8) */
#endif /* #if !defined(SXM_USE_SMS) */
