/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 * \file sxm_list.c
 * \author Alexander Pylchagin
 * \date 5/19/2014
 * \brief Implementation of Double Linked List implementation.
 *
 ******************************************************************************/

#include <util/sxm_list.h>
#include <util/sxm_memory_internal.h>

#ifndef SDKFILES_STANDALONE_BUILD
#include "sxm_common.h"
#else
#include "sdkfiles.h"
#endif

/**
 * Defines the max level of the merge sorting stack, in other words,
 * this is a power of to which gives nearest value to fit max allowed
 * entries within the list
 */
#define SXM_LIST_LIST_SORT_LEVEL_MAX (16) /* 2^16 items max in list */

/** Helper macro to get real memory address based on user's data
 * \param[in] _p user's input data
 */
#define SXM_LIST_ENTRY_FROM_USER(_p) \
    ((SXMListEntry*)(_p) - 1)

/** Helper macro to provide use the memory it can use based on
 * internal entry's address
 * \param[in] _p real entry address
 */
#define SXM_LIST_ENTRY_TO_USER(_p) \
    ((SXMListEntry*)(_p) + 1)

/** Merge intermediate structure */
typedef struct
{
    SXMListEntry *entry; //!< Address to the sorting list
    int level; //!< Level of the current list of entries
} SXMListMergeSortStackItem;

/**************************************************************************//**
 * Creates the linked list
 *
 * The linked list behavior can be tuned using different flags:
 *  - \ref SXM_LIST_NONE - default linked list;
 *  - \ref SXM_LIST_PREALLOCATED - the owner is aware about list entry memory
 *                                 allocation and release using dedicated API
 *                                 like \ref sxm_list_allocate() and
 *                                 \ref sxm_list_free()
 *
 * \param[in] list address of the memory where the list will be initialized
 * \param[in] flags list behavioral flags
 *
 * \retval SXM_E_OK in case of success,
 * \retval SXM_E_FAULT if the passed \c NULL argument
 *
 *****************************************************************************/
int sxm_list_create(SXMList *list, uint flags)
{
    SXMResultCode rc;
    if (list != NULL)
    {
        memset(list, 0, sizeof(*list));
        list->flags = flags;
        rc = SXM_E_OK;
    }
    else
    {
        rc = SXM_E_FAULT;
    }
    return rc;
}

/**************************************************************************//**
 * Destroys exiting list and all corresponded entries
 *
 * \param[in] list valid SXMList instance
 *
 *****************************************************************************/
void sxm_list_destroy(SXMList *list)
{
    if (list != NULL)
    {
        (void) sxm_list_remove_all(list);
        memset(list, 0, sizeof(*list));
    }
    return;
}

/**************************************************************************//**
 * Destroys exiting list and all corresponded entries, but list
 * remains valid. 
 * 
 * \param[in] list pointer to the list structure
 *
 * \retval SXM_E_OK all good,
 * \retval SXM_E_FAULT NULL parameter
 *
 *****************************************************************************/
int sxm_list_remove_all(SXMList *list)
{
    SXMResultCode rc;
    if (list == NULL)
    {
        rc = SXM_E_FAULT;
    }
    else 
    {
        if ((list->flags & SXM_LIST_PREALLOCATED) != SXM_LIST_PREALLOCATED)
        {
            SXMListEntry *entry = list->head;
            while (entry != NULL)
            {
                SXMListEntry *entry_to_remove;
                entry_to_remove = entry;
                entry = sxm_list_next(entry);
                sxe_free(entry_to_remove);
            }
        }
        /* In case of preallocated option the caller shall
         * take care about the memory for each of list's entries
         */
        list->head = list->tail = NULL;
        list->size = 0;
        rc = SXM_E_OK;
    }


    return rc;
}

/**************************************************************************//**
 * Adds new entry into the list.
 *
 * \param[in] list pointer to the list structure
 * \param[in] data entry data
 * \param[in] onTop \p TRUE if new item shall be put before the first one,
 *                  \p FALSE - after the last one.
 * \param[out] entry - output argument to return entry reference in case of
 *                     success. NULL can be passed if it's not needed, but
 *                     only if the list is not a preallocated one
 *                     (\ref SXM_LIST_PREALLOCATED).
 *
 * \retval SXM_E_OK - all good
 * \retval SXM_E_FAULT - null argument is passed
 * \retval SXM_E_NOMEM - no memory to add new entry
 *
 *****************************************************************************/
int sxm_list_add(SXMList *list, void *data, BOOL onTop, SXMListEntry** entry)
{
    SXMListEntry *newEntry;

    if (list == NULL)
    {
        return SXM_E_FAULT;
    }

    if ((list->flags & SXM_LIST_PREALLOCATED) == SXM_LIST_PREALLOCATED)
    {
        if (data == NULL)
        {
            return SXM_E_FAULT;
        }
        newEntry = SXM_LIST_ENTRY_FROM_USER(data);
    }
    else
    {
        // Create new entry
        newEntry = (SXMListEntry*)sxe_calloc(1, sizeof(SXMListEntry));
        if (newEntry == NULL)
        {
            return SXM_E_NOMEM;
        }
    }

    // Attach the data
    newEntry->data = data;

    if (onTop == TRUE)
    {
        sxm_list_next(newEntry) = list->head;
        sxm_list_prev(newEntry) = NULL;

        if (list->head != NULL)
        {
            sxm_list_prev(list->head) = newEntry;
            list->head = newEntry;
        }
    } 
    else 
    {
        sxm_list_next(newEntry) = NULL;
        sxm_list_prev(newEntry) = list->tail;
        if (list->tail != NULL)
        {
            sxm_list_next(list->tail) = newEntry;
            list->tail = newEntry;
        }
    }

    // Is this the first added item?
    if (list->size == 0)
    {
        list->head = list->tail = newEntry;
    }

    ++list->size;
    if (entry != NULL)
    {
        *entry = newEntry;
    }

    return SXM_E_OK;
}

/**************************************************************************//**
 * Removes entry from the list
 *
 * \warning After this operation in case of success the \c entry handle
 *          must not be used.
 * \param[in] list pointer to the list structure
 * \param[in] entry reference to entry to be removed
 * \retval SXM_E_OK all good
 * \retval SXM_E_FAULT null argument is passed
 *
 *****************************************************************************/
int sxm_list_remove(SXMList *list, SXMListEntry *entry)
{
    if ((list == NULL) || (entry == NULL))
    {
        return SXM_E_FAULT;
    }

    if (list->size == 1)
    {
        list->head = list->tail = NULL;
    }
    else if (entry == list->head)
    {
        list->head = sxm_list_next(entry);
        sxm_list_prev(list->head) = NULL;
    }
    else if (entry == list->tail)
    {
        list->tail = sxm_list_prev(entry);
        sxm_list_next(list->tail) = NULL;
    }
    else
    {
        SXMListEntry *after = sxm_list_next(entry);
        SXMListEntry *before = sxm_list_prev(entry);
        sxm_list_prev(after) = before;
        sxm_list_next(before) = after;
    }

    if (!(list->flags & SXM_LIST_PREALLOCATED))
    {
        sxe_free(entry);
    }
    else
    {
        sxm_list_next(entry) = NULL;
        sxm_list_prev(entry) = NULL;
    }

    --list->size;
    return SXM_E_OK;
}

/**************************************************************************//**
 * Looks through all entries within the list
 *
 * \param[in] list valid SXMList instance
 * \param[in] callback the function to be called to each entry
 * \param[in] arg callback argument which will be passed to each callback call
 *
 * \retval SXM_E_OK - all good,
 * \retval SXM_E_FAULT - null argument is passed
 *****************************************************************************/
int sxm_list_iterate(SXMList *list, SXMListCallack callback, void *arg)
{
    int rc;
    if ((list == NULL) || (callback == NULL))
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        /* This logic uses intermediate storage to allow harmless
         * entry deletion from the callback
         */
        SXMListEntry *entry_iter = list->head;
        while (entry_iter != NULL)
        {
            SXMListEntry *entry_to_pass = entry_iter;
            entry_iter = sxm_list_next(entry_iter);
            if (callback(list, entry_to_pass, arg) == 0)
            {
                break;
            }
        }
        rc = SXM_E_OK;
    }
    return rc;
}

/**************************************************************************//**
 * Merge two lists represented by their head references
 * in sorted order.
 * \param[in] entry1 the first entry for sorting
 * \param[in] entry2 the second entry for sorting
 * \param[in] callback pointer to the function used to compare data.
 * \param[in] arg callback's argument
 *
 * \return Head of the merged list
 *
 *****************************************************************************/
static SXMListEntry *sxm_list_merge_sort(SXMListEntry *entry1,
                                         SXMListEntry *entry2,
                                         SXMListSortCallack callback,
                                         void *arg)
{
    SXMListEntry *cur_entry, *result;
    int cmp_result;

    /* Select the merged list head entry */
    cmp_result = callback(sxm_list_data(entry1), sxm_list_data(entry2), arg);
    if (cmp_result <= 0)
    {
        cur_entry = entry1;
        entry1 = sxm_list_next(entry1);
    }
    else
    {
        cur_entry = entry2;
        entry2 = sxm_list_next(entry2);
    }

    /* From here result is the same as we selected */
    result = cur_entry;

    /* Merge rest pars of the lists */
    while ((entry1 != NULL) && (entry2 != NULL))
    {
        // Compare entries
        cmp_result = callback(sxm_list_data(entry1), sxm_list_data(entry2), arg);
        if (cmp_result <= 0)
        {
            /* The first is our candidate */
            sxm_list_next(cur_entry) = entry1;
            cur_entry = entry1;
            entry1 = sxm_list_next(entry1);
        }
        else
        {
            /* The second is our candidate */
            sxm_list_next(cur_entry) = entry2;
            cur_entry = entry2;
            entry2 = sxm_list_next(entry2);
        }
    }

    /* Finalize merging */
    sxm_list_next(cur_entry) = entry1 ? entry1 : entry2;

    return result;
}

/**************************************************************************//**
 * Performs merge sorting for the list based on criteria provided
 * by application through callback function.
 * 
 * The result of sorting operation will be the list in ascending order
 * based on the callback function implementations.
 *
 * \param[in] list pointer to the list structure
 * \param[in] callback comparison callback for entries
 * \param[in] arg callback argument
 * 
 * \retval SXM_E_OK all good,
 * \retval SXM_E_FAULT null argument is passed
 *
 *****************************************************************************/
extern int sxm_list_sort(SXMList *list,
                         SXMListSortCallack callback,
                         void *arg)
{
    if ((list == NULL) || (callback == NULL))
    {
        return SXM_E_FAULT;
    }
    if (sxm_list_size(list) == 2)
    {
        int cmp_result;

        cmp_result =
            callback(sxm_list_data(list->head),
                     sxm_list_data(list->tail), arg);
        if (cmp_result > 0)
        {
            SXMListEntry *tmp;
            // Need to swap
            tmp = list->head;
            list->head = list->tail;
            list->tail = tmp;

            // Update both-directions references
            sxm_list_prev(list->head) = NULL;
            sxm_list_next(list->head) = list->tail;
            sxm_list_prev(list->tail) = list->head;
            sxm_list_next(list->tail) = NULL;
        }
    }
    else if (sxm_list_size(list) > 2)
    {
        byte stack_pos = 0;
        SXMListEntry *item;
        SXMListMergeSortStackItem *stack_item;
        SXMListMergeSortStackItem stack[SXM_LIST_LIST_SORT_LEVEL_MAX];

        item = sxm_list_first(list);

        while (item != NULL)
        {
            /* Initialize */
            stack_item = &stack[stack_pos];
            stack_item->level = 1;
            stack_item->entry = item;

            /* Move next */
            item = sxm_list_next(item);
            sxm_list_next(stack_item->entry) = NULL;
            ++stack_pos;

            /* Merge loop */
            while ((stack_pos > 1) &&
                    (stack[stack_pos - 2].level == stack[stack_pos - 1].level))
            {
                stack_item = &stack[stack_pos - 2];
                stack_item->entry =
                    sxm_list_merge_sort(
                        stack_item->entry, stack[stack_pos - 1].entry,
                        callback, arg);
                ++(stack_item->level);
                --stack_pos;
            }
        }

        /* Final merge loop */
        while (stack_pos > 1)
        {
            stack_item = &stack[stack_pos - 2];
            stack_item->entry =
                sxm_list_merge_sort(
                    stack_item->entry, stack[stack_pos - 1].entry,
                    callback, arg);
            --stack_pos;
        }

        /* Update the reference to the head */
        list->head = stack[0].entry;

        /* Restore double linked list - prev references */
        item = list->head;
        sxm_list_prev(item) = NULL;
        while (sxm_list_next(item) != NULL)
        {
            sxm_list_prev(sxm_list_next(item)) = item;
            item = sxm_list_next(item);
        }
        list->tail = item;
    }
    return SXM_E_OK;
}

/**************************************************************************//**
 * Allocates memory together with room for entry descriptor.
 * 
 * The function allocates as many bytes as the caller asks in addition
 * to single list entry size. The memory is being allocated using sxe_calloc()
 * thus there is no need to initialize memory by zeros in case of success.
 *
 * \warning The memory allocated by this function must be release by 
 * \ref sxm_list_free() function.
 *
 * \param[in] s memory size to be allocated.
 * 
 * \return valid address of the allocated memory or \c NULL in case of error
 *
 *****************************************************************************/
void* sxm_list_allocate(size_t s)
{
    void* result = NULL;
    if (s != 0)
    {
        result = sxe_calloc(1, sizeof(SXMListEntry) + s);
        if (result != NULL)
        {
            result = SXM_LIST_ENTRY_TO_USER(result);
        }
    }
    return result;
}

/**************************************************************************//**
 * Freeses memory which was allocated earlier via \ref sxm_list_allocate(). 
 * 
 * \warning This function shall be used for releasing memory which is
 * allocated via sxm_list_allocate().
 *
 * \param[in] p pointer to the memory returned by allocation function
 * 
 *****************************************************************************/
void sxm_list_free(void *p)
{
    if (p != NULL)
    {
        sxe_free(SXM_LIST_ENTRY_FROM_USER(p));
    }
    return;
}

/**************************************************************************//**
 * Provides the list entry instance with corresponds to the provided memory.
 * \warning The pointer shell address the memory allocated via
 *          \ref sxm_list_allocate. Otherwise, the behavior is undefined
 *
 * \param[in] p pointer to the memory returned by allocation function
 * 
 *****************************************************************************/
SXMListEntry* sxm_list_entry(void *p)
{
    return (p != NULL) ? SXM_LIST_ENTRY_FROM_USER(p) : NULL;
}
