/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_image.c
* \author Alexander Pylchagin
* \date 7/03/2017
* \brief Implementation of SXe image interface.
*
********************************************************************************/

#include "sxm_build.h"
#include <util/sxm_image_internal.h>
#include <util/sxm_common_internal.h>

/***************************************************************************//**
 * This function provides the file path to the image in accordance with the
 * passed descriptor.
 *
 * \param[in] img   image descriptor.
 * \param[out] path address to the pre-allocated memory where to put the path
 * \param[in] size  the size in bytes of the memory addressed by \p path
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter(s) provided
 * \retval SXM_E_INVAL Invalid argument
 * \retval SXM_E_NOENT The image is absent
 * \retval SXM_E_NOMEM Not enough memory to generate the path
 ******************************************************************************/
int sxm_image_path(const SXMImage *img, char *path, size_t size) {
    SXMResultCode rc;
    if ((img == NULL) || (path == NULL)) {
        rc = SXM_E_FAULT;
    }
    else if ((size == 0U) ||
             (img->desc.module[0] == '\0') || (img->desc.file[0] == '\0'))
    {
        rc = SXM_E_INVAL;
    }
    else {
        char *res = sxm_file_make_path(img->desc.type,
                                       &img->desc.module[0],
                                       &img->desc.file[0], path, size);
        if (res != path) {
            rc = SXM_E_NOMEM;
        }
        else {
            /* check that file is accessible, in order to save CPU time there is
             * no need in calling some SXe api to check file existence since it
             * may cause additional make-path execution.
             */
            FILE *pFile = fopen(res, "r");
            if (pFile != NULL) {
                fclose(pFile);
                rc = SXM_E_OK;
            }
            else {
                rc = SXM_E_NOENT;
            }   
        }
    }
    return rc;
}

/***************************************************************************//**
 * This function provides the image's content in accordance with the passed
 * descriptor.
 *
 * \param[in] img   image descriptor.
 * \param[out] pData address to the pre-allocated memory where to put the image
 * \param[in] size  the size in bytes of the memory addressed by \p path
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_INVAL The image description doesn't match to the real file,
 *                     but content has been loaded and can be used if needed.
 * \retval SXM_E_FAULT NULL parameter(s) provided
 * \retval SXM_E_INVAL Invalid argument
 * \retval SXM_E_NOENT The image is absent
 ******************************************************************************/
int sxm_image_content(const SXMImage *img, void *pData, size_t size) {
    SXMResultCode rc;
    if ((img == NULL) || (pData == NULL) || (size == 0U)) {
        rc = SXM_E_FAULT;
    }
    else if ((img->desc.module[0] == '\0') || (img->desc.file[0] == '\0')) {
        rc = SXM_E_INVAL;
    }
    else {
        int ioRc;
        size_t outSize = 0U;
        /* load files content via common API */
        ioRc = sxm_file_load_type(img->desc.type, &img->desc.module[0],
                                  &img->desc.file[0], pData, size, &outSize);
        if (ioRc != 0) {
            rc = SXM_E_NOENT;
        }
        /* Seems like the image is incorrect */
        else if (outSize != img->size) {
            rc = SXM_E_INVAL;
        }
        else {
            rc = SXM_E_OK;
        }
    }
    return rc;
}

/***************************************************************************//**
 * This function retrieves the file size in accordance with the passed descriptor
 *
 * \param[in] img   image descriptor to be initialized
 * \param[out] size   the file's size in case of success
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_NOENT The image is absent
 ******************************************************************************/
int sxm_image_size(const SXMImage *img, size_t *size)
{   
    SXMResultCode rc;
    size_t outSize;
    if (sxm_file_size(img->desc.type, &img->desc.module[0], &img->desc.file[0], &outSize) == -1) {
        *size = 0U;
        rc = SXM_E_NOENT;
    }
    else {
        *size = outSize;
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * This function initializes the image's description in accordance with the passed
 * parameters.
 *
 * \param[out] img   image descriptor to be initialized
 * \param[in] type   file type
 * \param[in] module the name of the module requesting the file
 * \param[in] file   the name of the file, relative to the module name;
 *                   can be NULL to leave the value as-is
 * \param[in] size   the file's size, \p -1 in case if the caller wishes to retrieve it
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_INVAL Invalid argument
 * \retval SXM_E_NOENT The image is absent
 ******************************************************************************/
int sxm_image_init(SXMImage *img, char type, const char *module, const char *file, size_t size)
{
    SXMResultCode rc = SXM_E_OK;
    img->size = 0U;
    img->desc.type = type;
    strncpy(&img->desc.module[0], module, sizeof(img->desc.module) - 1);
    if (&img->desc.file[0] != file) {
        if (file != NULL) {
            strncpy(&img->desc.file[0], file, sizeof(img->desc.file) - 1);
        }
        /* at this point the file name must not be empty */
        if (img->desc.file[0] == '\0') {
            rc = SXM_E_INVAL;
        }
    }
    /* request the image size if the size has not been provided */
    if ((rc == SXM_E_OK) && (size == SXM_IMAGE_UNKNOWN_SIZE)) {
        rc = sxm_image_size(img, &size);
    }
    if (rc == SXM_E_OK) {
        img->size = size;
    }
    return rc;
}

/***************************************************************************//**
 * This function saves the image's content to the file which is described by
 * the image instance.
 *
 * \param[in] img    valid image descriptor
 * \param[in] pData  image's content which has at least SXMImage::size available bytes
 *                   taken from the \p img instance
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_PIPE Failed to perform I/O operations
 * \retval SXM_E_INVAL Invalid argument
 * \retval SXM_E_NOENT The image is absent
 ******************************************************************************/
int sxm_image_save(const SXMImage *img, const void *pData) {
    SXMResultCode rc;
    if ((img == NULL) || (pData == NULL)) {
        rc = SXM_E_FAULT;
    }
    else if ((img->desc.module[0] == '\0') || (img->desc.file[0] == '\0') || (img->size == 0U)) {
        rc = SXM_E_INVAL;
    }
    else {
        /* open the file for writing */
        FILE *pFile = sxm_file_open(img->desc.type, &img->desc.module[0], &img->desc.file[0], "wb");
        if (pFile != NULL) {
            /* try to write whole content */
            if (fwrite(pData, img->size, 1U, pFile) != 1U) {
                rc = SXM_E_PIPE;
            }
            else {
                rc = SXM_E_OK;
            }
            fclose(pFile);
            if (rc != SXM_E_OK) {
                /* remove file if the write has failed */
                sxm_file_remove(img->desc.type, &img->desc.module[0], &img->desc.file[0]);
            }
        }
        else {
            rc = SXM_E_PIPE;
        }
    }
    return rc;
}
