/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_grid.c
* \author Leslie French
* \date 25/04/2014
* \brief Common operations for services that use a 2x2 grid for location
*        referencing (Fuel, EV, Parking). Please, refer to \ref sxe_grid_page
*        for implementation details.
*
* \page sxe_grid_page GRID (Baseline Files Using Grid Squares)
*
* Some services divide their coverage area into a grid of 2&deg;x2&deg;
* (longitude x latitude) grid squares.  Baseline data are stored and indexed
* according to their grid square, and the broadcast transmission protocols
* divide their data into the same grid squares.  Examples of these services
* are Fuel Stations (both U.S. and Canada variants), Electric-Vehicle Charging
* Station Locations, and Parking Locations.
*
* The grid library routines support these services through common code. In order to
* use the grid library, the service must adhere to the following requirements:
*    -# The maximum size of a protocol Access Unit is 5,120 bytes
*    -# The maximum number of Access Units for any single grid square is 16
*    -# Only a single collection/extraction request is supported
*    -# The maximum distance over which the collection will be made lies within 4
*       (2x2) grid squares
*    -# The maximum number of entries returned in a single extraction is 2,048
*    -# Entries are returned sorted by distance from a supplied point.
*    -# The baseline database root block contains 2,048 grid-location indexes (regions)
*
* The Grid routines use a common service structure which is independent of the internal
* details of the module being supported. All these structures are shared with the service
* module. The library and the module co-operate to maintain the state of the structures:
*    - \ref SXMGridAU
*    - \ref SXMGridSavedAU
*    - \ref SXMGridBaseLine
*    - \ref SXMGridBaseLine
*    - \ref SXMGridExtraction
*    - \ref SXMGridLocation
*    - \ref SXMGridService
*
*******************************************************************************/

#define DEBUG_VAR ((service) ? service->ds.debug_level : (uint)(-1))
#define DEBUG_TAG service ? &service->ds.service[0] : "??"

#include <sxm_sdk.h>
#include <util/sxm_noname_internal.h>
#include <util/sxm_grid.h>
#include <util/sxm_common_internal.h>

/** GRID Service region is the valid region over which we are prepared
 *  to consider making collections and extractions.
 */
static SXMMBR grid_service_region =  {
#if SXM_USE_FIX
     {sxm_fix_int(-168), sxm_fix_int(12)},
     {sxm_fix_int(-42), sxm_fix_int(74)}
#else
     {-168.0, 12.0}, {-42.0, 74.0}
#endif
};

/***************************************************************************//**
 *  FORWARD DECLAERATION
 ******************************************************************************/
static int grid_lli_rfd_remove(SXMGridService *service, int id);

static int grid_lli_rfd_status(SXMGridService *service, int id, SXMRfdMetadata *rfd);

static int grid_lli_rfd_start(SXMGridService *service, SXMRfdMetadata *rfd);

static void grid_report_up(int type, int param, void *dsuser);

static int grid_user_poi_extract(SXMGridService *service, void *ret);

static void grid_timer_callback(void *pArg);

/***************************************************************************//**
 * \name Data Builders
 *
 * 'heap' is a heap of station indexes sorted by distance from the collection
 * point (the 'distance' function). The actual distance is R * sqrt(distance)
 * but the distance function is monotonic and good enough for compares.
 *
 * The heap is only built at the 'begin' point, and is destroyed during
 * extraction.
 *
 * @{
 ******************************************************************************/
/***************************************************************************//**
 * Allocates heap structure for sorting the stations entries during extraction.
 *
 * \param[in] grid a pointer to an initialized grid structure
 * \param[in] depth expressed as the number of station indexes
 *
 * \retval SXM_E_OK success
 * \retval SXM_E_NOMEM memory allocation failed
 *
 ********************************************************************************/
SXMResultCode sxm_grid_heap_alloc(SXMGridService *grid, size_t depth)
{
    SXMResultCode rc;
    const size_t heapsize = depth * sizeof(short),
                 locsize = depth * sizeof(SXMGridLocation);

    // attempt to create the heap structures
    grid->ploc = (SXMGridLocation *)sxe_malloc(locsize + heapsize);
    if (!grid->ploc) {
        // clear sizes
        grid->heapsize = grid->locsize = 0;
        rc = SXM_E_NOMEM;
    }
    else {
        grid->pheap = (short *)(grid->ploc + depth);
        grid->heapsize = heapsize;
        grid->locsize = locsize;
        rc = SXM_E_OK;
    }

    return rc;
}

/***************************************************************************//**
 * Destroys the heap structure for sorting the stations entries during extraction.
 *
 * \param[in] grid a pointer to an initialized grid structure
 *
 ********************************************************************************/
void sxm_grid_heap_free(SXMGridService *grid)
{
    // free any existing instances
    if (grid->ploc) {
        sxe_free(grid->ploc);
        grid->ploc = NULL;
        grid->locsize = 0;
        grid->pheap = NULL;
        grid->heapsize = 0;
    }

    return;
}

/***************************************************************************//**
 * Inserts a record into the heap according to distance from center point.
 *
 * Extractions are sorted from the supplied center sort point. The objects
 * (actually the indexes into the loc array) are held in a heap structure,
 * based on the distance metric in the loc array).
 *
 * \param[in] grid a pointer to an initialized grid structure
 * \param[in] rn record number to insert into heap
 *
 ********************************************************************************/
static void grid_heap_insert(SXMGridService *grid, ushort rn)
{
    int ix = grid->hc;
    const SXMFLOAT d0 = grid->ploc[rn].distance;

    grid->pheap[grid->hc++] = (short)rn;
    while (ix > 0) {
        if (d0 >= grid->ploc[grid->pheap[(ix - 1) / 2]].distance) {
            return;
        }
        grid->pheap[ix] = grid->pheap[(ix - 1) / 2];
        ix = (ix - 1) / 2;
        grid->pheap[ix] = (short)rn;
    }
}

/***************************************************************************//**
 * The routine returns the location at the top of the heap, removing that entry
 * and rebuilding the heap for the next-closest location
 *
 * \param[in] service a pointer to an initialized grid structure
 *
 * \return the loc index of the next closest object
 *
 ********************************************************************************/
static short grid_heap_extract(SXMGridService *service)
{
    short ret = service->pheap[0];
    int curr = 0;
    int n1, n2;
    short ix1, ix2;

    while ((2 * curr + 1) < service->hc) {
        n1 = 2 * curr + 1;
        n2 = 2 * curr + 2;

        if ((n1 + 1) == service->hc) {
            service->pheap[curr] = service->pheap[n1];
            curr = n1;
            break;
        }
        ix1 = service->pheap[n1];
        ix2 = service->pheap[n2];

        if (ix1 == -1) {
            if (ix2 == -1) {
                break;
            }
            service->pheap[curr] = ix2;
            curr = n2;
        }
        else if (ix2 == -1) {
            service->pheap[curr] = ix1;
            curr = n1;
        }
        else if (service->ploc[ix1].distance < service->ploc[ix2].distance) {
            service->pheap[curr] = ix1;
            curr = n1;
        }
        else {
            service->pheap[curr] = ix2;
            curr = n2;
        }
    }
    service->pheap[curr] = -1;
    return ret;
}

/***************************************************************************//**
 * Rebuilds the heap for a given center point.
 *
 * If the service is given a new sort point, but does not need to re-evaluate
 * the list of objects, or is starting a new extraction at the same point
 * before, the existing heap can be recreated. The \ref sxm_grid_heap_rebuild
 * routine constructs a heap from the existing loc array, updating the distance
 * values according to the new sort point.
 *
 * \param[in] grid a pointer to an initialized grid structure
 * \param[in] center the new center (sort) point
 *
 ********************************************************************************/
static void grid_heap_rebuild(SXMGridService *grid, const SXMPoint *center)
{
    ushort i;

    if (center) {
        grid->ext.center = *center;
        //  rebuild the distances
        for (i = 0; i < grid->sp; ++i) {
            grid->ploc[i].distance = sxm_point_distance(center, &grid->ploc[i].p);
        }
    }

    //  build the heap at this point, ready for extract
    grid->hc = 0;
    grid->pheap[0] = -1;
    for (i = 0; i < grid->sp; ++i) {
        grid_heap_insert(grid, i);
    }
}
/** @} */

/***************************************************************************//**
 * Returns the index of a region if it is loaded.
 *
 * The sxm_grid_baseix routine determines if a region is one that is currently
 * being collected, and returns the index in the bptr array if it is.
 *
 * \param[in] pGrid a pointer to an initialized grid structure
 * \param[in] region a region the caller is looking for
 *
 * \return value on range from 0 to #SXM_GRID_NEEDED_REGIONS if the region is
 *         being collected or #SXM_GRID_REGION_INVAL_IDX if the region not loaded
 *
 ********************************************************************************/
int sxm_grid_baseix(SXMGridService *pGrid, ushort region)
{
    uint i; 
    int index;

    for (i = 0, index = SXM_GRID_REGION_INVAL_IDX; i < ARRAY_SIZE(pGrid->bptr); ++i) {
        if  (pGrid->bptr[i]->region == region) {
            index = (int)i;
            break;
        }
    }

    return index;
}

/***************************************************************************//**
 * Check and save access unit (AU).
 *
 * If an incoming Access Unit lies within one of the regions being collected,
 * the \ref sxm_grid_aucheck routine will add that data to the collection, unless
 * it is a simple repeats (same signature) of data already received.
 *
 * \note It validates AU by computing CRC and saving with timestamp.
 * \note The AU may originate from the broadcast or from the CFile AU cache
 * during replay.
 *
 * \param[in] pGrid a pointer to an initialized grid structure
 * \param[in] hix index of loaded baselines for region containing the AU
 * \param[in] pFp a pointer to the common packet data structure
 * \param[in] aucnt the sequence number of this access unit in the region
 *
 * \retval SXM_E_OK if AU CRC check passed, or AU was already saved
 *                  (the service must process the packet)
 * \retval SXM_E_NOENT the AU is duplicate and may have nothing for the service
 *                     to process.
 * \retval SXM_E_INVAL if AU CRC check failed (no further processing required)
 *
 ********************************************************************************/
SXMResultCode sxm_grid_aucheck(SXMGridService *service, uint hix, SXMPacket *pFp, uint aucnt)
{
    // initialize baseline pointer for region containing the AU
    SXMGridBaseLine *pBl = service->bptr[hix];
    // Initialize return value to 'OK'
    SXMResultCode rc = SXM_E_OK;

    // check for and set replay mode state
    if (!((aucnt < ARRAY_SIZE(pBl->safe)) && (pBl->safe[aucnt].pkt == pFp->packet))) {
        SXMGridSavedAU *au =
            (aucnt < ARRAY_SIZE(pBl->safe)) ? &pBl->safe[aucnt] : NULL;

        // Check that that AU at this position has the same len and CRC in
        // order to detect duplicate. Assuming that the CRC kept by the AU
        // memory should be the same as the CRC kept by C-file. The C-file shall
        // have valid CRC for AU's content as well as passed AU at the end
        // so there is no need to recalculate it before check since it doesn't
        // matter if new AU will have the same CRC after calculation or not.
        // FYI: If not - the packet will be thrown out based on CRC check later
        if (au && (au->pl == pFp->pl) && ((uint)x4(pFp->packet, pFp->pl - CRC_BYTELEN) == au->crc)) {
            // Duplicate detected, but still need to update timestamp in order
            // to process data age out gracefully.
            au->ts = sxm_sxe_time_now();
            rc = SXM_E_NOENT;
        }
        // check CRC for broadcast AUs, (not replayed AUs, these have
        // already been checked)
        else if (!sxm_crc32_check(pFp->packet, pFp->pl, &pFp->crc)) {
            ERROR_PKT("CRC failed, aucnt %u", aucnt);
            rc = SXM_E_INVAL;
        }
        // process good AUs with sequence number within range received
        // from the broadcast and only process AUs not already saved
        else if (au) {
            if (!au->pkt) {
                // pick one off the heap
                const int ix = sxm_cfile_map_alloc(service->root, service->heap_sec, 1);
                if (ix >= 0) {
                    // populate AU metadata
                    au->pkt = (byte *)((SXMSector*)service->heap + (uint)ix * SXM_CSECTORCNT(*service->heap));
                    au->pktix = (uint)ix;
                }
                else {
                    PLOG_PKT("AU heap is full");
                }
            }

            // copy the AU to the safe
            if (au->pkt) {
                au->crc = pFp->crc;
                au->pl = pFp->pl;
                au->ts = sxm_sxe_time_now();
                memcpy(au->pkt, pFp->packet, (size_t)pFp->pl);
            }
        }
    }

    return rc;
}

/***************************************************************************//**
 * Determines what regions need to be collected for the center point.
 *
 * It is the responsibility of the service module to maintain the correct list
 * of 4 regions to be collected, through the mbr and center values in the
 * Extraction structure.  The sxm_grid_needed routine completed the needed array
 * in the Service structure once the service has determined the mbr and center
 * values.
 *
 * This code determines the quadrant in which the point lies, within the 2x2 region.
 *    - \c i is the latitude region index,
 *    - \c j is the longitude region index
 *    - \c i1 is 0 if in the lower half of region, 1 of in the upper half
 *    - \c j1 is 0 in the right-hand half, 1 if in the left-hand half
 *
 * so we get the 4 region indexes:
 *
 * ~~~~
 *                             i
 *                             ^
 *                     '11'    |     '10'
 *                 ------------+------------   > j
 *                     '01'    |     '00'
 *
 * ~~~~
 *
 * From that we pick the 3 extra regions closest to that quadrant.
 *
 * The calculation gives the wrong values if you are right at the edge of the map,
 * but that doesn't really matter, since there isn't any other useful region to
 * include anyway.
 *
 * \param[in] grid a pointer to an initialized grid structure
 *
 * \retval SXM_E_OK regions were selected.
 * \retval SXM_E_OK SXM_E_INVAL Center point outside service area.
 *
 ********************************************************************************/
static int grid_needed(SXMGridService *grid)
{
    SXMGridExtraction *ext = &grid->ext;
#if SXM_USE_FIX
    const int iLon = (ext->center.lon >= 0) ? (ext->center.lon >> 15) : -(-ext->center.lon >> 15);
    const int iLat = (ext->center.lat >= 0) ? (ext->center.lat >> 15) : -(-ext->center.lat >> 15);
#else
    const int iLon = (int)ext->center.lon;
    const int iLat = (int)ext->center.lat;
#endif
    // Determine the region
    const int jj = (iLon + 167) / 2; //TODOR: Why 167 while GRID lower-left if stated as -168!
    const int ii = (iLat - 12) / 2;

    //  always want the region containing the point
    const ushort i = (ushort)ii;
    const ushort j = (ushort)jj;

    //  center must be a valid square
    if ((jj < 0) || (jj > 64) || (ii < 0) || (ii > 32)) {
        return SXM_E_INVAL;
    }

    memset(grid->needed, 0, sizeof(grid->needed));

    grid->centerix = grid->needed[0] = (ushort)(j * 32 + i);

    // Determine which quadrant the point lies in
    switch ((iLat & 1) * 2 + (iLon & 1)) {
        case 0:
            grid->needed[1] = (ushort)(j * 32 + i - 1);
            grid->needed[2] = (ushort)((j + 1) * 32 + i); 
            grid->needed[3] = (ushort)((j + 1) * 32 + i - 1);
            break;
        case 1:
            grid->needed[1] = (ushort)(j * 32 + i - 1);
            grid->needed[2] = (ushort)((j - 1) * 32 + i);
            grid->needed[3] = (ushort)((j - 1) * 32 + i - 1);
            break;
        case 2:
            grid->needed[1] = (ushort)(j * 32 + i + 1);
            grid->needed[2] = (ushort)((j + 1) * 32 + i);
            grid->needed[3] = (ushort)((j + 1) * 32 + i + 1);
            break;
        case 3:
            grid->needed[1] = (ushort)(j * 32 + i + 1);
            grid->needed[2] = (ushort)((j - 1) * 32 + i);
            grid->needed[3] = (ushort)((j - 1) * 32 + i + 1);
            break;
    }

    if (!((grid->needed[0] < grid->needed[1]) &&
          (grid->needed[1] < grid->needed[2]) &&
          (grid->needed[2] < grid->needed[3])))
    {
        // Do hard-coded sorting network for four integer value in order
        // use binary search later for stations collection
        // (https://en.wikipedia.org/wiki/Sorting_network)
        // 0 --+-----+-----
        //     |     |
        // 1 --|--+--+--+--
        //     |  |     |
        // 2 --+--|--+--+--
        //        |  |
        // 3 -----+--+-----

        if (grid->needed[0] > grid->needed[2])
            SWAP_VALUES(ushort, grid->needed[0], grid->needed[2]);
        if (grid->needed[1] > grid->needed[3])
            SWAP_VALUES(ushort, grid->needed[1], grid->needed[3]);
        if (grid->needed[0] > grid->needed[1])
            SWAP_VALUES(ushort, grid->needed[0], grid->needed[1]);
        if (grid->needed[2] > grid->needed[3])
            SWAP_VALUES(ushort, grid->needed[2], grid->needed[3]);
        if (grid->needed[1] > grid->needed[2])
            SWAP_VALUES(ushort, grid->needed[1], grid->needed[2]);
    }

    return SXM_E_OK;
}

/***************************************************************************//**
 * This function calculates mbr which covers all needed regions.
 *
 * \param[in] grid a pointer to an initialized grid structure
 * \param[out] ll pointer to the memory where location to be populated with
 *                lower-left region id when successful
 * \param[out] ur pointer to the memory location to be populated with
 *                upper-right region id when successful
 * \param[out] mbr pointer to the memory location to be populated with
 *                 mbr with all needed regions.
 * \retval SXM_E_OK success
 * \retval SXM_E_NOENT there is no grid-regions
 * \retval SXM_E_FAULT NULL parameter
 ********************************************************************************/
static int grid_needed_corner_regions(SXMGridService *grid, uint *ll, uint *ur, SXMMBR *mbr)
{
    SXMResultCode rc = SXM_E_NOENT;
    if (!grid || (!ll && !ur && !mbr)) {
        rc = SXM_E_FAULT;
    }
    else {
        uint ix;

        // Since the list of needed regions is sorted here we need to
        // find the first valid one to get lower-left corner
        for (ix = 0; ix < ARRAY_SIZE(grid->needed); ++ix) {
            if (SXM_GRID_REGION_IS_VALID(grid->needed[ix])) {
                if (ll) {
                    *ll = grid->needed[ix];
                }
                break;
            }
        }
        if (ix < ARRAY_SIZE(grid->needed)) {
            rc = SXM_E_OK;

            if (mbr) {
                memset(mbr, 0, sizeof(*mbr));
#if SXM_USE_FIX
                mbr->ll.lat = grid_service_region.ll.lat + sxm_fix_int((SXM_GRID_REGION_LAT_UNPACK(grid->needed[ix]) << 1));
                mbr->ll.lon = grid_service_region.ll.lon + sxm_fix_int((SXM_GRID_REGION_LON_UNPACK(grid->needed[ix]) << 1));
#else
                mbr->ll.lat = grid_service_region.ll.lat + (SXMFLOAT)(SXM_GRID_REGION_LAT_UNPACK(grid->needed[ix]) << 1);
                mbr->ll.lon = grid_service_region.ll.lon + (SXMFLOAT)(SXM_GRID_REGION_LON_UNPACK(grid->needed[ix]) << 1);
#endif
            }

            // Looking for the valid region since the end of the list to find the upper-right region
            ix = ARRAY_SIZE(grid->needed);
            do {
                --ix;
                if (SXM_GRID_REGION_IS_VALID(grid->needed[ix])) {
                    if (ur) {
                        *ur = grid->needed[ix];
                    }
                    if (mbr) {
#if SXM_USE_FIX
                        mbr->ur.lat = grid_service_region.ll.lat + sxm_fix_int((SXM_GRID_REGION_LAT_UNPACK(grid->needed[ix]) << 1));
                        mbr->ur.lon = grid_service_region.ll.lon + sxm_fix_int((SXM_GRID_REGION_LON_UNPACK(grid->needed[ix]) << 1));
                        mbr->ur.lat += sxm_fix_int(2);
                        mbr->ur.lon += sxm_fix_int(2);
#else
                        mbr->ur.lat = grid_service_region.ll.lat + (SXMFLOAT)(SXM_GRID_REGION_LAT_UNPACK(grid->needed[ix]) << 1);
                        mbr->ur.lon = grid_service_region.ll.lon + (SXMFLOAT)(SXM_GRID_REGION_LON_UNPACK(grid->needed[ix]) << 1);
                        mbr->ur.lat = (SXMFLOAT)(mbr->ur.lat + 2.0);
                        mbr->ur.lon = (SXMFLOAT)(mbr->ur.lon + 2.0);
#endif
                    }
                    break;
                }
            } while (ix > 0);
        }
    }
    return rc;
}

/***************************************************************************//**
 * Reloads the regions required for a given extraction.
 *
 * The sxm_grid_reload routine populates the #SXM_GRID_NEEDED_REGIONS bptr
 * structures in the SXMGridService with the #SXM_GRID_NEEDED_REGIONS needed
 * values. It will retain any previously-loaded data, and saved Access Units.
 *
 * \param[in] grid a pointer to an initialized grid structure
 *
 ********************************************************************************/
static void grid_reload(SXMGridService *service)
{
    uint i, j;
    uint need = 15;
    uint ifree = 15;

    //  if we can re-use any of the save data, then do so, else reset
    for  (i = 0; i < ARRAY_SIZE(service->needed); i++) {
        for  (j = 0; j < ARRAY_SIZE(service->bptr); j++) {
            if  (service->bptr[j]->region == service->needed[i])
            {
                need &= ~(1U << i);		// don't need to load this one
                ifree &= ~(1U << j);		// and keep the block in-use
                break;
            }
        }
    }

    //  free any saved AUs from the unwanted regions
    for  (j = 0; j < ARRAY_SIZE(service->bptr); j++) {
        if  (ifree & (1U << j)) {
            SXMGridBaseLine * const bptr = service->bptr[j]; 
            for  (i = 0; i < ARRAY_SIZE(bptr->safe); i++) {
                if (bptr->safe[i].pkt) {
                    sxm_cfile_map_free(service->root, service->heap_sec,
                        bptr->safe[i].pktix, 1);
                }
            }
            if (bptr->dbentry) {
                sxe_free(bptr->dbentry);
            }
            memset(bptr, 0, sizeof(*bptr));
        }
    }

    for (i = 0; i < ARRAY_SIZE(service->needed); i++) {
        if (need & (1U << i)) {
            for  (j = 0; j < ARRAY_SIZE(service->bptr); j++) {
                if (ifree & (1U << j)) {
                    SXMGridBaseLine * const bptr = service->bptr[j]; 
                    bptr->region = service->needed[i];
                    if (sxm_grid_load_region(service, bptr, bptr->region, FALSE) != SXM_E_OK) {
                        memset(bptr, 0, sizeof(*bptr));
                    }
                    ifree &= ~(1U << j);		// used this block now
                    break;
                }
            }
        }
    }
}

/**
 * \name Database (T-File) Handling
 * @{
 */

/***************************************************************************//**
 * Checks existence of data for the specified region.
 *
 * The function simple checks existence of the region's data inside database
 * with no attempt to load it.
 *
 * \param[in] grid a pointer to an initialized grid structure, see \ref SXMGridService
 * \param[in] region the region number to be checked
 *
 * \retval FALSE there is no data for the region
 * \retval TRUE data available and can be loaded if needed
 *
 ********************************************************************************/
static BOOL grid_region_has_data(const SXMGridService *grid, ushort region)
{
    return (SXM_GRID_REGION_IS_VALID(region) && grid->dbroot[2 * region + 1]) ? TRUE : FALSE;
}

/***************************************************************************//**
 * Load a specific region from the baseline file into memory.
 *
 * The \c centerix value contains the grid index number of the grid square
 * containing the center point. Having determined the list of regions, the
 * service may bring the data into memory in one of two ways.
 *
 * \note The bl structure does not have to be one of the structures in the
 *       SXMGridService. This routine is also used to load baseline data during
 *       RFD updates.
 *
 * \param[in] service a pointer to an initialized grid structure
 * \param[in] bl a pointer to a baseline structure to contain the data
 * \param[in] region the region number to be loaded
 * \param[in] bChainNeeded \p TRUE if the chain handle is neede or \p FALSE - of not
 *
 * \retval SXM_E_OK there is no error
 * \retval SXM_E_INVAL the region is invalid
 * \retval SXM_E_NOMEM the region failed to load/create - memory allocation failed
 * \retval SXM_E_PIPE I/O error reading the region from the TFile
 *
 ********************************************************************************/
SXMResultCode sxm_grid_load_region(SXMGridService *service, SXMGridBaseLine *bl, ushort region, BOOL bChainNeeded)
{
    SXMResultCode rc = SXM_E_OK;

    // This check prevents the access to the memory out side the grid->dbroot
    // since per grid t-file design it can address certain number of item. In some
    // cases the region can be even less than 0 which is completely inaccessible
    // with possible SIGSEGV situation.
    if (SXM_GRID_REGION_IS_VALID(region)) {
        void *buffer;   // where the region (or dummy region) is loaded (created) in memory
        ushort blocks = 0;
        SXMTFileChain *chain = NULL;

        //  if this region has data, load it
        if (grid_region_has_data(service, region) == TRUE) {
            int intRc;
            // index and block count from the active root (directory) of the TFile
            ushort index = service->dbroot[2 * region];
            blocks = service->dbroot[2 * region + 1];

            // allocate a buffer and load the region into buffer from the TFile (database)
            buffer = sxm_tfile_alloc_and_read(service->dbfile, index, blocks,
                                              ((bChainNeeded == TRUE)? &chain : NULL), &intRc);
            rc = (SXMResultCode)intRc;
        }
        // region has no data
        else {
            // create an empty (dummy) region
            buffer = sxe_calloc(1, 16);
            if (!buffer) {
                rc = SXM_E_NOMEM;
            }
        }
        // if alloc and read failed, buffer will be NULL and rc will have the error code
        if (!buffer) {
            PLOG_DBF("Failed to load region %d (rc=%d)", region, rc);
        }
        // database load successfully
        else {
            // initialize baseline pointers for non-empty region
            bl->dbentry = (uint *)buffer;
            bl->region = region;
            bl->version = bl->dbentry[0] >> 16;
            bl->dbentry[0] = bl->dbentry[0] & 0xFFFF;
            bl->chain = chain;
            bl->base = bl->dbentry + bl->dbentry[0] + 1;
            bl->size = (uint)(blocks * sxm_tfile_bsize(service->dbfile) - 
                /* index area */ (bl->dbentry[0] + 1) * sizeof(bl->dbentry[0]));
            rc = SXM_E_OK;
        }
    }
    // invalid region
    else {
        ERROR_DBF("Invalid region id %d", region);
        rc = SXM_E_INVAL;
    }

    return rc;
}

/***************************************************************************//**
 * The routine opens the associated T-File which is assumed to be organized
 * as Grid.
 *
 * \param[in] service service instance
 * \param[out] ppTFile pointer to SXMTFile*
 * \param[in] srvc the module name of the service
 * \param[in] dbfile the name of the baseline (database) file
 * \param[in] mode the file open mode  ("rb" or "w+b" usually)
 * \param[in] format the internal data format unique identifier
 * \param[in] schema expected schema version
 *
 * \retval SXM_E_OK TFile opened
 * \retval SXM_E_FAULT NULL argument(s)
 * \retval SXM_E_BAD_DB TFile corrupted or has unsupported format
 * \retval SXM_E_NO_DB TFile failed to open
 *
 ********************************************************************************/
SXMResultCode sxm_grid_database_open(SXMGridService *service,
                           SXMTFile **ppTFile, const char *srvc,
                           const char *dbfile, const char *mode,
                           const char *format, uint schema)
{
    SXMResultCode rc = SXM_E_NO_DB;
    int intRc;
    SXMTFile *pTFile;
    uint overall_schema;

    // Validate input
    if (!service || !ppTFile || !srvc || !dbfile || !format || !mode) {
        return SXM_E_FAULT;
    }

    // set NULL until we have good pointer
    *ppTFile = NULL;

    // open the grid database
    pTFile = sxm_tfile_open('W', srvc, dbfile, mode, format, &overall_schema, &intRc);
    rc = (SXMResultCode)intRc;
    if (pTFile) {
        // if schema does not match, it is of no use
        if (SXM_GRID_TFILE_SCHEMA_VERSION != SXM_GRID_EXTRACT_OWN_SCHEMA(overall_schema)) {
            DEBUG_DBF(": unexpected grid t-file schema %u, expected %u",
                SXM_GRID_EXTRACT_OWN_SCHEMA(overall_schema), SXM_GRID_TFILE_SCHEMA_VERSION);
            sxm_tfile_close(pTFile);
            rc = SXM_E_BAD_DB;
        }
        else if (schema != SXM_GRID_EXTRACT_SERVICE_SCHEMA(overall_schema)) {
            DEBUG_DBF(": unexpected service's t-file schema %u, expected %u",
                SXM_GRID_EXTRACT_SERVICE_SCHEMA(overall_schema), schema);
            sxm_tfile_close(pTFile);
            rc = SXM_E_BAD_DB;
        }
        else {
            rc = SXM_E_OK;
        }
    }

    // return file pointer if we have it
    if (rc == SXM_E_OK) {
        *ppTFile = pTFile;
    }

    return rc;
}

/***************************************************************************//**
 * The routine initializes the library for a particular service.
 *
 * \param[in] grid a pointer to an empty GridService structure
 * \param[in] service the module name of the service
 * \param[in] dbfile the name of the baseline (database) file
 * \param[in] format the internal data format unique identifier
 * \param[in] schema a pointer to an unsigned integer to hold opened T-File schema
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_NO_DB TFile failed to open
 * \retval SXM_E_BAD_DB TFile corrupted or has unsupported format
 *
 ********************************************************************************/
SXMResultCode sxm_grid_database_start(SXMGridService *grid, const char *service,
                            const char *dbfile, const char *format,
                            uint schema)
{
    SXMResultCode rc;

    rc = sxm_grid_database_open(grid, &grid->dbfile, service, dbfile, "rb", format, schema);
    if (rc == SXM_E_OK) {
        grid->dbroot = (ushort *)sxm_tfile_root(grid->dbfile);
        grid->dbuser = sxm_tfile_user(grid->dbfile);

        if (grid->dbroot && grid->dbuser) {
            grid->db_version = (int)grid->dbuser[0];
        }
        else {
            sxm_tfile_close(grid->dbfile);
            grid->dbroot = NULL;
            grid->dbuser = NULL;
            grid->dbfile = NULL;
            rc = SXM_E_BAD_DB;
        }
    }
    return rc;
}

/***************************************************************************//**
 * Restarts the grid service.
 *
 * If the database file is updated (for example using a file delivered over RFD),
 * the grid library needs to refresh its copy of the data.
 * The \ref sxm_grid_database_restart routine reloads any data from the
 * baseline file.
 *
 * \param[in] grid a pointer to an empty GridService structure
 * \param[in] service the module name of the service
 * \param[in] dbfile the name of the baseline (database) file
 * \param[in] schema a pointer to an unsigned integer to hold opened T-File schema
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_NO_DB TFile failed to open
 * \retval SXM_E_BAD_DB TFile corrupted or has unsupported format
 *
 ********************************************************************************/
SXMResultCode sxm_grid_database_restart(SXMGridService *service, const char *svc,
                              const char *dbfile, uint schema) {
    SXMResultCode rc;

    char format[sizeof(service->dbfile->hdr->format)];

    LOCK(service->service_mutex);
    // Keep in the temp storage current format and sver
    memcpy(format, service->dbfile->hdr->format, sizeof(service->dbfile->hdr->format));

    // Close active DB
    sxm_tfile_close(service->dbfile);
    service->dbfile = NULL;

    rc = sxm_grid_database_start(service, svc, dbfile, format, schema);
    if (rc == SXM_E_OK) {
        uint i;
        for (i = 0; i < ARRAY_SIZE(service->bptr); ++i) {
            SXMGridBaseLine * const bptr = service->bptr[i];
            if (sxm_grid_load_region(service, bptr, bptr->region, FALSE) != SXM_E_OK) {
                memset(bptr, 0, sizeof(*bptr));
            }
        }
    }
    UNLOCK(service->service_mutex);
    return rc;
}

/********************************************************************//**
 *
 * The gets the current version of the database.
 *
 * \note The version will only change once an RFD-based file update has been
 * completely and successfully applied. The service will notify the
 * application through the service notification callback of the updated
 * service database. The application will decide when to restart the service
 * and cause the loading of the new database content.
 *
 * \param[in]  service     service instance
 * \param[out] pVer        a pointer uint variable in the application's memory
 *                         to be populated with the DB version.
 *
 * \retval  SXM_E_OK        DB version is retrieved and populated into pVer
 * \retval  SXM_E_FAULT     NULL parameter
 * \retval  SXM_E_STATE     Service in wrong state
 *
 ************************************************************************/
SXMResultCode sxm_grid_database_get_version(SXMGridService *service, uint *pVer) {
    SXMResultCode rc;

    if (!pVer) {
        rc = SXM_E_FAULT;
        ERROR_API("NULL parameter");
    }
    else if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else {
        *pVer = (uint)service->db_version;
        DEBUG_API("version=%u", *pVer);
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * Utility function to calculate coverage area of the MBR within the grid.
 *
 * This function allows caller to find out two regions upper-right and lower-left
 * inside the GRID which can be used to find all regions covered by the MBR
 *
 * \param[in] mbr the MBR for coverage calculation
 * \param[out] ll pointer to the memory where location to be populated with
 *                lower-left region id when successful
 * \param[out] ur pointer to the memory location to be populated with
 *                upper-right region id when successful
 *
 * \retval SXM_E_OK The region has intersection with grid service area and
 *                  both corner regions provided
 * \retval SXM_E_FAULT NULL Parameter
 * \retval SXM_E_INVAL Invalid MBR provided
 * \retval SXM_E_NOENT No regions are covered by the MBR
 *
 ********************************************************************************/
static int grid_mbr_corner_regions(const SXMMBR *mbr, uint *ll, uint *ur)
{
    int rc;
    if (!mbr || !ll || !ur) {
        rc = SXM_E_FAULT;
    }
    else if (!sxm_mbr_valid(mbr)) {
        rc = SXM_E_INVAL;
    }
    else if (!sxm_mbr_intersects(mbr, &grid_service_region)) {
        rc = SXM_E_NOENT;
    }
    else {
        int dlat, dlon;
#if SXM_USE_FIX
        int frac;
#endif
        // In order to calculate corner regions the mbr must be inside the
        // GRID service area. Thus, need to correct it's edges.
        SXMMBR gridMBR = *mbr;
        if (gridMBR.ll.lat < grid_service_region.ll.lat)
            gridMBR.ll.lat = grid_service_region.ll.lat;
        if (gridMBR.ll.lon < grid_service_region.ll.lon)
            gridMBR.ll.lon = grid_service_region.ll.lon;
        if (gridMBR.ur.lat > grid_service_region.ur.lat)
            gridMBR.ur.lat = grid_service_region.ur.lat;
        if (gridMBR.ur.lon > grid_service_region.ur.lon)
            gridMBR.ur.lon = grid_service_region.ur.lon;

        // Second step is to do the grid coordinates calculation
        dlon = (int)(gridMBR.ll.lon - grid_service_region.ll.lon) >> 1;
        dlat = (int)(gridMBR.ll.lat - grid_service_region.ll.lat) >> 1;
#if SXM_USE_FIX
        sxm_fix_printable(dlon, &dlon, &frac);
        sxm_fix_printable(dlat, &dlat, &frac);
#endif
        *ll = (uint)SXM_GRID_REGION_PACK(dlon, dlat);
        dlon = (int)(gridMBR.ur.lon - grid_service_region.ll.lon) >> 1;
        dlat = (int)(gridMBR.ur.lat - grid_service_region.ll.lat) >> 1;
#if SXM_USE_FIX
        sxm_fix_printable(dlon, &dlon, &frac);
        sxm_fix_printable(dlat, &dlat, &frac);
#endif
        *ur = (uint)SXM_GRID_REGION_PACK(dlon, dlat);
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * This function decodes packed phone number which read from the buffer
 *
 * \param[out] dst           decoded and formatted phone number
 * \param[in]  size          size of the \p dst
 *
 ******************************************************************************/
void sxm_grid_decode_phone(SXMByteBuff *pbuff, char *dst, size_t size) {
    byte phone[SXM_GRID_PHONE_BYTELEN];
    if (sxm_bytebuff_blob(pbuff, &phone[0], sizeof(phone)) == sizeof(phone)) {
        const uint area = (uint)(((phone[0] >> 4) * 100) + ((phone[0] & 0xF) * 10) + (phone[1] >> 4)),
                   exch = (uint)(((phone[1] & 0xF) * 100) + ((phone[2] >> 4) * 10) + (phone[2] & 0xF)),
                   num  = (uint)(((phone[3] >> 4) * 1000) + ((phone[3] & 0xF) * 100) + ((phone[4] >> 4) * 10) + (phone[4] & 0xF));
        if (!area && !exch && !num) {
            // Seems like empty phone detected
            dst[0] = '\0';
        }
        else {
            #ifdef _WIN32
            _snprintf
            #else
            snprintf
            #endif
                (dst, size, SXM_GRID_PHONE_FORMAT, area, exch, num);
        }
    }
    else {
        *dst = '\0';
    }
}

/***************************************************************************//**
 * This function encodes number which is represented by three components into
 * the provided buffer in binary form where each digit is represented by 4-bits.
 *
 * \param[out] pphone        encoded phone number, it must be 
 *                           #SXM_GRID_PHONE_BYTELEN bytes length
 * \param[in]  area          phone's area
 * \param[in]  exch          phone's exch
 * \param[in]  num           phone's num
 *
 ******************************************************************************/
void sxm_grid_encode_phone(byte *pphone, ushort area, ushort exch, ushort num) {
    pphone[0] = (byte)((((area / 100) & 0xF) << 4) | ((area % 100) / 10));
    pphone[1] = (byte)(((area % 10) << 4) | ((exch / 100) & 0xF));
    pphone[2] = (byte)((((exch % 100) / 10) << 4) | (exch % 10));
    pphone[3] = (byte)((((num / 1000) & 0xF) << 4) | ((num % 1000) / 100));
    pphone[4] = (byte)((((num % 100) / 10) << 4) | (num % 10));
}

/** @} */

/***************************************************************************//**
 * This function reads the service location coordinates
 *
 * \param[in] service        data service abstract instance
 * \param[out] loc           a pointer to an SXMPoint that will be filled up with
 *                           the location coordinates.
 * \param[in]  base          the information collection for a single grid square,
 *                           see \ref SXMGridBaseLine
 * \param[in]  id            station id identical to ID field in the baseline
 * 
 * \retval SXM_E_OK    if data has been extracted 
 * \retval SXM_E_NOENT gone past the end and not able to find an entry.
 * \retval SXM_E_INVAL id no longer in use
 * \retval SXM_E_PIPE  Something wrong with reading data from the DB
 *
 ******************************************************************************/
int sxm_grid_poi_read_location(SXMGridService *service, SXMPoint *loc,
                               const SXMGridBaseLine *base, ushort id)
{
    SXMResultCode rc;

    LOCK(service->service_mutex);
    DEBUG_API(": extracting coordinates for id = 0x%X", id);

    if ((uint)id <= base->dbentry[0]) {
        if ((id != 0) && (base->dbentry[id] != SXM_DB_INVALID_ENTRY)) {
            SXMByteBuff rpb, *prpb;

            prpb = sxm_bytebuff_init(&rpb, base->base, (int)base->size, (int)base->dbentry[id]);
            if (!prpb) {
                DEBUG_API(": failed to init buffer");
                rc = SXM_E_PIPE;
            }
            else {
#if SXM_USE_FIX
                loc->lon = (fix)sxm_bytebuff_g4(prpb);
                loc->lat = (fix)sxm_bytebuff_g4(prpb);
#else
                loc->lon = fix2float((fix)sxm_bytebuff_g4(prpb));
                loc->lat = fix2float((fix)sxm_bytebuff_g4(prpb));
#endif
                rc = SXM_E_OK;
            }
        }
        else {
            rc = SXM_E_INVAL;
        }
    }
    else {
        rc = SXM_E_NOENT;
    }
    UNLOCK(service->service_mutex);
    return rc;
}

/***************************************************************************//**
 * This function provides current station
 *
 ******************************************************************************/
int sxm_grid_poi_get_station(SXMGridService *service, uint *id,
                             void **ppRec, const PPOISpec **ppPpoiSpec,
                             byte *src_mask) {
    SXMGridServicePoiExtract *ppe = &service->PoiExtract;
    int rc = SXM_E_NOENT;
    *id = 0;
    *src_mask = 0;
    *ppRec = NULL;

    if (ppe->mask_src & SXM_POI_EXT_PPOI) {
        if (ppe->psuid_index_extract < *service->fav.count) {
            // Get the address of the fav location
            byte *favLoc = (byte*)service->fav.data +
                            service->fav.item_size * ppe->psuid_index_extract;
            *ppPpoiSpec = (const PPOISpec *)(void*)(favLoc + service->fav.ppoispec_offset);
            *id = (*ppPpoiSpec)->ppoi_id;
            *ppRec = favLoc + service->fav.data_offset;
            *src_mask = SXM_POI_EXT_PPOI;
            rc = SXM_E_OK;
        }
    }
    else if (ppe->mask_src & SXM_POI_EXT_EXTRACTION) {
        if (ppe->psuid_index_extract < service->sp) {
            *id = service->stations.ids[ppe->psuid_index_extract];
            *ppRec = service->stations.items +
                ppe->psuid_index_extract * service->stations.item_size;
            *src_mask = SXM_POI_EXT_EXTRACTION;
            rc = SXM_E_OK;
        }
    }
    else if (ppe->mask_src == SXM_POI_EXT_GRID) {
        const byte lonx =
            (byte)(ppe->lon_idx_start + ppe->region_index_extract % ppe->lon_idx_width);
        const byte laty =
            (byte)(ppe->lat_idx_start + ppe->region_index_extract / ppe->lon_idx_width);
        if (laty <= ppe->lat_idx_end) {
            *id = SXM_GRID_ID_PACK(SXM_GRID_REGION_PACK(lonx, laty), ppe->psuid_index_extract + 1U);
            *src_mask = SXM_POI_EXT_GRID;
            rc = SXM_E_OK;
        }
    }
    return rc;
}

/***************************************************************************//**
 * This function provides current station 
 *
 ******************************************************************************/
void sxm_grid_poi_next(SXMGridService *service, BOOL nextRegion,
                       SXMGridServicePoiExtract *PoiExtract, uint fav_count, uint sp)
{
    SXMGridServicePoiExtract * const ppe = PoiExtract;

    BOOL hasFallBack = FALSE;
    if (ppe->mask_src & ~SXM_POI_EXT_GRID) { // Something except GRID
        BOOL isValid = FALSE;
        if (ppe->mask_src & SXM_POI_EXT_PPOI) {
            ++ppe->psuid_index_extract;
            isValid = ppe->psuid_index_extract < fav_count;
            if (!isValid) {
                // Nothing to get from the PPOI list
                ppe->mask_src &= (byte)~SXM_POI_EXT_PPOI;
                ppe->region_index_extract = 0;
                ppe->psuid_index_extract = 0;
                nextRegion = FALSE;
                hasFallBack = TRUE;
                DEBUG_API(": leaving PPOI extraction");
            }
        }
        
        if ((ppe->mask_src & SXM_POI_EXT_EXTRACTION) && !isValid) {
            if (!hasFallBack) {
                ++ppe->psuid_index_extract;
            }
            if (ppe->psuid_index_extract >= sp) {
                // Nothing to get from the EXTRACTION list
                ppe->mask_src &= (byte)~SXM_POI_EXT_EXTRACTION;
                ppe->region_index_extract = 0;
                ppe->psuid_index_extract = 0;
                nextRegion = FALSE;
                hasFallBack = TRUE;
                DEBUG_API(": leaving EXTRACTION extraction");
            }
        }
    }
    if (ppe->mask_src == SXM_POI_EXT_GRID) {// Only grid left
        if (nextRegion == TRUE) {
            ++ppe->region_index_extract;
            ppe->psuid_index_extract = 0;
        }
        else if (!hasFallBack) {
            ++ppe->psuid_index_extract;
        }
    }
}

/***************************************************************************//**
 * Helper function to enhance c-like bsearch() function by performing additional
 * corner case checks which is usually is not done by stdlib implementation.
 *
 * \param[in] key this is the pointer to the object that serves as key for the
 *            search, type-casted as a void*
 * \param[in] base this is the pointer to the first object of the array where the
 *            search is performed, type-casted as a void*.
 * \param[in] nitems This is the number of elements in the array pointed by \p base
 * \param[in] size this is the size in bytes of each element in the array
 * \param[in] compare this is the function that compares two elements
 *
 * \return This function returns a pointer to an entry in the array that matches
 *         the search key. If key is not found, a \p NULL pointer is returned
 *
 ******************************************************************************/
static void* grid_service_bsearch(const void *key, const void *base, size_t nitems,
                                  size_t size, int (*compare)(const void*, const void*))
{
    return nitems ? bsearch(key, base, nitems, size, compare) : NULL;
}

/********************************************************************//** 
 * Initializes grid related part of the service.
 * -# Opens the c-file
 * -# Initializes common and service independent grid fields
 *
 * \param[in] service service instance
 * \param[out] out the c-file content memory provided by the caller
 * \param[in] out_size the size of the \p out in bytes
 *
 * \retval SXM_E_OK Success (file fully read)
 * \retval SXM_E_NOENT File has less data than required
 * \retval SXM_E_PIPE Failed to read the file
 * \retval SXM_E_NOMEM Memory related issued
 *                                                                      
 ************************************************************************/
int sxm_grid_init(SXMGridService *service, void **out, size_t out_size) {
    int rc;
    
    // Open the cycle file, make sure that goes okay
    service->cycle = sxm_cfile_open(&service->ds.service[0], SXM_GRID_CFILE_NAME);
    if (!service->cycle) {
        DEBUG_REQ("(No cycle file, failed to open or create)");
        rc = SXM_E_PIPE;
    }
    else {
        // Now load the data from the CFile into the service, and check the return code again
        rc = sxm_cfile_alloc_and_load(service->cycle, out_size, out);
        if (rc == -1) {
            if (*out != NULL) {
                sxe_free(*out);
                *out = NULL;
            }
            sxm_cfile_close(service->cycle, NULL);
            service->cycle = NULL;
            DEBUG_REQ("(OUT OF MEMORY)");
            rc = SXM_E_NOMEM;
        }
        else {
            // Expectations
            const size_t scount = (out_size + sizeof(SXMSector) - 1) / sizeof(SXMSector);
            rc = (scount == (size_t)rc) ? SXM_E_OK : SXM_E_NOENT;
            // Init remaining fields
            service->ext.state = SXM_GRID_EXT_STATE_IDLE;
            service->rfd.state = SXM_GRID_RFD_STATE_IDLE;
            service->stations.dirty_flag = FALSE;
            service->fav.dirty_flag = TRUE;
            service->reload_state = NULL;
            service->state = SXM_GRID_STATE_LOADED;
            service->ds.report = &grid_report_up;
            service->time_callback = &grid_timer_callback;
            DEBUG_REQ("(rc=%d)", rc);
        }
    }
    return rc;
}

/********************************************************************//** 
 * Cleans up resources allocated for c-file
 *
 * \param[in] service service instance
 * \param[in] data address of the c-file data
 *
 * \retval SXM_E_OK Success (file fully read)
 * \retval SXM_E_PIPE Failed I/O
 * \retval SXM_E_STATE Service not running
 *                                                                      
 ************************************************************************/
int sxm_grid_cycle_destroy(SXMGridService *service, const void* data) {
    int rc = SXM_E_OK;
    if (!service) {
        rc = SXM_E_STATE;
    }
    else if (service->cycle) {
        if (service->ds.auto_save_on_stop == TRUE) {
            rc = sxm_cfile_shutdown(service->cycle, service->root);
            if (rc != SXM_E_OK) {
                PLOG_PKT("Failed to shutdown CFile sections");
            }
            else {
                DEBUG_PKT("CFile write; secured all sections");
            }
        }
        else {
            // close without committing (writing) to the file system
            sxm_cfile_close(service->cycle, NULL);
        }
        service->cycle = NULL;
        service->root = NULL;

        if (data) {
            uint i;
            for (i = 0; i < ARRAY_SIZE(service->bptr); ++i) {
                if (service->bptr[i]) {
                    if (service->bptr[i]->dbentry) {
                        sxe_free(service->bptr[i]->dbentry);
                        service->bptr[i]->dbentry = NULL;
                    }
                }
            }
            sxe_free((void*)data);
        }
    }
    return rc;
}

/********************************************************************//** 
 * Collection invalidation per time change
 *                                                                      
 * \param[in] service service instance
 * 
 * \retval SXM_E_OK time is valid and data processed based on it
 * \retval SXM_E_NOENT time is not available
 *                                                                      
 ************************************************************************/
static int grid_process_timer(SXMGridService *service)
{
    int rc = SXM_E_NOENT;
    const uint time_now = sxm_cfile_time_now();
    if (time_now != 0U) {
        DEBUG_PKT(": time is %u", time_now);
        service->check_timestamp(service->ds.dsuser, time_now);

        service->state |= SXM_GRID_STATE_TIMER;
        if (service->state == SXM_GRID_STATE_ALL) {
            sxm_grid_reload_state(service); 
        }

        if (service->ext.flags & SXM_SDK_FLAG_NOTIFY) {
            LOCK(service->ext.mutex);
            service->ext.flags |= SXM_SDK_FLAG_INUSE;
            (*service->ext.notification)(&service->ext, service->ext.usercx);
            service->ext.flags &= (uint)(~(SXM_SDK_FLAG_INUSE | SXM_SDK_FLAG_NOTIFY));
            UNLOCK(service->ext.mutex);
        }
        rc = SXM_E_OK;
    }
    return rc;
}

/********************************************************************//** 
 * This function is called to handle timer event which is fired by
 * LLI only in case if time has come. Thus, this function should be used
 * to invalidate data in the collection are only if the time available.
 *                                                                      
 * \param[in] pArg service instance
 ************************************************************************/
static void grid_timer_callback(void *pArg)
{
    SXMGridService *service = (SXMGridService*)pArg;
    if (!(service->state & SXM_GRID_STATE_TIMER)) {
        int rc;
        rc = grid_process_timer(service);
        if (rc == SXM_E_OK) {
            // as soon as we have time available we start new timer to track data expiration
            rc = sxm_lli_timer(service->lli, service->time_callback, SXM_GRID_DATA_EXPIRATION_TIMER);
            if (rc != SXM_E_OK) {
                DEBUG_REQ(": failed to start new timer, rc=%d");
            }
        }
        else {
            DEBUG_REQ(": cannot process the time, wait for the next callback, rc=%d",
                rc);
        }
    }
    else {
        /* Do the callback if it exists */
        if (service->data_exp_callback != NULL) {
            service->data_exp_callback(service->ds.dsuser);
        }
    }
    return;
}

/********************************************************************//** 
 * This function calls the defined service level callback function.
 * The module-level callback is described in the "Error Detection and Reporting"
 * section of the SX-9845-0340 Introduction Design Guide. 
 *                                                                      
 * \param[in] type the type to be passed to the callback                
 *       The most common callback type is #SXM_CALLBACK_SUBS
 * \param[in] param the parameter to be passed to the callback
 *       If the type is #SXM_CALLBACK_SUBS then the most common param 
 *       would be #SXM_SUBS_FULL
 * \param[in] dsuser    Grid service instance
 *                                                                      
 ************************************************************************/
static void grid_report_up(int type, int param, ptr dsuser) {
    SXMGridService *service = (SXMGridService *)dsuser;

    CALLB_API("(type=%d, param=%d, dsuser=%p)", type, param, dsuser);

    //  now would be a good time to check the timestamp
    if (type == SXM_CALLBACK_SUBS) {
        int rc;
        if (service->state & SXM_GRID_STATE_TIMER) {
            // start timer to track data expiration
            rc = sxm_lli_timer(service->lli, service->time_callback, SXM_GRID_DATA_EXPIRATION_TIMER);
            DEBUG_REQ("(repeat)");
        }
        else {
            rc = grid_process_timer(service);
            if (rc != SXM_E_OK) {
                DEBUG_PKT(": time is not defined yet, start timer to wait for it");
                rc = sxm_lli_timer(service->lli, service->time_callback, 1U);
            }
            else {
                //start timer to track data expiration
                rc = sxm_lli_timer(service->lli, service->time_callback, SXM_GRID_DATA_EXPIRATION_TIMER);
            }
        }
        if (rc != SXM_E_OK) {
            DEBUG_PKT(": failed to set-up timers, rc=%d", rc);
        }
    }

    service->callback(type, param);
}

/********************************************************************//**                                                                    *
 * Parses RFD update metadata from an incoming packet and updates
 * the corresponding structure in the service.
 *
 * \param[in] service service instance
 * \param[in] pkt the SXM BitBuff containing packet data.
 * \param[in] fp a pointer to an packet
 *
 ************************************************************************/
void sxm_grid_rfd_handle_metadata(SXMGridService *service, SXMBitBuff *pkt, SXMPacket *fp)
{
    ENTER_PKT("()");

    switch (0) { default: {
        SXMRfdMetadata rfd;
        uint upd_version, old_version;
        int rc;

        if (service->rfd.state == SXM_GRID_RFD_STATE_APPLYING) {
            DEBUG_PKT(": skip the metadata since the DB is being applied");
            break;
        }

        // parse RFD
        if (sxm_rfd_parse(pkt, &rfd)) {
            DEBUG_PKT(": error during metadata parsing, skipping packet.");
            break;
        }

        rc = service->rfd.qualify(service->ds.dsuser, &rfd, &old_version, &upd_version);
        if (rc != SXM_E_OK) {
            DEBUG_PKT(": failed to qualify the RFD, rc=%d", rc);
            break;
        }

        if (service->db_version == (int)upd_version) {
            DEBUG_PKT(": RFD update %s from %d to %d is not needed for version %d",
                rfd.name, old_version, upd_version, service->db_version);
            break;
        }

        // assigning packet CRC to metadata for future use
        rfd.crc = fp->crc;

        // check if RFD update is collecting
        if (service->rfd.state == SXM_GRID_RFD_STATE_COLLECTING) {
            // checking if this metadata is processing right now
            int i;
            SXMRfdMetadata meta;

            for (i = 0; i < SXM_RFD_MAX_COLLECTION; ++i) {
                if (grid_lli_rfd_status(service, i, &meta) == SXM_E_OK) {
                    if (meta.crc == rfd.crc) {
                        DEBUG_PKT(": already collecting update %s, "
                            "skip metadata", meta.name);
                    }
                    else {
                        DEBUG_PKT(": cleaning service slot id %d, name %s, ",
                            i, meta.name);
                        (void)grid_lli_rfd_remove(service, i);
                        service->rfd.state = SXM_GRID_RFD_STATE_IDLE;
                    }
                    break;
                }
            }

            // if update is collecting, skip this metadata
            if (service->rfd.state == SXM_GRID_RFD_STATE_COLLECTING) {
                break;
            }
        }

        // let's start collecting RFD data if required
        DEBUG_PKT(": starting RFD %s", rfd.name);
        rc = grid_lli_rfd_start(service, &rfd);
        if (rc != SXM_E_OK) {
            DEBUG_PKT(": failed to start update (%d), waiting for next packet.", rc);
            break;
        }

        service->rfd.version = upd_version;
        service->rfd.state = SXM_GRID_RFD_STATE_COLLECTING;
    }}

    LEAVE_PKT("()");
}

/***************************************************************************//**
 * Helper function to perform request and PPIO notifications.
 *
 * \param[in] service service instance
 *
 ******************************************************************************/
void sxm_grid_notify_new_data(SXMGridService *service) {
    if ((service->stations.dirty_flag == TRUE) && (service->stations.au_flag == TRUE) &&
        (service->ext.state == SXM_GRID_EXT_STATE_ACTIVE))
    {
        service->stations.dirty_flag = FALSE;
        service->stations.au_flag = FALSE;

        LOCK(service->ext.mutex);

        service->ext.flags |= SXM_SDK_FLAG_INUSE;
        (*service->ext.notification)(&service->ext, service->ext.usercx);
        service->ext.flags &= (uint)~SXM_SDK_FLAG_INUSE;

        UNLOCK(service->ext.mutex);

        DEBUG_PKT("(notified)");
    }
    if ((service->fav.dirty_flag == TRUE) &&
        service->fav.active && service->fav.notification)
    {
        service->fav.dirty_flag = FALSE;
        LOCK(service->fav.mutex);

        service->ext.flags |= SXM_SDK_FLAG_INUSE;
        (*service->fav.notification)(service->fav.usercx,
            service->fav.user_flags);
        service->ext.flags &= (uint)~SXM_SDK_FLAG_INUSE;
        service->fav.user_flags = 0;

        UNLOCK(service->fav.mutex);

        DEBUG_PKT("(PPOI notified)");
    }
}

/***************************************************************************//**
 * Helper function to perform parking location id comparison over station ids.
 *
 * \param[in] p1 the first object to compare
 * \param[in] p2 the second object to compare
 * \return returns the difference between p1 and p2. Value can be positive or
 *                 negative.
 *
 ******************************************************************************/
static int grid_ids_compare_callback(const void *p1, const void *p2) {
    return (int)(*(const uint*)p1 - *(const uint*)p2);
}

/***************************************************************************//**
 * Helper function to let caller know the position of the location/station
 * with particular id.
 *
 * \param[in] service service instance
 * \param[in] index station unique id
 * \param[in] ix region index among SXM_GRID_NEEDED_REGIONS supported for collection
 * \param[out] out positions of the station within within the collection
 *
 * \return The address of the corresponding record or \p NULL in case if the record
 *         has not been found.
 ******************************************************************************/
void* sxm_grid_get_poi_data(SXMGridService *service, uint index, int ix)
{
    void *ret = NULL;
    const int nitems = service->hstarts[ix + 1] - service->hstarts[ix];
    if (nitems > 0) {
        const uint *pos = (const uint*)
            grid_service_bsearch(&index, &service->stations.ids[service->hstarts[ix]],
                (size_t)nitems, sizeof(index), grid_ids_compare_callback);
        if (pos) {
            // Calculate address of the record
            ret = service->stations.items +
                (uint)(pos - &service->stations.ids[0]) * service->stations.item_size;
        }
    }
    return ret;
}

/***************************************************************************//**
 * This function detects how many locations/stations lay inside the request mbr
 * and populate stations that fall with the extraction area specified by the MBR.
 *
 * Internally function checks all loaded regions and count all stations which
 * lay inside the request mbr. In order to do not waste CPU the counting process
 * stops once the number of found stations equal or greater than the max allowed.
 * .
 * \param[in] service service instance
 * \param[in] max_allowed number of max supported stations
 *
 * \note Assumption that the regions the application is interested in are asc sorted
 *       and taking into account that station id within region are being collected
 *       in asc order we can say that all stations inside the stations collection
 *       are sorted in asc order since unique id is a combination of region in
 *       16 MSB and station id inside the region in 16 LSB.
 *
 * \retval SXM_E_OK The region provides not more that allowed stations
 * \retval SXM_E_NOMEM There is no space to accommodate all stations from the region
 *
 ********************************************************************************/
static int grid_build_stations(SXMGridService *service, uint max_allowed) {
    uint ix;
    int rc = SXM_E_OK;

    ENTER_REQ("()");

    service->sp = 0;
    memset(service->stations.ids, 0, sizeof(*service->stations.ids) * service->stations.capacity);
    memset(service->ploc, 0, service->locsize);
    service->stations.dirty_flag = FALSE;
    service->stations.au_flag = FALSE;

    // process each region
    for (ix = 0; (ix < ARRAY_SIZE(service->bptr)) && (service->sp <= max_allowed) && (rc == SXM_E_OK); ++ix)
    {
        const SXMGridBaseLine *b = service->bptr[ix];
        service->hstarts[ix] = service->sp;

        if (b->dbentry) {
            ushort psuid;
            for (psuid = (ushort)service->stations.min_id;
                (psuid <= b->dbentry[0]) && (service->sp <= max_allowed) && (rc == SXM_E_OK);
                ++psuid) {
                if (b->dbentry[psuid] != SXM_DB_INVALID_ENTRY) {
                    SXMByteBuff rpb, *prpb;
                    SXMPoint p;

                    prpb = sxm_bytebuff_init(&rpb, b->base, (int)b->size, (int)b->dbentry[psuid]);
                    if (!prpb) {
                        DEBUG_EXT(": failed to init buffer station %u", psuid);
                        continue;
                    }
#if SXM_USE_FIX
                    p.lon = (fix)sxm_bytebuff_g4(prpb);
                    p.lat = (fix)sxm_bytebuff_g4(prpb);
#else
                    p.lon = fix2float((fix)sxm_bytebuff_g4(prpb));
                    p.lat = fix2float((fix)sxm_bytebuff_g4(prpb));
#endif
                    if (sxm_bytebuff_state(prpb) != SXM_BYTEBUFF_STATE_OK) {
                        // Just inform the log, but we can do almost nothing here
                        // since this is unlikely possible case
                        DEBUG_EXT(": unexpected buffer reading error REGION=%d, PSUID=%u",
                            b->region, psuid);
                    }
                    // populate only if within MBR
                    else if (sxm_mbr_inside(&service->ext.mbr, &p)) {
                        if (service->sp < max_allowed) {
                            service->stations.ids[service->sp] = SXM_GRID_ID_PACK(b->region, psuid);
                            service->ploc[service->sp].p = p;
                            service->ploc[service->sp].distance =
                                sxm_point_distance(&service->ext.center, &p);
                            service->sp++;
                        }
                        else {
                            // Here we've got one more station which out of the
                            // storage capacity, thus, stop this routine
                            rc = SXM_E_NOMEM;
                        }
                    }
                }
            }
        }
    }

    if (rc == SXM_E_OK) {
        service->hstarts[ix] = service->sp;
        LEAVE_REQ("(ix=%d start=%d)", ix, service->sp);
    }
    else {
        LEAVE_REQ(": more stations than allowed");
    }
    return rc;
}

/************************************************************************
 * The function reloads the dynamic and static content used by the service.
 * Dynamic from the AUs (broadcast) and static from the service's database.
 *
 * \param[in] service service instance
 *
 ************************************************************************/
void sxm_grid_reload_state(SXMGridService *service) {
    SXMGridExtraction *ext = &service->ext;
    SXMMBR origin = service->ext.orig_mbr;
    SXMMBR gridMbr, adjMbr;

    LOCK(service->service_mutex);
    ENTER_REQ("()");

    grid_reload(service);
    if (service->reload_state) {
        service->reload_state(service->ds.dsuser);
    }

    grid_needed_corner_regions(service, NULL, NULL, &gridMbr);
    if (sxm_mbr_intersects_get(&service->ext.mbr, &gridMbr, &adjMbr)) {
        service->ext.mbr = adjMbr;
    }

    while (grid_build_stations(service, service->stations.capacity) != SXM_E_OK) {
        sxm_mbr_reduce(&origin, SXM_GRID_DEFAULT_MBR_SHRINK_FACTOR,
            &ext->center, &ext->mbr);
        origin = ext->mbr;
    }
    UNLOCK(service->service_mutex);
    service->replay(service->ds.dsuser);

    LEAVE_REQ("()");
}

/********************************************************************//**
 * The function uses the application's request parameters to establish
 * collection of dynamic data from the broadcast.
 *
 * \param[in] service service instance
 * \param[in] ext collection/extraction handler
 *
 ************************************************************************/
static int grid_user_request(SXMGridService *service, SXMGridExtraction *ext) {
    ENTER_REQ("()");

    grid_needed(service);
    ext->state = SXM_GRID_EXT_STATE_ACTIVE;
    ext->flags = SXM_SDK_FLAG_NOTIFY;

    service->state |= SXM_GRID_STATE_REQUEST;
    if ((service->state & SXM_GRID_STATE_COMPLETE) == SXM_GRID_STATE_COMPLETE) {
        sxm_grid_reload_state(service); 
    }
    LEAVE_REQ("(OK)");
    return SXM_E_OK;
}

/*********************************************************************//**
 * The function sets up the request
 *
 * \param[in] pMbr        a pointer to the \ref SXMMBR structure in application 
 *                        memory defining the area over which service data 
 *                        is to be collected
 * \param[in] pLocation   NULL, OR a pointer to the \ref SXMPoint structure in 
 *                        application memory defining the sorting point for 
 *                        service data collection. If NULL is used then
 *                        the center point of the supplied SXMMBR is used.
 * \param[in] pNotify     function pointer of notification routine
 * \param[in] pUsercx     user context value supplied on the callback
 * \param[out] pHandle    pointer to returned request handle pointer
 *
 * \retval SXM_E_OK     %Request has been created.
 * \retval SXM_E_BUSY   Collection request is already active.
 * \retval SXM_E_FAULT  NULL parameter
 * \retval SXM_E_INVAL  Invalid parameter.
 * \retval SXM_E_STATE  service not started
 *
 ************************************************************************/
int sxm_grid_request(SXMGridService *service, const SXMMBR *pMbr,
                     const SXMPoint *pLocation, SXMGridExtCallback_t pNotify,
                     void *pUsercx, SXMGridExtraction **pHandle) {

    int rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (!pMbr || !pHandle || !pNotify) {
        rc = SXM_E_FAULT;
        ERROR_API("null argument, pMbr: %p, pHandle: %p,"
            "pNotify:%p", pMbr, pHandle, pNotify);
    }
    else if ((service->ext.state != SXM_GRID_EXT_STATE_IDLE) ||
             (service->PoiExtract.mask_src_origin != SXM_POI_EXT_INACTIVE)) {
        rc = SXM_E_BUSY;
        ERROR_API("collection is active");
    }
    else if (!sxm_mbr_valid(pMbr)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid MBR");
    }
    else {
        rc = SXM_E_OK;
        if (pLocation) {
            if (!sxm_point_valid(pLocation) || !sxm_mbr_inside(pMbr, pLocation)) {
                rc = SXM_E_INVAL;
                ERROR_API("invalid location point");
            }
            else {
                service->ext.center = *pLocation;
            }
        }
        else {
            sxm_mbr_center(pMbr, &service->ext.center);
        }

        if (rc == SXM_E_OK) {
            service->ext.mbr = *pMbr;
            service->ext.orig_mbr = *pMbr;
            service->ext.notification = pNotify;
            service->ext.usercx = pUsercx;
            sxm_mutex_init(&service->ext.mutex);

            rc = grid_user_request(service, &service->ext);
            if (rc == SXM_E_OK) {
                *pHandle = &service->ext;
            }
            else {
                sxm_mutex_destroy(&service->ext.mutex);
                memset(&service->ext, 0, sizeof(service->ext));
            }
        }
    }
    return rc;
}

/********************************************************************//**
 * The function uses the application's to modify parameters of the collection
 * of dynamic data from the broadcast.
 *
 * \param[in] service service instance
 * \param[in] mbr new collection area, optional
 * \param[in] center new sorting point, optional
 *
 ************************************************************************/
static int grid_user_modify(SXMGridService *service, const SXMMBR *mbr,
                            const SXMPoint *center) {
    int rc;
    const SXMPoint *updated_center = center ? center : &service->ext.center;
    const SXMMBR *updated_mbr = mbr ? mbr : &service->ext.mbr;

    ENTER_REQ("()");

    if (!sxm_mbr_inside(updated_mbr, updated_center)) {
        rc = SXM_E_INVAL;
    }
    else {
        if (center) {
            service->ext.center = *updated_center;
        }
        if (mbr) {
            service->ext.mbr = *updated_mbr;
            service->ext.orig_mbr = *updated_mbr;
        }
        rc = grid_user_request(service, &service->ext);
    }

    LEAVE_REQ("()");
    return rc;
}

/*********************************************************************//**
 * The routine will update the collection \ref SXMMBR 
 * (MBR- Map Bounding Rectangle) and optionally the \ref SXMPoint sort point.
 *
 * \note  If a new collection point value is not supplied, then the 
 * old value is retained (it does not reset to the center of the new 
 * MBR).  If a new MBR is not supplied, the old MBR is used.  
 * This is to support the most usual case of modifying a request, 
 * when a vehicle has moved sufficiently far from its previous 
 * position that a new collection is warranted, but the vehicle position 
 * is still close to the point of the last extraction.
 *
 * \param[out] pHandle    pointer to returned request handle pointer
 * \param[in] pMbr        a pointer to the \ref SXMMBR structure in application 
 *                        memory defining the area over which service's data 
 *                        is to be collected. If pMbr is not specified, 
 *                        the original MBR is used.
 * \param[in] pLocation   NULL, OR a pointer to the SXMPoint structure in 
 *                        application memory defining the sorting point for 
 *                        service's data collection. If NULL is used then
 *                        the center point of the supplied SXMMBR is used.
 * \param[in] min_id min possible station id
 *
 * \retval  SXM_E_OK       modify request successful
 * \retval  SXM_E_BUSY     collection request already active
 * \retval  SXM_E_INVAL    invalid parameter
 * \retval  SXM_E_FAULT    NULL pointer argument
 * \retval  SXM_E_STATE    service not started
 *
 *   sxm_ev_modify truth table
 *   |pHandle|   pMbr   | pLocation   |                      Description                               |
 *   |:-----:|:--------:|:-----------:|:--------------------------------------------------------------:|
 *   | NULL  |     X    |     X       | SXM_E_FAULT is returned, a valid "pHandle" must be provided.   |
 *   | Valid |   NULL   |    NULL     | SXM_E_FAULT is returned, must have one valid pMbr or pLocation |
 *   | Valid | New pMbr |    NULL     | Will use the "new" pMbr with the previous pLocation.           |
 *   | Valid |   NULL   |New pLocation| Will use the previous pMbr with the "new" pLocation.           |
 *
 ************************************************************************/
int sxm_grid_modify(SXMGridService *service, SXMGridExtraction *pHandle,
                    const SXMMBR *pMbr, const SXMPoint *pLocation) {
    int rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service inactive");
    }
    else if (!pHandle) {
        rc = SXM_E_FAULT;
        DEBUG_API("NULL argument(s) (pHandle=%p) ", pHandle);	
    }
    else if (!pMbr && !pLocation) {
        rc = SXM_E_FAULT;
        ERROR_API("MBR and LOC NULL (pMbr=%p, pLocation=%p) ", pMbr, pLocation);
    }
    else if ((pHandle != &service->ext) ||
             (service->ext.state == SXM_GRID_EXT_STATE_IDLE)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid handle (pHandle=%p) ", pHandle);
    }
    else if ((service->ext.state == SXM_GRID_EXT_STATE_EXTRACTING) ||
             (service->PoiExtract.mask_src_origin != SXM_POI_EXT_INACTIVE)) {
        rc = SXM_E_BUSY;
        ERROR_API("extraction is active");
    }
    else if (pMbr && !sxm_mbr_valid(pMbr)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid MBR");
    }
    else if (pLocation && !sxm_point_valid(pLocation)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid location point");
    }
    else {
        rc = grid_user_modify(service, pMbr, pLocation);
    }

    return rc;
}

/*********************************************************************//**
 *
 * The routine will remove a collection MBR, and all the
 * data associated with it. Once the request has been removed, the handle
 * should not be used on any subsequent calls.
 *
 * \param[in] service     service instance
 * \param[in] pHandle     request handle
 *
 * \retval  SXM_E_OK      collection request removed
 * \retval  SXM_E_BUSY    extraction in progress
 * \retval  SXM_E_FAULT   NULL pointer argument
 * \retval  SXM_E_INVAL   invalid parameter
 * \retval  SXM_E_STATE   service not started
 *
 ************************************************************************/
int sxm_grid_remove(SXMGridService *service, SXMGridExtraction *pHandle) {
    int rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (!pHandle) {
        rc = SXM_E_FAULT;
        ERROR_API("NULL argument, pHandle:%p", pHandle);
    }
    else if ((pHandle != &service->ext) ||
             (service->ext.state == SXM_GRID_EXT_STATE_IDLE)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid handle, (pHandle=%p)", pHandle);
    }
    else if ((service->ext.state == SXM_GRID_EXT_STATE_EXTRACTING) ||
             (service->PoiExtract.mask_src_origin != SXM_POI_EXT_INACTIVE)) {
        rc = SXM_E_BUSY;
        ERROR_API("extraction is active");
    }
    else if (sxm_lli_on_thread(service->lli) ||
            (service->ext.flags & SXM_SDK_FLAG_INUSE)) {
        ERROR_API("ERROR: Attempt to remove request from the servcie callback.");
        rc = SXM_E_BUSY;
    }
    else {
        sxm_grid_delete_request(service);
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * This function removes any existing collection requests
 *
 * \param[in] service service instance
 *
 ******************************************************************************/
void sxm_grid_delete_request(SXMGridService *service) {
    if (service->ext.state != SXM_GRID_EXT_STATE_IDLE) {
        LOCK(service->ext.mutex);
        service->ext.state = SXM_GRID_EXT_STATE_IDLE;
        UNLOCK(service->ext.mutex);
        pthread_mutex_destroy(&service->ext.mutex);
        memset(&service->ext, 0, sizeof(service->ext));
    }
}

/*********************************************************************//**
 * The routine starts an extraction. The will lock the application's copy
 * of the fuel data for extraction and will
 * not update the application's copy until the sxm_fuel_end is called.
 * The sxm_gfuel_begin function may also optionally change the sorting
 * location point. If a location reference point is specified, it must
 * lie within the current MBR being collected.  If the service is already
 * active with a previous request then additional calls to begin will
 * return a 'busy' error.

 *\note The notification routine specified with the collection request
 * only informs the application that new fuel data is available; it does
 * not process the data in any way.  The extraction request returns a list
 * of Fuel Stations sorted by distance from a Point, one at a time one
 * fuel station record per extraction request.  This allows the application
 * to extract as many stations as desired and then end the extraction.  The
 * extraction Point value may be specified on the extraction request.  If no
 * point is specified, extractions will proceed from the location point of
 * the current MBR in use. The application will the APIs in the order of
 * begin, extract and end to complete the service data information.
 *
 * \param[in] service     service instance
 * \param[in] pHandle     request handle
 * \param[in] pLocation   NULL, OR a pointer to the SXMPoint structure in
 *                        application memory defining the sorting point for
 *                        fuel data collection.
 *
 * \retval  SXM_E_OK      extraction begun
 * \retval  SXM_E_BUSY    extraction already active
 * \retval  SXM_E_FAULT   NULL argument
 * \retval  SXM_E_INVAL   invalid parameter
 * \retval  SXM_E_STATE   service not started
 *
 ************************************************************************/
int sxm_grid_begin(SXMGridService *service, SXMGridExtraction *pHandle,
                   const SXMPoint *pLocation) {
    int rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (!pHandle) {
        rc = SXM_E_FAULT;
        ERROR_API("NULL argument, pHandle:%p", pHandle);
    }
    else if ((pHandle != &service->ext) ||
             (service->ext.state == SXM_GRID_EXT_STATE_IDLE)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid handle, (pHandle=%p)", pHandle);
    }
    else if (service->ext.state == SXM_GRID_EXT_STATE_EXTRACTING) {
        rc = SXM_E_BUSY;
        ERROR_API("extraction is active");
    }
    //  the extraction center must lie within the MBR being collected
    else if (pLocation && !sxm_mbr_inside(&service->ext.mbr, pLocation)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid MBR location");
    }
    else if (!sxm_mbr_inside(&service->ext.mbr, &service->ext.center)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid MBR location");
    }
    else if ((service->state & SXM_GRID_STATE_COMPLETE) != SXM_GRID_STATE_COMPLETE) {
        rc = SXM_E_NOENT;
    }
    else {
        grid_heap_rebuild(service, pLocation);
        service->ext.state = SXM_GRID_EXT_STATE_EXTRACTING;
        rc = SXM_E_OK;
    }
    return rc;
}

/*********************************************************************//**
 *
 * The extraction routine terminates the extraction, and unlocks the data
 * for update.
 *
 * \param[in]  service  service instance
 * \param[in]  pHandle  request handle
 *
 * \retval  SXM_E_OK       extraction ended
 * \retval  SXM_E_FAULT    NULL argument
 * \retval  SXM_E_INVAL    invalid parameter
 * \retval  SXM_E_STATE    service or extraction not started
 *
 ************************************************************************/
int sxm_grid_end(SXMGridService *service, SXMGridExtraction *pHandle) {
    int rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (!pHandle) {
        rc = SXM_E_FAULT;
        ERROR_API("NULL argument pHandle: %p", pHandle);
    }
    else if ((pHandle != &service->ext) ||
             (service->ext.state == SXM_GRID_EXT_STATE_IDLE)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid handle, (pHandle=%p)", pHandle);
    }
    else if (service->ext.state != SXM_GRID_EXT_STATE_EXTRACTING) {
        rc = SXM_E_STATE;
        ERROR_API("extraction is inactive");
    }
    else if ((service->state & SXM_GRID_STATE_COMPLETE) != SXM_GRID_STATE_COMPLETE) {
        ERROR_API("(not ready)");
        rc = SXM_E_NOENT;
    }
    else {
        service->ext.state = SXM_GRID_EXT_STATE_ACTIVE;
        rc = SXM_E_OK;
    }
    return rc;
}

/*********************************************************************//**
 *
 * The sxm_gfuel_extract_station function will retrieve the individual
 * Fuel stations in distance-sorted order from the current location point.
 * Only one station is returned per call.
 *
 * \param[in]  service   service instance
 * \param[in]  pHandle   request handle
 * \param[out] pLocation a pointer to the location structure in the application's
 *                       memory where the service's specific station data
 *                       is placed.
 * \param[in] locSize    size of the memory addressed by \p pLocation in bytes
 *
 * \retval  SXM_E_OK      location service object extracted
 * \retval  SXM_E_NOENT   end of list
 * \retval  SXM_E_FAULT   NULL pointer argument
 * \retval  SXM_E_INVAL   invalid parameter
 * \retval  SXM_E_STATE   service or extraction not active
 * \retval  SXM_E_PIPE    Failed to access DB
 *
 ************************************************************************/
int sxm_grid_extract(SXMGridService *service, SXMGridExtraction *pHandle,
                    void *pLocation, size_t locSize) {
    int rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (!pHandle || !pLocation || !locSize) {
        rc = SXM_E_FAULT;
        ERROR_API("NULL argument pLocation: %p, pHandle: %p, locSize: %u",
            pLocation, pHandle, locSize);
    }
    else if ((pHandle != &service->ext) ||
             (service->ext.state == SXM_GRID_EXT_STATE_IDLE)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid handle, (pHandle=%p)", pHandle);
    }
    else if (service->ext.state != SXM_GRID_EXT_STATE_EXTRACTING) {
        rc = SXM_E_STATE;
        ERROR_API("extraction is inactive");
    }
    else {
        const short res = grid_heap_extract(service);
        if (res == -1) {
            rc = SXM_E_NOENT;
            DEBUG_API("end of list");
        }
        else {
            const uint rec_id = service->stations.ids[res];
            const int hix = sxm_grid_baseix(service, SXM_GRID_UNPACK_REGION(rec_id));
            if (hix == SXM_GRID_REGION_INVAL_IDX) {
                rc = SXM_E_NOENT;
                DEBUG_API("Region %d is not loaded", SXM_GRID_UNPACK_REGION(rec_id));
            }
            else {
                memset(pLocation, 0, locSize);
                DEBUG_EXT("Extracting index %d. Region %d uid %d. Heap Index %d",
                    res, SXM_GRID_UNPACK_REGION(rec_id), SXM_GRID_UNPACK_UID(rec_id), hix);
                LOCK(service->service_mutex);
                rc = service->extract(service->ds.dsuser, pLocation, res, rec_id, hix);
                UNLOCK(service->service_mutex);
            }
        }
    }
    LEAVE_API("(%d)", rc);
    return rc;
}

/***************************************************************************//**
 * This function start grid-based poi extraction 
 *
 ******************************************************************************/
static int grid_user_poi_begin(SXMGridService *service, const SXMMBR *mbr,
                               const ushort service_mask,
                               SXMGridServicePoiExtract *PoiExtract) {
    int rc = SXM_E_OK; 
    uint ll, ur;
    SXMGridServicePoiExtract *ppe = PoiExtract;

    // Initialize the state. Remaining part of the function will find data which
    // can be extracted based on application request and move extraction into the
    // proper state if needed.
    ppe->mask_src_origin = SXM_POI_EXT_INACTIVE;
    if (mbr) {
        BITSET(ppe->mask, POI_EXT_HAS_VALID_MBR_BIT);
        ppe->mbr = *mbr;
    }
    else {
        ppe->mask = 0; // No special conditions
    }

    if (service_mask == SXM_LOCATION_ALL) {
        // No specific mask set, thus, the application wants to see all
        // stations. However, the mbr can be provided and the service needs to
        // take it into account even for that case
        if (mbr) {
            DEBUG_API(": whole GRID area requested, but with MBR filtration");

            rc = grid_mbr_corner_regions(mbr, &ll, &ur);
            if (rc == SXM_E_OK) {
                ppe->mask_src_origin |= SXM_POI_EXT_GRID;

                DEBUG_API(": ll = %08x, ur = %08x", ll, ur);
                ppe->lat_idx_start = SXM_GRID_REGION_LAT_UNPACK(ll);
                ppe->lat_idx_end = SXM_GRID_REGION_LAT_UNPACK(ur);
                ppe->lon_idx_start = SXM_GRID_REGION_LON_UNPACK(ll);
                ppe->lon_idx_width = (byte)(SXM_GRID_REGION_LON_UNPACK(ur) -
                                     ppe->lon_idx_start + 1);
            }
        }
        else {
            DEBUG_API(": whole GRID area requested for POI extraction");
            memset(&ppe->mbr, 0, sizeof(ppe->mbr));
            ppe->mask_src_origin |= SXM_POI_EXT_GRID;
            BITCLR(ppe->mask, POI_EXT_HAS_VALID_MBR_BIT);
            BITSET(ppe->mask, POI_EXT_ALL_REGION_BIT);

            ppe->lat_idx_start = 0;
            ppe->lat_idx_end = (1 << SXM_GRID_REGION_LAT_BITSIZE) - 1;
            ppe->lon_idx_start = 0;
            ppe->lon_idx_width = (1 << SXM_GRID_REGION_LON_BITSIZE);
            rc = SXM_E_OK;
        }
    }
    else {
        if (service_mask & (SXM_LOCATION_IS_IN_COLLECTION |
                            SXM_LOCATION_IS_IN_EXTRACTION)) {
            // Application asks about the grid-like extraction. Here, we
            // need to see if the collection/request has been set.
            if (service->ext.state == SXM_GRID_EXT_STATE_IDLE) {
                DEBUG_API(": no active collection, no regions to look at for MBR");
                rc = SXM_E_NOENT;
            }
            // At least the request has been set. Since the collection set the
            // fact of having EXTRACTED mask doesn't heart anyone. The code
            // should look through the grid's regions and mark those stations
            // which belong to the extraction set by corresponding flag.
            else if (service_mask & SXM_LOCATION_IS_IN_COLLECTION) {
                if (mbr) {
                    SXMMBR collectionMbr, overlapMbr;

                    DEBUG_API(": collection selected, mbr filtration");

                    // In order to reduce set of regions for iteration the
                    // intersection between collection regions mbr and requested
                    // mbr can be used.
                    grid_needed_corner_regions(service, NULL, NULL, &collectionMbr);
                    if (!sxm_mbr_intersects_get(mbr, &collectionMbr, &overlapMbr)) {
                        // there is no intersection, thus no regions to iterate
                       DEBUG_API(": there is no overlapped region for both "
                                  "collection and requested mbrs");
                        rc = SXM_E_NOENT;
                    }
                    else {
                        rc = grid_mbr_corner_regions(&overlapMbr, &ll, &ur);
                        if (rc == SXM_E_OK) {
                            ppe->mbr = overlapMbr;
                            ppe->mask_src_origin |= SXM_POI_EXT_GRID;

                            DEBUG_API(": ll = %08x, ur = %08x", ll, ur);
                            ppe->lat_idx_start = SXM_GRID_REGION_LAT_UNPACK(ll);
                            ppe->lat_idx_end = SXM_GRID_REGION_LAT_UNPACK(ur);
                            ppe->lon_idx_start = SXM_GRID_REGION_LON_UNPACK(ll);
                            ppe->lon_idx_width = (byte)(SXM_GRID_REGION_LON_UNPACK(ur) -
                                                 ppe->lon_idx_start + 1);
                        }
                    }
                }
                else {
                    DEBUG_API(": collection selected, no mbr filtration");

                    // The request has been set, thus the extraction can be done
                    // for collection regions only. Thus, the set of regions can
                    // reduce down to regions under collection (aka 4 magic
                    // regions)
                    rc = grid_needed_corner_regions(service, &ll, &ur, &ppe->mbr);
                    if (rc == SXM_E_OK) {
                        ppe->mask_src_origin |= SXM_POI_EXT_GRID;
                        BITSET(ppe->mask, POI_EXT_HAS_VALID_MBR_BIT);

                        DEBUG_API(": all reduced to ll = %08x, ur = %08x", ll, ur);
                        ppe->lat_idx_start = SXM_GRID_REGION_LAT_UNPACK(ll);
                        ppe->lat_idx_end = SXM_GRID_REGION_LAT_UNPACK(ur);
                        ppe->lon_idx_start = SXM_GRID_REGION_LON_UNPACK(ll);
                        ppe->lon_idx_width = (byte)(SXM_GRID_REGION_LON_UNPACK(ur) -
                                             ppe->lon_idx_start + 1);   
                    }
                }
            }
            else if (service->sp > 0) { // SXM_LOCATION_IS_IN_EXTRACTION is set
                DEBUG_API(": extraction selected while the request has stations");
                // Only EXTRACTED items requested in the mask and there are
                // some station inside the extraction set.
                ppe->mask_src_origin |= SXM_POI_EXT_EXTRACTION;
            }
            else {
                DEBUG_API(": nothing to extract");
                // Nothing to extract.
                rc = SXM_E_NOENT;
            }
        }

        // Additionally check the PPOI flag to start with them if requested
        if ((service_mask & SXM_LOCATION_IS_IN_PPOIS) && (service->fav.active == TRUE)) {
            // Application wishes to see PPOI during this extraction.
            ppe->mask_src_origin |= SXM_POI_EXT_PPOI;
            rc = SXM_E_OK;
        }
    }

    if (rc == SXM_E_OK) {
        // Sync-up origin and working masks
        ppe->mask_src = ppe->mask_src_origin;
        if (ppe->mask_src_origin == SXM_POI_EXT_INACTIVE) {
            rc = SXM_E_NOENT;
        }
    }

    return rc;
}

/*****************************************************************************//**
 *
 * This function begins an POI extraction.
 *
 * \note Calls to request, modify and set ppoi list will return \ref SXM_E_BUSY
 *       until sxm_gfuel_poi_end is called.
 *
 * \param[in] service    service instance
 * \param[in] mbr        POI extraction MBR.
 * \param[in] filterMask condition for extracted entry. The filter mask is defined
 *                       by SXM_LOCATION_IS_IN_PPOIS, SXM_LOCATION_IS_IN_COLLECTION,
 *                       SXM_LOCATION_IS_IN_EXTRACTION and SXM_LOCATION_ALL.
 *                       8 LSB are reserved for own application flags which will be used
 *                       for PPOI filtrations.
 *
 * \retval SXM_E_OK      POI extraction has been started.
 * \retval SXM_E_BUSY    POI extraction in progress.
 * \retval SXM_E_INVAL   invalid MBR provided
 * \retval SXM_E_NOENT   there is no station inside the MBR or for the mask.
 * \retval SXM_E_STATE   service is not started
 *
 ********************************************************************************/
int sxm_grid_poi_begin(SXMGridService *service, const SXMMBR *mbr, ushort filterMask) {
    int rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (service->PoiExtract.mask_src_origin != SXM_POI_EXT_INACTIVE) {
        rc = SXM_E_BUSY;
        ERROR_API("POI extraction is in progress");
    }
    else if (mbr && !sxm_mbr_valid(mbr)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid MBR location");
    }
    else {
        const ushort service_mask = SXM_GRID_PPOI_SERVICE_MASK & filterMask;
        rc = grid_user_poi_begin(service, mbr, service_mask,
                                 &service->PoiExtract);
        if (rc == SXM_E_OK) {
            service->PoiExtract.service_filter = service_mask;
            service->PoiExtract.ppoi_filter = (byte)(SXM_GRID_PPOI_MASK & filterMask);
            service->PoiExtract.region_index_extract = 0;
            service->PoiExtract.psuid_index_extract = 0;
            service->PoiExtract.grid_buff_poi.region = (ushort)-1;
        }
    }
    return rc;
}

/*****************************************************************************//**
 *
 * This function extracts the next individual fuel station from the POI
 * extracted MBR in accordance with POI filter
 *
 * \param[in]  service   service instance
 * \param[out] pLocation a pointer to an station data storage
 *
 * \retval SXM_E_OK      data has been extracted.
 * \retval SXM_E_STATE   Service in wrong state or POI extraction has not been activated
 * \retval SXM_E_FAULT   NULL parameter.
 * \retval SXM_E_NOENT   There are no more entries. All entries have been extracted.
 * \retval SXM_E_INVAL   POI extraction is inactive
 *
 ********************************************************************************/
int sxm_grid_poi_extract_station(SXMGridService *service, void *pLocation) {
    int rc;
    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (!pLocation) {
        rc = SXM_E_FAULT;
        ERROR_API("NULL argument, pLocation:%p", pLocation);
    }
    else if (service->PoiExtract.mask_src_origin == SXM_POI_EXT_INACTIVE) {
        rc = SXM_E_STATE;
        ERROR_API("POI extraction has not been activated");
    }
    else {
        rc = grid_user_poi_extract(service, pLocation);
    }
    return rc;
}

/*****************************************************************************//**
 *
 * This function terminates the POI extraction and allows calls to service's request,
 * modify and set ppoi list to proceed without returning \ref SXM_E_BUSY.
 *
 * \param[in] service service instance
 *
 * \retval SXM_E_OK Extraction ended successfully.
 * \retval SXM_E_STATE Service is not started or POI extraction not in progress
 *
 ********************************************************************************/
int sxm_grid_poi_end(SXMGridService *service) {
    SXMResultCode rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK) ||
        (service->PoiExtract.mask_src_origin == SXM_POI_EXT_INACTIVE)) {
            rc = SXM_E_STATE;
            ERROR_API("service not started or POI extraction not in progress");
    }
    else {
        if (service->PoiExtract.grid_buff_poi.dbentry) {
            sxe_free(service->PoiExtract.grid_buff_poi.dbentry);
        }
        memset(&service->PoiExtract, 0, sizeof(service->PoiExtract));
        service->PoiExtract.mask_src = SXM_POI_EXT_INACTIVE;
        service->PoiExtract.mask_src_origin = SXM_POI_EXT_INACTIVE;
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * This function ensures that the required region exists and loaded from the
 * database.
 *
 * \param[in] service grid service instance
 * \param[in] region the regions id which the caller is interested in
 * \param[in] PoiExtract poi extraction handler
 * 
 * \retval SXM_E_NOENT there is no data for the passed region, thus, nothing is
 *                     loaded.
 * \retval SXM_E_OK region loaded, has data to provide and is ready to use
 *
 ******************************************************************************/
static int grid_poi_region_ensure(SXMGridService *service, ushort region,
                                  SXMGridServicePoiExtract *PoiExtract)
{
    int rc = SXM_E_OK;

    LOCK(service->service_mutex);

    if (region != PoiExtract->grid_buff_poi.region) {
        DEBUG_API(": reloading region from %X to %X",
            PoiExtract->grid_buff_poi.region, region);

        // Recycle previously loaded region if it exists and doesn't belong
        // to the grid regions.
        if ((PoiExtract->p_grid_base == &PoiExtract->grid_buff_poi) &&
            PoiExtract->grid_buff_poi.dbentry) {
            sxe_free(PoiExtract->grid_buff_poi.dbentry);
        }
        memset(&PoiExtract->grid_buff_poi, 0,
            sizeof(PoiExtract->grid_buff_poi));
        PoiExtract->grid_buff_poi.region = (ushort)-1;

        if (grid_region_has_data(service, region) == FALSE) {
            DEBUG_API("There is no data for region 0x%X", region);
            rc = SXM_E_NOENT;
        }
        else {
            // Do the check - possible we've got the same region already loaded
            const int ix = sxm_grid_baseix(service, region);
            if ((ix == SXM_GRID_REGION_INVAL_IDX) || !service->bptr[ix]) {
                rc = sxm_grid_load_region(service, &PoiExtract->grid_buff_poi, region, FALSE);
                if (rc != SXM_E_OK) {
                    ERROR_DBF("Unable to load entry table for region 0x%X", region);
                }
                else {
                    // Use own location for the baseline data
                    PoiExtract->p_grid_base = &PoiExtract->grid_buff_poi;
                    DEBUG_API("Loaded region 0x%X", region);
                }
            }
            else {
                PoiExtract->p_grid_base = service->bptr[ix];
                DEBUG_API("Use already loaded region 0x%X from the grid", region);
            }
        }
    }
    else {
        DEBUG_API(": region %X already loaded", region);
    }
    UNLOCK(service->service_mutex);
    return rc;
}

/***************************************************************************//**
 * This function returns pointer to the PPOI data structure which matches the
 * station id (region << 16) | psuid).
 *
 * \param[in] service service instance
 * \param[in] index unique id of location (region << 16) | psuid)
 *
 * \retval NULL no record found
 * \retval not-NULL valid pointer to corresponding record
 *
 ******************************************************************************/
void* sxm_grid_is_poi_from_favorite_list(SXMGridService *service, uint index)
{
    void *ret;
    SXMGridFavorites *pf = &service->fav;
    PPOISpec * const spec = (PPOISpec*)(void*)((byte*)pf->key + pf->ppoispec_offset);

    // Only this field will be used, thus, do designated init
    spec->ppoi_id = index;

    ret = grid_service_bsearch(pf->key, pf->data, *pf->count, pf->item_size,
                               pf->index_compare_callback);

    return ret;
}

/***************************************************************************//**
 * The sxm_gfuel_is_favorite_region function checks if there are any fuel
 * PPOIs in the region.
 *
 * \param[in] service service instance
 * \param[in] region Region id to search for a PPOI fuel location.
 *
 * \retval 0 if there are no PPOIs in the region
 * \retval not-0 otherwise.
 *
 ******************************************************************************/
int sxm_grid_is_favorite_region(SXMGridService *service, int region) {
    void *ret;
    SXMGridFavorites *pf = &service->fav;
    PPOISpec * const spec = (PPOISpec*)(void*)((byte*)pf->key + pf->ppoispec_offset);

    // Only this field will be used, thus, do designated init
    spec->ppoi_id = SXM_GRID_ID_PACK(region, 0 /* doesn't matter */); 

    ret = grid_service_bsearch(pf->key, pf->data, *pf->count, pf->item_size,
                               pf->region_compare_callback);
    return ret ? !0 : 0;
}

/***************************************************************************//**
 * This function extracts an individual parking location with the POI filter
 * settings and from expected POI extracted MBR
 *
 * \param[in] service service instance
 * \param[out] ret a pointer to an location instance.
 *
 * \retval SXM_E_OK    if data has been extracted
 * \retval SXM_E_NOENT if no entry found
 * \retval SXM_E_INVAL if found entry is out of expected POI MBR
 *
 ******************************************************************************/
static int grid_user_poi_extract(SXMGridService *service, void *ret)
{
    int rc = SXM_E_NOENT;
    BOOL extracted = FALSE;
    SXMGridServicePoiExtract *ppe = &service->PoiExtract;

    ENTER_EXT("(ret=%p)", ret);

    while (extracted == FALSE) {
        SXMPoint loc;
        void *pRec;
        const PPOISpec *pPpoiSpec;
        int hix;
        ushort current_region;
        uint id;
        ushort uid;
        byte id_src_mask;

        rc = sxm_grid_poi_get_station(service, &id, (void**)&pRec, &pPpoiSpec, &id_src_mask);
        if (rc != SXM_E_OK) {
            // Leave the loop since no more regions available for extraction
            DEBUG_EXT(": leaving extraction loop since no more regions to extract");
            break;
        }

        // Explode the id into region and station ids.
        current_region = SXM_GRID_UNPACK_REGION(id);
        uid = SXM_GRID_UNPACK_UID(id);

        rc = grid_poi_region_ensure(service, current_region, ppe);
        if (rc != SXM_E_OK) {
            DEBUG_EXT(": failed to load region 0x%X (rc=%d), move to the next",
                current_region, rc);
            sxm_grid_poi_next(service, TRUE, ppe, *service->fav.count, service->sp);
            continue;
        }

        // Extract geographical parking location from the DB
        rc = sxm_grid_poi_read_location(service, &loc, ppe->p_grid_base, uid);
        if (rc == SXM_E_OK) {
            // Either region should be valid and station belongs to it,
            // or all regions have been requested.
            if (!ISBITSET(ppe->mask, POI_EXT_ALL_REGION_BIT) &&
                ISBITSET(ppe->mask, POI_EXT_HAS_VALID_MBR_BIT) &&
                !sxm_mbr_inside(&service->PoiExtract.mbr, &loc)) {
                rc = SXM_E_INVAL;
            }
        }
        // At this point we known that station should be provided to the caller
        // since it lays inside the area of interest. The next step is to figure
        // out whether the station shall be extracted right now or the extraction
        // can be postponed till next type of station source.
        if (rc == SXM_E_OK) {
            // The station extracted from the PPOI collection and the code
            // in order to avoid extra searches can ignore extraction for that
            // station right now.
            if (id_src_mask == SXM_POI_EXT_PPOI) {
                // If application is expecting to see data from EXTRACTION
                // collection the station will be populated later since it's a
                // part of EXTRACTION collection
                if ((ppe->mask_src & SXM_POI_EXT_EXTRACTION) &&
                    sxm_mbr_inside(&service->ext.mbr, &loc)) {
                    rc = SXM_E_BUSY; // This value just used for the following
                                     // lines in order to detect difference from
                                     // other several cases
                }
            }
            // The station belongs to the EXTRACTION collection and the code
            // in order to avoid extra searches
            else if (id_src_mask == SXM_POI_EXT_EXTRACTION) {
                // It may happen that the station belong to one of regions
                // which will be looked trough later if the caller asked about
                // them, thus, skip extraction for that station for now.
                if (ppe->mask_src & SXM_POI_EXT_GRID) {
                    const byte lonx = SXM_GRID_REGION_LON_UNPACK(current_region);
                    const byte laty = SXM_GRID_REGION_LAT_UNPACK(current_region);
                    if ((lonx >= ppe->lon_idx_start) && (lonx < ppe->lon_idx_start + ppe->lon_idx_width) &&
                        (laty >= ppe->lat_idx_start) && (laty <= ppe->lat_idx_end)) {
                        rc = SXM_E_BUSY; // This value just used for the following
                                         // lines in order to detect difference from
                                         // other several cases
                    }
                }
            }
            // The last extraction source is GRID and if one of the stations
            // from these regions already extraction before the code should
            // do it again
            else if (id_src_mask == SXM_POI_EXT_GRID) {
                // It may happened only if the caller asked about other sources.
                // *) If PPOI asked and that point belong to the PPOI
                // *) If EXTRACTION asked and that point belong to the EXTRACTION
                if (((ppe->mask_src_origin & SXM_POI_EXT_PPOI) &&
                     sxm_grid_is_poi_from_favorite_list(service, id)) ||
                    ((ppe->mask_src_origin & SXM_POI_EXT_EXTRACTION) &&
                     sxm_mbr_inside(&service->ext.mbr, &loc))) {
                    rc = SXM_E_BUSY; // This value just used for the following
                                     // lines in order to detect difference from
                                     // other several cases
                }
            }
        }

        // Still valid?
        if (rc == SXM_E_OK) {
            void *rec = NULL;
            ushort flags = 0;

            DEBUG_EXT("loaded psuid 0x%X", uid);
            id = SXM_GRID_ID_PACK(current_region, uid);

            hix = sxm_grid_baseix(service, current_region);
            if (id_src_mask == SXM_POI_EXT_EXTRACTION)             {
                flags |= (SXM_LOCATION_IS_IN_COLLECTION | SXM_LOCATION_IS_IN_EXTRACTION);
            }
            else if (hix != SXM_GRID_REGION_INVAL_IDX) {
                flags |= SXM_LOCATION_IS_IN_COLLECTION;

                // Based on current logic the MBR kept by extraction data structure
                // contains only points with dynamic data, thus, this check is
                // enough to say that the point has dynamic data from the collection
                if (sxm_mbr_inside(&service->ext.mbr, &loc)) {
                    flags |= SXM_POI_EXT_EXTRACTION;
                }
            }

            if (id_src_mask == SXM_POI_EXT_PPOI) {
                flags |= (ushort)(SXM_LOCATION_IS_IN_PPOIS | pPpoiSpec->type_flags);
                rec = pRec;
            }
            else if (service->fav.active == TRUE) {
                byte * const ifav = (byte*)sxm_grid_is_poi_from_favorite_list(service, id);
                if (ifav) {
                    const PPOISpec *spec = ((const PPOISpec*)(void*)(ifav + service->fav.ppoispec_offset));
                    flags |= (ushort)(SXM_LOCATION_IS_IN_PPOIS | spec->type_flags);

                    // Keep this pointer for future usage if the point data is outside
                    // the extraction area.
                    rec = ifav + service->fav.data_offset;
                }
            }

            // Check service and user flags to detect necessity of the station
            if (ppe->service_filter == SXM_LOCATION_ALL) {
                extracted = TRUE;
            }
            else if (flags & ppe->service_filter) {
                if ((flags & ppe->service_filter) == SXM_LOCATION_IS_IN_PPOIS) {
                    extracted = (!ppe->ppoi_filter || (flags & ppe->ppoi_filter)) ? TRUE : FALSE;
                }
                else {
                    extracted = TRUE;
                }
            }

            if (extracted == TRUE) {
                // Since we're here the dynamic data should be provided to the application.
                // Thus, we can use either PPOI storage or regular stations collection storage.
                // At this moment data from PPOIs storage might be already found and taking into
                // account that data is the same in both places or at least PPOI data is better
                // there is no need to search for the data among regular stations location
                // collection.
                if (!rec && (flags & SXM_LOCATION_IS_IN_EXTRACTION)) {
                    rec = sxm_grid_get_poi_data(service, id, hix);
                }

                LOCK(service->service_mutex);
                rc =  service->poi_extract(service->ds.dsuser, ret, id, flags, rec);
                UNLOCK(service->service_mutex);
                if (rc != SXM_E_OK) {
                    DEBUG_EXT(": failed to extract data for UID 0x%X [%d]", uid, rc);
                    extracted = FALSE; /* No extraction performed */
                }
            }
            sxm_grid_poi_next(service, FALSE, ppe, *service->fav.count, service->sp);
        }
        else if (rc == SXM_E_BUSY) {
            DEBUG_EXT("there is no need to provide 0x%X from 0x%X right now. move to next psuid",
                uid, current_region);
            sxm_grid_poi_next(service, FALSE, ppe, *service->fav.count, service->sp);
        }
        else if (rc == SXM_E_INVAL)         {
            DEBUG_EXT("there is no uid 0x%X in 0x%X. move to next psuid",
                uid, current_region);
            sxm_grid_poi_next(service, FALSE, ppe, *service->fav.count, service->sp);
        }
        else if (rc == SXM_E_NOENT)         {
            DEBUG_EXT("there are no more uids in 0x%X. move to the next region",
                current_region);
            sxm_grid_poi_next(service, TRUE, ppe, *service->fav.count, service->sp);
        }
    }

    LEAVE_EXT("(rc=%d)", rc);
    return rc;
}

/**************************************************************************//**
* The function populates the PPOI structure with information from the fuel
* service database and readies the PPOIs for use by the service.
*
* \param[in] service service instance
* \param[in] pPpoiIndex pointer to the array of ids provided by the application,
*                        see corresponding service location id.
* \param[in] maxCount the number of PPOIs provided by the application.
* \param[in] notification notification callback.
* \param[in] usercx user data passed to each callback call
*
* \retval SXM_E_OK Success
* \retval SXM_E_NOMEM Failed to compose cumulative list of PPOIs due to
*                     lack of memory
*
******************************************************************************/
static int grid_user_ppoi_modify(SXMGridService *service, const PPOISpec *pPpoiIndex,
                                 const uint maxCount,
                                 SXMGridPPOIExtCallback_t notification, void *usercx)
{
    int rc = SXM_E_OK;
    uint i;
    byte *p_loc;
    SXMGridFavorites *pf = &service->fav;
    byte * const data = (byte*)pf->data;

    if (maxCount == 0) {
        // Cleanup internal storage
        *pf->count = 0;
        memset(data, 0, pf->capacity * pf->item_size);
        rc = SXM_E_INVAL; // This code is used for internal needs only and
                          // won't be seen by the caller
    }
    else {
        uint uniq = 0;

        // Check current ppoi set - it may have data to share for new set
        if (*pf->count != 0) {
            memset(pf->mask, 0, sizeof(pf->mask));
            memset(pf->new_mask, 0, sizeof(pf->new_mask));

            // Find all record which shall remain after update marking each of them
            // inside two independent masks to avoid additional search during internal
            // ppoi set update among apparently not sorted list of new ppois.
            for (i = 0; i < maxCount; ++i) {
                byte * const ifav =
                    (byte*)sxm_grid_is_poi_from_favorite_list(service, pPpoiIndex[i].ppoi_id);
                if (ifav) {
                    // Set bit for the existing record which belong to the new set
                    BITS(pf->mask, (uint)(ifav - data) / pf->item_size);
                    // Set bit for the new record which belong to existing set
                    BITS(pf->new_mask, i);
                    // Update just user flags
                    *((PPOISpec*)(void*)(ifav + pf->ppoispec_offset)) = pPpoiIndex[i];
                }
                else {
                    ++uniq;
                }
            }
            if (uniq == maxCount) {
                // No duplicates detected, thus pretend we've got nothing
                // to keep among existing points and can blindly apply new set
                DEBUG_REQ(": all ppois are new, apply blindly all of them");
                *pf->count = 0;
            }
            else if (!uniq && (maxCount == *pf->count)) {
                DEBUG_REQ(": there are no new PPOIs, skip this request");
                rc = SXM_E_NOENT;
            }
        }

        // Check current ppoi set - it may still have data to share for new set
        if (*pf->count == 0) {
            DEBUG_EXT(": apply full set of new ppois");

            // Just put all ids into internal storage
            *pf->count = maxCount;
            for (p_loc = data, i = 0; i < *pf->count; ++i, p_loc += pf->item_size) {
                // Clean up only in case if we here due to completely new set of ppois
                if (uniq == maxCount) {
                    memset(p_loc, 0, pf->item_size);
                }
                *((PPOISpec*)(void*)(p_loc + pf->ppoispec_offset)) = pPpoiIndex[i];
            }
        }
        else if (rc == SXM_E_OK) {
            uint p_loc_idx = 0;

            // Put new record into the fav-list using just releases slots or
            // slots which have not been used by current set of PPOIs.
            for (i = 0; i < maxCount; ++i) {
                if (!BITP(pf->new_mask, i)) {
                    p_loc = NULL;
                    // The station is out of the list detected, need to find place for it
                    for (;p_loc_idx < pf->capacity; ++p_loc_idx) {
                        if (!BITP(pf->mask, p_loc_idx)) {
                            p_loc = data + pf->item_size * p_loc_idx;
                            break;
                        }
                    }
                    if (p_loc) {
                        memset(p_loc, 0, pf->item_size); // Clean up the slot
                        *((PPOISpec*)(void*)(p_loc + pf->ppoispec_offset)) = pPpoiIndex[i]; // Update id & flags
                        BITS(pf->mask, p_loc_idx); // Mark it as used
                    }
                    else {
                        // This is a barely possible case since there is a check that
                        // list of newly added and remaining PPOIs shall not exceed
                        // SXM_PPOI_MAX records
                        DEBUG_PKT("Failed to find empty slot for new ppoi, stop update");
                        rc = SXM_E_NOMEM;
                        break;
                    }
                }
            }

            // Move the tail of the array if it still has mix of used and unused slots
            if (*pf->count > (p_loc_idx + 1)) {
                int used_idx = (int)*pf->count - 1,
                    free_idx = 0;

                while (free_idx < used_idx) {
                    // Looking for free
                    for (; free_idx < (int)pf->capacity; ++free_idx) {
                        if (!BITP(pf->mask, free_idx)) {
                            // Found
                            break;
                        }
                    }
                    // Looking for used
                    for (; used_idx >= 0; --used_idx) {
                        if (BITP(pf->mask, used_idx)) {
                            // Found
                            break;
                        }
                    }

                    // Move data from used one to the free one
                    if (free_idx < used_idx) {
                        // Mark free slot as used on
                        BITS(pf->mask, free_idx);
                        // Mark found used slot as free one since it
                        // can be reused.
                        BITC(pf->mask, used_idx);
                        // Copy the data
                        memcpy(data + pf->item_size * (uint)free_idx,
                            data + pf->item_size * (uint)used_idx,
                            (size_t)pf->item_size);
                    }
                }

                *pf->count = (uint)free_idx;
            }
            else {
                // Update number of PPOIs
                *pf->count = p_loc_idx + 1;
            }
        }

        if (rc == SXM_E_OK) {
            // sort favorite fuel stations by id
            qsort(pf->data, *pf->count,
                pf->item_size,
                pf->index_compare_callback);
        }
    }

    /* Either the request has something new or the the PPOI set from C-File
     * has the same values as the caller set, but it's the PPOI creation */
    if ((rc == SXM_E_OK) || ((rc == SXM_E_NOENT) && !pf->active)) {
        pf->dirty_flag = FALSE;
        pf->user_flags = 0;
        pf->notification = notification;
        pf->usercx = usercx;
        pf->active = TRUE;

        // Replay all aus since they may have data for ppois
        service->replay(service->ds.dsuser);

#ifndef SXM_DEBUG_PRODUCTION
        DEBUG_PKT("ppoi list has %u item(s)", *pf->count);
        for (p_loc = data, i = 0; i < *pf->count; ++i, p_loc += pf->item_size) {
            const PPOISpec *spec = ((PPOISpec*)(void*)(p_loc + pf->ppoispec_offset));
            DEBUG_PKT("station %08x [%4u:%4u] [%04x]",
                spec->ppoi_id,
                SXM_GRID_UNPACK_REGION(spec->ppoi_id),
                SXM_GRID_UNPACK_UID(spec->ppoi_id),
                spec->type_flags);
        }
#endif
        rc = SXM_E_OK;
    }
    else if ((rc == SXM_E_INVAL) || (rc == SXM_E_NOENT)) {
        // All set w/o real PPOI content modifications.
        rc = SXM_E_OK;
    }

    return rc;
}

/*****************************************************************************//**
 *
 * This function is used to set or clear the Preferred Points of Interests (PPOIs)
 *
 * This function also sets a notification callback \p pNotify. The callback is
 * called when the dynamic data has changed for 1 or more PPOIs. The 1st argument
 * to the callback is a regurgitation of the \p pUsercx, the second argument to the
 * callback is a bitwise-or of all the \ref PPOISpec::ppoi_user_flag values of all the PPOIs
 * that changed.
 *
 * \note In order to clean up current PPOI list this function can be called with
 *       \p specCnt == 0: values of \p specs, \p pNotify and \p pUsercx will be ignored.
 *
 * \param[in]  specCnt number of elements in the \p specs.
 * \param[in]  specs array of points of interests, see corresponding grid location id field.
 * \param[in]  notification notification callback.
 * \param[in]  usercx user data passed to each callback call
 *
 * \retval SXM_E_OK     on success
 * \retval SXM_E_STATE  if service is not running
 * \retval SXM_E_FAULT  if \p ids is \p NULL while \p specCnt is greater than 0
 * \retval SXM_E_INVAL  if \p specCnt is more than \ref SXM_PPOI_MAX
 * \retval SXM_E_BUSY   POI or request extraction in progress.
 *
 ********************************************************************************/
int sxm_grid_set_ppoi_list(SXMGridService *service, const uint specCnt,
                           const PPOISpec *specs,
                           SXMGridPPOIExtCallback_t notification, void *usercx)
{
    int rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if ((service->ext.state == SXM_GRID_EXT_STATE_EXTRACTING) ||
             (service->PoiExtract.mask_src_origin != SXM_POI_EXT_INACTIVE)) {
        rc = SXM_E_BUSY;
        ERROR_API("POI or request extraction in progress");
    }
    else if ((specCnt > 0) && !specs) {
        rc = SXM_E_FAULT;
        ERROR_API("specs is NULL while specCnt is greater than 0");
    }
    else if (specCnt > SXM_PPOI_MAX) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid specCnt");
    }
    else {
        // At this point the PPOI engine has not been initialized, thus, we're
        // free to init mutex which will be used later
        if (service->fav.active == FALSE) {
            sxm_mutex_init(&service->fav.mutex);
        }

        rc = grid_user_ppoi_modify(service, specs, specCnt, notification, usercx);
        if (rc == SXM_E_OK) {
            if (specCnt == 0) {
                if (service->fav.active == TRUE) {
                    sxm_mutex_destroy(&service->fav.mutex);
                }
                service->fav.notification = NULL;
                service->fav.usercx = NULL;
                service->fav.active = FALSE;
            }
        }
        else if (service->fav.active == FALSE) {
            // Apparently here there is a need to destroy the mutex since
            // it has been created earlier under the same condition.
            sxm_mutex_destroy(&service->fav.mutex);
        }
    }
    return rc;
}

/***************************************************************************//**
 * Initialize the grid-related db updater.
 *
 * \param[in] service   service instance
 * \param[in] upd       not initialized grid updater
 * \param[in] size      size of the index area, as a rule, number of stations
 *                      allowed in a region
 * \param[in] rfd       parsed RFD update metadata
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_NOMEM Failed to allocate memory
 * \retval SXM_E_PIPE Failed on t-file operations
 *
 ******************************************************************************/
int sxm_grid_update_init(SXMGridService *service, SXMGridUpdate *upd, ushort size, SXMRfdMetadata *rfd) {
    int rc;
    switch (0) { default: {
        const ushort map_size = (ushort)((size + SXM_ARCH_INT_BITS - 1) / SXM_ARCH_INT_BITS);

        upd->root = (ushort *)sxm_tfile_root(upd->dbfile);
        if (!upd->root) {
            rc = SXM_E_ERROR;
            break;
        }

        // Prepare update source file
        rc = sxm_rfd_file_buffer(service->lli, rfd, &upd->gzb);
        if (rc != SXM_E_OK) {
            DEBUG_UPD(": failed to open RFD file (%d)", rc);
            break;
        }

        // Allocate memory for updater needs
        upd->index_area =
            (uint *)sxe_calloc((size_t)(size + map_size * 2 /* new + del */), SXM_ARCH_INT_SIZE);
        if (!upd->index_area) {
            rc = SXM_E_NOMEM;
            break;
        }
        upd->nw = upd->index_area + size;
        upd->del = upd->nw + map_size;
        upd->region_size = size;
        upd->map_size = (ushort)(map_size * SXM_ARCH_INT_SIZE);
    }}

    if (rc != SXM_E_OK) {
        sxm_grid_update_uninit(service, upd);
    }
    return rc;
}

/***************************************************************************//**
 * Releases resources associated with the the grid-related db updater.
 *
 * \param[in] service   service instance
 * \param[in] upd       grid updater
 *
 * \retval SXM_E_OK Success
 *
 ******************************************************************************/
int sxm_grid_update_uninit(SXMGridService *service, SXMGridUpdate *grid) {
    UNUSED_VAR(service);

    if (grid->gzb) {
        sxm_bitbuff_close(grid->gzb);
        sxe_free(grid->gzb);
        grid->gzb = NULL;
    }

    if (grid->base.dbentry) {
        sxe_free(grid->base.dbentry);
        grid->base.dbentry = NULL;
    }
    if (grid->stream) {
        sxm_tstream_destroy(grid->stream);
        grid->stream = NULL;
    }
    if (grid->index_area) {
        sxe_free(grid->index_area);
        grid->index_area = grid->nw = grid->del = NULL;
        grid->region_size = 0;
    }
    return SXM_E_OK;
}

/***************************************************************************//**
 * This function commits the stream to the t-file and get info back in case
 * of success
 *
 * \param[in] service   service instance
 * \param[in] upd       update structure
 * \param[in] region    the region number
 *
 *  \note this function runs in RFD thread context
 *
 ******************************************************************************/
int sxm_grid_stream_commit(SXMGridService *service, SXMGridUpdate *upd, uint region)
{
    uint bs, bc;
    int rc;

    rc = sxm_tstream_commit(upd->stream, &bs, &bc);
    if (rc == SXM_E_OK) {
        ushort olds, oldc;
        if (upd->base.chain) {
            sxm_tfile_free(upd->dbfile, upd->base.chain);
            olds = sxm_tfile_chain_start(upd->base.chain);
            oldc = sxm_tfile_chain_count(upd->base.chain);
            sxm_tfile_chain_free(upd->base.chain);
            upd->base.chain = NULL;
        }
        else {
            olds = oldc = 0U;
        }
        DEBUG_UPD("Update Region [%4d] from %d,%d to %d,%d",
            region, olds, oldc, bs, bc);
        upd->root[2 * region] = (ushort)bs;
        upd->root[2 * region + 1] = (ushort)bc;
    }
    else {
        PLOG_DBF("Failed to commit stream (rc=%d, stream state=%d)",
                 rc, sxm_tstream_state(upd->stream));
    }
    return rc;
}
/***************************************************************************//**
 * This function is called to merge the updated database index and records into
 * a contiguous set of blocks in the TFile for the current region.  The number
 * of blocks required is calculated and then allocated.  The indexes are written,
 * followed by the newly updated/modified records.  Records in the region which
 * were not modified or deleted, are merged into the newly allocated blocks from
 * the old blocks before being written.
 *
 * Once the indexes and records are written to the TFile, the blocks holding
 * the stale records are released.
 *
 * \param[in] service   service instance
 * \param[in] upd       update structure
 * \param[in] omax      the largest old non-deleted ID
 * \param[in] region    the region number
 *
 *  \note this function runs in RFD thread context
 *
 ******************************************************************************/
int sxm_grid_merge_data(SXMGridService *service, SXMGridUpdate *upd, uint omax, uint region)
{
    const uint nmax = (upd->maxs > omax) ? upd->maxs : omax;
    uint idx;
    int rc;
    SXMTFileStream *stream = upd->stream;

    upd->index_area[0] = nmax;

    for  (idx = service->stations.min_id; idx <= nmax; ++idx) {
        if ((idx <= upd->maxs) && BITP(upd->nw, idx)) {
            ; //  already added
        }
        else if (!BITP(upd->del, idx) && (idx <= omax) && (upd->base.dbentry[idx] != SXM_DB_INVALID_ENTRY)) {
            // copy in OLD data
            upd->index_area[idx] = (uint)sxm_tstream_tell(stream);
            upd->merge_record(service, stream, (const char*)upd->base.base + upd->base.dbentry[idx]);
        }
    }

    // Put the index area on top
    sxm_tstream_write(upd->stream, upd->index_area,
                      (nmax + 1) * sizeof(*upd->index_area), TRUE);

    // commit the stream to the t-file and get info back in case
    // of success.
    rc = sxm_grid_stream_commit(service, upd, region);
    return rc;
}

/*****************************************************************************//**
 * This function returns actual request MBR.
 *
 * During request creation or modification the service can reduce collection
 * area based taking into account provided MBR and center point. As result
 * the actual area there data are being collected can be different than application
 * knows. This function allows application to get this real MBR and find out
 * actual collection area.
 *
 * \param[in] service     grid service instance
 * \param[in] handle      the collection handle created previously.
 * \param[out] pMbr       the actual collection MBR.
 *
 * \retval SXM_E_OK       Success.
 * \retval SXM_E_STATE    Service is not running
 * \retval SXM_E_FAULT    NULL parameter(s).
 * \retval SXM_E_INVAL    Invalid request handle
 *
 ********************************************************************************/
int sxm_grid_get_mbr(SXMGridService *service, SXMGridExtraction *handle, SXMMBR *pMbr) {
    SXMResultCode rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (!handle || !pMbr) {
        rc = SXM_E_FAULT;
        ERROR_API("null argument, pMbr: %p, handle: %p,", pMbr, handle);
    }
    else if ((handle != &service->ext) ||
             (service->ext.state == SXM_GRID_EXT_STATE_IDLE)) {
        rc = SXM_E_INVAL;
        ERROR_API("invalid handle (handle=%p) ", handle);
    }
    else {
        *pMbr = service->ext.mbr;
        rc = SXM_E_OK;
    }

    return rc;
}

/***************************************************************************//**
 *
 * The sxm_grid_point_closest and sxm_grid_point_furthest routines allow
 * the application to adjust the collection MBR as described in section
 * "When to Modify?" of SX-9845-0340 Introduction Design Guide. The distance
 * from the sort point to the closest edge of the MBR. The units reported
 * (miles or kilometers) is set by the global SDK "units" option.
 *
 * \param[in]   service service instance
 * \param[in]   pSort   a pointer to the SXMPoint structure referencing the
 *                      point from which to measure the distance.
 * \param[out]  pDist   a pointer to a float value that will contain the distance
 *                      from the sort point (pSort) to the closest edge of
 *                      the MBR (SXMMBR).
 *
 * \retval  SXM_E_OK      Distance calculated successfully.
 * \retval  SXM_E_STATE   Service is not running
 * \retval  SXM_E_FAULT   NULL parameter.
 * \retval  SXM_E_INVAL   sort point lies outside MBR
 *
 ********************************************************************************/
int sxm_grid_point_closest(SXMGridService *service, const SXMPoint *pSort, SXMFLOAT *pDist) {
    SXMResultCode rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (!pSort || !pDist) {
        rc = SXM_E_FAULT;
        ERROR_API("NULL argument pSort: %p, pDist: %p", pSort, pDist);
    }
    else {
        SXMFLOAT dist;
        dist = sxm_mbr_closest(&service->ext.mbr, pSort);
        if (sxm_is_negative(dist)) {
            rc = SXM_E_INVAL;
            ERROR_API("sort point lies outside MBR");
        }
        else {
            *pDist = dist;
            rc = SXM_E_OK;
        }
    }
    return rc;
}

/*********************************************************************//**
 *
 * The sxm_grid_point_closest and sxm_grid_point_furthest routines allow
 * the application to adjust the collection MBR as described in section
 * "When to Modify?" of SX-9845-0340 Introduction Design Guide. The distance
 * from the sort point to the closest edge of the MBR. The units reported
 * (miles or kilometers) is set by the global SDK "units" option.
 *
 * \param[in]   service service instance
 * \param[in]   pSort   a pointer to the SXMPoint structure referencing the
 *                      point from which to measure the distance.
 * \param[out]  pDist   a pointer to a float value that will contain the distance
 *                      from the sort point (pSort) to the closest edge of
 *                      the MBR (SXMMBR).
 *
 * \retval  SXM_E_OK      Distance calculated successfully.
 * \retval  SXM_E_STATE   Service is not running
 * \retval  SXM_E_FAULT   NULL parameter
 * \retval  SXM_E_INVAL   sort point lies outside MBR
 ********************************************************************************/
int sxm_grid_point_furthest(SXMGridService *service, const SXMPoint *pSort, SXMFLOAT *pDist) {
    SXMResultCode rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (!pSort || !pDist) {
        rc = SXM_E_FAULT;
        ERROR_API("NULL argument pSort: %p, pDist: %p", pSort, pDist);
    }
    else {
        SXMFLOAT dist;
        dist = sxm_mbr_furthest(&service->ext.mbr, pSort);
        if (sxm_is_negative(dist)) {
            rc = SXM_E_INVAL;
            ERROR_API("sort point lies outside MBR");
        }
        else {
            *pDist = dist;
            rc = SXM_E_OK;
        }
    }
    return rc;
}

/***************************************************************************//**
 * This private function to commit service's cycle file sections.
 *
 * \param[in] service service instance
 * \param[in] first the first section to commit
 * \param[in] end the next after the last needed section for the commit
 * \param[in] all_mask corresponding all sections mask
 * \param[out] pStat I/O statistics if requested
 *
 * \retval  SXM_E_OK     Data has been written to file system successfully.
 * \retval  SXM_E_PIPE   Error writing service data to file system
 *
 * \note        The requested section is written to disk, the section
 *              values are updated in the root block, and the root block is the
 *              written.
 *
 ********************************************************************************/
static int grid_cfile_commit(SXMGridService *service, int begin, int end, byte all_mask, SXMIOStat *pStat) {
    int num, idx;
    int rc = SXM_E_OK;
    const uint sections_update = service->sections_update;

    for (idx = begin; idx < end; ++idx) {
        if (ISBITSET(service->sections_update, idx)) {
            num = sxm_cfile_secure(service->cycle, service->root, (uint)idx, pStat);
            if (num <= 0) {
                DEBUG_API("unable to store %d C-File section.", idx);
                BITCLR(service->sections_update, idx);
            }
        }
    }

    /* commit if at least one section has been updated */
    if (service->sections_update & all_mask) {
        num = sxm_cfile_commit(service->cycle, service->root, pStat);
        if (num <= 0) {
            rc = SXM_E_PIPE;
        }
    }

    if (sections_update != service->sections_update) {
        rc = SXM_E_PIPE;
    }
    service->sections_update ^= sections_update;
    return rc;
}


/***************************************************************************//**
 *
 * The sxm_grid_save_data routine flushes any cached service data
 * required to be persisted over power-cycles to the persisted file system
 * such as non-volatile memory (NVM).
 *
 * \param[in] service service instance
 * \param[in] first the first section to commit
 * \param[in] end the next after the last needed section for the commit
 * \param[in] all_mask corresponding all sections mask
 * \param[out] pStat I/O statistics if requested
 *
 * \retval  SXM_E_OK     Data has been written to file system successfully.
 * \retval  SXM_E_STATE  Service data not available
 * \retval  SXM_E_PIPE   Error writing service data to file system
 *
 ********************************************************************************/
int sxm_grid_save_data(SXMGridService *service, int begin, int end, byte all_mask, SXMIOStat *pStat) {
    int rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        ERROR_API("service unavailable");
        rc = SXM_E_STATE;
    }
    else {
        LOCK(service->service_mutex);
        rc = grid_cfile_commit(service, begin, end, all_mask, pStat);
        UNLOCK(service->service_mutex);
    }
    return rc;
}

/*********************************************************************//**
 *
 * The function allows the application to enable or disable the service's stop
 * routine from writing the cached service's data to the file system.
 *
 * \note if the application disables writing of fuel cached data during the stop function
 * then the application shall be responsible for persisting this data across service
 * restarts and power cycling using the service's save data api call.
 *
 * \param[in]            service service instance
 * \param[in]            auto_save_on_stop controls the autosave of cached service
 *                       data on call service's stop routine.
 *
 * \retval  SXM_E_OK     Request to save data on stop is successfully.
 * \retval  SXM_E_STATE  Service data not available
 *
************************************************************************/
int sxm_grid_set_auto_save(SXMGridService *service, BOOL auto_save_on_stop) {
    SXMResultCode rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else {
        service->ds.auto_save_on_stop = auto_save_on_stop;
        rc = SXM_E_OK;
    }

    return rc;
}

/*********************************************************************//**
 *
 * The function allows the application to change the logging level on
 * the module once it has been initialized.
 *
 * \param[in] service    service instance
 * \param[in] debugLevel Level of debugging desired.
 *
 * \retval SXM_E_OK     set debug successful
 * \retval SXM_E_STATE  if service is not running
 *
 ************************************************************************/
int sxm_grid_set_debug(SXMGridService *service, int debugLevel) {
    int rc;
    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
    }
    else {
        service->ds.debug_level = (uint)debugLevel;
        DEBUG_API("debugLevel=%d", debugLevel);
        rc = SXM_E_OK;
    }
    return rc;
}

/*********************************************************************//**
 *
 * The function is used to get the debug level for the service.
 *
 * \param[in]           service       service instance
 * \param[out]          pDebugLevel   returned debug level
 *
 * \retval SXM_E_OK     on success
 * \retval SXM_E_STATE  if service is not running
 * \retval SXM_E_FAULT  if pDebugLevel is NULL
 *
 ************************************************************************/
int sxm_grid_get_debug(SXMGridService *service, int *pDebugLevel) {
    int rc;

    if (!service || (service->ds.status->service != SXM_SERVICE_OK)) {
        rc = SXM_E_STATE;
        ERROR_API("service not started");
    }
    else if (!pDebugLevel) {
        rc = SXM_E_FAULT;
        ERROR_API("NULL argument");
    }
    else {
        *pDebugLevel = (int)service->ds.debug_level;
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * This function syncs state of the service's RFD collection tracker and
 * real situation reported by the RFD engine
 *
 * \param[in] service Pointer to the Parking information.
 *
 ******************************************************************************/
void sxm_grid_rfd_sync(SXMGridService *service) {
    int slot_idx, found_slot_idx;
    BOOL bFound = FALSE;
    SXMRfdMetadata found_slot_rfd;

    ENTER_API("(service=%p)", service);
    
    memset(&found_slot_rfd, 0, sizeof(found_slot_rfd));

    // Load current RFD collection state
    for (slot_idx = 0, found_slot_idx = 0; slot_idx < SXM_RFD_MAX_COLLECTION; ++slot_idx) {
        SXMRfdMetadata slot_rfd;
        if (grid_lli_rfd_status(service, slot_idx, &slot_rfd) == SXM_E_OK) {
            if (bFound == FALSE) {
                found_slot_rfd = slot_rfd;
                found_slot_idx = slot_idx;
                DEBUG_PKT(": the first collection in slot #%d has been found",
                    slot_idx);
                bFound = TRUE;
            }
            else {
                // NOTE: This shall never happen since current code handles
                // situation with multiply RFD collections. In case if *better*
                // one has come all (but, again, only one) active collection
                // should be removed/terminated. But if it has happened by some
                // reason let's choose the best one based on the lifetime

                if (found_slot_rfd.lifetime < slot_rfd.lifetime) {
                    // Remove previously found
                    grid_lli_rfd_remove(service, found_slot_idx);
                    // Store the better one
                    found_slot_rfd = slot_rfd;
                    found_slot_idx = slot_idx;

                    DEBUG_PKT(": selecting better slot #%d since lifetime old %u < new %u",
                        slot_idx, found_slot_rfd.lifetime, slot_rfd.lifetime);
                }
            }
        }
    }

    if (bFound == TRUE) {
        service->rfd.state = SXM_GRID_RFD_STATE_COLLECTING;
        service->rfd.version = (uint)strtol(&found_slot_rfd.name[1], NULL, 10);
    }
    LEAVE_API("()");
}

/***************************************************************************//**
 * This function starts the Low Level Interface (LLI)
 *
 * \param[in] service data service abstract instance
 *
 * \retval  SXM_E_OK        LLI started successfully.
 * \retval  SXM_E_PIPE      Invalid service handle.
 * \retval  SXM_E_NOMEM     Memory could not be allocated.
 * \retval  SXM_E_THREAD    Failed due to error starting the pix thread.
 *
 ******************************************************************************/
int sxm_grid_lli_start(SXMGridService *service) {
    int rc;

    if (!service) {
        rc = SXM_E_FAULT;
    }
    else {
        CALLR_LLI("()");
        rc = sxm_lli_start(&service->ds, &service->lli);
        if (rc == SXM_E_OK) {
            rc = sxm_lli_rfd_init(service->lli);
            if (rc != SXM_E_OK) {
                sxm_lli_stop(service->lli);
                service->lli = NULL;
                rc = SXM_E_PIPE;
            }
        }
    }
    return rc;
}

/***************************************************************************//**
 * This function stops the low level interface
 *
 * \param[in] service data service abstract instance
 *
 * \retval  SXM_E_OK    LLI started successfully.
 * \retval  SXM_E_BUSY  Service is running, cannot be stopped.
 ******************************************************************************/
int sxm_grid_lli_stop(SXMGridService *service) {
    if (!service) {
        return SXM_E_FAULT;
    }
    CALLR_LLI("()");
    return sxm_lli_stop(service->lli);
}

/***************************************************************************//**
 * This function start the low level interface to collect RFD
 *
 * \param[in] service data service abstract instance
 * \param[in] rfd parsed RFD metadata info
 *
 * \retval  SXM_E_OK     Success.
 * \retval  SXM_E_FAULT  NULL parameter
 * \retval  SXM_E_NOENT  if RFD failed to start (e.g. RFD slot file issue)
 * \retval  SXM_E_NOMEM  if not enough memory to start RFD processing
 ******************************************************************************/
static int grid_lli_rfd_start(SXMGridService *service, SXMRfdMetadata *rfd) {
    if (!service || !rfd) {
        return SXM_E_FAULT;
    }
    CALLR_LLI("()");
    return sxm_lli_rfd_start(service->lli, rfd);
}

/***************************************************************************//**
 * This function returns status of Data Service RFD slot
 *
 * \param[in] service data service abstract instance
 * \param[in] id Data Service slot number [0 to SXM_RFD_MAX_COLLECTION-1]
 * \param[out] meta RFD update metadata structure assigned to the slot.
 *
 * \retval SXM_E_OK Data Service slot is busy.
 * \retval SXM_E_NOENT Data Service slot is not used.
 * \retval SXM_E_FAULT NULL parameter(s)
 *
 ********************************************************************************/
static int grid_lli_rfd_status(SXMGridService *service, int id, SXMRfdMetadata *rfd) {
    if (!service || !rfd) {
        return SXM_E_FAULT;
    }
    CALLR_LLI("()");
    return sxm_lli_rfd_status(service->lli, id, rfd);
}

/***************************************************************************//**
 * This function changes Data Service RFD slot status from busy to not used
 *
 * \param[in] service data service abstract instance
 * \param[in] id Data Service slot number
 *
 * \retval SXM_E_OK Success.
 * \retval SXM_E_FAULT NULL parameter
 *
 ********************************************************************************/
static int grid_lli_rfd_remove(SXMGridService *service, int id) {
    if (!service) {
        return SXM_E_FAULT;
    }
    CALLR_LLI("()");
    return sxm_lli_rfd_remove(service->lli, id);
}

/***************************************************************************//**
 * This function adds data to an RFD.
 *
 * \note In case of error this function internally fires up a notification
 *       through service callback about RFD issues.
 *
 * \param[in] service data service abstract instance
 * \param[in] packet Pointer to the data packet to add to the RFD.
 * \param[in] pl Starting place in the packet for the data to add.
 *
 * \retval SXM_E_OK    Slave started successfully.
 * \retval SXM_E_FAULT NULL parameter
 *
 ********************************************************************************/
int sxm_grid_lli_rfd_add(SXMGridService *service, const void *packet, int pl) {
    if (!service || !packet) {
        return SXM_E_FAULT;
    }
    CALLR_LLI(": Adding RFD packet in %d byte(s)", pl);
    return sxm_lli_rfd_add(service->lli, (byte*)packet, pl);
}

/***************************************************************************//**
 * Opens RFD segment for reading providing file handle
 *
 * \param[in] service data service abstract instance
 * \param[in] rfd Pointer to RFD file meta data.
 * \param[in] frag RFD file fragment.
 *
 * \return file handle to the RFD file or \c NULL in case of error
 *
 ********************************************************************************/
FILE *sxm_grid_lli_rfd_open(SXMGridService *service, SXMRfdMetadata *rfd, int frag) {
    if (!service || !rfd) {
        return NULL;
    }
    CALLR_LLI(": Opening RFD fragment %d", frag);
    return sxm_lli_rfd_open(service->lli, rfd, frag);
}
