/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_file.c
* \author Leslie French
* \date 8/20/2013
* \brief File-Related operations. Please, refer to \ref sxe_file_page for more details
*
* \page sxe_file_page File System Storage Class Types
*
* In many embedded systems, the amount of nvRAM available for writing is very
* limited, and resides on a separate device from the main filing system. The
* type parameter indicates that the file path should be allocated on a device
* with particular characteristics as shown in the following table:
*
* | Type | Characteristic                                                      |
* |:----:|:--------------------------------------------------------------------|
* |  F   | A 'regular' file in the filing system that can be created, deleted, read, written |
* |  R   | A read-only file, that will never be written to (once it has been created) |
* |  T   | A transient save file that will be written to during normal vehicle operation |
* |  W   | A file that will be written to (updated) during normal vehicle operation |
*
* Transient files are used to hold data over power cycles. This includes
* partially-gathered RFD files and state that is saved across power cycles. These
* files may be written quite frequently (one or more times per power cycle). However,
* they are non-critical, in that if the write fails, the contents can be discarded.
* In contrast, the Writable files contain baseline information that must be preserved. 
* Updates to these files are infrequent (typically every four or six months). Every
* effort should be made to ensure the files are synchronized correctly, though the SXe
* SDK ensures that any failure to complete a write will not damage the file.
*
* The SXe SDK is able to pre-allocate the file space for files of type \c T and \c W since
* their sizes and names can be known in advance. In contrast the generic F-type files
* are more like traditional user files.  A typical use of regular files is to hold
* channel-graphic logos, whose names are not pre-determined.
*
* When shutting the system down, priority should be given to ensure the \c W (baseline)
* files are synchronized, then regular F files, and the lastly the transient files.
*
* Read-Only files contain information that is only ever updated in stable operating
* conditions (i.e. at factory install or dealer upgrades). These files are only ever
* read during normal operation. 
*
* The file type above does not indicate the open mode of the file. An updatable
* baseline file must be allocated in the \c W file system, though most of the
* time it is opened read-only.  The combination of module and file together provide
* a unique name for the file being requested.
*
* A system with separate \c nvRAM types (usually a small writable device and a much
* larger read-only device) may allocate, for example:
*
* | Type | Mount point            | Mode            |
* |:----:|:-----------------------|:----------------|
* |  R   | mmc0:/sxm-data/...     | large read-only |
* |  F   | mmc1:/filesys/...      | writable        |
* |  T   | mmc1:/transient/...    | writable        |
* |  W   | mmc1:/sxm-databases/...| writable        |
*
* for two different grades of flash memory.
*
*******************************************************************************/

#define DEBUG_TAG "file"

#include <sxm_build.h>

#include <stdarg.h>

#ifndef WIN32
    #include <unistd.h>
    #include <sys/select.h>
#endif

#include <sys/stat.h>

#include <sxm_common.h>
#include <sxm_stdtrace.h>
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>
#include <util/sxm_timers_incl.h>
#include <util/sxm_iomanager_internal.h>

#if SXM_USE_CANFUEL
#include <sxm_canfuel.h>
#endif
#if SXM_USE_CCLOGO
#include <sxm_cclogo.h>
#endif
#if SXM_USE_CUDS
#include <sxm_cuds8.h>
#endif
#if SXM_USE_EPG
#include <sxm_epg.h>
#endif
#if SXM_USE_EV
#include <sxm_ev.h>
#endif
#if SXM_USE_GMD
#include <sxm_gmd.h>
#endif
#if SXM_USE_IGS
#include <sxm_igs.h>
#endif
#if SXM_USE_IVSM
#include <sxm_ivsm.h>
#endif
#if SXM_USE_FUEL
#include <sxm_fuel.h>
#endif
#if SXM_USE_MOVIES
#include <sxm_movies.h>
#endif
#if SXM_USE_PARKING
#include <sxm_parking.h>
#endif
#if SXM_USE_PHONETICS
#include <sxm_phonetics.h>
#endif
#if SXM_USE_SPORTS
#include <sxm_sports.h>
#endif
#if SXM_USE_WBMD
#include <sxm_wbmd.h>
#endif
#if SXM_USE_WXAGW
#include <sxm_wxagw.h>
#endif
#if SXM_USE_WXMA
#include <sxm_wxma.h>
#endif
#if SXM_USE_WXTAB
#include <sxm_wxtab.h>
#endif

#ifdef WIN32
#define snprintf _snprintf
#endif

#define SXM_PRINTF_STR(_s) ((_s) ? (_s) : "(null)")

#if SXM_FILE_PATH_HEAP_ALLOCATION
    #define SXM_FILE_PATH_ALLOC(_name, _size) \
        char *_name = (char*)sxe_malloc(_size); \
        if (!(_name)) { \
            sxm_file_log("Failed to allocate memory at line %d (%d bytes)", \
                         __LINE__, (_size)); } else
    #define SXM_FILE_PATH_FREE(_name) \
        sxe_free(_name)
#else
    /**
     * Array allocation method which behaves depending on
     * \ref SXM_FILE_PATH_HEAP_ALLOCATION value.
     * \param[in] _name variable name to keep the allocated memory
     * \param[in] _size the array size in bytes
     */
    #define SXM_FILE_PATH_ALLOC(_name, _size) \
        char _name[_size];

    /**
     * Array release method which behaves depending on
     * \ref SXM_FILE_PATH_HEAP_ALLOCATION value.
     * \param[in] _name variable name which keeps allocated memory
     *                  address.
     */
    #define SXM_FILE_PATH_FREE(_name)
#endif

/** Shareable between SXe component pthread mutex attributes */
pthread_mutexattr_t sxm_mutexattr_normal;
/** Keeps current device path. \sa sxm_sdk_start */
static char* sxm_device_path = NULL;
/** Keeps current make path callback \sa sxm_sdk_start */
static SXMMakePathCallback_t sxm_makepath;
/** Keeps current logging callback \sa sxm_sdk_start */
static SXMLogHandler_t sxm_logit = NULL;
#ifdef SXM_EXT_LOGGING
/** Keeps extended logging callback \sa sxm_sdk_set_extlog */
static SXMExtLogHandler_t sxm_log_ext = NULL;
#endif /* #ifdef SXM_EXT_LOGGING */

#if (SXM_USE_GEN8)
/** Keeps default SDK configuration */
static const SXMSdkConfig gDefaultConfig = {
    /* At this point all components make sense for the user, so set all */
    /* flags = */       SXM_SDK_CONFIG_I2S | 
                        SXM_SDK_CONFIG_PLAY_SECONDS | 
                        SXM_SDK_CONFIG_LANGUAGE |
                        SXM_SDK_CONFIG_UNIT |
                        SXM_SDK_CONFIG_RFD_MODE |
                        SXM_SDK_CONFIG_IOMAN_PROFILE,
    /* playSeconds = */ SCAN_ITEM_PLAY_SEC_DFLT,
    /* I2SControl = */  SXM_SDK_I2S_MASTER,
    /* lang = */        SXM_SDK_LANG_ENGLISH,
    /* unit = */        SXM_SDK_UNIT_IMPERIAL,
    /* mode = */        SXM_SDK_RFD_FILE_SYSTEM,
    /* ioman_profile = */{ &SXM_IOMANAGER_POWER_DOWN_PROFILE, 0U, 0U, 0U }
};

/** Keeps current SDK configuration */
static SXMSdkConfig gCurrentConfig = {
    /* At this point all components make sense for the user, so set all */
    /* flags = */       SXM_SDK_CONFIG_I2S | 
                        SXM_SDK_CONFIG_PLAY_SECONDS | 
                        SXM_SDK_CONFIG_LANGUAGE |
                        SXM_SDK_CONFIG_UNIT | 
                        SXM_SDK_CONFIG_RFD_MODE |
                        SXM_SDK_CONFIG_IOMAN_PROFILE,
    /* playSeconds = */ SCAN_ITEM_PLAY_SEC_DFLT,
    /* I2SControl = */  SXM_SDK_I2S_MASTER,
    /* lang = */        SXM_SDK_LANG_ENGLISH,
    /* unit = */        SXM_SDK_UNIT_IMPERIAL,
    /* mode = */        SXM_SDK_RFD_FILE_SYSTEM,
    /* ioman_profile = */{ &SXM_IOMANAGER_POWER_DOWN_PROFILE, 0U, 0U, 0U }
};
#else
/** Keeps default SDK configuration */
static const SXMSdkConfig gDefaultConfig = {
    /* At this point all components make sense for the user, so set all */
    /* flags = */ SXM_SDK_CONFIG_I2S | 
                  SXM_SDK_CONFIG_MREF | 
                  SXM_SDK_CONFIG_LANGUAGE |
                  SXM_SDK_CONFIG_UNIT |
                  SXM_SDK_CONFIG_RFD_MODE |
                  SXM_SDK_CONFIG_CFM_WAIT |
                  SXM_SDK_CONFIG_IOMAN_PROFILE,
#if SXM_USE_WBMD
    /* MRefEnabled = */ TRUE,
#else
    /* MRefEnabled = */ FALSE,
#endif
    /* ConfirmationWaitTime = */ 4U /* 400ms */,
    /* I2SControl = */ SXM_SDK_I2S_MASTER,
    /* lang = */ SXM_SDK_LANG_ENGLISH,
    /* unit = */ SXM_SDK_UNIT_IMPERIAL,
    /* mode = */ SXM_SDK_RFD_FILE_SYSTEM,
    /* ioman_profile = */{ &SXM_IOMANAGER_POWER_DOWN_PROFILE, 0U, 0U, 0U }
};

/** Keeps current SDK configuration */
static SXMSdkConfig gCurrentConfig = {
    /* At this point all components make sense for the user, so set all */
    /* flags = */ SXM_SDK_CONFIG_I2S | 
                  SXM_SDK_CONFIG_MREF | 
                  SXM_SDK_CONFIG_LANGUAGE |
                  SXM_SDK_CONFIG_UNIT |
                  SXM_SDK_CONFIG_RFD_MODE |
                  SXM_SDK_CONFIG_CFM_WAIT |
                  SXM_SDK_CONFIG_IOMAN_PROFILE,
#if SXM_USE_WBMD
    /* MRefEnabled = */ TRUE,
#else
    /* MRefEnabled = */ FALSE,
#endif
    /* ConfirmationWaitTime = */ 4U /* 400ms */,
    /* I2SControl = */ SXM_SDK_I2S_MASTER,
    /* lang = */ SXM_SDK_LANG_ENGLISH,
    /* unit = */ SXM_SDK_UNIT_IMPERIAL,
    /* mode = */ SXM_SDK_RFD_FILE_SYSTEM,
    /* ioman_profile = */ {&SXM_IOMANAGER_POWER_DOWN_PROFILE, 0U, 0U, 0U}
};

#endif


#if (SXM_USE_GEN8) 
/**
 * \name SXe8 Release Version
 * @{
 */
/** SXe Major Version Number */
#define SXE_SW_VER_MAJOR        2
/** SXe Major Version Number */
#define SXE_SW_VER_MINOR        30
/** SXe incremental Version Number */
#define SXE_SW_VER_INC          13

/** SXe release month */
#define SXE_SW_VER_MONTH        5
/** SXe release day */
#define SXE_SW_VER_DAY          2
/** SXE release year */
#define SXE_SW_VER_YEAR         2018
#else 
/**
 * \name SXe Release Version
 * @{
 */
/** SXe Major Version Number */
#define SXE_SW_VER_MAJOR        1
/** SXe Major Version Number */
#define SXE_SW_VER_MINOR        30
/** SXe incremental Version Number */
#define SXE_SW_VER_INC          13

/** SXe release month */
#define SXE_SW_VER_MONTH        5
/** SXe release day */
#define SXE_SW_VER_DAY          2
/** SXE release year */
#define SXE_SW_VER_YEAR         2018

#endif 

/** Keeps current SXe version data */
static SXMSxeVerNum Sxe_ver_num =
{
    SXE_SW_VER_MAJOR,
    SXE_SW_VER_MINOR,
    SXE_SW_VER_INC,
    SXE_SW_VER_MONTH,
    SXE_SW_VER_DAY,
    SXE_SW_VER_YEAR
};
/** @} */

/** Keeps current SXe SDK status */
static SXMStatus status = {SXM_SERVICE_STOPPED, SXM_SUBS_UNKNOWN};

/**
 * String representation for Debug Area values.
 * NOTE: Indexes must correspond to \ref SxmDebugArea values.
 */
static const char* dbgAreaToText[] = {
    "API",  /* API */
    "USR",  /* User */
    "REQ",  /* Request */
    "EXT",  /* Extraction */
    "PKT",  /* Protocol */
    "UPD",  /* RFD Update */
    "DBF",  /* Database (TFile) */
    "LLI",  /* Low-level Interface */
    "UTL"   /* Utility */
};

ARRAY_SIZE_STATIC_ASSERT(dbgAreaToText, SXM_DBG_NUM_AREAS);

/**
 * String representation for Debug Type values.
 * NOTE: Indexes must correspond to \ref SxmDebugLevel values.
 */
static const char* dbgTypeToText[] = {
    "CLL",  /* Call */
    "EVT",  /* Event */
    "DBG",  /* Debug */
    "ERR",  /* Error */
    "WRN",  /* Warning */
    "DMP",  /* Dump */

    "PLE" /* Production Log Error */
};

ARRAY_SIZE_STATIC_ASSERT(dbgTypeToText, SXM_DBG_NUM_TYPES);

/****************************************************************************//**
 * Converts an SXM_E_* error code to a human readable string.
 *
 * This function may be called at any time during the execution of the SXe SDK
 * (even before calling #sxm_sdk_start()).
 *
 * \param err one of the SXM_E_* error codes (SXM_E_OK, SXM_E_BUSY, etc)
 *
 * \returns a NUL-terminated string representation of the error code (e.g. "SXM_E_OK(0)")
 *
 * \note Use of an unknown value for \p err results in "UNKNOWN ERROR CODE".
 *******************************************************************************/
SXESDK_API const char *sxm_sdk_format_result(int err) {
    /* Use the result executor macro to create the cases and stringify the name */
    switch(err) {
    #define RESULT_ACTION_(a,b) case a: return #a "(" #b ")";
        RESULT_ACTION_EXECUTOR_
    #undef RESULT_ACTION_
    default:
        /* can't append the actual error number to unknown codes because we'd
           need to store the string in memory and that memory would have to be
           shared (preventing it from being a thread-safe operation). */
        return "UNKNOWN ERROR CODE";
    }
}

/****************************************************************************//**
 * Gets one of the options that cause individual modules to report values differently.
 * See #sxm_sdk_set_option() for supported option keys and values
 *
 * This function may be called at any time during the execution of the SXe SDK
 * (even before calling #sxm_sdk_start()).
 *
 * \param key a pointer to a NUL-terminated string indicating the desired option
 *
 * \returns the option's value or '\0' if \p key is NULL or an invalid option
 *******************************************************************************/
SXESDK_API char sxm_sdk_get_option(const char *key) {
    char rc = '\0';
    if (key) {
        if (strequ(key, "lang")) {
            switch (gCurrentConfig.lang) {
                case SXM_SDK_LANG_ENGLISH: rc = 'E'; break;
                case SXM_SDK_LANG_FRENCH: rc = 'F'; break;
                case SXM_SDK_LANG_SPANISH: rc = 'S'; break;
            }
        }
        else if (strequ(key, "unit")) {
            switch (gCurrentConfig.unit) {
                case SXM_SDK_UNIT_METRIC: rc = 'm'; break;
                case SXM_SDK_UNIT_IMPERIAL: rc = 'i'; break;
            }
        }
    }

    return rc;
}

/****************************************************************************//**
 * Sets one of the options that cause individual modules to report values differently.
 *
 * There are two options that can be set globally for the whole of the SXe SDK
 * that cause individual modules to report values differently.
 *
 *  Key  |Value|Meaning
 *  -----|-----|---------------------------------------------
 * "lang"|'E'  |Use English as the preferred output language (THIS IS THE DEFAULT)
 * "lang"|'F'  |Use French as the preferred output language
 * "lang"|'S'  |Use Spanish as the preferred output language
 * "unit"|'i'  |Use imperial units to report measured quantities (THIS IS THE DEFAULT)
 * "unit"|'m'  |Use metric units to report measured quantities
 *
 * This function may be called at any time during the execution of the SXe SDK
 * (even before calling #sxm_sdk_start()). The behavior of SXe is undefined if an option
 * value is changed while extractions are being processed; they may report results using
 * the old option or the new option.
 *
 * \param[in] key   a pointer to a NUL-terminated string indicating which option to set
 * \param[in] value the desired value to set for the option
 *
 * \retval SXM_E_OK    success.
 * \retval SXM_E_FAULT \p key was NULL.
 * \retval SXM_E_INVAL \p key is an invalid option name
 *******************************************************************************/
SXESDK_API int sxm_sdk_set_option(const char *key, char value) {
    int rc = SXM_E_OK;
    if (key == NULL) {
        rc = SXM_E_FAULT;
    }
    else if (strequ(key, "lang")) {
        switch (value) {
            case 'E': gCurrentConfig.lang = SXM_SDK_LANG_ENGLISH; break;
            case 'F': gCurrentConfig.lang = SXM_SDK_LANG_FRENCH; break;
            case 'S': gCurrentConfig.lang = SXM_SDK_LANG_SPANISH; break;
            default: rc = SXM_E_INVAL; break;
        }
    }
    else if (strequ(key, "unit")) {
        switch (value) {
            case 'm': gCurrentConfig.unit = SXM_SDK_UNIT_METRIC; break;
            case 'i': gCurrentConfig.unit = SXM_SDK_UNIT_IMPERIAL; break;
            default: rc = SXM_E_INVAL; break;
        }
    }
    else {
        rc = SXM_E_INVAL;
    }

    return rc;
}

/****************************************************************************//**
 * Sets SDK configuration items which are used by the SDK and it's components.
 * The function should be called right before sxm_sdk_start function 
 * to guarantee that configuration items will be picked up.
 * 
 * \param[in] config structure with set of items the application wishes to set
 *                   explicitely. The field \ref SXMSdkConfig::flags allows to
 *                   choose only needed components to configure and use un-set 
 *                   ones by default. In order to reset configuration to default
 *                   the application can call this function with \p NULL or simply
 *                   with SXMSdkConfig::flags equals to #SXM_SDK_CONFIG_NONE.
 * \retval SXM_E_OK Configuration has been set.
 *******************************************************************************/
SXESDK_API int sxm_sdk_set_config(const SXMSdkConfig *config) {
    if ((NULL == config) || (config->flags == SXM_SDK_CONFIG_NONE)) {
        gCurrentConfig = gDefaultConfig;
    }
    else { 
        if (config->flags & SXM_SDK_CONFIG_I2S) {
            gCurrentConfig.I2SControl = config->I2SControl;
        }
#if (SXM_USE_GEN8)
        if (config->flags & SXM_SDK_CONFIG_PLAY_SECONDS) {
            gCurrentConfig.playSeconds = config->playSeconds;
        }
#else
        if (config->flags & SXM_SDK_CONFIG_MREF) {
            gCurrentConfig.MRefEnabled = config->MRefEnabled;
        }
        if (config->flags & SXM_SDK_CONFIG_CFM_WAIT) {
            gCurrentConfig.ConfirmationWaitTime = config->ConfirmationWaitTime;
        }
#endif
        if (config->flags & SXM_SDK_CONFIG_LANGUAGE) {
            gCurrentConfig.lang = config->lang;
        }
        if (config->flags & SXM_SDK_CONFIG_UNIT) {
            gCurrentConfig.unit = config->unit;
        }
        if (config->flags & SXM_SDK_CONFIG_RFD_MODE) {
            gCurrentConfig.mode = config->mode;
        }
        if (config->flags & SXM_SDK_CONFIG_IOMAN_PROFILE) {
            gCurrentConfig.ioman_profile = config->ioman_profile;
        }
    }
    return SXM_E_OK;
}

/****************************************************************************//**
 * Provides current SDK configuration.
 * 
 * \param[out] config provides full set of currently used configurable items
 * \retval SXM_E_OK Configuration provided
 * \retval SXM_E_FAULT NULL Parameter
 *******************************************************************************/
SXESDK_API int sxm_sdk_get_config(SXMSdkConfig *config) {
    int rc;
    if (NULL != config) {
        *config = gCurrentConfig;
        rc = SXM_E_OK;
    }
    else {
        rc = SXM_E_FAULT;
    }
    return rc;
}

/****************************************************************************//**
 *
 * Initializes the SXe SDK.
 *
 * The first main call into the SXe SDK should be this function.
 *
 * \param device_path a string which refers a device to be used to communication
 *                    with a satellited module. This parameter is passed unchanged
 *                    to the communication driver implementation (\ref sxm_hal_hw_init)
 *                    to let SXe integrator use this information to initialize
 *                    and control communication. For examples, in common implementations,
 *                    it could be a non-NULL pointer to NUL-terminated string that
 *                    indicates the path to the satellite module device interface
 *                    (e.g. "\\.\COM10", "/dev/ser1", etc). Another more advanced
 *                    sample could be device and some additional arguments like 
 *                    GPIOs, e.g. "/dev/ttyS0,5,11" where '5' is the reset pin and
 *                    '11' is for power.
 *                    Please note, this function copies the path (if not NULL) to
 *                    an internal variable, it is OK if the memory pointed to by
 *                    \p devicePath is free or reused after calling this function.
 *
 * \param make_path_func a non-NULL pointer to a function that generates file paths
 *                       in an implementation specific way for the different files
 *                       created and/or used by SXe. Refer to #SXMMakePathCallback_t
 *                       for the requirements of this function.
 *
 * \param log_func a non-NULL pointer to a function that consumes log messages
 *                 in an implementation specific way (e.g. store to debug log
 *                 file, output to console, store in RAM buffer, ignored, etc.).
 *                 The \p log_func routine is used for all printable output generated
 *                 by the SXe SDK library. See #SXMLogHandler_t
 *
 * \retval SXM_E_OK    the SDK was successfully initialized.
 * \retval SXM_E_FAULT one of the parameters is NULL.
 * \retval SXM_E_NOMEM couldn't allocate memory needed for initialization
 * \retval SXM_E_STATE the SDK is already initialized
 *
 * \see sxm_sdk_status()
 * \see sxm_sdk_stop()
 *
 ********************************************************************************/
SXESDK_API int sxm_sdk_start(const char *device_path,
                             SXMMakePathCallback_t make_path_func,
                             SXMLogHandler_t log_func)
{
    /* No NULL arguments allowed */
    if ((make_path_func == NULL) || (log_func == NULL))
    {
        return SXM_E_FAULT;
    }

    /* Check that service isn't already initialized */
    if (status.service != SXM_SERVICE_STOPPED) 
    {
        return SXM_E_STATE;
    }

    if (device_path != NULL)
    {
        sxm_device_path = (char *) sxe_malloc(strlen(device_path) + 1);
        /* Check that device_path and it's NUL terminator
           will fit in sxm_device_path */
        if (sxm_device_path == NULL) {
            return SXM_E_NOMEM;
        }
        strcpy(sxm_device_path, device_path);
    }
    else
    {
        sxm_device_path = NULL;
    }

    sxm_makepath = make_path_func;
    sxm_logit = log_func;

    pthread_mutexattr_init(&sxm_mutexattr_normal);
    pthread_mutexattr_settype(&sxm_mutexattr_normal, PTHREAD_MUTEX_NORMAL);
    sxm_crc16_initialize();
    sxm_crc32_initialize();

    /* SDK is initialized */
    status.service = SXM_SERVICE_OK;
    return SXM_E_OK;
}

/****************************************************************************//**
 *
 * Clears cycle files for all supported services. May be used to reset SDk to Factory
 * Default state.
 *
 * The function shall be called either before SDK startup (before \ref sxm_sdk_start)
 * or after SDK is stopped (\ref sxm_sdk_stop)
 *
 * \param make_path_func a non-NULL pointer to a function that generates file paths
 *                       in an implementation specific way for the different files
 *                       created and/or used by SXe. THe funciton is expected to be
 *                       the same as passed as parameter into \ref sxm_sdk_start.
 *                       Refer to #SXMMakePathCallback_t for the detailed requirements.
 *
 * \param log_func a non-NULL pointer to a function that consumes log messages
 *                 in an implementation specific way (e.g. store to debug log
 *                 file, output to console, store in RAM buffer, ignored, etc.).
 *                 The \p log_func routine is used for all printable output generated
 *                 by the SXe SDK library. See #SXMLogHandler_t
 *
 * \retval SXM_E_OK    all cycle files were removed
 * \retval SXM_E_FAULT one of the parameters is NULL.
 * \retval SXM_E_NOMEM couldn't allocate required amount of memory
 * \retval SXM_E_STATE Attempt to call the function while SDK is working
 *
 ********************************************************************************/
SXESDK_API int sxm_sdk_cfiles_clean(SXMMakePathCallback_t make_path_func,
    SXMLogHandler_t log_func)
{
    /* No NULL arguments allowed */
    if ((make_path_func == NULL) || (log_func == NULL))
    {
        return SXM_E_FAULT;
    }

    /* Check that service isn't already initialized */
    if (status.service != SXM_SERVICE_STOPPED)
    {
        return SXM_E_STATE;
    }

    /* Assigning MakePath and Logging functions. Those will be rewritten by sxm_sdk_start() */
    sxm_makepath = make_path_func;
    sxm_logit = log_func;

    /* Removing cycle files one by one */
#if ((SXM_USE_GEN8 == 0) && (SXM_USE_SMS == 0))
    sxm_sxi_cfile_clean();
#endif

#if SXM_USE_CANFUEL
    sxm_canfuel_cfile_clean();
#endif

#if SXM_USE_CCLOGO
    sxm_cclogo_cfile_clean();
#endif

#if SXM_USE_CUDS
    sxm_cuds8_cfile_clean();
#endif

#if SXM_USE_EPG
    sxm_epg_cfile_clean();
#endif

#if SXM_USE_EV
    sxm_ev_cfile_clean();
#endif

#if SXM_USE_FUEL
    sxm_fuel_cfile_clean();
#endif

#if SXM_USE_GMD
    sxm_gmd_cfile_clean();
#endif

#if SXM_USE_IGS
    sxm_igs_cfile_clean();
#endif

#if SXM_USE_IVSM
    sxm_ivsm_cfile_clean();
#endif

#if SXM_USE_MOVIES
    sxm_movies_cfile_clean();
#endif

#if SXM_USE_PARKING
    sxm_parking_cfile_clean();
#endif

#if SXM_USE_PHONETICS
    sxm_phn_cfile_clean();
#endif

#if SXM_USE_RFD
    sxm_rfd_cfile_clean();
#endif

#if SXM_USE_SPORTS
    sxm_sports_cfile_clean();
#endif

#if SXM_USE_WBMD
    sxm_wbmd_cfile_clean();
#endif

#if SXM_USE_WXAGW
    sxm_wxagw_cfile_clean();
#endif

#if SXM_USE_WXMA
    sxm_wxma_cfile_clean();
#endif

#if SXM_USE_WXTAB
    sxm_wxtab_cfile_clean();
#endif

    return SXM_E_OK;
}

/****************************************************************************//**
 * Queries the status of the SXe SDK.
 *
 * After successful completion of this function, 'ret->service' will indicate
 * if the SDK is initialized (#SXM_SERVICE_OK) or not (#SXM_SERVICE_STOPPED)
 *
 * \param[out] ret a pointer to the structure into which the current status should
 *                 be copied.
 *
 * \retval SXM_E_OK    the status query completed successfully.
 * \retval SXM_E_FAULT \p ret was NULL
 *
 * \see sxm_sdk_start()
 * \see sxm_sdk_stop()
 *
 ********************************************************************************/
SXESDK_API int sxm_sdk_status(SXMStatus *ret) {

    /* ret parameter verification */
    if (ret == NULL)
    {
        return SXM_E_FAULT;
    }

    /* copy over the status of the sxi thread */
    *ret = status;

    /* report back if the audio thread is running or not */
    return SXM_E_OK;
}

/****************************************************************************//**
 * Frees any resources created by the SXe SDK.
 *
 * \retval SXM_E_OK    the SDK was successfully uninitialized.
 * \retval SXM_E_STATE the SDK is already stopped
 *
 * \see sxm_sdk_status()
 * \see sxm_sdk_stop()
 *
 ********************************************************************************/
SXESDK_API int sxm_sdk_stop(void) {
    /* SDK is started */
    if (status.service != SXM_SERVICE_OK)
    {
        return SXM_E_STATE;
    }

    /* stop the I/O manager */
    (void) sxm_iomanager_uninit();

    /* stop timers service (does nothing if service is not running)*/
    (void) sxm_timers_stop();

    /* release the mutex resources */
    pthread_mutexattr_destroy(&sxm_mutexattr_normal);

    /* uninit rfd resources */
    sxm_rfd_uninit();

    /* free the memory holding the device path */
    if (NULL != sxm_device_path)
    {
        sxe_free(sxm_device_path);
        sxm_device_path = NULL;
    }

    /* move to stopped stated since we're not ready to work */
    status.service = SXM_SERVICE_STOPPED;

    return SXM_E_OK;
}

/*****************************************************************************//**
 * Queries the version of the SXe SDK.
 *
 * The SXe SDK maintains an internal version number which can be retrieved and
 * presented to the user or developer (possibly through a hidden engineering or
 * debug screen).
 *
 * \param[out] sxe_ver_num_ptr a pointer to the structure into which the current
 *                             version info should be copied.
 *
 * \retval SXM_E_OK    the version query completed successfully.
 * \retval SXM_E_FAULT \p ret was NULL
 *
 * \see SXMSxeVerNum
 *
 ********************************************************************************/
SXESDK_API int sxm_sdk_get_version(SXMSxeVerNum *sxe_ver_num_ptr) {

    int rc = SXM_E_OK;

    /* Is the input pointer valid ? */
    if (sxe_ver_num_ptr == NULL) {
        /* not a valid pointer */
        rc = SXM_E_FAULT;
    } else {
        /* return the version number */
        *sxe_ver_num_ptr = Sxe_ver_num;
    }

    return rc;
}

/*****************************************************************************//**
 * Gets the path to the interface that communicates with the satellite module.
 * The path was set by a call to #sxm_sdk_start()
 *
 * \returns a pointer to the NUL-terminated string that describes the path
 *
 ********************************************************************************/
extern SXESDK_API const char *sxm_sdk_get_device_path(void)
{
    return sxm_device_path;
}

/***************************************************************************//**
* Opens a file (in the filing system) as a FILE structure
*
* \param[in] type file type (\ref sxe_file_page)
* \param[in] module the name of the module requesting the file
* \param[in] file the name of the file, relative to the module name
* \param[in] mode open mode ("rb" etc.)
*
* \return an open file description, or NULL in case of error
*
*******************************************************************************/
FILE *sxm_file_open(char type, const char *module, const char *file, const char *mode)
{
    FILE *result = NULL;
    if ((module != NULL) && (file != NULL) && (mode != NULL))
    {
        SXM_FILE_PATH_ALLOC(temp, SXE_FILE_PATH_MAX_SIZE)
        {
            const char *path;
            path = sxm_makepath(temp, SXE_FILE_PATH_MAX_SIZE, type, module, file);
            if (path != NULL)
            {
                result = fopen(path, mode);
                if (result == NULL)
                {
                    PLOG_UTL("Failed to open file (%c/%s/%s) as %s in mode %s",
                        type, SXM_PRINTF_STR(module), SXM_PRINTF_STR(file), path,
                        SXM_PRINTF_STR(mode));
                }
            }
            else
            {
                PLOG_UTL("No path provided for (%c/%s/%s) to open",
                    type, SXM_PRINTF_STR(module), SXM_PRINTF_STR(file));
            }
            SXM_FILE_PATH_FREE(temp);
        }
    }
    return result;
}

/***************************************************************************//**
* Opens a file (in the filing system) as a FILE structure in accordance
* with requested mode. If file is absent it will be created and provided to
* the caller in the required mode.
*
* \param[in] type file type (\ref sxe_file_page)
* \param[in] module the name of the module requesting the file
* \param[in] file the name of the file, relative to the module name
* \param[in] mode open mode ("rb" etc.)
*
* \return an open file description, or NULL in case of error
*
*******************************************************************************/
FILE *sxm_file_open_or_create(char type, const char *module, const char *file, const char *mode)
{
    FILE *result = NULL;
    if ((module != NULL) && (file != NULL) && (mode != NULL))
    {
        SXM_FILE_PATH_ALLOC(temp, SXE_FILE_PATH_MAX_SIZE)
        {
            const char *path;
            path = sxm_makepath(temp, SXE_FILE_PATH_MAX_SIZE, type, module, file);
            if (path != NULL)
            {
                /* Open file in reading mode to check its existence */
                result = fopen(path, "r+b");
                if (result == NULL)
                {
                    /* Try to create this file */
                    result = fopen(path, "w+b");
                }

                if (result != NULL)
                {
                    /* Close somehow previously opened file */
                    fclose(result);
                    /* One of the previous commands were successfully, so,
                       just re-open file in the requested mode. */
                    result = fopen(path, mode);
                }
                /* Eventually do the last check if it still no opened file */
                if (result == NULL)
                {
                    PLOG_UTL("Failed to open or create file "
                             "'%c/%s/%s' as '%s' in mode '%s'",
                             type, SXM_PRINTF_STR(module),
                             SXM_PRINTF_STR(file), path,
                             SXM_PRINTF_STR(mode));
                }
            }
            else
            {
                PLOG_UTL("No path provided for '%c/%s/%s'",
                    type, SXM_PRINTF_STR(module), SXM_PRINTF_STR(file));
            }
            SXM_FILE_PATH_FREE(temp);
        }
    }
    return result;

}

/***************************************************************************//**
* Closes a file (in the filing system) as a FILE structure
*
* \param[in] FILE pointer
*
* \retval SXM_E_OK Success
* \retval SXM_E_FAULT Null parameter
* \retval SXM_E_PIPE File operation failed
*
*******************************************************************************/
int sxm_file_close(FILE *pFile)
{
    int rc;

    /* sanity check */
    if (pFile == NULL)
    {
        rc = SXM_E_FAULT;
    }
    /* now close the file stream */
    else if (fclose(pFile) != 0)
    {
        rc = SXM_E_PIPE;
    }
    else
    {
        rc = SXM_E_OK;
    }

    return rc;

}

/***************************************************************************//**
* Loads file into the memory provides by the caller
* 
* \param[in] type the file type (\ref sxe_file_page)
* \param[in] module the name of the module requesting the file
* \param[in] file the name of the file, relative to the module name
* \param[in] buffer the callers buffer to accommodate file content
* \param[in] buffer_size the size of the \c buffer in bytes
* \param[out] size provides number of read bytes
*
* \retval 0 in case of success
* \retval -1 in case of error
*
*******************************************************************************/
int sxm_file_load_type(char type, const char *module, const char *file, void *buffer,
                       size_t buffer_size, size_t *size)
{
    int rc = -1;
    size_t fsize;
    
    if (sxm_file_size(type, module, file, &fsize) != -1) {
        if (fsize <= buffer_size) {
            FILE *in = sxm_file_open(type, module, file, "rb");
            if (in != NULL) {
                *size = fread(buffer, 1, fsize, in);
                sxm_file_close(in);
                if (*size == fsize) {
                    rc = 0;
                }
            }
        }
    }

    return rc;
}

/***************************************************************************//**
* Loads regular file (type ['F'](\ref sxe_file_page)) into the memory provides by the caller
* 
* \param[in] module the name of the module requesting the file
* \param[in] file the name of the file, relative to the module name
* \param[in] buffer the callers buffer to accommodate file content
* \param[in] buffer_size the size of the \c buffer in bytes
* \param[out] size provides number of read bytes
*
* \retval 0 in case of success
* \retval -1 in case of error
*
*******************************************************************************/
int sxm_file_load(const char *module, const char *file, void *buffer,
                  size_t buffer_size, size_t *size)
{
    return sxm_file_load_type('F', module, file, buffer, buffer_size, size);
}

/***************************************************************************//**
* Deletes a file in the filing system.
*
* \param[in] type file type (\ref sxe_file_page)
* \param[in] module the name of the module requesting the file
* \param[in] file the name of the file, relative to the module name
*
* \retval 0 in case of success and -1 in case of error.
*
*******************************************************************************/
int sxm_file_remove(char type, const char *module, const char *file)
{
    int result = -1;
    if ((module != NULL) && (file != NULL))
    {
        SXM_FILE_PATH_ALLOC(temp, SXE_FILE_PATH_MAX_SIZE)
        {
            const char *path = sxm_makepath(temp, SXE_FILE_PATH_MAX_SIZE, type, module, file);
            if (path != NULL)
            {
                result =
#ifdef WIN32
                    _unlink
#else
                    unlink
#endif
                    (path);
                if ((result == -1) && (errno != ENOENT))
                {
                    PLOG_UTL("Failed to remove file '%c/%s/%s' as '%s'",
                        type, SXM_PRINTF_STR(module), SXM_PRINTF_STR(file), path);
                }
            }
            else
            {
                PLOG_UTL("No path provided for '%c/%s/%s'",
                    type, SXM_PRINTF_STR(module), SXM_PRINTF_STR(file));
            }

            SXM_FILE_PATH_FREE(temp);
        }
    }
   
    return result;
}

/***************************************************************************//**
* The rename routine changes the name of a file.
* \note This is only supported for files of the same type and within the same module.
*
* \param[in] type file type (\ref sxe_file_page)
* \param[in] module the name of the module requesting the file
* \param[in] from file name of the existing file
* \param[in] to the file name the file will have in case of success
*
* \return On success, zero is returned. On error, -1 is returned.
*
*******************************************************************************/
int sxm_file_rename(char type, const char *module, const char *from, const char *to)
{
    int result = -1;

    if ((module != NULL) && (from != NULL) && (to != NULL))
    {
        SXM_FILE_PATH_ALLOC(tempf, SXE_FILE_PATH_MAX_SIZE)
        {
            SXM_FILE_PATH_ALLOC(tempt, SXE_FILE_PATH_MAX_SIZE)
            {
                const char *from_path =
                    sxm_makepath(tempf, SXE_FILE_PATH_MAX_SIZE, type, module, from);
                const char *to_path =
                    sxm_makepath(tempt, SXE_FILE_PATH_MAX_SIZE, type, module, to);
                if ((from_path != NULL) && (to_path != NULL))
                {
                    result = rename(from_path, to_path);
                    if (result == -1)
                    {
                        PLOG_UTL("Failed to rename file '%c/%s/%s'"
                                 " to '*/*/%s' as '%'s to '%s'",
                                 type, SXM_PRINTF_STR(module),
                                 SXM_PRINTF_STR(from), SXM_PRINTF_STR(to),
                                 from_path, to_path);
                    }
                }
                else
                {
                    PLOG_UTL("No path provided for '%c/%s/%s' and/or '*/*/%s'",
                             type, SXM_PRINTF_STR(module), SXM_PRINTF_STR(from),
                             SXM_PRINTF_STR(to));
                }
                SXM_FILE_PATH_FREE(tempt);
            }
            SXM_FILE_PATH_FREE(tempf);
        }
    }

    return result;
}

/***************************************************************************//**
* Makes a copy of the existing file.
*
* \note This is only supported for files of the same type and within the same module.
*
* \param[in] type file type (\ref sxe_file_page)
* \param[in] module the name of the module requesting the file
* \param[in] from file name of the existing file
* \param[in] to the name of the copied file
*
* \retval SXM_E_OK Success
* \retval SXM_E_ERROR internal file operation has failed
*
*******************************************************************************/
int sxm_file_copy(char type, const char *module, const char *from, const char *to)
{
    FILE *pSrcFile;
    SXMResultCode rc;

    pSrcFile = sxm_file_open(type, module, from, "rb");
    if (pSrcFile == NULL)
    {
        rc = SXM_E_ERROR;
    }
    else
    {
        FILE *pDstFile = sxm_file_open(type, module, to, "wb");
        if (pDstFile == NULL)
        {
            fclose(pSrcFile);
            rc = SXM_E_ERROR;
        }
        else {
            byte readBuf[SXM_ARCH_FILE_SECTOR_SIZE];
            size_t bytesRead;

            rc = SXM_E_OK;
            /* Just copy the input file to the output file */
            while ((bytesRead = fread(readBuf, 1, sizeof(readBuf), pSrcFile)) > 0)
            {
                if (fwrite(readBuf, bytesRead, 1, pDstFile) != 1)
                {
                    rc = SXM_E_ERROR;
                    break;
                }
            }
            fclose(pDstFile);
            fclose(pSrcFile);
        }
    }

    return rc;
}

/***************************************************************************//**
* Saves data into the regular file (type ['F'](\ref sxe_file_page))
* 
* \param[in] module the name of the module requesting the file
* \param[in] file the name of the file, relative to the module name
* \param[in] buffer the data to be saved into the file
* \param[out] size the size of the \c buffer in bytes
*
* \retval 0 in case of success and -1 in case of error.
*
*******************************************************************************/
int sxm_file_save(const char *module, const char *file, const void *buffer, size_t size)
{
    int rc;
    FILE *out;

    out = sxm_file_open('F', module, file, "wb");
    if (out == NULL) {
        rc = -1;
    }
    else {
        if ((buffer != NULL) && (size > 0U)) {
            rc = (fwrite(buffer, size, 1U, out) == 1) ? 0 : -1;
        }
        else {
            rc = (size == 0U) ? 0 : -1;
        }
        fclose(out);
    }
    return rc;
}

/***************************************************************************//**
* The size routine returns the size (in bytes) of a file.
* This routine may also be used to check if the file exists or not
* (by checking the return code).
*
* \param[in] type file type (\ref sxe_file_page)
* \param[in] module the name of the module requesting the file
* \param[in] file the name of the file, relative to the module name
* \param[out] size size of the file in bytes in case of success
*
* \return 0 in case of success or -1 in case of error.
*
*******************************************************************************/
int sxm_file_size(char type, const char *module, const char *file, size_t *size) {

    int result = -1;
    if ((module != NULL) && (file != NULL) && (size != NULL))
    {
        SXM_FILE_PATH_ALLOC(temp, SXE_FILE_PATH_MAX_SIZE)
        {
            const char *path;
            path = sxm_makepath(temp, SXE_FILE_PATH_MAX_SIZE, type, module, file);
            if (path != NULL)
            {
                struct stat ss;
                result = stat(path, &ss);
                if (result == 0)
                {
                    *size = (size_t)ss.st_size;
                }
                else if (errno != ENOENT)
                {
                    PLOG_UTL("Failed to get stat for file '%c/%s/%s' as '%s'",
                        type, SXM_PRINTF_STR(module), SXM_PRINTF_STR(file), path);
                }
            }
            else
            {
                PLOG_UTL("No path provided for '%c/%s/%s'",
                    type, SXM_PRINTF_STR(module), SXM_PRINTF_STR(file));
            }
            SXM_FILE_PATH_FREE(temp);
        }
    }

    return result;
}

/***************************************************************************//**
* This function allows to generate the file path via \ref sxm_makepath callback.
*
* \param[in] type file type (\ref sxe_file_page)
* \param[in] module the name of the module requesting the file
* \param[in] file the name of the file, relative to the module name
* \param[out] out a pointer to a character buffer used to contain the final file path
* \param[in] size size of the memory addressed by the \p out 
* 
* \return address of the memory where to find the generated path or NULL in case of error
*
*******************************************************************************/
char *sxm_file_make_path(char type, const char *module, const char *file, char *out, size_t size)
{
    return sxm_makepath(out, size, type, module, file);
}

/***************************************************************************//**
* The log routine formats a message, and presents it for printing through
* the SXe SDK output routine (\ref sxm_logit)
*
* \param[in] fmt the format control (sprintf controls)
* \param[in] ... any remaining arguments
*
*******************************************************************************/
void sxm_file_log(const char *fmt, ...) {
    char buffer[SXM_PBUF_SIZE];
    va_list ap;

    if (NULL != sxm_logit)
    {
        va_start(ap, fmt);
        vsnprintf(buffer, sizeof(buffer), fmt, ap);
        buffer[SXM_PBUF_SIZE - 1] = '\0';
        sxm_logit(buffer);
        va_end(ap);
    }

    return;
}

/***************************************************************************//**
 * The log routine formats a message and presents it for printing through
 * the SXe SDK output routine (\ref sxm_logit).
 *
 * \param[in] debugArea debug area for the reported log message
 * \param[in] debugType debug type for the reported log message
 * \param[in] tag SXe component which the log message belongs to
 * \param[in] prefix prefix to denote call type
 * \param[in] function name of the function which the log is printed from
 * \param[in] fmt the format control (sprintf controls)
 * \param[in] ... any remaining arguments
 *
 ******************************************************************************/
void sxm_file_log_ext(SxmDebugArea debugArea,
                      SxmDebugType debugType,
                      const char *tag,
                      const char *prefix,
                      const char *function,
                      const char *fmt, ...) {
    char buffer[SXM_PBUF_SIZE];
    int prefixLen = 0;
    va_list ap;

    if (NULL != sxm_logit) {
        prefixLen = snprintf(buffer, SXM_PBUF_SIZE,
                             "%s%c%s%c%-10s%c%s%c%s%c",
                             dbgAreaToText[debugArea],
                             SXM_DBG_LOG_COLUMN_DELIMITER,
                             dbgTypeToText[debugType],
                             SXM_DBG_LOG_COLUMN_DELIMITER,
                             tag,
                             SXM_DBG_LOG_COLUMN_DELIMITER,
                             prefix,
                             SXM_DBG_LOG_COLUMN_DELIMITER,
                             function,
                             SXM_DBG_LOG_COLUMN_DELIMITER);
        if ((prefixLen < 0) || (prefixLen >= SXM_PBUF_SIZE)) {
            sxm_file_log("log_ext: insufficient buffer size");
            prefixLen = 0;
        }

        va_start(ap, fmt);
        vsnprintf(&buffer[prefixLen],
                  (size_t)(SXM_PBUF_SIZE - prefixLen),
                  fmt, ap);
        buffer[SXM_PBUF_SIZE - 1] = '\0';
        sxm_logit(buffer);
        va_end(ap);
    }
#ifdef SXM_EXT_LOGGING
    if (NULL != sxm_log_ext) {
        sxm_log_ext(debugArea, debugType, tag,
            prefix, function, &buffer[prefixLen]);
    }
#endif /* #ifdef SXM_EXT_LOGGING */
}

/***************************************************************************//**
* The dump routine prints an area of memory in hex, using the SXe SDK output
* routine (\ref sxm_logit).
*
* \param[in] data a pointer to the area to be dumped
* \param[in] bc the count of bytes to be dumped
*
*******************************************************************************/
void sxm_file_dump(const void *data, size_t bc) {

    char printBuf[SXM_PBUF_SIZE];
    uint ix;
    int printRet;
    int numPrinted = 0;

    if ((NULL == sxm_logit) || (NULL == data) || (bc == 0U)) {
        return;
    }

    for (ix = 0; ix < bc; ix++) {
        /* Print data start index */
        if (0 == (ix & SXM_DUMP_NUM_BYTES_IN_LINE)) {
            numPrinted = snprintf(&printBuf[0],
                                  SXM_PBUF_SIZE,
                                  "%6u:", ix);
            if ((numPrinted < 0) || (numPrinted >= SXM_PBUF_SIZE)) {
                sxm_file_log("dump: insufficient buffer size");
                return;
            }
        }

        /* Print data in hex */
        printRet = snprintf(&printBuf[numPrinted],
                            (size_t)(SXM_PBUF_SIZE - numPrinted),
                            " %02x", *((const byte*)data + ix));
        if ((printRet < 0) || (printRet >= (SXM_PBUF_SIZE - numPrinted))) {
            sxm_file_log("dump: insufficient buffer size");
            return;
        }

        numPrinted += printRet;

        if (SXM_DUMP_NUM_BYTES_IN_LINE == (ix & SXM_DUMP_NUM_BYTES_IN_LINE)) {
            sxm_logit(&printBuf[0]);
        }
    }

    if (0 != (ix & SXM_DUMP_NUM_BYTES_IN_LINE)) {
        sxm_logit(&printBuf[0]);
    }

    return;
}

/***************************************************************************//**
 * The dump routine prints an area of memory in hex, using the SXe SDK output
 * routine (\ref sxm_logit) along with extra parameters.
 *
 * \param[in] debugArea debug area for the reported log message
 * \param[in] debugType debug type for the reported log message
 * \param[in] tag SXe component which the log message belongs to
 * \param[in] prefix prefix to denote call type
 * \param[in] function name of the function which the log is printed from
 * \param[in] data a pointer to the data to be dumped
 * \param[in] bc the count of bytes to be dumped
 *
*******************************************************************************/
void sxm_file_dump_ext(SxmDebugArea debugArea,
                       SxmDebugType debugType,
                       const char *tag,
                       const char *prefix,
                       const char *function,
                       const void *data,
                       size_t bc) {
    char printBuf[SXM_PBUF_SIZE];
    char *pDataBuf;
    uint ix;
    int printRet;
    int numPrinted = 0;
    int dataBufSize;

    if (NULL == sxm_logit) {
        return;
    }

    /* Print prefix */
    printRet = snprintf(printBuf, SXM_PBUF_SIZE,
                        "%s%c%s%c%-10s%c%s%c%s%c",
                        dbgAreaToText[debugArea],
                        SXM_DBG_LOG_COLUMN_DELIMITER,
                        dbgTypeToText[debugType],
                        SXM_DBG_LOG_COLUMN_DELIMITER,
                        tag,
                        SXM_DBG_LOG_COLUMN_DELIMITER,
                        prefix,
                        SXM_DBG_LOG_COLUMN_DELIMITER,
                        function,
                        SXM_DBG_LOG_COLUMN_DELIMITER);
    if ((printRet < 0) || (printRet >= SXM_PBUF_SIZE)) {
        sxm_file_log("dump_ext: insufficient buffer size");
        return;
    }

    pDataBuf = &printBuf[printRet];
    dataBufSize = SXM_PBUF_SIZE - printRet;

    /* Print data */
    for (ix = 0; ix < bc; ix++) {
        /* Print data start index */
        if (0 == (ix & SXM_DUMP_NUM_BYTES_IN_LINE)) {
            numPrinted = snprintf(&pDataBuf[0],
                                  (size_t)dataBufSize,
                                  "%6u:", ix);
            if ((numPrinted < 0) || (numPrinted >= dataBufSize)) {
                sxm_file_log("dump_ext: insufficient buffer size");
                return;
            }
        }

        /* Print data in hex */
        printRet = snprintf(&pDataBuf[numPrinted],
                            (size_t)(dataBufSize - numPrinted),
                            " %02x", *((const byte*)data + ix));
        if ((printRet < 0) || (printRet >= (dataBufSize - numPrinted))) {
            sxm_file_log("dump_ext: insufficient buffer size");
            return;
        }

        numPrinted += printRet;

        if (SXM_DUMP_NUM_BYTES_IN_LINE == (ix & SXM_DUMP_NUM_BYTES_IN_LINE)) {
            /* Print the whole message including prefix */
            sxm_logit(&printBuf[0]);
#ifdef SXM_EXT_LOGGING
            if (NULL != sxm_log_ext) {
                /* Pass data buffer only (without prefix) */
                sxm_log_ext(debugArea, debugType, tag,
                            prefix, function, &pDataBuf[0]);
            }
#endif /* #ifdef SXM_EXT_LOGGING */
        }
    }

    if (0 != (ix & SXM_DUMP_NUM_BYTES_IN_LINE)) {
        /* Print the whole message including prefix */
        sxm_logit(&printBuf[0]);
#ifdef SXM_EXT_LOGGING
        if (NULL != sxm_log_ext) {
            /* Pass data buffer only (without prefix) */
            sxm_log_ext(debugArea, debugType, tag,
                        prefix, function, &pDataBuf[0]);
        }
#endif /* #ifdef SXM_EXT_LOGGING */
    }

    return;
}

/***************************************************************************//**
* The routine sets the function that generates file paths.
*
* \param[in]  make_path_func a pointer to a function that generates file paths.
*                            Refer to #SXMMakePathCallback_t
*                            for the requirements of this function.
*
* \retval SXM_E_OK Success
* \retval SXM_E_FAULT NULL parameter
*
*******************************************************************************/
int sxm_sdk_set_makepath(SXMMakePathCallback_t make_path_func)
{
    if (NULL == make_path_func)
    {
        return SXM_E_FAULT;
    }

    sxm_makepath = make_path_func;
    return SXM_E_OK;
}

/***************************************************************************//**
* The routine gets the current function that generates file paths.
*
* \retval Pointer to currently set makepath function
*
*******************************************************************************/
SXMMakePathCallback_t sxm_sdk_get_makepath( void )
{
    return sxm_makepath;
}

#ifdef SXM_EXT_LOGGING
/***************************************************************************//**
* The routine sets the extended debug handler.
*
* \param[in] log_func a pointer to a function that implements extended
*                     debug handler
*
*******************************************************************************/
void sxm_sdk_set_extlog(SXMExtLogHandler_t log_func)
{
    sxm_log_ext = log_func;

    return;
}
#endif /* #ifdef SXM_EXT_LOGGING */
