/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_csv.c
* \author Sergey Kotov
* \date 9/01/2015
* \brief CSV Parser
*
*******************************************************************************/

#define DEBUG_TAG "csv"

#include <ctype.h>
#include <limits.h>
#include <errno.h>

#include <util/sxm_csv.h>

#ifndef SDKFILES_STANDALONE_BUILD
#include <sxm_stdtrace.h>
#include <util/sxm_common_internal.h>

/** Debug macro redefinition */
#ifndef SXM_DEBUG_PRODUCTION
#undef DEBUG_VAR
static uint debugVar = 0; /* Disabled by default. -1 enables all logging */
#define DEBUG_VAR debugVar
#endif /* #ifndef SXM_DEBUG_PRODUCTION */

#else /* #ifndef SDKFILES_STANDALONE_BUILD */

#include "sdkfiles.h"

#undef ERROR_UTL
#define ERROR_UTL non_fatal

#endif /* #ifndef SDKFILES_STANDALONE_BUILD ... #else ... */

#define SXM_CSV_DIVIDER_MAX (100000)

struct _csv
{
    char separator;
    uint lineSizeMax;
    uint options;

    char *headerLine;
    char *inputLine;
    char **headerColumns;
    char **inputColumns;
    uint numColumnsMax;
    uint currentLine;

    /* Implementation specific for different CSV types
       (SxmCsvBufParser or SxmCsvFileParser) */

    /* Function to read next line from the input to lineBuf */
    int (*readNextLine)(SxmCsvParser *pCsv, char *lineBuf);
    /* Function to return the current position */
    int (*getPos)(SxmCsvParser *pCsv, size_t *pPos);
    /* Function to set (seek) to the specified position */
    int (*setPos)(SxmCsvParser *pCsv, size_t pos);
    /* Function to releases allocated resources */
    void (*release)(SxmCsvParser *pCsv);
};

typedef struct
{
    /* SxmCsvParser must be the first field */
    SxmCsvParser csvParser;

    char *pData;
    size_t dataSize;
    size_t currentIndex;
} SxmCsvBufParser;

typedef struct
{
    /* SxmCsvParser must be the first field */
    SxmCsvParser csvParser;

    FILE *pFile;
} SxmCsvFileParser;

/******************************************************************************
 *                                                                            *
 *            Internal functions                                              *
 *            ==================                                              *
 *                                                                            *
 ******************************************************************************/

/* Helpers */

static int copyField(const char *src, size_t srcSize,
                     char *dst, size_t dstSizeMax,
                     char separator)
{
    const char *srcPointer = src;

    while ((srcSize > 0) && (dstSizeMax > 0))
    {
        if (*srcPointer == separator)
        {
            srcPointer++;
            break;
        }
        else
        {
            *dst++ = *srcPointer++;
            srcSize--;
            dstSizeMax--;
        }
    }

    if (0 == dstSizeMax)
    {
        ERROR_UTL("Max size exceeded. "
                  "No line or field separator found.");
        return -1;
    }
    else
    {
        *dst = '\0';
    }

    return (int)(srcPointer - src);
}

static char * strTrim(char *buffer)
{
    size_t len = strlen(buffer);
    char *start = buffer;

    while ((len > 0) && isspace((unsigned char)(buffer[len - 1])))
    {
        buffer[--len] = '\0';
    }

    /* Skip whitespaces */
    while (isspace((unsigned char)(*start)))
    {
        start++;
    }

    return start;
}

static int strSplit(SxmCsvParser *pCsv,
                    char *input, char separator,
                    char *output[], uint maxColumns)
{
    uint numColumns = 0;

    while ('\0' != *input)
    {
        /* Check if column text is quoted */
        if (('"' == *input) &&
            (0 == (pCsv->options & SXM_CSV_OPTION_IGNORE_QUOTES)))
        {
            char *pResultInput = ++input;

            output[numColumns++] = input;

            /* Move to the closing quotes, saving escaped quotes */
            while ('\0' != *input)
            {
                if ('"' == input[0])
                {
                    if ('"' == input[1])
                    {
                        *(pResultInput++) = '"';
                        input += 2;
                    }
                    else
                    {
                        break;
                    }
                }
                else
                {
                    *(pResultInput++) = *(input++);
                }
            }

            if ('"' != *input)
            {
                ERROR_UTL("Unterminated quoted string."
                          " Line: %u, column: %u",
                          pCsv->currentLine, numColumns);
                return SXM_E_CORRUPT;
            }

            input++;
            *pResultInput = '\0';

            if (('\0' != *input) && (separator != *input))
            {
                ERROR_UTL("Quoted string not followed by separator."
                          " Line: %u, column: %u",
                          pCsv->currentLine, numColumns);
                return SXM_E_CORRUPT;
            }
        }
        else
        {
            output[numColumns++] = input;

            /* Move to the separator or to the string end */
            while (('\0' != *input) && (separator != *input))
            {
                input++;
            }
        }

        if (separator == *input)
        {
            *input = '\0';

            if (numColumns == maxColumns)
            {
                ERROR_UTL("Too many columns (%u > %u). Line: %u",
                          numColumns + 1, maxColumns, pCsv->currentLine);
                return SXM_E_CORRUPT;
            }

            input++;

            /* Check if last column is empty */
            if ('\0' == *input)
            {
                output[numColumns++] = input;
            }
        }
    }

    return SXM_E_OK;
}

static int strToInt(const char *in, int *pRet)
{
    if (isspace((unsigned char)(in[0])))
    {
        ERROR_UTL("Cannot convert '%s' to int: "
                  "white space encountered", in);
        return SXM_E_CORRUPT;
    }
    else
    {
        char *pEndPtr;
        long int lVal;

        errno = 0;
        lVal = strtol(in, &pEndPtr, 10);

        if ((pEndPtr == in) || (pEndPtr[0] != '\0') ||
            (((LONG_MAX == lVal) || (LONG_MIN == lVal)) && (errno == ERANGE)))
        {
            ERROR_UTL("Cannot convert '%s' to int", in);
            return SXM_E_CORRUPT;
        }
        else if ((lVal > INT_MAX) || (lVal < INT_MIN))
        {
            ERROR_UTL("Value '%s' is out of limit", in);
            return SXM_E_CORRUPT;
        }
        else
        {
            *pRet = (int)lVal;
        }
    }

    return SXM_E_OK;
}

static int initCsvParser(SxmCsvParser *pCsv)
{
    if (0 == (pCsv->options & SXM_CSV_OPTION_NO_HEADER))
    {
        int rc;
        char *inputPointer;

        pCsv->headerLine = (char *)sxe_malloc(pCsv->lineSizeMax);
        if (NULL == pCsv->headerLine)
        {
            ERROR_UTL("Failed to allocate memory for headerLine");
            return SXM_E_NOMEM;
        }

        /* Read header line */
        rc = pCsv->readNextLine(pCsv, pCsv->headerLine);
        if (SXM_E_OK != rc)
        {
            return rc;
        }

        pCsv->currentLine = 1;

        /* Calculate number of columns */
        inputPointer = strTrim(pCsv->headerLine);
        pCsv->numColumnsMax = 1;
        while ('\0' != *inputPointer)
        {
            if (*inputPointer == pCsv->separator)
            {
                pCsv->numColumnsMax++;
            }
            inputPointer++;
        }

        pCsv->headerColumns = (char **)sxe_calloc(pCsv->numColumnsMax, sizeof(char *));
        if (NULL == pCsv->headerColumns)
        {
            ERROR_UTL("Failed to allocate memory for headerColumns");
            return SXM_E_NOMEM;
        }

        rc = strSplit(pCsv, pCsv->headerLine, pCsv->separator,
                      pCsv->headerColumns, pCsv->numColumnsMax);
        if (SXM_E_OK != rc)
        {
            ERROR_UTL("Failed to split line %u", pCsv->currentLine);
            return rc;
        }
    }
    else
    {
        /* Set maximum possible number of columns */
        pCsv->numColumnsMax =
            (pCsv->lineSizeMax / 2) + (pCsv->lineSizeMax % 2);
    }

    pCsv->inputLine = (char *)sxe_malloc(pCsv->lineSizeMax);
    if (NULL == pCsv->inputLine)
    {
        ERROR_UTL("Failed to allocate memory for inputLine");
        return SXM_E_NOMEM;
    }

    pCsv->inputColumns =  (char **)sxe_calloc(pCsv->numColumnsMax, sizeof(char*));
    if (NULL == pCsv->inputColumns)
    {
        ERROR_UTL("Failed to allocate memory for inputColumns");
        return SXM_E_NOMEM;
    }

    return SXM_E_OK;
}

static void deinitCsvParser(SxmCsvParser *pCsv)
{
    if (NULL != pCsv->headerLine)
    {
        sxe_free(pCsv->headerLine);
        pCsv->headerLine = NULL;
    }

    if (NULL != pCsv->inputLine)
    {
        sxe_free(pCsv->inputLine);
        pCsv->inputLine = NULL;
    }

    if (NULL != pCsv->headerColumns)
    {
        sxe_free(pCsv->headerColumns);
        pCsv->headerColumns = NULL;
    }

    if (NULL != pCsv->inputColumns)
    {
        sxe_free(pCsv->inputColumns);
        pCsv->inputColumns = NULL;
    }

    return;
}

static int getStrVal(SxmCsvParser *pCsv, uint index, const char **ppRet)
{
    int rc;

    if (index >= pCsv->numColumnsMax)
    {
        ERROR_UTL("Invalid CSV index: %u > %u",
                  index, pCsv->numColumnsMax);
        rc = SXM_E_INVAL;
    }
    else if (NULL == pCsv->inputColumns[index])
    {
        rc = SXM_E_UNSUPPORTED;
    }
    else
    {
        if ('\0' == *pCsv->inputColumns[index])
        {
            rc = SXM_E_NOENT;
        }
        else
        {
            rc = SXM_E_OK;
        }

        *ppRet = pCsv->inputColumns[index];
    }

    return rc;
}

static int getIntVal(SxmCsvParser *pCsv, uint index, int *pRet)
{
    const char *strVal;
    int rc = getStrVal(pCsv, index, &strVal);

    if (SXM_E_OK == rc)
    {
        rc = strToInt(strVal, pRet);
    }

    return rc;
}

static int strToFix(const char *in, fix *pRet)
{
    uint intPart = 0;
    uint fractionPart = 0;
    uint fractionDivider = 1;
    int sign = 1;
    const char *inPointer = in;

    /* Sign */
    if ('-' == *inPointer)
    {
        inPointer++;
        sign = -1;
    }

    /* Integer part */
    while (isdigit((unsigned char)(*inPointer)))
    {
        intPart = intPart * 10U + (uint)(*inPointer++ - '0');
    }

    /* Fraction */
    if (*inPointer == '.')
    {
        inPointer++;

        while (isdigit((unsigned char)(*inPointer)))
        {
            fractionPart = fractionPart * 10U + (uint)(*inPointer++ - '0');
            fractionDivider *= 10;
            if (SXM_CSV_DIVIDER_MAX == fractionDivider)
            {
                break;
            }
        }
    }

    if ((SXM_CSV_DIVIDER_MAX != fractionDivider) && ('\0' != *inPointer))
    {
        ERROR_UTL("Cannot convert '%s' to fix: "
                  "unacceptable character encountered: '%c'",
                  in, *inPointer);
        return SXM_E_CORRUPT;
    }

    *pRet = sign * (int)((intPart << 15) + ((fractionPart << 15) / fractionDivider));

    return SXM_E_OK;
}

static int getFixVal(SxmCsvParser *pCsv, uint index, fix *pRet)
{
    const char *strVal;
    int rc = getStrVal(pCsv, index, &strVal);

    if (SXM_E_OK == rc)
    {
        rc = strToFix(strVal, pRet);
    }

    return rc;
}

/* CSV Data Buffer Parser implementation */

static int csvBufParserReadNextLine(SxmCsvParser *pCsvParser,
                                    char *lineBuf)
{
    SxmCsvBufParser *pBufParser = (SxmCsvBufParser *)pCsvParser;
    int bytesRead;

    bytesRead = copyField(&pBufParser->pData[pBufParser->currentIndex],
                          pBufParser->dataSize - pBufParser->currentIndex,
                          lineBuf, pCsvParser->lineSizeMax, '\n');
    if (bytesRead == 0)
    {
        return SXM_E_NOENT;
    }
    else if (bytesRead < 0)
    {
        ERROR_UTL("Failed to read CSV line from buffer");
        return SXM_E_CORRUPT;
    }

    pBufParser->currentIndex += (uint)bytesRead;
    return SXM_E_OK;
}

static int csvBufParserGetPos(SxmCsvParser *pCsvParser, size_t *pPos)
{
    *pPos = ((SxmCsvBufParser*)pCsvParser)->currentIndex;

    return SXM_E_OK;
}

static int csvBufParserSetPos(SxmCsvParser *pCsvParser, size_t pos)
{
     SxmCsvBufParser *pBufParser = (SxmCsvBufParser*)pCsvParser;

    if (pos >= pBufParser->dataSize)
    {
        return SXM_E_NOENT;
    }

    pBufParser->currentIndex = pos;
    return SXM_E_OK;
}

static void csvBufParserRelease(SxmCsvParser *pCsvParser)
{
    SxmCsvBufParser *pBufParser = (SxmCsvBufParser*)pCsvParser;

    if (0 != (pBufParser->csvParser.options & SXM_CSV_OPTION_DATA_OWNER))
    {
        sxe_free((void*)pBufParser->pData);
    }

    deinitCsvParser(pCsvParser);
    sxe_free(pCsvParser);

    return;
}

static int createBufParser(char *pData,
                           size_t dataSize,
                           uint lineSizeMax,
                           uint options,
                           char separator,
                           SxmCsvParser **ppRet)
{
    int rc = SXM_E_OK;
    SxmCsvBufParser *pBufParser =
        (SxmCsvBufParser*)sxe_calloc(1, sizeof(SxmCsvBufParser));

    if (NULL == pBufParser)
    {
        ERROR_UTL("Failed to allocate memory for SxmCsvBufParser");
        return SXM_E_NOMEM;
    }

    pBufParser->pData = pData;
    pBufParser->dataSize = dataSize;

    pBufParser->csvParser.separator = separator;
    pBufParser->csvParser.lineSizeMax = lineSizeMax;
    pBufParser->csvParser.options = options;
    pBufParser->csvParser.readNextLine = csvBufParserReadNextLine;
    pBufParser->csvParser.getPos = csvBufParserGetPos;
    pBufParser->csvParser.setPos = csvBufParserSetPos;
    pBufParser->csvParser.release = csvBufParserRelease;

    rc = initCsvParser(&pBufParser->csvParser);
    if (SXM_E_OK != rc)
    {
        pBufParser->csvParser.options = (uint)(pBufParser->csvParser.options & 
            ((uint)(~SXM_CSV_OPTION_DATA_OWNER)));
        csvBufParserRelease(&pBufParser->csvParser);
    }
    else
    {
        *ppRet = &pBufParser->csvParser;
    }

    return rc;
}

/* CSV File Parser implementation */

static int csvFileParserReadNextLine(SxmCsvParser *pCsvParser,
                                     char *lineBuf)
{
    SxmCsvFileParser *pCsvFileParser = (SxmCsvFileParser *)pCsvParser;

    lineBuf[pCsvParser->lineSizeMax - 2] = '\0';

    if (NULL != fgets(lineBuf, (int)pCsvParser->lineSizeMax, pCsvFileParser->pFile))
    {
        if (0 == feof(pCsvFileParser->pFile))
        {
            if ((lineBuf[pCsvParser->lineSizeMax - 2] != '\0') &&
                (lineBuf[pCsvParser->lineSizeMax - 2] != '\n'))
            {
                ERROR_UTL("Line size exceeds the max size (%u)",
                          pCsvParser->lineSizeMax);
                return SXM_E_CORRUPT;
            }
        }

        return SXM_E_OK;
    }
    else if (0 != feof(pCsvFileParser->pFile))
    {
        return SXM_E_NOENT;
    }

    ERROR_UTL("Failed to read string from CSV file");
    return SXM_E_PIPE;
}

static int csvFileParserGetPos(SxmCsvParser *pCsvParser, size_t *pPos)
{
    SxmCsvFileParser *pCsvFileParser = (SxmCsvFileParser *)pCsvParser;
    long int pos = ftell(pCsvFileParser->pFile);
    if (-1 == pos)
    {
        return SXM_E_PIPE;
    }

    *pPos = (uint)pos;
    return SXM_E_OK;
}

static int csvFileParserSetPos(SxmCsvParser *pCsvParser, size_t pos)
{
    SxmCsvFileParser *pCsvFileParser = (SxmCsvFileParser *)pCsvParser;

    if (0 != fseek(pCsvFileParser->pFile, (long)pos, SEEK_SET))
    {
        return SXM_E_PIPE;
    }

    return SXM_E_OK;
}

static void csvFileParserRelease(SxmCsvParser *pCsvParser)
{
    SxmCsvFileParser *pCsvFileParser = (SxmCsvFileParser *)pCsvParser;

    deinitCsvParser(pCsvParser);

    if (NULL != pCsvFileParser->pFile)
    {
        fclose(pCsvFileParser->pFile);
    }

    sxe_free(pCsvFileParser);

    return;
}

static int createFileParser(const char* filePath, uint lineSizeMax,
                            char separator, uint options, SxmCsvParser **ppRet)
{
    int rc = SXM_E_OK;
    SxmCsvFileParser *pFileParser =
        (SxmCsvFileParser *)sxe_calloc(1, sizeof(SxmCsvFileParser));

    if (NULL == pFileParser)
    {
        ERROR_UTL("Failed to allocate memory for SxmCsvFileParser");
        return SXM_E_NOMEM;
    }

    pFileParser->pFile = fopen(filePath, "rb");
    if (NULL == pFileParser->pFile)
    {
        ERROR_UTL("Failed to open file: %s", filePath);
        sxe_free(pFileParser);
        return SXM_E_PIPE;
    }

    pFileParser->csvParser.separator = separator;
    pFileParser->csvParser.lineSizeMax = lineSizeMax;
    pFileParser->csvParser.options = options;
    pFileParser->csvParser.readNextLine = csvFileParserReadNextLine;
    pFileParser->csvParser.getPos = csvFileParserGetPos;
    pFileParser->csvParser.setPos = csvFileParserSetPos;
    pFileParser->csvParser.release = csvFileParserRelease;

    rc = initCsvParser(&pFileParser->csvParser);
    if (SXM_E_OK != rc)
    {
        csvFileParserRelease(&pFileParser->csvParser);
    }
    else
    {
        *ppRet = &pFileParser->csvParser;
    }

    return rc;
}

static int loadFileToBuffer(const char* filePath, void **ppBuf, uint *pBufSize)
{
    int rc = SXM_E_OK;
    FILE *pFile = NULL;
    uint fileSize = 0;
    void *pBuf = NULL;

    switch(0) { default: {

        pFile = fopen(filePath, "rb");
        if (NULL == pFile)
        {
            ERROR_UTL("Failes to open file: %s", filePath);
            rc = SXM_E_PIPE;
            break;
        }

        /* Calculate file size */
        if ((fseek(pFile, 0, SEEK_END)) == -1)
        {
            ERROR_UTL("Failed to seek in file: %s", filePath);
            rc = SXM_E_PIPE;
            break;
        }

        fileSize = (uint)ftell(pFile);

        /* Allocate buffer */
        pBuf = sxe_malloc(fileSize);
        if (NULL == pBuf)
        {
            ERROR_UTL("Failed to allocate %u bytes", fileSize);
            rc = SXM_E_NOMEM;
            break;
        }

        /* Copy data from file to buffer */
        if ((fseek(pFile, 0, SEEK_SET)) == -1)
        {
            ERROR_UTL("Failed to seek in file: %s", filePath);
            rc = SXM_E_PIPE;
            break;
        }

        if (fread(pBuf, fileSize, 1, pFile) != 1)
        {
            ERROR_UTL("Failed to read file: %s", filePath);
            rc = SXM_E_PIPE;
            break;
        }
    }}

    if (SXM_E_OK == rc)
    {
        *ppBuf = pBuf;
        *pBufSize = fileSize;
    }
    else if (NULL != pBuf)
    {
        sxe_free(pBuf);
    }

    if (NULL != pFile)
    {
        fclose(pFile);
    }

    return rc;
}

static int createBufParserFromFile(const char* filePath,
                                   uint lineSizeMax,
                                   char separator,
                                   uint options,
                                   SxmCsvParser **ppRet)
{
    int rc = SXM_E_OK;
    void *pBuf = NULL;
    uint bufSize = 0;

    rc = loadFileToBuffer(filePath, &pBuf, &bufSize);
    if (SXM_E_OK != rc)
    {
        return rc;
    }

    rc = createBufParser(pBuf, bufSize,
                         lineSizeMax,
                         options | SXM_CSV_OPTION_DATA_OWNER,
                         separator, ppRet);
    if (SXM_E_OK != rc)
    {
        sxe_free(pBuf);
        ERROR_UTL("Failed to create CSV parser");
        return rc;
    }

    return SXM_E_OK;
}

/******************************************************************************
 *                                                                            *
 *            Interface Functions                                             *
 *            ===================                                             *
 *                                                                            *
 ******************************************************************************/

/***************************************************************************//**
 *
 * Create CSV Parser which uses data buffer as input.
 *
 * \param[in] pData a pointer to the input data buffer
 * \param[in] dataSize input buffer size
 * \param[in] lineSizeMax maximum size of a single CSV line
 *                        (including line terminator)
 * \param[in] options a combination of SXM_CSV_OPTION_... bit flags
 * \param[in] separator CSV fields separator
 * \param[out] ppRet a pointer to a pointer to a SxmCsvParser structure
 *                   which is allocated and initialized by the call
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_INVAL Invalid parameter
 * \retval SXM_E_NOMEM Out of memory
 *
 ******************************************************************************/
int sxm_csv_buffer_new(char *pData, size_t dataSize, uint lineSizeMax,
                       uint options, char separator, SxmCsvParser **ppRet)
{
    int rc;

    if ((NULL == pData) || (NULL == ppRet))
    {
        rc = SXM_E_FAULT;
    }
    else if ((0 == dataSize) || (0 == lineSizeMax))
    {
        rc = SXM_E_INVAL;
    }
    else
    {
        rc = createBufParser(pData, dataSize, lineSizeMax,
                             options, separator, ppRet);
    }

    return rc;
}

/***************************************************************************//**
 *
 * Create CSV Parser which uses a file as input.
 *
 * \param[in] fileName input file name
 * \param[in] lineSizeMax maximum size of a single CSV line
 * \param[in] separator CSV fields separator
 * \param[in] options a combination of SXM_CSV_OPTION_... bit flags
 * \param[out] ppRet a pointer to a pointer to a SxmCsvParser structure
 *                   which is allocated and initialized by the call
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_INVAL Invalid parameter
 * \retval SXM_E_NOMEM Out of memory
 * \retval SXM_E_NOENT file is empty
 * \retval SXM_E_CORRUPT No line separator found
 *
 ******************************************************************************/
int sxm_csv_file_new(const char* fileName, uint lineSizeMax,
                     char separator, uint options, SxmCsvParser **ppRet)
{
    int rc;

    if ((NULL == fileName) || (NULL == ppRet))
    {
        rc = SXM_E_FAULT;
    }
    else if (0 == lineSizeMax)
    {
        rc = SXM_E_INVAL;
    }
    else
    {
        if (0 != (options & SXM_CSV_OPTION_LOAD_TO_MEMORY))
        {
            rc = createBufParserFromFile(fileName,
                                         lineSizeMax,
                                         separator,
                                         options,
                                         ppRet);
        }
        else
        {
            rc = createFileParser(fileName,
                                  lineSizeMax,
                                  separator,
                                  options,
                                  ppRet);
        }
    }

    return rc;
}

/***************************************************************************//**
 *
 * Delete CSV parser
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 *
 ******************************************************************************/
void sxm_csv_delete(SxmCsvParser *pCsv)
{
    if (pCsv)
    {
        pCsv->release(pCsv);
    }

    return;
}

/***************************************************************************//**
 *
 * Get current position in CSV Parser.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 * \param[out] pPos a pointer to a variable filled in with the current position
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_PIPE File I/O failed
 *
 ******************************************************************************/
int sxm_csv_get_pos(SxmCsvParser *pCsv, size_t *pPos)
{
    int rc;

    if ((NULL == pCsv) || (NULL == pPos))
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        rc = pCsv->getPos(pCsv, pPos);
    }

    return rc;
}

/***************************************************************************//**
 *
 * Set current position in CSV Parser.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 * \param[in] pos specified position to set
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_NOENT pos is out of range
 * \retval SXM_E_PIPE I/O operation failed
 *
 ******************************************************************************/
int sxm_csv_set_pos(SxmCsvParser *pCsv, size_t pos)
{
    int rc;

    if (NULL == pCsv)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        rc = pCsv->setPos(pCsv, pos);
    }

    return rc;
}

/***************************************************************************//**
 *
 * Reads next CSV line without parsing.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_PIPE File read failed
 * \retval SXM_E_NOENT No more lines to read
 *
 ******************************************************************************/
int sxm_csv_skip_line(SxmCsvParser *pCsv)
{
     int rc = pCsv->readNextLine(pCsv, pCsv->inputLine);
     if (SXM_E_OK == rc)
     {
         pCsv->currentLine++;
     }

     return rc;
}

/***************************************************************************//**
 *
 * Reset CSV Parser to start reading from the beginning.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_PIPE File I/O failed
 *
 ******************************************************************************/
int sxm_csv_reset(SxmCsvParser *pCsv)
{
    int rc = sxm_csv_set_pos(pCsv, 0);

    if (SXM_E_OK == rc)
    {
        /* Reset line number */
        pCsv->currentLine = 0;

        if (0 == (pCsv->options & SXM_CSV_OPTION_NO_HEADER))
        {
            /* Skip header */
            rc = sxm_csv_skip_line(pCsv);
        }
    }

    return rc;
}

/***************************************************************************//**
 *
 * Read and parse next CSV line.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_PIPE File read failed
 * \retval SXM_E_NOENT No more lines to read
 * \retval SXM_E_CORRUPT The number of columns in the read line is greater
 *                       than the number of columns in the header line.
 *
 ******************************************************************************/
int sxm_csv_next_line(SxmCsvParser *pCsv)
{
    int rc;

    if (NULL == pCsv)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        rc = pCsv->readNextLine(pCsv, pCsv->inputLine);
        if (SXM_E_OK == rc)
        {
            pCsv->currentLine++;

            memset(pCsv->inputColumns, 0, pCsv->numColumnsMax * sizeof(char*));

            rc = strSplit(pCsv, strTrim(pCsv->inputLine), pCsv->separator,
                          pCsv->inputColumns, pCsv->numColumnsMax);
            if (SXM_E_OK != rc)
            {
                ERROR_UTL("Failed to split line %u", pCsv->currentLine);
            }
        }
    }

    return rc;
}

/***************************************************************************//**
 *
 * Get column index by column name (case insensetive).
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 * \param[in] colName a column name
 *
 * \return Valid index if colName is valid and is presented in the header line
 *         or negative value otherwise
 *
 ******************************************************************************/
int sxm_csv_index(SxmCsvParser *pCsv, const char *colName)
{
    uint i;

    if ((NULL == pCsv) || (NULL == colName))
    {
        return -1;
    }

    for (i = 0; i < pCsv->numColumnsMax; i++)
    {
        if (0 == strcmp(colName, pCsv->headerColumns[i]))
        {
            return (int)i;
        }
    }

    return -1;
}

/***************************************************************************//**
 *
 * Return string value of the column cpecified by column index.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 * \param[in] index a column index
 * \param[out] SXe error code (Can be NULL if not required)
 *             SXM_E_OK Success
 *             SXM_E_FAULT NULL parameter
 *             SXM_E_INVAL Invalid parameter
 *             SXM_E_UNSUPPORTED The column is not present in the row
 *             SXM_E_NOENT Column has empty value
 *
 * \return String value pointer on success, or NULL otherwise
 *
 ******************************************************************************/
const char * sxm_csv_str_val(SxmCsvParser *pCsv,
                             uint index,
                             int *pRc)
{
    int rc;
    const char *pStrVal = NULL;

    if (NULL == pCsv)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        rc = getStrVal(pCsv, index, &pStrVal);
    }

    if (pRc)
    {
        *pRc = rc;
    }

    return pStrVal;
}

/***************************************************************************//**
 *
 * Get string value of the column cpecified by column index.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 * \param[in] index a column index
 * \param[out] pDstBuf a pointer to a destination buffer
 * \param[in] dstBufSize destination buffer size
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_UNSUPPORTED The column is not present in the row
 * \retval SXM_E_NOENT Column has empty value
 * \retval SXM_E_RESOURCE Insufficient buffer size. Truncated string is written
 *                        into the destination buffer, '\0' is appended.
 *
 ******************************************************************************/
int sxm_csv_get_str_val(SxmCsvParser *pCsv, uint index,
                        char *pDstBuf, uint dstBufSize)
{
    int rc;

    if ((NULL == pCsv) || (NULL == pDstBuf))
    {
        rc = SXM_E_FAULT;
    }
    else if (0 == dstBufSize)
    {
        rc = SXM_E_INVAL;
    }
    else
    {
        const char *pSrc;
        rc = getStrVal(pCsv, index, &pSrc);
        if (SXM_E_OK == rc)
        {
            rc = copyField(pSrc, dstBufSize,
                           pDstBuf, dstBufSize, '\0');
            if (rc < 0)
            {
                /* Max size exceeded.
                   Append '\0' and indicate this case
                   by the appropriate return code. */
                pDstBuf[dstBufSize - 1] = '\0';
                rc = SXM_E_RESOURCE;
            }
            else
            {
                rc = SXM_E_OK;
            }
        }
    }

    return rc;
}

/***************************************************************************//**
 *
 * Get a single character value of the column cpecified by column index.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 * \param[in] index a column index
 * \param[out] pRet a pointer to a character which is filled by the call
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_INVAL Invalid parameter
 * \retval SXM_E_UNSUPPORTED The column is not present in the row
 * \retval SXM_E_NOENT Column has empty value
 *
 ******************************************************************************/
int sxm_csv_get_char_val(SxmCsvParser *pCsv, uint index, char *pRet)
{
    int rc;

    if ((NULL == pCsv) || (NULL == pRet))
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        const char *pSrc;
        rc = getStrVal(pCsv, index, &pSrc);
        if (SXM_E_OK == rc)
        {
            if ('\0' != pSrc[1])
            {
                ERROR_UTL("The field at index %u "
                          "is longer than 1 character", index);
                rc = SXM_E_INVAL;
            }
            else
            {
                *pRet = pSrc[0];
            }
        }
        else if (SXM_E_NOENT == rc)
        {
            *pRet = '\0';
        }
    }

    return rc;
}

/***************************************************************************//**
 *
 * Return integer value of the column cpecified by column index.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 * \param[in] index a column index
 * \param[out] SXe error code (Can be NULL if not required)
 *             SXM_E_OK Success
 *             SXM_E_FAULT NULL parameter
 *             SXM_E_INVAL Invalid parameter
 *             SXM_E_CORRUPT Invalid column value
 *             SXM_E_UNSUPPORTED The column is not present in the row
 *
 * \return value of the column cpecified by column index, 0 in case of error
 *
 ******************************************************************************/
extern SXESDK_INTERNAL_API int sxm_csv_int_val(SxmCsvParser *pCsv,
                                               uint index,
                                               int *pRc)
{
    int rc;
    int retVal = 0;

    if (NULL == pCsv)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        rc = getIntVal(pCsv, index, &retVal);
    }

    if (pRc)
    {
        *pRc = rc;
    }

    return retVal;
}

/***************************************************************************//**
 *
 * Get integer value of the column cpecified by column index.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 * \param[in] index a column index
 * \param[out] pRet a pointer to an integer which is filled by the call
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_INVAL Invalid parameter
 * \retval SXM_E_CORRUPT Invalid column value
 * \retval SXM_E_UNSUPPORTED The column is not present in the row
 *
 ******************************************************************************/
int sxm_csv_get_int_val(SxmCsvParser *pCsv, uint index, int *pRet)
{
    if ((NULL == pCsv) || (NULL == pRet))
    {
        return SXM_E_FAULT;
    }

    return getIntVal(pCsv, index, pRet);
}

/***************************************************************************//**
 *
 * Get fix value of the column cpecified by column index.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 * \param[in] index a column index
 * \param[out] pRet a pointer to a fix variable which is filled by the call
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_INVAL Invalid parameter
 * \retval SXM_E_CORRUPT Invalid column value
 * \retval SXM_E_UNSUPPORTED The column is not present in the row
 *
 ******************************************************************************/
extern SXESDK_INTERNAL_API int sxm_csv_get_fix_val(SxmCsvParser *pCsv,
                                                   uint index,
                                                   fix *pRet)
{
    if ((NULL == pCsv) || (NULL == pRet))
    {
        return SXM_E_FAULT;
    }

    return getFixVal(pCsv, index, pRet);
}

/***************************************************************************//**
 *
 * Return current line number.
 *
 * \param[in] pCsv a pointer to a SxmCsvParser structure
 *
 * \return Current line number
 *
 ******************************************************************************/
uint sxm_csv_current_line(SxmCsvParser *pCsv)
{
    return pCsv->currentLine;
}
