/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_crc.c
* \author Leslie French
* \date 8/20/2013
* \brief CRC checkers
*
*******************************************************************************/

#include <util/sxm_common_internal.h>

#ifndef SXM_USE_FAST_CRC
#define SXM_USE_FAST_CRC        1
#endif

/*
 * When generating fast CRC calculation code (SXM_USE_FAST_CRC == 1), the preprocessor
 * code below automatically detects whether the target platform is little endian or not.
 * For the case of little endian target platform some additional optimizations are applied.
 * The automatic target platform type detection can be turned off by defining one of
 * the following:
 *
 * Define CRC32_LITTLE_ENDIAN for the case SXe code is compiled for a little endian
 * platform but the preprocessor code below is not able to recognize it.
 *
 * Define CRC32_ENDIAN_INDEPENDENT for the case when little endian platform optimization
 * has to be avoided in CRC calculation. In this case the generated CRC calculation code
 * will be suitable for any target platform type.
 */

#if SXM_USE_FAST_CRC
#  if !defined(CRC32_LITTLE_ENDIAN) && !defined(CRC32_ENDIAN_INDEPENDENT)     // autodetect endianess
#    if defined(_MSC_VER)               // MS Visual Studio -- always little endian
#      define CRC32_LITTLE_ENDIAN
#    elif defined(__GLIBC__)            // glibc
#      include <endian.h>
#      if (__BYTE_ORDER == __LITTLE_ENDIAN)
#        define CRC32_LITTLE_ENDIAN
#      endif
#    elif defined(__BYTE_ORDER__)       // gcc
#      if (__BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__)
#        define CRC32_LITTLE_ENDIAN
#      endif
#    elif defined(__QNX__)              // qnx
#      include <gulliver.h>
#      if defined(__LITTLEENDIAN__)
#        define CRC32_LITTLE_ENDIAN
#      endif
#    endif
#  endif
#endif

/************************************************************************
 *                                                                      *
 *            CRC generator -- 32 bit XM version                        *
 *            ==================================                        *
 *                                                                      *
 ************************************************************************/

static uint crc32_table[256];                   /**< Precomputed values for CRC-32 */

#if SXM_USE_FAST_CRC
static uint crc32_table_ext[7][256];            /**< Additional precomputed values for CRC-32 */

/***************************************************************************//**
 * CRC-32 additive calculation -- slicing-by-8 implementation
 *
 * \param[in] c init polynomial or crc from previous run
 * \param[in] buf input data
 * \param[in] len size of the \c buf in bytes
 *
 * \return calculated value
 *
 ******************************************************************************/
static uint sxm_crc32_part_fast(uint c, const void *buf, uint len)
{
    const byte * p = (const byte *)buf;         // current byte/block
    const byte * pe = p + len - 8U;              // the beginning of last block for slicing-by-8

    // In case of empty buffer just return the same value which was passed
    // with no modifications
    if (!buf || !len)
    {
        return c;
    }

    if (p <= pe)        // check whether the buffer is big enough for slicing-by-8 algorithm
    {
#ifdef CRC32_LITTLE_ENDIAN
        // ensure the pointer is properly aligned
        const byte * pa = p + (sizeof(UINT32) - 1 - ((size_t)(p - 1) % sizeof(UINT32)));
        while (p < pa)
            c = crc32_table[(c & 0xffU) ^ *p++] ^ (c >> 8U);
#endif

        for (;p <= pe; p += 8)  // slicing-by-8 main loop
        {
#ifdef CRC32_LITTLE_ENDIAN
            // little endian platform allows reading whole 32-bit word at once
            uint u = *(const UINT32 *)(const void *)p ^ c;
            uint v = *(const UINT32 *)(const void *)(p + 4);

            c = crc32_table_ext[6][u & 0xFF] ^
                crc32_table_ext[5][(u >> 8U) & 0xFF] ^
                crc32_table_ext[4][(u >> 16U) & 0xFF] ^
                crc32_table_ext[3][u >> 24U] ^
                crc32_table_ext[2][v & 0xFF] ^
                crc32_table_ext[1][(v >> 8U) & 0xFF] ^
                crc32_table_ext[0][(v >> 16U) & 0xFF] ^
                crc32_table[v >> 24U];
#else
            // other platforms use endian-independent way
            uint u = (p[0] + (p[1] << 8U) + (p[2] << 16U) + (p[3] << 24U)) ^ c;

            c = crc32_table_ext[6][u & 0xFF] ^
                crc32_table_ext[5][(u >> 8U) & 0xFF] ^
                crc32_table_ext[4][(u >> 16U) & 0xFF] ^
                crc32_table_ext[3][u >> 24U] ^
                crc32_table_ext[2][p[4]] ^
                crc32_table_ext[1][p[5]] ^
                crc32_table_ext[0][p[6]] ^
                crc32_table[p[7]];
#endif
        }
    }

    // tail of the buffer
    while (p < pe + 8U)
        c = crc32_table[(c & 0xff) ^ *p++] ^ (c >> 8U);

    return c;
}
#else
/***************************************************************************//**
 * CRC-32 additive calculation -- standard implementation
 *
 * \param[in] c init polynomial or crc from previous run
 * \param[in] buf input data
 * \param[in] len size of the \c buf in bytes
 *
 * \return calculated value
 *
 ******************************************************************************/
static uint sxm_crc32_part_std(uint c, const void *buf, uint len)
{
    uint i;
    for (i = 0; i < len; i++)
        c = crc32_table[(c ^ *((const byte*)buf + i)) & 0xff] ^ (c >> 8U);
    return c;
}
#endif

/***************************************************************************//**
 * CRC-32 additive calculation
 *
 * \param[in] c init polynomial or crc from previous run
 * \param[in] buf input data
 * \param[in] len size of the \c buf in bytes
 *
 * \return calculated value
 *
 ******************************************************************************/
uint sxm_crc32_part(uint c, const void *buf, uint len)
{
#if SXM_USE_FAST_CRC
    return sxm_crc32_part_fast(c, buf, len);
#else
    return sxm_crc32_part_std(c, buf, len);
#endif
}

/***************************************************************************//**
 * CRC-32 calculation (init polynomial is \ref SXM_CRC32_INIT_VAL)
 *
 * \param[in] buf input data
 * \param[in] len size of the \c buf in bytes
 *
 * \return calculated value
 *
 ******************************************************************************/
uint sxm_crc32_calculate(const void *buf, uint len)
{
    return SXM_CRC32_INIT_VAL ^ sxm_crc32_part(SXM_CRC32_INIT_VAL, buf, len);
}

/***************************************************************************//**
 * CRC-32 Initialization routine
 *
 * \note This function should be called during SXe SDK initialization.
 *
 ******************************************************************************/
void sxm_crc32_initialize(void)
{
    uint c;
    uint n, k;

    // Generate CRC table
    for (n = 0; n < 256; n++)
    {
        c = n;
        for (k = 0; k < 8; k++)
            if (c & 1)
                c = SXM_CRC32_POLYNOMIAL ^ (c >> 1);
            else
                c = c >> 1;

        crc32_table[n] = c;
    }

#if SXM_USE_FAST_CRC
    // Generate additional CRC tables
    for (n = 0; n < 256; n++)
    {
        c = crc32_table[n];
        for (k = 0; k < 7; k++)
        {
            c = crc32_table[c & 0xFF] ^ (c >> 8);
            crc32_table_ext[k][n] = c;
        }
    }
#endif
}

/***************************************************************************//**
 * Check the buffer check sum over the provided data which includes
 * it's own CRC-32 value.
 *
 * Access units contain a 32-bit cyclic redundancy check word in the last 4
 * bytes of the payload.  The check routine calculates the CRC value of the
 * supplied buffer, excluding the last 4 bytes, stores the value in supplied
 * crc variable, and returns true if it also matches the value in the last 4
 * bytes of the buffer.
 *
 * \param[in] pBuff input data
 * \param[in] iBufSize size of the \c buf in bytes
 * \param[out] pCrc calculation result. Can be NULL if it's not needed.
 *
 * \retval 1 if the CRC-32 matches the buffer value
 * \retval 0 otherwise
 *
 ******************************************************************************/
uint sxm_crc32_check(const void *pBuff, uint unBufSize, uint *pCrc)
{
    const byte *pBuf = (const byte*)pBuff;

    uint unCalculatedCrc = sxm_crc32_calculate(pBuf, unBufSize - 4U);

    uint unCrc = (uint)(pBuf[unBufSize-4] << 24U) | (uint)(pBuf[unBufSize-3] << 16U) |
                 (uint)(pBuf[unBufSize-2] << 8U)  | (uint)(pBuf[unBufSize-1]);

    if (NULL != pCrc)
    {
        *pCrc = unCrc;
    }

    return (unCalculatedCrc == unCrc);
}

/************************************************************************
 *                                                                      *
 *            CRC-16 generator                                          *
 *            ================                                          *
 *                                                                      *
 ************************************************************************/

static ushort sxm_crc16_table[256];             /**< Precomputed values for CRC-16 calculation */

#if SXM_USE_FAST_CRC
static ushort sxm_crc16_table_ext[7][256];      /**< Additional precomputed values for CRC-16 */

/***************************************************************************//**
 * CRC-16 additive calculation -- slicing-by-8 implementation
 *
 * \param[in] c init polynomial or  crc from previous run
 * \param[in] buf input data
 * \param[in] count size of the \c buf in bytes
 *
 * \return calculated value
 *
 ******************************************************************************/
static ushort sxm_crc16_part_fast(ushort c, const void *buf, uint count)
{
    const byte * p = (const byte *)buf;         // current byte/block
    const byte * pe = p + count - 8U;           // the beginning of last block for slicing-by-8

    for (;p <= pe; p += 8U)
    {
        c = sxm_crc16_table_ext[6][p[0] ^ (c >> 8U)] ^
            sxm_crc16_table_ext[5][p[1] ^ (c & 0xFFU)] ^
            sxm_crc16_table_ext[4][p[2]] ^
            sxm_crc16_table_ext[3][p[3]] ^
            sxm_crc16_table_ext[2][p[4]] ^
            sxm_crc16_table_ext[1][p[5]] ^
            sxm_crc16_table_ext[0][p[6]] ^
            sxm_crc16_table[p[7]];
    }

    // tail of the buffer
    while (p < pe + 8U)
        c = (ushort)((c << 8U) ^ sxm_crc16_table[(c >> 8U) ^ *p++]);

    return c;
}
#else
/***************************************************************************//**
 * CRC-16 additive calculation -- standard implementation
 *
 * \param[in] c init polynomial or  crc from previous run
 * \param[in] buf input data
 * \param[in] count size of the \c buf in bytes
 *
 * \return calculated value
 *
 ******************************************************************************/
static ushort sxm_crc16_part_std(ushort c, const void *buf, uint count)
{
    uint i;
    for (i = 0; i < count; ++i)
        c = (ushort)((c << 8U) ^ sxm_crc16_table[(byte)(*((const byte*)buf + i) ^ (c >> 8U))]);
    return c;
}
#endif

/***************************************************************************//**
 * CRC-16 Initialization routine
 *
 * \note This function should be called during SXe SDK initialization.
 *
 ******************************************************************************/
void sxm_crc16_initialize(void)
{
    // Generate CRC table
    ushort c;
    uint n, k;

    for (n = 0; n < 256; n++)
    {
        c = (ushort)(n << 8U);
        for (k = 0; k < 8; k++)
            if (c & 0x8000U)
                c = (ushort)(0x1021U ^ ((uint)c << 1U));
            else
                c = (ushort)(c << 1U);

        sxm_crc16_table[n] = c;
    }

#if SXM_USE_FAST_CRC
    // Generate additional CRC tables
    for (n = 0; n < 256; n++)
    {
        c = sxm_crc16_table[n];
        for (k = 0; k < 7; k++)
        {
            c = (ushort)(sxm_crc16_table[c >> 8U] ^ (c << 8U));
            sxm_crc16_table_ext[k][n] = c;
        }
    }
#endif
}

/***************************************************************************//**
 * CRC-16 additive calculation
 *
 * \param[in] c init polynomial or crc from previous run
 * \param[in] buf input data
 * \param[in] count size of the \c buf in bytes
 *
 * \return calculated value
 *
 ******************************************************************************/
ushort sxm_crc16_part(ushort c, const void *buf, uint count)
{
#if SXM_USE_FAST_CRC
    return sxm_crc16_part_fast(c, buf, count);
#else
    return sxm_crc16_part_std(c, buf, count);
#endif
}

/***************************************************************************//**
 * CRC-16 calculation (init polynomial is \c 0x1d0f)
 * 
 * \param[in] buf input data
 * \param[in] count size of the \c buf in bytes
 *
 * \return calculated value
 * 
 ******************************************************************************/
ushort sxm_crc16_calculate(const void *buf, uint count)
{
    return sxm_crc16_part(0x1D0F, buf, count);
}
