/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_cfile.h
* \author Leslie French, Alexander Pylchagin
* \date 8/20/2013
*
* Public header file for C-File definitions. Refer to \ref cfile_page for
* implementation details.
*
*******************************************************************************/

#ifndef SXM_CFILE_H
#define SXM_CFILE_H

#include <stdio.h>
#include <sxm_build.h>
#include <sxm_common.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** C-File Format length */
#define SXM_CFILE_FORMAT_LEN (16)
/** C-File magic number */
#define SXM_CFILE_MAGIC (('S' << 24) | ('X' << 16) | ('M' << 8) | 'C')
/** C-File own schema version */
#define SXM_CFILE_VERSION (2)

/** Sector size via power of 2 */
#define SXM_ARCH_FILE_SECTOR_DIVISOR (9)

/** Defines single sector size */
#define SXM_ARCH_FILE_SECTOR_SIZE   (1 << SXM_ARCH_FILE_SECTOR_DIVISOR)

/** Defines invalid section number for auto-detection */
#define SXM_CFILE_SECTION_INVALID_ID ((uint)-1)

/** Sector data type */
typedef byte SXMSector[SXM_ARCH_FILE_SECTOR_SIZE];

/** The macro calculate the number of sectors required to hold an
 * array of structures
 * \param[in] _type a type name
 * \param[in] _count the number of structures in the array
 */
#define CSIZE(_type, _count) \
    ((uint)((sizeof(_type) * (_count) + sizeof(SXMSector) - 1) / sizeof(SXMSector)))

/** Defines min number of sectors to accommodate the data type
 * \param[in] _type data type
 */
#define SXM_CSECTORCNT(_type) CSIZE(_type, 1)

 /** The macro calculate the number of sectors required to hold an
 * array of structures. It has to be used \b only to declare regular
 * sections.
 * \param[in] _type a type name
 * \param[in] _variable the name of the array to declare
 * \param[in] _count the number of structures in the array
 */
#define CSECT(_type, _variable, _count) \
    union {        \
        _type _variable[_count]; \
        SXMSector csect[CSIZE(_type, _count)]; \
    }

/** The macro calculate the number of sectors required to hold an
 * array of structures where no sharable sectors between structures.
 * It has to be used \b only to declare dynamic/heap sections.
 * \sa sxm_cfile_add_dsection
 * \param[in] _type a type name
 * \param[in] _variable the name of the array to declare
 * \param[in] _count the number of structures in the array
 */
#define CDSECT(_type, _variable, _count) \
    struct { union {        \
        _type _variable; \
        SXMSector csect[SXM_CSECTORCNT(_type)]; \
    } _variable[_count]; }

/** Defines number of supported sections in c-file */
#define SXM_CFILE_SEC_COUNT (8U)

/** Defines number of max supported masked elements */
#define SXM_CFILE_MASK_ITEMS_COUNT (2048U)

/** Defines user section size in bytes.
 * \note Any modification of this definition can make
 * \ref SXMCFileRootBlock data structure bigger
 * causing usage more sectors in the file.
 */
#define SXM_CFILE_USER_AREA_SIZE (60)

/** \name C-File Masks
 * @{
 */
typedef enum {
    /** Default mask */
    SXM_CFILE_MASK_NONE      = 0x0000,
    /** Heap base section */
    SXM_CFILE_MASK_HEAP      = 0x0001,
    /** Heap base section */
    SXM_CFILE_MASK_CORRUPTED = 0x0002,
} SXMCFileSectorMask;
/** @} */

/** Representation of the each c-file section */
typedef struct {
    uint crc;           //!< CRC-32 over meaningful data in the section
    uint ts;            //!< timestamp when the section was last updated
    ushort off;         //!< offset of the start of the section, in sectors
    ushort length;      //!< length of the section, in sectors
    ushort version;     //!< version number of the section data, where applicable
    ushort count;       //!< count of values within the section, where applicable
    ushort mask;        //!< mask for various needs (\ref SXMCFileSectorMask)
    /** Offset within the \ref SXMCFileRootBlock::map in case of dynamic section */
    ushort heap_map_off;
} SXMCFileSection;

/** The C-File root block. The CRC for the root guarantees that the sections'
 * descriptions as well as CRC are valid.
 * \note The size of this structure must be divided by sizeof(SXMSector).
 */
typedef struct {
    /** The magic sequence (\ref SXM_CFILE_MAGIC) */
    uint magic;
    /** The C-File engine/format version */
    uint cver;
    /** User dependent content's format defined by C-File creator */
    char format[SXM_CFILE_FORMAT_LEN];
    /** User dependent content's schema version defined by C-File creator */
    uint sver;
    /** Timestamp of the last write */
    uint ts;
    /** Describes section keepers */
    SXMCFileSection sections[SXM_CFILE_SEC_COUNT];
    /** Heap map. can be used as an allocation bit-map for heap-based
     * sectors (up to \ref SXM_CFILE_MASK_ITEMS_COUNT entries) */
    uint map[(SXM_CFILE_MASK_ITEMS_COUNT + sizeof(uint) * 8 - 1) / (sizeof(uint) * 8)];
    /** User values started from here until crc field which lays at the end of sector
     * defined by size of \ref SXMSector
     */
    uint u[SXM_CFILE_USER_AREA_SIZE / sizeof(uint)];
    /** Calculated CRC32 for all field above including gaps caused by
     * possible structure alignment.
     * \note This field must be the last in the structure. All data which
     * lays after it inside the root block is not protected by the CRC.
     */
    uint crc;
} SXMCFileRootBlock;

/** Defines validation results */
typedef enum {
    /** The root sector is nor valid so the whole file
     * obviously invalid.
     */
    SXM_CFILE_INVALID = 0,
    /** The root sector is valid, but some active sections
     * are not. The owner is free to do own check to find
     * those valid ones
     */
    SXM_CFILE_PARTIALY_VALID = 1,
    /** The file is valid and it's content can be used
     * w/o extra checks
     */
    SXM_CFILE_FULLY_VALID = 2
} SXMCFileValidationResult;

extern SXESDK_INTERNAL_API void sxm_cfile_set_format(SXMCFileRootBlock *,
                                            const char *,
                                            uint);

extern SXESDK_INTERNAL_API int sxm_cfile_check_format(SXMCFileRootBlock *,
                                             const char *,
                                             uint *);

extern SXESDK_INTERNAL_API int sxm_cfile_add_section(SXMCFileRootBlock *,
                                             uint,
                                             uint);

extern SXESDK_INTERNAL_API int sxm_cfile_add_dsection(SXMCFileRootBlock *,
                                             uint,
                                             uint,
                                             uint);

extern SXESDK_INTERNAL_API int sxm_cfile_alloc_and_load(FILE *,
                                               size_t,
                                               ptr *);

extern SXESDK_INTERNAL_API void* sxm_cfile_base_section(SXMCFileRootBlock *,
                                             uint);

extern SXESDK_INTERNAL_API int sxm_cfile_check(SXMCFileRootBlock *,
                                      uint,
                                      BOOL);

extern SXESDK_INTERNAL_API int sxm_cfile_clear_section(SXMCFileRootBlock *,
                                               uint);

extern SXESDK_INTERNAL_API void sxm_cfile_close(FILE *,
                                       SXMCFileRootBlock *);

extern SXESDK_INTERNAL_API int sxm_cfile_commit(FILE *,
                                       SXMCFileRootBlock *,
                                       SXMIOStat *);

extern SXESDK_INTERNAL_API int sxm_cfile_map_init(SXMCFileRootBlock *,
                                                   uint,
                                                   uint);

extern SXESDK_INTERNAL_API int sxm_cfile_map_alloc(SXMCFileRootBlock *,
                                          uint,
                                          uint);

extern SXESDK_INTERNAL_API int sxm_cfile_map_free(SXMCFileRootBlock *,
                                          uint,
                                          uint,
                                          uint);

extern SXESDK_INTERNAL_API FILE *sxm_cfile_open(const char *,
                                       const char *);

extern SXESDK_INTERNAL_API int sxm_cfile_secure(FILE *,
                                       SXMCFileRootBlock *,
                                       uint,
                                       SXMIOStat *);

extern SXESDK_INTERNAL_API int sxm_cfile_secure_all(FILE *,
                                           SXMCFileRootBlock *,
                                           SXMIOStat *);

extern SXESDK_INTERNAL_API int sxm_cfile_shutdown(FILE *,
                                         SXMCFileRootBlock *);

extern SXESDK_INTERNAL_API SXMCFileValidationResult sxm_cfile_validate(SXMCFileRootBlock *);

extern SXESDK_INTERNAL_API int sxm_cfile_write(FILE *,
                                         uint,
                                         uint,
                                         const void *);

extern SXESDK_INTERNAL_API uint sxm_cfile_time_now(void);

extern SXESDK_INTERNAL_API int sxm_cfile_uset(SXMCFileRootBlock *,
                                            uint, 
                                            uint);

extern SXESDK_INTERNAL_API int sxm_cfile_uget(const SXMCFileRootBlock *,
                                            uint, 
                                            uint *);

#ifdef __cplusplus
}
#endif

#endif // SXM_CFILE_H
