/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_callback.c
* \author Eugene Lopatukhin $
* \date 10/5/2013
*
* Implementation file for application callback service. This is done so that
* callbacks occur under separate thread and have no effect on the actual processing
* of incoming sxi messages. If something bad is done in application callback
* routine of it will not have any ill effect on the actual processing thread
* callbacks are isolated of by separate thread.
*******************************************************************************/

#include <util/sxm_msg_queue_incl.h>
#include <util/sxm_callback_incl.h>
#include <util/sxm_common_internal.h>

/** Thread name */
#define CALLBACK_THREAD_NAME   	"CALLBACK_THREAD"

enum {
    /** Callback service thread message queue size */
    CALLBACK_MSG_QUEUE_SIZE	= 64
};

/** types of application callback messages */
typedef enum {
    APP_CALLBACK_MSG_TYPE, //!< Application callback message
    CALLBACK_STOP //!< Stop message
} SXMCallbackMsgType;

/** Callback message structure */
typedef struct {
    void(*callback)(int, int); //!< Callback function to be called from the thread
    uint type; //!< Callback argument 1 - type
    uint param; //!< Callback argument 2 - param
} SXMCallbackMsg;

/** Callback service structure */
typedef struct {
    /** processing thread */
    pthread_t tid;
    /** message queue for timers service thread */
    MSG_QUEUE_STRUCT queue;
} SXMCallbackService;

/** callback service status */
static SXMStatus status = {SXM_SERVICE_STOPPED, SXM_SUBS_NONE};

#define IS_SERVICE_RUNNING_OK() (service && (status.service == SXM_SERVICE_OK))
#define IS_SERVICE_STOPPED() (status.service == SXM_SERVICE_STOPPED)

/** Timers service instance */
static SXMCallbackService *service = NULL;

/* queue mutex */
static pthread_mutex_t queueMutex = PTHREAD_MUTEX_INITIALIZER;

/* private functions */
static void *callback_thread(void *pData);
static SXMResultCode callback_allocate_service_resources(SXMCallbackService **ppService);
static void callback_free_service_resources(SXMCallbackService *pService);

/***************************************************************************//**
* This function allocates some of the resource needed for service.
*
* \param[out] ppService service instance in case of success
* \return Corresponding error code
********************************************************************************/
static SXMResultCode callback_allocate_service_resources(SXMCallbackService **ppService) {
    SXMResultCode rc = SXM_E_ERROR;
    SXMCallbackService *pService;

    pService = (SXMCallbackService *)sxe_calloc(1, sizeof(SXMCallbackService));
    if (!pService) {
        rc = SXM_E_NOMEM;
    }
    else {
        rc = sxm_queue_create(&pService->queue, CALLBACK_MSG_QUEUE_SIZE,
                              &queueMutex, sizeof(SXMCallbackMsg), "callback", NULL);
        if (rc != SXM_E_OK) {
            sxe_free(pService);
            pService = NULL;
        }
    }
    if (rc == SXM_E_OK) {
        *ppService = pService;
    }
    return rc;
}

/***************************************************************************//**
* This function frees some of the resource needed for service.
*
* \param[in] pService service to be released
********************************************************************************/
static void callback_free_service_resources(SXMCallbackService *pService) {
    if (pService) {
        sxm_queue_destroy(&pService->queue);
        sxe_free(pService);
    }
}

/***************************************************************************//**
* This function starts application callback service.
*
* \retval SXM_E_OK on successful
* \retval SXM_E_STATE Service already running
* \retval SXM_E_NOMEM No memory to start the service
* \retval SXM_E_THREAD Thread creation failure
********************************************************************************/
int sxm_callback_start(void) {
    SXMResultCode rc = SXM_E_OK;
    /* make sure not running before starting */
    if (IS_SERVICE_STOPPED() ) {
        rc = callback_allocate_service_resources(&service);
        if (rc == SXM_E_OK) {
            SXMSem_t initCompletedSem;
            /* initialize signaling semaphore for thread initialization completion */
            if (sxm_sem_init(&initCompletedSem, 0) == SXM_E_OK) {
                /* processing thread creation */
                if (sxm_pthread_create(&service->tid, NULL, callback_thread, &initCompletedSem) == 0) {
                    /* init thread name */
                    sxe_thread_setname(service->tid, CALLBACK_THREAD_NAME);
                    /* wait for thread initialization to complete */
                    rc = sxm_sem_wait(&initCompletedSem);
                    if (rc != SXM_E_OK) {
                        PLOG_UTL("Failed to wait semaphore %d, should not be a problem", rc);
                        rc = SXM_E_OK;
                    }
                }
                else {
                    callback_free_service_resources(service);
                    service = NULL;
                    rc = SXM_E_THREAD;
                }
                /* destroy init semaphore */  
                sxm_sem_destroy(&initCompletedSem);
            }
            else {
                callback_free_service_resources(service);
                service = NULL;
                rc = SXM_E_RESOURCE;
            }
        }
        else {
            rc = SXM_E_NOMEM;
        }
    }
    else {
        /* service already running */
        rc = SXM_E_STATE;
    }

    return rc;
}

/***************************************************************************//**
* This function stops application callback service.

* \retval SXM_E_OK on successful
* \retval SXM_E_STATE incorrect state
********************************************************************************/
int sxm_callback_stop(void) {
    SXMResultCode rc;

    /* make sure service is running */
    if (!IS_SERVICE_STOPPED()) {
        /* tell thread to kill itself */
        rc = sxm_queue_put(&service->queue, NULL, CALLBACK_STOP, 0, MSG_PRIO_NORMAL);
        if (rc == SXM_E_OK) {
            /* wait for thread to complete */
            pthread_join(service->tid, NULL);

            /* release acquired resources */
            callback_free_service_resources(service);
            service = NULL;
        }
    }
    else {
        rc = SXM_E_STATE;
    }

    return rc;
}

/***************************************************************************//**
* This function requests callback service to perform callback.
*
* \param[in] callback the function to be called from the callback's context
* \param[in] type the first argument to be passed to the \p callback upon call
* \param[in] param the second argument to be passed to the \p callback upon call
*
* \retval SXM_E_OK on success
* \retval SXM_E_NOMEM cannot allocate memory for message data
* \retval SXM_E_RESOURCE queue was full and message was not placed on the queue
********************************************************************************/
int sxm_callback_req(void(*callback)(int, int), uint type, uint param) {

    SXMResultCode rc = SXM_E_OK;

    /* if callback service is not running ignore it */
    if (IS_SERVICE_RUNNING_OK()) {
        SXMCallbackMsg msgPayloadData;
        msgPayloadData.callback = callback;
        msgPayloadData.type = type;
        msgPayloadData.param = param;

        rc = sxm_queue_put(&service->queue, &msgPayloadData,
                           APP_CALLBACK_MSG_TYPE, sizeof(msgPayloadData),
                           MSG_PRIO_NORMAL);
    }

    return rc;
}

/***************************************************************************//**
* High level processing thread for callback related messages
* 
* \param[in] pData thread argument, in this case the initialization semaphore
* \return Thread execution result
********************************************************************************/
static void *callback_thread(void *pData) {
    SXMQueueMsg msg = { 0, 0, NULL };
    SXMSem_t *pInitCompletedSem = (SXMSem_t *)pData;

    /* service is operational */
    status.service = SXM_SERVICE_OK;
 
    /* notify service start routine that service thread initialization has completed */
    sxm_sem_post(pInitCompletedSem);

    while (!IS_SERVICE_STOPPED()) {
        /* pend on message */
        if (sxm_queue_get(&service->queue, &msg, SXM_QUEUE_INFINITE_TIMEOUT) == SXM_E_OK) {
            /* process message */
            switch (msg.msgType) {
                case APP_CALLBACK_MSG_TYPE: {
                    SXMCallbackMsg *pMsg = (SXMCallbackMsg*)sxm_queue_msg_data(&msg);
                    if ((pMsg != NULL) && (pMsg->callback != NULL)) {
                        SXMCallbackMsg cbMsg = *pMsg;
                        sxm_queue_msg_release(&msg); /* release memory as soon as possible */
                        cbMsg.callback((int)cbMsg.type, (int)cbMsg.param);
                    }
                }
                break;
                case CALLBACK_STOP: {
                    /* service no longer  running */
                    status.service = SXM_SERVICE_STOPPED;
                }
                break;
                default:
                break;
            }

            /* free message memory, if it has not been released yet */
            sxm_queue_msg_release(&msg);
        }
    }

    return NULL;
}
