/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_bitbuff.h
* \author Leslie French, Sergey Kotov
* \date 8/20/2013
*
* Header file for BitStream Parsing.
* The BitBuff (bitstream buffer) is the utility interface for parsing protocol
* units.
*
*******************************************************************************/

#ifndef SXM_BITBUFF_H
#define SXM_BITBUFF_H

#include <stdio.h>
#include <sxm_build.h>
#include <sxm_typ.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** Baudot mode */
#define BAUDOT_MODE_START_WITH_LETTERS       (0U)
#define BAUDOT_MODE_START_WITH_SYMBOLS       (1U)
#define BAUDOT_MODE_START_WITH_LOWER_LETTERS (2U)

/** File buffer size */
#define BIT_FILE_BUF_SIZE                   (4096) /* bytes */

/** A simple structure to track the current position with the bit buffer data */
struct _bitbuff
{
    byte *ob;       //!< original buffer pointer
    byte *b;        //!< current buffer pointer
    byte *e;        //!< pointer to end of buffer
    uint valid;     //!< number of valid bits in seed
    uint seed;      //!< accumulated bits
    byte err;       //!< error flag
    int(*replenish)(struct _bitbuff *pBuff);
    int(*seek)(struct _bitbuff *pBuff, size_t offset);
    int(*close)(struct _bitbuff *pBuff);
    size_t(*read_bytes)(struct _bitbuff *pBuff, size_t len, byte *out);
};

typedef struct _bitbuff SXMBitBuff;

/** A simple structure to track file bit buffer data */
typedef struct
{
    SXMBitBuff b;               // the actual bitbuff (must be first)
    byte fb[BIT_FILE_BUF_SIZE]; // input buffer for file
    byte eof;                   // end of file on input
    FILE *in;                   // input file
    uint size;                  // size of file to be used
    uint pos;                   // current position in file
} SXMBitFileBuff;

/** A structure containing Huffman codes for a single block */
typedef struct
{
    uint *litdist;
    uint *litvalues;
    uint nlit;
    uint ndist;
    uint nbit;
} SXMDeflateCompBlock;

/** Zip buffer states */
typedef enum
{
    SXM_BITZIP_BUFF_STATE_INITIAL = 0,
    SXM_BITZIP_BUFF_STATE_UNCOMP,
    SXM_BITZIP_BUFF_STATE_FCOMP,
    SXM_BITZIP_BUFF_STATE_DCOMP,
    SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED,
    SXM_BITZIP_BUFF_STATE_ALL_COMPLETED,
    SXM_BITZIP_BUFF_STATE_ERROR
} SxmBitZipBuffState;

/** Max size of deflate window supported by bitzipbuffer */
#define SXM_BITZIP_BUFF_WINDOW_SIZE (65536)

/** Zip stream structure */
typedef struct
{
    SXMBitBuff b;                               /** the actual bitbuff (must be first) */
    byte ringbuf[SXM_BITZIP_BUFF_WINDOW_SIZE];  /** ring buffer containing uncompressed data */
    byte *bufhead;                              /** header of uncompressed data in ring buffer */
    byte *buftail;                              /** tail of uncompressed data in ring buffer */
    byte *bufend;                               /** pointer to end of buffer */
    uint bufcount;                              /** a number of uncompressed bytes in buffer */
    SXMBitBuff *input;                          /** input bit buffer containing compressed data */
    byte bfinal;                                /** last block flag */
    SxmBitZipBuffState state;                   /** current state of zip buffer */
    uint rlen;                                  /** remaining length (different meaning depending on state) */
    uint pos;                                   /** position in uncompressed data */
    SXMDeflateCompBlock bfixed;                 /** fixed block related data */
    SXMDeflateCompBlock *bdynamic;              /** dynamic block related data */
} SXMBitZipBuff;

extern SXESDK_INTERNAL_API int sxm_bitzipbuff_setup(SXMBitZipBuff *,    // this
                                                    SXMBitBuff *);      // input

extern SXESDK_INTERNAL_API int sxm_bitzipbuff_close(SXMBitZipBuff *);   // this

extern SXESDK_INTERNAL_API byte *sxm_bitbuff_align(SXMBitBuff *);       // this

extern SXESDK_INTERNAL_API byte *sxm_bitbuff_align_debug(SXMBitBuff *,  // this
                                                         int);          // flag

extern SXESDK_INTERNAL_API int sxm_bitbuff_baudot(SXMBitBuff *,         // this
                                                  char *,               // dstBuf
                                                  uint,                 // mode (one of BAUDOT_MODE_...)
                                                  int,                  // numReadMax
                                                  int);                 // dstBufSize

extern SXESDK_INTERNAL_API int sxm_bitbuff_baudot_debug(SXMBitBuff *,   // this
                                                        const char *,   // var
                                                        char *,         // dstBuf
                                                        uint,           // mode (one of BAUDOT_MODE_...)
                                                        int,            // numReadMax
                                                        int,            // dstBufSize
                                                        int);           // flag

extern SXESDK_INTERNAL_API int sxm_bitbuff_consume_rev(SXMBitBuff *,    // this
                                                       uint,            // value
                                                       uint);           // len

extern SXESDK_INTERNAL_API void sxm_bitbuff_cstring(SXMBitBuff *,       // this
                                                    char *,             // desc
                                                    int);               // len

extern SXESDK_INTERNAL_API void sxm_bitbuff_iso(SXMBitBuff *,           // this
                                                char *,                 // desc
                                                int);

extern SXESDK_INTERNAL_API int sxm_bitbuff_cstring_debug(SXMBitBuff *,  // this
                                                         const char *,  // var
                                                         char *,        // desc
                                                         int,           // len
                                                         int);          // flag

extern SXESDK_INTERNAL_API int sxm_bitbuff_iso_debug(SXMBitBuff *,      // this
                                                     const char *,      // var
                                                     char *,            // desc
                                                     int,               // len
                                                     int);              // flag

extern SXESDK_INTERNAL_API uint sxm_bitbuff_flag_debug(SXMBitBuff *,    // this
                                                       int);            // flag

extern SXESDK_INTERNAL_API uint sxm_bitbuff_read(SXMBitBuff *,          // this
                                                 uint);                 // len

extern SXESDK_INTERNAL_API void sxm_bitbuff_skip(SXMBitBuff *,          // this
                                                 uint);                 // len

extern SXESDK_INTERNAL_API uint sxm_bitbuff_read_debug(SXMBitBuff *,    // this
                                                       const char *,    // var
                                                       uint,            // len
                                                       int);            // flag

extern SXESDK_INTERNAL_API void sxm_bitbuff_skip_cstring_debug(SXMBitBuff *, // this
                                                               int);         // flag

extern SXESDK_INTERNAL_API void sxm_bitbuff_skip_debug(SXMBitBuff *,    // this
                                                       uint,            // len
                                                       int);            // flag

extern SXESDK_INTERNAL_API uint sxm_bitbuff_read_rev(SXMBitBuff *,      // this
                                                     uint);             // len

extern SXESDK_INTERNAL_API void sxm_bitbuff_setup(SXMBitBuff *,         // this
                                                  byte *,               // b
                                                  uint);                // len

extern SXESDK_INTERNAL_API int sxm_bitfilebuff_setup(SXMBitFileBuff *,  // this
                                                     FILE *,            // pFile
                                                     uint,              // offset
                                                     uint);             // size

extern SXESDK_INTERNAL_API SXMBitFileBuff* sxm_bitfilebuff_open(char,   // type
                                                     const char *,      // module
                                                     const char *,      // file
                                                     const char *,      // mode
                                                     int *);            // pointer to return code

extern SXESDK_INTERNAL_API int sxm_bitfilebuff_close(SXMBitFileBuff *); // this

extern SXESDK_INTERNAL_API void sxm_bitbuff_skip_baudot(SXMBitBuff *,   // this
                                                        uint);          // mode (one of BAUDOT_MODE_...)

extern SXESDK_INTERNAL_API void sxm_bitbuff_skip_cstring(SXMBitBuff *); // this

extern SXESDK_INTERNAL_API void sxm_bitbuff_skip_iso(SXMBitBuff *,      // this
                                                     uint);             // len

extern SXESDK_INTERNAL_API int sxm_bitbuff_close(SXMBitBuff *);         // this

extern SXESDK_INTERNAL_API int sxm_bitbuff_seek(SXMBitBuff *,           // this
                                                uint);                  // len

extern SXESDK_INTERNAL_API size_t sxm_bitbuff_read_bytes(SXMBitBuff *,  // this
                                                         size_t,        // len
                                                         byte *);       // out

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SXM_BITBUFF_H */
