/************************************************************************
 *                                                                      *
 *            SXI_INTERNAL.H	                                        *
 *            ==============                                            *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 *    Internal routines for the SXi module                              *
 *                                                                      *
 ************************************************************************/

#ifndef SXI_INTERNAL_H
#define SXI_INTERNAL_H

#include "sxm_build.h"

#if (!SXM_USE_GEN8)
#include <sxm_sdk.h>

#ifdef __cplusplus
extern "C" {
#endif

/*************************/
/* SXI CYCLE FILE  DATA  */
/*************************/
/* SXI Data Saved over a power cycle,
WARNING:
changing any of these values will invalidate any
saved date file.  (Delete the sxm/cycle file) */
#define MAX_LEAGUE          128
#define MAX_TEAM            1024
#define MAX_TRAFFIC         32
#define MAX_PRESET          32
#define MAX_ITUNE           1
#define MAX_CHAN            SXI_SP_MAX_SID_CNT

/* The list of possible Sxi callback types */
typedef enum {
    SXM_SXI_CALLBACK_ERROR,
    SXM_SXI_CALLBACK_RADIO,
    SXM_SXI_CALLBACK_POWER,
    SXM_SXI_CALLBACK_PACKAGE,
    SXM_SXI_CALLBACK_SUBS,
    SXM_SXI_CALLBACK_TIME,
    SXM_SXI_CALLBACK_SMON,
    SXM_SXI_CALLBACK_EVENT,
    SXM_SXI_CALLBACK_DISPLAY,
    SXM_SXI_CALLBACK_CHANNEL,
    SXM_SXI_CALLBACK_CATEGORY,
    SXM_SXI_CALLBACK_TUNE,
    SXM_SXI_CALLBACK_PAD,
    SXM_SXI_CALLBACK_LOOKAHEAD,
    SXM_SXI_CALLBACK_GLOBAL,
    SXM_SXI_CALLBACK_RECORD,
    SXM_SXI_CALLBACK_ERASE,
    SXM_SXI_CALLBACK_NEXT,
    SXM_SXI_CALLBACK_DEBUG,
    SXM_SXI_CALLBACK_MEMORY,
    SXM_SXI_CALLBACK_FA,
    SXM_SXI_CALLBACK_BULLETIN,
    SXM_SXI_CALLBACK_BROWSE,
    SXM_SXI_CALLBACK_FLASH,
    SXM_SXI_CALLBACK_CONTENT_BUFFER,
    SXM_SXI_CALLBACK_EXT_CHANNEL,
    SXM_SXI_CALLBACK_IPDEVAUTH,
    SXM_SXI_CALLBACK_CMD_TIMEOUT,
    SXM_SXI_CALLBACK_AUTHSTATE,

    SXM_SXI_CALLBACK_INVALID,
    SXM_SXI_CALLBACK_COUNT = SXM_SXI_CALLBACK_INVALID
} SxmSxiCallbackTypes;

/**************************/
/* SXI I2S SUPPORT       */
/*************************/
#define SXM_SXI_I2SS_PARAMETERS_REGISTER_ADDR (0x3180U)
#define SXM_SXI_I2SS_FRAME_LEN_MASK  (0x30U)

/***********************************/
/* SXI TFILE and CFILE DEFINITIONS */
/***********************************/
#define SXI_MODULE_MANE "sxi"
#define SXI_TFILE_TYPE 'R'
#define SXI_TFILE_NAME "metadata"
#define SXI_FILE_FORMAT SXI_MODULE_MANE
#define SXI_TFILE_SCHEMA_VERSION (2)
#define SXI_CFILE_TYPE 'T'
#define SXI_CFILE_NAME  "cycle"

typedef enum {
    SXI_TFILE_SECT_TRAFFIC = 0,
    SXI_TFILE_SECT_LEAGUES,
    SXI_TFILE_SECT_TEAMS,
    SXI_TFILE_SECT_MAX
} SXIMetadataTypes;

typedef struct {
    UINT16 version; //!< Table version
    UINT16 count; //!< Number of record
    UINT8 idx; //!< The start block
} SXIMetadataMap;

typedef struct {
    /** Common version to differentiate between
     * generated databases
     */
    UINT16 version;

    /** Versions of metadata */
    SXIMetadataMap meta_map[SXI_TFILE_SECT_MAX];
} SXIRootHeader;

/*************************/
/* EXTENDED METADATA     */
/* COLLECTION STATUS     */
/*************************/
enum {
    EXT_METADATA_IDLE          = 0U,
    EXT_METADATA_WAIT_COUNT    = 1U,
    EXT_METADATA_START_REQ     = 2U,
    EXT_METADATA_COLLECTING    = 3U,
    EXT_METADATA_STOP_REQ      = 4U
};


/***********************************/
/* GENERAL SXI SIZE                */
/***********************************/
#define SXI_MOD_CFG_CAP_SIZE            4
#define SXI_PROGRAMID_SIZE              4
#define SXI_EVENT_DATA_TYPE_SIZE        2
#define SXI_EVENT_DATA_ARRAY_SIZE       (SXI_EVENT_DATA_SIZE*SXI_EVENT_DATA_TYPE_SIZE)
#define SXI_SID_VECTOR_SIZE             48
#define SXI_ARRAY_HASH_SIZE             12
#define SXI_PKG_MAC_SIZE                12
#define SXI_FWU_BYTES_IN_PACKET_SIZE    4
#define SXI_FWU_NUM_PACKETS_SIZE        4
/* This size is enough to accommodate MAX(SXI_CATID_ALL_NUM + 1, SXI_CATID_MAX_NUM_CNT)
 * categories.
 */
#define SXM_CATEGORY_MAP_SIZE           8U

/***********************************/
/* METADATA FIELD SIZES            */
/***********************************/
#define SXI_CM_SUM_OF_ENTRIES_LEN_MAX       256
#define SXI_TM_SUM_OF_ENTRIES_LEN_MAX       188

/* Channel Metadata Identifiers */
#define CHANNEL_SHORT_DESCRIPTION           64
#define CHANNEL_LONG_DESCRIPTION            65
#define RELATED_CHANNEL_LIST                66
#define CHANNEL_LIST_ORDER                  68
#define PLAY_ON_SELECT                      69
#define CHANNEL_CONTENT_TYPE                70
#define PLAYBACK_NAVIGATION_CLASS           71
#define SHADOW_RECORD_CLASS                 72

/****************************/
/* SXI OFFSETS              */
/****************************/
/* general offset */
#define SXI_OPCODE_MSB_OFFSET                   0
#define SXI_OPCODE_LSB_OFFSET                   1
#define SXI_TRANSACTION_ID_OFFSET               2

/* extended metadata command */
#define SXI_EXT_METADATA_MON_CMD_SELECTION      3
#define SXI_EXT_METADATA_MON_CMD_CHANGE_TYPE    4

/* response offset */
#define SXI_RESP_OPCODE_MSB_OFFSET      0
#define SXI_RESP_OPCODE_LSB_OFFSET      1
#define SXI_RESP_TRANSACTION_ID_OFFSET  2
#define SXI_RESP_STATUS_CODE_OFFSET     3

/* Indication offset */
#define SXI_IND_OPCODE_MSB_OFFSET      0
#define SXI_IND_OPCODE_LSB_OFFSET      1
#define SXI_IND_TRANSACTION_ID_OFFSET  2
#define SXI_IND_STATUS_CODE_OFFSET     3
#define SXI_IND_PAYLOAD_OFFSET         4

/* confirm offset */
#define SXI_CFM_OPCODE_MSB_OFFSET       0
#define SXI_CFM_OPCODE_LSB_OFFSET       1
#define SXI_CFM_TRANSACTION_ID_OFFSET   2
#define SXI_CFM_HDR_SIZE                3

/****************************/
/* SXI OPCODE MASK          */
/****************************/
#define SXI_OPCODE_FROM_HOST        0x0000
#define SXI_OPCODE_FROM_MODULE      0x8000

#define SXI_OPCODE_MSG_NOT_REPLY    0x0000
#define SXI_OPCODE_MSG_IS_REPLY     0x4000

#define SXI_OPCODE_MSG_TYPE_MASK    0xC000
#define SXI_OPCODE_MASK             0x3FFF

#define SXI_OPCODE_COMMAND          (SXI_OPCODE_FROM_HOST |  SXI_OPCODE_MSG_NOT_REPLY)
#define SXI_OPCODE_RESPONSE         (SXI_OPCODE_FROM_MODULE | SXI_OPCODE_MSG_IS_REPLY)
#define SXI_OPCODE_INDICATION       (SXI_OPCODE_FROM_MODULE | SXI_OPCODE_MSG_NOT_REPLY)
#define SXI_OPCODE_CONFIRMATION     (SXI_OPCODE_FROM_HOST |  SXI_OPCODE_MSG_IS_REPLY)

/****************************/
/* SXI MESSAGE SIZE         */
/****************************/
#define SXI_MODULE_CFG_CMD_SIZE             13
#define SXI_PWR_MODE_CMD_SIZE               4
#define SXI_TIME_CFG_CMD_SIZE               5
#define SXI_STATUS_MON_CMD_SIZE             5   /*  variable length --- (5 + SXI_SP_MAX_STATUS_MON_LIST_CNT) */
#define SXI_FEATURE_MON_CMD_SIZE            5   /*  variable length --- (5 + SXI_SP_MAX_FEAT_MON_LIST_CNT) */
#define SXI_EXT_METADATA_MON_CMD_SIZE       6   /*  variable length --- (6 +2*SXI_SP_MAX_EMD_MON_LIST_CNT) */
#define SXI_PING_CMD_SIZE                   3
#define SXI_AUDIO_MUTE_CMD_SIZE             4
#define SXI_AUDIO_VOL_CMD_SIZE              4
#define SXI_AUDIO_TONAL_CMD_SIZE            5
#define SXI_AUDIO_EQ_CMD_SIZE               13
#define SXI_TONE_GEN_CMD_SIZE               6
#define SXI_CHAN_BROWSE_CMD_SIZE            8
#define SXI_USER_CAT_CFG_CMD_SIZE           53  /*  variable length --- (53 + sizeof user defined category name) */
#define SXI_CHAN_SELECT_CMD_SIZE            8
#define SXI_CHAN_ATTRIB_CFG_CMD_SIZE        52
#define SXI_CHAN_SELECT_CFG_CMD_SIZE        10
#define SXI_CHAN_ATTRIB_LIST_CMD_SIZE       6   /*  variable length --- (6 + SXI_SP_MAX_SID[383]) */
#define SXI_SEEK_MON_CMD_SIZE               10  /*  variable length --- (10 + SXI_SP_MAX_MON_VALUE_LEN*SXI_SP_MAX_MON_VALUE_CNT + 2*SXI_SP_MAX_TMI_MON_LIST_CNT)  */
#define SXI_IR_PLAYBACK_CONTROL_CMD_SIZE    8
#define SXI_DATA_SERVICE_MON_CMD_SIZE       6
#define SXI_FWU_PROG_CMD_SIZE               3
#define SXI_FWU_UCFG_CMD_SIZE               11
#define SXI_FWU_PACKET_CMD_SIZE             7   /*  variable length --- (7 + BytesInPacket) */
#define SXI_FA_CMD_SIZE                     6   /*  variable length --- (6 + sizeof model string) */
#define SXI_PKG_CMD_SIZE                    5
#define SXI_PLAYBACK_AUDIO_PACKET_CMD_SIZE  9   /*  variable length --- (9 + NumDataBytes) */


/* buffer sizes */
/* per SX-9845-0098_SXi_UART_Link_Draft_V1.4_2011-05-05 document, section 2.1.5,
the maxmimum payload for the any SXI messages in bytes is 1400. The SXi control message
max is 1024, SXi data message is 1042, SXI Audio Message is 1400. Therefore, the maximum
size of 1400 bytes (SXI Audio Message) is selected for the command and receive buffer */
#define SXI_CMD_BUFFER_SIZE             1400
#define SXI_RX_BUFFER_SIZE              1400


/***********************************/
/* SXI DSI                         */
/***********************************/
#define SXI_DSI_MIN_NUM                 0
#define SXI_DSI_MAX_NUM                 (SXI_SP_MAX_DSI_CNT - 1)

/******************************************************************************/
/*==========================> SXI COMMAND DEFINITION  <=======================*/
/******************************************************************************/

/***********************************/
/* SXI COMMAND OPCODE              */
/***********************************/
#define SXI_MODULE_CFG_CMD              0x0020
#define SXI_PWR_MODE_CMD                0x0021
#define SXI_TIME_CFG_CMD                0x0060
#define SXI_STATUS_MON_CMD              0x00A0
#define SXI_FEATURE_MON_CMD             0x00A1
#define SXI_EXT_METADATA_MON_CMD        0x00A2
#define SXI_PING_CMD                    0x00E0
#define SXI_IP_DEVICE_AUTH_CMD          0x00F0
#define SXI_AUTH_STATE_CMD              0x00F1
#define SXI_AUDIO_MUTE_CMD              0x0100
#define SXI_AUDIO_VOL_CMD               0x0101
#define SXI_AUDIO_TONAL_CMD             0x0102
#define SXI_AUDIO_EQ_CMD                0x0104
#define SXI_TONE_GEN_CMD                0x0180
#define SXI_CHAN_BROWSE_CMD             0x0200
#define SXI_USER_CAT_CFG_CMD            0x0202
#define SXI_CHAN_SELECT_CMD             0x0280
#define SXI_CHAN_ATTRIB_CFG_CMD         0x0282
#define SXI_CHAN_SELECT_CFG_CMD         0x0283
#define SXI_CHAN_ATTRIB_LIST_CMD        0x0284
#define SXI_SEEK_MON_CMD                0x0304
#define SXI_IR_PLAYBACK_CONTROL_CMD     0x0402
#define SXI_BULLETIN_MON_CMD            0x0420
#define SXI_PLAYBACK_AUD_PACKET_CMD     0x0443
#define SXI_DATA_SERVICE_MON_CMD        0x0500
#define SXI_DATA_SERVICE_FILTER_CMD     0x0501
#define SXI_FWU_PROG_CMD                0x0E81
#define SXI_FWU_CFG_CMD                 0x0E82
#define SXI_FWU_PACKET_CMD              0x0E83
#define SXI_FA_CMD                      0x0EC0
#define SXI_FA_PKG_CMD                  0x0ED0
#define SXI_MODULE_RESET_CMD            0x0F00
#define SXI_MEM_WRITE_CMD               0x0F03
#define SXI_MEM_MON_SETUP_CMD           0x0F04
#define SXI_DELETE_MEM_MON_CMD          0x0F05
#define SXI_TUNNEL_RESP_CMD             0x0F07
#define SXI_DEBUG_ENABLE_CMD            0x0F09



/***********************************/
/* SXI MODULE CONFIGURE COMMAND    */
/***********************************/
/*byte 3 - fade */
#define SXI_MOD_CFG_FADE_ENABLED    0x01

/* byte 4 - category label length */
#define SXI_MOD_CFG_CAT_08_LEN      0x00
#define SXI_MOD_CFG_CAT_10_LEN      0x01
#define SXI_MOD_CFG_CAT_16_LEN      0x02

/* byte 5 - channel label length */
#define SXI_MOD_CFG_CHN_08_LEN      0x00
#define SXI_MOD_CFG_CHN_10_LEN      0x01
#define SXI_MOD_CFG_CHN_16_LEN      0x02

/* byte 6 - metadata label length */
#define SXI_MOD_CFG_META_08_LEN     0x00
#define SXI_MOD_CFG_META_10_LEN     0x01
#define SXI_MOD_CFG_META_16_LEN     0x02

/* byte 7 - maximum number of indications */
#define SXI_MOD_CFG_01_IND          0x00
#define SXI_MOD_CFG_04_IND          0x01

/* byte 8 - confirmation wait time */
#define SXI_MOD_CFG_CFM_WAIT_200MS   0
#define SXI_MOD_CFG_CFM_WAIT_400MS   4
#define SXI_MOD_CFG_CFM_WAIT_4000MS  76
#define SXI_MOD_CFG_CFM_WAIT_STEP_MS 50

/* byte 9 - Instant replay function (IRControl) */
#define SXI_MOD_CFG_IR_DISABLED     0x00
#define SXI_MOD_CFG_IR_ENABLED      0x01

/* byte 10 - Instant replay delete on tune */
#define SXI_MOD_CFG_KEEP_ON_TUNE    0x00
#define SXI_MOD_CFG_DELETE_ON_TUNE  0x01

/* byte 11 - Instant replay mark new track */
#define SXI_MOD_CFG_PID_NEW_TRACK   0x00
#define SXI_MOD_CFG_LABEL_NEW_TRACK 0x01

/* byte 12 - RecordControl */
/* bit 7 to bit 3 are reserved */
#define SXI_MOD_CFG_RECORD_DISABLED         0x00
#define SXI_MOD_CFG_RECORD_MODULE_MANAGE    0x01        /* bit 0 - Module managed recording */
#define SXI_MOD_CFG_RECORD_HOST_MANAGE      0x02        /* bit 1 - Host managed recording */
#define SXI_MOD_CFG_LISTEN_RECORD_ENABLED   0x04        /* bit 2 - ListenRecEnable */

/* byte 13 - ExtendedControl */
/* bit 7 to bit 3 are reserved */
#define SXI_MOD_CFG_EXTENDED_CTL_I2S_MASTER                         0x01        /* bit 0 I2S Control Master */
#define SXI_MOD_CFG_EXTENDED_CTL_CONTENT_BUFFERED_IND_ENABLED       0x02        /* bit 1 Content Buffered Indications enabled */
#define SXI_MOD_CFG_EXTENDED_CTL_MREF_IND_ENABLED                   0x04        /* bit 2 MRef Mode enabled */

/***********************************/
/* SXI MODULE CFG INDICATION       */
/***********************************/

/* byte 25 to byte 28 */
/* Capability */
/* bit 31 to bit 6: reserved */
#define SXI_INSTANT_REPLAY_PRESENT          0x000001    /* bit 0 - Instant Replay */
#define SXI_AUDIO_RECORDING_PRESENT         0x000002    /* bit 1 - Audio Recording */
#define SXI_EXTENDED_CHAN_PRESENT           0x000004    /* bit 2 - Extended Channels */
#define SXI_I2S_SLAVE_MODE_SUPPORTED        0x000008    /* bit 3 - I2S Slave */
#define SXI_ADV_IR_FEATURE_AVAIL            0x000010    /* bit 4 - Advanced IR Available */
#define SXI_EXPANDED_SFAV_SUPPORTED         0x000020    /* bit 5 - Expanded Smart Favorites */

/* offsets */
#define SXI_MOD_CFG_IND_MODULE_TYPE_OFFSET      4
#define SXI_MOD_CFG_IND_MODULE_HW_REV_OFFSET    7
#define SXI_MOD_CFG_IND_MODULE_SW_REV_OFFSET    10
#define SXI_MOD_CFG_IND_SXI_REV_OFFSET          13
#define SXI_MOD_CFG_IND_BB_REV_OFFSET           16
#define SXI_MOD_CFG_IND_HDEC_REV_OFFSET         19
#define SXI_MOD_CFG_IND_RF_REV_OFFSET           22
#define SXI_MOD_CFG_IND_CAPABILITY_OFFSET       25
#define SXI_MOD_CFG_IND_SPL_REV_OFFSET          31
#define SXI_MOD_CFG_IND_MAX_SMART_FAV_OFFSET    34
#define SXI_MOD_CFG_IND_MAX_TUNE_MIX_OFFSET     35
#define SXI_MOD_CFG_IND_MAX_SPORT_FLASH_OFFSET  36
#define SXI_MOD_CFG_IND_MAX_TW_NOW_OFFSET       37

/***********************************/
/* SXI POWER MODE COMMAND          */
/***********************************/
/* byte 3 - power mode */
/* See POWER MODES in sxm_sxi.h */
#define SXI_PWR_MODE_MIN_NUM        SXI_PWR_MODE_DOWN
#define SXI_PWR_MODE_MAX_NUM        SXI_PWR_EXIT_LOW_POWER

#define SXI_PWR_MODE_UP             (SXI_PWR_MODE_MAX_NUM + 1)

#define SXI_PWR_MODE_IND_FACTORY_TIMEOUT (10000)
#define SXI_PWR_MODE_IND_DOWN_TIMEOUT (2000)

/***********************************/
/* SXI FEATURE MONITOR COMMAND     */
/***********************************/
/*byte 3 - MonitorOperation */
#define SXI_MONITOR_OP_LIMITED      0x00
#define SXI_MONITOR_OP_FULL         0x01
#define SXI_MONITOR_OP_LIMIT_ALL    0x02
#define SXI_MONITOR_OP_MAX_NUM      SXI_MONITOR_OP_LIMIT_ALL

/* byte 4 - FeatureListCnt */
#define SXI_MONITOR_FEATURE_LIST_MAX_CNT    16
#define SXI_MONITOR_FEATURE_LIST_MAX_NUM    SXI_MONITOR_FEATURE_LIST_MAX_CNT

/* byte 5 - FeatureMonitorID */
#define SXI_MONITOR_FEATURE_TIME            0x00
#define SXI_MONITOR_FEATURE_CHAN            0x01
#define SXI_MONITOR_FEATURE_CAT             0x02
#define SXI_MONITOR_FEATURE_META            0x03
#define SXI_MONITOR_FEATURE_IR_RECORD_META  0x04


/************************************/
/* SXI EXT METADATA MONITOR COMMAND */
/************************************/
/* byte 3 - MonitorSelection */
#define SXI_EXT_TRACK_MON_ACTIVE_BROWSED    0x00
#define SXI_EXT_TRACK_MON_ALL               0x01
#define SXI_EXT_CHAN_MON_ACTIVE_BROWSED     0x02
#define SXI_EXT_CHAN_MON_ALL                0x03
#define SXI_EXT_GLOBAL_MON                  0x04
#define SXI_EXT_LOOK_AHEAD_TRACK_MON        0x05

#define SXI_EXT_MONITOR_SELECTION_MIN_NUM   SXI_EXT_TRACK_MON_ACTIVE_BROWSED
#define SXI_EXT_MONITOR_SELECTION_MAX_NUM   SXI_EXT_LOOK_AHEAD_TRACK_MON

/* byte 4 - MonitorChangeType */
/* SEE DEFINITION OF MONITOR CHANGE TYPE IN SXI STATUS MONITOR COMMAND */

/* byte 5 - ExtMetadataCnt */
#define SXI_EXT_METADATACNT_TMI_CMI_MAX_NUM 32

#define SXI_EXT_METADATACNT_MIN_NUM         1
#define SXI_EXT_METADATACNT_MAX_NUM         SXI_SP_MAX_GLOBAL_MON_LIST_CNT

/* byte 6 - EMI */
/* TMI - Track Metadata Identifier */
/* see sxm_sxi.h */

/* CMI - Channel Metadata Identifier */
#define SXI_CMI_CHAN_SHORT_DESCRIPTION      64
#define SXI_CMI_CHAN_LONG_DESCRIPTION       65
#define SXI_CMI_SIMILAR_CHAN_LIST           66
#define SXI_CMI_CHAN_LIST_ORDER             68
#define SXI_CMI_PLAY_ON_SELECT              69
#define SXI_CMI_CHANNEL_CONTENT_TYPE        70
#define SXI_CMI_PLAYBACK_NAVIGATION_CLASS   71
#define SXI_CMI_SHADOW_RECORD_CLASS         72

/* GMI - Global Metadata Identifier */
#define SXI_GMI_CHAN_METADATA_TABLE_VERSION             258
#define SXI_GMI_CHAN_METADATA_RECORD_COUNT              514
#define SXI_GMI_ITUNES_SXM_URL                          130
#define SXI_GMI_TRAFFIC_WEATHER_TABLE_VERSION           260
#define SXI_GMI_TRAFFIC_WEATHER_RECORD_COUNT            516
#define SXI_GMI_TRAFFIC_WEATHER_ID                      141
#define SXI_GMI_TRAFFIC_WEATHER_ABBREVIATION            142
#define SXI_GMI_TRAFFIC_WEATHER_NAME                    143
#define SXI_GMI_SPORTS_TEAM_TABLE_VERSION               261
#define SXI_GMI_SPORTS_TEAM_RECORD_COUNT                517
#define SXI_GMI_SPORTS_TEAM_ID                          151
#define SXI_GMI_SPORTS_TEAM_ABBREVIATION                152
#define SXI_GMI_SPORTS_TEAM_NAME                        153
#define SXI_GMI_SPORTS_TEAM_NICKNAME                    154
#define SXI_GMI_SPORTS_TEAM_LEAGUE_ID                   155
#define SXI_GMI_SPORTS_TEAM_TIER_LIST                   156
#define SXI_GMI_SPORTS_LEAGUE_TABLE_VERSION             262
#define SXI_GMI_SPORTS_LEAGUE_RECORD_COUNT              518
#define SXI_GMI_SPORTS_LEAGUE_ID                        161
#define SXI_GMI_SPORTS_LEAGUE_SHORT_NAME                162
#define SXI_GMI_SPORTS_LEAGUE_LONG_NAME                 163
#define SXI_GMI_SPORTS_LEAGUE_TYPE                      164
#define SXI_GMI_SPORTS_LEAGUE_SF_ENABLED                165
#define SXI_GMI_SPORTS_LEAGUE_SF_TIERS                  166
#define SXI_GMI_FEATURED_FAVORITES_TABLE_VERSION        263
#define SXI_GMI_FEATURED_FAVORITES_RECORD_COUNT         519
#define SXI_GMI_FEATURED_FAVORITES_BANK_ORDER           81
#define SXI_GMI_FEATURED_FAVORITES_BANK_ID              82
#define SXI_GMI_FEATURED_FAVORITES_BANK_SEQUENCE        83
#define SXI_GMI_FEATURED_FAVORITES_BANK_TITLE_SHORT     84
#define SXI_GMI_FEATURED_FAVORITES_BANK_TITLE_LONG      85
#define SXI_GMI_FEATURED_FAVORITES_BANK_TITLE_VERBOSE   86
#define SXI_GMI_FEATURED_FAVORITES_BANK_DESCRIPTION     87
#define SXI_GMI_FEATURED_FAVORITES_BANK_PURPOSE         88
#define SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT1    89
#define SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT2    90
#define SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT3    91
#define SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT4    92

/* GMI array data type size */
#define SXI_GMI_SPORTS_TEAM_SPORT_ID_DATA_TYPE_SIZE         1
#define SXI_GMI_SPORTS_TEAM_TIER_LIST_DATA_TYPE_SIZE        1

/* GMI MAX SIZE */
#define SXI_GMI_CHAN_METADATA_TABLE_VERSION_MAX_SIZE        2
#define SXI_GMI_CHAN_METADATA_RECORD_COUNT_MAX_SIZE         2
#define SXI_GMI_ITUNES_SXM_URL_MAX_SIZE                     128
#define SXI_GMI_TRAFFIC_WEATHER_TABLE_VERSION_MAX_SIZE      2
#define SXI_GMI_TRAFFIC_WEATHER_RECORD_COUNT_MAX_SIZE       2
#define SXI_GMI_TRAFFIC_WEATHER_ID_MAX_SIZE                 1
#define SXI_GMI_TRAFFIC_WEATHER_ABBREVIATION_MAX_SIZE       3
#define SXI_GMI_TRAFFIC_WEATHER_NAME_MAX_SIZE               24
#define SXI_GMI_SPORTS_TEAM_TABLE_VERSION_MAX_SIZE          2
#define SXI_GMI_SPORTS_TEAM_RECORD_COUNT_MAX_SIZE           2
#define SXI_GMI_SPORTS_TEAM_ID_MAX_SIZE                     2
#define SXI_GMI_SPORTS_TEAM_ABBREVIATION_MAX_SIZE           6
#define SXI_GMI_SPORTS_TEAM_NAME_MAX_SIZE                   24
#define SXI_GMI_SPORTS_TEAM_NICKNAME_MAX_SIZE               16
#define SXI_GMI_SPORTS_TEAM_SPORT_ID_MAX_SIZE               (SXI_GMI_SPORTS_TEAM_SPORT_ID_DATA_TYPE_SIZE*8)
#define SXI_GMI_SPORTS_TEAM_TIER_LIST_MAX_SIZE              (SXI_GMI_SPORTS_TEAM_TIER_LIST_DATA_TYPE_SIZE*8)
#define SXI_GMI_SPORTS_LEAGUE_TABLE_VERSION_MAX_SIZE        2
#define SXI_GMI_SPORTS_LEAGUE_RECORD_COUNT_MAX_SIZE         2
#define SXI_GMI_SPORTS_LEAGUE_ID_MAX_SIZE                   1
#define SXI_GMI_SPORTS_LEAGUE_SHORT_NAME_MAX_SIZE           8
#define SXI_GMI_SPORTS_LEAGUE_LONG_NAME_MAX_SIZE            36
#define SXI_GMI_SPORTS_LEAGUE_TYPE_MAX_SIZE                 16

/***********************************/
/* SXI AUDIO TONAL COMMAND         */
/***********************************/
/* See AUDIO TONE in sxm_sxi.h */
/* byte 3 - Bass */
/* byte 4 - Treble */

/***********************************/
/* SXI EQUALIZER COMMAND           */
/***********************************/
/* byte 3 to byte 12 - BandGain */
/* See AUDIO EQUALIZER in sxm_sxi.h */
#define SXI_BAND_GAIN_SIZE          10

/***********************************/
/* SXI TONE GENERATION COMMAND     */
/***********************************/
/* See TONE GENERATION in sxm_sxi.h */
/* byte 3 - ToneFreqControl */
/* byte 4 - AudioToneOption */
/* bit 5 to bit 2: Alert */
#define SXI_AUDIO_TONE_ALERT_MASK       0x2C
#define SXI_AUDIO_TONE_ALERT_MAX_NUM    SXI_AUDIO_TONE_ALERT_1

/* bit 1 to bit 0: left right enable */
#define SXI_AUDIO_TONE_OPTION_MASK      0x3F

/* byte 5 - ToneVolumeAdj */

/***********************************/
/* SXI AUDIO MUTE COMMAND          */
/***********************************/
/* byte 3 - Mute */
/* See AUDIO MUTE in sxm_sxi.h */
#define SXI_MUTE_MIN_NUM            SXI_UNMUTE
#define SXI_MUTE_MAX_NUM            SXI_MUTE_AUDIO_AND_I2S

/***********************************/
/* SXI AUDIO VOLUME COMMAND        */
/***********************************/
/* byte 3 - VolumeAdj */
/* See AUDIO VOLUME ADJUSTMENT in sxm_sxi.h */

/***********************************/
/* SXI TIME CONFIGURE COMMAND      */
/***********************************/
/* byte 3 - TimeZone */
/* See TIME ZONES in sxm_sxi.h */
#define SXI_TIME_ZONE_MAX_NUM       SXI_HAWAIIAN_TIME_ZONE

/* byte 4 - DSTObs */
/* See DST ADJUSTMENTS in sxm_sxi.h */
#define SXI_DST_OBS_MAX_NUM         SXI_AUTO_DST_ADJUSTMENT

/***********************************/
/* SXI CHANNEL SELECT COMMAND      */
/***********************************/
/* byte 3: SelectionType */
#define SXI_USE_SID_TO_TUNE                         0x00
#define SXI_USE_CHAN_TO_TUNE                        0x01
#define SXI_CAT_UP_TUNE                             0x02
#define SXI_CAT_DWN_TUNE                            0x03
#define SXI_SCAN_CHAN_IN_CATID                      0x04
#define SXI_SCAN_INCLUDE_LIST                       0x05
#define SXI_SCAN_SMART_FAVORITE_CONTENT             0x06
#define SXI_SCAN_SMART_FAVORITE_MUSIC               0x07
#define SXI_SCAN_SKIP_BACK                          0x08
#define SXI_SCAN_SKIP_FORWARD                       0x09
#define SXI_SCAN_STOP_CURRENT_TRACK                 0x0A
#define SXI_SCAN_ABORT_RESUME_PLAYBACK              0x0B
#define SXI_SCAN_SKIP_ONE_APPOS_OF_CUR_DIRECTION    0x0C
#define SXI_SCAN_CONTENT_CRITERIA_REPORT            0x0D
#define SXI_SCAN_MUSIC_ONLY_CRITERIA                0x0E
#define SXI_PLAY_BULLETIN                           0x0F
#define SXI_ABORT_BULLETIN                          0x10
#define SXI_FLASH_EVENT                             0x11
#define SXI_REMAIN_ON_FLASH_EVENT_CHANNEL           0x12
#define SXI_ABORT_FLASH_EVENT                       0x13
#define SXI_START_HOST_PLAYBACK                     0x14
#define SXI_ABORT_HOST_PLAYBACK                     0x15


/* byte 4 and byte 5 - ChanIDorSID */
/* FlashEventID and BulletinEventID can be any 16 bit number */
#define SXI_CHANIDORSID_MIN_NUM                    0
#define SXI_CHANIDORSID_MAX_NUM                    0xFFFF

#define SXI_SELECTIONTYPE_MIN_NUM                  SXI_USE_SID_TO_TUNE
#define SXI_SELECTIONTYPE_MAX_NUM                  SXI_ABORT_HOST_PLAYBACK

/* byte 6: CatID */
#define SXI_CATID_DONTCARE                         0x00
#define SXI_USER_CATID_MIN_NUM              SXI_USER_CATID_64
#define SXI_USER_CATID_MAX_NUM              SXI_USER_CATID_66

/* byte 7: Overrides */
/* See OVERRIDES in sxm_sdk.h */
#define SXI_OVERRIDE_MASK                          0x3f

/* byte 8: Routing */

/* bit 7: reserved */
/* bit 6: reserved */
/* bit 5: reserved */
/* bit 4: reserved */
/* bit 3: reserved */
/* bit 2: reserved */

/* bit 1 - Recorded Channel */
#define SXI_RECORD_PATH_DISABLED    0x00
#define SXI_RECORD_PATH_ENABLED     0x02

/* bit 0 - I2S Audio Path */
#define SXI_I2S_PATH_DISABLED       0x00
#define SXI_I2S_PATH_ENABLED        0x01

/* overall Routing definition */
#define SXI_ROUTING_MASK            0x03

/***********************************/
/* SXI CHAN BROWSE COMMAND         */
/***********************************/
/* byte 3 - channel browse type */
#define SXI_BROWSE_BY_SID                   0x00
#define SXI_BROWSE_BY_CHANID                0x01
#define SXI_BROWSE_NEXT_CHAN_IN_CAT         0x02
#define SXI_BROWSE_PREV_CHAN_IN_CAT         0x03
#define SXI_BROWSE_FIRST_CHAN_NEXT_CAT      0x04
#define SXI_BROWSE_FIRST_CHAN_PREV_CAT      0x05

#define SXI_BROWSE_TYPE_MIN_NUM             SXI_BROWSE_BY_SID
#define SXI_BROWSE_TYPE_MAX_NUM             SXI_BROWSE_FIRST_CHAN_PREV_CAT

/* byte 7 - overrides */
/* SEE DEFINITION OF OVERRIDES IN SXI CHANNEL SELECT COMMAND */

/***********************************/
/* SXI USER CAT CFG COMMAND        */
/***********************************/

/* byte 3 - CatID */
/* See User Categories in sxm_sxi.h */

/***********************************/
/* SXI CHAN ATTRIBUTE CFG COMMAND  */
/***********************************/
/* byte 3 - AttributeChangeType */
/* see CONFIGURABLE CHANNEL ATTRIBUTES in sxm_sdk.h */
#define SXI_ATTRIBUTE_CHANGE_MIN_NUM    SXI_ATTRIBUTE_APPLY_SKIP
#define SXI_ATTRIBUTE_CHANGE_MAX_NUM    SXI_ATTRIBUTE_SCAN_EXCLUDE

/***********************************/
/* SXI CHAN SELECT CFG COMMAND     */
/***********************************/
/* byte 3 - FeatureControl */
/* reserved */

/* byte 4 - PlayPoint */
#define SXI_ADV_IR_PLAY_AT_LIVE         0x00
#define SXI_ADV_IR_START_AT_NEW_TRACK   0x01
#define SXI_ADV_IR_AUTO_SELECT          0x02

#define SXI_PLAYPOINT_MIN_NUM          SXI_ADV_IR_PLAY_AT_LIVE
#define SXI_PLAYPOINT_MAX_NUM          SXI_ADV_IR_AUTO_SELECT


/* byte 5 - PlaySeconds */
/* num of seconds to play a track before going to next scan track */
/* valid values are from 2 to 15 seconds */
#define PLAY_SEC_DEFAULT                6

/* byte 6,7 - ChannelScanInclude */
/* bit 15 to bit 9 are reserved */

/* bit 8 - User Cat3 Cats */
#define SXI_USER_CAT3_CATS_EXCLUDE  0x0000
#define SXI_USER_CAT3_CATS_INCLUDE  0x0100

/* bit 7 - User Cat2 Cats */
#define SXI_USER_CAT2_CATS_EXCLUDE  0x0000
#define SXI_USER_CAT2_CATS_INCLUDE  0x0080

/* bit 6 - User Cat1 Cats */
#define SXI_USER_CAT1_CATS_EXCLUDE  0x0000
#define SXI_USER_CAT1_CATS_INCLUDE  0x0040

/* bit 5 - User Cat3 */
#define SXI_USER_CAT3_EXCLUDE       0x0000
#define SXI_USER_CAT3_INCLUDE       0x0020

/* bit 4 - User Cat2 */
#define SXI_USER_CAT2_EXCLUDE       0x0000
#define SXI_USER_CAT2_INCLUDE       0x0010

/* bit 3 - User Cat1 */
#define SXI_USER_CAT1_EXCLUDE       0x0000
#define SXI_USER_CAT1_INCLUDE       0x0008

/* bit 2 - Active Category */
#define SXI_ACT_CAT_EXCLUDE         0x0000
#define SXI_ACT_CAT_INCLUDE         0x0004

/* bit 1 - SmartFavs */
#define SXI_SMT_FAV_EXCLUDE         0x0000
#define SXI_SMT_FAV_INCLUDE         0x0002

/* bit 0 - Included */
#define SXI_INC_LIST_EXCLUDE        0x0000
#define SXI_INC_LIST_INCLUDE        0x0001

/* byte 8,9 - ChannelScanExclude */
/* bit 15 to bit 1 are reseved */

/* bit 1 - Excluded */
#define SXI_EXC_LIST_INCLUDE        0x0000
#define SXI_EXC_LIST_EXCLUDE        0x0001

/***********************************/
/* SXI CHAN ATTRIBUTE LIST COMMAND */
/***********************************/
/* byte 3 - AttributeChangeType */
#define SXI_ATTRIBUTE_APPLY_SCAN_INCLUDE            0x00
#define SXI_ATTRIBUTE_APPLY_SMART_FAVORITE          0x01
#define SXI_ATTRIBUTE_APPLY_SMART_FAVORITE_SCAN     0x02

/* reserved from 3 to 31 */
#define SXI_ATTRIBUTE_APPLY_RESERVED03              0x03
#define SXI_ATTRIBUTE_APPLY_RESERVED31              0x1F

/* sport flash */
#define SXI_ATTRIBUTE_APPLY_SPORTS_FLASH            0x20

/* reserved from 33 to 63 */
#define SXI_ATTRIBUTE_APPLY_RESERVED33              0x21
#define SXI_ATTRIBUTE_APPLY_RESERVED63              0x3F

/* Tune Mix */
#define SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH1            0x40
#define SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH2            0x41
#define SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH3            0x42
#define SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH4            0x43
#define SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH5            0x44
#define SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH6            0x45
#define SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH7            0x46
#define SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH8            0x47
#define SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH9            0x48
#define SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH10           0x49

/* max number */
#define SXI_ATTRIBUTE_APPLY_MIN_NUM                 SXI_ATTRIBUTE_APPLY_SCAN_INCLUDE
#define SXI_ATTRIBUTE_APPLY_MAX_NUM                 SXI_ATTRIBUTE_APPLY_TUNE_MIX_CH10

/***********************************/
/* SXI IR PLAYBACK CONTROL COMMAND */
/***********************************/
/* byte 3 - Control */
#define SXI_LIVE                        0x01
#define SXI_PLAY                        0x02
#define SXI_PAUSE                       0x03
#define SXI_NEXT                        0x04
#define SXI_PREVIOUS                    0x05
#define SXI_JUMP_TO_START               0x06
#define SXI_JUMP_TO_PLAYBACKID          0x07
#define SXI_JUMP_TO_TIMEOFFSET          0x08
#define SXI_NEXT_RESUME                 0x09
#define SXI_PREVIOUS_RESUME             0x0A
#define SXI_JUMP_TO_START_RESUME        0x0B
#define SXI_JUMP_TO_PLAYBACKID_RESUME   0x0C
#define SXI_JUMP_TO_TIMEOFFSET_RESUME   0x0D

#define SXI_PLAYBACKCONTROL_MIN_NUM     SXI_LIVE
#define SXI_PLAYBACKCONTROL_MAX_NUM     SXI_JUMP_TO_TIMEOFFSET_RESUME

/* byte 4 and byte 5 - TimeOffset */
#define SXI_TIMEOFFSET_MIN_NUM          -32768
#define SXI_TIMEOFFSET_MAX_NUM           32767

/* byte 6 and byte 7 - PlaybackID */
#define SXI_PLAYBACKID_MIN_NUM          0
#define SXI_PLAYBACKID_MAX_NUM          65535

/***********************************/
/* SXI IR RECORD INFO INDICATION   */
/***********************************/
/* byte 4 - RecordState */
/* See IR RECORDING in sxm_sxi.h */

/***********************************/
/* SXI BULLETIN MONITOR COMMAND    */
/***********************************/
/* byte 3 - BulletinType */
#define SXI_TRAFFIC_WEATHER_NOW         0x00

#define SXI_BULLETINTYPE_MIN_NUM       SXI_TRAFFIC_WEATHER_NOW
#define SXI_BULLETINTYPE_MAX_NUM       SXI_TRAFFIC_WEATHER_NOW

/* byte 4 - BulletinParamCnt */
#define SXI_BULLETINPARAMCNT_MIN_NUM    0
#define SXI_BULLETINPARAMCNT_MAX_NUM    255

/***********************************/
/* SXI STATUS MONITOR COMMAND      */
/***********************************/
/* byte 3 - MonitorChangeType */
/* See MONITOR CHANGE TYPES in sxm_sxi.h */
#define SXI_MONITOR_CHANGE_TYPE_MIN_NUM  SXI_MONITOR_DISABLED
#define SXI_MONITOR_CHANGE_TYPE_MAX_NUM  SXI_MONITOR_ALL_DISABLED

/* byte 4 StatusListCnt */
#define SXI_STATUS_LIST_CNT_MIN_NUM     1
#define SXI_STATUS_LIST_CNT_MAX_NUM     SXI_SP_MAX_STATUS_MON_LIST_CNT

/* byte 5 - StatusMonitorItem */
/* See STATUS MONITOR ITEMS in sxm_sxi.h */
/* Additional items: */
#define SXI_MONITOR_CHANNEL_IMPULSE_RESPONSE    0x32
#define SXI_MONITOR_EXTRACTION_MANAGER_STATUS   0x33
#define SXI_MONITOR_DECODER_STATUS              0x34
#define SXI_MONITOR_SIGNAL_OFFSETS              0x35
#define SXI_MONITOR_OVERLAY_PIPE_CONFIGURATION  0x36
#define SXI_MONITOR_DATA_LAYER_STATUS           0x37
#define SXI_MONITOR_MODULE_QUEUE_STATUS         0x39
#define SXI_MONITOR_MFC_ONE_COUNTER             0x3A
#define SXI_MONITOR_AUDIO_DECODER_STATUS        0x3B
#define SXI_MONITOR_UPC_VALUES                  0x3C
#define SXI_MONITOR_EXTENDED_SIGNAL_QUALITY     0x3D
#define SXI_MONITOR_OVERLAY_SIGNAL_QUALITY      0x3E
#define SXI_MONITOR_TYPE_APPROVAL_MONITOR       0x3F

/*************************************/
/* SXI RECORD CONTENT SELECT COMMAND */
/*************************************/

/* byte 3 - SelectionType */
#define SXI_REC_LIVE                    0x00
#define SXI_REC_START                   0x01
#define SXI_REC_PAUSE                   0x02
#define SXI_REC_RESUME                  0x03
#define SXI_REC_START_BLOCK             0x04
#define SXI_REC_START_TRACK             0x05
#define SXI_REC_ORDERED_FILTER          0x06

#define SXI_REC_MIN_NUM                 SXI_REC_LIVE
#define SXI_REC_MAX_NUM                 SXI_REC_ORDERED_FILTER


/*************************************/
/* SXI PLAYBACK AUDIO PACKET COMMAND */
/*************************************/

/* byte 7 - AudioEncoderType */
#define SXI_AUDIO_ENCODER_USAC          0x06

/* byte 8 - AudioBitRate */
#define SXI_AUDIO_BITRATE_8 8
#define SXI_AUDIO_BITRATE_10 10
#define SXI_AUDIO_BITRATE_12 12
#define SXI_AUDIO_BITRATE_20 20
#define SXI_AUDIO_BITRATE_22 22
#define SXI_AUDIO_BITRATE_24 24
#define SXI_AUDIO_BITRATE_32 32

/* byte 9 - playback status */
#define SXI_AUDIO_PLAYBACK_STATUS_LAST_MASK (0x01)

/* byte 10 and byte 11 - NumDataBytes */
#define SXI_AUDIO_DATA_MIN_NUM          1
#define SXI_AUDIO_DATA_MAX_NUM          SXI_SP_MAX_AUDIO_PACKET_BYTES

/***********************************/
/* SXI DATA SERVICE MONITOR COMMAND*/
/***********************************/
/* byte 3 - UpdateType */
#define SXI_DATA_START_MONITOR          0x00
#define SXI_DATA_HALT_SERVICE_MONITOR   0x01
#define SXI_DATA_HALT_ALL_MONITOR       0x02

#define SXI_UPDATETYPE_MIN_NUM          SXI_DATA_START_MONITOR
#define SXI_UPDATETYPE_MAX_NUM          SXI_DATA_HALT_ALL_MONITOR


/* byte 4 - DSI */
#define SXI_ELECTRONIC_PROGRAM_GUIDE    300
#define SXI_CHANNEL_GRAPHICS_UPDATE     311
#define SXI_PHONETICS_UPDATE            330
#define SXI_XM_NAV_TRAFFIC              400
#define SXI_XM_WX_WEATHER_APP_ID_10     410
#define SXI_XM_WX_WEATHER_APP_ID_230    230
#define SXI_XM_WX_WEATHER_APP_ID_231    231
#define SXI_XM_WX_WEATHER_APP_ID_232    232
#define SXI_XM_WX_WEATHER_APP_ID_234    234
#define SXI_XM_WX_WEATHER_APP_ID_235    235
#define SXI_XM_WX_WEATHER_APP_ID_236    236
#define SXI_XM_WX_WEATHER_APP_ID_237    237
#define SXI_XM_WX_WEATHER_APP_ID_238    238
#define SXI_XM_NAV_WEATHER              430
#define SXI_SXM_WEATHER_TABULAR         440
#define SXI_SXM_WEATHER_ALERTS          441
#define SXI_SXM_WEATHER_GRAPHICAL       442
#define SXI_SXM_MARINE_INLAND           443
#define SXI_SXM_MARINE_MARINER          444
#define SXI_SXM_MARINE_VOYAGER          445
#define SXI_SXM_AVIATION_BASIC          446
#define SXI_SXM_AVIATION_STANDARD       447
#define SXI_SXM_AVIATION_PREMIUM        448
#define SXI_TRAFFIC                     480
#define SXI_APOGEE_TRAFFIC_0            490
#define SXI_APOGEE_TRAFFIC_1            491
#define SXI_APOGEE_TRAFFIC_CAMERAS      492
#define SXI_STOCK_TICKER                500
#define SXI_AIR_TRAVEL                  520
#define SXI_SXM_SPORTS                  530
#define SXI_FUEL_PRICES                 600
#define SXI_MOVIE_TIMES                 610
#define SXI_SAFETY_CAMERA               620

/***********************************/
/* SXI DATA SERVICE FILTER COMMAND */
/***********************************/

/* byte 3 - UpdateType */
#define SXI_FILTER_DMI_START            0
#define SXI_FILTER_DMI_HALT             1

#define SXI_FILTER_UPDATETYPE_MIN_NUM   SXI_FILTER_DMI_START
#define SXI_FILTER_UPDATETYPE_MAX_NUM   SXI_FILTER_DMI_HALT

/* byte 6 - DMICnt */
#define SXI_FILTER_DMICNT_MIN_NUM       0
#define SXI_FILTER_DMICNT_MAX_NUM       SXI_SP_MAX_DMI_CNT


/***********************************/
/* SXI FWU CONFIGURE COMMAND       */
/***********************************/
/* byte 3 */
#define SXI_FWU_BYTES_IN_PACKET_64      64
#define SXI_FWU_BYTES_IN_PACKET_256     SXI_SP_MAX_FWU_BYTES_PER_PACKET

/***********************************/
/* SXI FA ACTIVATION COMMAND       */
/***********************************/

/* byte 3 - Delay */
#define SXI_FA_DELAY_MIN_NUM               0
#define SXI_FA_DELAY_MAX_NUM               0xFF

/* byte 4 and 5 - RetryCnt  */
#define SXI_FA_ACT_RETRYCNT_MIN_NUM        0
#define SXI_FA_ACT_RETRYCNT_MAX_NUM        0xFFFF

/* byte 6 - Model_string */
#define SXI_FA_ACT_MODEL_STRING_MIN_NUM   1
#define SXI_FA_ACT_MODEL_STRING_MAX_NUM   60

/***************************************/
/* SXI MEMORY MONITOR SETUP            */
/***************************************/

/* byte 3 */
#define SXI_MEMMON_ONETIMEREAD_ID (0)

#define SXI_SP_MAX_MEMORY_MONITORS (16)

/* byte 4 */
#define SXI_MEMMON_REGION_DIRECT      0
#define SXI_MEMMON_REGION_TUNER       1
#define SXI_MEMMON_REGION_HDEC        2
#define SXI_MEMMON_REGION_EEPROM      3
#define SXI_MEMMON_REGION_CDEC        4
#define SXI_MEMMON_REGION_SDEC        5
#define SXI_MEMMON_REGION_DAP         6
#define SXI_MEMMON_REGION_RESERVED0   7
#define SXI_MEMMON_REGION_FLASH       8
#define SXI_MEMMON_REGION_RUDE_WORDS  9
#define SXI_MEMMON_REGION_SID_LISTS   10
#define SXI_MEMMON_REGION_DECODER     11
#define SXI_MEMMON_REGION_TSCW        12
#define SXI_MEMMON_REGION_DSMT        13
#define SXI_MEMMON_REGION_TDS         14
#define SXI_MEMMON_REGION_SID2PCIDPRC 15

#define SXM_MEMMON_REGION_MIN         SXI_MEMMON_REGION_DIRECT
#define SXM_MEMMON_REGION_MAX         SXI_MEMMON_REGION_SID2PCIDPRC

/* byte 11 */
#define SXI_MEMMON_DATATYPE_UNIT8    0
#define SXI_MEMMON_DATATYPE_UNIT16   1
#define SXI_MEMMON_DATATYPE_UNIT32   2

#define SXI_MEMMON_DATATYPE_MIN       SXI_MEMMON_DATATYPE_UNIT8
#define SXI_MEMMON_DATATYPE_MAX       SXI_MEMMON_DATATYPE_UNIT32

/***********************************/
/* SXI MEMORY WRITE COMMAND        */
/***********************************/

/* byte 3 Regions */
#define SXI_MEMWRITE_REGION_DIRECT      0
#define SXI_MEMWRITE_REGION_TUNER       1
#define SXI_MEMWRITE_REGION_HDEC        2
#define SXI_MEMWRITE_REGION_EEPROM      3
#define SXI_MEMWRITE_REGION_CDEC        4
#define SXI_MEMWRITE_REGION_SDEC        5
#define SXI_MEMWRITE_REGION_DAP         6
#define SXI_MEMWRITE_REGION_RESERVED0   7
#define SXI_MEMWRITE_REGION_FLASH       8
#define SXI_MEMWRITE_REGION_RESERVED1   9
#define SXI_MEMWRITE_REGION_RESERVED2   10

#define SXM_MEMWRITE_REGION_MIN         SXI_MEMWRITE_REGION_DIRECT
#define SXM_MEMWRITE_REGION_MAX         SXI_MEMWRITE_REGION_FLASH


/* byte 10 */
#define SXI_MEMWRITE_DATATYPE_UNIT8    0
#define SXI_MEMWRITE_DATATYPE_UNIT16   1
#define SXI_MEMWRITE_DATATYPE_UNIT32   2

#define SXI_MEMWRITE_DATATYPE_MIN       SXI_MEMWRITE_DATATYPE_UNIT8
#define SXI_MEMWRITE_DATATYPE_MAX       SXI_MEMWRITE_DATATYPE_UNIT32

/***********************************/
/* SXI SEEK COMMAND                */
/***********************************/
/* byte 3 - SeekMonID */
#define SXI_SEEK_MONITOR_0      0
#define SXI_SEEK_MONITOR_1      1
#define SXI_SEEK_MONITOR_2      2
#define SXI_SEEK_MONITOR_3      3
#define SXI_SEEK_MONITOR_4      4
#define SXI_SEEK_MONITOR_5      5
#define SXI_SEEK_MONITOR_6      6
#define SXI_SEEK_MONITOR_7      7

#define SXI_SEEK_MONITOR_MIN_NUM    SXI_SEEK_MONITOR_0
#define SXI_SEEK_MONITOR_MAX_NUM    SXI_SEEK_MONITOR_7

/* byte 4 - MonChangeType */
/* SEE DEFINITION OF MONITOR CHANGE TYPE IN SXI STATUS MONITOR COMMAND */

/* byte 5 and byte 6 - MonTMI */
/* see TMI - TRACK METADATA IDENTIFIERS in sxm_sxi.h */

/* byte 7 - MonValueCnt */
#define SXI_SEEK_MONVALUECNT_MIN_NUM    1
#define SXI_SEEK_MONVALUECNT_MAX_NUM    SXI_SP_MAX_MON_VALUE_CNT

/* byte 8 - MonValueLen */
#define SXI_SEEK_MON_VALUE_UINT8    1
#define SXI_SEEK_MON_VALUE_UINT16   2
#define SXI_SEEK_MON_VALUE_UINT32   4

/* byte 9 - ReportTMICnt */
#define SXI_SEEK_REPORTTMICNT_MIN_NUM   0
#define SXI_SEEK_REPORTTMICNT_MAX_NUM   32

/* byte x - SeekControl */
/* bit 1 */
#define SXI_SEEK_IMMEDIATE_DISABLED     0x00
#define SXI_SEEK_IMMEDIATE_ENABLED      0x02

/* bit 0 */
#define SXI_SEEK_END_DISABLED           0x00
#define SXI_SEEK_END_ENABLED            0x01

/******************************************************************************/
/*==========================> SXI RESPONSE DEFINITION  <======================*/
/******************************************************************************/

/***********************************/
/* SXI RESPONSE OPCODE             */
/***********************************/
#define SXI_MODULE_CFG_RESP             0xC020
#define SXI_PWR_MODE_RESP               0xC021
#define SXI_TIME_CFG_RESP               0xC060
#define SXI_STATUS_MON_RESP             0xC0A0
#define SXI_FEATURE_MON_RESP            0xC0A1
#define SXI_EXT_METADATA_MON_RESP       0xC0A2
#define SXI_PING_RESP                   0xC0E0
#define SXI_IP_DEVICE_AUTH_RESP         0xC0F0
#define SXI_AUTH_STATE_RESP             0xC0F1
#define SXI_AUDIO_MUTE_RESP             0xC100
#define SXI_AUDIO_VOL_RESP              0xC101
#define SXI_AUDIO_TONAL_RESP            0xC102
#define SXI_AUDIO_EQ_RESP               0xC104
#define SXI_TONE_GEN_RESP               0xC180
#define SXI_CHAN_BROWSE_RESP            0xC200
#define SXI_USER_CAT_CFG_RESP           0xC202
#define SXI_CHAN_SELECT_RESP            0xC280
#define SXI_CHAN_ATTRIB_CFG_RESP        0xC282
#define SXI_CHAN_SELECT_CFG_RESP        0xC283
#define SXI_CHAN_ATTRIB_LIST_RESP       0xC284
#define SXI_SEEK_RESP                   0xC304
#define SXI_IR_PLAYBACK_CONTROL_RESP    0xC402
#define SXI_BULLETIN_MON_RESP           0xC420
#define SXI_DATA_SERVICE_MON_RESP       0xC500
#define SXI_DATA_SERVICE_FILTER_RESP    0xC501
#define SXI_FWU_PROG_RESP               0xCE81
#define SXI_FWU_CFG_RESP                0xCE82
#define SXI_FWU_PKG_RESP                0xCE83
#define SXI_FA_RESP                     0xCEC0
#define SXI_FA_PKG_RESP                 0xCED0
#define SXI_MODULE_RESET_RESP           0xCF00
#define SXI_MEM_WRITE_RESP              0xCF03
#define SXI_MEM_MON_SETUP_RESP          0xCF04
#define SXI_DELETE_MEM_MON_RESP         0xCF05
#define SXI_TUNNEL_RESP                 0xCF07
#define SXI_DEBUG_ENABLE_RESP           0xCF09

/***********************************/
/* SXI RESPONSE TIMOUTS            */
/***********************************/
#define SXI_COMMAND_RESP_DEFAULT_TIMEOUT (200)
#define SXI_FWU_PROG_RESP_TIMEOUT        (5000)
#define SXI_FWU_PKG_RESP_TIMEOUT         (3000)

/******************************************************************************/
/*==========================> SXI INDICATION DEFINITION  <====================*/
/******************************************************************************/

/***********************************/
/* SXI INDICATION OPCODE           */
/***********************************/
#define SXI_MODULE_CFG_IND              0x8020
#define SXI_PWR_MODE_IND                0x8021
#define SXI_TIME_IND                    0x8060
#define SXI_EVENT_IND                   0x8080
#define SXI_STATUS_IND                  0x80A0
#define SXI_DISP_ADVISORY_IND           0x80C0
#define SXI_SUB_STATUS_IND              0x80C1
#define SXI_IP_DEVICE_AUTH_IND          0x80F0
#define SXI_AUTH_STATE_IND              0x80F1
#define SXI_CHAN_BROWSE_IND             0x8200
#define SXI_CAT_INFO_IND                0x8201
#define SXI_CHAN_SELECT_IND             0x8280
#define SXI_CHAN_INFO_IND               0x8281
#define SXI_META_DATA_IND               0x8300
#define SXI_CHAN_METADATA_IND           0x8301
#define SXI_GLOBAL_METADATA_IND         0x8302
#define SXI_LA_METADATA_IND             0x8303
#define SXI_SEEK_IND                    0x8304
#define SXI_MREF_IND                    0x8310
#define SXI_LAMREF_IND                  0x8311
#define SXI_IR_PLAYBACK_INFO_IND        0x8402
#define SXI_IR_PLAYBACK_METADATA_IND    0x8403
#define SXI_IR_RECORD_INFO_IND          0x8404
#define SXI_IR_RECORD_METADATA_IND      0x8405
#define SXI_BULLETIN_STATUS_IND         0x8420
#define SXI_FLASH_IND                   0x8421
#define SXI_CONTENT_BUFFERED_IND        0x8422
#define SXI_AUDIO_REQUEST_IND           0x8441
#define SXI_DATA_SERVICE_STATUS_IND     0x8500
#define SXI_DATA_PACKET_IND             0x8510
#define SXI_FWU_ERASE_IND               0x8E84
#define SXI_FA_IND                      0x8EC0
#define SXI_FA_PKG_IND                  0x8ED0
#define SXI_MEM_MON_IND                 0x8F01
#define SXI_DEBUG_MODULE_IND            0x8F08

/***********************************/
/* SXI TIME INDICATION             */
/***********************************/
/* offset */
#define SXI_TIME_IND_MINUTE_OFFSET              4
#define SXI_TIME_IND_HOUR_OFFSET                5
#define SXI_TIME_IND_DAY_OFFSET                 6
#define SXI_TIME_IND_MONTH_OFFSET               7
#define SXI_TIME_IND_YEAR_OFFSET                8

/***********************************/
/* SXI EVENT INDICATION            */
/***********************************/
/* byte 3 - EventCode */
/* see SXI EVENT in sxm_sxi.h */

/***********************************/
/* SXI DISPLAY ADVISORY INDICATION */
/***********************************/
/*byte 4 */
#define SXI_CHANINFO_INVALID        0x00
#define SXI_CHANINFO_VAILD          0x01

/***********************************/
/* SXI CHANNEL SELECT INDICATION   */
/***********************************/
/* byte 12: channel attributes */

/* bit 7: reserved */
/* bit 6: reserved */
/* See CHANNEL ATTRIBUTES in sxm_sdk.h */

/* byte 13 - RecordRestriction  */

/* bit 7: reserved */
/* bit 6: reserved */
/* bit 5: reserved */
/* bit 4: reserved */
/* bit 3: reserved */

/* bit 2: channel record */
/* See RECORD RESTRICTIONS in sxm_sdk.h */

/***********************************/
/* SXI CHANNEL BROWSE INDICATION   */
/***********************************/
/* byte 8: ChanAttributes */
/* SEE SXI CHANNEL SELECT INDICATION FOR DEFINITION OF THIS BYTE */

/* byte 15 - RecordRestriction */
/* SEE SXI CHANNEL SELECT INDICATION FOR DEFINITION OF THIS BYTE */

/* Indication timeout after Response */
#define SXI_CHAN_BROWSE_IND_TIMEOUT (2000) /* There is no SXi spec for this */

/*************************************/
/* SXI CHANNEL INFOMATION INDICATION */
/*************************************/
/* byte 8 - ChanAttributes */
/* SEE SXI CHANNEL SELECT INDICATION FOR DEFINITION OF THIS BYTE */

/* byte 9 - RecordRestrictions */
/* SEE SXI CHANNEL SELECT INDICATION FOR DEFINITION OF THIS BYTE */


/***********************************/
/* SXI META DATA INDICATION        */
/***********************************/
/* byte 12 - RecordRestriction  */
/* SEE SXI CHANNEL SELECT INDICATION FOR DEFINITION OF THIS BYTE */

/***********************************/
/* SXI IR PLAYBACK INFO INDICATION */
/***********************************/
/* byte 4 - PlaybackState */
/* see Playback States in sxm_sdk.h */

/***************************************/
/* SXI IR PLAYBACK METADATA INDICATION */
/***************************************/
/* byte 4 - Status */
/* See Content Storage Status in sxm_sdk.h */

/***************************************/
/* SXI IR RECORD METADATA INDICATION   */
/***************************************/
/* byte 4 - Status */
#define SXI_IR_ENDING_NOT_STORED    0x00
#define SXI_IR_ENDING_STORED        0x10

#define SXI_IR_BEGINNING_NOT_STORED 0x00
#define SXI_IR_BEGINNING_STORED     0x08

#define SXI_IR_RECORING_ALLOWED     0x00
#define SXI_IR_RECORDING_RESTRICTED 0x80

/***************************************/
/* SXI BULLETIN STATUS INDICATION      */
/***************************************/
/* byte 4 - BulletinType */
/* SEE SXI BULLETIN MONITOR COMMAND FOR DEFINITION OF THIS BYTE */
#define SXI_BULLETIN_TYPE_OFFSET        4

/* byte 8 - BulletinStatus */
#define SXI_NO_BULLETIN_AVAILABLE       0x00
#define SXI_NEW_BULLETIN_AVAILABLE      0x01
#define SXI_PLAYED_BULLETIN_AVAILABLE   0x02


/***************************************/
/* SXI FLASH INDICATION                */
/***************************************/
/* byte 13 FlashType */
/* See Flash Type in sxm_sdk.h */
#define FLASH_EVENT_TYPE_OFFSET         13

/* byte 14 - FlashStatus */
/* see FLASH STATUSES in sxm_sdk.h */

/***********************************/
/* SXI STATUS INDICATION           */
/***********************************/
/* byte 4 - MonitorItemID */
/* See STATUS MONITOR ITEMS in sxm_sxi.h */

/* byte 5 - MonitorItemValue */
/* See STATUS VALUES in sxm_sxi.h */

/***************************************/
/* SXI MEMORY MONITOR INDICATION       */
/***************************************/
/* byte 6: DataType */
/* See MEMORY MONITOR DATA TYPES in sxm_sxi.h */

/* Indication wait timeout for one-time-read monitor */
#define SXI_MEM_MON_ONETIMEREAD_IND_TIMEOUT     2000

/***********************************/
/* SXI SUBSCRIPTION STATUS IND     */
/***********************************/
/* See SUBSCRIPTION STATUS in sxm_sxi.h */
/* byte 13 - subscription status */
#define SXI_SUB_STATUS_OFFSET       13

/* byte 18 - reason text size */
/* byte xx - phone number text size */

/***************************************/
/* SXI DATA SERVICE STATUS INDICATION  */
/***************************************/
/* byte 5 - DataServiceStatus */
#define SXI_FULLY_SUBSCRIBED            0x00
#define SXI_PARTIALLY_SUBSCRIBED        0x01
#define SXI_UNSUBSCRIBED                0x02
#define SXI_INVALID_DSI                 0x03
#define SXI_INSUFF_SYS_RESOURCES        0x04

/* byte 8 - DMI - (APPNID) */
#define SXI_XM_NAV_TRAFFIC_DMI           8
#define SXI_APPID_10_WX_PRODUCTS        10
#define SXI_XM_NAV_WEATHER_DMI          14
#define SXI_PROPRIETARY_0               20
#define SXI_PROPRIETARY_1               21
#define SXI_PROPRIETARY_2               22
#define SXI_STOCK_SYMBOLS               50
#define SXI_STOCK_VALUES                51
#define SXI_STOCK_EXTENDED_VALUES       52
#define SXI_STOCK_SPORTS_PROVIDER_ID    53
#define SXI_SPORTS_MENU_ITEM            60
#define SXI_SPORTS_MENU_LEVEL           61
#define SXI_SPORTS_MENU_DISPLAY         62
#define SXI_SPORTS_MENU_ALERT           63
#define SXI_PROPRIETARY_3               64
#define SXI_PROPRIETARY_4               65
#define SXI_CHANNEL_GRAPICS_LOGOS       70
#define SXI_CHANNEL_GRAPICS_REFERENCES  71
#define SXI_PROPRIETARY_5               77
#define SXI_PROPRIETARY_6               101
#define SXI_PROPRIETARY_7               102
#define SXI_PROPRIETARY_8               103
#define SXI_PROPRIETARY_9               104
#define SXI_APPID_230_WX_PRODUCTS       230
#define SXI_APPID_231_WX_PRODUCTS       231
#define SXI_APPID_232_WX_PRODUCTS       232
#define SXI_APPID_234_WX_PRODUCTS       234
#define SXI_APPID_235_WX_PRODUCTS       235
#define SXI_APPID_236_WX_PRODUCTS       236
#define SXI_APPID_237_WX_PRODUCTS       237
#define SXI_APPID_238_WX_PRODUCTS       238
#define SXI_PROPRIETARY_10              256
#define SXI_PROPRIETARY_11              319

/***************************************/
/* SXI DATA PACKET INDICATION          */
/***************************************/
/* byte 3: PacketType */
#define SXI_SDTP_PACKET         0x00
#define SXI_XMAPP_PACKET        0x01
#define SXI_RAW_PACKET          0x02

/***************************************/
/* SXI BULLETIN STATUS INDICATION      */
/***************************************/
/* byte 4: BulletinType */
#define SXI_STATUS_IND_BULLETIN_TYPE_TWN        0

/*******************************/
/* SXI FWU ERASE COMMAND       */
/*******************************/
#define SXI_FWU_ERASE_IND_TIMEOUT (20000)

/*******************************/
/* SXI MREF IND                */
/*******************************/
/** Message length */
#define SXI_MREF_IND_SIZE              (15)
/** Offset to the MrefAssociation field */
#define SXI_MREF_IND_MREF_ASSOC_OFFSET (4)

/** \name mRef Associations
 * @{
 */
#define SXI_MREF_IND_ASSOC_LIVE        (1)
#define SXI_MREF_IND_ASSOC_PLAYPOINT   (2)
#define SXI_MREF_IND_ASSOC_BOTH        \
    (SXI_MREF_IND_ASSOC_LIVE | SXI_MREF_IND_ASSOC_PLAYPOINT)
/** @} */

/******************************************************************************/
/*==========================> SXI CONFIRMATION DEFINITION  <==================*/
/******************************************************************************/

/***********************************/
/* SXI CONFIRMATION OPCODE         */
/***********************************/
#define SXI_MODULE_CFG_CFM              0x4020
#define SXI_PWR_MODE_CFM                0x4021
#define SXI_TIME_CFM                    0x4060
#define SXI_EVENT_CFM                   0x4080
#define SXI_STATUS_CFM                  0x40A0
#define SXI_DISP_ADVISORY_CFM           0x40C0
#define SXI_SUB_STATUS_CFM              0x40C1
#define SXI_IP_DEVICE_AUTH_CFM          0x40F0
#define SXI_AUTH_STATE_CFM              0x40F1
#define SXI_CHAN_BROWSE_CFM             0x4200
#define SXI_CAT_INFO_CFM                0x4201
#define SXI_CHAN_SELECT_CFM             0x4280
#define SXI_CHAN_INFO_CFM               0x4281
#define SXI_METADATA_CFM                0x4300
#define SXI_CHAN_METADATA_CFM           0x4301
#define SXI_GLOBAL_METADATA_CFM         0x4302
#define SXI_LA_METADATA_CFM             0x4303
#define SXI_SEEK_CFM                    0x4304
#define SXI_MREF_CFM                    0x4310
#define SXI_LAMREF_CFM                  0x4311
#define SXI_IR_PLAYBACK_INFO_CFM        0x4402
#define SXI_IR_PLAYBACK_METADATA_CFM    0x4403
#define SXI_IR_RECORD_INFO_CFM          0x4404
#define SXI_IR_RECORD_METADATA_CFM      0x4405
#define SXI_BULLETIN_STATUS_CFM         0x4420
#define SXI_FLASH_CFM                   0x4421
#define SXI_CONTENT_BUFF_CFM            0x4422
#define SXI_DATA_SERVICE_STATUS_CFM     0x4500
#define SXI_FWU_ERASE_CFM               0x4e84
#define SXI_FA_CFM                      0x4ec0
#define SXI_FA_PKG_CFM                  0x4ed0
#define SXI_MEM_MON_CFM                 0x4f01

/******************************************************************************/
/*==========================> SXI CODE DEFINITION  <==========================*/
/******************************************************************************/

/****************************/
/* SXI STATUS CODE          */
/****************************/
#define SC_MSG_RECEIVED     0x00
#define SC_MSG_COMPLETE     0x01
#define SC_MSG_FAIL         0x02
#define SC_MSG_UNKNOWN      0x03

/******************************************************************************/
/*==================> EXTENDED METADATA DEFINITON  <==========================*/
/******************************************************************************/
/* featured favorites - purpose field definition, see FEATURED FAVORITES in sxm_sxi.h */

/*****************************************/
/*    EXTENDED CHANNEL METADATA          */
/*****************************************/
#define NO_EXT_CHANNEL_METADATA          0

/*************************/
/* SXM PWR MODE          */
/*************************/
typedef struct {
    byte code;                                          /* SXI Indication Code for SXM Module Power Mode or Power Down State */
} SXMPwrMode;

/*************************/
/* SXM CHANNEL EXTENDED  */
/*************************/
typedef struct {
    //  extended channel metadata (channel descriptions, channel list order, play on select,...)
    byte valid;                                         /* Extended Channel Metadata vaild status --
                                                           Flags used to mark which fields are actually valid */
    char descs[SXI_CHANNEL_SHORT_DESCRIPTION_MAX];      /* Channel Short Description */
    char descl[SXI_CHANNEL_LONG_DESCRIPTION_MAX];       /* Channel Long Description */
    ushort ymal[SXI_RELATED_CHAN_LIST_MAX];             /* Related Channel List */
    ushort clo;                                         /* Channel List Order */
    byte pos;                                           /* Play On Select */
    byte cct;                                           /* Channel Content Type */
    byte irnc;                                          /* Playback Navigation Class */
    byte src;                                           /* Shadow Record Class */
}SXMChannelExt;

/*************************/
/* SXM Buffered Content  */
/*************************/
typedef struct {
    byte reserved;                                  /* reserved by the SXI message specification */
    ushort unplaytrack;                             /* number of available scan items not been marked as play */
    ushort unscantrack;                             /* number of available scan items not previously scan  */
} SXMScanItems;

/**********************/
/* SXI Channel Select */
/* Configuration      */
/**********************/
typedef struct {
    int FeatureControl;                             /* Feature Control - (deprecated) */
    int PlayPoint;                                  /* Playpoint - (where track begins when playing) */
    int PlaySeconds;                                /* PlaySeconds - (number of seconds to scan) */
    int ChannelScanInclude;                         /* ChannelScanInclude - (group of channel to include in scan) */
    int ChannelScanExclude;                         /* ChannelScanExclude - (group of channel to exclude from scan) */
}ChanSelectCfg;

/**********************/
/* SXI Smart Favorite */
/* List               */
/**********************/
typedef struct {
    int SidCount;                                   /* Smart Favorite SID count */
    ushort SidList[SXI_SP_MAX_SID_CNT];             /* Smart Favorite SID list */
}SmartFavList;

/*************************/
/* AUDIO TUNE            */
/*************************/
/* Audio tune channel */
typedef struct {
    byte code;          /* indication code from channel selection indication */
    ushort sid;         /* sid of the tune channel */
    SXMChannelBas cb;   /* channel basic metadata  - channel number, channel name,...,etc */
    SXMChannelExt ce;   /* channel extended metadata - (channel descriptions, channel list order, play on select,...) */
    SXMPad p;           /* normal PAD data like artist name, song title,...,etc  */
    ushort secid;       /* secondary ID related to tuner request for Bulletins and Flash events */
    byte valid;
}AudioTune;

/*************************/
/* Parsed ChanInfoInd    */
/*************************/
typedef struct {
    byte code;                                          /* Indication code */
    ushort id;                                          /* Channel Number (aka Identifier)*/
    ushort sid;                                         /* SID */
    byte attr;                                          /* Attributes of the channel and content */
    byte rr;                                            /* Recording restriction flags */
    byte catid;                                         /* Category Identifier (Nework Category Identifier) */
    char names[SXI_CHAN_NAME_SHORT_MAX];                /* Channel Name, Short format */
    char namem[SXI_CHAN_NAME_MED_MAX];                  /* Channel Name, Medium format */
    char namel[SXI_CHAN_NAME_LONG_MAX];                 /* Channel Name, Long format */
} ChanInfoInd;

/****************************/
/* Parsed MetadataInd       */
/****************************/
typedef struct {
    ushort id;
    ushort sid;
    SXMPad p;
} MetadataInd;

/****************************/
/* Parsed LAMetadataInd     */
/****************************/
typedef struct {
    ushort id;
    ushort sid;
    SXMLAPad l;
} LAMetadataInd;

/****************************/
/* Parsed IRPlaybackInfoInd */
/****************************/
typedef struct {
    byte code;
    SXMReplay r;
} IRPlaybackInfoInd;

/********************************/
/* Parsed IRPlaybackMetadataInd */
/********************************/
typedef struct {
    byte code;
    SXMTrackIdStatus is;
    SXMChannelBas cb;
    SXMChannelExt ce;
    SXMPad p;
} IRPlaybackMetadataInd;

/****************************/
/* Parsed IRRecordInfoInd   */
/****************************/
typedef struct {
    byte code;
    SXMReplayInfo ri;
} IRRecordInfoInd;

/******************************/
/* Parsed IRRecordMetadataInd */
/******************************/
typedef struct {
    byte code;
    SXMTrackIdStatus is;
    SXMTrackDur d;
    SXMChannelBas cb;
    SXMChannelExt ce;
    SXMPad p;
} IRRecordMetadataInd;

/******************************/
/* Parsed BulletinStatusInd   */
/******************************/
typedef struct {
    byte code;
    SXMBulletinStatus bs;
} BulletinStatusInd;

/******************************/
/* Parsed FlashInd            */
/******************************/
typedef struct {
    byte code;
    SXMFlashStatus fs;
} FlashInd;

/******************************/
/* Parsed AudioRequestInd     */
/******************************/
typedef struct {
    uint rbi;
} AudioRequestInd;

/******************************/
/* SXi Audio Tune Param       */
/******************************/
typedef struct {
    int sid;  //!< the sid to make initial tune once the SXi is started
    int overrides; //!< Overrides channel selection limits defined by channel
                   // attribute filters for the initial tune to the \v sid
    INT8 volume; //!< Desired volume level
    byte mute;   //!< Desired mute state
    INT8 bass; //!< Amplitude adjustment to be applied to low frequencies of the audio port.
    INT8 treble; //!< Amplitude adjustment to be applied to high frequencies of the audio port.
    INT8 equ[SXI_BAND_GAIN_SIZE]; //!< Equalizer
} SXiStartParam;

/*************************/
/* SXI GLOBAL API        */
/*************************/
extern uint sxm_sxi_debug_level;

extern int sxm_sxi_global_start(BOOL loadStorage);
extern int sxm_sxi_global_stop(void);
extern int sxm_sxi_tune_init(void);

extern int sxm_sxi_build_sxm_channel_no_lock(SXMChannel *, const SXMChannelBas *, const SXMChannelExt *);

extern int sxm_sxi_parse_scan_available_items(byte *, int, SXMScanItems *);
extern int sxm_sxi_parse_disp_advisory_ind(byte *, int, SXMDisplay *);
extern int sxm_sxi_parse_chan_select_ind(byte *, int, AudioTune *);
extern int sxm_sxi_parse_chan_info_ind(byte *, int, ChanInfoInd *);
extern int sxm_sxi_parse_metadata_ind(byte *, int, MetadataInd *);
extern int sxm_sxi_parse_la_metadata_ind(byte *, int, LAMetadataInd *);
extern int sxm_sxi_parse_ir_playback_info_ind(byte *, int, IRPlaybackInfoInd *);
extern int sxm_sxi_parse_ir_playback_metadata_ind(byte *, int, IRPlaybackMetadataInd *);
extern int sxm_sxi_parse_ir_record_info_ind(byte *, int, IRRecordInfoInd *);
extern int sxm_sxi_parse_ir_record_metadata_ind(byte *, int, IRRecordMetadataInd *);
extern int sxm_sxi_parse_bulletin_status_ind(byte *, int, BulletinStatusInd *);
extern int sxm_sxi_parse_flash_ind(byte *, int, FlashInd *);
extern int sxm_sxi_parse_audio_request_ind(byte *, int, AudioRequestInd *);
extern int sxm_sxi_parse_seek_ind(byte *, int, SXMSeek *);
extern int sxm_sxi_parse_content_buffer(byte *, int, SXMContentBuf *);

extern BOOL sxm_sxi_update_auth_state(byte *, int);
extern void sxm_sxi_update_category(byte *, int);
extern void sxm_sxi_update_channel_tune_from(const AudioTune *);
extern void sxm_sxi_update_channel_browse(byte *, int);
extern SXMSXiChannelAction sxm_sxi_update_channel_info_from(const ChanInfoInd *);
extern void sxm_sxi_update_channel_meta(byte *, int);
extern void sxm_sxi_update_debug(byte *, int);
extern void sxm_sxi_update_display_from(const SXMDisplay *);
extern void sxm_sxi_update_dmilist(byte *, int);
extern void sxm_sxi_update_event(byte *, int);
extern void sxm_sxi_update_fa(byte *, int);
extern void sxm_sxi_update_global(byte *, int);
extern BOOL sxm_sxi_update_ip_device_auth(byte *, int);
extern void sxm_sxi_update_memory(byte *, int);
extern void sxm_sxi_update_package(byte *, int);
extern void sxm_sxi_update_pad_from(const MetadataInd *);
extern void sxm_sxi_update_la_pad_from(const LAMetadataInd *);
extern void sxm_sxi_update_playback(byte *, int);
extern void sxm_sxi_update_radio(byte *, int);
extern void sxm_sxi_update_power_mode(byte *, int );
extern void sxm_sxi_update_record_from(const IRRecordInfoInd *);
extern void sxm_sxi_update_record_meta_from(const IRRecordMetadataInd *);
extern void sxm_sxi_update_record_track(byte *, int);
extern void sxm_sxi_update_replay_from(const IRPlaybackInfoInd *);
extern void sxm_sxi_update_replay_meta_from(const IRPlaybackMetadataInd *);
extern void sxm_sxi_update_bulletin_status_from(const BulletinStatusInd *);
extern void sxm_sxi_update_flash_status_from(const FlashInd *);
extern void sxm_sxi_update_mref(byte *, int);
extern void sxm_sxi_update_smon(byte *, int);
extern void sxm_sxi_update_scan_available_items_from(const SXMScanItems *);
extern void sxm_sxi_update_subscription(byte *, int);
extern void sxm_sxi_update_time(byte *, int);

/*************************/
/* SXI SERVICE           */
/*************************/
extern SXESDK_INTERNAL_API int sxm_sxi_start(void(*callback)(int, int), const SXiStartParam *tuneParam, int debugLevel);
extern SXESDK_INTERNAL_API int sxm_sxi_status(SXMStatus *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_stop(void);
extern SXESDK_INTERNAL_API int sxm_sxi_abort(void);
extern SXESDK_INTERNAL_API int sxm_sxi_set_dsi_callback(int dsi, void(*cb)(int, int, void*), void *arg);
extern SXESDK_INTERNAL_API int sxm_sxi_cfile_clean(void);
extern SXESDK_INTERNAL_API int sxm_sxi_set_debug(int debugLevel);
extern SXESDK_INTERNAL_API int sxm_sxi_get_debug(int *pDebugLevel);

/*************************/
/* SXI MESSAGES          */
/*************************/
extern SXESDK_INTERNAL_API int sxm_sxi_power(int PowerMode);
extern SXESDK_INTERNAL_API int sxm_sxi_chan_attrib_cfg(int AttributeChangeType, uint *sidv);
extern SXESDK_INTERNAL_API int sxm_sxi_chan_attrib_list(int AttributeChangeType, int count, const ushort *sids);
extern SXESDK_INTERNAL_API int sxm_sxi_time(int TimeZone, int DSTObs);
extern SXESDK_INTERNAL_API int sxm_sxi_chan_select_cfg(int FeatureControl, int PlayPoint, int PlaySeconds, int ChannelScanInclude, int ChannelScanExclude);
extern SXESDK_INTERNAL_API int sxm_sxi_feature_mon(int MonitorOperation, int FeatureListCnt, byte *FeatureMonitorID);
extern SXESDK_INTERNAL_API int sxm_sxi_chan_browse(int BrowseType, int ChanIDorSID, int CatID, int Overrides);
extern SXESDK_INTERNAL_API int sxm_sxi_usercat_cfg(int CatID , char *CategoryName, uint *SIDV);
extern SXESDK_INTERNAL_API int sxm_sxi_tune(int SelectionType, int ChanIDorSID, int CatID, int Overrides, int Routing);
extern SXESDK_INTERNAL_API int sxm_sxi_seekmon(int SeekMonID, int MonChangeType, int MonTMI, int MonValueCnt, int MonValueLen,
                                               ptr MonValue, int ReportTMICnt, ushort *ReportTMI, int SeekControl);
extern SXESDK_INTERNAL_API int sxm_sxi_extmon(int MonitorSelection, int MonitorChangeType, int ExtMetadataCnt, ushort *EMI);
extern SXESDK_INTERNAL_API int sxm_sxi_volume(INT8 VolumeAdj);
extern SXESDK_INTERNAL_API int sxm_sxi_mute(int Mute);
extern SXESDK_INTERNAL_API int sxm_sxi_tone(INT8 Bass, INT8 Treble);
extern SXESDK_INTERNAL_API int sxm_sxi_equalizer(INT8 *BandGain);
extern SXESDK_INTERNAL_API int sxm_sxi_ir_playback_control(int Control, int TimeOffset, int PlaybackID);
extern SXESDK_INTERNAL_API int sxm_sxi_bulletin_mon(int BulletinType, int BulletinParamCnt, ushort *BulletinParamlList);
extern SXESDK_INTERNAL_API int sxm_sxi_fwup_prog(void);
extern SXESDK_INTERNAL_API int sxm_sxi_fwup_configure(int BytesInPacket, int NumPackets);
extern SXESDK_INTERNAL_API int sxm_sxi_fwup_packet(ushort PacketID, byte *PacketData);
extern SXESDK_INTERNAL_API int sxm_sxi_monitor_dsi(int UpdateType, int DSI);
extern SXESDK_INTERNAL_API int sxm_sxi_filter_dsi(int UpdateType, int DSI, unsigned char DMICnt, ushort *DMI);
extern SXESDK_INTERNAL_API int sxm_sxi_playback_audio_packet(uint ReadByteIndex, byte AudioEncoderType, byte AudioBitRate,
        byte PlaybackStatus, ushort NumDataBytes, const byte *AudioData);
extern SXESDK_INTERNAL_API int sxm_sxi_get_chan_select_cfg(ChanSelectCfg *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_get_smart_favorite_list(SmartFavList *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_is_sid_in_smart_favorite_list(ushort sid);
extern SXESDK_INTERNAL_API int sxm_sxi_get_mute_status(int *mute_status_request);

/*************************/
/* SXI EXTRACTIONS       */
/*************************/
extern SXESDK_INTERNAL_API int sxm_sxi_extract_category(int id, SXMCategory *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_category_map(uint ret[SXM_CATEGORY_MAP_SIZE]);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_channel(int sid, SXMChannel *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_channel_id(int sid, SXMChannelId *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_channel_id_name(int sid, SXMChannelIdName *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_channel_map(uint ret[12], byte category);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_channel_content_type(int sid, byte *cct);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_channel_attributes(int sid, byte *attr);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_channel_category(int sid, byte *catId);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_display(SXMDisplay *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_dmis(int dsi, SXMDMIList *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_event(SXMEvent *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_power_mode(SXMPwrMode *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_la_pad(int sid, SXMLAPad *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_pad(int sid, SXMPad *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_pad_chn_scan(SXMPad *pPad, SXMChannelBas *pCb);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_package(SXMPackage *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_signal(SXMSignal *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_subscription(SXMSubscription *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_time(SXMTime *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_ir_replay_channel(SXMTrackIdStatus *id, SXMChannel *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_ir_replay_pad(SXMTrackIdStatus *id, SXMPad *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_ir_cur_play_info(SXMTrackIdStatus *id, SXMChannel *ch, SXMPad *pad);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_ir_cur_playback_id(ushort *id);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_ir_replay(SXMReplay *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_ir_buffer_channel(SXMTrackIdStatus *id, SXMChannel *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_ir_buffer_pad(SXMTrackIdStatus *id, SXMTrackDur *rd, SXMPad *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_ir_buffer_track_info(SXMTrackIdStatus *id, SXMChannel *rc, SXMTrackDur *rd, SXMPad *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_ir_buffer_info(SXMReplayInfo *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_er_record_replay(SXMReplay *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_er_record_channel(SXMChannel *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_er_record_pad(SXMPad *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_er_record(SXMRecording *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_tune_code(ushort *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_tune_sid(ushort *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_tune_channel(SXMChannel *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_tune_channel_num(ushort *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_tune_pad(SXMPad *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_tune_secondary_id(ushort *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_browse_code(ushort *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_browse_sid(ushort *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_browse_channel(SXMChannel *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_browse_pad(SXMPad *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_bulletin_status(SXMBulletinStatus *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_flash_status(SXMFlashStatus *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_extract_scan_track(SXMScanItems *ret);

/*************************/
/* SXI EXTENDED METADATA */
/*************************/
extern SXESDK_INTERNAL_API int sxm_sxi_global_league_count(uint *out);
extern SXESDK_INTERNAL_API int sxm_sxi_global_league(int s, SXMLeague *out);
extern SXESDK_INTERNAL_API int sxm_sxi_global_league_id_valid(int s);
extern SXESDK_INTERNAL_API int sxm_sxi_global_league_ix(int ix, SXMLeague *out);
extern SXESDK_INTERNAL_API int sxm_sxi_global_team_count(int *out);
extern SXESDK_INTERNAL_API int sxm_is_team_in_league(SXMTeam *t, int lid);
extern SXESDK_INTERNAL_API int sxm_is_team_in_league_and_tier_good_for_sf(SXMTeam *t, SXMLeague *l);
extern SXESDK_INTERNAL_API int sxm_sxi_global_team(int tn, int s, SXMTeam *out);
extern SXESDK_INTERNAL_API int sxm_sxi_global_team_ix(int ix, SXMTeam *out);
extern SXESDK_INTERNAL_API int sxm_sxi_global_traffic_count(int *out);
extern SXESDK_INTERNAL_API int sxm_sxi_global_traffic(int af, const char *s, SXMTraffic *out);
extern SXESDK_INTERNAL_API int sxm_sxi_global_traffic_ix(int ix, SXMTraffic *out);
extern SXESDK_INTERNAL_API int sxm_sxi_global_traffic_id(int id, SXMTraffic *out);
extern SXESDK_INTERNAL_API int sxm_sxi_global_traffic_is_valid_market_id(int id);
extern SXESDK_INTERNAL_API int sxm_sxi_global_chan_ext_metadata_version(int *out);

extern SXESDK_INTERNAL_API int sxm_sxi_global_league_collect_status(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_league_collect_start(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_league_collect_stop(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_team_collect_status(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_team_collect_start(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_team_collect_stop(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_preset_collect_status(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_preset_collect_start(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_preset_collect_stop(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_traffic_collect_status(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_traffic_collect_start(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_traffic_collect_stop(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_itune_url_collect_status(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_itune_url_collect_stop(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_chan_ext_metadata_collect_status(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_chan_ext_metadata_collect_start(int *ret);
extern SXESDK_INTERNAL_API int sxm_sxi_global_chan_ext_metadata_collect_stop(int *ret);

extern SXESDK_INTERNAL_API void sxm_sxi_global_begin_league(void);
extern SXESDK_INTERNAL_API void sxm_sxi_global_end_league(void);

/*************************/
/* SXI PID               */
/*************************/
extern SXESDK_INTERNAL_API int sxm_sxi_get_pid(ushort sid, SXMPid *out);
extern SXESDK_INTERNAL_API int sxm_sxi_make_pid(SXMPid *out);
extern SXESDK_INTERNAL_API int sxm_sxi_decode_pid(uint pid, SXMPid *out);

/*************************/
/* STATUS MONITOR        */
/*************************/
extern SXESDK_INTERNAL_API int sxm_sxi_update_smon_invalidate(int StatusListCnt, byte *StatusMonitorItem);

#ifdef __cplusplus
}
#endif

#endif /* !SXM_USE_GEN8 */

#endif /* SXI_INTERNAL_H */

