/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 * 
 * \file sxm_sports_request.c
 * \author Leslie French, Sergey Kotov
 * \date 8/20/2013
 *
 * Implementation of the requests management for Sports
 *
 ******************************************************************************/

#define DEBUG_TAG "sports"

#include "sxm_sports_internal.h"

/************************************************************************
 *                                                                      *
 *            Private prototypes                                        *
 *            ======================                                    *
 *                                                                      *
 ************************************************************************/

static const LeagueMapEntry *getLeagueMap(const char *pLeague);

/***************************************************************************//**
 *
 * This private function searches the League Map for a well-known league.
 *
 * \param[in] pLeague league name string pointer
 *
 * \return address of the LeagueMapEntry for the specified league or NULL
 *
 ******************************************************************************/
static const LeagueMapEntry* getLeagueMap(const char* pLeague)
{
    uint i;
    const LeagueMapEntry *pLeagueMap = NULL;

    /* look for a match */
    for(i=0; i<ARRAY_SIZE(SportsLeagueMap); i++)
    {
        if(strcmp(pLeague, SportsLeagueMap[i].league) == 0)
        {
            pLeagueMap = &SportsLeagueMap[i];
            break;
        }
    }

    return pLeagueMap;
}

/***************************************************************************//**
 *
 * This function is called to add a new request for a league of interest.
 * Standard requests operate at the league level.  Extended requests operate
 * at the root affiliation (AFID) level.
 *
 * \param[in] service structure pointer
 * \param[in] league league name
 * \param[in] notification notification callback function
 * \param[in] usercx opaque user context for callback
 * \param[in] handle pointer to handle pointer
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Service has not been started
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_INVAL Invalid parameter (out of range)
 * \retval SXM_E_BUSY Maximum requests active
 *
 * \note
 * Multilple requests (up to SXM_SPORTS_NUM_REQUESTS_MAX) are allowed to
 * be outstanding at any particular point in time.  This is in contrast
 * to other data services which allow only a single request at a time.
 *
 * \note
 * Requests made using the Standard Interface specify the string name
 * of a well know league, for example, "nfl".
 *
 * \par
 * Requests made using the Extended Interface specify the string
 * representation of the numeric AFID with a '#' character prefix. For
 * example, a request for Affilation ID 123 would take the form: "#123".
 *
 ******************************************************************************/
int sxm_sports_request_add(SportsService* service,
                           const char* league,
                           SPORTS_REQUEST_CALLBACK notification,
                           ptr usercx,
                           ptr* handle)
{
    uint i;
    int rc;
    SportsRequest* r;
    Affiliation* a = NULL;
    const LeagueMapEntry* e;

    ENTER_REQ("(%p) league=%s", handle, league);

    /* Must lock to keep bsc affiliate upheaval at bay. */
    LOCK(service->mutex);

    do
    {
        /* Find a free request. */
        for(i=0; i<ARRAY_SIZE(service->request); i++)
        {
            if(service->request[i].sportId == INVALID_ID)
            {
                break;
            }
        }

        /* check if any free request was found */
        if(i == ARRAY_SIZE(service->request))
        {
            DEBUG_REQ("(no free requests) max=%d", ARRAY_SIZE(service->request));
            rc = SXM_E_BUSY;
            break;
        }

        // the address of the request slot
        r = &service->request[i];

        // Extended API.  Parse afid user is interested in.
        if(league[0] == '#'){
            if(!SPORTS_HAVE_CONFIG() || !SPORTS_HAVE_AFTABLE())
            {
                DEBUG_REQ("(#format league but no config and/or aff table) league=%s", league);
                rc = SXM_E_INVAL;
                break;
            }

            rc = sscanf(league, "#%u", &i);
            if(rc != 1){
                DEBUG_REQ("(#format league failed to parse number) league=%s", league);
                rc = SXM_E_INVAL;
                break;
            }

            if(i>MAX_AFFILIATE_ID)
            {
                DEBUG_REQ("(#format league afid too big) league=%s", league);
                rc = SXM_E_INVAL;
                break;
            }

            if(ABLOCK[i] == INVALID_ID)
            {
                DEBUG_REQ("(#format league aff not present) league=%s", league);
                rc = SXM_E_INVAL;
                break;
            }

            // guard against out-of-range index
            if (ABLOCK[i] >= ARRAY_SIZE(AFF) )
            {
                DEBUG_REQ("invalid AFF index (i=%d,ABLOCK[i]=%d)", i, ABLOCK[i]);
                rc = SXM_E_INVAL;
                break;
            }

            // affiliation descriptor
            a = &AFF[ABLOCK[i]];

            // Only root affiliates can be the target of a request.
            if(a->level != 0)
            {
                DEBUG_REQ("(#format league not root level) league=%s", league);
                rc = SXM_E_INVAL;
                break;
            }

            LOCK(r->mutex);
            r->sportId = a->spid;
            strcpy(r->afname, AFF[ABLOCK[i]].name);
            // Keep the root affiliate id for usage later.
            r->afid = a->afid;
        }
        else
        {

            // standard API - check if this league is well known (in the map)
            e = getLeagueMap(league);
            if(!e)
            {
                DEBUG_REQ("(known format league not found) league=%s", league);
                rc = SXM_E_INVAL;
                break;
            }

            LOCK(r->mutex);
            r->sportId = e->sportId;
            strncpy(r->afname, e->rootAffiliate, sizeof(r->afname)-1);
            r->afname[sizeof(r->afname)-1] = '\0';
            /*
             * Set the root affiliate id invalid.  It may be available now but not guaranteed.
             * Extract begin will find an afid for a request if it is invalid.  This is so we
             * can create requests using the known API before the affiliates are present.
             *
             */
            r->afid = INVALID_ID;
        }

        r->bsc = FALSE;
        r->addNotify = TRUE;
        r->ext = NULL;
        r->callback = notification;
        r->usercx = usercx;

        // return the request handle
        *handle = (ptr)r;
        rc = SXM_E_OK;

        r->deleteAtEnd = FALSE;
        r->rdelete = FALSE;
        UNLOCK(r->mutex);
    }
    while(0);
    UNLOCK(service->mutex);

    LEAVE_REQ("(%p) rc=%d", handle, rc );
    return rc;
}

/***************************************************************************//**
 *
 * Function to remove an outstanding notification request.
 *
 * The routine will remove the notifications for that league.
 * Data are still being collected within the SDK and
 * will continue to be available to any subsequent notification request.
 *
 * \param[in] service pointer to the service structure
 * \param[in] handle validated request handle to be removed
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_BUSY Attempt to remove request currently extracting
 *
 ******************************************************************************/
int sxm_sports_request_remove(SportsService* service, ptr handle)
{
    int rc = SXM_E_NOENT;
    uint i;
    SportsRequest* r;

    ENTER_REQ("(%p)", handle);

    /*
     * TODO: Evaluate need for service level lock.  Current thought is it is not needed since
     * no data set related resources are ever accessed.
     */
    for(i=0; i<ARRAY_SIZE(service->request); i++)
    {
        r = &service->request[i];
        if(((ptr)r) != handle)
        {
            continue;
        }

        /* if active extraction, request can't be removed */
        if(r->ext)
        {
            DEBUG_REQ("(req has active extract: busy) req=%p ext=%p", handle, r->ext);
            rc = SXM_E_BUSY;
            break;
        }

        /* if request is not active, or if pending removal, return OK */
        if( (r->sportId == INVALID_ID) || (r->rdelete == TRUE) )
        {
            DEBUG_REQ("(req not active or pending: delete after callback) req=%p sportId=%d delete=%d", handle, r->sportId, r->rdelete);
            rc = SXM_E_OK;
            break;
        }

        /* mark it for deletion */
        r->rdelete = TRUE;

        /* check for inuse (active callback) and lli thread context */
        if( (sxm_lli_on_thread(service->lli)) && (r->inuse == TRUE) )
        {
            r->deleteAtEnd = TRUE;
            DEBUG_REQ("(req inuse and on lli thread: mark for delete when inuse ends) req=%p", handle);
            rc = SXM_E_OK;
            break;
        }

        /* now mark inactive by setting 'invalid' sport ID */
        LOCK(service->mutex);
        LOCK(r->mutex);
        r->sportId = INVALID_ID;
        UNLOCK(r->mutex);
        UNLOCK(service->mutex);
        rc = SXM_E_OK;
        break;
    }

    LEAVE_REQ("(%p) rc=%d", handle, rc);
    return rc;
}
