/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 *
 * \file sxm_sports_protocol.c
 * \author Leslie French, Sergey Kotov
 * \date 8/20/2013
 *
 * Implementation of the bitstream decoder for Sports
 *
 ******************************************************************************/

#include "sxm_build.h"

#define DEBUG_TAG "sports"

#if (SXM_USE_GEN8)
#include "sxm_gmd.h"
#endif

#include "sxm_sports_internal.h"

static SportsService *service = NULL;

#ifdef SXM_DEBUG_PKT_METRICS
SXESDK_API uint sports_packets = 0;
SXESDK_API uint sports_bytes = 0;
#endif /*#ifdef SXM_DEBUG_PKT_METRICS */

/************************************************************************
 *                                                                      *
 *            Private prototypes                                        *
 *            ======================                                    *
 *                                                                      *
 ************************************************************************/

static void init_cycle(void);

static void drop_affilates(void);

static void drop_all_affiliate_data(void);

static void drop_table_defintions(void);

static BOOL salvage_cycle(void);

static int drop_table_data_check_use(Table *t);

static void drop_table_data_ingore_use(int tabno, Table *t, int count);

static void drop_all_table_data_ignore_use(int tableid);

static int save_table_packet(Table *t, ushort pl, byte *b);

static void purge_old_data(uint epochNow);

static void notify_schedule(SportsRequest *r, Table *t,
    int count, ushort afid, uint epochNow);

static void notify_other(SportsRequest *r, Table *t, int count,
    ushort afid, uint startIndex);

static void notify_all_affiliates(SportsRequest *r, uint epochNow, BOOL emitOthers);

static void check_epoch(uint epochNow);

static void new_request_notify(uint epochNow);

static BOOL set_bsc(void);

static void clear_bsc(void);

static void notify_updating(BOOL bscStarting);

static void process_global_versions(void);

static void clear_au(SportsAUAssemble *assemble);

static uint assemble_data_au(SportsAUAssemble *assemble, byte *dataStart,
    uint aucnt, uint auaid, uint autot);

static uint assemble_au(SportsAUAssemble *assemble, byte *dataStart, uint aucnt,
    uint auauid, uint autot);

static uint assemble_pkt(SportsAUAssemble *assemble, byte *dataStart,
    uint aucnt, uint auaid, uint autot);

static uint assemble_data_pkt(SportsAUAssemble *assemble, byte *dataStart,
    uint aucnt, uint auaid, uint autot);

static void configCarousel(SXMBitBuff *pkt, SportsAUAssemble *assemble);

static int findParent(Affiliation *c, uint afcount);

static void affiliationCarousel(SXMBitBuff *pkt, SportsAUAssemble *assemble);

static void tablesCarousel(SXMBitBuff *pkt);

static void dataCarousel(SXMBitBuff *pkt, SportsAUAssemble *assemble,
    uint crc, uint epochNow);


/************************************************************************
 *									                                    *
 *            Cycle File Management					                    *
 *            =====================					                    *
 *									                                    *
 ************************************************************************/

/***************************************************************************//**
 * Secures all 'dirty' cycle file sections and commits root.
 *
 * \param[out] pStat I/O statistics if requested
 *
 * \retval SXM_E_OK Data has been written to file system successfully.
 * \retval SXM_E_PIPE Unable to commit C-File.
 *
 * \note runs in either service thread or application thread context 
 *
 ********************************************************************************/
int sxm_sports_commit(SXMIOStat *pStat)
{
    int rc = SXM_E_OK;
    int num, idx;
    const byte sections_update = service->sections_update;

    ENTER_DBF("pStat: %p", pStat);

    /* secure the section if the check dirty bit is set */
    for (idx = SPORTS_SECTION_FIRST; idx < SPORTS_SECTIONS_COUNT; ++idx)
    {
        if (ISBITSET(service->sections_update, idx))
        {
            num = sxm_cfile_secure(service->cycle, ROOT, idx, pStat);
            DEBUG_PKT("secure %d C-File section.", idx);
            if (num <= 0)
            {
                ERROR_PKT("unable to store %d C-File section.", idx);
                BITCLR(service->sections_update, idx);
            }
        }
    }

    /* commit if at least one section has been updated */
    if (service->sections_update & SPORTS_SECTIONS_ALL)
    {
        num = sxm_cfile_commit(service->cycle, ROOT, pStat);
        if (num <= 0)
        {
            rc = SXM_E_PIPE;
        }
    }

    /* error if one or more sections failed to secure */
    if (sections_update != service->sections_update)
    {
        rc = SXM_E_PIPE;
    }

    /* reset the dirty bits */
    service->sections_update ^= sections_update;

    LEAVE_DBF("%s", sxm_sdk_format_result(rc));
    return rc;
}

/**************************************************************************//**
 *
 * Private function to completely initialize the cycle file.
 *
 * This function is called during service start if the cycle file doesn't
 * exist or is corrupt and needs to be completely restarted.
 *
 * Also called to reinitialize the cycle file when a new service configuration
 * table is loaded from the broadcast.
 *
 * \return NONE
 *
 *****************************************************************************/
static void init_cycle(void)
{
    uint j;

    /*  clear the individual sections */
    sxm_cfile_clear_section(ROOT, SPORTS_SECTION_SPORTS);
    sxm_cfile_clear_section(ROOT, SPORTS_SECTION_AFFS);
    sxm_cfile_clear_section(ROOT, SPORTS_SECTION_TABLES);
    sxm_cfile_clear_section(ROOT, SPORTS_SECTION_HEAP);

    /* initialize the HEAP section (wipe the map) */
    sxm_cfile_map_init(ROOT, SXM_CFILE_SECTION_INVALID_ID, MAX_HEAP_BLOCK_COUNT);

    /* initialize the SPORTS (CONFIG) section */
    service->h->g.sections[SPORTS_SECTION_SPORTS].version = INVALID_VERSION;
    for (j = 0; j <= MAX_SPORT_ID; j++)
    {
        SBLOCK[j] = INVALID_ID;
    }

    /* initialize the TABLES section */
    for (j = 0; j <= MAX_TABLE_ID; j++)
    {
        TBLOCK[j] = INVALID_ID;
    }

    /* reset the AFFILIATION section */
    service->h->g.sections[SPORTS_SECTION_AFFS].version = INVALID_VERSION;
    for (j = 0; j <= MAX_AFFILIATE_ID; j++)
    {
        ABLOCK[j] = INVALID_ID;
    }

    for (j = 0; j < MAX_AFFILIATE_COUNT; j++)
    {
        AFF[j].ver = INVALID_VERSION;
        AFF[j].afid = INVALID_ID;
        AFF[j].parentid = INVALID_PARENT;
    }

    for (j = 0; j < MAX_TABLE_COUNT; j++)
    {
        TABLE[j].ver = INVALID_VERSION;
    }

    /* all sections are updated */
    service->sections_update = SPORTS_SECTIONS_ALL;

}

/**************************************************************************//**
 *
 * Private function to drop affiliates and all associated sports data.
 *
 * This function is called during service start if the affilliate section
 * of the cycle file is corrupt.
 *
 * Also called to reinitialize the affilation section of the cycle file when
 * a new service affiliation table is loaded from the broadcast.
 *
 * \return NONE
 *
 *****************************************************************************/
static void drop_affiliates(void)
{
    uint j;

    /* clear the AFFS section and mark it as invalid version */
    sxm_cfile_clear_section(ROOT, SPORTS_SECTION_AFFS);

    /* version info in kept in the 'root' block */
    service->h->g.sections[SPORTS_SECTION_AFFS].version = INVALID_VERSION;

    /* Drop all collected data by freeing all block in the heap map which
     * is maintained in the root block of the cycle file.
     */
    sxm_cfile_map_init(ROOT, SXM_CFILE_SECTION_INVALID_ID, MAX_HEAP_BLOCK_COUNT);

    /* mark all the map entries invalid */
    for (j = 0; j <= MAX_AFFILIATE_ID; j++)
    {
        ABLOCK[j] = INVALID_ID;
    }

    /* mark all the table entries invalid */
    for (j = 0; j < MAX_AFFILIATE_COUNT; j++)
    {
        AFF[j].ver = INVALID_VERSION;
        AFF[j].afid = INVALID_ID;
        AFF[j].parentid = INVALID_PARENT;
    }

    /* AFFILIATES section has updated */
    BITSET(service->sections_update, SPORTS_SECTION_AFFS);
}

/**************************************************************************//**
 *
 * Private function to drop all sports data.
 *
 * This function is called during service start while salvaging the cycle file.
 * 
 * \note
 * The Sports and Affs section were found to be good, but we are going to
 * delete all the associated sports data even though the HEAP section is not
 * necessarily bad. This is because the whole cycle file CRC has failed, and
 * we don't trust the data.
 *
 * \return NONE
 *
 *****************************************************************************/
static void drop_all_affiliate_data(void)
{
    int j;
    Affiliation *a;

    /* Wipe the map. */
    sxm_cfile_map_init(ROOT, SXM_CFILE_SECTION_INVALID_ID, MAX_HEAP_BLOCK_COUNT);

    /* Clear the tables.(set size to zero) */
    for (j = 0; j <= MAX_AFFILIATE_ID; j++)
    {
        if (ABLOCK[j] == INVALID_ID)
        {
            continue;
        }

        if (ABLOCK[j] >= ARRAY_SIZE(AFF))
        {
            DEBUG_PKT("invalid AFF index (j=%d,ABLOCK[j]=%d)", j, ABLOCK[j]);
            continue;
        }

        a = &AFF[ABLOCK[j]];
        memset(a->others, 0, sizeof(a->others));
        memset(a->schedule, 0, sizeof(a->schedule));
    }

    /* AFFILIATES section has updated */
    BITSET(service->sections_update, SPORTS_SECTION_AFFS);
}

/**************************************************************************//**
 *
 * Private function to drop the table defintions.
 *
 * This function is called during service start to delete the table defintions.
 *
 * \return NONE
 *
 *****************************************************************************/
static void drop_table_definitions(void)
{
    uint j;

    /* clear the table defs sections */
    sxm_cfile_clear_section(ROOT, SPORTS_SECTION_TABLES);

    /* invalidate the entire map */
    for (j = 0; j <= MAX_TABLE_ID; j++)
    {
        TBLOCK[j] = INVALID_ID;
    }

    /* mark all the tables invalid */
    for (j = 0; j < ARRAY_SIZE(TABLE); j++)
    {
        TABLE[j].ver = INVALID_VERSION;
    }

    /* TABLES section has updated */
    BITSET(service->sections_update, SPORTS_SECTION_TABLES);
}

/**************************************************************************//**
 *
 * Private function to salvage as much as possible from the cycle file.
 *
 * \return 0 if cycle file could not be salvaged, 1 otherwise
 *
 *****************************************************************************/
static BOOL salvage_cycle(void)
{
    int sok, aok, tok;

    sok = sxm_cfile_check(ROOT, SPORTS_SECTION_SPORTS, FALSE);
    aok = sxm_cfile_check(ROOT, SPORTS_SECTION_AFFS, FALSE);
    tok = sxm_cfile_check(ROOT, SPORTS_SECTION_TABLES, FALSE);

    /* The sport section must be ok or we cannot use any of the data. */
    if (!sok)
    {
        ERROR_PKT("Sport section CRC failure");
        return FALSE;
    }

    /*
     * No matter what we drop all data.  The whole file crc failed and it cannot be trusted.
     */

    if (!aok)
    {
        /* Affiliate section corrupt: drop all affiliates and data.(does not examine file data) */
        drop_affiliates();
        ERROR_PKT("Affiliate section CRC failure");
    }
    else
    {
        /* Affiliate section ok.  Drop all data. (needs valid affiliate data). */
        drop_all_affiliate_data();
    }

    /* The table definitions are corrupt: drop all table definitions.(data was already dumped) */
    if (!tok)
    {
        drop_table_definitions();
        ERROR_PKT("Table def section crc failure");
    }

    return TRUE;
}

/**************************************************************************//**
 *
 * Private function to drop the table data if not currently in use.
 *
 * \param[in] t table defintion pointer for this data
 *
 * \return 1 if table dropped, 0 if table in use
 *
 *****************************************************************************/
static SXE_INLINE int drop_table_data_check_use(Table* t) {

    /* check in use flag */
    if (!t->inuse)
    {

        /* release the blocks */
        if (t->heapix != INVALID_HEAPIX)
        {
            sxm_cfile_map_free(ROOT, SXM_CFILE_SECTION_INVALID_ID, t->heapix,
                ((t->size + SXM_ARCH_FILE_SECTOR_SIZE - 1 ) >> SXM_ARCH_FILE_SECTOR_DIVISOR));
        }

        /* invalidate this entry */
        t->tabno = INVALID_ID;
        t->heapix = INVALID_HEAPIX;
        t->size = 0;

        /* mark the CFILE section dirty bit */
        BITSET(service->sections_update, SPORTS_SECTION_AFFS);

        return 1;
    }

    return 0;
}

/**************************************************************************//**
 *
 * Private function to drop the table data for a number of common tables
 * regardless if currently in use.
 *
 * \param[in] tabno table number
 * \param[in] t table defintion pointer for this data
 * \param[in] count number of common tables to drop
 *
 * \return NONE
 *
 *****************************************************************************/
static void drop_table_data_ignore_use(int tabno, Table* t, int count)
{
    int i;

    /* for the following common tables... */
    for (i = 0; i < count; i++, t++)
    {
        if (t->tabno == tabno)
        {

            /* ..release the blocks... */
            if (t->heapix != INVALID_HEAPIX)
            {
                sxm_cfile_map_free(ROOT, SXM_CFILE_SECTION_INVALID_ID, t->heapix,
                    ((t->size + SXM_ARCH_FILE_SECTOR_SIZE - 1 ) >> SXM_ARCH_FILE_SECTOR_DIVISOR));
            }

            /* ...invalidate the table */
            t->tabno = INVALID_ID;
            t->heapix = INVALID_HEAPIX;
            t->size = 0;

            /* mark the CFILE section dirty bit */
            BITSET(service->sections_update, SPORTS_SECTION_AFFS);
        }
    }
}

/**************************************************************************//**
 *
 * Private function to drop all table data ('schedule' and 'others' for this
 * table ID for all affiliations regardless of current usage.
 *
 * \param[in] tableid  table ID
 *
 * \return NONE
 *
 *****************************************************************************/
static void drop_all_table_data_ignore_use(int tableid) {
    int j;

    /* just skip any invalid entries in the affiliation map */
    for (j = 0; j <= MAX_AFFILIATE_ID; j++)
    {
        if (ABLOCK[j] == INVALID_ID)
        {
            continue;
        }

        /* should not happen, but make sure map is in range */
        if (ABLOCK[j] >= ARRAY_SIZE(AFF))
        {
            DEBUG_PKT("invalid index (j=%d,ABLOCK[j]=%d)", j, ABLOCK[j]);
            continue;
        }

        /* drop 'others' */
        drop_table_data_ignore_use(tableid, AFF[ABLOCK[j]].others,
                ARRAY_SIZE(AFF[ABLOCK[j]].others));

        /* drop 'schedules' */
        drop_table_data_ignore_use(tableid, AFF[ABLOCK[j]].schedule,
                ARRAY_SIZE(AFF[ABLOCK[j]].schedule));
    }
}

/**************************************************************************//**
 *
 * Private function to save the table data to the heap.
 *
 * This function is called by the dataCarousel processing function.
 *
 * \param[in] t table defintion pointer for this data
 * \param[in] pl data length in bytes
 * \param[in] b pointer to the packet data
 *
 * \return 1 if data was saved, zero otherwise. 
 * 
 * \note heap memory is allocated if needed.
 *
 *****************************************************************************/
static int save_table_packet(Table *t, ushort pl, byte *b) {
    byte *dest;
    int heapix;
    const int nb = (pl + SXM_ARCH_FILE_SECTOR_SIZE - 1) >> SXM_ARCH_FILE_SECTOR_DIVISOR;

    /* allocate the memory for this packet if needed */
    if (t->heapix == INVALID_HEAPIX)
    {
        heapix = sxm_cfile_map_alloc(ROOT, SXM_CFILE_SECTION_INVALID_ID, nb);

        /* no memory of required size available from heap */
        if (heapix == -1)
        {
            t->size = 0;
            BITSET(service->sections_update, SPORTS_SECTION_AFFS);
            ERROR_PKT("Out of memory: blocks=%d", nb);
            return 0;
        }

        /* remember the heap index */
        t->heapix = (ushort)heapix;
    }

    /* now save the data in the heap */
    dest = (byte *) &HEAP[t->heapix];
    t->size = pl;
    memcpy(dest, b, pl);

    /* both HEAP and AFFS sections updated */
    BITSET(service->sections_update, SPORTS_SECTION_HEAP);
    BITSET(service->sections_update, SPORTS_SECTION_AFFS);

    return 1;
}

/**************************************************************************//**
 *
 * Private function called at service start to delete stale data.
 *
 * \param[in] epochNow the current epoch
 *
 * \return NONE 
 * 
 * \note this function is called in application thread context
 *
 *****************************************************************************/
static void purge_old_data(uint epochNow)
{
    int epochDelta;
    uint j, i, saveEpoch;
    Affiliation* a;

    LOCK(service->mutex);

    /*
     * We need to drop any table dated between our shutdown and start up.
     * We cannot interpret the data.  Avoid odd things like a past day of events
     * that is current day data with games in progress, etc.
     */
     saveEpoch = SPORTS_EPOCH_MAKE(ROOT->ts);
     /*
     * Save epoch in the future should not happen, but easy and minor to check.
     *
     */
    if(saveEpoch > epochNow)
    {
        saveEpoch = epochNow;
    }

    DEBUG_PKT(" epochNow=%u saveEpoch=%u", epochNow, saveEpoch);

    for (j = 0; j <= MAX_AFFILIATE_ID; j++)
    {
        if (ABLOCK[j] == INVALID_ID)
        {
            continue;
        }

        if (ABLOCK[j] >= ARRAY_SIZE(AFF))
        {
            DEBUG_PKT("invalid AFF index (j=%d,ABLOCK[j]=%d)", j, ABLOCK[j]);
            continue;
        }

        a = &AFF[ABLOCK[j]];

        /* Drop schedule data for today and any other day outside our window. */
        for (i = 0; i < ARRAY_SIZE(a->schedule); i++)
        {
            if (a->schedule[i].size)
            {
                /* clear any in use this is a start up: cleanse of the file. */
                if (a->schedule[i].inuse)
                {
                    ERROR_PKT(
                            "Sched table in use: inuse=%d idx=%u tabno=%d afid=%d",
                            a->schedule[i].inuse, i, a->schedule[i].tabno,
                            a->afid);
                }
                a->schedule[i].inuse = 0;
                epochDelta = (int)(a->schedule[i].epoch - (int)epochNow);
                if( ((a->schedule[i].epoch >= saveEpoch) && (a->schedule[i].epoch <= epochNow)) ||
                    (epochDelta < -8) ||
                    (epochDelta >  8) )
                {
                        DEBUG_PKT("(DROP) afid=%d afver=%d name=%s tabno=%d epoch=%d",
                            a->afid, a->ver, a->name, a->schedule[i].tabno, a->schedule[i].epoch);
                        drop_table_data_ignore_use(a->schedule[i].tabno,
                            &a->schedule[i], 1);
                }
            }
        }

        /* Drop all other data besides today. */
        for (i = 0; i < ARRAY_SIZE(a->others); i++)
        {
            /* clear any in use this is a start up: cleanse of the file. */
            if (a->others[i].inuse)
            {
                ERROR_PKT(
                        "Others table in use: inuse=%d idx=%u tabno=%d afid=%d",
                        a->others[i].inuse, i, a->others[i].tabno, a->afid);
            }
            a->others[i].inuse = 0;
            if (a->others[i].size && (a->others[i].epoch != epochNow))
            {
                DEBUG_PKT("(DROP) afid=%d afver=%d name=%s tabno=%d epoch=%d",
                        a->afid, a->ver, a->name, a->others[i].tabno, a->others[i].epoch);
                drop_table_data_ignore_use(a->others[i].tabno, &a->others[i], 1);
            }
        }
    }

    UNLOCK(service->mutex);
}

/**************************************************************************//**
 *
 * Private function is called to announce the arrival of new schedule data.
 *
 * \param[in] r pointer to an outstanding request
 * \param[in] t pointer to table (schedule) data
 * \param[in] count number of notifications to issue
 * \param[in] afid affiliation
 * \param[in] epochNow current epoch
 *
 * \return NONE 
 * 
 * \note runs in service thread context with service mutex held
 *
 *****************************************************************************/
static void notify_schedule(SportsRequest* r, Table* t, int count, ushort afid,
        uint epochNow)
{
    int i, param;
    int index;

    /* notify for each... */
    for (i = 0; i < count; i++)
    {
        if (t[i].size)
        {
            index = (int)(t[i].epoch - (int)epochNow + 8);
            if ((index >= 0) && (index <= MAX_SCHEDULE_INDEX))
            {
                param = (int)(SPORTS_TABLE_PARAM(afid, index));

                /* ... if has not been removed */
                if (r->rdelete == FALSE)
                {
                    r->inuse = TRUE;
                    UNLOCK(service->mutex);
                    r->callback((ptr) r, SXM_SPORTS_TABLE, param, r->usercx);
                    LOCK(service->mutex);
                    r->inuse = FALSE;
                }
                else
                {
                    break;
                }
            }
        }
    }
}

/**************************************************************************//**
 *
 * Private function is called to announce the arrival sports data other than
 * schedule data.
 *
 * \param[in] r pointer to an outstanding request
 * \param[in] t pointer to table (schedule) data
 * \param[in] count number of notifications to issue
 * \param[in] afid affiliation
 * \param[in] startIndex starting table index
 *
 * \return NONE 
 * 
 * \note runs in service thread context with service mutex held
 *
 *****************************************************************************/
static void notify_other(SportsRequest* r, Table* t, int count, ushort afid,
        uint startIndex)
{
    uint i, param;

	for (i = 0; i < (uint)count; i++)
    {
        if (t[i].size)
        {
            param = (int)(SPORTS_TABLE_PARAM(afid, startIndex + i));
            if (r->rdelete == FALSE)
            {
                r->inuse = TRUE;
                UNLOCK(service->mutex);
                r->callback((ptr) r, SXM_SPORTS_TABLE, param, r->usercx);
                LOCK(service->mutex);
                r->inuse = FALSE;
            }
            else
            {
                break;
            }
        }
    }
}

static void notify_all_affiliates(SportsRequest* r, uint epochNow,
        BOOL emitOthers)
{
    uint i;
    Affiliation* a;
    Affiliation* b;

    a = sxm_sports_shared_find_root(r);
    if (!a)
    {
        return;
    }

    for (i = 0; i <= MAX_AFFILIATE_ID; i++)
    {
        if (ABLOCK[i] == INVALID_ID)
        {
            continue;
        }

        if (ABLOCK[i] >= ARRAY_SIZE(AFF))
        {
            ERROR_PKT("invalid AFF index (i=%d,ABLOCK[i]=%d)", i, ABLOCK[i]);
            continue;
        }

        b = &AFF[ABLOCK[i]];
        /* rootid of a root is itself.  This emits the tables of root affiliates also. */
        if (b->rootid != a->rootid) {
            continue;
        }

        if (emitOthers == TRUE)
        {
            notify_other(r, b->others, ARRAY_SIZE(b->others), b->afid, MAX_SCHEDULE_INDEX+1);
        }
        notify_schedule(r, b->schedule, ARRAY_SIZE(b->schedule), b->afid, epochNow);

        if (r->rdelete == TRUE)
        {
            break;
        }
    }
}

static void check_epoch(uint epochNow)
{
    uint i;
    SportsRequest* r;

    if (epochNow == service->lastEpoch)
    {
        return;
    }

    LOCK(service->mutex);
    service->lastEpoch = epochNow;

    for (i = 0; i < ARRAY_SIZE(service->request); i++)
    {
        r = &service->request[i];
        LOCK(r->mutex);
        if ((r->sportId == INVALID_ID)
            || (r->rdelete == TRUE) || (r->addNotify == TRUE) || (r->bsc == TRUE) )
        {
            UNLOCK(r->mutex);
            continue;
        }

        notify_all_affiliates(r, epochNow, FALSE);

        if (r->deleteAtEnd == TRUE)
        {
            r->sportId = INVALID_ID;
        }
        UNLOCK(r->mutex);
    }

    UNLOCK(service->mutex);
}

/**************************************************************************//**
 *
 * Private function to notify any new requests.
 *
 * \return NONE
 *
 *****************************************************************************/
static void new_request_notify(uint epochNow)
{
    uint i;
    SportsRequest* r;

    LOCK(service->mutex);

    /* if BSC is active, nothing to do */
    if (service->bsc == TRUE)
    {
        UNLOCK(service->mutex);
        return;
    }

    /* look for active new requests (addNotify == TRUE)... */
    for (i = 0; i < ARRAY_SIZE(service->request); i++)
    {
        r = &service->request[i];
        LOCK(r->mutex);
        if ((r->sportId == INVALID_ID) || (r->rdelete == TRUE) 
            || (r->bsc == TRUE) || (r->addNotify == FALSE) )
        {
            UNLOCK(r->mutex);
            continue;
        }

        /* ... and notify */
        r->inuse = TRUE;   /* set callback active state */
        UNLOCK(service->mutex);
        r->callback((ptr) r, SXM_SPORTS_UPDATED, 0, r->usercx);
        LOCK(service->mutex);
        r->inuse = FALSE;   /* reset callback active state */

        /* notify all affiliates */
        notify_all_affiliates(r, epochNow, TRUE);

        /* removing request after callback */
        if (r->deleteAtEnd == TRUE)
        {
            r->sportId = INVALID_ID;
        }

        /* reset 'new request' flag */
        r->addNotify = FALSE;

        /* unlock the request */
        UNLOCK(r->mutex);
    }

    UNLOCK(service->mutex);
}

/**************************************************************************//**
 *
 * Private function to set global BSC (Broad Scope Change) event flag and
 * to mark BSC event in all outstanding collection requests.
 *
 * \return TRUE : BSC was triggered
 * \return FALSE : BSC was already triggered
 *
 *****************************************************************************/
static BOOL set_bsc(void)
{
    uint j;

    /* if global BSC state is not active, set it active and then ... */
    if (service->bsc == FALSE)
    {
        /* set global BSC state */
        service->bsc = TRUE;

        /* ... synchronously mark BSC active in all outstanding requests */
        for (j = 0; j < ARRAY_SIZE(service->request); j++)
        {
            LOCK(service->request[j].mutex);
            if (service->request[j].sportId != INVALID_ID)
            {
                service->request[j].bsc = TRUE;
            }
            UNLOCK(service->request[j].mutex);
        }

        /* BSC was triggered */
        return TRUE;
    }

    /* BSC was already active, no need to do anything */
    return FALSE;
}

/**************************************************************************//**
 *
 * Private function to clear global BSC (Broad Scope Change) start event and
 * notify application of BSC completion via the service-level callback.
 *
 * \return NONE
 *
 *****************************************************************************/
static void clear_bsc(void)
{

#if (SXM_USE_GEN8)
#if (SXM_USE_GMD)
    int ver;

    /* If BSC not active, nothing to do */
    if (service->bsc == FALSE)
    {
        return;
    }

    sxm_gmd_table_begin();

    /* Need sports, affiliation, and global league and teams */
    if (SPORTS_HAVE_CONFIG()
        && SPORTS_HAVE_AFTABLE()
        && (sxm_gmd_league_version_extract(&ver) == SXM_E_OK)
        && (sxm_gmd_team_version_extract(&ver) == SXM_E_OK)){

        /* reset the global BSC state */
        service->bsc = FALSE;

        /* notify application that BSC is completed */
        if (service->ds.status->service == SXM_SERVICE_OK)
        {
            service->callback(SXM_CALLBACK_SPORTS_UPDATED, 0);
        }
    }

    sxm_gmd_table_end();
#endif
#else

    int ver;

    /* If BSC not active, nothing to do */
    if (service->bsc == FALSE)
    {
        return;
    }

    sxm_sxi_global_table_begin();

    /* Need sports, affiliation, and global league and teams */
    if (SPORTS_HAVE_CONFIG()
        && SPORTS_HAVE_AFTABLE()
        && (sxm_sxi_global_league_version(&ver) == SXM_E_OK)
        && (sxm_sxi_global_team_version(&ver) == SXM_E_OK) )
    {

        /* reset the global BSC state */
        service->bsc = FALSE;

        /* notify application that BSC is completed */
        if (service->ds.status->service == SXM_SERVICE_OK)
        {
            service->callback(SXM_CALLBACK_SPORTS_UPDATED, 0);
        }
    }
    sxm_sxi_global_table_end();

#endif
}

/**************************************************************************//**
 *
 * Private function to conditionally notify (service-level notification)
 * that BSC (Broad Scope Change) is in progress
 *
 * \param[in] bscStarting TRUE if bsc is in progress, FALSE otherwise
 *
 * \return NONE
 *
 *****************************************************************************/
static void notify_updating(BOOL bscStarting)
{

    if ((service->ds.status->service == SXM_SERVICE_OK) && (bscStarting == TRUE))
    {
        service->callback(SXM_CALLBACK_SPORTS_UPDATING, 0);
    }
}

/**************************************************************************//**
 *
 * Private function invoked periodically to check for new versions of the
 * global team or/or league  tables.  If there are new versions available,
 * then the version tags are updated, and a BSC event is announced.
 *
 * \return NONE
 *
 *****************************************************************************/
static void process_global_versions(void)
{

    /* retrieve the global league and team table versions */
#if (SXM_USE_GEN8)
#if (SXM_USE_GMD)
    int lv, tv, rc;
    BOOL bsc = FALSE;

   sxm_gmd_table_begin();
    rc = sxm_gmd_league_version_extract(&lv);

    if (rc == SXM_E_OK)
    {
        rc = sxm_gmd_team_version_extract(&tv);
    }
    sxm_gmd_table_end();

    /* we need both league and team tables */
    if (rc != SXM_E_OK)
        return;

    DEBUG_PKT("s->lv=%d s->tv=%d g->lv=%d g->tv=%d",
                service->globalLeagueVer, service->globalTeamVer, lv, tv);

    /* Get start up out of the way. */
    if(service->globalTeamVer == GLOBAL_VERSION_STARTUP)
    {
        service->globalLeagueVer = lv;
        service->globalTeamVer = tv;
        return;
    }

    /* new Global League table version triggers a BSC */
    if(service->globalLeagueVer != lv)
    {
        service->globalLeagueVer = lv;
        bsc = TRUE;
    }

    /* new Global Team table version triggers a BSC */
    if(service->globalTeamVer != tv)
    {
        service->globalTeamVer = tv;
        bsc = TRUE;
    }

    /* now check BSC trigger and notify application of BSC event */
    if(bsc == TRUE)
    {
        set_bsc();
        notify_updating(bsc);
    }
#else 
    /* NO GMD service */
    return;
#endif
#else
   int lv, tv, rc;
    BOOL bsc = FALSE;

    /* retrieve the global league and team table versions */
    sxm_sxi_global_table_begin();
    rc = sxm_sxi_global_league_version(&lv);

    if (rc == SXM_E_OK)
    {
        rc = sxm_sxi_global_team_version(&tv);
    }
    sxm_sxi_global_table_end();

    /* we need both league and team tables */
    if (rc != SXM_E_OK)
        return;

    DEBUG_PKT("s->lv=%d s->tv=%d g->lv=%d g->tv=%d",
                service->globalLeagueVer, service->globalTeamVer, lv, tv);

    /* Get start up out of the way. */
    if(service->globalTeamVer == GLOBAL_VERSION_STARTUP)
    {
        service->globalLeagueVer = lv;
        service->globalTeamVer = tv;
        return;
    }

    /* new Global League table version triggers a BSC */
    if(service->globalLeagueVer != lv)
    {
        service->globalLeagueVer = lv;
        bsc = TRUE;
    }

    /* new Global Team table version triggers a BSC */
    if(service->globalTeamVer != tv)
    {
        service->globalTeamVer = tv;
        bsc = TRUE;
    }

    /* now check BSC trigger and notify application of BSC event */
    if(bsc == TRUE)
    {
        set_bsc();
        notify_updating(bsc);
    }
#endif
}

/**************************************************************************//**
 *
 * Private function invoked to reset the assembly structure/buffers.
 *
 * \return NONE
 *
 *****************************************************************************/
static SXE_INLINE void clear_au(SportsAUAssemble* assemble)
{
    assemble->autot = 0;
}

/**************************************************************************//**
 * The routine sequentially concatenates independent Access Units with a
 * common AU group ID into a single AU for post processing.
 *
 * \param[in] assemble a pointer to the AU assembly buffer
 * \param[in] dataStart byte offset to AU payload data
 * \param[in] aucnt AU sequence number
 * \param[in] auaid AU group ID
 * \param[in] autot total number of AUs in the group
 *
 * \return integer return code
 * \retval 0 = assembly incomplete, or just starting
 * \retval 1 = assembly complete
 * \retval 2 = assembly sequence broken, try again (restart)
 *
 * \note
 * This function runs in service thread context
 *
 *****************************************************************************/
static uint assemble_data_au(SportsAUAssemble* assemble, byte* dataStart,
    uint aucnt, uint auaid, uint autot)
{
    int size;
    int offset;

    /* Data packets concat aus' and parse as one unit. */
    offset = (int)(dataStart - assemble->pktbuf);

    /* No active multi au */
    if(assemble->autot == 0)
    {

        /* Only start on the first entry. */
        if(aucnt != 0)
        {
            return 0;
        }

        /* guard against AU count overflow */
        if(autot > MAX_AU_COUNT)
        {
            return 0;
        }

        /* First AU */
        assemble->autot = (ushort)autot;
        assemble->auaid = (ushort)auaid;
        assemble->startOff = (ushort)offset;
        assemble->auend = assemble->aubuf + assemble->pktlen - 4;
        memcpy(assemble->aubuf, assemble->pktbuf, (size_t)(assemble->pktlen - 4));

        assemble->trefmap[0].offset = assemble->pktlen;
        assemble->trefmap[0].delta = 0;
    }

    /* Active au */
    else
    {

        /* if out of sequence, start over */
        if(aucnt != (assemble->aucur + 1))
        {
            clear_au(assemble);
            return 2;
        }

        /* must be in same group, otherwise start over  */
        if(auaid != assemble->auaid)
        {
            clear_au(assemble);
            return 2;
        }

        /* total AU counts must match */
        if(autot != assemble->autot)
        {
            clear_au(assemble);
            return 2;
        }

        size = assemble->pktlen - offset - 4;

        /* Range check to end of buffer before copying it in */
        if( (assemble->auend + size) > (assemble->aubuf + MAX_CAROUSEL_DATA_SIZE))
        {
            DEBUG_PKT("(max data au size exceeded) max=%d received=%d aucnt=%d autot=%d ausize=%d",
                    MAX_CAROUSEL_DATA_SIZE,
                    (assemble->auend + size) - assemble->aubuf,
                    aucnt,
                    autot,
                    size);

            clear_au(assemble);
            return 0;
        }

        /* append (concatenate) this AU to the previouos one */
        memcpy(assemble->auend, assemble->pktbuf + offset, (size_t)size);
        assemble->auend += size;

        assemble->trefmap[aucnt].offset = (ushort)(assemble->trefmap[aucnt-1].offset + assemble->pktlen);
        assemble->trefmap[aucnt].delta = (ushort)(assemble->trefmap[aucnt-1].delta + offset + 4);
    }

    //sxm_file_log("(trefmap) aucnt=%d offset=%d delta=%d", aucnt, assemble->trefmap[aucnt].offset, assemble->trefmap[aucnt].delta);

    /* returns '1' if we've finished all of them in this group */
    assemble->aucur = (ushort)aucnt;
    return ((aucnt + 1) == autot) ? 1 : 0;
}

/**************************************************************************//**
 * The routine sequentially assembles independent Access Units with a
 * common AU group ID into a single assembly buffer for post processing.
 *
 * \param[in] assemble a pointer to the AU assembly buffer
 * \param[in] dataStart byte offset to AU payload data
 * \param[in] aucnt AU sequence number
 * \param[in] auaid AU group ID
 * \param[in] autot total number of AUs in the group
 *
 * \return integer return code
 * \retval 0 = assembly incomplete, or just starting
 * \retval 1 = assembly complete
 * \retval 2 = assembly sequence broken, try again (restart)
 *
 * \note
 * This function runs in service thread context
 *
 *****************************************************************************/
static uint assemble_au(SportsAUAssemble* assemble, byte* dataStart, uint aucnt,
    uint auaid, uint autot)
{

    /*
     * Config and affiliate do not concat payloads.  They parse each au independently
     * and accumulate the data.
     *
     */
    if(autot == 0)
    {
        ERROR_PKT("Invalid autot 0");
        return 0;
    }

    /*
     * Not multi-au, set up the assemble structure to use the input packet so the processing loop
     * does not care where the data is.
     */
    if(autot == 1)
    {

        assemble->auoff[0] = (ushort)(dataStart - assemble->pktbuf);
        assemble->ausize[0] = (ushort)(assemble->pktlen - 4);
        assemble->au[0] = assemble->pktbuf;
        assemble->autot = (ushort)autot;
        return 1;
    }

    /* No active multi au */
    if(assemble->autot == 0)
    {

        // Only start on the first entry.
        if(aucnt != 0)
        {
            return 0;
        }

        /* limit check */
        if(autot > MAX_AU_COUNT)
        {
            ERROR_PKT("(max au count exceeded) max=%d received=%d", MAX_AU_COUNT, autot);
            return 0;
        }

        /* the first AU of a Multi-AU assembly */
        assemble->autot = (ushort)autot;
        assemble->auaid = (ushort)auaid;
        assemble->auend = assemble->aubuf;
    }

    /* Active au */
    else
    {

        /* if sequence is broken, start over */
        if(aucnt != (assemble->aucur + 1))
        {
            clear_au(assemble);
            return 2;
        }

        /* if foreign AU group ID, start over */
        if(auaid != assemble->auaid)
        {
            clear_au(assemble);
            return 2;
        }

        /* if different number of AUs in the group, start over */
        if(autot != assemble->autot)
        {
            clear_au(assemble);
            return 2;
        }
    }

    /* add this AU to the assembly buffer */
    assemble->auoff[aucnt] = (ushort)(dataStart - assemble->pktbuf);
    assemble->ausize[aucnt] = (ushort)(assemble->pktlen - 4);
    assemble->au[aucnt] = assemble->auend;

    /* Range check to end of buffer before we copy it in. */
    if((assemble->au[aucnt] + assemble->ausize[aucnt]) > (assemble->aubuf + MAX_CAROUSEL_DATA_SIZE))
    {
        ERROR_PKT("(max config/aff au data size exceeded) max=%d received=%d aucnt=%d autot=%d ausize=%d",
                MAX_CAROUSEL_DATA_SIZE,
                assemble->auend + assemble->ausize[aucnt],
                aucnt,
                autot,
                assemble->ausize[aucnt]);

        clear_au(assemble);
        return 0;
    }

    /* now OK to copy in this AU */
    assemble->auend += assemble->ausize[aucnt];
    memcpy(assemble->au[aucnt], assemble->pktbuf, assemble->ausize[aucnt]);

    /* keep track of the current AU */
    assemble->aucur = (ushort)aucnt;

    /* returns '1' if we've assembled all of them in this group */
    return ((aucnt + 1) == autot) ? 1 : 0;
}

/**************************************************************************//**                                                                         *
 * This helper inline function invokes the AU assembly routine.  In the
 * case where the assembly sequence is broken, the assembly is restarted.
 *
 * \param[in] assemble a pointer to the AU assembly buffer
 * \param[in] dataStart byte offset to AU payload data
 * \param[in] aucnt AU sequence number
 * \param[in] auaid AU group ID
 * \param[in] autot total number of AUs in the group
 *
 * \return integer return code
 * \retval 0 = assembly incomplete, or just starting
 * \retval 1 = assembly complete
 *
 * \note
 * This function runs in service thread context
 *
 *****************************************************************************/
static SXE_INLINE uint assemble_pkt(SportsAUAssemble* assemble, byte* dataStart, uint aucnt,
    uint auaid, uint autot)
{

    //sxm_file_log("(au info)(autot=%d aucnt=%d auaid=%d pl=%d)",
    //			autot, aucnt, auaid, assemble->pktlen);

    /*
     * 0 - nothing to process,
     * 1 - process,
     * 2 - try again, gather sequence broken but this packet may be a starting point.
     *
     */
    int rc = assemble_au(assemble, dataStart, aucnt, auaid, autot);
    if(rc==2)
    {
        rc = assemble_au(assemble, dataStart, aucnt, auaid, autot);
    }
    return (rc == 1);
}

static SXE_INLINE uint assemble_data_pkt(SportsAUAssemble* assemble,
    byte* dataStart, uint aucnt, uint auaid, uint autot)
{

    //sxm_file_log("(au info)(autot=%d aucnt=%d auaid=%d pl=%d)",
    //			autot, aucnt, auaid, assemble->pktlen);

    /*
     * 0 - nothing to process,
     * 1 - process,
     * 2 - try again, gather sequence broken but this packet my be a starting point.
     */
    uint rc = assemble_data_au(assemble, dataStart, aucnt, auaid, autot);
    if(rc==2)
    {
        rc = assemble_data_au(assemble, dataStart, aucnt, auaid, autot);
    }
    return (rc == 1);
}

/************************************************************************
 *									                                    *
 *            Collection Layer						                    *
 *            ================						                    *
 *									                                    *
 ************************************************************************/
/**************************************************************************//**
 * The routine processes the Sports Service Configuration Carousel.
 *
 * \param[in] pkt a pointer bit buffer
 * \param[in] assemble a pointer to the AU assembly buffer
 *
 * \note
 * The receipt of a new Service Configuration Table triggers
 * a Broad-Scope-Change (bsc) and sets 'sports updating' state.
 * All previously recieved data are dropped.
 *
 * \note
 * This function runs in Service Thread context.
 *
 *****************************************************************************/
static void configCarousel(SXMBitBuff *pkt, SportsAUAssemble* assemble)
{
    uint rfu, cfgver, k, j;
    uint ctsize = 0, autot = 1, aucnt = 1, auaid = 0, sno;
    uint spid, tno, end, total;
    byte* cur;
    BOOL bsc;

    /* parse out the common header */
    NEXT_TO(uint, rfu, 1);       /* rfu */
    NEXT_TO(uint, cfgver, 8);    /* service configuration version */

    /* if FLAG is set, there is more that a single AU */
    if (FLAG()) {
		NEXT_TO_ADD(uint, ctsize, 4, 1);           /* AUTOT and AUCT field width */
		NEXT_TO_ADD(uint, autot, ctsize, 1);       /* total AU count */
		NEXT_TO(uint, aucnt, ctsize);              /* this AU's sequence number */
		NEXT_TO(uint, auaid, 16);                  /* AU Group ID */
	}

    /* align the bitbuff to the next byte boundary */
    cur = ALIGN();

    /* parse error check */
    if(pkt->err)
    {
        ERROR_PKT("Parse error 1: cfgver=%d", cfgver);
        return;
    }

    /*  just ignore this if we've got this version already? */
    if (cfgver == service->h->g.sections[SPORTS_SECTION_SPORTS].version)
    {
        DEBUG_PKT("(Same Config) cfgver=%d", cfgver);
        return;
    }

    /* debug trace */
    DEBUG_PKT("(New Config) rfu=%d cfgver=%d oldver=%d",
        rfu, cfgver,service->h->g.sections[SPORTS_SECTION_SPORTS].version);

    /* the AU(s) are assembled and then parsed through later individually */
    j = assemble_pkt(assemble, cur, aucnt, auaid, autot);
    if(!j)
    {
        return;
    }

    /* Sync with application thread */
    LOCK(service->mutex);

    /* A new configuration triggers a broad-scope-change - drop all data */
    init_cycle();

    /* now parse through the individual AU(s) in order */
    for(k=0,total=0; k<assemble->autot; k++)
    {
        sxm_bitbuff_setup(pkt, assemble->au[k], assemble->ausize[k]);
        SKIPBYTES(assemble->auoff[k]);

        /* the number of sports in this AU */
        NEXT_TO_ADD(uint, sno, 8, 1);
        if(pkt->err) {
            ERROR_PKT("ConfigCarousel parse error 2: cfgver=%d", cfgver);
            break;
        }

        /* we support a limited number of sports */
        end = MIN(total + sno, MAX_SPORT_COUNT);

        /* now parse out the rest of the service configuration payload */
        for (; total<end; total++)
        {
            Sport *s;
            char sname[MAX_SPORT_NAME_CHAR];

            /* the name of the sport */
            BAUDOT(sname, BAUDOT_MODE_START_WITH_LETTERS,
                SMALL_SYMCOUNT(MAX_SPORT_NAME_CHAR), MAX_SPORT_NAME_CHAR);

            /* the sport ID */
            NEXT_TO(uint, spid, 8);

            /* use the 'spid' as an index, use the current count ('total') as the key */
            SBLOCK[spid] = (ushort)total;
            s = &SPORT[total];

            /* the sport NAME */
            strncpy(s->name, sname, sizeof(s->name));
            s->name[sizeof(s->name) - 1] = '\0';

            /* the number of tables associated with this sport */
            NEXT_TO_ADD(uint, tno, 4, 1);
            for (j = 0; j < tno; j++)
            {
                NEXT_TO(ushort, s->tables[j], 10);
            }
            s->tno = tno;
        }

        /* check for parsing errors */
        if(pkt->err)
        {
            ERROR_PKT("Parse error 3: cfgver=%d pkt->err=%d", cfgver, pkt->err);
            break;
        }

        /* log message if we had more sports that we will support and had to ignore some of them */
        if(total >= MAX_SPORT_COUNT)
        {
            ERROR_PKT("(Exceeded configured sport count) cfgver=%d total=%d sno=%d maxsno=%d", cfgver, total, sno, MAX_SPORT_COUNT);
            break;
        }
    }

    /* reset the assembly buffer */
    clear_au(assemble);

    /* any parsing error, drop all data */
    if(pkt->err)
    {
        ERROR_PKT("Parse error 4: cfgver=%d pkt->err=%d", cfgver, pkt->err);
        init_cycle();
        cfgver = INVALID_VERSION;
    }

    /* now update the configuration version tag */
    service->h->g.sections[SPORTS_SECTION_SPORTS].version = (ushort)cfgver;

    /* mark Broad Scope Change */
    bsc = set_bsc();

    /* release sync */
    UNLOCK(service->mutex);

    /* notify of BSC */
    notify_updating(bsc);

    /* print tables in DEBUG mode */
    DSPORT_PRINT_SPORTS();
    DSPORT_PRINT_ALL();

}

/**************************************************************************//**
 * The routine determines and assigns the parent node for the affilation.
 *
 * \param[in] c pointer to the Affilation descriptor
 * \param[in] afcount number of affiliations to search
 *
 * \return TRUE if parent was found, FALSE otherwise
 *
 * \note
 * 
 * Root (level 0) is a special case.  It has no parent.
 *
 * \note
 * This function runs in Service Thread context.
 *
 *****************************************************************************/
static BOOL findParent(Affiliation* c, uint afcount)
{
    size_t plen;
    uint i;
    Affiliation* a;
    int look = 2;
    ushort sportid;
    char pname[MAX_AFFILIATE_NAME_CHAR];

    /* Roots are easy. */
    if (c->level == 0)
    {
        c->rootid = c->afid;
        c->parentid = INVALID_ID;
        return TRUE;
    }

    /* Isolate the parent name. */
    strcpy(pname, c->name);
    plen = strlen(pname);
    while(plen)
    {
        if(pname[plen] == ':')
        {
            pname[plen] = '\0';
            break;
        }
        plen--;
    }

    /* First look for a match with the same sport id. */
    sportid = c->spid;

    while(look)
    {
        for (i = 0; i<afcount; i++)
        {
            a = &AFF[i];

            /* Must be one level higher. */
            if (a->level != (c->level - 1))
            {
                continue;
            }

            /* Wrong id. */
            if (a->spid != sportid)
            {
                continue;
            }

            /* No match. */
            if (strcmp(a->name, pname) != 0)
            {
                continue;
            }

            /* assign the parent ID */
            c->parentid = a->afid;
            return TRUE;
        }

        /* Second look for a match with the aggregate sport id. */
        sportid = SPID_MULTI;
        look--;
    };

    /* No parent found. */
    return FALSE;
}

/**************************************************************************//**
 * The routine processes the Affililiation Carousel.
 *
 * \param[in] pkt a pointer bit buffer
 * \param[in] assemble a pointer to the AU assembly buffer
 *
 * \return None
 *
 * \note called in service thread context
 *
 *****************************************************************************/
static void affiliationCarousel(SXMBitBuff *pkt, SportsAUAssemble *assemble)
{
    uint rfu, cfgver, adver, actual;
    uint ctsize, autot = 1, aucnt = 1, auaid = 0;
    uint i, j, k;
    int cnt, ok;
    char* brk;
    char adname[MAX_AFFILIATE_NAME_CHAR];
    byte* cur;
    Affiliation* a;
    BOOL bsc, parent;

    /* Parse the AU header */
    NEXT_TO(uint, rfu, 1);       /* reserved for future use */
    NEXT_TO(uint, cfgver, 8);    /* configuration version */
    NEXT_TO(uint, adver, 8);     /* affiliation version */

    /* if 'flag' is set, then more than 1 AU */
    if (FLAG())
    {
        NEXT_TO_ADD(uint, ctsize, 4, 1);         /* autot and aucnt field width */
		NEXT_TO_ADD(uint, autot, ctsize, 1);     /* number of AUs */
		NEXT_TO(uint, aucnt, ctsize);            /* sequence number this AU */
		NEXT_TO(uint, auaid, 16);                /* AU group ID */
    }

    /* align bitbuff to next byte boundary */
    cur = ALIGN();

    /* now check for parsing errors before we go any further */
    if(pkt->err)
    {
        ERROR_PKT("Parse error 1: cfgver=%d adver=%d", cfgver, adver);
        return;
    }

    /* the configuration versions must match up, otherwise ignore this AU */
    if (cfgver != service->h->g.sections[SPORTS_SECTION_SPORTS].version)
    {
        DEBUG_PKT("(Config Mismatch) cfgver=%d storedcfgver=%d", cfgver, service->h->g.sections[SPORTS_SECTION_SPORTS].version);
        return;
    }

    /* ignore this AU if we've already got this version */
    if (adver == service->h->g.sections[SPORTS_SECTION_AFFS].version)
    {
        DEBUG_PKT("(SAME) cfgver=%d adver=%d", cfgver, adver);
        return;
    }

    /* log some trace info */
    DEBUG_PKT("(New) rfu=%d cfgver=%d adver=%d oldadver=%d",
        rfu, cfgver, adver, service->h->g.sections[SPORTS_SECTION_AFFS].version);

    /* assemble AU(s) */
    i = assemble_pkt(assemble, cur, aucnt, auaid, autot);
    if(!i)
    {
        return;
    }

    /* Drop all existing affiliates and all collected table data. */
    LOCK(service->mutex);
    drop_affiliates();

    /* now parse out the assembled AU(s) */
    for(k=0,actual=0; k<assemble->autot; k++)
    {

        /* automatic (stack) variables */
        uint adno = 0, gdref = 0, afid = 0, spid= 0;

        sxm_bitbuff_setup(pkt, assemble->au[k], assemble->ausize[k]);
        SKIPBYTES(assemble->auoff[k]);

        /* number of affiliate definitions in this AU */
        NEXT_TO_ADD(uint, adno, 10, 1);
        if(pkt->err)
        {
            ERROR_PKT("Parse error 2: cfgver=%d adver=%d", cfgver, adver);
            break;
        }

        /* Parse the affiliates. */
        for (i=0; i<adno; i++)
        {

            /* range check */
            if(actual >= MAX_AFFILIATE_COUNT)
            {
                ERROR_PKT("(Exceeded configured affiliate count) cfgver=%d actual=%d adno=%d maxadno=%d", cfgver, actual, adno, MAX_AFFILIATE_COUNT);
                break;
            }

            /* affiliation name */
            BAUDOT(adname, BAUDOT_MODE_START_WITH_LETTERS,
                BIG_SYMCOUNT(MAX_AFFILIATE_NAME_CHAR), MAX_AFFILIATE_NAME_CHAR);

            /* affiliation ID */
  			NEXT_TO(uint, afid, 10);
			if (FLAG()) 
            {
				NEXT_TO(uint, gdref, 7);
			}
            else
            {
                gdref = INVALID_GDREF;
            }

            /* the Sports ID ties to this affiliation */
			NEXT_TO(uint, spid, 8);

            /* Make sure we are interested in this sport. */
            if(SBLOCK[spid] == INVALID_ID)
            {
                continue;
            }

            /* validate the string */
            cnt = (int)strlen(adname);
            if(cnt == 0)
            {
                ERROR_PKT("(invalid adname 1) %s", adname);
                continue; /* cannot have empty name */
            }

            if(adname[0] == ':')
            {
                ERROR_PKT("(invalid adname 2) %s", adname);
                continue; /* cannot begin with ':' */
            }

            if(adname[cnt-1] == ':')
            {
                ERROR_PKT("(invalid adname 3) %s", adname);
                continue; /* cannot end with ':' */
            }

            /* drop affiliates with an empty name not at the ends and get the level. */
            for(brk=adname, ok=1, cnt=0;;)
            {
                brk = strpbrk(brk,":");

                /* if no delimiter, then this is single level */
                if(!brk)
                {
                    break;
                }

                cnt++;
                brk++; /* safe cannot end with ':' by check above */
                if(*brk == ':') 
                {
                    ok = 0;
                    break;
                }
            }

            /* make sure name is valid */
            if(!ok)
            {
                ERROR_PKT("(invalid adname 4) %s", adname);
                continue;
            }

            /* can only support up to max Level */
            if(cnt > MAX_AFFILIATE_NAME_LEVEL)
            {
                ERROR_PKT("(invalid adname 5) %s", adname);
                continue;
            }

            /* assign the index */
            ABLOCK[afid] = (ushort)actual;

            /* and now fill in the table */
            a = &AFF[actual];
            a->global = (ushort)gdref;  /* global data reference */
            a->spid = (ushort)spid;     /* sport ID */
            a->afid = (ushort)afid;     /* affiliation ID */
            a->ver = (ushort)adver;     /* affiliation database version */
            a->level = (ushort)cnt;     /* we use this to simplify parent-child determination. */
            strcpy(a->name, adname);    /* affiliation name */

            /* Init variables used for 'others' table presence. */
            for(j=0;j<ARRAY_SIZE(a->others);j++)
            {
                a->others[j].heapix = INVALID_HEAPIX;
                a->others[j].tabno = INVALID_ID;
                a->others[j].size = 0;
            }

            /* Init variables used for 'schedule' tables presence */
            for(j=0;j<ARRAY_SIZE(a->schedule);j++)
            {
                a->schedule[j].heapix = INVALID_HEAPIX;
                a->schedule[j].tabno = INVALID_ID;
                a->schedule[j].size = 0;
            }

            /* bump the count of affiliations */
            actual++;
        }

        /* check for parsing errors */
        if(pkt->err)
        {
            ERROR_PKT("(parse error 3) cfgver=%d adver=%d", cfgver, adver);
            break;
        }

        /* only support a certain number of affiliates */
        if(actual >= MAX_AFFILIATE_COUNT)
        {
            ERROR_PKT("(Exceeded configured affiliate count) cfgver=%d actual=%d adno=%d maxadno=%d", cfgver, actual, adno, MAX_AFFILIATE_COUNT);
            break;
        }
    }

    /* reset the assembly buffer */
    clear_au(assemble);

    /* if there are any parsing errors, can't keep the new table */
    if(pkt->err)
    {
        drop_affiliates();
        adver = INVALID_VERSION;
    }
    else
    {
        /* Find parents.  We can use the non mapped access since we have packed
         * the array and all entries are to be considered valid.
         */
        for(i=0; i<actual; i++)
        {
            a = &AFF[i];
            parent = findParent(a, actual);
            if(parent == FALSE)
            {
                ABLOCK[a->afid] = INVALID_ID;
                ERROR_PKT("affiliationCarousel(parent less) cfgver=%d adver=%d afid=%d adname=%s", cfgver, adver, a->afid, a->name);
            }
        }

        /*
         * Assign root links.  We make a pass over each level starting from highest to lowest. Roots have already been assigned:
         * we are propagating them down.  Must use the map array since we may have invalidated some affiliates
         * (ones we did not find a parent for).
         */
        for(j=1; j<=MAX_AFFILIATE_NAME_LEVEL; j++)
        {
            for(i=0; i<=MAX_AFFILIATE_ID; i++)
            {
                if(ABLOCK[i] == INVALID_ID)
                {
                    continue;
                }

                if (ABLOCK[i] >= ARRAY_SIZE(AFF))
                {
                    ERROR_PKT("invalid AFF index (i=%d,ABLOCK[i]=%d)", i, ABLOCK[i]);
                    continue;
                }

                a = &AFF[ABLOCK[i]];
                if(a->level == j)
                {
                    a->rootid = AFF[ABLOCK[a->parentid]].rootid;
                }
            }
        }
    }

    /* new version for affilation section */
    service->h->g.sections[SPORTS_SECTION_AFFS].version = (ushort)adver;

    /* an updated affiliation section triggers a broad-scope-change */
    bsc = set_bsc();

    /* release sync */
    UNLOCK(service->mutex);

    /* notify */
    notify_updating(bsc);

    /* reset BSC */
    clear_bsc();

    /* aff section updated */
    BITSET(service->sections_update, SPORTS_SECTION_AFFS);

    DSPORT_PRINT_AFFILIATES();
    DSPORT_PRINT_ALL();

}

/**************************************************************************//**
 * The routine processes the Table Definitions Carousel.
 *
 * \param[in] pkt a pointer bit buffer
 *
 * \return NONE
 *
 * \note table defintion is present in a single access unit
 *
 *****************************************************************************/
static void tablesCarousel(SXMBitBuff *pkt)
{
    uint rfu, cfgver;
    uint tabid, tabver, tclass, extcnt;
    uint lbno, i, tix;
    STable *t;
    TCol *l;
    BOOL bsc, drop = FALSE;

    /* parse the header */
    NEXT_TO(uint, rfu, 1);
    NEXT_TO(uint, cfgver, 8);
	NEXT_TO(uint, tabid, 10);
	NEXT_TO(uint, tabver, 8);
	NEXT_TO(uint, tclass, 4);
	ALIGN();
	NEXT_TO_ADD(uint, lbno, 6, 1);

    /* if any parse error, don't go any further */
    if	(pkt->err)
    {
        ERROR_PKT("(parse error) 1. cfgver=%d", cfgver);
        return;
    }

    /* ignore this table definition if it is out-of-sync with the current service configuration table */
    if (cfgver != service->h->g.sections[SPORTS_SECTION_SPORTS].version)
    {
        DEBUG_PKT("(Config Mismatch) cfgver=%d storedcfgver=%d", cfgver, service->h->g.sections[SPORTS_SECTION_SPORTS].version);
        return;
    }

    // ignore this table definition if it is not a support class
    if (!(TCLASS_SUPPORTED & (1<<tclass))) {
        DEBUG_PKT("(Unsupported table tclass) cfgver=%d tabid=%d tabver=%d tclass=%d lbno=%d", cfgver, tabid, tabver, tclass, lbno);
        return;
    }

    /* we may already have this table if the index is valid */
    tix = TBLOCK[tabid];

    if ((tix >= ARRAY_SIZE(TABLE)) && (tix != INVALID_ID)) 
    {
        ERROR_PKT("(Invalid table index) cfgver=%d tabid=%d tabver=%d tclass=%d lbno=%d tix=%d", cfgver, tabid, tabver, tclass, lbno, tix);
        return;
    }

    /* is this a new table ? */
    if(tix == INVALID_ID)
    {

        DEBUG_PKT("(NEW) cfgver=%d tabid=%d tabver=%d tclass=%d lbno=%d", cfgver, tabid, tabver, tclass, lbno);
        for(i=0; i<ARRAY_SIZE(TABLE); i++)
        {
            if(TABLE[i].ver == INVALID_VERSION)
            {
                tix = i;
                break;
            }
        }

        /* no room to store new table, so nothing else to do */
        if(i == MAX_TABLE_COUNT)
        {
            ERROR_PKT("(No free table entries) max=%d", i);
            return;
        }
    }
    else
    {

        /* already have this table, ignore it */
        if(TABLE[tix].ver == tabver)
        {
            DEBUG_PKT("(SAME) cfgver=%d tabid=%d tabver=%d tclass=%d lbno=%d tix=%d",
                cfgver, tabid, tabver, tclass, lbno, tix);
            return;
        }

        /* Replacing existing need to drop all data that uses it. */
        drop = TRUE;
    }

    DEBUG_PKT("(SAVE) rfu=%d cfgver=%d tabid=%d tabver=%d tclass=%d lbno=%d tix=%d",
        rfu, cfgver, tabid, tabver, tclass, lbno, tix);

    /* New table cannot be stored due to resource limits.
     * Drop all old data and invalidate the existing table.
     */
    if(lbno > MAX_TABLE_LABEL_COUNT)
    {
        LOCK(service->mutex);
        drop_all_table_data_ignore_use(tabid);
        bsc = set_bsc();

        /* invalidate the existing table defintion */
        TABLE[tix].ver = INVALID_VERSION;
        TBLOCK[tabid] = INVALID_ID;

        /* TABLES section has updated */
        BITSET(service->sections_update, SPORTS_SECTION_TABLES);

        ERROR_PKT("(Label count too big. Table def dropped) lbno=%d max=%d", lbno, MAX_TABLE_LABEL_COUNT);
        UNLOCK(service->mutex);
        notify_updating(bsc);
        return;
    }

    LOCK(service->mutex);

    /* replacing table, drop all old data */
    if(drop == TRUE)
    {
        drop_all_table_data_ignore_use(tabid);
    }

    /* new or replaced table */
    TBLOCK[tabid] = (ushort)tix;
    t = &TABLE[tix];

    // parse the table defintion payload label section
    for (i = 0; i < lbno; i++)
    {
        l = &t->l[i];
        NEXT_TO(byte, l->lbid, 8);
        NEXT_TO(byte, l->dtype, 1);
        NEXT_TO(byte, l->lbtype, 8);
        NEXT_TO(byte, l->havedmod, 1);
        if (l->havedmod)
        {
            NEXT_TO(byte, l->dmod, 3);
        }
        NEXT_TO(byte, l->prio, 3);
        NEXT_TO(ushort, l->lbsize, 12)+1;
        NEXT_TO(byte, l->havetext, 1);
        if (l->havetext)
        {
            BAUDOT(l->text, BAUDOT_MODE_START_WITH_LETTERS,
                SMALL_SYMCOUNT(MAX_LABEL_TEXT_COUNT), MAX_LABEL_TEXT_COUNT);
        }

        /* extra data */
        if (FLAG())
        {
            NEXT(extcnt, 8)+1;
            SKIPBYTES(extcnt);
        }

        /* check for any parsing error */
        if(pkt->err)
        {
            ERROR_PKT("(parse error 2) cfgver=%d label=%d lbno=%d", cfgver, i, lbno);
            break;
        }
    }

    /* make sure we processed all the lables we were expecting */
    if(i != lbno)
    {
        TABLE[tix].ver = INVALID_VERSION;
        TBLOCK[tabid] = INVALID_ID;
    }
    else
    {
        t->tclass = (ushort)tclass;
        t->ver = (ushort)tabver;
        t->lc = (ushort)lbno;
        DSPORT_PRINT_TABLEDEF(&TABLE[TBLOCK[tabid]], tabid);
    }

    /* if dropped, set the bsc active */
    if(drop == TRUE)
    {
        bsc = set_bsc();
    }

    UNLOCK(service->mutex);

    // notify if dropped
    if(drop == TRUE)
    {
        notify_updating(bsc);
    }

    /* TABLE section has been updated */
    BITSET(service->sections_update, SPORTS_SECTION_TABLES);

    DSPORT_PRINT_ALL();
}

/**************************************************************************//**
 * The routine processes the Table Data Carousel.
 *
 * \param[in] pkt a pointer to the bit buffer
 * \param[in] assemble pointer to the assembly buffer
 * \param[in] crc crc of AU
 * \param[in] epochNow the current epoch
 *
 *****************************************************************************/
static void dataCarousel(SXMBitBuff *pkt, SportsAUAssemble* assemble,
    uint crc, uint epochNow)
{
    Affiliation *a;
    STable *t;
    Table *data;
    Table *freeTable;
    byte* cur;
    byte* buffer;
    uint rfu, cfgver, afid, season, tid, tver, toffset, pl;
    uint autot = 1, aucnt = 1, auaid = 0, ctsize = 0;
    int newBlocks, oldBlocks;
    int epochDelta;
    int tableIsSchedule = 0;
    uint epoch;
    uint timeNow, i, tidx=0;

    /* parse out the header */
   	NEXT_TO(uint, rfu, 1);
	NEXT_TO(uint, cfgver, 8);
	NEXT_TO(uint, afid, 10);
	NEXT_TO(uint, epoch, 16);
	NEXT_TO(uint, season, 1);

    /* if FLAG is set, there is more that a single AU */
	if (FLAG()) 
    {
		NEXT_TO_ADD(uint, ctsize, 4, 1);
		NEXT_TO_ADD(uint, autot, ctsize, 1);
		NEXT_TO(uint, aucnt, ctsize);
		NEXT_TO(uint, auaid, 16);
    }

    /* align up to the next byte boundary */
    cur = ALIGN();

    /* any parsing errors, ignore this AU */
    if(pkt->err)
    {
        ERROR_PKT("(Parse error) 1 cfgver=%d", cfgver);
        return;
    }

    /*  ignoring this AU if it is out-of-sync with the configuration */
    if (cfgver != service->h->g.sections[SPORTS_SECTION_SPORTS].version)
    {
        DEBUG_PKT("(Config Mismatch) cfgver=%d storedcfgver=%d", cfgver, service->h->g.sections[SPORTS_SECTION_SPORTS].version);
        return;
    }

//	sxm_file_log("(au info)(ctsize=%d autot=%d aucnt=%d auaid=%d pl=%d) cfgver=%d afid=%d epoch=%d season=%d crc=%d",
//				ctsize, autot, aucnt, auaid, assemble->pktlen, cfgver, afid, epoch, season, crc);

    // single AU table
    if(autot == 1)
    {

        /*
         * Optimization to avoid recalculating the crc and bitbuff setup stuff.
         * We always clear any au gather so this is ok to do.
         */
        pl = assemble->pktlen;
        buffer = assemble->pktbuf;
        toffset = cur - assemble->pktbuf;
    }
    else
    {

        /* assemble multiple AU table */
        i = assemble_data_pkt(assemble, cur, aucnt, auaid, autot);

        /* don't process until all the AUs are collected */
        if(!i)
        {
            return;
        }

        /* Point to the assembled packet now. */
        pl = assemble->auend - assemble->aubuf;
        buffer = assemble->aubuf;
        toffset = assemble->startOff;
        sxm_bitbuff_setup(pkt, buffer, pl);
        SKIPBYTES(toffset);
        crc = sxm_crc32_calculate(buffer, pl);	/* use crc over combined packet. */
    }

    /* reset the assembly buffer */
    clear_au(assemble);

    /* table ID and version */
    NEXT_TO(uint, tid, 10);
    NEXT_TO(uint, tver, 8);
    if(pkt->err)
    {
        ERROR_PKT("(Parse error) 2 cfgver=%d", cfgver);
        return;
    }

    DEBUG_PKT("(Table data) rfu=%d cfgver=%d afid=%d epoch=%d season=%d tid=%d tver=%d autot=%d ABLOCK=%d tblock=%d, crc=%d",
            rfu, cfgver, afid, epoch, season, tid, tver, autot, ABLOCK[afid], TBLOCK[tid], crc);

    /* Need affiliate and table definition to collect. */
    if (ABLOCK[afid] == INVALID_ID || TBLOCK[tid] == INVALID_ID)
    {
        DEBUG_PKT("(No affiliate or table def)");
        return;
    }

    /* Guard against AFF invalid index */
    if (ABLOCK[afid] >= ARRAY_SIZE(AFF))
    {
        ERROR_PKT("invalid AFF index (afid=%d,ABLOCK[afid]=%d)", afid, ABLOCK[afid]);
        return;
    }

    /* Guard against TABLE invalid index */
    if (TBLOCK[tid] >= ARRAY_SIZE(TABLE))
    {
        ERROR_PKT("invalid TABLE index (tid=%d,TBLOCK[tid]=%d)", tid, TBLOCK[tid]);
        return;
    }

    /* The received table def version must match the version of our stored table def. */
    t = &TABLE[TBLOCK[tid]];
    if(t->ver != tver)
    {
        DEBUG_PKT("(table def version mismatch) stored=%d tver=%d", t->ver, tver);
        return;
    }

    LOCK(service->mutex);

    /* affiliation reference */
    a = &AFF[ABLOCK[afid]];
    epochDelta = epoch - (int)epochNow;
    timeNow = sxm_sxe_time_now();

    /* schedule table */
    if ((t->tclass == TCLASS_H2H_EVENT) || (t->tclass == TCLASS_RL_EVENT))
    {

        /* can't use an out-of-range schedule */
        if((epochDelta < -8) || (epochDelta > 8))
        {
            ERROR_PKT("(schedule epoch out of range) now=%d table->epoch=%d", epochNow, epoch);
            UNLOCK(service->mutex);
            return;
        }

        /* Find an existing schedule table for this epoch. */
        for(i=0; i<ARRAY_SIZE(a->schedule); i++) {
            if(a->schedule[i].size && (a->schedule[i].epoch == epoch)) {
                break;
            }
        }

        /* No table exists for the epoch */
        if(i == ARRAY_SIZE(a->schedule))
        {
            DEBUG_PKT("(NEW)");

            /* Try and drop any old tables not in use */
            for(i=0; i<ARRAY_SIZE(a->schedule); i++)
            {
                if(a->schedule[i].size && (((uint)(a->schedule[i].epoch) + 8) < epochNow))
                {
                    DEBUG_PKT("(Drop old) tabno=%d heapix=%d size=%d epoch=%d ts=%u inuse=%u crc=%d",
                            a->schedule[i].tabno,
                            a->schedule[i].heapix,
                            a->schedule[i].size,
                            a->schedule[i].epoch,
                            a->schedule[i].timeStamp,
                            a->schedule[i].inuse,
                            a->schedule[i].crc );
                    drop_table_data_check_use(&a->schedule[i]);
                }
            }

            /* Find a free table entry */
            for(i=0; i<ARRAY_SIZE(a->schedule); i++)
            {
                if(!a->schedule[i].size)
                {
                    break;
                }
            }
        }

        /* Table exists for epoch. */
        else
        {

            /* Table is the same if CRCs match, just update the timestamp */
            if(a->schedule[i].crc == crc)
            {
                a->schedule[i].timeStamp = timeNow;
                BITSET(service->sections_update, SPORTS_SECTION_AFFS);
                DEBUG_PKT("(SAME) ts=%u", timeNow);
                UNLOCK(service->mutex);
                return;
            }

            /* Table is in use, can't update it now */
            if(a->schedule[i].inuse)
            {
                DEBUG_PKT("(INUSE) %u", (uint)a->schedule[i].inuse);
                UNLOCK(service->mutex);
                return;
            }

            DEBUG_PKT("(UPDATE) oldcrc=%d", a->schedule[i].crc);
        }

        /* No resources to save this table. */
        if(i == ARRAY_SIZE(a->schedule))
        {
            ERROR_PKT("(No schedule resource available)");
            UNLOCK(service->mutex);
            return;
        }

        /* assign slot for the table */
        data = &a->schedule[i];
        tableIsSchedule = 1;
    }

    /* others table */
    else {

        /* only 'today' tables will be cached */
        if(epochDelta != 0)
        {
            DEBUG_PKT("(Others epoch not today)");
            UNLOCK(service->mutex);
            return;
        }

        data = freeTable = NULL;

        /* News is always the 0th entry. */
        if(t->tclass == TCLASS_RL_NEWS)
        {
            tidx = 0;
            if(a->others[0].size)
            {
                data = &a->others[0];
            }
            else
            {
                freeTable = &a->others[0];
            }
        }
        else
        {

            /* Find the table. */
            for (i=1; i<ARRAY_SIZE(a->others); i++)
            {
                if(a->others[i].size)
                {
                    if((a->others[i].epoch == epoch) && (a->others[i].tabno == tid))
                    {
                        data = &a->others[i];
                        tidx = i;
                        break;
                    }
                }
                else
                {
                    freeTable = &a->others[i];
                    tidx = i;
                }
            }
        }

        /* Found a populated instance of the table. */
        if(data)
        {

            /* Table is the same, just update the timestamp. */
            if(data->crc == crc)
            {
                data->timeStamp = timeNow;
                BITSET(service->sections_update, SPORTS_SECTION_AFFS);
                DEBUG_PKT("(SAME) ts=%u", data->timeStamp);
                UNLOCK(service->mutex);
                return;
            }

            /* the table is in use, nothing to do */
            if(data->inuse)
            {
                DEBUG_PKT("(INUSE) %u", (uint)data->inuse);
                UNLOCK(service->mutex);
                return;
            }

            DEBUG_PKT("(UPDATE) oldcrc=%u newcrc=%u", data->crc, crc);
        }
        else {

            if(!freeTable)
            {
                DEBUG_PKT("(NO FREE OTHERS SLOT)");

                /* Drop an expired table to find one. */
                for (i=1; i<ARRAY_SIZE(a->others); i++)
                {
                    data = &a->others[i];
                    if(data->size && (data->epoch != epochNow))
                    {
                        DEBUG_PKT("(drop old) tabno=%d heapix=%d size=%d epoch=%d ts=%u crc=%d",
                                data->tabno,
                                data->heapix,
                                data->size,
                                data->epoch,
                                data->timeStamp,
                                data->crc );
                        if(drop_table_data_check_use(data))
                        {
                            freeTable = data;
                            tidx = i;
                            break;
                        }
                    }
                }

                /* can't find a table to drop, nothing else to do */
                if(!freeTable)
                {
                    UNLOCK(service->mutex);
                    return;
                }
                DEBUG_PKT("(FREED UP OTHERS SLOT)");
            }

            data = freeTable;
        }
    }

    /*
     * If we get here: Either new table and a slot to keep one or an existing table with different crc.
     */

    /* fill in the table metadata */
    data->crc = crc;
    data->epoch = (ushort)epoch;
    data->season = (ushort)season;
    data->tabno = (ushort)tid;
    data->tabver = (ushort)tver;
    data->timeStamp = timeNow;
    data->offset = (ushort)toffset;
    data->inuse = 0;

    /* TREF map */
    if(autot != 1)
    {
        memcpy(data->trefmap, assemble->trefmap, sizeof(assemble->trefmap));
    }
    else {
        memset(data->trefmap, 0, sizeof(assemble->trefmap));
    }

    /* now calculate the new and old block requirements */
    newBlocks = (pl + SXM_ARCH_FILE_SECTOR_SIZE - 1) >> SXM_ARCH_FILE_SECTOR_DIVISOR;
    oldBlocks = (data->size + SXM_ARCH_FILE_SECTOR_SIZE - 1) >> SXM_ARCH_FILE_SECTOR_DIVISOR;

    /* oldBlocks is zero if a new table */
    if (oldBlocks == 0)
    {
        data->heapix = INVALID_HEAPIX;
    }

    /* old table and new table are have same block requirement, leave the map alone */
    else if (oldBlocks == newBlocks)
    {
    }

    /* free the old blocks */
    else
    {
        sxm_cfile_map_free(ROOT, SXM_CFILE_SECTION_INVALID_ID, data->heapix, (uint)oldBlocks);
        data->heapix = INVALID_HEAPIX;
    }

    /* if not enough room to save the new table, drop the old table */
    if(!save_table_packet(data, (ushort)pl, buffer))
    {
        drop_table_data_ignore_use(data->tabno, data, 1);
    }
    else
    {

        SportsRequest* r;
        Affiliation *b;

        /* now notify all outstanding requests with common root affiliate */
        for(i=0; i<ARRAY_SIZE(service->request); i++)
        {
            r = &service->request[i];
            LOCK(r->mutex);
            if((r->sportId == INVALID_ID) || (r->rdelete == TRUE)
                || (r->bsc == TRUE) || (r->addNotify == TRUE) )
            {
                UNLOCK(r->mutex);
                continue;
            }

            /* find root affiliate */
            b = sxm_sports_shared_find_root(r);
            if(!b)
            {
                ERROR_PKT("(unable to find root affiliate for request) afname=%s", r->afname);
            }
            else
            {

                /* notify if common root affiliate */
                if(b->rootid == a->rootid)
                {
                    if(tableIsSchedule)
                    {
                        notify_schedule(r, data, 1, a->afid, epochNow);
                    }
                    else
                    {
                        notify_other(r, data, 1, a->afid, (MAX_SCHEDULE_INDEX+1) + tidx);
                    }
                    if(r->deleteAtEnd == TRUE)
                    {
                        r->sportId = INVALID_ID;
                    }
                }
            }
            UNLOCK(r->mutex);
        }
    }

    DSPORT_PRINT_ALL();
    UNLOCK(service->mutex);

}

/************************************************************************
 *									                                    *
 *            Change processing						                    *
 *            ==============						                    *
 *									                                    *
 ************************************************************************/

/**************************************************************************//**
 *
 * This is the service-thread callback for processing AUs received OTA for
 * the Sports data service. This function pre-processes all the Access Units
 * (AUs) for the service and then dispatches them for further processing based
 * on carousel ID.
 *
 * \param[in] buff a pointer to the buffer containing the AU
 * \param[in] pl packet length in bytes
 * \param[in] ptr opaque user context pointer
 *
 * \note
 * Runs in service thread context
 * 
 * \note
 * Multiple carousels may be in various states of processing concurrently.
 *
 *****************************************************************************/
void sxm_sports_complete_au(byte *buff, int pl, ptr dsuser)
{
    SXMBitBuff apkt, *pkt;
    uint pvn, cid, crc;
    uint epochNow;
    
    UNUSED_VAR(dsuser);

    /* We will use the same epoch for the entire process of this data packet. */
    epochNow = SPORTS_EPOCH_NOW();
    process_global_versions();
    clear_bsc();
    new_request_notify(epochNow);
    check_epoch(epochNow);

#ifdef SXM_DEBUG_PKT_METRICS
    /* Update statistics */
    sports_packets++;
    sports_bytes += (uint)pl;
#endif /* #ifdef SXM_DEBUG_PKT_METRICS */

    sxm_bitbuff_setup(&apkt, buff, (uint)(pl - 4));
    pkt = &apkt;

    NEXT(pvn, 4);
    NEXT(cid, 3);

    /* Protocol Version check */
    if (pvn != 1)
    {
        ERROR_PKT("(Invalid pvn) %d", pvn);
        return;
    }

    /* CRC check - toss it if it's bad */
    if (sxm_crc32_check(buff, (uint)pl, &crc) == 0)
    {
        ERROR_PKT("Checksum Failure");
        return;
    }

    /* process the AU by Carousel ID */
    switch (cid) 
    {

    case SPORTS_CAROUSEL_SPORTS:
        service->aus[0].pktbuf = buff;
        service->aus[0].pktlen = (ushort)pl;
        configCarousel(pkt, &service->aus[0]);
        break;

    case SPORTS_CAROUSEL_AFFS:
        service->aus[1].pktbuf = buff;
        service->aus[1].pktlen = (ushort)pl;
        affiliationCarousel(pkt, &service->aus[1]);
        clear_bsc();  /* Immediately notify aff update complete */
        break;

    case SPORTS_CAROUSEL_TABLES:
        tablesCarousel(pkt);
        clear_bsc(); /* Immediately notify table def update complete */
        break;

    case SPORTS_CAROUSEL_DATA0:
        service->aus[2].pktbuf = buff;
        service->aus[2].pktlen = (ushort)pl;
        dataCarousel(pkt, &service->aus[2], crc, epochNow);
        break;

    case SPORTS_CAROUSEL_DATA1:
        service->aus[3].pktbuf = buff;
        service->aus[3].pktlen = (ushort)pl;
        dataCarousel(pkt, &service->aus[3], crc, epochNow);
        break;

    default:
        WARN_PKT("(Unknown/Reserved carid) %d", cid);
        break;
    }

}


void sxm_sports_timer(void *pArg)
{
	uint epochNow;
//	sxm_file_log("TIMER CALLBACK FIRED");
	UNUSED_VAR(pArg);

    /* we need both the SPORTS and AFFILATION tables */
    if( (service->h->g.sections[SPORTS_SECTION_SPORTS].version == INVALID_VERSION) ||
        (service->h->g.sections[SPORTS_SECTION_AFFS].version == INVALID_VERSION))
    {
        return;
    }

    /* EPOCH NOW */
    epochNow = SPORTS_EPOCH_NOW();

    /* if new versions of global tables, set BSC event and notify 'updating' */
    process_global_versions();

    /* if BSC event due to new tables, clear BSC event and notify 'updated' */
    clear_bsc();

    /* notify new requests since BSC event */
    new_request_notify(epochNow);

    check_epoch(epochNow);

}

/************************************************************************
 *									                                    *
 *            Protocol Level						                    *
 *            ==============						                    *
 *									                                    *
 ************************************************************************/

/**************************************************************************//**
 *
 * The routine is called during service start to spin up the protocol handler.
 * The function will restore the service data from NVM (cycle file) and
 * validate its contents.  Corrupted service data will be initialized to
 * empty data.
 *
 * \note runs in application thread context
 *
 *****************************************************************************/
int sxm_sports_protocol_start(SportsService *ims)
{
    int rc;
    BOOL ok = FALSE;

    service = ims;

    ENTER_PKT("");

    /* open/create the cycle file */
    if ((service->cycle = sxm_cfile_open(SPORTS_SERVICE_NAME, SPORTS_CFILE_NAME)) == NULL)
    {
        ERROR_PKT("CFile opening failed");
        LEAVE_PKT("%s", sxm_sdk_format_result(SXM_E_PIPE));
        return SXM_E_PIPE;
    }

    /* load the cycle file into memory */
    rc = sxm_cfile_alloc_and_load(service->cycle, sizeof(SportSave),
                                  (ptr*) &service->h);
    if (rc == -1)
    {
        ERROR_PKT("CFile reading failed");
        LEAVE_PKT("%s", sxm_sdk_format_result(SXM_E_NOMEM));
        return SXM_E_NOMEM;
    }

    /* check it */
    if (rc == SPORT_BLOCKS)
    {
        uint schema;
        rc = sxm_cfile_check_format(ROOT, SPORTS_SERVICE_NAME, &schema);
        if ((rc == SXM_E_OK) && (SPORTS_CFILE_SCHEMA_VERSION != schema))
        {
            rc = SXM_E_ERROR;
        }
        if (rc == SXM_E_OK)
        {
            SXMCFileValidationResult vc;
            vc = sxm_cfile_validate(ROOT);
            switch (vc)
            {

            case SXM_CFILE_INVALID:
                /* root crc failed. Toast. */
                ERROR_PKT("Cycle root CRC failure");
                ok = FALSE;
                break;

            case SXM_CFILE_PARTIALY_VALID:
                /* root crc ok but whole file crc failed.  Maybe we can keep some of the sections. */
                ERROR_PKT("Cycle whole CRC failure");
                ok = salvage_cycle();
                break;

            case SXM_CFILE_FULLY_VALID:
                /* both root and whole crc passed. */
                ok = TRUE;
                break;

            default:
                ERROR_PKT("Unknown cfile_validate return code: %d", rc);
                break;
            }
        }
        else
        {
            ERROR_PKT("Cycle read unsupported format");
        }
    }
    else
    {
        ERROR_PKT("cycle read size failure.  readcnt=%d SPORT_BLOCKS=%d",
            rc, SPORT_BLOCKS);
    }

    // the cycle file is corrupt, or doesn't exist yet
    if (ok == FALSE)
    {
        uint section;

        /* zero fill the SPORTS Save, set the schema version */
        memset(service->h, 0, sizeof(*(service->h)));
        sxm_cfile_set_format(ROOT, SPORTS_SERVICE_NAME, SPORTS_CFILE_SCHEMA_VERSION);

        /* now add the individual sections in order of ascending section number */
        for (section = SPORTS_SECTION_FIRST; section < SPORTS_SECTIONS_COUNT; ++section)
        {
            switch(section)
            {

                /* adding the SPORTS section */
                case SPORTS_SECTION_SPORTS:
                {
                    sxm_cfile_add_section(ROOT, section, CSIZE(SportSection, 1));
                    DEBUG_PKT(" adding SportSection to cycle file (%u Sectors)",
                        CSIZE(SportSection, 1));
                }
                break;
                
                /* adding the AFFILIATION section */
                case SPORTS_SECTION_AFFS:
                {
                    sxm_cfile_add_section(ROOT, section, CSIZE(AffiliationSection, 1));
                    DEBUG_PKT(" adding AffiliationSection to cycle file (%u Sectors)",
                        CSIZE(AffiliationSection, 1));
                }
                break;
                
                /* adding the TABLES section */
                case SPORTS_SECTION_TABLES:
                {
                    sxm_cfile_add_section(ROOT, section, CSIZE(STableSection, 1));
                    DEBUG_PKT(" adding Table Section to cycle file (%u Sectors)",
                        CSIZE(STableSection, 1));
                }
                break;

                /* adding the HEAP section */
                case SPORTS_SECTION_HEAP:
                {
                    sxm_cfile_add_section(ROOT, section, CSIZE(HeapSection, 1));
                    DEBUG_PKT(" adding HeapSection to cycle file (%u Sectors)",
                        CSIZE(HeapSection, 1));
                }
                break;
            }
        }

        /* initialize the cycle file */
        init_cycle();

        DEBUG_PKT("New cycle file");
    }

    //DSPORT_PRINT_SPORTS();
    //DSPORT_PRINT_AFFILIATES();
    //DSPORT_PRINT_TABLEDEFS();
    DSPORT_PRINT_ALL();

    /* delete stale data */
    service->lastEpoch = SPORTS_EPOCH_NOW();
    purge_old_data(service->lastEpoch);

    /* global table versions at startup */
    service->globalLeagueVer = GLOBAL_VERSION_STARTUP;
    service->globalTeamVer = GLOBAL_VERSION_STARTUP;

    LEAVE_PKT("%s", sxm_sdk_format_result(SXM_E_OK));
    return SXM_E_OK;
}

/**************************************************************************//**
 *
 * The routine is called during service stop to shutdown the protocol handler.
 * The function will save (commit) the service data to NVM if the autosave
 * option is in effect.
 *
 * \param[in] NONE
 *
 * \return NONE
 *
 * \note runs in application thread context
 *
 *****************************************************************************/
void sxm_sports_protocol_stop(void)
{

    int rc;

    /* commit the cycle file to NVM if 'autosave' is enabled */
    if (service->ds.auto_save_on_stop)
    {
        rc = sxm_sports_commit(NULL);
        if(rc != SXM_E_OK)
        {
            ERROR_PKT("sxm_sports_commit(err) rc=%d blockCnt=%d", rc, SPORT_BLOCKS);
        }
    }

    /* done with the cycle file, close it, no commit */
    sxm_cfile_close(service->cycle, NULL);

    service = NULL;
}

/************************************************************************//*
 *									                                    *
 *            Shared				                                    *
 *            ======				                                    *
 *                                                                      *
 * Functions defined in this section are referred to as 'shared' since  *
 * they are externally referenced from sports functions defined in      *
 * other source files.                                                  *
 *                                                                      *
 *									                                    *
 ************************************************************************/

/**************************************************************************//**
 *
 * The routine is called to locate the root affilation associated with the
 * request.  The root affiliation may be located in the request, or we may
 * have to look it up since it may not have been known at the time the
 * request was issued.
 *
 * \param[in] r a pointer to the SportsRequest
 *
 * \return a pointer to the Affiliation for this request, on NULL if not found
 *
 * \note may run in application thread or service thread context
 *
 *****************************************************************************/
Affiliation* sxm_sports_shared_find_root(SportsRequest* r)
{
    int ok;
    uint i;
    Affiliation* a;

    /* the root affiliate may be in the request, so just return it */
    if(r->afid != INVALID_ID)
    {
        if((r->afid <= MAX_AFFILIATE_ID) && (ABLOCK[r->afid] != INVALID_ID))
        {
            /* Guard against invalid index */
            if (ABLOCK[r->afid] < ARRAY_SIZE(AFF))
            {
                a = &AFF[ABLOCK[r->afid]];
                return a;
            }
            DEBUG_PKT("invalid AFF index (r->afid=%d,ABLOCK[r->afid]=%d)",
                r->afid, ABLOCK[r->afid]);
        }
        return NULL;
    }

    /* it wasn't in the request, try to find it. */
    for (i = 0; i <= MAX_AFFILIATE_ID; i++)
    {
        if (ABLOCK[i] == INVALID_ID)
        {
            continue;
        }

        /* Guard against AFF invalid index */
        if (ABLOCK[i] >= ARRAY_SIZE(AFF))
        {
            ERROR_PKT("invalid AFF index (i=%d,ABLOCK[i]=%d)", i, ABLOCK[i]);
            continue;
        }

        a = &AFF[ABLOCK[i]];

        /* Only roots can match a request. */
        if (a->level != 0)
        {
            continue;
        }

        /* the affilation and the sport must match */
        ok = strcmp(r->afname, a->name);
        if ((ok == 0) && (r->sportId == a->spid))
        {
            return a;
        }
    }

    /* no affilation for this request */
    return NULL;
}
