/************************************************************************
 *                                                                      *
 *            SXM_PARKING_UPDATE.C                                      *
 *            ====================                                      *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 *                         Apply the update file to the saved database  *
 *                                                                      *
 ************************************************************************/
/***************************************************************************//**
 *
 * \file sxm_parking_update.c
 * \author Leslie French
 * \date 8/20/2013
 *
 * Apply the update file to the saved database
 *
 ******************************************************************************/

#define DEBUG_TAG "parking"

#include "sxm_parking_internal.h"

#ifdef _WIN32
#define snprintf _snprintf
#endif

/** Keeps reference to the service only for RFD callback */
static ParkingService *gService = NULL;

/***************************************************************************//**
 * This function is called to parse over a Parking Station record in the update
 * file without processing any of the data.  Essentially, a single record is
 * discarded or 'skipped over', because it contains invalid data.  For example,
 * the Parking Station ID of the incoming record exceeds the maximum allowed by
 * the architecture.
 *
 * \param[in] pkt   SXMBitBuff address
 * \param[in] ut    update type
 * 
 * \note this function runs in RFD thread context
 *
 ******************************************************************************/
static void parking_skip_station(SXMBitBuff *pkt, ParkingUpdateTypes ut)
{
    if (ut != PARKING_UTYPE_DELETE) {
        if (FLAG()) {   // F+LOC
            uint loc;
            NEXT(loc, 2);
            if  (loc == 3) SKIP(8);
            if  (loc == 2 || loc == 3) SKIP(8);
            if  (loc != 0) SKIP(12);
        }

        if (FLAG()) SKIP(16);    // F+LOC2

        if (FLAG()) BSKIP(0);    // F+PNAME
        if (FLAG()) BSKIP(1);    // F+ADDRESS1
        if (FLAG()) BSKIP(1);    // F+ADDRESS2
        if (FLAG()) BSKIP(0);    // F+CITY
        if (FLAG()) SKIP(7);     // F+STATE

        if (FLAG()) {            // F+ZIP
            if (!FLAG()) {
                SKIP(17);
            }
            else {
                BSKIP(0);
            }
        }

        if (FLAG()) SKIP(34);   // F+PHONE
        if (FLAG()) SKIP(7);    // F+HGT1
        if (FLAG()) SKIP(7);    // F+HGT2
        if (FLAG()) SKIP(16);   // F+SPACES

        if (FLAG()) {           // F+OPERATION

            int i;
            for (i = 0; i < 7; i++) {
                uint uvalue;
                NEXT(uvalue, 2);
                if (uvalue == 2) {
                    SKIP(14);
                }
            }
        }
        if (FLAG()) SKIP(8);     // F+PRICE
        if (FLAG()) BSKIP(1);    // F+COMM
        if (FLAG()) SKIP(16);    // F+AM1
        if (FLAG()) SKIP(16);    // F+AM2
    }
    if (FLAG()) {                // F+EXT_SIZE
        uint esize;
        NEXT(esize, 8);
        SKIP(esize * SXM_ARCH_BITS_IN_BYTE);
    }
}

/***************************************************************************//**
 * This function is called to merge a single unmodified and undeleted record
 * from the about-to-be deleted blocks with the records in the new blocks.
 *
 * \param[in] service   service instance
 * \param[in] stream    active stream
 * \param[in] rec       byte address of old record
 *
 * \note this function runs in RFD thread context
 * 
 ******************************************************************************/
static void parking_merge1(SXMGridService *service, SXMTFileStream *stream, const char *rec) {
    int j;
    const char *prec = rec +
        sizeof(fix) /* lon */ + sizeof(fix) /* lat */ +
        sizeof(fix) /* dlon */ + sizeof(fix) /* dlat */ + 
        sizeof(ushort) /* ver */ +
        sizeof(ushort) /* hgt1 */ + sizeof(ushort) /* hgt2 */ +
        sizeof(ushort) /* spaces */ + sizeof(ushort) /* prices */ +
        sizeof(ushort) /* amen1 */ + sizeof(ushort) /* amen2 */ + 
        SXM_GRID_PHONE_BYTELEN;

    UNUSED_VAR(service);

    /* name, address1, address2, city, state, zip, operation, comm.
     * The order doesn't matter since all those fields are nil-terminated strings
     * and can be copied over regardless their real meaning.
     */
    for (j = 0; j < 8 /* eight c-strings */; ++j) {
        prec += strlen(prec) + 1;
    }

    /* put whole record */
    sxm_tstream_putb(stream, rec, (size_t)(prec - rec));
    /* round up */
    sxm_tstream_alignup(stream, sizeof(uint));
}

/***************************************************************************//**
 * This function is called to perform the update.  The logic sequence can
 * be described as follows:
 *
 *    Process Update Logic
 *
 *    - get region number from update file
 *    - load region from database; initialise record pointers
 *    - process update from file; update record pointers
 *    - start transaction
 *    - allocate blocks for new region
 *    - write new region records
 *    - delete old blocks
 *    - secure transaction
 *    - update file pointers for next region
 *
 *  \param[in] service  service instance
 *  \param[in] upd      updater instance
 *  \param[in] pkt      SXMBitBuff address
 *  \param[in] nver     new database version
 *
 *  \retval             SXM_E_OK     Success
 *  \retval             SXM_E_ERROR  Database was not changed due to 
 *                                   parsing error in data file
 *  \note this function runs in RFD thread context
 *
 ************************************************************************/
static int parking_process_update(ParkingService *service, ParkingUpdate *upd,
                                  SXMBitBuff *pkt, int nver)
{
    uint i, j, psize, psuid;
    uint rcount;
    uint omax;
    ParkingUpdateTypes ut;
    int section;
    int loopc = 0;
    char ptemp[128];
    int rc = SXM_E_ERROR;
    byte is_db_updated = 0;
    int isUpdated = SXM_E_ERROR;
    SXMGridUpdate * const gupd = &upd->grid;

    ENTER_UPD("");

    // set RFD state applying
    service->grid.rfd.state = SXM_GRID_RFD_STATE_APPLYING;

    NEXT(rcount, 11);
    DEBUG_UPD("Processing %d regions", rcount);

    while (rcount-- > 0) {
        int version, loc;
        ushort region;
        int subr = 0, div = 0, subd = 0;
        uint bix, zip;
        fix lon = 0,lat = 0,d1;
        ushort attempt = 0;

        NEXT_TO(ushort, region, 11);
        NEXT(psize, 4) + 1;
        DEBUG_UPD("starting update for region %d. (psize+1)=%d", region, psize);

        rc = sxm_grid_load_region(&service->grid, &upd->grid.base, region, TRUE);
        if (rc != SXM_E_OK) {
            if (rc == SXM_E_INVAL) {
                ERROR_UPD("Unsupported Region %u, ignoring this record", region);
                continue;
            }
            ERROR_UPD("Region %u loading failed", region);
            break;
        }

        // Clean out our new & delete flags
        memset(gupd->nw, 0, gupd->map_size);
        memset(gupd->del, 0, gupd->map_size);

        section = 0;
        omax = gupd->base.dbentry[0];
        if (gupd->stream) {
            rc = sxm_tstream_clean(gupd->stream);
        }
        else {
            rc = sxm_tstream_create(gupd->dbfile, &gupd->stream);
        }
        if (rc != SXM_E_OK) {
            ERROR_UPD("Failed to init tfile stream");
            break;
        }
        memset(&gupd->index_area[0], 0xFF, sizeof(*gupd->index_area) * gupd->region_size);
        gupd->maxs = 0;

        while ((pkt->err == 0) &&
               (sxm_tstream_state(gupd->stream) == SXM_TFILE_STREAM_OK) &&
               (NEXT_TO(ParkingUpdateTypes, ut, 2)) != PARKING_UTYPE_EOL)
        {
            ushort ustmp;
            char term = ' ';

            //  psuid and version are always present
            NEXT(psuid, psize);
            NEXT_TO(int, version, 10);

            // ignore station IDs that are out of range
            if ((psuid < PARKING_MIN_STATION_ID) || (psuid > PARKING_MAX_STATION_ID)) {
                parking_skip_station(pkt, ut);
                ERROR_UPD("Unsupported Station ID: %u, region %u", psuid, region);
                continue;
            }

            if (ut == PARKING_UTYPE_DELETE) {
                BITS(gupd->del, psuid);
                DEBUG_UPD("Deleting id %d", psuid);
                parking_skip_station(pkt, ut);
                continue;
            }

            BITS(gupd->nw, psuid);
            gupd->index_area[psuid] = (uint)sxm_tstream_tell(gupd->stream);

            DEBUG_UPD("Update id %d, type %d", psuid, ut);
            if (psuid > gupd->maxs) {
                gupd->maxs = psuid;
            }

            // Clear out our working location
            memset(&upd->fs, 0, sizeof(upd->fs));
            memset(&upd->location, 0, sizeof(upd->location));

            //  load any existing values into an SXMLocationStation entry if modify;
            if ((ut == PARKING_UTYPE_MODIFY) && (psuid <= gupd->base.dbentry[0]) &&
                (gupd->base.dbentry[psuid] != SXM_DB_INVALID_ENTRY) )
            {
                SXMByteBuff obb;
                if (sxm_bytebuff_init(&obb, gupd->base.base, (int)gupd->base.size, (int)gupd->base.dbentry[psuid]) == NULL)
                {
                    DEBUG_UPD("Failed to init buffer");
                    break; // stop doing the update
                }

                DEBUG_UPD("Loading old values");

                upd->location.lon = (fix)sxm_bytebuff_g4(&obb);
                upd->location.lat = (fix)sxm_bytebuff_g4(&obb);
                upd->location.dlon = (fix)sxm_bytebuff_g4(&obb);
                upd->location.dlat = (fix)sxm_bytebuff_g4(&obb);

                upd->fs.version = sxm_bytebuff_g2(&obb);
                upd->fs.height = sxm_bytebuff_g2(&obb);
                upd->fs.height_sec = sxm_bytebuff_g2(&obb);
                upd->fs.capacity = sxm_bytebuff_g2(&obb);
                ustmp = sxm_bytebuff_g2(&obb);
                upd->fs.price_hour = (byte)(ustmp / 10);
                upd->fs.price_day = (byte)(ustmp % 10);
                upd->fs.amen = (uint)(sxm_bytebuff_g2(&obb) << 16);
                upd->fs.amen |= (uint)sxm_bytebuff_g2(&obb);
                sxm_bytebuff_blob(&obb, &upd->fs.phone[0], SXM_GRID_PHONE_BYTELEN);
                sxm_bytebuff_string(&obb, &upd->fs.name[0], sizeof(upd->fs.name) - 1);
                sxm_bytebuff_string(&obb, &upd->fs.address[0], sizeof(upd->fs.address) - 1);
                sxm_bytebuff_string(&obb, &upd->fs.address_sec[0], sizeof(upd->fs.address_sec) - 1);
                sxm_bytebuff_string(&obb, &upd->fs.city[0], sizeof(upd->fs.city) - 1);
                sxm_bytebuff_string(&obb, &upd->fs.state[0], sizeof(upd->fs.state) - 1);
                sxm_bytebuff_string(&obb, &upd->fs.zip[0], sizeof(upd->fs.zip) - 1);

                //  opening times
                sxm_bytebuff_string(&obb, &ptemp[0], sizeof(ptemp) - 1);
                for (i = 0, j= 0; i < 7; ++i) {
                    char *t = upd->fs.days[i].times;
                    while (ptemp[j]) {
                        if (ptemp[j] == ' ') {
                            ++j;
                            break;
                        } 
                        *t++ = ptemp[j++];
                    }
                }
                sxm_bytebuff_string(&obb, &upd->fs.comment[0], sizeof(upd->fs.comment) - 1);
                DEBUG_UPD("Loaded old values");
            }

            //  now process all the fields in the update record
            upd->fs.version = (ushort)version;

            DEBUG_UPD("New version %d", version);
            if (FLAG()) {
                //  update location
                NEXT_TO(int, loc, 2);
                if (loc == 3) {
                    NEXT_TO(int, subr, 8);
                }
                if ((loc == 2) || (loc == 3)) {
                    NEXT_TO(int, div, 8);
                }
                if (loc != 0) {
                    NEXT_TO(int, subd, 12);
                }
                //  convert to longitude and latitude
                if (loc != 0) {
                    section = 1;
                }

                //  only apply lon/lat changes once we have a valid location
                if (section == 1) {
                    lon = sxm_fix_int(-168 + 2 * (int)SXM_GRID_REGION_LON_UNPACK(region));
                    lat = sxm_fix_int(12 + 2 * SXM_GRID_REGION_LAT_UNPACK(region));

                    d1 = sxm_fix_int(1) >> 3;
                    lat += (subr & 15)*d1;
                    lon += (subr >> 4)*d1;
                    d1 >>= 4;
                    lat += (div & 15)*d1;
                    lon += (div >> 4)*d1;
                    d1 >>= 6;
                    lat += (subd & 63)*d1;
                    lon += (subd >> 6)*d1;
                }

                upd->location.lon = lon;
                upd->location.lat = lat;
                DEBUG_UPD("Modified location %d %d", lon, lat);
            }

            if (FLAG()) {
                signed char dlat, dlon;
                //  update secondary location

                NEXT_TO(signed char, dlat, 8);
                NEXT_TO(signed char, dlon, 8);

                upd->location.dlon = 4 * dlon; //4: convert 1/8192 to fix
                upd->location.dlat = 4 * dlat;

                DEBUG_UPD("Modified location offset %d %d",
                          upd->location.dlon, upd->location.dlat);
            }

            if (FLAG()) {
                BAUDOT(upd->fs.name, BAUDOT_MODE_START_WITH_LETTERS,
                    54, sizeof(upd->fs.name));
            }
            if (FLAG()) {
                BAUDOT(upd->fs.address, BAUDOT_MODE_START_WITH_SYMBOLS,
                    54, sizeof(upd->fs.address));
            }
            if (FLAG()) {
                BAUDOT(upd->fs.address_sec, BAUDOT_MODE_START_WITH_SYMBOLS,
                54, sizeof(upd->fs.address_sec));
            }

            DEBUG_UPD("%s %s/%s", upd->fs.name, upd->fs.address, upd->fs.address_sec);

            if  (FLAG())
                BAUDOT(upd->fs.city, BAUDOT_MODE_START_WITH_LETTERS,
                54, sizeof(upd->fs.city));

            if (FLAG()) {
                NEXT(bix, 7);
                strncpy(&upd->fs.state[0], sxm_states_abbr((int)bix), sizeof(upd->fs.state) - 1);
                upd->fs.state[sizeof(upd->fs.state) - 1] = '\0';
            }
            if (FLAG()) {
                if (!FLAG()) {
                    NEXT(zip, 17);
                    sprintf(&upd->fs.zip[0], "%05d", zip);
                }
                else {
                    BAUDOT(upd->fs.zip, BAUDOT_MODE_START_WITH_LETTERS,
                        15, sizeof(upd->fs.zip));
                }
            }
            DEBUG_UPD("%s, %s, %s", upd->fs.city, upd->fs.state, upd->fs.zip);

            if (FLAG()) {
                ushort area, exch, num;
                NEXT_TO(ushort, area, 10);
                NEXT_TO(ushort, exch, 10);
                NEXT_TO(ushort, num, 14);
                sxm_grid_encode_phone((byte*)&upd->fs.phone[0], area, exch, num);
            }
            DEBUG_UPD("phone=%x%x%x%x%x",
                upd->fs.phone[1], upd->fs.phone[1], upd->fs.phone[2],
                upd->fs.phone[3], upd->fs.phone[4]);

            if (FLAG()) {
                NEXT_TO(ushort, upd->fs.height, 7);
                upd->fs.height = (ushort)(upd->fs.height + 72);
            }
            if (FLAG()) {
                NEXT_TO(ushort, upd->fs.height_sec, 7);
                upd->fs.height_sec = (ushort)(upd->fs.height_sec + 72);
            }
            if (FLAG()) {
                NEXT_TO(ushort, upd->fs.capacity, 16);
            }

            DEBUG_UPD("height=%d height_sec=%d capacity=%d",
                upd->fs.height, upd->fs.height_sec, upd->fs.capacity);

            if (FLAG()) {
                //  hours of operation
                for (i = 0; i < 7; i++)
                {
                    uint code, o1, c1;

                    NEXT(code, 2);
                    if (code == 0) {
                        strcpy(upd->fs.days[i].times, "C");
                    } 
                    else if (code == 1) {
                        strcpy(upd->fs.days[i].times, "O");
                    }
                    else if (code == 2) {
                        NEXT(o1, 7);
                        NEXT(c1, 7);
                        snprintf(upd->fs.days[i].times, sizeof(upd->fs.days[i].times),
                                 "%02d:%02d-%02d:%02d",
                                 o1 >> 2, (o1 & 3) * 15, c1 >> 2, (c1 & 3) * 15);
                    }
                    DEBUG_UPD("times[%d] (Code %d): %s", i, code, upd->fs.days[i].times);
                }
            }

            if (FLAG()) {
                NEXT_TO(byte, upd->fs.price_hour, 4);
                NEXT_TO(byte, upd->fs.price_day, 4);
            }
            DEBUG_UPD("price-ranges: %d/hr %d/day", upd->fs.price_hour, upd->fs.price_day);

            if (FLAG()) {
                BAUDOT(upd->fs.comment, BAUDOT_MODE_START_WITH_SYMBOLS,
                    350, sizeof(upd->fs.comment));
            }
            DEBUG_UPD("comment %s", upd->fs.comment);

            if (FLAG()) {
                uint amen;
                NEXT(amen, 16) << 16;
                upd->fs.amen = (upd->fs.amen & 0x0000ffff) | amen;
            }

            if (FLAG()) {
                uint amen;
                NEXT(amen, 16);
                upd->fs.amen = (upd->fs.amen & 0xffff0000) | amen;
            }
            DEBUG_UPD("amen <%08x>", upd->fs.amen);

            if (FLAG()) {    // F+EXT_SIZE
                uint esize;
                NEXT(esize, 8);
                SKIP(esize * SXM_ARCH_BITS_IN_BYTE);
            }

            sxm_tstream_put(gupd->stream, upd->location.lon);
            sxm_tstream_put(gupd->stream, upd->location.lat);
            sxm_tstream_put(gupd->stream, upd->location.dlon);
            sxm_tstream_put(gupd->stream, upd->location.dlat);
            sxm_tstream_put(gupd->stream, upd->fs.version);
            sxm_tstream_put(gupd->stream, upd->fs.height);
            sxm_tstream_put(gupd->stream, upd->fs.height_sec);
            sxm_tstream_put(gupd->stream, upd->fs.capacity);
            ustmp = (ushort)(upd->fs.price_hour * 10 + upd->fs.price_day);
            sxm_tstream_put(gupd->stream, ustmp);
            ustmp = (ushort)(upd->fs.amen >> 16);
            sxm_tstream_put(gupd->stream, ustmp);
            ustmp = (ushort)(upd->fs.amen & 0xFFFF);
            sxm_tstream_put(gupd->stream, ustmp);
            sxm_tstream_putb(gupd->stream, &upd->fs.phone[0], SXM_GRID_PHONE_BYTELEN);
            sxm_tstream_puts(gupd->stream, &upd->fs.name[0]);
            sxm_tstream_puts(gupd->stream, &upd->fs.address[0]);
            sxm_tstream_puts(gupd->stream, &upd->fs.address_sec[0]);
            sxm_tstream_puts(gupd->stream, &upd->fs.city[0]);
            sxm_tstream_puts(gupd->stream, &upd->fs.state[0]);
            sxm_tstream_puts(gupd->stream, &upd->fs.zip[0]);

            for (i = 0; i < ARRAY_SIZE(upd->fs.days); ++i) {
                if (i > 0) {
                    sxm_tstream_put(gupd->stream, term);
                }
                sxm_tstream_putb(gupd->stream,
                                 &upd->fs.days[i].times[0],
                                 (uint)strlen(&upd->fs.days[i].times[0]));
            }
            term = '\0';
            sxm_tstream_put(gupd->stream, term);
            sxm_tstream_puts(gupd->stream, &upd->fs.comment[0]);
            sxm_tstream_alignup(gupd->stream, sizeof(uint));

            DEBUG_UPD("Updated id %d", psuid);
        }

        if (sxm_tstream_state(gupd->stream) != SXM_TFILE_STREAM_OK) {
            PLOG_UPD("Output stream error detected. Aborting");
            break;
        }

        if (pkt->err) {
            PLOG_UPD("Parsing error in data file. Aborting");
            break;
        }

        //  find largest non-deleted old value
        if (gupd->maxs < omax) {
            while ((omax > 0) && BITP(gupd->del, omax)) {
                --omax;
            }
        }

        if (loopc == 0) {
            sxm_tfile_start(gupd->dbfile);
        }

        while (attempt < 2) 
        {
            if (attempt == 0)
                sxm_grid_merge_data(&service->grid, gupd, omax, region);
            else 
                sxm_grid_stream_commit(&service->grid, gupd, region);
            attempt++;

            if (sxm_tstream_state(gupd->stream) != SXM_TFILE_STREAM_OK) {
                if (attempt == 1 && loopc > 0) {
                    sxm_tfile_commit(gupd->dbfile);
                    is_db_updated = 1;
                    loopc = 0;
                }
                else if (attempt == 2) {
                    DEBUG_UPD("Region buffer is not ok after merge, aborting");
                    if (sxm_tfile_transaction_state(gupd->dbfile) == TRUE) {
                       sxm_tfile_cancel(gupd->dbfile);
                       DEBUG_UPD("Cancel current transaction");
                    }
                }
            }
            else {
                if (++loopc == 16) {
                    sxm_tfile_commit(gupd->dbfile);
                    is_db_updated = 1;
                    loopc = 0;
                }
                break;
            }
        }
        sxe_free(gupd->base.dbentry);
        gupd->base.dbentry = NULL;
    }

    if (gupd->base.dbentry) {
        sxe_free(gupd->base.dbentry);
        gupd->base.dbentry = NULL;
    }

    if (pkt->err) {
        ERROR_UPD("Parsing error. Aborting");
    }
    else {
        DEBUG_UPD("Completed all regions");
        sxm_tfile_update_version(gupd->dbfile, (uint)nver);
        sxm_tfile_commit(gupd->dbfile);
        is_db_updated = 1;
        service->grid.db_version = nver;
    }

    if (is_db_updated) {
        isUpdated = SXM_E_OK;
    }
    /* In case there was a file commit before, sxm_tfile_cancel will return
     * SXM_E_STATE but we do not care. The call is needed to only cancel the
     * transaction in case some parsing error occurred before any commit
     * happened
     */
    sxm_tfile_cancel(gupd->dbfile);

    sxm_parking_database_restart(service);
    sxm_grid_reload_state(&service->grid);

    DEBUG_UPD("version = %d", isUpdated, nver);

    LEAVE_UPD("%s", sxm_sdk_format_result(isUpdated));
    return isUpdated;
}

/***************************************************************************//**
 * This function is invoked by the RFD thread to start the update.
 *
 * Once the indexes and records are written to the TFile, the blocks holding
 * the stale records are released.
 *
 * \param[in] rfd   SXMRfdMetadata address
 *
 * \retval SXM_E_OK     update successfully completed
 * \retval SXM_E_NOMEM  heap memory allocation failed
 * \retval SXM_E_PIPE   failed to open file
 * \retval SXM_E_BAD_DB database schema mismatch
 * \retval SXM_E_STATE  the callback has been called in wrong state
 *
 * \note this function runs in RFD thread context
 *
 ******************************************************************************/
int sxm_parking_update_start(SXMRfdMetadata *rfd)
{
    int rc;
    ParkingService *const service = gService;
    ParkingUpdate *upd;
    int isUpdated = SXM_E_ERROR;

    ENTER_UPD("(rfd->name: %s)", rfd->name);
    DEBUG_UPD("File %s: size %d. Compression mode %d", rfd->name, rfd->size, rfd->comp);

    if (service->upd) {
        ERROR_UPD("Update in progress");
        return SXM_E_STATE;
    }

    service->upd = (ParkingUpdate *)sxe_calloc(1, sizeof(ParkingUpdate));
    if (!service->upd) {
        PLOG_UPD("ParkingUpdate allocation failed");
        return SXM_E_NOMEM;
    }
    upd = service->upd;

    rc = sxm_grid_database_open(&service->grid, &upd->grid.dbfile, PARKING_SERVICE_NAME,
                                PARKING_SERVICE_DB_NAME,
                                "r+b", PARKING_SERVICE_NAME, PARKING_TFILE_SCHEMA_VERSION);
    if (rc != SXM_E_OK) {
        PLOG_UPD("Unable to open DB for update (rc=%d)", rc);
        rc = SXM_E_PIPE;
    }
    else {
        rc = sxm_grid_update_init(&service->grid, &upd->grid, PARKING_MAX_STATIONS, rfd);
        if (rc == SXM_E_OK) {
            upd->grid.merge_record = &parking_merge1;
            isUpdated = parking_process_update(service, upd, upd->grid.gzb, 
                atoi(rfd->name+1) % 1000);
            //cleanup
            sxm_grid_update_uninit(&service->grid, &upd->grid);
        }
        sxm_tfile_close(upd->grid.dbfile);
    }
    sxe_free(service->upd);
    service->upd = NULL;

    if (isUpdated == SXM_E_OK) {
        sxm_lli_rfd_set_db_updated(service->grid.lli);
    }

    // set RFD back to idle state
    service->grid.rfd.state = SXM_GRID_RFD_STATE_IDLE;

    LEAVE_UPD("%s", sxm_sdk_format_result(rc));
    return rc;
}

/***************************************************************************//**
 * This function is invoked by the sxm_parking_start() function.  Global variables
 * initialized and the service is registered to use the states database.
 *
 * \param[in] service   service structure address
 *
 ******************************************************************************/
void sxm_parking_update_init(ParkingService *service)
{
    gService = service;
    sxm_states_register();
    CALLR_UPD("");
}

/***************************************************************************//**
 * This function is invoked by the sxm_parking_stop() function.  Global variables
 * initialized and the service is unregistered to use the states database.
 *
 * \param[in] service service instance
 *
 ******************************************************************************/
void sxm_parking_update_uninit(ParkingService *service)
{
    if (service && service->upd) {
        sxm_grid_update_uninit(&service->grid, &service->upd->grid);
        sxe_free(service->upd);
        service->upd = NULL;
    }
    gService = NULL;
    sxm_states_unregister();

    CALLR_UPD("");
}
