/************************************************************************
 *                                                                      *
 *            SXM_PARKING_PROTOCOL.C                                    *
 *            ======================                                    *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 *           Implementation of the bitstream decoder for Parking prices *
 *                                                                      *
 ************************************************************************/
/***************************************************************************//**
 *
 * \file sxm_parking_protocol.c
 * \author Leslie French
 * \date 8/20/2013
 *
 * Implementation of the bitstream decoder for Parking prices
 *
 ******************************************************************************/

#define DEBUG_TAG "parking"

#include <stddef.h>
#include "sxm_parking_internal.h"

#ifdef SXM_DEBUG_PKT_METRICS
SXESDK_API uint parking_packets = 0;
SXESDK_API uint parking_bytes = 0;
#endif

/************************************************************************
 *                                                                      *
 *            Private prototypes                                        *
 *            ======================                                    *
 *                                                                      *
 ************************************************************************/

static int parking_read_location_info(ParkingService *,
                                      SXMParkingLocation *,
                                      SXMGridBaseLine *,
                                      ushort);

static int parking_locations_region_compare_callback(const ParkingFavLocation *p1,
                                                     const ParkingFavLocation *p2);

static int parking_index_compare_callback(const ParkingFavLocation *p1,
                                          const ParkingFavLocation *p2);

static void parking_check_timestamp(ParkingService *service, const uint time_now);

static void parking_replay(ParkingService *service);

static void parking_regions_reload(ParkingService *service);

static int parking_extract(ParkingService *service, SXMParkingLocation *ret,
                           short res, uint rec_id, int hix);

static int parking_poi_extract(ParkingService *service, SXMParkingLocation *ret,
                               uint id, ushort flags, const ParkingRecord *rec);

/***************************************************************************//**
 * This function checks and saves access unit (AU) for Parking Fullness data.
 *
 * \note It validates AU by computing CRC and saving with timestamp.
 * \note The AU may originate from the broadcast or from the CFile AU cache
 * during replay.
 *
 * \param[in] service service instance
 * \param[in] hix index of loaded baselines for region containing the AU
 * \param[in] pFp a pointer to the common packet data structure
 * \param[in] aucnt the sequence number of this access unit in the region
 *
 * \retval SXM_E_OK if AU CRC check passed, or AU was already saved
 *                  (the service must process the packet)
 * \retval SXM_E_NOENT the AU is duplicate and may have nothing for the service
 *                     to process.
 * \retval SXM_E_INVAL if AU CRC check failed (no further processing required)
 *
 ******************************************************************************/
static int parking_aucheck(ParkingService *service, uint hix, SXMPacket *pFp, uint aucnt)
{
    // Initialize return value to 'OK'
    int rc = SXM_E_OK;

    // check for and set replay mode state
    if (!((aucnt < ARRAY_SIZE(service->fullness_au[hix].safe)) &&
        (service->fullness_au[hix].safe[aucnt].pkt == pFp->packet))) {
        SXMGridSavedAU *au =
            (aucnt < ARRAY_SIZE(service->fullness_au[hix].safe)) ? &service->fullness_au[hix].safe[aucnt] : NULL;

        // Check that AU at this position has the same len and CRC in
        // order to detect duplicate. Assuming that the CRC kept by the AU
        // memory should be the same as the CRC kept by C-file. The C-file shall
        // have valid CRC for AU's content as well as passed AU at the end
        // so there is no need to recalculate it before check since it doesn't
        // matter if new AU will have the same CRC after calculation or not.
        // FYI: If not - the packet will be thrown out based on CRC check later
        if (au && (au->pl == pFp->pl) && ((uint)x4(pFp->packet, pFp->pl - CRC_BYTELEN) == au->crc)) {
            // Duplicate detected, but still need to update timestamp in order
            // to process data age out gracefully.
            au->ts = sxm_sxe_time_now();
            rc = SXM_E_NOENT;
        }
        // check CRC for broadcast AUs, (not replayed AUs, these have
        // already been checked)
        else if (!sxm_crc32_check(pFp->packet, pFp->pl, &pFp->crc)) {
            ERROR_PKT("CRC failed, aucnt %u", aucnt);
            rc = SXM_E_INVAL;
        }
        // process good AUs with sequence number within range received
        // from the broadcast and only process AUs not already saved
        else if (au) {
            if (!au->pkt) {
                // pick one off the heap
                const int ix =
                    sxm_cfile_map_alloc(&service->fullness_data->g, PARKING_FULLNESS_SAVE_HEAP, 1);
                if (ix >= 0) {
                    // populate AU metadata
                    au->pkt = (byte *)((SXMSector*)&service->fullness_data->heap.heap[0] +
                        (uint)ix * SXM_CSECTORCNT(service->fullness_data->heap.heap[0]));
                    au->pktix = (uint)ix;
                }
                else {
                    PLOG_PKT("AU heap is full");
                    rc = SXM_E_NOMEM;
                }
            }

            // copy the AU to the safe
            if (au->pkt) {
                au->crc = pFp->crc;
                au->pl = pFp->pl;
                au->ts = sxm_sxe_time_now();
                memcpy(au->pkt, pFp->packet, (size_t)pFp->pl);
            }
        }
    }

    return rc;
}

/************************************************************************
 *                                                                      *
 *            Carousel Handlers                                         *
 *            =================                                         *
 *                                                                      *
 ************************************************************************/
/***************************************************************************//**
 *
 * Processes the Parking Availability Carousel.
 *
 * Parses Parking Availability AU packets for each region being monitored, and updates
 * the price data in the service objects.
 *
 * \param[in] service service instance
 * \param[in] pkt pointer to the SXMBitBuff containing packet data.
 * \param[in] fp SXMPacket (AU) pointer
 *
 ********************************************************************************/
static void parking_availability(ParkingService *service, SXMBitBuff *pkt, SXMPacket *fp)
{
    ushort region;
    uint atime, psize;
    uint ctsize, autot, aucnt;
    uint psuid = 0;
    int hix;
    ParkingFavLocation *ifav = NULL;
    ParkingRecord *pih = NULL;
    int rc = SXM_E_OK;

    NEXT_TO(ushort, region, 11);

    DEBUG_PKT("region: %d", region);

    // if the region of this AU is not one of the regions we're collecting, ignore it
    hix = sxm_grid_baseix(&service->grid, region);
    if ((hix == SXM_GRID_REGION_INVAL_IDX) && !sxm_grid_is_favorite_region(&service->grid, region)) {
        DEBUG_PKT("Ignore AUs since it doesn't belong to neither GRID nor PPOI");
        return;
    }

    // Stop here if neither regular nor favorite points exists
    if ((service->grid.sp == 0) && (service->fav_data->count == 0)) {
        DEBUG_PKT("No low-level processing since no one is interested in it");
        return;
    }

    NEXT(atime, 11);
    NEXT(psize, 4);

    // one or more AUs for this region if 'ctsize' is present
    if (FLAG()) {
        NEXT(ctsize, 4) + 1;
        NEXT(autot, ctsize) + 1;
        NEXT(aucnt, ctsize);
    }
    else {
        autot = 1;
        aucnt = 0;  // only one AU for this region
    }

    if (hix != -1) {
        LOCK(service->grid.service_mutex);
        // cache the AU in the CFile (if room) and if it passes CRC check
        rc = parking_aucheck(service, (uint)hix, fp, aucnt);
        UNLOCK(service->grid.service_mutex);
        if (rc == SXM_E_INVAL) {
            // Nothing to process further
            return;
        }
        // Check the availability time
        if (atime != service->avail_au[hix].new_group_au_flag)
        {
            // erase any accumulated working copy information
            memset(&service->avail_au[hix], 0, sizeof(SXMGridDataAU));

            service->avail_au[hix].new_group_au_flag = atime;
            DEBUG_PKT("New Availability time %d found.", atime);
        }

        if (aucnt < SXM_GRID_AUS_PER_REGION) {
            BITSET(service->avail_au[hix].au_cnt, aucnt);
        }
        // Set the total value
        service->avail_au[hix].tot_au = SXM_GRID_COMPLETE_AU_MASK(autot);
    }
    //  check this AU
    else if (sxm_crc32_check(fp->packet, fp->pl, &fp->crc) == 0) {
         return;
    }

    if (rc == SXM_E_OK) {
        //  now process the AU Data
        while (pkt->b < pkt->e) {
            uint delta = 0;
            uint status;
            byte full[5];
            uint code;
            ParkingRecord *rec = NULL;
            ushort valid = 0;

            memset(full, 0, sizeof(full));
            if (FLAG() == 1)
                delta = 1;
            else if (FLAG() == 1)
                delta = 2;
            else if (FLAG() == 1)
                delta = 3;
            else if (FLAG() == 1)
                delta = 4;
            else if (FLAG() == 1)
                delta = 5;
            else if (FLAG() == 1)
                delta = 6;
            else
                NEXT(psuid, psize + 1);

            //  watch out for overflow on the last byte!

            if (pkt->err) {
                ERROR_PKT("Parsing error detected");
                break;
            }
            psuid += delta;
            code = SXM_GRID_ID_PACK(region, psuid);

            //  find the heap record for this station, if it is in the scan list
            if (hix != SXM_GRID_REGION_INVAL_IDX) {
                pih = (ParkingRecord *) sxm_grid_get_poi_data(&service->grid, code, hix);
                if (pih) {
                    rec = &service->staged;
                    *rec = *pih;
                }
            }
            ifav = (ParkingFavLocation *)sxm_grid_is_poi_from_favorite_list(&service->grid, code);
            if (ifav && !rec) {
                rec = &service->staged;
                *rec = ifav->fav_station;
            }

            //  still need to parse out the data, even if we are ignoring the station

            NEXT(status, 3);

            if (FLAG()) {
                valid = SXM_PARKING_VALID_FULL0;
                NEXT_TO(byte, full[0], 4);
                if (FLAG()) {
                    valid = SXM_PARKING_VALID_FULL;        //  all valid
                    NEXT_TO(byte, full[1], 4);
                    NEXT_TO(byte, full[2], 4);
                    NEXT_TO(byte, full[3], 4);
                    NEXT_TO(byte, full[4], 4);
                }
            }

            //  if we have a valid station entry, copy the data
            if (rec) {
                rec->status = (byte)status;
                rec->avail_time = atime;

                //  clear the FULL flags
                rec->valid &= (ushort)~SXM_PARKING_VALID_FULL;
                rec->valid |= valid | SXM_PARKING_VALID_STATUS;
                memcpy(rec->full, full, sizeof(rec->full));

                if (pih && (memcmp(rec, pih, sizeof(ParkingRecord)) != 0)) {
                    // if record has changed, update it and set the dirty flag
                    *pih = *rec;
                    service->grid.stations.dirty_flag = TRUE;
                }
                if (ifav && (memcmp(rec, &ifav->fav_station, sizeof(ParkingRecord)) != 0)) {
                    // if record has changed, update it and set the dirty flag
                    LOCK(service->grid.service_mutex);
                    ifav->fav_station = *rec;
                    UNLOCK(service->grid.service_mutex);
                    service->grid.fav.dirty_flag = TRUE;
                    service->grid.fav.user_flags |= ifav->ppoi_spec.type_flags;
                    BITSET(service->grid.sections_update, PARKING_PPOI);
                }
            }

            if (FLAG()) { // F+EXT_SIZE+EXT
                uint skip;
                NEXT(skip, 7);
                SKIP(skip + 1);
            }
        }
    }

    // Set au_flag to TRUE when we processed all the AUs for the region 
    if (hix != SXM_GRID_REGION_INVAL_IDX)
    {
        if (service->avail_au[hix].au_cnt == service->avail_au[hix].tot_au)
        {
            service->grid.stations.au_flag = TRUE;
        }
    }

    sxm_grid_notify_new_data(&service->grid);
}

/***************************************************************************//**
 *
 * Processes the Parking Price Carousel.
 *
 * Parses Parking Price AU packets for each region being monitored, and updates
 * the price data in the service objects.
 *
 * \param[in] service service instance
 * \param[in] pkt SXMBitBuff pointer
 * \param[in] fp SXMPacket (AU) pointer
 *
 ********************************************************************************/
static void parking_prices(ParkingService *service, SXMBitBuff *pkt, SXMPacket *fp)
{
    ushort region;
    uint ctsize, autot, aucnt, psize, numsize, hoursize, daysize, hres, dres, price_ver;
    uint psuid = 0;
    int hix;
    ParkingRecord *pih = NULL;
    ParkingFavLocation *ifav = NULL;
    static ushort res[] = {5, 10, 25, 100};
    int rc = SXM_E_OK;

    NEXT_TO(ushort, region, 11);

    DEBUG_PKT("region: %d", region);

    // if the region of this AU is not one of the regions we're collecting ignore it
    hix = sxm_grid_baseix(&service->grid, region);
    if ((hix == SXM_GRID_REGION_INVAL_IDX) && !sxm_grid_is_favorite_region(&service->grid, region)) {
        DEBUG_PKT("Ignore AUs since it doesn't belong to neither GRID nor PPOI");
        return;
    }

    NEXT(price_ver, 8); // version
    NEXT(psize, 4);
    NEXT(numsize, 4);
    NEXT(hoursize, 4);
    NEXT(hres, 2);
    NEXT(daysize, 4);
    NEXT(dres, 2);

    // one or more AUs for this region if 'ctsize' is present
    if (FLAG()) {
        NEXT(ctsize, 4) + 1;
        NEXT(autot, ctsize) + 1;
        NEXT(aucnt, ctsize);
    }
    else {
        autot = 1; 
        aucnt = 0;    // only one AU for this region
    }

    // cache the AU in the CFile (if room) and if it passes CRC check
    if (hix != SXM_GRID_REGION_INVAL_IDX) {
        LOCK(service->grid.service_mutex);
        // cache the AU in the CFile (if room) and if it passes CRC check
        rc = sxm_grid_aucheck(&service->grid, (uint)hix, fp, aucnt);
        if ((rc == SXM_E_OK) || (rc == SXM_E_NOENT)) {
            // Either all content or timestamp only has been updated
            BITSET(service->grid.sections_update, PARKING_BASE);
            if (rc == SXM_E_OK) {
                BITSET(service->grid.sections_update, PARKING_HEAP);
            }
        }
        UNLOCK(service->grid.service_mutex);
        if (rc == SXM_E_INVAL) {
            // Nothing to process further
            return;
        }
        // Check the version number
        if (price_ver != service->price_au[hix].new_group_au_flag)
        {
            // erase any accumulated working copy information
            memset(&service->price_au[hix], 0, sizeof(SXMGridDataAU));

            service->price_au[hix].new_group_au_flag = price_ver;
            DEBUG_PKT("New Parking price version %d found.", price_ver);
        }
        if (aucnt < SXM_GRID_AUS_PER_REGION) {
             BITSET(service->price_au[hix].au_cnt, aucnt);
        }
        // Set the total value
        service->price_au[hix].tot_au = SXM_GRID_COMPLETE_AU_MASK(autot);
    }
    else if (sxm_crc32_check(fp->packet, fp->pl, &fp->crc) == 0) {
        ERROR_PKT("CRC check failed");
        return;
    }

    // Stop here if neither regular nor favorite points exists
    if ((service->grid.sp == 0) && (service->fav_data->count == 0)) {
        DEBUG_PKT("No low-level processing since no one is interested in it");
        return;
    }

    if (rc == SXM_E_OK) {
        //  now process the AU Data
        while (pkt->b < pkt->e) {
            uint delta = 0;
            uint total;
            uint code;
            ushort prices[24], ptype, ip;
            ushort valid = 0;
            ParkingRecord *rec = NULL;

            if (FLAG() == 1)
                delta = 1;
            else if (FLAG() == 1)
                delta = 2;
            else if (FLAG() == 1)
                delta = 3;
            else if (FLAG() == 1)
                delta = 4;
            else if (FLAG() == 1)
                delta = 5;
            else if (FLAG() == 1)
                delta = 6;
            else
                NEXT(psuid, psize + 1);

            //  watch out for overflow on the last byte!

            if (pkt->err) {
                break;
            }
            psuid += delta;
            code = SXM_GRID_ID_PACK(region, psuid);

            //  find the heap record for this station, if it is in the scan list
            if (hix != SXM_GRID_REGION_INVAL_IDX) {
                pih = (ParkingRecord*)sxm_grid_get_poi_data(&service->grid, code, hix);
                if (pih) {
                    rec = &service->staged;        // staged record
                    *rec = *pih;   // copy current record contents
                    rec->total = 0;
                }
            }
            ifav = (ParkingFavLocation*)sxm_grid_is_poi_from_favorite_list(&service->grid, code);
            if (ifav && !rec) {
                rec = &service->staged;
                *rec = ifav->fav_station;
            }

            //  still need to parse out the data, even if we are ignoring the station

            if (FLAG()) {
                NEXT(total, numsize + 1);

                if (rec) {
                    rec->total = (ushort)total;
                    valid |= SXM_PARKING_VALID_NUMTOTAL;
                }
            }

            while ((NEXT_TO(ushort, ptype, 3)) != 0) {
                ip = 0;
                memset(prices, 0, sizeof(prices));
                if (ptype < 4) {
                    NEXT_TO(ushort, prices[ip], hoursize+1);
                    prices[ip] = (ushort)(prices[ip] * res[hres]);
                    ip++;
                    while (FLAG()) {
                        NEXT_TO(ushort, prices[ip], hoursize+1);
                        prices[ip] = (ushort)(prices[ip] * res[hres]);
                        ip++;
                    }

                    if (rec) {
                        if (ptype == 1) {
                            valid |= SXM_PARKING_VALID_WK_HR;
                            memcpy(rec->weekday_hour, prices, sizeof(prices));
                        }
                        else if (ptype == 2) {
                            valid |= SXM_PARKING_VALID_SA_HR;
                            memcpy(rec->saturday_hour, prices, sizeof(prices));
                        }
                        else if (ptype == 3) {
                            valid |= SXM_PARKING_VALID_SU_HR;
                            memcpy(rec->sunday_hour, prices, sizeof(prices));
                        }
                    }
                }
                else if (ptype == 4) {
                    if (FLAG()) {
                        NEXT_TO(ushort, prices[0], daysize+1); 
                        if (rec) {
                            valid |= SXM_PARKING_VALID_WK_DY;
                            rec->weekday_day = (ushort)(prices[0] * res[dres]);
                        }
                    }
                    if (FLAG()) {
                        NEXT_TO(ushort, prices[0], daysize+1);
                        if (rec) {
                            valid |= SXM_PARKING_VALID_SA_DY;
                            rec->saturday_day = (ushort)(prices[0] * res[dres]);
                        }
                    }
                    if (FLAG()) {
                        NEXT_TO(ushort, prices[0], daysize+1);
                        if (rec) {
                            valid |= SXM_PARKING_VALID_SU_DY;
                            rec->sunday_day = (ushort)(prices[0] * res[dres]);
                        }
                    }
                }
                else {
                    NEXT_TO(ushort, prices[0], daysize+1);
                    if (rec) {
                        if (ptype == 5) {
                            valid |= SXM_PARKING_VALID_EBIRD;
                            rec->ebird = (ushort)(prices[0] * res[dres]);
                        }
                        else if (ptype == 6) {
                            valid |= SXM_PARKING_VALID_EVENT;
                            rec->event = (ushort)(prices[0] * res[dres]);
                        }
                        else if (ptype == 7) {
                            valid |= SXM_PARKING_VALID_OTHER;
                            rec->other = (ushort)(prices[0] * res[dres]);
                        }
                    }
                }
            }

            if (FLAG()) {
                uint skip;
                NEXT(skip, 7);
                SKIP(skip + 1);
            }

            if (rec) {
                //  keep the FULL flags
                rec->valid &= (SXM_PARKING_VALID_FULL | SXM_PARKING_VALID_STATUS);
                //  add the new ones
                rec->valid |= valid;


                if (pih && (memcmp(rec, pih, sizeof(ParkingRecord)) != 0)) {
                   // if record has changed, update it and set the dirty flag
                    *pih = *rec;  // copy changed record
                    service->grid.stations.dirty_flag = TRUE;
                }
                if (ifav && (memcmp(rec, &ifav->fav_station, sizeof(ParkingRecord)) != 0)) {
                    // if record has changed, update it and set the dirty flag
                    LOCK(service->grid.service_mutex);
                    ifav->fav_station = *rec;
                    UNLOCK(service->grid.service_mutex);
                    service->grid.fav.dirty_flag = TRUE;
                    service->grid.fav.user_flags |= ifav->ppoi_spec.type_flags;
                    BITSET(service->grid.sections_update, PARKING_PPOI);
                }
            }
        }
    }

    // Set au_flag to TRUE when we processed all the AUs for the region 
    if (hix != SXM_GRID_REGION_INVAL_IDX)
    {
        if (service->price_au[hix].au_cnt == service->price_au[hix].tot_au)
        {
            service->grid.stations.au_flag = TRUE;
        }
    }

    sxm_grid_notify_new_data(&service->grid);
}

/********************************************************************//**                                                                    *
 * Qualifies RFD metadata and provides corresponding version if it's ok.
 *
 * \param[in] service service instance
 * \param[in] rfd parsed RFD metadata instance
 * \param[out] old_version extracted old/from version
 * \param[out] upd_version extracted upd/to version
 *
 * \retval SXM_E_OK Update is good and versions provided
 * \retval SXM_E_INVAL Update seems incorrect
 *
 ************************************************************************/
static int parking_qualify_metadata(ParkingService *service, const SXMRfdMetadata *rfd,
                                    uint *old_version, uint *upd_version)
{
    int rc;
    ENTER_PKT("");

    // check to see if we support this update file by looking at the prefix/size
    if ((rfd->name[0] != PARKING_UPDATE_FNAME_PREFIX) ||
        (strlen(rfd->name) != PARKING_UPDATE_FNAME_SIZE)) {
        ERROR_PKT("Invalid update filename %s", rfd->name);
        rc = SXM_E_INVAL;
    }
    else {
        // For PARKING, the updates come in the form Uxxxyxy; we can do some
        // simple arithmetic to get the old & new versions from the name.
        *old_version = (uint)atoi(rfd->name + 1);
        *upd_version = *old_version % 1000;
        *old_version = *old_version / 1000;
        rc = SXM_E_OK;
    }

    LEAVE_PKT("%s", sxm_sdk_format_result(rc));
    return rc;
}

/********************************************************************//**
 * The complete function is called when a complete AU is
 * received. The complete function determines the type of
 * AU for the service and passes it to the appropriate function.
 *
 * \param[in] service service instance
 * \param[in] pkt buffer instance associated with the data
 * \param[in] packet buffer with the raw data 
 * \param[in] pl size of the data available via \p packet
 ************************************************************************/
static void parking_complete(ParkingService *service, SXMBitBuff *pkt, const void *packet, uint pl) {
    SXMPacket fp;
    uint pvn;
    ParkingCarouselIDs cid;

    ENTER_PKT("");

    //  if it's the wrong PVN we can stop now
    NEXT(pvn, 4);
    if (pvn != 1) {
        ERROR_PKT("Wrong pvn %d", pvn);
        LEAVE_PKT("");
        return;
    }

    NEXT_TO(ParkingCarouselIDs, cid, 3);

    // don't check CRC on data until we know we need it
    fp.packet = (byte*)packet;
    fp.pl = pl;

    switch (cid) {
        case PARKING_CARID_AVAL_DATA_0:
        case PARKING_CARID_AVAL_DATA_1: {
            //  only collect if we are ready for it.
            if ((service->grid.state == SXM_GRID_STATE_ALL) ||
                ((service->grid.state == SXM_GRID_STATE_READY) && service->fav_data->count)) {
                    parking_availability(service, pkt, &fp);
            }
        }
        break;
        case PARKING_CARID_PRICE_AND_CAPACITY: {
            if ((service->grid.state == SXM_GRID_STATE_ALL) ||
                ((service->grid.state == SXM_GRID_STATE_READY) && service->fav_data->count)) {
                    parking_prices(service, pkt, &fp);
            }
        }
        break;
        case PARKING_CARID_RFD_DATA: {
            // Shall not pass further if no RFD in progress
            if (service->grid.rfd.state != SXM_GRID_RFD_STATE_COLLECTING) {
                break;
            }
        } /* go further */
        case PARKING_CARID_RFD_METADATA: {
            if (sxm_crc32_check(packet, pl, &fp.crc) == 0) {
                ERROR_PKT("Invalid CRC CARID=%d", (int)cid);
                break;
            }
            if (cid == PARKING_CARID_RFD_DATA) {
                sxm_grid_lli_rfd_add(&service->grid, packet, (int)pl);
            }
            else {
                sxm_grid_rfd_handle_metadata(&service->grid, pkt, &fp);
            }
        }
        break;
        default: {
            ERROR_PKT("Unknown Carousel %d", cid);
        }
        break;
    }

    LEAVE_PKT("");
}

/***************************************************************************//**
 * Dispatch AU for processing.
 * Called by the monitor thread when a new Access Unit (AU) is ready for processing.
 *
 * \param[in] pBuff points to buffer containing the AU
 * \param[in] pl packet (AU) length in bytes
 * \param[in] dsuser service instance
 *
 ********************************************************************************/
void sxm_parking_complete_au(byte *buff, int pl, ptr dsuser) {
    SXMBitBuff pkt;
    ParkingService *service = (ParkingService*)dsuser;

#ifdef SXM_DEBUG_PKT_METRICS
    // Update statistics
    parking_packets++;
    parking_bytes += (uint)pl;
#endif

    sxm_bitbuff_setup(&pkt, buff, (uint)(pl - CRC_BYTELEN));
    parking_complete(service, &pkt, buff, (uint)pl);
}

/********************************************************************//**
 * The function initializes the necessary service information.
 *
 * \param[in] service service instance
 *
 ************************************************************************/
int sxm_parking_protocol_start(ParkingService *service) {
    uint i, j;
    int rc;

    ENTER_PKT("");

    service->fullness_data =
        (ParkingFullnessSave *)sxe_calloc(1, sizeof(ParkingFullnessSave));
    if (!service->fullness_data) {
        ERROR_PKT("ParkingFullnessSave allocation failed");
        LEAVE_PKT("%s", sxm_sdk_format_result(SXM_E_NOMEM));
        return SXM_E_NOMEM;
    }

    sxm_cfile_add_dsection(&service->fullness_data->g,
        PARKING_FULLNESS_SAVE_HEAP, sizeof(SXMGridAU),
        ARRAY_SIZE(service->fullness_data->heap.heap));

    // Initialize the grid part
    rc = sxm_grid_init(&service->grid, (void**)&service->data, sizeof(ParkingSave));
    if ((rc != SXM_E_OK) && (rc != SXM_E_NOENT)) {
        sxe_free(service->fullness_data);
        service->fullness_data = NULL;
        ERROR_PKT("sxm_grid_init failed (rc=%d)", rc);
        LEAVE_PKT("%s", sxm_sdk_format_result(rc));
        return rc;
    }

    service->grid.root = &service->data->g;
    service->grid.heap_sec = PARKING_HEAP;
    service->grid.heap = &(service->data->heap.heap[0].heap);
    for (i = 0; i < ARRAY_SIZE(service->grid.bptr); ++i) {
        service->grid.bptr[i] = &service->data->baseb.baseb[i];
    }
    service->fav_data = &service->data->fav_loc.fav_loc[0];

    // Initialize the PPOI part of the grid
    service->grid.fav.count = &service->fav_data->count;
    service->grid.fav.capacity = ARRAY_SIZE(service->fav_data->locations);
    service->grid.fav.item_size = sizeof(ParkingFavLocation);
    service->grid.fav.index_compare_callback =
        (SXMGridCompareCallaback_t) &parking_index_compare_callback;
    service->grid.fav.region_compare_callback = 
        (SXMGridCompareCallaback_t) &parking_locations_region_compare_callback;
    service->grid.fav.data = &service->fav_data->locations[0];
    service->grid.fav.ppoispec_offset = offsetof(ParkingFavLocation, ppoi_spec);
    service->grid.fav.data_offset = offsetof(ParkingFavLocation, fav_station);
    service->grid.fav.key = &service->key;

    service->grid.check_timestamp = (SXMGridCheckTimeStamp_t)&parking_check_timestamp;
    service->grid.replay = (SXMGridReplay_t)&parking_replay;
    service->grid.reload_state = (SXMGridReloadState_t)parking_regions_reload;
    service->grid.extract = (SXMGridExtract_t) &parking_extract;
    service->grid.poi_extract = (SXMGridPOIExtract_t) &parking_poi_extract;

    service->grid.stations.min_id = PARKING_MIN_STATION_ID;
    service->grid.stations.items = (byte*)&service->stations.items[0];
    service->grid.stations.item_size = sizeof(service->stations.items[0]);
    service->grid.stations.capacity = ARRAY_SIZE(service->stations.items);
    service->grid.stations.ids = &service->stations.ids[0];

    service->grid.rfd.qualify = (SXMGridQualify_t)&parking_qualify_metadata;

    // Now see if we need to do anything to fix up the CFile;
    // check to see if we got the right # of sectors from
    // the CFile; if so, we double-check the CRCS
    if (rc == SXM_E_OK) {
        SXMCFileValidationResult val;
        uint schema;
        int intRc;

        intRc = sxm_cfile_check_format(service->grid.root, PARKING_SERVICE_NAME, &schema);
        if ((intRc == SXM_E_OK) && (PARKING_CFILE_SCHEMA_VERSION == schema)) {
            val = sxm_cfile_validate(service->grid.root);
        }
        else {
            val = SXM_CFILE_INVALID;
        }

        if (val == SXM_CFILE_INVALID) {
            rc = SXM_E_NOENT;
        }
        else if (val == SXM_CFILE_PARTIALY_VALID) {
            // See if we can keep any data

            if (sxm_cfile_check(service->grid.root, PARKING_PPOI, FALSE) == 0) {
                sxm_cfile_clear_section(service->grid.root, PARKING_PPOI);
                BITSET(service->grid.sections_update, PARKING_PPOI);
            }

            if ((sxm_cfile_check(service->grid.root, PARKING_BASE, FALSE) == 0) ||
                (sxm_cfile_check(service->grid.root, PARKING_HEAP, FALSE)) == 0) {
                sxm_cfile_clear_section(service->grid.root, PARKING_BASE);
                sxm_cfile_map_init(service->grid.root, PARKING_HEAP, 0);

                for  (j = 0; j < ARRAY_SIZE(service->grid.bptr); ++j) {
                    memset(service->grid.bptr[j], 0, sizeof(*(service->grid.bptr[j])));
                }
                BITSET(service->grid.sections_update, PARKING_BASE);
            }
        }
        else {
            for (i = 0; i < ARRAY_SIZE(service->grid.bptr); ++i) {
                SXMGridBaseLine * const bptr = service->grid.bptr[i];
                if (sxm_grid_load_region(&service->grid, bptr, bptr->region, FALSE) == SXM_E_OK) {
                    //  convert index values back into pointers
                    for (j = 0; j < ARRAY_SIZE(bptr->safe); ++j) {
                        SXMGridSavedAU * const safe = &bptr->safe[j];
                        if (safe->pl) {
                            safe->pkt = (byte *)&service->data->heap.heap[safe->pktix];
                        }
                    }
                }
                else {
                    memset(bptr, 0, sizeof(*bptr));
                }
            }
        }
    }

    if (rc != SXM_E_OK) {
        uint section;
        memset(service->data, 0, sizeof(*service->data));
        sxm_cfile_set_format(service->grid.root,
                             PARKING_SERVICE_NAME, PARKING_CFILE_SCHEMA_VERSION);
        for (section = PARKING_SECTION_FIRST; section < PARKING_SECTIONS_COUNT; ++section) {
            switch (section) {
                case PARKING_BASE: {
                    sxm_cfile_add_section(service->grid.root, section,
                        CSIZE(SXMGridBaseLine, ARRAY_SIZE(service->data->baseb.baseb)));
                }
                break;
                case PARKING_HEAP: {
                    sxm_cfile_add_dsection(service->grid.root, section,
                        sizeof(SXMGridAU), ARRAY_SIZE(service->data->heap.heap));
                }
                break;
                case PARKING_PPOI: {
                    sxm_cfile_add_section(service->grid.root, section,
                        CSIZE(ParkingFavPersData, ARRAY_SIZE(service->data->fav_loc.fav_loc)));
                }
                break;
            }
        }

        // now mark all CFILE sections for update
        service->grid.sections_update = PARKING_UPDATE_ALL;

        ERROR_PKT("WRONG LENGTH");
    }

    LEAVE_PKT("%s", sxm_sdk_format_result(SXM_E_OK));
    return SXM_E_OK;
}

/********************************************************************//**
 * The function persists identified content as well as frees up the service.
 *
 * \param[in] service service instance
 *
 ************************************************************************/
int sxm_parking_protocol_stop(ParkingService *service) {
    int rc;
    ENTER_PKT("");

    if (service) {
        rc = sxm_grid_cycle_destroy(SXM_2GRID(service), service->data);
        if (service->fullness_data) {
            sxe_free(service->fullness_data);
            service->fullness_data = NULL;
        }
    }
    else  {
        rc = SXM_E_STATE;
    }

    LEAVE_PKT("%s", sxm_sdk_format_result(rc));
    return rc;
}

/************************************************************************
 * Replays all cached data through the service parser to update current
 * collection.
 *
 * \param[in] service service instance
 *
 ************************************************************************/
static void parking_replay(ParkingService *service)
{
    uint ix;
    DEBUG_PKT("reply AUs");

    /* Remove any previous saved pricing data */
    memset(&service->stations.items[0], 0,
        sizeof(service->stations.items[0]) * service->grid.sp);

    //  re-process any saved access units for the different regions
    for (ix = 0; ix < ARRAY_SIZE(service->grid.bptr); ++ix) {
        uint j;
        SXMBitBuff pkt;
        for (j = 0; j < ARRAY_SIZE(service->grid.bptr[ix]->safe); ++j) {
            SXMGridSavedAU * const safe = &(service->grid.bptr[ix]->safe[j]);
            if (safe->pl) {
                sxm_bitbuff_setup(&pkt, safe->pkt, safe->pl);
                parking_complete(service, &pkt, safe->pkt, safe->pl);
            }
        }
        for (j = 0; j < ARRAY_SIZE(service->fullness_au[ix].safe); ++j) {
            SXMGridSavedAU * const safe = &(service->fullness_au[ix].safe[j]);
            if (safe->pl) {
                sxm_bitbuff_setup(&pkt, safe->pkt, safe->pl);
                parking_complete(service, &pkt, safe->pkt, safe->pl);
            }
        }
    }
}

/***************************************************************************//**
 * Reloads the regions required for a given extraction.
 *
 * The parking_regions_reload routine populates the "region" field in the
 * 4 fullness_au structures in the ParkingService with the 4 needed values,
 * see \ref ParkingService. It will retain any previously-loaded data,
 * and saved Access Units for fullness data.
 * The order of "region" fields in the fullness_au[4] should be the same
 * as in the the 4 bptr structures in the GridService.
 *
 * \param[in] service service instance
 * \param[in] grid a pointer to an initialized grid structure
 *
 ********************************************************************************/
static void parking_regions_reload(ParkingService *service)
{
    SXMGridService *grid = &service->grid;
    uint i, j;
    uint need = 0xF;  // bit mask that indicates which regions from 'needed' list
                      // should be loaded
    uint ifree = 0xF; // bit mask that indicates, data for what regions, should be
                      // removed from service->fullness_au

    //  if we can re-use any of the save data, then do so, else reset
    for  (i = 0; i < ARRAY_SIZE(grid->needed); ++i) {
        for  (j = 0; j < ARRAY_SIZE(service->fullness_au); j++) {
            if  (service->fullness_au[j].region == grid->needed[i]) {
                need &= ~(1U << i);        // don't need to load this one
                ifree &= ~(1U << j);        // and keep the block in-use
                break;
            }
        }
    }
    //  free any saved AUs from the unwanted regions
    for (j = 0; j < ARRAY_SIZE(service->fullness_au); ++j) {
        if (ifree & (1U << j)) {
            for (i = 0; i < ARRAY_SIZE(service->fullness_au[j].safe); ++i) {
                if  (grid->bptr[j]->safe[i].pkt)
                    sxm_cfile_map_free(&service->fullness_data->g, PARKING_FULLNESS_SAVE_HEAP,
                                       service->fullness_au[j].safe[i].pktix, 1);
            }
            memset(&service->fullness_au[j], 0, sizeof(SXMParkingFullnesSavedAU));
        }
    }
    for (i = 0; i < ARRAY_SIZE(grid->needed); ++i) {
        if (need & (1U << i)) {
            for (j = 0; j < ARRAY_SIZE(service->fullness_au); ++j) {
                if (ifree & (1U << j)) {
                    service->fullness_au[j].region = grid->needed[i];
                    ifree &= ~(1U << j);        // used this block now
                    break;
                }
            }
        }
    }
}

/********************************************************************//**
 * The function checks the timestamp and invalidates expired data
 * notifying application via request callback if available and required.
 *
 * \param[in] service service instance
 * \param[in] time_now value of the NOW time in minutes
 *
 ************************************************************************/
static void parking_check_timestamp(ParkingService *service, const uint time_now) {
    ENTER_PKT("");

    if (time_now < service->grid.root->ts) {
        DEBUG_PKT("Ignore this check since section's time in future");
    }
    else {
        //  check the timestamp  (< 24 hours old)
        const uint age = time_now - service->grid.root->ts;
        if (age > PARKING_MAX_MINUTES) {
            uint idx;
            DEBUG_REQ("File too old (%d)", age);
            for (idx = 0; idx < ARRAY_SIZE(service->grid.bptr); ++idx) {
                service->grid.bptr[idx]->region = 0;
                memset(service->grid.bptr[idx]->safe, 0, sizeof(service->grid.bptr[idx]->safe));
            }
            sxm_cfile_clear_section(&service->data->g, PARKING_BASE);
            sxm_cfile_clear_section(&service->data->g, PARKING_PPOI);
            BITSET(service->grid.sections_update, PARKING_BASE);
            BITSET(service->grid.sections_update, PARKING_PPOI);
            sxm_cfile_map_init(service->grid.root, PARKING_HEAP, 0);
        }
    }

    LEAVE_PKT("");
}

/************************************************************************
 *                                                                      *
 *            User Collection and Extraction Requests                   *
 *            =======================================                   *
 *                                                                      *
 ************************************************************************/

/***************************************************************************//**
 *
 * Extract Parking location service object.
 *
 * Returns the next closest parking location entry sorted by distance from location.
 *
 * \param[in] service service instance
 * \param[in] ret points to the buffer receiving the service object.
 * \param[in] res the loc index of the next closest object
 * \param[in] rec_id location's unique id
 * \param[in] hix region data storage index
 *
 * \retval SXM_E_OK if service object found
 * \retval SXM_E_NOENT if service object not found
 * \retval SXM_E_PIPE if failed to access location data within DB
 *
 ********************************************************************************/
static int parking_extract(ParkingService *service, SXMParkingLocation *ret,
                           short res, uint rec_id, int hix) {
    SXMGridBaseLine *b;
    ParkingFavLocation *ifav;
    SXMByteBuff dbrecb, *prpb;
    int i, j;
    char ptemp[128];
    ParkingRecord *rec;
    fix dlon, dlat;
    ushort ustmp;
    const uint pu = SXM_GRID_UNPACK_UID(rec_id);
    SXMSdkConfig config;
    BOOL is_secondary = FALSE;

    ENTER_EXT("");

    rec = &service->stations.items[res];

    b = service->grid.bptr[hix];

    if (pu > b->dbentry[0]) {
        LEAVE_EXT("%s", sxm_sdk_format_result(SXM_E_NOENT));
        return SXM_E_NOENT;
    }
    // guard against INVALID entry
    else if (b->dbentry[pu] == SXM_DB_INVALID_ENTRY) {
        LEAVE_EXT("%s", sxm_sdk_format_result(SXM_E_NOENT));
        return SXM_E_NOENT;
    }

    if ((prpb = sxm_bytebuff_init(&dbrecb, b->base, (int)b->size, (int)b->dbentry[pu])) == NULL) {
        LEAVE_EXT("%s", sxm_sdk_format_result(SXM_E_PIPE));
        return SXM_E_PIPE;
    }

    ret->id = rec_id;
    ret->lon = service->grid.ploc[res].p.lon;
    ret->lat = service->grid.ploc[res].p.lat;
    ret->distance = sxm_point_true_distance(service->grid.ploc[res].distance);
    sxm_bytebuff_skip(&dbrecb, sizeof(fix)/*lon*/ + sizeof(fix) /* lat */);
    dlon = (fix)sxm_bytebuff_g4(&dbrecb);
    dlat = (fix)sxm_bytebuff_g4(&dbrecb);
    if ((dlon == 0) && (dlat == 0)) {
        ret->lon_sec = 0;
        ret->lat_sec = 0;
    }
    else {
        is_secondary = TRUE;
#if SXM_USE_FIX
        ret->lon_sec = ret->lon + dlon;
        ret->lat_sec = ret->lat + dlat;
#else
        ret->lon_sec = ret->lon + fix2float(dlon);
        ret->lat_sec = ret->lat + fix2float(dlat);
#endif
    }

    ret->version = sxm_bytebuff_g2(&dbrecb);
    ret->height = sxm_bytebuff_g2(&dbrecb);
    ret->height_sec = sxm_bytebuff_g2(&dbrecb);
    sxm_sdk_get_config(&config);
    if (config.unit == SXM_SDK_UNIT_METRIC) {
        //  convert inches to centimeters
        ret->height = (ushort)((ret->height * 254) / 100);
        ret->height_sec = (ushort)((ret->height_sec * 254) / 100);
    }
    // If a secondary entrance presents and its height equals primary entrance height 
    // (i.e. height_sec == 0), 'height_sec' field is set with value from 'height' field.
    if (is_secondary && (ret->height != 0) && (ret->height_sec == 0)) {
        ret->height_sec = ret->height;
    }
    ret->capacity = sxm_bytebuff_g2(&dbrecb);
    ustmp = sxm_bytebuff_g2(&dbrecb);
    ret->price_hour = (byte)(ustmp / 10);
    ret->price_day = (byte)(ustmp % 10);
    ret->amen = (uint)(sxm_bytebuff_g2(&dbrecb) << 16);
    ret->amen |= (uint)sxm_bytebuff_g2(&dbrecb);
    sxm_grid_decode_phone(&dbrecb, &ret->phone[0], sizeof(ret->phone) - 1);
    sxm_bytebuff_string(&dbrecb, &ret->name[0], sizeof(ret->name) - 1);
    sxm_bytebuff_string(&dbrecb, &ret->address[0], sizeof(ret->address) - 1);
    sxm_bytebuff_string(&dbrecb, &ret->address_sec[0], sizeof(ret->address_sec) - 1);
    sxm_bytebuff_string(&dbrecb, &ret->city[0], sizeof(ret->city) - 1);
    sxm_bytebuff_string(&dbrecb, &ret->state[0], sizeof(ret->state) - 1);
    sxm_bytebuff_string(&dbrecb, &ret->zip[0], sizeof(ret->zip) - 1);

    sxm_bytebuff_string(&dbrecb, &ptemp[0], sizeof(ptemp) - 1);
    for  (i = 0, j= 0; i < 7; ++i) {
        char *t = ret->days[i].times;

        while (ptemp[j]) {
            if (ptemp[j] == ' ') {
                j++;
                break;
            }
            else { 
                *t++ = ptemp[j++];
            }
        }
    }
    sxm_bytebuff_string(&dbrecb, &ret->comment[0], sizeof(ret->comment) - 1);

    ret->user_flag = SXM_LOCATION_IS_IN_COLLECTION | SXM_LOCATION_IS_IN_EXTRACTION;
    if (service->grid.fav.active &&
        ((ifav = (ParkingFavLocation*)sxm_grid_is_poi_from_favorite_list(&service->grid, ret->id)) != NULL))
    {
        ret->user_flag |= (ushort)(SXM_LOCATION_IS_IN_PPOIS | ifav->ppoi_spec.type_flags);
        // NOTE: Assumption that the ppoi storage should have better or the same data
        //       in comparison to the regular location. This is happening due to sync
        //       between regular locations and ppoi storages during AU processing from
        //       either set ppoi routine or broadcast data processing by *complete* function.
        DEBUG_EXT("Using PPOI dynamic data");
        rec = &ifav->fav_station;
    }

    //  now add the variable data
    ret->valid = rec->valid;
    ret->status = rec->status;
    ret->avail_time = rec->avail_time;
    memcpy(ret->fullness, rec->full, sizeof(ret->fullness));
    memcpy(ret->weekday_hour, rec->weekday_hour, sizeof(ret->weekday_hour));
    memcpy(ret->saturday_hour, rec->saturday_hour, sizeof(ret->saturday_hour));
    memcpy(ret->sunday_hour, rec->sunday_hour, sizeof(ret->sunday_hour));
    ret->weekday_day = rec->weekday_day;
    ret->saturday_day = rec->saturday_day;
    ret->sunday_day = rec->sunday_day;
    ret->ebird = rec->ebird;
    ret->event = rec->event;
    ret->other = rec->other;
    ret->numtotal = rec->total;

    if (sxm_bytebuff_state(prpb) != SXM_BYTEBUFF_STATE_OK) {
        LEAVE_EXT("%s", sxm_sdk_format_result(SXM_E_PIPE));
        return SXM_E_PIPE;
    }

    LEAVE_EXT("%s", sxm_sdk_format_result(SXM_E_OK));
    return SXM_E_OK;
}

/***************************************************************************//**
 * This function reads the Parking location information for parking with id
 * from the parking service database.
 *
 * \param[in]  service      service instance
 * \param[out] parking_loc   a pointer to an SXM Parking Location that will be
 *                           filled up with parking data, see \ref SXMParkingLocation
 * \param[in]  base          the information collection for a single grid square,
 *                           see \ref SXMGridBaseLine
 * \param[in]  puid          parking id identical to ID field in the baseline
 *
 * \retval SXM_E_OK    if data has been extracted
 * \retval SXM_E_NOENT if no entry found for the provided parking id.
 * \retval SXM_E_INVAL if found entry is out of expected POI MBR
 * \retval SXM_E_PIPE  if failed to load station data from the t-file
 *
 ******************************************************************************/
static int parking_read_location_info(ParkingService *service, SXMParkingLocation *parking_loc,
                                      SXMGridBaseLine *base, ushort puid)
{
    int rc = SXM_E_NOENT;

    memset(parking_loc, 0, sizeof(*parking_loc));

    DEBUG_EXT("Extracting  location for puid = 0x%X", puid);

    if (puid <= base->dbentry[0]) {
        if (base->dbentry[puid] != SXM_DB_INVALID_ENTRY) {
            char ptemp[128];
            fix dlon, dlat, flon1, flat1;
            ushort ustmp;
            SXMByteBuff dbrecb, *prpb;
            BOOL is_secondary = FALSE;

            if ((prpb = sxm_bytebuff_init(&dbrecb, base->base, (int)base->size, (int)base->dbentry[puid])) == NULL) {
                rc = SXM_E_PIPE;
            }
            else {
                uint i, j;
                SXMSdkConfig config;

                flon1 = (fix)sxm_bytebuff_g4(&dbrecb);
                flat1 = (fix)sxm_bytebuff_g4(&dbrecb);

    #if SXM_USE_FIX
                parking_loc->lon = flon1;
                parking_loc->lat = flat1;
    #else
                parking_loc->lon = fix2float(flon1);
                parking_loc->lat = fix2float(flat1);
    #endif
                dlon = (fix)sxm_bytebuff_g4(&dbrecb);
                dlat = (fix)sxm_bytebuff_g4(&dbrecb);
                if ((dlon == 0) && (dlat == 0)) {
                    parking_loc->lon_sec = 0;
                    parking_loc->lat_sec = 0;
                }
                else {
                    is_secondary = TRUE;
    #if SXM_USE_FIX
                    parking_loc->lon_sec = parking_loc->lon + dlon;
                    parking_loc->lat_sec = parking_loc->lat + dlat;
    #else
                    parking_loc->lon_sec = parking_loc->lon + fix2float(dlon);
                    parking_loc->lat_sec = parking_loc->lat + fix2float(dlat);
    #endif
                }

                parking_loc->version = sxm_bytebuff_g2(&dbrecb);
                parking_loc->height = sxm_bytebuff_g2(&dbrecb);
                parking_loc->height_sec = sxm_bytebuff_g2(&dbrecb);
                sxm_sdk_get_config(&config);
                if (config.unit == SXM_SDK_UNIT_METRIC) {
                    //  convert inches to centimeters
                    parking_loc->height = (ushort)((parking_loc->height * 254) / 100);
                    parking_loc->height_sec = (ushort)((parking_loc->height_sec * 254) / 100);
                }
                // If a secondary entrance presents and its height equals primary entrance height 
                // (i.e. height_sec == 0), 'height_sec' field is set with value from 'height' field
                if (is_secondary && (parking_loc->height != 0) && (parking_loc->height_sec == 0)) {
                    parking_loc->height_sec = parking_loc->height;
                }
                parking_loc->capacity = sxm_bytebuff_g2(&dbrecb);
                ustmp = sxm_bytebuff_g2(&dbrecb);
                parking_loc->price_hour = (byte)(ustmp / 10);
                parking_loc->price_day = (byte)(ustmp % 10);
                parking_loc->amen = (uint)(sxm_bytebuff_g2(&dbrecb) << 16);
                parking_loc->amen |= (uint)sxm_bytebuff_g2(&dbrecb);
                sxm_grid_decode_phone(&dbrecb, &parking_loc->phone[0], sizeof(parking_loc->phone) - 1);
                sxm_bytebuff_string(&dbrecb, &parking_loc->name[0], sizeof(parking_loc->name) - 1);
                sxm_bytebuff_string(&dbrecb, &parking_loc->address[0], sizeof(parking_loc->address) - 1);
                sxm_bytebuff_string(&dbrecb, &parking_loc->address_sec[0], sizeof(parking_loc->address_sec) - 1);
                sxm_bytebuff_string(&dbrecb, &parking_loc->city[0], sizeof(parking_loc->city) - 1);
                sxm_bytebuff_string(&dbrecb, &parking_loc->state[0], sizeof(parking_loc->state) - 1);
                sxm_bytebuff_string(&dbrecb, &parking_loc->zip[0], sizeof(parking_loc->zip) - 1);

                sxm_bytebuff_string(&dbrecb, &ptemp[0], sizeof(ptemp) - 1);
                for  (i = 0, j = 0; i < ARRAY_SIZE(parking_loc->days); i++) {
                    char *t = parking_loc->days[i].times;

                    while (ptemp[j])
                        if  (ptemp[j] == ' ') {
                            j++;
                            break;
                        }
                        else {
                            *t++ = ptemp[j++];
                        }
                }

                sxm_bytebuff_string(&dbrecb, &parking_loc->comment[0], sizeof(parking_loc->comment) - 1);

                rc = (sxm_bytebuff_state(prpb) == SXM_BYTEBUFF_STATE_OK) ? SXM_E_OK : SXM_E_PIPE;
            }
        }
        else {
            rc = SXM_E_INVAL;
        }
    }
    return rc;
}

/***************************************************************************//**
 * Helper function to perform parking location id comparison over
 * ParkingFavLocation.
 *
 * \param[in] p1 the first object to compare
 * \param[in] p2 the second object to compare
 * \return returns the difference between p1 and p2. Value can be positive or
 *                 negative.
 *
 ******************************************************************************/
static int parking_index_compare_callback(const ParkingFavLocation *p1,
                                          const ParkingFavLocation *p2)
{
    return (int)(p1->ppoi_spec.ppoi_id - p2->ppoi_spec.ppoi_id);
}

/***************************************************************************//**
 * Helper function to perform parking location region comparison over
 * ParkingFavLocation.
 *
 * \param[in] p1 the first object to compare
 * \param[in] p2 the second object to compare
 * \return returns the difference between p1 and p2. Value can be positive or
 *                 negative.
 *
 ******************************************************************************/
static int parking_locations_region_compare_callback(const ParkingFavLocation *p1,
                                                     const ParkingFavLocation *p2)
{
    return (int)(SXM_GRID_UNPACK_REGION(p1->ppoi_spec.ppoi_id) -
            SXM_GRID_UNPACK_REGION(p2->ppoi_spec.ppoi_id));
}

/***************************************************************************//**
 * This function extracts an individual parking location with the POI filter
 * settings and from expected POI extracted MBR
 *
 * \param[in] service service instance
 * \param[out] ret a pointer to an SXM Parking Location that will be
 *                   filled up with parking data, see \ref SXMParkingLocation
 * \param[in] id unique location id
 * \param[in] flags collection of POI flags
 * \param[in] rec dynamic data if exists, NULL if there is no data for the location
 *
 * \retval SXM_E_OK    if data has been extracted
 * \retval SXM_E_NOENT if no entry found
 * \retval SXM_E_INVAL if found entry is out of expected POI MBR
 *
 ******************************************************************************/
static int parking_poi_extract(ParkingService *service, SXMParkingLocation *ret,
                               uint id, ushort flags, const ParkingRecord *rec)
{
    int rc;
    ENTER_EXT("(ret: %p, id=%08x, flags: %04x, rec: %p", ret, id, flags, rec);

    memset(ret, 0, sizeof(*ret));

    rc = parking_read_location_info(service, ret, service->grid.PoiExtract.p_grid_base,
                                    SXM_GRID_UNPACK_UID(id));
    if (rc == SXM_E_OK) {
        ret->id = id;
        ret->user_flag = flags;
        /* no distance even if available */
#if SXM_USE_FIX
        ret->distance = sxm_fix_int(-1);
#else
        ret->distance = -1;
#endif

        if (rec) {
            ret->valid = rec->valid;
            ret->status = rec->status;
            ret->avail_time = rec->avail_time;
            memcpy(ret->fullness, rec->full, sizeof(ret->fullness));
            memcpy(ret->weekday_hour, rec->weekday_hour, sizeof(ret->weekday_hour));
            memcpy(ret->saturday_hour, rec->saturday_hour, sizeof(ret->saturday_hour));
            memcpy(ret->sunday_hour, rec->sunday_hour, sizeof(ret->sunday_hour));
            ret->weekday_day = rec->weekday_day;
            ret->saturday_day = rec->saturday_day;
            ret->sunday_day = rec->sunday_day;
            ret->ebird = rec->ebird;
            ret->event = rec->event;
            ret->other = rec->other;
            ret->numtotal = rec->total;
        }
    }
    else {
        DEBUG_EXT("Failed to load uid %u", SXM_GRID_UNPACK_UID(id));
    }

    LEAVE_EXT("%s", sxm_sdk_format_result(rc));
    return rc;
}
