/************************************************************************
 *                                                                      *
 *            SXM_PARKING_DATABASE.C                                    *
 *            ======================                                    *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 *                              The baseline file for Parking locations *
 *                                                                      *
 ************************************************************************/
/***************************************************************************//**
 *
 * \file sxm_parking_database.c
 * \author Leslie French
 * \date 8/20/2013
 * 
 * The baseline file for Parking stations
 *
 * \page parking_record "Parking DB Structure"
 *
 * The file is organized as a transaction file with an certain block size
 * defined by \ref PARKING_BLOCK_SIZE which has been defined by service
 * designed.
 *
 * \section parking_root Parking Root Block
 *
 * The root block is 2048 4-byte entries, one per region.
 * 
 * Each entry is:
 * - \b offset of first data block : 2 bytes
 * - \b number of data blocks : 2 bytes
 *
 * \section parking_data_block Parking Data Block(s)
 *
 * The data blocks for each region comprise a header followed by a variable
 * number of variable length records. The index into the record is the FSUID
 * used to identify the station
 * 
 * \par Header
 *  - number of records in data block : 4 bytes
 *  - offset to record [number of records] * 4 bytes
 *
 * \par Record
 *  - \b longitude in fixpoint : 4 bytes
 *  - \b latitude in fixpoint : 4 bytes
 *  - \b delta longitude in fixpoint : 4 bytes
 *  - \b delta latitude in fixpoint : 4 bytes
 *  - \b version of record : 2 bytes
 *  - \b primary height : 2 bytes
 *  - \b secondary height : 2 bytes
 *  - \b price : 2 bytes
 *  - \b amenity 1 fields : 2 bytes
 *  - \b amenity 2 fields : 2 bytes
 *  - \b phone : 5 bytes (4 bits per digit)
 *  - \b name : nil-terminated string
 *  - \b primary address : nil-terminated string
 *  - \b secondary address : nil-terminated string
 *  - \b city : nil-terminated string
 *  - \b state : nil-terminated string
 *  - \b zip : nil-terminated string
 *  - \b days : nil-terminated string
 *  - \b comment : nil-terminated string
 *  - \b padding bytes: up to sizeof(unsigned int)
 *
 ******************************************************************************/

#define DEBUG_TAG "parking"

#include "sxm_parking_internal.h"

/***************************************************************************//**
 * Initializes DB environment
 *
 * \param[in] service service instance
 *
 ******************************************************************************/
int sxm_parking_database_start(ParkingService *service) {
    int rc;

    ENTER_DBF("");

    rc = sxm_grid_database_start(&service->grid, PARKING_SERVICE_NAME,
                                 PARKING_SERVICE_DB_NAME, PARKING_SERVICE_NAME,
                                 PARKING_TFILE_SCHEMA_VERSION);
    if (rc == SXM_E_PIPE) {
        rc = SXM_E_BAD_DB;
    }

    LEAVE_DBF("%s", sxm_sdk_format_result(rc));
    return rc;
}

/***************************************************************************//**
 * Uninitializes DB environment
 *
 * \param[in] service service instance
 *
 ******************************************************************************/
void sxm_parking_database_stop(ParkingService *service) {
    if (service) {
        ENTER_DBF("");
        if (service->grid.dbfile) {
            DEBUG_DBF("version=%d", service->grid.db_version);
            sxm_tfile_close(service->grid.dbfile);
            service->grid.dbfile = NULL;
        }
        LEAVE_DBF("");
    }
}

/***************************************************************************//**
 * Restarts the Parking database.
 *
 * If the database file is updated (for example using a file delivered over RFD),
 * the grid library needs to refresh its copy of the data.
 * The \ref sxm_parking_database_restart routine reloads any data from the Parking
 * baseline file.
 *
 * \param[in] service service instance
 *
 * \retval SXM_E_OK Success 
 * \retval SXM_E_NOENT TFile failed to open
 * \retval SXM_E_CORRUPT TFile corrupted or has unsupported schema
 * \retval SXM_E_PIPE Error reading data from the TFILE
 * \retval SXM_E_FAULT database never started
 *
 ********************************************************************************/
int sxm_parking_database_restart(ParkingService *service) {
    SXMResultCode rc;

    ENTER_DBF("");

    if (service) {

        rc = sxm_grid_database_restart(&service->grid,
                    PARKING_SERVICE_NAME, PARKING_SERVICE_DB_NAME,
                    PARKING_TFILE_SCHEMA_VERSION);
        if (rc == SXM_E_OK) {
            DEBUG_DBF("(version=%d)", service->grid.db_version);
        }
    }
    else {
        rc = SXM_E_FAULT;
    }

    LEAVE_DBF("%s", sxm_sdk_format_result(rc));
    return rc;
}
