/************************************************************************
 *
 *            SXM_APOGEE_INTERNAL.H
 *            =====================
 *
 *                                 Copyright 2013 Sirius XM Radio, Inc. 
 *                                                 All Rights Reserved. 
 *               Licensed Materials - Property of Sirius XM Radio, Inc. 
 *
 *    For passing around inside the Apogee Module
 *
 ************************************************************************/

#ifndef SXM_APOGEE_INTERNAL_H
#define SXM_APOGEE_INTERNAL_H


#include <sxm_apogee.h>
#ifndef SDKFILES_BUILDER
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>
#include <sxm_stdtrace.h>
#endif // SDKFILES_BUILDER 

#define APOGEE_SERVICE_NAME     "apogee"

/* Data Service Identifiers for Appogee Service */
#define APOGEE_DSI_COUNT                2

/* Data Service Identifier for Locations, Construction and Incidents */
#define APOGEE_DSI_490                  490
/* Fixed Data Multiplex Indexes for this DSI */
#define APOGEE_DSI_490_DATA_DMI_START   640
#define APOGEE_DSI_490_DATA_DMI_COUNT   32
#define APOGEE_DSI_490_RFD_DMI_START    736
#define APOGEE_DSI_490_RFD_DMI_COUNT    4

/* Data Service Identifier for Flow, Ramps and Predictive */
#define APOGEE_DSI_491                  491
/* Fixed Data Multiplex Indexes for this DSI */
#define APOGEE_DSI_491_DATA_DMI_START   672
#define APOGEE_DSI_491_DATA_DMI_COUNT   64

/* Memory Pool Sizes, each Pool is defined by (a) the size of each 32-block group and (b) the number of 32-block groups */
#define APOGEE_LINEAR_HEAP_BLOCK_SIZE   1024
#define APOGEE_LINEAR_HEAP_BLOCKS       30
#define APOGEE_AU_HEAP_BLOCK_SIZE       1024
#define APOGEE_AU_HEAP_BLOCKS           16

#define APOGEE_PROTOCOL_VERSION                     1
#define APOGEE_PROTOCOL_MAX_CAROUSEL_ID             7
#define APOGEE_PROTOCOL_MIN_PACKET_BYTE_COUNT       6 // 2 + 4 first 16 bits plus 4 byte crc
#define APOGEE_PROTOCOL_PRED_DATASET_COUNT          16
#define APOGEE_PROTOCOL_FORECAST_DATASET_COUNT      16
#define APOGEE_PROTOCOL_CARID_BITLEN                4

// Patterns database
#define APOGEE_PDB_FOLDER_NAME                      APOGEE_SERVICE_NAME
#define APOGEE_PDB_NAME                             "patterns"
#define APOGEE_PDB_FILE_FORMAT                      "patterns"
#define APOGEE_PDB_FILENAME                         "patternsdb"
#define APOGEE_PDB_TFILE_TYPE                       'W'
#define APOGEE_PDB_TFILE_SCHEMA_VERSION             (1)
#define APOGEE_PDB_TFILE_BLOCK_SIZE                 (2048)
#define APOGEE_PDB_TFILE_BSA_DATA_SIZE              (APOGEE_PDB_TFILE_BLOCK_SIZE/sizeof(ushort) - ((SXM_APOGEE_MAX_TABLE_COUNT + 1)*sizeof(ushort)))
#define APOGEE_PDB_TFILE_MAX_PATTERN_INDEX          (255)
#define APOGEE_PDB_TFILE_NUM_HISTORY_IDX            (SXM_APOGEE_HIST_NUM_SEASONS * SXM_APOGEE_HIST_NUM_DAYS * SXM_APOGEE_HIST_NUM_INTERVALS)
#define APOGEE_PDB_TFILE_SEASONS_OFFSET             (SXM_APOGEE_HIST_NUM_DAYS * SXM_APOGEE_HIST_NUM_INTERVALS)
#define APOGEE_PDB_TFILE_DAYS_OFFSET                (SXM_APOGEE_HIST_NUM_INTERVALS)
#define APOGEE_PDB_BINFILE_TYPE                     'F'

// Locations database 
#define SXM_LOCATION_DB_FOLDER_NAME                 APOGEE_SERVICE_NAME
#define SXM_LOCATION_DB_FILE_NAME                   "locations"
#define SXM_LOCATION_DB_FILE_FORMAT                 "locations"
#define SXM_LOCATION_DB_TFILE_TYPE                  'R'
#define SXM_LOCATION_DB_TFILE_SCHEMA_VERSION        (3)
#define SXM_LOCATION_DB_TFILE_BLOCK_SIZE            (512)
#define SXM_LOCATION_DB_TFILE_BLOCK_SIZE_SHIFT      (9) // (1 << 9) == 512

// SXM Ramps database (Layout same as Locations database)  
#define SXM_SXMRAMP_DB_FOLDER_NAME                  APOGEE_SERVICE_NAME
#define SXM_SXMRAMP_DB_FILE_NAME                    "rampsdb"
#define SXM_SXMRAMP_DB_FILE_FORMAT                  "sxmramps"
#define SXM_SXMRAMP_DB_TFILE_TYPE                   'W'
#define SXM_SXMRAMP_DB_TFILE_SCHEMA_VERSION         (1)
#define SXM_SXMRAMP_DB_TFILE_BLOCK_SIZE             (512)

#define APOGEE_FLOW_MAX_LANES                       8


//
//  Structures stored in the Locations/ SXM Ramps baseline files
//
typedef struct {
    fix ll_lon;
    fix ll_lat;
    fix ur_lon;
    fix ur_lat;
    ushort offset;
    ushort count;
}Table;

typedef struct {
    fix ll_lon;
    fix ll_lat;
    fix ur_lon;
    fix ur_lat;
    ushort linear;
    ushort lcount;
    ushort ramp;
    ushort rcount;
}BSA;

typedef struct {
    fix ll_lon;
    fix ll_lat;
    fix ur_lon;
    fix ur_lat;
    ushort tmc1;
    ushort tmc2;
    ushort count;
    ushort type;
}Linear;


//
//  Structures stored in the Historical db file
//
typedef struct {
    ushort offset[SXM_APOGEE_MAX_TABLE_COUNT+1];
    ushort BSA_data[APOGEE_PDB_TFILE_BSA_DATA_SIZE];
} PTRN_ROOT_BLOCK;

typedef struct {
    uint patterns;
    ushort ptrn_ver[APOGEE_PDB_TFILE_MAX_PATTERN_INDEX+1];
    byte history_index[APOGEE_PDB_TFILE_NUM_HISTORY_IDX];
} PTRN_DATA_BLOCK;


//
//  Structures stored in the Patterns binary files
//
typedef struct {
    ushort version;
    uint ramp_offset;
} PTRN_BIN_HEADER;


#ifndef SDKFILES_BUILDER

#define APOGEE_UPDATE_TEMPFILE_TYPE                 'O'

//
// Carousel ids.
//
#define CARID_RT_ROADS                              0
#define CARID_RT_RAMPS                              1
#define CARID_RT_CONSTRUCTION                       2
#define CARID_RT_INCIDENT                           3
#define CARID_PREDICTIVE_DATA                       4
#define CARID_FORCAST_DATA                          5
#define CARID_RFD_FILE                              6
#define CARID_RFD_METADATA                          7

#define CARID_CISTRINGBUFF_OFFSET                   2

// Specification of the locations database.
#define LDB_RAMP_TMC                                0
#define LDB_RAMP_SXM                                1
#define LDB_LINEAR                                  2

// Force purge of data set
#define FORCE_PURGE                                 1
#define DO_NOT_FORCE_PURGE                          0

// Location database
#define LDB_TABLE_COUNT                             SXM_APOGEE_MAX_TABLE_COUNT
#define LDB_BLOCK_SIZE_SHIFT                        SXM_LOCATION_DB_TFILE_BLOCK_SIZE_SHIFT
#define LDB_BSA_BUFFER_SIZE                         MODUP(sizeof(BSA)*SXM_APOGEE_MAX_TABLE_BSA_COUNT, LDB_BLOCK_SIZE_SHIFT)
#define LDB_FILTER_BUFFER_SIZE                      MODUP(MAX(sizeof(Linear)*SXM_APOGEE_MAX_BSA_LINEAR_COUNT,sizeof(BSA)*SXM_APOGEE_MAX_TABLE_BSA_COUNT), LDB_BLOCK_SIZE_SHIFT)

#define TABLE_INVALID_ID                            0

#define APOGEE_FORMAT_CI_ID(_marker, _offset, _dir, _id, _ci, _syn) ((uint)((_marker) & 0xff) | (uint)((_offset)<<16U) | (uint)((_dir)<<19U) | (uint)((_id)<<20U) | (uint)((_ci)<<28U) | (uint)((_syn)<<29U))

#define APOGEE_IMPL_PRED_DATASET_COUNT              SXM_APOGEE_PRED_DATASET_COUNT

#define CLEVEL_NULL_FLOW (7)
#define BUCKET_NULL_SPEED (15)
#define CLEVEL_EXISTING_FLOW (5)
#define BUCKET_EXISTING_SPEED (0)

#define SXM_APOGEE_MAX_COLLECTION_TYPES             32
#define SXM_APOGEE_COUNTRY_CODE                     1

enum SAVE_IDX_ENUM{
    SAVE_IDX_FLOW,
    SAVE_IDX_RAMP,
    SAVE_IDX_CONSTRUCTION,
    SAVE_IDX_INCIDENT,
    SAVE_IDX_P1,
    SAVE_IDX_P2,
    SAVE_IDX_P3,
    SAVE_IDX_P4,
    SAVE_IDX_FORECAST,
    SAVE_IDX_COUNT,
};

enum SBUFF_IDX_ENUM{
    SBUFF_IDX_CONSTRUCTION,
    SBUFF_IDX_INCIDENT,
    SBUFF_IDX_COUNT
};

// localized return codes
#define APOGEE_E_AU_KEEP_PROCESSING                 1
#define APOGEE_E_AU_STOP_PROCESSING                 0

#define APOGEE_E_NO_AU_DATA                         -1
#define APOGEE_E_AU_DATA_AGEOUT                     -2

#define APOGEE_E_AU_OFFSET_FOUND                    1
#define APOGEE_E_AU_OFFSET_NOT_FOUND                0

#define APOGEE_E_SKIP_LINEAR_SUCCESS                1
#define APOGEE_E_SKIP_LINEAR_FAIL                   0

// item tracker 
#define APOGEE_ITEM_TOUCHED                         1
#define APOGEE_ITEM_NOT_TOUCHED                     0

// request tracker
#define APOGEE_REQUEST_IN_USE                       1
#define APOGEE_REQUEST_NOT_IN_USE                   0

#define APOGEE_AU_TABLE_ID_SIZE                     8
#define APOGEE_AU_CTSIZE_SIZE                       4
#define APOGEE_AU_SEQUENCE_SIZE                     4
#define APOGEE_AU_ID_SIZE                           16
#define APOGEE_AU_PCOUNT_SIZE                       4U
#define APOGEE_AU_POFFSET_SIZE                      5U
#define APOGEE_AU_BSACOUNT_SIZE                     8
#define APOGEE_AU_BSABID_SIZE                       8
#define APOGEE_AU_BSAOFFSET_SIZE                    16
#define APOGEE_AU_BSASIG_SIZE                       16
#define APOGEE_AU_CHECKSUM_SIZE                     4
#define APOGEE_AU_FC_OFFSET_SIZE                    6
#define APOGEE_AU_FC_DELTA_SIZE                     4
#define APOGEE_AU_FC_FCOUNT_SIZE                    4

// Linear Data AU fields
#define APOGEE_LINEAR_CODE_END                      0x00
#define APOGEE_LINEAR_CODE_POSITIVE                 0x01
#define APOGEE_LINEAR_CODE_NEGATIVE                 0x02
#define APOGEE_LINEAR_CODE_BOTH                     0x03
// Linear Data field sizes
#define APOGEE_LINEAR_CODE_SIZE                     2
#define APOGEE_LINEAR_BUCKET_SIZE                   4
#define APOGEE_LINEAR_CLEVEL_SIZE                   3
#define APOGEE_LINEAR_EXTENT_SIZE                   7
// Ramp Data field sizes
#define APOGEE_LINEAR_COUNT1_SIZE                   12
#define APOGEE_LINEAR_COUNT2_SIZE                   12
#define APOGEE_LINEAR_RLEVEL1_SIZE                  2
#define APOGEE_LINEAR_RLEVEL2_SIZE                  2
// CI Data field sizes
#define APOGEE_CI_ETYPE_SIZE                        1
// 8.17 fixed point (Longitude), as per Protocol Spec.
#define APOGEE_CI_GEO_LOC0_SIZE                     25
// 7.17 fixed point (Latitude), as per Protocol Spec.
#define APOGEE_CI_GEO_LOC1_SIZE                     24
#define APOGEE_CI_GEO_LOC_TEXT_SIZE                 16
#define APOGEE_CI_TMC_LOC_SIZE                      16
#define APOGEE_CI_TMC_OFFSET_SIZE                   3
#define APOGEE_CI_TMC_DIRECTION_SIZE                1
#define APOGEE_CI_TMC_EXTENT_SIZE                   7
#define APOGEE_CI_TMC_START_SIZE                    16
#define APOGEE_CI_TMC_START_OFFSET_SIZE             3
#define APOGEE_CI_EVENT_CODE_SIZE                   11
#define APOGEE_CI_EVENT_SEVERITY_SIZE               2
#define APOGEE_CI_EVENT_ID_SIZE                     8
#define APOGEE_CI_LANG_SIZE                         2
#define APOGEE_CI_STRING_OFFSET_SIZE                16
#define APOGEE_CI_EXT_SIZE                          7

// Lane Flow fields
#define APOGEE_FLOW_LANE_COUNT_SIZE                 3
#define APOGEE_FLOW_LANE_TYPE_SIZE                  3
#define APOGEE_FLOW_LINEAR_COUNT_SIZE               14

// Predictive Parameters
#define APOGEE_PRED_TIMEINC_SEC                     300

// Forecast Data field sizes
#define APOGEE_FC_PATTERN_SIZE                      8
#define APOGEE_FC_VER_SIZE                          10

// Forecast Parameters
#define APOGEE_FC_TIMEINC_SEC                       300

// Historical helper macros
#define APOGEE_HIST_SUBTYPE_SEASON(_stype)          ((_stype)>>12)
#define APOGEE_HIST_SUBTYPE_DAYOFWEEK(_stype)       (((_stype)>>8) & 0x07)
#define APOGEE_HIST_SUBTYPE_INTERVAL(_stype)        ((_stype) & 0xFF)


// from protocol spec, must skip EXT field if present as this version of Apogee does not decode it
// this is the formula for calculating bits to skip
#define APOGEE_CI_SKIP_EXTEND(_ext)                 (_ext+1)*8


/*************************************************/
/*                     RFD                       */
/*************************************************/

#define APOGEE_UPDATE_FNAME_SIZE        (4)
enum APOGEE_UPDATETYPES      { APOGEE_UPDATETYPE_PATTERN, APOGEE_UPDATETYPE_RAMP, NUM_APOGEE_UPDATETYPES };

/** RFD processing control structure */
typedef struct
{
//    ApogeeRFDState state; //!< Update state
    BOOL bIsProcessing; //!< An update file is being processed
    BOOL bIsCollecting[NUM_APOGEE_UPDATETYPES]; //!< Is currently collecting
    uint version[NUM_APOGEE_UPDATETYPES]; //!< Currently collecting version
} ApogeeRFD;


//
//  Pattern bin file structures
//

enum PTRN_BUFF_STATE{
    PTRN_BUFF_STATE_NONE,
    PTRN_BUFF_STATE_LANE,
    PTRN_BUFF_STATE_RAMP
};

typedef struct {
    PTRN_BIN_HEADER header;
    SXMBitFileBuff  BitBuf;
    enum PTRN_BUFF_STATE state;
    uint table;
    uint bsa;
    uint pattern;
} PTRN_BIN_BUFFER;


typedef struct {
    ushort map[SXM_APOGEE_MAX_TABLE_COUNT+1][SXM_APOGEE_MAX_TABLE_BSA_COUNT];
    ushort tableInfo[SXM_APOGEE_MAX_TABLE_COUNT+1];
}DatasetMask;

#define REQ_TRANS_ACTION_NONE                           -1
#define REQ_TRANS_ACTION_TABLE_REMOVE                   0
#define REQ_TRANS_ACTION_TABLE_ADD                      1
#define REQ_TRANS_ACTION_TABLE_MOD_PRUNE                2
#define REQ_TRANS_ACTION_BSA_REMOVE                     3
#define REQ_TRANS_ACTION_BSA_MOD_PRUNE                  4
#define REQ_TRANS_ACTION_BSA_MOD_ADD                    5
#define REQ_TRANS_ACTION_BSA_ADD                        6

// Masks to specify items required within an action.
// We need to manage allocation of sig storage for these types.
#define REQ_BSA_MOD_ADD_LINEAR_SIG_MASK (SXM_APOGEE_REQ_FLOW | SXM_APOGEE_REQ_RAMPS | SXM_APOGEE_REQ_PREDICTIVE)
// We need to prune string buffers for these types.
#define REQ_TABLE_MOD_PRUNE_STRING_BUFF (SXM_APOGEE_REQ_CONSTRUCTION | SXM_APOGEE_REQ_INCIDENT)

// Masks to qualify actions.
#define REQ_QUALIFY_BSA_MOD_ADD REQ_BSA_MOD_ADD_LINEAR_SIG_MASK
#define REQ_QUALIFY_TABLE_MOD_PRUNE REQ_TABLE_MOD_PRUNE_STRING_BUFF

#define REQ_TRANS_LIST_EXECUTOR_ \
    _ACTION_(REQ_TRANS_LIST_BSA_ADD                  ,0) \
    _ACTION_(REQ_TRANS_LIST_BSA_REMOVE               ,1) \
    _ACTION_(REQ_TRANS_LIST_MOD_PRUNE               ,2) \
    _ACTION_(REQ_TRANS_LIST_MOD_ADD                 ,3) \
    _ACTION_(REQ_TRANS_LIST_TABLE_ADD               ,4) \
    _ACTION_(REQ_TRANS_LIST_TABLE_REMOVE            ,5) \
    _ACTION_(REQ_TRANS_LIST_HISTORY                 ,6) \
    _ACTION_(REQ_TRANS_LIST_REVERTLOG               ,7) \
    _ACTION_(REQ_TRANS_LIST_MAX                     ,8)

// Produce the transaction list enums.
enum REQ_TRANS_LIST_ENUM {
#define _ACTION_(a,b) a = (b),
    REQ_TRANS_LIST_EXECUTOR_
#undef _ACTION_
};

typedef struct _CITrackingItem {
    struct _CITrackingItem *link;
    uint id;
    ushort sig;
    ushort sequence;
    ushort touched;
}CITrackingItem;

typedef struct _TransactionItem {
    struct _TransactionItem *link;
    ushort bsaref;
    ushort flags;
    ushort action;
}TransactionItem;

typedef struct {
    TransactionItem* root[REQ_TRANS_LIST_MAX];
}RequestTransaction;

typedef struct {
    ushort index;
    ushort sig;
    ushort offset;
    short change;
}BsaDirEntry;

#define MULTI_AU_GATHER_MAX_PACKETS                     8   // ushort bit mask is used for presence of each packet.
#define MULTI_AU_GATHER_AGE_OUT_INTERVAL                20
#define MULTI_AU_PACKET_SIZE                            5120
#define MULTI_AU_PACKET_COUNT                           (MULTI_AU_GATHER_MAX_PACKETS*5)

typedef struct _MultiAuPacket {
    struct _MultiAuPacket *link;
    byte b[MULTI_AU_PACKET_SIZE];
    uint count;
}MultiAuPacket;

typedef struct _MultiAuGather {
    struct _MultiAuGather *link;
    ushort table;
    ushort carid;
    ushort sig;
    ushort bsa;
    uint timeStamp;
    ushort startOffset;
    ushort packetNeedMask;
    ushort packetHaveMask;
    byte packetTot;
    MultiAuPacket* packets[MULTI_AU_GATHER_MAX_PACKETS];
    BsaDirEntry bsaDir[SXM_APOGEE_MAX_TABLE_BSA_COUNT+1];  // +1 for calculation of data size for the last 'real' entry.
    uint bsaDirCount;
}MultiAuGather;

typedef void (*ApogeeRequestCallback)(ptr, int, int, int, ptr);

typedef struct {
    byte* packet;
    SXMMemLoc packetLoc;
    ushort packetLen;       // actual packet length
    ushort sig;             // BSA signature of saved copy
    short useCount;
    uint *dataSig;          // data signatures for that BSA (used for change detection)
    SXMMemLoc dataSigLoc;
    uint timeStamp;
}AUSave;

// StringBuff related data
#define PACKED_STRING_MIN                                   6

typedef struct {
    int useCount;
    byte* expanded;                                         // index to expanded data
    SXMMemLoc expandedLoc;
    byte* packed;                                           // index to packed data
    SXMMemLoc packedLoc;
    uint expandedLen;                                       // expanded length
    uint packedLen;                                         // packed length
}StringBuff;

typedef struct {
    Linear *ldbLinear;                                      // linear data from locations database
    SXMMemLoc linearLoc;
    Linear *ldbRamp;                                        // TMC ramp data from locations database
    SXMMemLoc rampLoc;
    Linear *rdbRamp;                                        // SXM ramp data from sxmramps database
    SXMMemLoc rampSXM;
    AUSave aus[SAVE_IDX_COUNT];                             // saved AUS for each flow/cons type
    CITrackingItem* ciTrackingRoot[2];
}BsaCollection;

typedef struct {
    int table;                                              // TMC table number TABLE_INVALID_ID means free for use.
    BSA *ldbBsa;                                            // Location db bsa info.
    SXMMemLoc ldbBsaLoc;
    BSA *rdbBsa;                                            // SXM Ramps db bsa info.
    SXMMemLoc rdbBsaLoc;
    BsaCollection bsa[SXM_APOGEE_MAX_TABLE_BSA_COUNT];      // Collected bsas of the table.
    StringBuff strings[2];                                  // Strings for cons and inc.
    ushort fcount;                                          // Number of forecasts in group
}TableCollection;

typedef struct {
    int type;
    int subType;
    ushort table;
    ushort bsaIndex;
    ushort auSaveIndex;
    ushort packedIndex;
    TableCollection* col;                                   // Handle to data entry point cached in begin().
    SXMBitBuff bitParser;
    SXMBitBuff *pBitParser;

    int ldbLinearCount;                                     // Retrieved from locations.db: number of linears the user is aware of.
/* Retrieved from locations.db or sxmramps db: number of TMC or SXM ramps the user is aware of. */
    int dbRampCount;                                       

    // Apogee Forecast and Historical elements
#if (SXM_USE_APOGEE_FULL == 1)
    int patternIdx;                                       
#endif

    //
    // RT - Road Flow Processing
    //

    // Parsed from bitstream.
    int laneCount;
    int linearCount;
    byte laneType[APOGEE_FLOW_MAX_LANES];

    // Current location in bitstream.
    int laneCursor;
    int linearCursor;

    // Last successful user request.
    int lastLinearIndex;
    int lastLaneIndex;
}Extraction;

typedef struct {
    SXMApogeeRequest req;
    byte inuse;
    byte provisioned;
    int types;
    int userTypes;
    ApogeeRequestCallback notifyCallback;
    ptr userData;
    Extraction extracts[1 + APOGEE_IMPL_PRED_DATASET_COUNT];
    int extractCount;

#if (SXM_USE_APOGEE_FULL == 1)
    PTRN_BIN_BUFFER PtrnBuf;
#endif
}Request;

typedef struct {
    SXMDataService ds;
    SXMTFile* dbfile;
    SXMTFile* dbfile_ramps;
    ptr lliHandle;

    SXMMemPool *linearHeap;
    SXMMemPool *auHeap;

    Request request[SXM_APOGEE_MAX_REQ_COUNT+1];            // +1 for running speculative add/modify requests.
    pthread_mutex_t colMutex;
    TableCollection collection[SXM_APOGEE_COL_TABLE_COUNT];

    DatasetMask currentMask;
    DatasetMask newMask;

    // Used for parsing.
    BsaDirEntry bsaDir[SXM_APOGEE_MAX_TABLE_BSA_COUNT+1];  // +1 for calculation of data size for the last 'real' entry.
    uint bsaDirCount;

    CITrackingItem ciTrackingBuffer[SXM_APOGEE_COL_TABLE_COUNT*SXM_APOGEE_MAX_COL_TABLE_CI_COUNT];
    CITrackingItem* ciTrackingRoot;

    TransactionItem transBuffer[2*(SXM_APOGEE_MAX_REQ_BSAREF_COUNT+SXM_APOGEE_COL_TABLE_COUNT)];
    TransactionItem* transRoot;

    MultiAuPacket auPacketBuffer[MULTI_AU_PACKET_COUNT];
    MultiAuPacket* auPacketRoot;

    MultiAuGather auGatherBuffer[MULTI_AU_PACKET_COUNT/2];
    MultiAuGather* auGatherFreeRoot;
    MultiAuGather* auGatherActiveRoot;

    time_t predTimeStamp;
    uint predCount;
    ushort predOffset[APOGEE_IMPL_PRED_DATASET_COUNT];
    byte assembleBuffer[MULTI_AU_GATHER_MAX_PACKETS*MULTI_AU_PACKET_SIZE];

    // Apogee Forecast and Historical elements
#if (SXM_USE_APOGEE_FULL == 1)
    time_t forecastTimeStamp;
    ushort forecastOffset;
    uint forecastDelta;
    uint forecastCount;
    SXMTFile* dbfile_ptrn;
#endif

    ApogeeRFD rfd;   // RFD related data
}ApogeeService;

//
//  protocol
extern void sxm_apogee_protocol_complete_au(byte *, int, ptr);

extern void sxm_apogee_protocol_start(ApogeeService *);

extern void sxm_apogee_protocol_rfd_sync(ApogeeService *);

extern void sxm_apogee_protocol_stop(void);

//
//  databases
//
// The locations database is always fixed point MBR's.  We will convert the input
// MBR to fixed, if required, inside sxm_agogee_locations_xxx API.
//

extern int sxm_apogee_locations_start(ApogeeService* s);

extern void sxm_apogee_locations_stop(void);

extern int sxm_apogee_locations_add_bsas(SXMApogeeRequest *,
                     SXMMBR *);

extern int sxm_apogee_locations_add_linears(SXMApogeeBSAFilter *,
                        SXMMBR *,
                        ushort);

extern int sxm_apogee_locations_add_all_linears(SXMApogeeFilter *,
                        SXMMBR *,
                        SXMApogeeRequest *);

extern int sxm_apogee_locations_add_ramps(SXMApogeeBSAFilter *,
                      SXMMBR *,
                      ushort,
                      int);

extern int sxm_apogee_locations_add_all_ramps(SXMApogeeFilter *,
                          SXMMBR *,
                          SXMApogeeRequest *,
                          int);

extern int sxm_apogee_locations_version(uint *pVer);


// The sxm ramps database
extern int sxm_apogee_sxmramps_open(SXMTFile **ppTFile, const char *mode);

extern int sxm_apogee_sxmramps_start(ApogeeService* s);

extern void sxm_apogee_sxmramps_stop(void);

extern int sxm_apogee_sxmramps_restart(void);

extern int sxm_apogee_sxmramps_version(uint *pVer);


// The patterns database
extern void sxm_apogee_patternsdb_update_inmem(uint dataIndex, ushort newOffset);

extern int sxm_apogee_patternsdb_open(SXMTFile **ppTFile, const char *mode);

extern int sxm_apogee_patterns_start(ApogeeService* s);

extern void sxm_apogee_patterns_stop(void);

extern int sxm_apogee_patterns_restart(void);

extern int sxm_apogee_patterns_version(uint *pVer);

extern int sxm_apogee_patterns_open_hist_pattern(uint tableNumber, uint bsa, 
                                           uint season_idx, uint day_idx, uint interval_idx, 
                                           PTRN_BIN_BUFFER *pPtrnBuf);

extern int sxm_apogee_patterns_open_pattern(uint tableNumber, uint bsa, uint pattern, PTRN_BIN_BUFFER *pPtrnBuf);

extern int sxm_apogee_patterns_load_lane(PTRN_BIN_BUFFER *pPtrnBuf);

extern int sxm_apogee_patterns_load_ramp(PTRN_BIN_BUFFER *pPtrnBuf);

extern void sxm_apogee_patterns_close_pattern(PTRN_BIN_BUFFER *pPtrnBuf);

// Helper
extern int sxm_apogee_helper_ldbTableGetBsas(int, BSA *);

extern int sxm_apogee_helper_ldbLinearTypeGetRequiredBytes(ushort);

extern int sxm_apogee_helper_ldbBsaGetLinears(BSA *, Linear *);

extern int sxm_apogee_helper_ldbBsaGetRamps(BSA *, Linear *);

extern int sxm_apogee_helper_rdbTableGetBsas(int, BSA *);

extern int sxm_apogee_helper_rdbLinearTypeGetRequiredBytes(ushort);

extern int sxm_apogee_helper_rdbBsaGetRamps(BSA *, Linear *);

//
//  Requests
//
extern int sxm_apogee_request_add(SXMApogeeRequest* bsaRefs,
                  int types,
                  void (*notification)(ptr, int, int, int, ptr),
                  ptr userData,
                  ptr* handle);

extern int sxm_apogee_request_modify(ptr handle,
                     SXMApogeeRequest* bsaRefs,
                     int types);

extern int sxm_apogee_request_remove(ptr handle);

extern void sxm_apogee_request_start(ApogeeService *);
extern void sxm_apogee_request_stop(void);

extern void sxm_apogee_helper_FreeStringBuff(StringBuff* sbuff);

extern void sxm_apogee_helper_freeGather(MultiAuGather *);

// Helper
extern int sxm_apogee_helper_reqValidateSXMRequest(SXMApogeeRequest* bsaRefs, int* types);

extern Request* sxm_apogee_helper_locked_GetRequest(ptr handle);

//
// Extraction
//

extern int sxm_apogee_extraction_begin(ptr req,
                       int type,
                       int subType,
                       int bsa_ix_ref,
                       BOOL isRef,
                       int *c1,
                       int *c2);

extern int sxm_apogee_extraction_extract_flow(ptr req,
                          int lane,
                          int linearIndex,
                          SXMApogeeFlowVector *out,
                          int *changed);

extern int sxm_apogee_extraction_extract_ramp(ptr req,
                          int rampType,
                          SXMApogeeRampVector *out);

extern int sxm_apogee_extraction_extract_ci(ptr req,
                        SXMApogeeCIMessage *out);

extern int sxm_apogee_extraction_end(ptr req);

extern int sxm_apogee_extraction_get_lanes(ptr req, int *vec);

extern int sxm_apogee_extraction_get_predictive(ptr req, time_t* timeref, int *vec);

extern int sxm_apogee_extraction_get_forecast(ptr req, time_t* timeref, uint* delta, uint* count);

extern void sxm_apogee_extraction_start(ApogeeService *);

extern void sxm_apogee_extraction_stop(void);

//
//  Update
//
extern void sxm_apogee_update_init( ApogeeService *svc );

extern int sxm_apogee_update_start( SXMRfdMetadata *rfd );	// rfd

extern void sxm_apogee_update_uninit( void );

//
//  LLI: either SDK or SMS
//
extern int sxm_apogee_lli_start(ApogeeService* s);

extern int sxm_apogee_lli_stop(ApogeeService* s);

extern int sxm_apogee_lli_set_filter(ApogeeService* s, int dsi, int dmi, int enable);

extern int sxm_apogee_lli_rfd_start( SXMRfdMetadata *rfd ); 

extern int sxm_apogee_lli_rfd_status(int id, SXMRfdMetadata *rfd);

extern int sxm_apogee_lli_rfd_remove(int id);

extern int sxm_apogee_lli_rfd_add( byte *packet, int pl ); 


/************************************************************************
 *
 *            Development debugging related.
 *            ==============================
 *
 *    In a non-PRODUCTION build, the debug routines are compiled
 *    into the code, but the macros to call them are disabled.
 *    To enable the calls, uncomment the definition lines below
 *
 *    Do not attempt to mix modes, calling these routines in a
 *    production build will either fail to link, or cause a segment
 *    violation at run time.
 *
 *    The routines do not have to be all enabled at the same time
 *    Pick whiever you need to see.
 *
 ************************************************************************/

#ifdef SXM_DEBUG_PRODUCTION
#define SXM_APOGEE_CI_TRACKING_CHECK()
#define SXM_APOGEE_TRANS_LIST_CHECK()
#define SXM_APOGEE_AU_PACKET_CHECK()
#define SXM_APOGEE_AU_GATHER_CHECK()
#define SXM_APOGEE_DREQ_DATA_SET_MASK(_a, _b)
#define SXM_APOGEE_DREQ_TRANSACTION(_a, _b)
#define SXM_APOGEE_DEXT_PRINT_ROADS(_a, _b, _c, _d)
#define SXM_APOGEE_DEXT_PRINT_RAMPS(_a, _b, _c, _d)
#define SXM_APOGEE_DEXT_PRINT_CI(_a, _b)
#define SXM_APOGEE_DEXT_PRINT_SBUFF(_a, _b, _c, _d, _e)
#define SXM_APOGEE_DEXT_PRINT_FORECAST(_a, _b, _c, _d, _e)
#define SXM_APOGEE_DPROTO_PRINT_BSADIR()
#define SXM_APOGEE_MEMPOOL_TRACE(_a, _b)
#else
#define SXM_APOGEE_CI_TRACKING_CHECK()                      //sxm_apogee_debug_checkCITrackingList()
#define SXM_APOGEE_TRANS_LIST_CHECK()                       //sxm_apogee_debug_checkTransList()
#define SXM_APOGEE_AU_PACKET_CHECK()                        //sxm_apogee_debug_checkAuPacketList()
#define SXM_APOGEE_AU_GATHER_CHECK()                        //sxm_apogee_debug_checkAuGatherList()
#define SXM_APOGEE_DREQ_DATA_SET_MASK(_a, _b)               //sxm_apogee_debug_printDatasetMask(_a, _b)
#define SXM_APOGEE_DREQ_TRANSACTION(_a, _b)                 //sxm_apogee_debug_printTransaction(_a, _b)
#define SXM_APOGEE_DEXT_PRINT_ROADS(_a, _b, _c, _d)         //sxm_apogee_debug_print_roads(_a, _b, _c, _d)
#define SXM_APOGEE_DEXT_PRINT_RAMPS(_a, _b, _c, _d)         //sxm_apogee_debug_print_ramps(_a, _b, _c, _d)
#define SXM_APOGEE_DEXT_PRINT_CI(_a, _b)                    //sxm_apogee_debug_print_ci(_a, _b)
#define SXM_APOGEE_DEXT_PRINT_SBUFF(_a, _b, _c, _d, _e)     //sxm_apogee_debug_print_sbuff(_a, _b, _c, _d, _e)
#define SXM_APOGEE_DEXT_PRINT_FORECAST(_a, _b, _c, _d, _e)  sxm_apogee_debug_print_forecast(_a, _b, _c, _d, _e)
#define SXM_APOGEE_DPROTO_PRINT_BSADIR()                    //sxm_apogee_debug_print_BSADir()
#define SXM_APOGEE_MEMPOOL_TRACE(_a, _b)                    //sxm_mempool_dump_state(_a, _b)

extern void sxm_apogee_debug_start(ApogeeService *);

extern void sxm_apogee_debug_stop(void);

extern void sxm_apogee_debug_checkTransList(void);

extern void sxm_apogee_debug_checkCITrackingList(void);

extern void sxm_apogee_debug_checkAuPacketList(void);

extern void sxm_apogee_debug_checkAuGatherList(void);

extern void sxm_apogee_debug_printDatasetMask(DatasetMask *, char *);

extern void sxm_apogee_debug_printTransaction(RequestTransaction *, char *);

extern void sxm_apogee_debug_print_BSADir(void);

extern void sxm_apogee_debug_print_roads(byte *, int, int, int);

extern void sxm_apogee_debug_print_ramps(byte *, int, int, int);

extern void sxm_apogee_debug_print_ci(SXMApogeeCIMessage *, int);

extern void sxm_apogee_debug_print_sbuff(byte *, int, int, int, int);

extern void sxm_apogee_debug_print_forecast(byte *, int, int, int, int);

#endif // SXM_DEBUG_PRODUCTION

#endif // SDKFILES_BUILDER 

#endif // SXM_APOGEE_INTERNAL_H
