/************************************************************************
 *                                 Copyright 2014 Sirius XM Radio, Inc. 
 *                                                 All Rights Reserved. 
 *               Licensed Materials - Property of Sirius XM Radio, Inc. 
  *************************************************************************//**
 * \file sxm_parking.h
 * \author Leslie French
 * \date 5/20/2014
 *
 * Header for the production Parking service module
 ************************************************************************/

#ifndef SXM_PARKING_H
#define SXM_PARKING_H

#include <sxm_sdk.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \defgroup parking_pubapi Parking Public API
 * \brief This section contains Parking Public API
 */
/**@{*/
/**
 * \name Validity Bits
 * @{
 */
/** defines that all fullness fields are valid */
#define SXM_PARKING_VALID_FULL      0x1f
/** defines fullness[0], current occupancy */
#define SXM_PARKING_VALID_FULL0     (1 << 0)
/** defines fullness[1], +15 min forecast */
#define SXM_PARKING_VALID_FULL1     (1 << 1)
/** defines fullness[2], +30 min forecast */
#define SXM_PARKING_VALID_FULL2     (1 << 2)
/** defines fullness[3], +45 min forecast */
#define SXM_PARKING_VALID_FULL3     (1 << 3)
/** defines fullness[4], +60 min forecast */
#define SXM_PARKING_VALID_FULL4     (1 << 4)

/** defines weekday_hour */
#define SXM_PARKING_VALID_WK_HR     (1 << 5)
/** defines saturday_hour */
#define SXM_PARKING_VALID_SA_HR     (1 << 6)
/** defines sunday_hour */
#define SXM_PARKING_VALID_SU_HR     (1 << 7)
/** defines weekday_day */
#define SXM_PARKING_VALID_WK_DY     (1 << 8)
/** defines saturday_day */
#define SXM_PARKING_VALID_SA_DY     (1 << 9)
/** defines sunday_day */
#define SXM_PARKING_VALID_SU_DY     (1 << 10)
/** defines ebird */
#define SXM_PARKING_VALID_EBIRD     (1 << 11)
/** defines event */
#define SXM_PARKING_VALID_EVENT     (1 << 12)
/** defines other */
#define SXM_PARKING_VALID_OTHER     (1 << 13)
/** defines numtotal */
#define SXM_PARKING_VALID_NUMTOTAL  (1 << 14)
/** defines status */
#define SXM_PARKING_VALID_STATUS    (1 << 15)
/** @} */

/** Defines the maximum size of the \ref SXMParkingLocation::name field */
#define SXM_PARKING_NAME_SIZE_MAX       (40)
/** Defines the maximum size of the \ref SXMParkingLocation::address field */
#define SXM_PARKING_ADDR_SIZE_MAX       (40)
/** Defines the maximum size of the \ref SXMParkingLocation::city field */
#define SXM_PARKING_CITY_SIZE_MAX       (40)
/** Defines the maximum size of the \ref SXMParkingLocation::state field */
#define SXM_PARKING_STATE_SIZE_MAX      (4)
/** Defines the maximum size of the \ref SXMParkingLocation::zip field */
#define SXM_PARKING_ZIP_SIZE_MAX        (16)
/** Defines the maximum size of the \ref SXMParkingLocation::phone field */
#define SXM_PARKING_PHONE_SIZE_MAX      (16)
/** Defines the maximum size of the \ref SXMParkingLocation::comment field */
#define SXM_PARKING_COMMENT_SIZE_MAX    (256)

/**
 * The object represents the basic service's entity. It provides information 
 * about single Parking: state, name, address, phone, location.
 * It contains data about weekday prices and prices for each hour of occupancy during a weekday.
 * In addition, the field state provides information about current and predicted percentage fullness
 * Some fields are optional and in order to check existence of such
 *       field the \a valid field can be used.
 * \note The \a height variables are in inches when the units are set to imperial, and centimeters when 
 * the units are set to metric.  \a Prices are always in the local currency, \a times are in the local 
 * time-zone for that location, \a avail_time is in minutes, since 00:00 UTC.
 * If secondary entrance is absent, then \a lon_sec and \a lat_sec should be set to 0
 *
 */
typedef struct {
    uint id;                  //!< the parking location unique identifier
    ushort user_flag;         //!< a bitfield indicating the type of parking location
    SXMFLOAT lon;             //!< the longitude of the primary entrance to the parking location
    SXMFLOAT lat;             //!< the latitude of the primary entrance to the parking location
    SXMFLOAT lon_sec;         //!< the longitude of the secondary entrance to the parking location
    SXMFLOAT lat_sec;         //!< latitude of the secondary entrance to the parking location
    SXMFLOAT distance;        //!< the distance in miles from the extraction center point
    char name[SXM_PARKING_NAME_SIZE_MAX];       //!< the name of the parking location
    char address[SXM_PARKING_ADDR_SIZE_MAX];    //!< the street address (number and street name) of the primary entrance
    char address_sec[SXM_PARKING_ADDR_SIZE_MAX];//!< the street address (number and street name) of the secondary entrance
    char city[SXM_PARKING_CITY_SIZE_MAX];       //!< the city name
    char state[SXM_PARKING_STATE_SIZE_MAX];     //!< the 2 or 3 letter abbreviation for the State, Province etc
    char zip[SXM_PARKING_ZIP_SIZE_MAX];         //!< the ZIP or Postal Code
    char phone[SXM_PARKING_PHONE_SIZE_MAX];     //!< the telephone number
    char comment[SXM_PARKING_COMMENT_SIZE_MAX]; //!< additional text

    struct {
        char times[16];       //!< opening times
    } days[7];                //!< an array of strings describing the opening times for each day of the week

    uint amen;                //!< the amenities bitfield – 16 × 2-bit values
    ushort capacity;          //!< the number of parking spaces available

    ushort version;           //!< the version number of this record in the parking database
    ushort height;            //!< the height of the primary entrance
    ushort height_sec;        //!< height of secondary entrance

    //  pricing structures

    ushort valid;             //!< a bitfield indicating the validity of the following fields (see below)

    ushort numtotal;          //!< temporary total number of Parking Spaces at the Parking Location
    ushort weekday_hour[24];  //!< an array of prices for each hour of occupancy during a weekday
    ushort saturday_hour[24]; //!< an array of prices for each hour of occupancy during a Saturday
    ushort sunday_hour[24];   //!< an array of prices for each hour of occupancy during a Sunday
    ushort weekday_day;       //!< the price of all-day parking on a weekday
    ushort saturday_day;      //!< the price of all-day parking on a Saturday
    ushort sunday_day;        //!< the price of all-day parking on a Sunday

    ushort ebird;             //!< the price for parking during an "early-bird" special pricing period
    ushort event;             //!< the price for parking during a "special event" pricing period
    ushort other;             //!< the price for parking during an "other period" when active 
    uint avail_time;          //!< the time when availability data received from the broadcast
    byte fullness[5];         //!< an array of values giving current and predicted percentage fullness

    byte price_hour;          //!< a generic indicator of relative pricing per hour compared to nearby locations
    byte price_day;           //!< a generic indicator of relative pricing per day compared to nearby locations
    byte status;
} SXMParkingLocation;

/************************************************************************
*
*  Module interface
*
************************************************************************/

extern SXESDK_API int sxm_parking_start(void (*)(int, int), // callback
                                        int);               // debugLevel

extern SXESDK_API int sxm_parking_status(SXMStatus *);      // status

extern SXESDK_API int sxm_parking_stop(void);

extern SXESDK_API int sxm_parking_request(const SXMMBR *,// mbr
                   const SXMPoint *,                     // sort point
                   void (*)(ptr, ptr),                   // notification
                   ptr,                                  // user context
                   ptr *);                               // returned handle

extern SXESDK_API int sxm_parking_modify(ptr,           // request
                  const SXMMBR *,                       // mbr
                  const SXMPoint *);                    // sort point

extern SXESDK_API int sxm_parking_remove(ptr);          // request

extern SXESDK_API int sxm_parking_begin(ptr,            // request
                  const SXMPoint *);                    // sort point

extern SXESDK_API int sxm_parking_extract_location(ptr, // request
                    SXMParkingLocation *);              //out

extern SXESDK_API int sxm_parking_end(ptr);            // request

extern SXESDK_API int sxm_parking_set_debug(int);

extern SXESDK_API int sxm_parking_get_debug(int *);

extern SXESDK_API int sxm_parking_get_version(uint *);

extern SXESDK_API int sxm_parking_point_closest(const SXMPoint *, SXMFLOAT *);

extern SXESDK_API int sxm_parking_point_furthest(const SXMPoint *, SXMFLOAT *);

extern SXESDK_API int sxm_parking_get_mbr(ptr, SXMMBR *);

extern SXESDK_API int sxm_parking_set_ppoi_list(const uint, const PPOISpec *,
                                                void (*)( ptr, byte), ptr);

extern SXESDK_API int sxm_parking_poi_begin(const SXMMBR *, ushort);

extern SXESDK_API int sxm_parking_poi_extract_station(SXMParkingLocation *);

extern SXESDK_API int sxm_parking_poi_end(void);

extern SXESDK_API int sxm_parking_cfile_clean(void);

extern DEPRECATED SXESDK_API int sxm_parking_save_data(void);

extern DEPRECATED SXESDK_API int sxm_parking_set_auto_save(BOOL);

/**
 * @}
 */
 
#ifdef __cplusplus
}
#endif

#endif
