/************************************************************************
 *                                                                      *
 *            SXM_COMMON.H                                              *
 *            ============                                              *
 *                                                                      *
 *  Copyright 2013 Sirius XM Radio, Inc.                                *
 *  All Rights Reserved.                                                *
 *  Licensed Materials - Property of Sirius XM Radio, Inc.              *
 *                                                                      *
 *    Common Types and definitions for the SDK interface                *
 *                                                                      *
 *    Also includes all the utility layer routines                      *
 *                                                                      *
 ************************************************************************/

#ifndef SXM_COMMON_H
#define SXM_COMMON_H

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sxm_build.h>
#include <sxm_typ.h>
#include <sxm_iomanager.h>

#ifdef __cplusplus
extern "C" {
#endif

/************************************************************************
 *                                                                      *
 *            Common Constants                                          *
 *            ================                                          *
 *                                                                      *
 ************************************************************************/

#define SXM_SERVICE_NAME_LEN_MAX    (16) // Including '\0'
#define SXM_DB_INVALID_ENTRY        (0xffffffff) // no DB entry

/************************************************************************
 *                                                                      *
 *            Architectural Constants                                   *
 *            =======================                                   *
 *                                                                      *
 *    The following constants are 'fixed by architecture', changing     *
 *    these numbers WILL break something, so don't even think about it  *
 *                                                                      *
 ************************************************************************/

#define SXM_ARCH_BITS_IN_BYTE       (8)
#define SXM_ARCH_INT_SIZE           (sizeof(uint))
#define SXM_ARCH_INT_BITS           (SXM_ARCH_INT_SIZE * SXM_ARCH_BITS_IN_BYTE)
#define SXM_ARCH_INT_MAX_VAL        (uint)(-1)
#if SXM_USE_WIDEBAND
#define SXM_ARCH_NUM_SIDS           (1024)
#else
#define SXM_ARCH_NUM_SIDS           (384)
#endif
#define SXM_ARCH_NUM_CATEGORIES     (256)
#define SXM_ARCH_NUM_CHANNELS       (1024)
#define SXM_ARCH_NUM_DSIS           (1024)
#define SXM_ARCH_NUM_DMIS           (1024)

// Common useful constants
#define SXM_SXE_TIME_EPOCH0         (15340U)

#define SECONDS_IN_MINUTE           (60)
#define MINUTES_IN_HOUR             (60)
#define HOURS_IN_DAY                (24)
#define MINUTES_IN_DAY              (HOURS_IN_DAY * MINUTES_IN_HOUR)
#define SECONDS_IN_HOUR             (SECONDS_IN_MINUTE * MINUTES_IN_HOUR)
#define SECONDS_IN_DAY              (SECONDS_IN_HOUR * HOURS_IN_DAY)

// CRC related constants
#define CRC_BYTELEN                 (4)
#define CRC_BITLEN                  (AU_CRC_BYTELEN * BITS_IN_BYTE)
#define CRC_INVERSION_VALUE         (0xFFFFFFFFL)

/* Defines the float point main data type */
#if SXM_USE_FIX
typedef fix SXMFLOAT;
#else
typedef float SXMFLOAT;
#endif

/************************************************************************
 *                                                                      *
 *            Error Codes                                               *
 *            ===========                                               *
 *                                                                      *
 ************************************************************************/

#define RESULT_ACTION_EXECUTOR_ \
    RESULT_ACTION_(SXM_E_OK                         ,0) \
    RESULT_ACTION_(SXM_E_BUSY                       ,1) \
    RESULT_ACTION_(SXM_E_ERROR                      ,2) \
    RESULT_ACTION_(SXM_E_FAULT                      ,3) \
    RESULT_ACTION_(SXM_E_INVAL                      ,4) \
    RESULT_ACTION_(SXM_E_NOENT                      ,5) \
    RESULT_ACTION_(SXM_E_NOMEM                      ,6) \
    RESULT_ACTION_(SXM_E_PIPE                       ,7) \
    RESULT_ACTION_(SXM_E_STATE                      ,8) \
    RESULT_ACTION_(SXM_E_THREAD                     ,9) \
    RESULT_ACTION_(SXM_E_TIMEOUT                    ,10) \
    RESULT_ACTION_(SXM_E_CORRUPT                    ,11) \
    RESULT_ACTION_(SXM_E_UNSUPPORTED                ,12) \
    RESULT_ACTION_(SXM_E_RESOURCE                   ,13) \
    RESULT_ACTION_(SXM_E_UPDATING                   ,14) \
    RESULT_ACTION_(SXM_E_RESET                      ,15) \
    RESULT_ACTION_(SXM_E_NO_DB                      ,16) \
    RESULT_ACTION_(SXM_E_BAD_DB                     ,17) \
    RESULT_ACTION_(SXM_E_SIZE                       ,200) \
    RESULT_ACTION_(SXM_E_BAD_PACKET                 ,201) \
    RESULT_ACTION_(SXM_E_ARRAY_SIZE                 ,202) \
    RESULT_ACTION_(SXM_E_BUF_SIZE                   ,203) \
    RESULT_ACTION_(SXM_E_BAD_STRING                 ,204) \
    RESULT_ACTION_(SXM_E_BAD_LABEL                  ,205) \
    RESULT_ACTION_(SXM_E_NO_LOCS                    ,514) \
    RESULT_ACTION_(SXM_E_BAD_LOCS                   ,515) \
    RESULT_ACTION_(SXM_E_NO_STATES                  ,516) \
    RESULT_ACTION_(SXM_E_BAD_STATES                 ,517) \
    RESULT_ACTION_(SXM_E_NO_SUBNOTIFICATION         ,518) \
    RESULT_ACTION_(SXM_E_BAD_CRC                    ,600) \
    RESULT_ACTION_(SXM_E_PATHFILE_ERROR             ,601) \
    RESULT_ACTION_(SXM_E_LINK_ESTABLISHED           ,700) \
    RESULT_ACTION_(SXM_E_LINK_PENDING               ,701) \
    RESULT_ACTION_(SXM_E_LINK_LOSS                  ,702) \

// define the macro calls that are inside the RESULT_ACTION_EXECUTOR_
// so that it produces the error code enum values.
#define RESULT_ACTION_(a,b) a = (b),

//use RESULT_ACTION_EXECUTOR_ macro to generate the enum values
typedef enum sxm_result_code {
    RESULT_ACTION_EXECUTOR_
} SXMResultCode;

// un-define the macro calls that are inside the RESULT_ACTION_EXECUTOR_
#undef RESULT_ACTION_

/***********************************/
/* SYSTEM PARAMETERS               */
/***********************************/
#define SXI_SP_EQ_GAINS_STEPS               25
#define SXI_SP_MAX_FWU_BYTES_PER_PACKET     256
#define SXI_SP_MAX_CHANID                   999
#define SXI_SP_MAX_NETWORK_CATEGORIES       63
#define SXI_SP_MAX_NETWORK_CATEGORIES_CNT   (SXI_SP_MAX_NETWORK_CATEGORIES + 1)
#define SXI_SP_USER_CATID_BASE              64
#define SXI_SP_MAX_USER_CATEGORIES          3
#define SXI_SP_MAX_SID                      (SXM_ARCH_NUM_SIDS - 1)
#define SXI_SP_MAX_SID_CNT                  (SXI_SP_MAX_SID + 1)
#define SXI_SP_MAX_BASIC_SONG_TITLE_CHARS   16
#define SXI_SP_MAX_SONG_TITLE_CHARS         36
#define SXI_SP_MAX_CONTENT_INFO_CHARS       36
#define SXI_SP_MAX_AUDIO_PACKET_BYTES       1296
#define SXI_SP_MAX_DMI_CNT                  96
#define SXI_SP_ALL_CHANNEL_CATEGORY         255
#define SXI_SP_VOL_MINUNIT                  1
#define SXI_SP_VOLADJ_MAX                   24
#define SXI_SP_VOLADJ_MIN                  -96
#define SXI_SP_TON_VOLADJ_MAX               0
#define SXI_SP_TON_VOLADJ_MIN              -26
#define SXI_SP_MAX_STATUS_MON_LIST_CNT      24
#define SXI_SP_MAX_FEAT_MON_LIST_CNT        16
#define SXI_SP_MAX_TMI_MON_LIST_CNT         32
#define SXI_SP_MAX_CMI_MON_LIST_CNT         32
#define SXI_SP_MAX_GLOBAL_MON_LIST_CNT      64
#define SXI_SP_MAX_EMD_MON_LIST_CNT         SXI_SP_MAX_GLOBAL_MON_LIST_CNT
#define SXI_SP_MAX_GLOBAL_FIELD_CNT         255
#define SXI_SP_MAX_DSI_CNT                  1024
#define SXI_SP_MAX_MEMORY_WRITES            64

#define SXI_SP_MAX_MON_VALUE_LEN            4
#define SXI_SP_MAX_MON_VALUE_CNT           128

#define SXI_SP_MAX_CMI_SIMILAR_CHAN_LIST   8

#define SXI_SP_MAX_USER_CAT_NAME_LEN       17
#define SXI_SP_MAX_BAND_GAIN_CNT           10

#define SXI_SP_MIN_TUNEMIXID                4097
#define SXI_SP_MAX_TUNEMIXID                4106

#define SXI_CATID_MIN_NUM                   1
#define SXI_CATID_MAX_NUM                   (SXI_SP_MAX_NETWORK_CATEGORIES + SXI_SP_MAX_USER_CATEGORIES)
#define SXI_CATID_ALL_NUM                   255
#define SXI_CATID_MAX_NUM_CNT               (SXI_CATID_MAX_NUM + 1)

/************************************************************************
 *                                                                      *
 *            File Operations                                           *
 *            ===============                                           *
 *                                                                      *
 ************************************************************************/

extern SXESDK_API FILE *sxm_file_open(char,         // type
                                      const char *, // module
                                      const char *, // file
                                      const char *);// mode

extern SXESDK_API FILE *sxm_file_open_or_create(char,         // type
                                                const char *, // module
                                                const char *, // file
                                                const char *);// mode

extern SXESDK_API int sxm_file_close(FILE *);       // FILE pointer

extern SXESDK_API int sxm_file_remove(char,         // type
                                      const char *, // module
                                      const char *);// file

extern SXESDK_API int sxm_file_rename(char,         // type
                                      const char *, // module
                                      const char *, // from
                                      const char *);// to

extern SXESDK_API int sxm_file_copy(char,           // type
                                    const char *,   // module
                                    const char *,   // from
                                    const char *);  // to

extern SXESDK_API int sxm_file_size(char,           // type
                                    const char *,   // module
                                    const char *,   // file
                                    size_t *);      // size

/************************************************************************
 *                                                                      *
 *            FixPoint Arithmetic and GeoSpatial Data                   *
 *            =======================================                   *
 *                                                                      *
 *    The first two structures are for external users passing data      *
 *    into the SDK in either fix or float formats.                      *
 *    The second two structures are for the internal operations         *
 *    that know they are in fix mode.                                   *
 *                                                                      *
 ************************************************************************/

typedef struct
{
    SXMFLOAT lon;
    SXMFLOAT lat;
} SXMPoint;

typedef struct
{
    SXMPoint ll;
    SXMPoint ur;
} SXMMBR;

typedef struct
{
    fix lon;
    fix lat;
} SXMFixPoint;

typedef struct
{
    SXMFixPoint ll;
    SXMFixPoint ur;
} SXMFixMBR;

// check for negative distance value

#if SXM_USE_FIX
#define sxm_is_negative(_i)     ((_i) < 0)
#else
#define sxm_is_negative(_i)     ((_i) < 0.0)
#endif

//   these are always fixpoint routines

/***************************************************************************//**
 * Converts an integer into a fixpoint value
 * \param[in] _i input as an integer value
 * \return fixpoint representation of the input
 ******************************************************************************/
#define sxm_fix_int(_i)        ((fix)((_i) * 32768))

extern SXESDK_API int sxm_fix_real(int _i, uint _j);

extern SXESDK_API fix sxm_fix_cos(fix);                     // this

extern SXESDK_API fix sxm_fix_sin(fix);                     // this

extern SXESDK_API fix sxm_fix_mul(fix,                      // a
                                  fix);                     // b

extern SXESDK_API fix sxm_fix_div(fix,                      // a
                                  fix);                     // b

extern SXESDK_API void sxm_fix_printable(fix,               // this
                                         int *,             // intpart
                                         int *);            // fracpart

extern SXESDK_API char *sxm_fix_sprint(char *,              // b
                                       fix);                // in

extern SXESDK_API fix sxm_fix_read(const char **);          // inp

extern SXESDK_API fix sxm_fix_root(fix);                    // this

extern SXESDK_API void sxm_fixmbr_about(SXMFixMBR *,        // this
                                        const SXMFixPoint *,// center
                                        fix);               // miles

extern SXESDK_API int sxm_fixmbr_inside(const SXMFixMBR *,      // this
                                        const SXMFixPoint *);   // where

extern SXESDK_API int sxm_fixmbr_intersects(const SXMFixMBR *,  // one
                                            const SXMFixMBR *); // two

extern SXESDK_API void sxm_fixmbr_center(const SXMFixMBR *, // mbr
                                         SXMFixPoint *);    // center
//
//  these are polymorphic routines for SXMFLOAT type
//

extern SXESDK_API void sxm_mbr_about(SXMMBR *,              // mbr
                                     const SXMPoint *,      // center
                                     SXMFLOAT);             // miles

extern SXESDK_API SXMFLOAT sxm_mbr_closest(const SXMMBR *,  // mbr
                                        const SXMPoint *);  // point

extern SXESDK_API SXMFLOAT sxm_mbr_furthest(const SXMMBR *, // mbr
                                         const SXMPoint *); // point

extern SXESDK_API int sxm_mbr_inside(const SXMMBR *,        // mbr
                                     const SXMPoint *);     // where

extern SXESDK_API int sxm_mbr_inside_mbr(const SXMMBR *,    // mbr
                                         const SXMMBR *);   // inner_mbr

extern SXESDK_API int sxm_mbr_valid(const SXMMBR *);        // mbr

extern SXESDK_API int sxm_mbr_valid_eq(const SXMMBR *);     // mbr

extern SXESDK_API void sxm_mbr_reduce(const SXMMBR *,       // mbr
                                      SXMFLOAT,             // factor
                                      const SXMPoint *,     // center
                                      SXMMBR *);            // reduced_mbr

extern SXESDK_API void sxm_mbr_scale(const SXMMBR *,        // mbr
                                     SXMFLOAT,              // factor
                                     SXMMBR *);             // scaled_mbr 

extern SXESDK_API void sxm_mbr_center(const SXMMBR *,       // mbr
                                      SXMPoint *);          // center

extern SXESDK_API int sxm_mbr_intersects(const SXMMBR *,    // one
                                         const SXMMBR *);   // two

extern SXESDK_API int sxm_mbr_intersects_get(const SXMMBR *,    // one
                                             const SXMMBR *,    // two
                                             SXMMBR *);         // out

extern SXESDK_API int sxm_mbr_intersects_polygon(const SXMMBR *,    // mbr
                                                 const SXMPoint *,  // points
                                                 uint,              // count
                                                 int);              // closed

extern SXESDK_API SXMFLOAT sxm_point_distance(const SXMPoint *, // one
                                           const SXMPoint *);   // two

extern SXESDK_API SXMFLOAT sxm_point_true_distance(SXMFLOAT);   // distance2

extern SXESDK_API int sxm_point_valid(const SXMPoint *);        // this

/************************************************************************
 *                                                                      *
 *            Common Status Reporting                                   *
 *            =======================                                   *
 *                                                                      *
 ************************************************************************/
#define SXM_SERVICE_OK          (0)
#define SXM_SERVICE_STOPPED     (1)
#define SXM_SERVICE_STOPPING    (2)

#define SXM_SUBS_NONE           (0)
#define SXM_SUBS_PARTIAL        (1)
#define SXM_SUBS_FULL           (2)
#define SXM_SUBS_UNKNOWN        (3)

typedef struct
{
    int service;    // service-level status
    int substate;   // subscription-level status
} SXMStatus;

/************************************************************************
 *                                                                      *
 *            States and Counties                                       *
 *            ===================                                       *
 *                                                                      *
 ************************************************************************/

/** This is states SXM index max value. Full table of states SXM indexes 
 * could be found in SX-9845-0031, SX-9845-0023 and SX-9845-0027
 */
#define SXM_STATES_INDEX_MAX (127)

extern SXESDK_API int sxm_states_register(void);

extern SXESDK_API void sxm_states_unregister(void);

extern SXESDK_API char *sxm_states_abbr(int);           // i

extern SXESDK_API void sxm_states_lookup(SXMMBR *,      // mbr
                                         uint *,        // st
                                         int);          // doc

extern SXESDK_API char *sxm_states_name(int);           // i

extern SXESDK_API int sxm_states_inside(int,            // i
                                        SXMPoint *);    // point

/************************************************************************
 *                                                                      *
 *            RFD Decoding                                              *
 *            ============                                              *
 *                                                                      *
 ************************************************************************/

/** Defines max length of the file name */
#define SXM_RFD_METADATA_FILE_NAME_LEN_MAX  (16) // including '\0'
/** Defines max number of supported segments */
#define SXM_RFD_NUM_FRAGMENTS_MAX           (16)

typedef struct
{
    char name[SXM_RFD_METADATA_FILE_NAME_LEN_MAX];  // includes '\0'
    uint size;
    byte carid;
    byte alg;       // One of RFD_ALGORITHM_...
    byte comp;      // One of RFD_COMPRESSION_...
    ushort lifetime;
    byte nfrags;
    struct {
        ushort id;
        ushort blksize;
        uint fragsize;
        uint crc;
    } fragment[SXM_RFD_NUM_FRAGMENTS_MAX];
    ushort dmi;
    UINT32 crc;
} SXMRfdMetadata;

/************************************************************************
 *                                                                      *
 *            Data Service LLI                                          *
 *            ====================                                      *
 *                                                                      *
 ************************************************************************/

 /** Defines set of I/O related statistics */
typedef struct _sxm_io_stat {
    size_t sectorsWritten; //!< Number of written SXMSector elements
} SXMIOStat;

/** Defines data service timer callback.
 * \param[in] pArg value of \ref SXMDataService::dsuser
 */
typedef void (*SXMDataServiceTimerCallback)(void *pArg);

/** Cycle data saving callback.
* \param[in] pStat the save operation statistics
* \param[in] pArg the callback argument which was passed on registration
* \return Corresponding error code
*/
typedef int (*SXMDataServiceSaveDataCallback)(SXMIOStat *pStat, void *pArg);

/** Defines I/O Manager registration info */
typedef struct {
    /** the callback function to be called once the manager decides
    * to let the I / O to the registration. */
    SXMDataServiceSaveDataCallback pFunc;
    /** the argument which is passed thru the callback */
    void *pArg;
    /** the max quota required by the service per tick in bytes */
    size_t tQuota;
    /** human readable tag */
    const char *psTag;
} SXMIOManagerRegInfo;

enum {
    /** Defines number of supported DSIs per data service */
    SXM_LLI_DATA_SERVICE_DSI_COUNT = 6,
    /** Defines size of the DMI filter array */
    SXM_LLI_DATA_SERVICE_DMI_FILTER_SIZE = (SXM_ARCH_INT_BITS + SXI_SP_MAX_DMI_CNT - 1) / SXM_ARCH_INT_BITS
};

typedef struct
{
    ushort dmis[SXI_SP_MAX_DMI_CNT];
    ushort dmic;
    ushort status;
#if (SXM_USE_GEN8)
    /* signature needed for Gen8 to check for repeats */
    UINT32 sig[SXI_SP_MAX_DMI_CNT];
#endif
} SXMDMIList;

typedef struct
{
    ushort dsi;
    SXMDMIList dmi;
    SXMDMIList pending;
    uint filter[SXM_LLI_DATA_SERVICE_DMI_FILTER_SIZE];
    /** Subscription state for each DSI in service with multiple DSIs */
    ushort substate;
} SXMDataDsi;

typedef struct
{
    /** Service name */
    char service[SXM_SERVICE_NAME_LEN_MAX];
    SXMDataDsi dsis[SXM_LLI_DATA_SERVICE_DSI_COUNT];
    SXMStatus *status;
    uint debug_level;
    uint timer_mins;
    uint maxau;
    uint carsize;                           //!< for RFD parser
    void (*report)(int, int, ptr);          //!< status report
    void (*complete)(byte *, int, ptr);     //!< AU complete
    /** AU complete (extended version) */
    void (*complete_ex)(void *, size_t, void*, ushort dsi, ushort dmi);
    SXMDataServiceTimerCallback timer;      //!< timer callback
    /** queue callback to handle any custom messages which were not handled by the LLI */
    void (*queue_callback)(uint msgType, uint msgSize, void *pData);
    int (*collected)(SXMRfdMetadata *);
    SXMIOManagerRegInfo iom;                 //!< I/O Reg info
    DEPRECATED BOOL auto_save_on_stop;       //!< Enables saving of cycle file data on stop
    ptr dsuser;                              //!< User data for service callback
} SXMDataService;

extern int sxm_lli_on_thread(ptr);              // handle

extern int sxm_lli_remove(ptr,                  // handle
                          int,                  // dsi
                          int);                 // dmi

extern int sxm_lli_request(ptr,                 // handle
                           int,                 // dsi
                           int);                // dmi

extern int sxm_lli_update(ptr,                  // handle
                          int,                  // dsi
                          ushort,               // dmic
                          ushort *);            // dmis

extern int sxm_lli_rfd_add(ptr,                 // handle
                           byte *,              // packet
                           int);                // pl

extern int sxm_lli_rfd_init(ptr);               // handle

extern FILE *sxm_lli_rfd_open(ptr,              // handle
                              SXMRfdMetadata *, // rfd
                              int);             // frag

extern int sxm_lli_rfd_start(ptr,               // handle
                             SXMRfdMetadata *); // rfd

extern int sxm_lli_rfd_permil_complete(ptr,     // handle
                                       int,     // id
                                       int *);  // permilComplete)

extern int sxm_lli_rfd_status(ptr,              // handle
                              int,              // id
                              SXMRfdMetadata *);// rfd

extern int sxm_lli_rfd_remove(ptr,              // handle
                              int);             // id

extern int sxm_lli_start(SXMDataService *,      // ds
                         ptr *);                // handle

extern int sxm_lli_stop(ptr);                   // handle

extern int sxm_lli_timer(ptr,                   // handle
                         SXMDataServiceTimerCallback, // callback
                         uint);                 // mins

extern int sxm_lli_send_event(ptr,              // handle
                              const void *,     // pData
                              uint,             // msgType
                              uint);            // msgSize

extern void sxm_lli_rfd_stop(ptr);              // handle

extern void sxm_lli_rfd_set_db_updated(ptr);    // handle

/************************************************************************
 *                                                                      *
 *            TimeStamps                                                *
 *            ==========                                                *
 *                                                                      *
 ************************************************************************/

typedef struct
{
    ushort year;
    byte mon;
    byte day;
    byte hour;
    byte min;
    byte sec;
} SXMTime;

#define sxm_time_epoch_from_time(_t)    (ushort)((_t) / SECONDS_IN_DAY)

#define sxm_time_epoch_to_time(_e)      (uint)((_e) * SECONDS_IN_DAY)

extern SXESDK_API void sxm_time_decode(time_t,     // now
                                       SXMTime *); // t

extern SXESDK_API time_t sxm_time_encode(SXMTime *); // t

extern SXESDK_API time_t sxm_time_now(void);

extern SXESDK_API time_t sxm_time_now_utc(void);

extern SXESDK_API int sxm_time_set_utc_offset(int);

extern SXESDK_API int sxm_time_get_utc_offset(void);

/************************************************************************
 *                                                                      *
 *            Starter and Common Routines for the SDK                   *
 *            =======================================                   *
 *                                                                      *
 ************************************************************************/

/*************************/
/* SXE VERSION NUMBER    */
/*************************/
typedef struct
{
    byte major_ver;     //!< SXe Major Version Number
    byte minor_ver;     //!< SXe Major Version Number
    byte inc_ver;       //!< SXe incremental Version Number
    byte month;         //!< SXe release month
    byte day;           //!< SXe release day
    ushort year;        //!< SXE release year
} SXMSxeVerNum;

/**
 * \defgroup core_pubapi Core API
 * \brief The core APIs for the SXe SDK
 *
 * @{
 */

/********************************************************************************/
/**
 * A typdef for a pointer to a function that generates file paths in an
 * implementation specific way for files used by SXe.
 *
 * The function must return a string that is suitable for passing to \c fopen() to
 * open the file.
 *
 * The function should generate a unique file path for unique combinations of
 * \p fileType, \p moduleName, and \p fileName. The function should always return
 * the same file path when presented with the same combination of \p fileType,
 * \p moduleName, and \p fileName.
 *
 * The \p fileType param indicates the requested file system storage class type
 * for the file.
 *
 * Type| File System Storage Characteristics
 * ----| -------------------------------------------------------------------------------
 * 'F' | A regular file in the filing system that can be created, deleted, read, written
 * 'R' | A read-only file, that was created outside of SXe and will never be written to by SXe
 * 'T' | A transient save file that will be written to during normal vehicle operation
 * 'W' | A file that will be written to (updated) during normal vehicle operation
 *
 * \param[out] path a never-NULL pointer to a character buffer into which this
 *                  function shall store the generated file path
 * \param[in]  pathSize the size in bytes of the character buffer referenced by
 *                      \p path (i.e. \p path can hold strings with a \c strlen() of
 *                      up to \p pathSize - 1 characters)
 * \param[in]  fileType a single char ('F', 'R', 'T', or 'W') indicating the
 *                      file system storage class type
 * \param[in]  moduleName the name of the module requesting the path
 * \param[in]  fileName a module-unique name for the file
 *
 * \return either \p path (to indicate success) or NULL (pathSize was too small
 *         to hold the generated path)
 *
 ********************************************************************************/
typedef char *(*SXMMakePathCallback_t)(char *path,
                                       uint pathSize,
                                       char fileType,
                                       const char *moduleName,
                                       const char *fileName);

/***************************************************************************//**
 *
 * A typedef for a pointer to a function that consumes printable text that can be
 * used to output log messages in an implementation specific way.
 *
 * Examples are:
 *  - output to a console
 *  - output to a debug port
 *  - written to a debug log
 *  - stored in a RAM buffer
 *  - ignored
 *
 * The prototype of this function is deliberately defined to be identical to the
 * standard C \c puts() function so that variables of this type can simply contain
 * a pointer to the \c puts() function. Alternatively the application can point to
 * a custom logging function. As with \c puts(),
 * the function must supply the trailing newline character at the end of the text.
 * This also isolates the SXe SDK from different operating system line-termination
 * requirements.
 *
 * \param[in] log_line a pointer to a NUL-terminated line of printable text (not
 *                     including the new-line character)
 *
 * \return The return value of the function is ignored by the library.
 ********************************************************************************/
typedef int (*SXMLogHandler_t)(const char * log_line);
#ifdef SXM_EXT_LOGGING
#include <sxm_stdtrace.h>
/***************************************************************************//**
 *
 * A typedef for a pointer to a function that consumes additional debug
 * parameters which allow the application to filter debug messages.
 *
 * \param[in] debugArea debug area for the reported log message
 * \param[in] debugType debug type for the reported log message
 * \param[in] logTag name of the SXe component which the log message belongs to
 * \param[in] prefix prefix to denote call type
 * \param[in] function name of the function which the log is printed from
 * \param[in] logMessage a pointer to a NUL-terminated line of printable text
 *                       (not including the new-line character)
 *
 ******************************************************************************/
typedef void (*SXMExtLogHandler_t)(SxmDebugArea debugArea,
                                   SxmDebugType debugType,
                                   const char *logTag,
                                   const char *prefix,
                                   const char *function,
                                   const char *logMessage);
#endif /* #ifdef SXM_EXT_LOGGING */

/** SXMSdkConfig fields validity masks */
typedef enum {
    SXM_SDK_CONFIG_NONE         = 0x0000, //!< Nothing is set
    SXM_SDK_CONFIG_I2S          = 0x0001, //!< I2S
#if (SXM_USE_GEN8)
    SXM_SDK_CONFIG_PLAY_SECONDS = 0x0002, //!< Scan item play sec
#else
    SXM_SDK_CONFIG_MREF         = 0x0002, //!< MRef Enabling
#endif
    SXM_SDK_CONFIG_LANGUAGE     = 0x0004,  //!< Language
    SXM_SDK_CONFIG_UNIT         = 0x0008,  //!< Units
    SXM_SDK_CONFIG_RFD_MODE     = 0x0010,  //!< RFD operating mode
#if (!SXM_USE_GEN8)
    SXM_SDK_CONFIG_CFM_WAIT     = 0x0020,  //!< Confirmation Wait Time
#endif
    SXM_SDK_CONFIG_IOMAN_PROFILE = 0x0040   //!< I/O Manager Profile
} SXMSdkConfigFlags;

/** Defines possible I2S modes for configurations */
typedef enum {
    SXM_SDK_I2S_SLAVE = 0, //!< Slave
    SXM_SDK_I2S_MASTER = 1 //!< Master
} SXMSdkI2SControl;

/** Defines supported languages */
typedef enum {
    SXM_SDK_LANG_ENGLISH = 0,
    SXM_SDK_LANG_SPANISH,
    SXM_SDK_LANG_FRENCH
} SXMSdkLanguage;

/** Defines supported units */
typedef enum {
    SXM_SDK_UNIT_IMPERIAL = 0,
    SXM_SDK_UNIT_METRIC
} SXMSdkUnit;

/** Defines RFD operating modes */
typedef enum {
    SXM_SDK_RFD_FILE_SYSTEM = 0,
    SXM_SDK_RFD_RAM_BUFFER
} SXMSdkRfdMode;


#if (SXM_USE_GEN8)
#define SCAN_ITEM_PLAY_SEC_MIN      (5)
#define SCAN_ITEM_PLAY_SEC_MAX      (15)
#define SCAN_ITEM_PLAY_SEC_DFLT     (6)
#endif

/** Defines I/O Manager profile */
typedef struct {
    const void *pSig; //!< Signature; Has to be \p NULL
    time_t cadence; //!< Cadence in seconds
    size_t quota; //!< Per-cadence quota
    size_t cycle_quota; //!< Service power cycle quota
} SXMIOManagerProfile;

/** \name I/O Manager Profiles
  * @{
  */
/** Immediate profile */
SXESDK_API extern const SXMIOManagerProfile SXM_IOMANAGER_IMMEDIATE_PROFILE;
/** Small profile */
SXESDK_API extern const SXMIOManagerProfile SXM_IOMANAGER_SMALL_PROFILE;
/** Medium profile */
SXESDK_API extern const SXMIOManagerProfile SXM_IOMANAGER_MEDIUM_PROFILE;
/** Large profile */
SXESDK_API extern const SXMIOManagerProfile SXM_IOMANAGER_LARGE_PROFILE;
/** Power-Down profile */
SXESDK_API extern const SXMIOManagerProfile SXM_IOMANAGER_POWER_DOWN_PROFILE;
/** @} */

/** SDK basic configuration items */
typedef struct {
    /** Validity flags for the following fields */
    uint flags;

#if (SXM_USE_GEN8)
    /** Number of seconds to play scan item before going to the next,
     * valid if #SXM_SDK_CONFIG_PLAY_SECONDS mask is set
     */
    byte playSeconds;
#else
    /** Metadata Reference Mode (\p TRUE - enable, \p FALSE - disable),
     * valid if #SXM_SDK_CONFIG_MREF mask is set
     */
    BOOL MRefEnabled;
    /** Selects the time the Module will wait for the Host to send the
     * confirmation of an indication message before sending the next indication.
     * Refer to SX-9845-0097 (MdouleCfgCmd) for allowed values. 
     * Valid if #SXM_SDK_CONFIG_CFM_WAIT mask is set
     */
    byte ConfirmationWaitTime;
#endif
    /** Defines I2S mode. valid if #SXM_SDK_CONFIG_I2S mask is set */
    SXMSdkI2SControl I2SControl;
    /** SXe SDK Language. valid if #SXM_SDK_CONFIG_LANGUAGE mask is set */
    SXMSdkLanguage lang;
    /** SXe SDK Unit. valid if #SXM_SDK_CONFIG_UNIT mask is set */
    SXMSdkUnit unit;
    /** SXe SDK RFD operating mode. valid if #SXM_SDK_RFD_MODE mask is set */
    SXMSdkRfdMode mode;
    /** SXe I/O Manager profile. valid if SXM_SDK_IOMANAGER_PROFILE is set */
    SXMIOManagerProfile ioman_profile;
} SXMSdkConfig;

extern SXESDK_API int sxm_sdk_set_config(const SXMSdkConfig *);

extern SXESDK_API int sxm_sdk_get_config(SXMSdkConfig *);

extern SXESDK_API int sxm_sdk_start(const char *,
                                    SXMMakePathCallback_t,
                                    SXMLogHandler_t);

extern SXESDK_API int sxm_sdk_cfiles_clean(SXMMakePathCallback_t,
                                    SXMLogHandler_t);

extern SXESDK_API int sxm_sdk_status(SXMStatus *);

extern SXESDK_API int sxm_sdk_stop(void);

DEPRECATED extern SXESDK_API char sxm_sdk_get_option(const char *); // key

DEPRECATED extern SXESDK_API int sxm_sdk_set_option(const char *,   // key
                                                    char);          // value

extern SXESDK_API int sxm_sdk_get_version(SXMSxeVerNum *);

extern SXESDK_API const char *sxm_sdk_get_device_path(void);

extern SXESDK_API const char *sxm_sdk_format_result(int err);

#ifdef SXM_EXT_LOGGING
extern SXESDK_API void sxm_sdk_set_extlog(SXMExtLogHandler_t);
#endif /* #ifdef SXM_EXT_LOGGING */
/** @} */ //ends the \defgroup core_pubapi

/************************************************************************
 *                                                                      *
 *            Platform Support                                          *
 *            ================                                          *
 *                                                                      *
 ************************************************************************/

extern SXESDK_API void sxe_sleep(uint sec);

#ifdef __cplusplus
}
#endif

#endif // SXM_COMMON_H
