/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/**********************************************************************//**
 * \file    sxm_apogee.h
 * \author  Leslie French, David Franklin
 * \date 8/20/2013
 *
 * This is the Header for the production Apogee Traffic Service.
 ************************************************************************/
#ifndef SXM_APOGEE_H
#define SXM_APOGEE_H

#ifndef SXM_SDK_H
#include "sxm_sdk.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

#define SXM_APOGEE_REQ_FLOW                 1       // real-time flow on main linears 
#define SXM_APOGEE_REQ_RAMPS                2       // real-time flow on ramps
#define SXM_APOGEE_REQ_CONSTRUCTION         4       // construction messages
#define SXM_APOGEE_REQ_INCIDENT             8       // incident messages
#define SXM_APOGEE_REQ_P1                   16      // first predictive flow
#define SXM_APOGEE_REQ_P2                   32      // second predictive flow
#define SXM_APOGEE_REQ_P3                   64      // third predictive flow
#define SXM_APOGEE_REQ_P4                   128     // fourth predictive flow
#define SXM_APOGEE_REQ_FORECAST             256     // all forecast flows
#define SXM_APOGEE_REQ_HISTORICAL           512     // historical flows

#define SXM_APOGEE_REQ_REALTIME             15      // all real-time components
#define SXM_APOGEE_REQ_PREDICTIVE           240     // all predictive components

#define SXM_APOGEE_MAX_FLOW_COUNT           1024
#define SXM_APOGEE_MAX_RAMP_COUNT           4096

#define SXM_APOGEE_MAX_TABLE_COUNT          36
#define SXM_APOGEE_MAX_TABLE_BSA_COUNT      64
#define SXM_APOGEE_MAX_BSA_LINEAR_COUNT     1600
#define SXM_APOGEE_MAX_REQ_COUNT            8
#define SXM_APOGEE_MAX_REQ_BSAREF_COUNT     72
#define SXM_APOGEE_COL_TABLE_COUNT          8
#define SXM_APOGEE_MAX_COL_TABLE_CI_COUNT   64
#define SXM_APOGEE_HIST_NUM_SEASONS         2
#define SXM_APOGEE_HIST_NUM_DAYS            7
#define SXM_APOGEE_HIST_NUM_INTERVALS       96
#define SXM_APOGEE_PRED_DATASET_COUNT       4

/**
    The FlowVector object contains the speed and flow data for a single 
    linear in a BSA.

    The first row of speed values represents the positive direction in RDS/Alert-C and runs from tmc_start 
    to tmc_end,  the second row represents the negative direction and runs from tmc_end to tmc_start. All 
    four arrays are in units of 1/8 TMC segment. The valid value is the maximum of the two directions.  
    For a flow vector that terminates in both directions, valid would be 8 X (number_of_tmc_points - 1).
    In the case where the flow vector continues into another vector in one or both the other directions, 
    valid would be 8 X number_of_tmc_points. The actual number of points is determined by the topology in the TMC tables. 
*/
typedef struct {
    /** The starting TMC Point index in the linear. */
    ushort tmc_start;
    /** The ending TMC Point index in the linear. */
    ushort tmc_end;
    /** Up to SXM_APOGEE_MAX_FLOW_COUNT speed bucket values can be stored in each direction. */
    byte speed[SXM_APOGEE_MAX_FLOW_COUNT][2];
    /** Up to SXM_APOGEE_MAX_FLOW_COUNT flow (congestion) bucket values can be stored in each direction. */
    byte flow[SXM_APOGEE_MAX_FLOW_COUNT][2];
    /** The number of valid entries in the speed and flow vectors. */
    ushort count;
}SXMApogeeFlowVector;

/** 
    The RampVector object contains all the Ramp flow values for a single BSA, 
    either the TMC-defined ramps or the SXM-defined ramps. 

    Each byte contains the congestion level for a single ramp, in BSA index order.
*/
typedef struct {
    /** Up to SXM_APOGEE_MAX_RAMP_COUNT ramp congestion values can be stored. */
    byte flow[SXM_APOGEE_MAX_RAMP_COUNT];
    /** TMC associated with the ramp congestion value */
    ushort tmc[SXM_APOGEE_MAX_RAMP_COUNT];
    /** Ramp topology for SXM defined ramps */
    byte type[SXM_APOGEE_MAX_RAMP_COUNT];
    /** The number of valid entries in the flow vector. */
    ushort count;
}SXMApogeeRampVector;

#define CI_ETYPE_TMC                0
#define CI_ETYPE_GEO_LOC            1

#define CI_MSG_MAX_SIZE             256
/** 
    The CIMessage object contains either a Construction Message or an Incident Message, 
    with the same format.

    The message may be in TMC or SXM format.
*/
typedef struct {
    /** The Alert-C event code for this message. */
    ushort event;
    /** The type of location reference in the union: 0=> tmc, 1=>ll. */
    byte etype;
    union {
        struct {
            /** The starting TMC point location within the BSA. */
            ushort tmc;
            /** The sub-segment offset of the starting location. */
            byte offset;
            /** The Alert-C direction (of queue growth) for the message. */
            byte direction;
            /** The number of 1/8-TMC sub-segments for which the event extends. */
            byte extent;
            /** The starting TMC point location if not within this BSA. */
            ushort start_tmc;
            /** The starting sub-segment offset. */
            byte start_offset;
        }tmc;
        struct {
            /** A geo-reference point (longitude and latitude values). */
            SXMPoint geo;
            /** A mappable street address for the geo-reference. */
            char desc[CI_MSG_MAX_SIZE];
        }ll;
    }loc;
    /** The severity level of this message. */
    byte severity;
    /** Free text description of the event (in addition to the event code). */
    char text[CI_MSG_MAX_SIZE];
    /** A unique identifier for each CI message. */
    uint id;
    /** A sequence number incremented by 1 for each new message. */
    ushort sequence;
}SXMApogeeCIMessage;

/**
    Apogee collection requests operate on a geographic area defined as a list of table
    and BSA pairs.
    The table number is equivalent to the LTN (Location Table Number).
    These are passed across the API using the SXMApogeeRequest structure.
*/
typedef struct {
    /** An array of up to SXM_APOGEE_MAX_REQ_BSAREF_COUNT pairs, each pair packed in one entry. */
    ushort bsaref[SXM_APOGEE_MAX_REQ_BSAREF_COUNT];     // table << 8 | bsa
    /** The number of valid entries in the bsaref array. */
    ushort bsac;
    /** The country code. */
    ushort country_code;
}SXMApogeeRequest;

/**
    Apogee BSA Filters indicate which linears within a given BSA are to be filtered.
*/
typedef struct {
    /** This is the bitfield for linears. */
    uint lf[(SXM_APOGEE_MAX_BSA_LINEAR_COUNT + 31) >> 5];
    /** The count of active filter bits. */
    uint lc;
}SXMApogeeBSAFilter;

/**
    Apogee Filters indicate which set of BSA filters are included for a given request.
*/
typedef struct {
    /** The list of BSA filters to include in the data filtering. */
    SXMApogeeBSAFilter f[SXM_APOGEE_MAX_REQ_BSAREF_COUNT];
    /** The count of active BSAs. */
    uint bsac;         
}SXMApogeeFilter;

//  module interface

#define BSAREF(_table, _bsaindex) (((ushort)(_table)<<8) | (_bsaindex))
#define BSAREF_TABLE(_ref) ((_ref)>>8)
#define BSAREF_INDEX(_ref) ((_ref) & 0xff)

/*****************************************************************************//**
 *
 * This function starts the Apogee Traffic service.
 *
 * \param[in] callback      This is a pointer to a routine that will handle apogee
 *                          module-level event indications.
 * \param[in] debugLevel    This sets the debug level for the Apogee module.
 *
 * \retval SXM_E_OK         Service start is successful.
 * \retval SXM_E_STATE      One of: already started; sdk not started; link not started.
 * \retval SXM_E_FAULT      NULL parameter passed to the function.
 * \retval SXM_E_NOMEM      Memory Allocation Failed.
 * \retval SXM_E_THREAD     Thread creation Failed.
 * \retval SXM_E_PIPE       No radio device was found.
 * \retval SXM_E_NO_LOCS    No Apogee Locations baseline database was found.
 * \retval SXM_E_BAD_LOCS   The Apogee Locations baseline database is corrupt.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_start(void (*callback)(int, int), int debugLevel);

/*****************************************************************************//**
 * This function reports the current Apogee service status.
 *
 * \param[out] ret      A pointer to the SXMStatus object that is populated with the 
 *                      current status information.
 *
 * \retval SXM_E_OK     The status fields were populated successfully.
 * \retval SXM_E_FAULT  A NULL parameter ret was passed into the function.
 *
 * \note    Traffic relies on two DSI's for data. Make sure both DSI's are
 *          subscribed, otherwise the service must report a substate of Unsubscribed.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_status(SXMStatus *ret);

/*****************************************************************************//**
 *
 * This function stops the Apogee service.
 *
 * \retval SXM_E_OK     Success
 * \retval SXM_E_STATE  Service is not running
 * \retval SXM_E_FAULT  NULL pointer to service handle encountered.
 * \retval SXM_E_BUSY   Attempt to stop the service from its own callback
 * \retval SXM_E_NOENT  reference to service structure not found
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_stop(void);

//  request

/*****************************************************************************//**
 *
 * This function submits an Apogee request and starts collecting its data.
 *
 * \param[in] req           A pointer to a request structure containing the
 *                          list of BSAs to be collected.
 * \param[in] types         One of more of the SXM_APOGEE_REQ values.
 * \param[in] notification  A pointer to the notification callback routine.
 * \param[in] callbackData  Application data which will be passed during
 *                          each callback execution.
 * \param[out] handle       A reference for the collection request, if it
 *                          is created.
 *
 * \retval SXM_E_OK        The request has been created.
 * \retval SXM_E_BAD_LOCS  The Apogee Locations baseline database is corrupt.
 * \retval SXM_E_BUSY      The maximum number of requests are already active.
 * \retval SXM_E_NOMEM     Insufficient memory was allocated to complete the request.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_INVAL     A request parameter is invalid.
 * \retval SXM_E_STATE     The Apogee Service is in a state that cannot accept a request.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_request(SXMApogeeRequest *req,
                                    int types, 
                                    void (*notification)(ptr, int, int, int, ptr), 
                                    ptr callbackData, 
                                    ptr *handle);

/*****************************************************************************//**
 *
 * This function will change the area or collection types of an active request.
 *
 * \param[in] handle    A handle for the collection request.
 * \param[in] req -     A pointer to a collection request.
 * \param[in] types -   The set of types to be collected.
 *
 * \retval SXM_E_OK        The request has been modified.
 * \retval SXM_E_BAD_LOCS  The Apogee Locations database is corrupt.
 * \retval SXM_E_BUSY      The request is currently extracting and cannot be changed.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_NOMEM     There is insufficient memory allocated to complete the request.
 * \retval SXM_E_INVAL     An Invalid parameter was passed, (e.g. request handle invalid).
 * \retval SXM_E_STATE     The Apogee Service is in a state that cannot accept a request change.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_modify(ptr handle, SXMApogeeRequest *req, int types);

/*****************************************************************************//**
 *
 * This function deletes a request. 
 *
 * \param[in] handle    A handle for the collection request to delete.
 *
 * \retval SXM_E_OK     The request has been removed.
 * \retval SXM_E_BUSY   The request currently extracting and cannot be deleted.
 * \retval SXM_E_FAULT  A NULL parameter was passed to the function.
 * \retval SXM_E_INVAL  An Invalid parameter was passed (e.g. request handle).
 * \retval SXM_E_STATE  The Apogee Service is in a state that cannot delete a request.
 *
 * \note    Once a request is deleted, the handle for that request must not be used again.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_remove(ptr handle);

//  extract

/*****************************************************************************//**
 *
 * This helper macro formats the subtype needed for a historical extraction.
 *
 * \param[in] _season     Season index (Summer = 0, Winter = 1). Use DST to determine season.
 * \param[in] _dayofweek  Day of the week index (Sun = 0, ... Fri = 5, Sat = 6).
 * \param[in] _isholiday  Holiday setting (Is NOT Holiday = 0, Is Holiday = 1).
 * \param[in] _interval   Interval index (00:00-00:15UTC = 0, 00:15-00:30UTC = 1, ...).
 *
 * \retval subtype        Subtype needed for extraction.
 *
 ********************************************************************************/
#define APOGEE_FORMAT_HIST_SUBTYPE(_season, _dayofweek, _isholiday, _interval) (((_season)<<12) | ( (_isholiday)?0:(_dayofweek)<<8) | (_interval))

/*****************************************************************************//**
 *
 * This function starts an extraction of requested Apogee Traffic data.
 *
 * \param[in] handle    A handle for the collection request to extract.
 * \param[in] type      One (only) of the collection types in the request.
 * \param[in] subType   A sub-type of a specific collection type.
 * \param[in] bsaix     An index of elements in the collection request bsaref
 *                      array representing the BSA to be extracted.
 * \param[out] c1       A returned counter depending on the type of data extracted.
 * \param[out] c2       A returned counter depending on the type of data extracted.
 *
 * \retval SXM_E_OK         The extraction has started.
 * \retval SXM_E_BUSY       The request is currently extracting.
 * \retval SXM_E_FAULT      A NULL parameter was passed to the function.
 * \retval SXM_E_NOENT      No Entry was found (e.g. end of list).
 * \retval SXM_E_INVAL      A parameter was passed with an invalid value.
 * \retval SXM_E_STATE      The Apogee Service or the request is in a state that 
 *                          cannot extract data.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_begin(ptr handle, 
                                        int type, 
                                        int subType, 
                                        int bsaix, 
                                        int *c1, 
                                        int *c2);

/*****************************************************************************//**
 *
 * This function starts an extraction of requested Apogee Traffic data 
 * (user supplied BSAREF variant).
 *
 * \param[in] handle    A handle for the collection request to extract.
 * \param[in] type      One (only) of the collection types in the request.
 * \param[in] subType   A sub-type of a specific collection type.
 * \param[in] bsaref    The BSA reference representing the BSA to be extracted.
 * \param[out] c1       A returned counter depending on the type of data extracted.
 * \param[out] c2       A returned counter depending on the type of data extracted.
 *
 * \retval SXM_E_OK         The extraction has started.
 * \retval SXM_E_BUSY       The request is currently extracting.
 * \retval SXM_E_FAULT      A NULL parameter was passed to the function.
 * \retval SXM_E_NOENT      No Entry was found (e.g. end of list).
 * \retval SXM_E_INVAL      A parameter was passed with an invalid value.
 * \retval SXM_E_STATE      The Apogee Service or the request is in a state that 
 *                          cannot extract data.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_begin_bsaref(ptr handle, 
                                              int type, 
                                              int subType, 
                                              int bsaref, 
                                              int *c1, 
                                              int *c2);

/*****************************************************************************//**
 *
 * This function extracts traffic flow information for linears.
 *
 * \param[in] handle        A handle for the collection request to use.
 * \param[in] lane          The desired lane index to extract.
 * \param[in] linearIndex   The linear index within the BSA to extract.
 * \param[out] out          A pointer to the FlowVector which is filled in
 *                          by the SDK.
 * \param[out] changed      A pointer to a change marker indicating whether 
 *                          or not the extracted data has changed.
 *
 * \retval SXM_E_OK        The extraction was successful.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_NOENT     One of: end of list;
 *                         lane index out of bounds;
 *                         linear index out of bounds.
 * \retval SXM_E_INVAL     A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE     The Apogee Service or the request is in a state that 
 *                         cannot extract data.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_extract_flow(ptr handle, 
                                        int lane, 
                                        int linearIndex, 
                                        SXMApogeeFlowVector *out, 
                                        int *changed);

/*****************************************************************************//**
 *
 * This function extracts traffic flow information for ramp data.
 *
 * \param[in] handle    A handle for the collection request to extract.
 * \param[out] out      A pointer to a RampVector which is filled in by the SDK.
 *
 * \retval SXM_E_OK        The extraction was successful.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_NOENT     The end of the list was encountered before the entry was found.
 * \retval SXM_E_INVAL     A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE     The Apogee Service or the request is in a state that 
 *                         cannot extract data.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_extract_ramp(ptr handle, SXMApogeeRampVector *out);

/*****************************************************************************//**
 *
 * This function extracts construction and incident messages.
 *
 * \param[in] handle    A handle for the collection request to extract.
 * \param[out] out      A pointer to a message struct which is filled in by the SDK.
 *
 * \retval SXM_E_OK        The extraction was successful.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_NOENT     The end of the list was encountered before the entry was found.
 * \retval SXM_E_INVAL     A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE     The Apogee Service or the request is in a state that 
 *                         cannot extract data.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_extract_ci(ptr handle, SXMApogeeCIMessage *out);

/*****************************************************************************//**
 *
 * This function terminates the given extraction.
 *
 * \param[in] handle    A handle for the collection request to end extraction on.
 *
 * \retval SXM_E_OK        The extraction terminated successfully.
 * \retval SXM_E_FAULT     A NULL parameter was passed for the handle.
 * \retval SXM_E_STATE     The Apogee Service or the request is in a state that 
 *                         cannot stop extractions.
 *
 ********************************************************************************/
extern SXESDK_API  int sxm_apogee_end(ptr handle);

//
//  utilities
//

/*****************************************************************************//**
 *
 * This function retrieves all available lane types and flow data for a linear.
 *
 * param[in] handle     A handle for the collection request to use.
 * param[out] ltypes    An array of 8 integers showing the lane types in the request.
 *
 * \retval SXM_E_OK        Successful extraction.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_INVAL     A parameter passed contains an incorrect value.
 * \retval SXM_E_NOENT     No Entry was found (e.g. end of list).
 * \retval SXM_E_STATE     The Apogee Service or the request is in a state that 
 *                         cannot perform extractions.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_get_lanes(ptr handle, int *ltypes);

/*****************************************************************************//**
 *
 * This function retrieves the time ranges of the Predictive datasets currently-available.
 *
 * \param[in] handle    A handle for the collection request to use.
 * \param[out] timeref  A pointer to an time_t value that contains the
 *                      time-reference point (seconds since 00:00:00 Jan 1, 1970)
 *                      from which the predictive times are offsets.
 * \param[out] vec      A pointer to an array of 4 integers that is filled in to
 *                      the contain the offsets, in seconds, of the available
 *                      predictive datasets.
 *
 * \retval SXM_E_OK        The extraction was successful.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_NOENT     No entry was found (e.g. end of list reached).
 * \retval SXM_E_INVAL     A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE     The Apogee Service or the request is in a state that 
 *                         cannot perform extractions.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_get_predictive(ptr handle, time_t* timeref, int *vec);

/*****************************************************************************//**
 *
 * This function retrieves the time ranges of the Forecast datasets currently-available.
 *
 * \param[in] handle    A handle for the collection request to use.
 * \param[out] timeref  A pointer to an time_t value that contains the
 *                      time-reference point (seconds since 00:00:00 Jan 1, 1970)
 *                      from which the forecast times are offsets.
 * \param[out] delta    A pointer to an integer value that is filled in to
 *                      the contain the delta time between offsets, in seconds.
 * \param[out] count    A pointer to an integer value that is filled in to
 *                      the contain the count of available forecast datasets.
 *
 * \retval SXM_E_OK           The extraction was successful.
 * \retval SXM_E_FAULT        A NULL parameter was passed to the function.
 * \retval SXM_E_NOENT        No entry was found (e.g. end of list reached).
 * \retval SXM_E_INVAL        A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE        The Apogee Service or the request is in a state that 
 *                            cannot perform extractions.
 * \retval SXM_E_UNSUPPORTED  API is unsupported.
 *
 * \note
 * Note that if full Apogee featureset is not selected selected at build-time
 * then this function will return SXM_E_UNSUPPORTED.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_get_forecast(ptr handle, time_t* timeref, uint* delta, uint* count);

/*****************************************************************************//**
 *
 * This function extracts traffic flow information for ramp data in a 
 * forecast/historical dataset.
 *
 * \param[in] handle    A handle for the collection request to extract.
 * \param[in] ramptype  The ramp type to retrieve.
 * \param[out] out      A pointer to a RampVector which is filled in by the SDK.
 *
 * \retval SXM_E_OK           The extraction was successful.
 * \retval SXM_E_FAULT        A NULL parameter was passed to the function.
 * \retval SXM_E_NOENT        The end of the list was encountered before the entry was found.
 * \retval SXM_E_INVAL        A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE        The Apogee Service or the request is in a state that 
 *                            cannot extract data.
 * \retval SXM_E_UNSUPPORTED  API is unsupported.
 *
 * \note
 * Note that if full Apogee featureset is not selected selected at build-time
 * then this function will return SXM_E_UNSUPPORTED.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_get_stored_rampdata(ptr handle, int rampType, SXMApogeeRampVector *out);

/*****************************************************************************//**
 *
 * This function converts a geographic rectangle to a list of BSAs and stores it
 * in the collection request.
 *
 * \param[in] req   A pointer to a collection request structure.
 * \param[in]  mbr  A pointer to map bounding rectangle for the collection request.
 *
 * \retval SXM_E_OK        The conversion was successful.
 * \retval SXM_E_BAD_LOCS  The Locations database is corrupt.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_NOMEM     There is insufficient memory to perform the function.
 * \retval SXM_E_INVAL     A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE     The Apogee Service is in a state that cannot perform 
 *                         the function.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_add_bsas(SXMApogeeRequest *req, SXMMBR *mbr);

/*****************************************************************************//**
 *
 * This function builds a linears filter for a single BSA reference using
 * the provided MBR.
 *
 * \param[out] filter   A pointer to a filter structure to be filled.
 * \param[in] mbr       The map bounding rectangle for the extraction request.
 * \param[in] bsaref    The BSA reference for this filter.
 *
 * \retval SXM_E_OK        The filter was built successfully.
 * \retval SXM_E_BAD_LOCS  The Locations database is corrupt.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_NOMEM     There is insufficient memory to complete the filter.
 * \retval SXM_E_INVAL     A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE     The Apogee Service is in a state that cannot perform 
 *                         the function.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_add_linears(SXMApogeeBSAFilter *filter, 
                                        SXMMBR *mbr, 
                                        ushort bsaref);

/*****************************************************************************//**
 *
 * This function builds a ramp filter for a single BSA reference using 
 * the provided MBR.
 *
 * \param[out] filter   A pointer to a filter structure to be filled.
 * \param[in] mbr       The map bounding rectangle for the extraction request.
 * \param[in] bsaref    The BSA reference for this filter.
 * \param[in] type      A value that indicates the type of ramp data, either an TMC
 *                      or an SXM defined ramp. Values are LDB_RAMP_TMC and LDB_RAMP_SXM.
 *
 * \retval SXM_E_OK        The filter was built successfully.
 * \retval SXM_E_BAD_LOCS  The Locations database is corrupt.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_NOMEM     There is insufficient memory to complete the filter.
 * \retval SXM_E_INVAL     A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE     The Apogee Service is in a state that cannot perform 
 *                         the function.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_add_ramps(SXMApogeeBSAFilter *filter, 
                                        SXMMBR *mbr, 
                                        ushort bsaref,
                                        int type);

/*****************************************************************************//**
 *
 * This function builds a linears filter for all the BSAs in the collection request.
 *
 * \param[out] filters  A pointer to a structure containing an array of filters.
 * \param[in] mbr       A map bounding rectangle for the extraction request.
 * \param[in] req       A pointer to a request structure initialized with a BSA list.
 *
 * \retval SXM_E_OK        The filter was built successfully.
 * \retval SXM_E_BAD_LOCS  The Locations database is corrupt.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_NOMEM     There is insufficient memory to complete the filter.
 * \retval SXM_E_INVAL     A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE     The Apogee Service is in a state that cannot perform 
 *                         the function.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_add_all_linears(SXMApogeeFilter *filters, 
                                        SXMMBR *mbr, 
                                        SXMApogeeRequest *req);

/*****************************************************************************//**
 *
 * This function builds a ramps filter for all the BSAs in the collection request.
 *
 * \param[out] filters  A pointer to a structure containing an array of filters.
 * \param[in] mbr       A map bounding rectangle for the extraction request.
 * \param[in] req       A pointer to a request structure initialized with a BSA list.
 * \param[in] type      A ramp type.
 *
 * \retval SXM_E_OK        The filter was built successfully.
 * \retval SXM_E_BAD_LOCS  The Locations database is corrupt.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_NOMEM     There is insufficient memory to complete the filter.
 * \retval SXM_E_INVAL     A parameter was passed with an incorrect value.
 * \retval SXM_E_STATE     The Apogee Service is in a state that cannot perform 
 *                         the function.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_add_all_ramps(SXMApogeeFilter *filters, 
                                        SXMMBR *mbr, 
                                        SXMApogeeRequest *req, 
                                        int type);

 /*****************************************************************************//**
 *
 * This function returns the current locations, SXM Ramps and patterns database version.
 *
 * \param[out] pVerLoc  A pointer to the returned Locations DB version.
 * \param[out] pVerRamp A pointer to the returned SXM Ramps DB version.
 * \param[out] pVerPtrn A pointer to the returned Patterns DB version.
 *
 * \retval SXM_E_OK        The DB version was retrieved and populated into pVer.
 * \retval SXM_E_FAULT     A NULL parameter was passed to the function.
 * \retval SXM_E_STATE     The Apogee service is in a state that cannot return the version.
 * \retval SXM_E_NOENT     No version is available.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_get_version(uint *pVerLoc, uint *pVerRamp, uint *pVerPtrn);

//
//  debug
//

/*****************************************************************************//**
 *
 * This function sets the debug output level.
 *
 * \param[in] debugLevel    The debug level, a 32-bit mask.
 *
 * \retval SXM_E_OK        The debug level was set successfully.
 * \retval SXM_E_STATE     The Apogee service is in a state that cannot set the debug level.
 *
 * \note    If the service has not been started, this function does nothing.
 *          Refer to Section 8 (DEBUGGING), in the User Guide for the SiriusXM
 *          Embedded Software Development Kit (SXe) - Core.  Document number
 *          SX-9880-0058.  A simple approach is to set the debug bit mask to all
 *          ones (0xffffffff) which turns on all the debugging levels.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_set_debug(int debugLevel);

/*****************************************************************************//**
 *
 * This function reports the current debug level.
 *
 * \param[out] pDebugLevel  A pointer to the value for the current debug level.
 *
 * \retval SXM_E_OK      The debug level was saved.
 * \retval SXM_E_STATE   The service is not running.
 * \retval SXM_E_FAULT   The input parameter is NULL.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_get_debug(int *pDebugLevel);

/*****************************************************************************//**
 *
 * This function enables or disables the requested DMI for the requested Apogee DSI.
 *
 * \param[in] dsi       The DSI to filter.
 * \param[in] dmi       The DMI to filter.
 * \param[in] enable    0 to disable the selected DMI; 1 to enable the selected DMI.
 *
 * \retval SXM_E_OK      The DMI was filtered successfully.
 * \retval SXM_E_STATE   The Apogee service is not running.
 * \retval SXM_E_NOENT   The selected DSI/DMI pair is not found.
 *
 ********************************************************************************/
extern SXESDK_API int sxm_apogee_set_filter_debug(int dsi,
                                                  int dmi, 
                                                  int enable);

#ifdef __cplusplus
}
#endif

#endif
