/************************************************************************
 *                                                                      *
 *            Build and Analyse the various SDK files                   *
 *            =======================================                   *
 *                                                                      *
 *  Copyright 2013 Sirius XM Radio, Inc.                                *
 *  All Rights Reserved.                                                *
 *  Licensed Materials - Property of Sirius XM Radio, Inc.              *
 *                                                                      *
 *    Header for SDK file tool                                          *
 *                                                                      *
 ************************************************************************/

#ifndef SDKFILES_COMMON_H
#define SDKFILES_COMMON_H

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>

#include <util/sxm_cfile.h>
#include <util/sxm_tfile.h>
#include <util/sxm_csv.h>
#include <util/sxm_grid.h>
#include <util/sxm_files_input.h>

#include "sdk_tql.h"

#ifdef __cplusplus
extern "C" {
#endif

// Common definitions
// ==================
typedef char *String;

#define SDK_FILE_PATH_LEN_MAX   (255)

#define UNUSED_VAR(_v)  ((void)_v)

#define NUM_ALIGNED_BLOCKS(_size, _d) (((_size) + ((_d) - 1)) / (_d))

#define SDKFILES_MAKEPATH_TEMPLATE_SPECIFIER    '*'

// SXe-like file operation stubs
// =============================

extern void sdkfiles_start(SXMMakePathCallback_t makePath);

extern void sdkfiles_stop(void);

extern const char *sdkfiles_get_path(char type, const char *module, const char *file);

extern FILE *sdkfiles_file_open_or_create(char type, const char *module, const char *file, const char *mode);

extern FILE *sdkfiles_file_open(char type, const char *module, const char *file, const char *mode);

extern void sdkfiles_file_remove(char type, const char *module, const char *file);

extern int sdkfiles_file_close(FILE *pFile);

extern int sdkfiles_sxi_extract_time(SXMTime *ret);

// Builders and checkers
// =====================

extern void apogee_patterns_ver(const char *, BOOL, void *);
extern int apogee_patterns_build(const SXMFilesInput*, int);
extern void apogee_patterns_check_tfile(const char*);

extern void apogee_sxmramp_ver(const char *, BOOL, void *);
extern int apogee_sxmramp_build(const SXMFilesInput*, int);
extern void apogee_sxmramp_check_tfile(const char*);

extern void fuel_ver(const char *, BOOL, void *);
extern int fuel_build(const SXMFilesInput*, int);
extern void fuel_check_cfile(void);
extern void fuel_check_tfile(const char*);

extern int canfuel_build(const SXMFilesInput*, int);
extern void canfuel_check_cfile(void);
extern void canfuel_check_tfile(const char*);

extern void locations_ver(const char *, BOOL, void *);
extern int locations_build(const SXMFilesInput*, int);
extern void locations_check_tfile(const char*);

extern void parking_ver(const char *, BOOL, void *);
extern int parking_build(const SXMFilesInput*, int);
extern void parking_check_cfile(void);
extern void parking_check_tfile(const char*);

extern void sports_check_cfile(void);

extern int states_build(const SXMFilesInput*, int);
extern void states_check_tfile(const char*);

// Utilities
// =========

// Common functions
// ----------------
extern void fatal(const char*, ...);

extern void non_fatal(const char*, ...);

#define string_limit_warning(_fieldName,\
                             _originalString,\
                             _maxBufSize,\
                             _truncatedString,\
                             _csvLine)\
    non_fatal("%s: '%s'. Max size exceeded (%u >= %u). Truncated: '%s'. Line: %u",\
              _fieldName, _originalString, strlen(_originalString),\
              (_maxBufSize), _truncatedString, (_csvLine));

extern void dump(const byte *b, uint len);

// Buffer read/write
// -----------------

extern byte *bput2(byte *, short);

extern byte *bput4(byte *, int);

extern byte *bput1(byte *, unsigned char);

extern byte *bputs(byte *, String);

extern void salign(SXMTFileStream *, size_t);

extern void sputb(SXMTFileStream *, const void *, size_t);

extern void sputs(SXMTFileStream *, const char*);

extern void sput4(SXMTFileStream *, int);

extern void sput2(SXMTFileStream *, short);

extern void sput1(SXMTFileStream *, unsigned char);

extern byte *bget1(byte *, unsigned char*);

extern byte *bget2(byte *, ushort *);

extern byte *bget4(byte *, int *);

extern byte *bgets(byte *, String *);

extern String strtrim(String);

extern uint strsplit(String, char, String[], uint);

extern fix fixread(String);

const char *get_file_name_pointer(const char *pFilePath);

int strncmp_lowercase(const char *pStr1, const char *pStr2, size_t len);

const char * strstr_lowercase(const char *pStr1, const char *pStr2);

// Bit Buffer
// ----------

typedef struct {
    byte * _buff;
    int _size;
    int _w_bit_pos;
    int _w_byte_pos;
    int _r_bit_pos;
    int _r_byte_pos;
    int _err;
}BIT_BUFFER;

extern BIT_BUFFER *bitbuff_create(int size);
extern void bitbuff_destroy(BIT_BUFFER *pBuff);
extern void bitbuff_clear(BIT_BUFFER *pBuff);
extern int bitbuff_add(BIT_BUFFER *pBuff, uint val, int bits);
extern uint bitbuff_next(BIT_BUFFER *pBuff, int size);
extern void bitbuff_align_w(BIT_BUFFER *pBuff);
extern void bitbuff_align_r(BIT_BUFFER *pBuff);
extern int bitbuff_length(BIT_BUFFER *pBuff);
extern byte *bitbuff_get(BIT_BUFFER *pBuff, int offset);
extern void bitbuff_write(BIT_BUFFER *pBuff, int offset, int size, void *pData);

// CRC calculation
// ---------------

extern uint doCRC(const void *buf, uint len);

extern uint doCRC_part(uint c, const void *buf, uint len);

// C-File implementation
// ---------------------

extern int cfile_open(const char*, const char*, const char*);

extern uint cfile_schema(void);

extern void cfile_close(void);

extern byte *cfile_load(uint);

extern uint *cfile_user(void);

extern uint cfile_count(uint);

extern int cfile_version(uint);

extern ushort cfile_length(uint);

extern uint cfile_timestamp(uint);

// Time functions
// --------------

extern char *time_decode(uint);
extern char *unix_time_decode(uint);

// CSV parser implementation
// -------------------------

extern int create_buffer_psv_parser(const SXMFilesInputRecord *pFileRecord,
                                    char separator, uint lineSizeMax,
                                    SxmCsvParser **ppRet);

// GRID Related
// -------------------
extern int grid_check_location(ushort region, fix lat, fix lon);

extern int grid_saveregion(SXMTFile *dbfile, SXMTFileStream *stream, int region,
                           ushort *ib, uint *index_area);

extern const void* grid_encode_phone(void *buff, size_t s, const char *phone);

extern const char *grid_decode_phone(char *phone, size_t s, const void *buff);

extern ushort grid_encode_amen(const char *amen);

extern void grid_cfile_sec_stat(const SXMCFileRootBlock *root, int sno);

// File implementation
// -------------------

extern byte *fileLoad(const char *, uint *);

// Output formatting
// -----------------

// Function to print formatted column with '|' separator
extern void printColumn(uint colWidth,
                        const char *format,
                        ...);

extern void printFooter(uint length);

#ifdef __cplusplus
}
#endif

#endif
